"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCache = exports.MajorVersion = exports.ECPUPerSecond = exports.DataStorage = exports.StorageUnit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const util_1 = require("./util");
/**
 * Storage unit for data storage in ElastiCache Serverless.
 */
var StorageUnit;
(function (StorageUnit) {
    /**
     * Gigabytes
     */
    StorageUnit["GB"] = "GB";
})(StorageUnit || (exports.StorageUnit = StorageUnit = {}));
/**
 * The data storage limit.
 */
class DataStorage {
    /**
     * Creates data storage settings with gigabytes as the unit.
     * @param options The configuration options containing min and/or max values.
     */
    static gb(options) {
        return new DataStorage(StorageUnit.GB, options.minimum, options.maximum);
    }
    constructor(unit, minimum, maximum) {
        this.validate(minimum, maximum);
        this.minimum = minimum;
        this.maximum = maximum;
        this.unit = unit;
    }
    /**
     * Render dataStorage property.
     *
     * @internal
     */
    _render() {
        return {
            unit: this.unit,
            maximum: this.maximum,
            minimum: this.minimum,
        };
    }
    validate(minimum, maximum) {
        const validMinimum = minimum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(minimum);
        const validMaximum = maximum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(maximum);
        if (validMinimum && (minimum < 1 || minimum > 5000)) {
            throw new Error(`\`dataStorage.minimum\` must be between 1 and 5000, got: ${minimum}.`);
        }
        if (validMaximum && (maximum < 1 || maximum > 5000)) {
            throw new Error(`\`dataStorage.maximum\` must be between 1 and 5000, got: ${maximum}.`);
        }
        if (validMinimum && validMaximum && maximum < minimum) {
            throw new Error(`\`dataStorage.maximum\` must be greater than or equal to \`dataStorage.minimum\`, got: maximum ${maximum}, minimum ${minimum}.`);
        }
    }
}
exports.DataStorage = DataStorage;
_a = JSII_RTTI_SYMBOL_1;
DataStorage[_a] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.DataStorage", version: "0.1.1" };
/**
 * The configuration for the number of ElastiCache Processing Units (ECPU) the cache can consume per second.
 */
class ECPUPerSecond {
    /**
     * Creates ECPU per second settings.
     *
     * @param options The configuration options containing min and/or max values.
     */
    static of(options) {
        return new ECPUPerSecond(options.minimum, options.maximum);
    }
    constructor(minimum, maximum) {
        this.validate(minimum, maximum);
        this.minimum = minimum;
        this.maximum = maximum;
    }
    /**
     * Render ecpuPerSecond property.
     *
     * @internal
     */
    _render() {
        return {
            maximum: this.maximum,
            minimum: this.minimum,
        };
    }
    validate(minimum, maximum) {
        const validMinimum = minimum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(minimum);
        const validMaximum = maximum !== undefined && !aws_cdk_lib_1.Token.isUnresolved(maximum);
        if (validMinimum && (minimum < 1000 || minimum > 15000000)) {
            throw new Error(`\`ecpuPerSecond.minimum\` must be between 1000 and 15000000, got: ${minimum}.`);
        }
        if (validMaximum && (maximum < 1000 || maximum > 15000000)) {
            throw new Error(`\`ecpuPerSecond.maximum\` must be between 1000 and 15000000, got: ${maximum}.`);
        }
        if (validMinimum && validMaximum && maximum < minimum) {
            throw new Error(`\`ecpuPerSecond.maximum\` must be greater than or equal to \`ecpuPerSecond.minimum\`, got: maximum ${maximum}, minimum ${minimum}.`);
        }
    }
}
exports.ECPUPerSecond = ECPUPerSecond;
_b = JSII_RTTI_SYMBOL_1;
ECPUPerSecond[_b] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.ECPUPerSecond", version: "0.1.1" };
/**
 * The version number of the engine the serverless cache is compatible with.
 */
var MajorVersion;
(function (MajorVersion) {
    /**
     * Version 7
     */
    MajorVersion["VER_7"] = "7";
    /**
     * Version 8
     */
    MajorVersion["VER_8"] = "8";
})(MajorVersion || (exports.MajorVersion = MajorVersion = {}));
/**
 * A new or imported serverless cache.
 */
class ServerlessCacheBase extends aws_cdk_lib_1.Resource {
    /**
     * Imports an existing ServerlessCache from attributes
     */
    static fromServerlessCacheAttributes(scope, id, attrs) {
        class Import extends ServerlessCacheBase {
            constructor() {
                super(...arguments);
                this.serverlessCacheName = attrs.serverlessCacheName;
                this.endpointAddress = attrs.endpointAddress;
                this.endpointPort = attrs.endpointPort;
                this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(attrs.endpointPort),
                });
                this.serverlessCacheArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'serverlesscache',
                    resourceName: attrs.serverlessCacheName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity the specified actions.
     * @param grantee the identity to be granted the actions
     * @param actions the data-access actions
     *
     * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html
     */
    grant(grantee, ...actions) {
        return aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'serverlesscache',
                    resourceName: this.serverlessCacheName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        });
    }
    /**
     * Permits an IAM principal to perform connect to the serverless cache.
     *
     * Actions: Connect
     *
     * @param grantee The principal to grant access to.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    grantConnect(grantee) {
        return this.grant(grantee, 'elasticache:Connect');
    }
    /**
     * Create a CloudWatch metric for serverless cache.
     *
     * @param metricName name of the metric.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events-redis.html
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events.memcached.html
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                clusterId: this.serverlessCacheName,
            },
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the total number of bytes used by the data stored in your cache.
     *
     * @default - average over 5 minutes
     */
    metricBytesUsedForCache(props) {
        return this.metric('BytesUsedForCache', { statistic: 'Average', ...props });
    }
    /**
     * Metric for the total number of ElastiCacheProcessingUnits (ECPUs) consumed by the requests executed on your cache.
     *
     * @default - average over 5 minutes
     */
    metricElastiCacheProcessingUnits(props) {
        return this.metric('ElastiCacheProcessingUnits', { statistic: 'Average', ...props });
    }
}
/**
 * Represents an ElastiCache Serverless Cache construct in AWS CDK.
 *
 * @example
 * declare const vpc: aws_ec2.IVpc;
 *
 * const serverlessCache = new ServerlessCache(
 *   stack,
 *   'ServerlessCache',
 *   {
 *     serverlessCacheName: 'my-serverlessCache',
 *     engine: Engine.VALKEY,
 *     vpc,
 *   },
 * );
 */
class ServerlessCache extends ServerlessCacheBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.serverlessCacheName ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, { separator: '-', maxLength: 40 }).toLowerCase(),
                }),
        });
        this.props = props;
        this.securityGroups = props.securityGroups ?? [
            this.createSecurityGroup(this, 'SecurityGroup', {
                vpc: this.props.vpc,
                description: 'Automatic generated security group for ElastiCache Serverless Security Group',
            }),
        ];
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.validateServerlessCacheName();
        this.validateDescription();
        this.validateAutomaticBackupSettings();
        this.validateFinalSnapshotName();
        this.validateUserGroup();
        const serverlessCache = this.createResource(this, 'Resource', {
            engine: this.props.engine,
            serverlessCacheName: this.physicalName,
            cacheUsageLimits: this.renderCacheUsageLimits(),
            dailySnapshotTime: props.dailySnapshotTime?.toTimestamp(),
            description: this.props.description,
            finalSnapshotName: this.props.finalSnapshotName,
            kmsKeyId: this.props.kmsKey?.keyArn,
            majorEngineVersion: this.props.majorEngineVersion,
            securityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            subnetIds: this.props.vpc.selectSubnets(this.vpcSubnets).subnetIds,
            snapshotArnsToRestore: this.props.snapshotArnsToRestore,
            snapshotRetentionLimit: this.props.snapshotRetentionLimit,
            userGroupId: this.props.userGroup?.userGroupId,
        });
        this.serverlessCacheArn = serverlessCache.attrArn;
        this.serverlessCacheName = serverlessCache.ref;
        this.endpointAddress = serverlessCache.attrEndpointAddress;
        this.endpointPort = Number(serverlessCache.attrEndpointPort);
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(this.endpointPort),
        });
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_elasticache.CfnServerlessCache(scope, id, props);
    }
    createSecurityGroup(scope, id, props) {
        return new aws_cdk_lib_1.aws_ec2.SecurityGroup(scope, id, props);
    }
    renderCacheUsageLimits() {
        const usageLimits = this.props.cacheUsageLimits;
        if (usageLimits === undefined) {
            return undefined;
        }
        const dataStorage = usageLimits.dataStorage;
        const ecpuPerSecond = usageLimits.ecpuPerSecond;
        if (dataStorage === undefined && ecpuPerSecond === undefined) {
            return undefined;
        }
        return {
            dataStorage: dataStorage ? dataStorage._render() : undefined,
            ecpuPerSecond: ecpuPerSecond ? ecpuPerSecond._render() : undefined,
        };
    }
    /**
     * Validates a serverless cache name.
     */
    validateServerlessCacheName() {
        const serverlessCacheName = this.props.serverlessCacheName;
        if (aws_cdk_lib_1.Token.isUnresolved(serverlessCacheName) || serverlessCacheName === undefined) {
            return;
        }
        if (/\s/.test(serverlessCacheName)) {
            throw new Error(`\`serverlessCacheName\` must not contain spaces, got: ${serverlessCacheName}.`);
        }
        if (serverlessCacheName.length < 1 || serverlessCacheName.length > 40) {
            throw new Error(`\`serverlessCacheName\` must be between 1 and 40 characters, got: ${serverlessCacheName.length} characters.`);
        }
    }
    /**
     * Validates a description.
     */
    validateDescription() {
        const description = this.props.description;
        if (aws_cdk_lib_1.Token.isUnresolved(description) || description === undefined) {
            return;
        }
        if (/<|>/.test(description)) {
            throw new Error(`\`description\` must not contain < and >, got: ${description}`);
        }
        if (description.length > 255) {
            throw new Error(`\`description\` must not exceed 255 characters, got: ${description.length} characters.`);
        }
    }
    /**
     * Validates an automatic backup settings.
     */
    validateAutomaticBackupSettings() {
        const dailySnapshotTime = this.props.dailySnapshotTime;
        const snapshotRetentionLimit = this.props.snapshotRetentionLimit;
        if (aws_cdk_lib_1.Token.isUnresolved(snapshotRetentionLimit)) {
            return;
        }
        if (dailySnapshotTime !== undefined && snapshotRetentionLimit === undefined) {
            throw new Error('`snapshotRetentionLimit` must be specified when `dailySnapshotTime` is set.');
        }
        if (snapshotRetentionLimit !== undefined && (snapshotRetentionLimit < 1 || snapshotRetentionLimit > 35)) {
            throw new Error(`\`snapshotRetentionLimit\` must be between 1 and 35, got: ${snapshotRetentionLimit}.`);
        }
    }
    /**
     * Validates final snapshot name.
     */
    validateFinalSnapshotName() {
        const finalSnapshotName = this.props.finalSnapshotName;
        if (aws_cdk_lib_1.Token.isUnresolved(finalSnapshotName) || finalSnapshotName === undefined)
            return;
        if (!/^[a-z][a-z0-9]*(-[a-z0-9]+)*$/.test(finalSnapshotName)) {
            throw new Error(`\`finalSnapshotName\` must consist only of lowercase alphanumeric characters or hyphens, with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens, got: ${finalSnapshotName}.`);
        }
        if (finalSnapshotName.length > 255) {
            throw new Error(`\`finalSnapshotName\` must not exceed 255 characters, got: ${finalSnapshotName.length} characters.`);
        }
    }
    /**
     * Validates an engine and a user group combination.
     */
    validateUserGroup() {
        if (this.props.userGroup === undefined)
            return;
        if (![util_1.Engine.REDIS, util_1.Engine.VALKEY].includes(this.props.engine)) {
            throw new Error(`\`userGroup\` is available for Valkey and Redis OSS only, got engine: ${this.props.engine}.`);
        }
    }
}
exports.ServerlessCache = ServerlessCache;
_c = JSII_RTTI_SYMBOL_1;
ServerlessCache[_c] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.ServerlessCache", version: "0.1.1" };
//# sourceMappingURL=data:application/json;base64,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