"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const util_1 = require("./util");
/**
 * Represents a user group construct in AWS CDK.
 *
 * @example
 * declare const user: User;
 *
 * const userGroup = new UserGroup(
 *   stack,
 *   'UserGroup',
 *   {
 *      users: [user],
 *   },
 * );
 */
class UserGroup extends aws_cdk_lib_1.Resource {
    /**
     * Imports an existing user group from attributes
     */
    static fromUserGroupId(scope, id, userGroupId) {
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.userGroupId = userGroupId;
                this.userGroupArn = aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'elasticache',
                    resource: 'usergroup',
                    resourceName: userGroupId,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.userGroupId ??
                aws_cdk_lib_1.Lazy.string({
                    produce: () => aws_cdk_lib_1.Names.uniqueResourceName(this, {
                        maxLength: 40,
                        separator: '-',
                    }).toLowerCase(),
                }),
        });
        this.props = props;
        this.users = this.props.users;
        this.validateUserGroupId();
        this.node.addValidation({ validate: () => this.validateDefaultUser() });
        this.node.addValidation({ validate: () => this.validateDuplicateUsernames() });
        const userGroup = this.createResource(this, 'Resource', {
            engine: util_1.Engine.REDIS,
            userGroupId: this.physicalName,
            userIds: aws_cdk_lib_1.Lazy.list({ produce: () => this.users.map(user => user.userId) }),
        });
        this.userGroupArn = userGroup.attrArn;
        this.userGroupId = userGroup.ref;
    }
    createResource(scope, id, props) {
        return new aws_cdk_lib_1.aws_elasticache.CfnUserGroup(scope, id, props);
    }
    /**
     * Validates user group id.
     */
    validateUserGroupId() {
        const userGroupId = this.props.userGroupId;
        if (aws_cdk_lib_1.Token.isUnresolved(userGroupId) || userGroupId === undefined) {
            return;
        }
        if (userGroupId.length < 1 || userGroupId.length > 40) {
            throw new Error(`\`userGroupId\` must be between 1 and 40 characters, got ${userGroupId.length} characters.`);
        }
        if (!/^[A-Za-z][A-Za-z0-9]*(-[A-Za-z0-9]+)*$/.test(userGroupId)) {
            throw new Error(`\`userGroupId\` must consist only of alphanumeric characters or hyphens, with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens, got: ${userGroupId}.`);
        }
    }
    /**
     * Validates default user.
     */
    validateDefaultUser() {
        const userNameList = this.users.map(user => user.userName);
        if (userNameList.some(userName => aws_cdk_lib_1.Token.isUnresolved(userName))) {
            return [];
        }
        const errors = [];
        if (!userNameList.includes('default')) {
            errors.push('A user with the username `default` must be included in `users`.');
        }
        return errors;
    }
    /**
     * Validates that there are no duplicate usernames in the user group
     */
    validateDuplicateUsernames() {
        const userNameList = this.users.map(user => user.userName);
        if (userNameList.some(username => aws_cdk_lib_1.Token.isUnresolved(username))) {
            return [];
        }
        const seenUsernames = new Set();
        const errors = [];
        for (const username of userNameList) {
            if (seenUsernames.has(username)) {
                errors.push(`Duplicate username found in user group: \`${username}\` is duplicated. Each username must be unique within a user group.`);
            }
            seenUsernames.add(username);
        }
        return errors;
    }
    /**
     * Adds a user to the user group
     *
     * @param user the user to add
     */
    addUser(user) {
        this.users.push(user);
    }
}
exports.UserGroup = UserGroup;
_a = JSII_RTTI_SYMBOL_1;
UserGroup[_a] = { fqn: "@open-constructs/aws-cdk.aws_elasticache.UserGroup", version: "0.1.1" };
//# sourceMappingURL=data:application/json;base64,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