import { Duration, SecretValue, aws_ec2, aws_fsx } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { DailyAutomaticBackupStartTime } from './daily-automatic-backup-start-time';
import { MaintenanceTime } from './maintenance-time';
/**
 * The different kinds of file system deployments used by NetApp ONTAP.
 */
export declare enum OntapDeploymentType {
    /**
     * A high availability file system configured for Multi-AZ redundancy to tolerate temporary Availability Zone (AZ) unavailability.
     * This is a first-generation FSx for ONTAP file system.
     */
    MULTI_AZ_1 = "MULTI_AZ_1",
    /**
     * A high availability file system configured for Multi-AZ redundancy to tolerate temporary AZ unavailability.
     * This is a second-generation FSx for ONTAP file system.
     */
    MULTI_AZ_2 = "MULTI_AZ_2",
    /**
     * A file system configured for Single-AZ redundancy.
     * This is a first-generation FSx for ONTAP file system.
     */
    SINGLE_AZ_1 = "SINGLE_AZ_1",
    /**
     * A file system configured with multiple high-availability (HA) pairs for Single-AZ redundancy.
     * This is a second-generation FSx for ONTAP file system.
     */
    SINGLE_AZ_2 = "SINGLE_AZ_2"
}
/**
 * The throughput capacity per HA pair for an Amazon FSx for NetApp ONTAP file system.
 */
export declare abstract class ThroughputCapacityPerHaPair {
    readonly capacity: number;
    /**
     * The deployment type of the throughput capacity.
     */
    abstract readonly deploymentType: OntapDeploymentType;
    protected constructor(capacity: number);
}
/**
 * The throughput capacity for the Single-AZ 1 deployment type.
 */
export declare class SingleAz1ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    /**
     * The throughput capacity of 128 MBps per HA pair.
     */
    static readonly MB_PER_SEC_128: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 256 MBps per HA pair.
     */
    static readonly MB_PER_SEC_256: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 512 MBps per HA pair.
     */
    static readonly MB_PER_SEC_512: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 1024 MBps per HA pair.
     */
    static readonly MB_PER_SEC_1024: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 2048 MBps per HA pair.
     */
    static readonly MB_PER_SEC_2048: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 4096 MBps per HA pair.
     */
    static readonly MB_PER_SEC_4096: SingleAz1ThroughputCapacityPerHaPair;
    /**
     * The deployment type of the throughput capacity.
     */
    readonly deploymentType = OntapDeploymentType.SINGLE_AZ_1;
}
/**
 * The throughput capacity for the Multi-AZ 1 deployment type.
 */
export declare class MultiAz1ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    /**
     * The throughput capacity of 128 MBps per HA pair.
     */
    static readonly MB_PER_SEC_128: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 256 MBps per HA pair.
     */
    static readonly MB_PER_SEC_256: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 512 MBps per HA pair.
     */
    static readonly MB_PER_SEC_512: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 1024 MBps per HA pair.
     */
    static readonly MB_PER_SEC_1024: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 2048 MBps per HA pair.
     */
    static readonly MB_PER_SEC_2048: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 4096 MBps per HA pair.
     */
    static readonly MB_PER_SEC_4096: MultiAz1ThroughputCapacityPerHaPair;
    /**
     * The deployment type of the throughput capacity.
     */
    readonly deploymentType = OntapDeploymentType.MULTI_AZ_1;
}
/**
 * The throughput capacity for the Single-AZ 2 deployment type.
 */
export declare class SingleAz2ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    /**
     * The throughput capacity of 1536 MBps per HA pair.
     */
    static readonly MB_PER_SEC_1536: SingleAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 3072 MBps per HA pair.
     */
    static readonly MB_PER_SEC_3072: SingleAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 6144 MBps per HA pair.
     */
    static readonly MB_PER_SEC_6144: SingleAz2ThroughputCapacityPerHaPair;
    /**
     * The deployment type of the throughput capacity.
     */
    readonly deploymentType = OntapDeploymentType.SINGLE_AZ_2;
}
/**
 * The throughput capacity for the Multi-AZ 2 deployment type.
 */
export declare class MultiAz2ThroughputCapacityPerHaPair extends ThroughputCapacityPerHaPair {
    /**
     * The throughput capacity of 384 MBps per HA pair.
     */
    static readonly MB_PER_SEC_384: MultiAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 768 MBps per HA pair.
     */
    static readonly MB_PER_SEC_768: MultiAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 1536 MBps per HA pair.
     */
    static readonly MB_PER_SEC_1536: MultiAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 3072 MBps per HA pair.
     */
    static readonly MB_PER_SEC_3072: MultiAz2ThroughputCapacityPerHaPair;
    /**
     * The throughput capacity of 6144 MBps per HA pair.
     */
    static readonly MB_PER_SEC_6144: MultiAz2ThroughputCapacityPerHaPair;
    /**
     * The deployment type of the throughput capacity.
     */
    readonly deploymentType = OntapDeploymentType.MULTI_AZ_2;
}
/**
 * The configuration for the Amazon FSx for NetApp ONTAP file system.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-fsx-filesystem-ontapconfiguration.html
 */
export interface OntapConfiguration {
    /**
     * The number of days to retain automatic backups.
     *
     * Setting this property to 0 disables automatic backups.
     * You can retain automatic backups for a maximum of 90 days.
     *
     * @default - 30 days
     */
    readonly automaticBackupRetention?: Duration;
    /**
     * Start time for 30-minute daily automatic backup window in Coordinated Universal Time (UTC).
     *
     * @default - no backup window
     */
    readonly dailyAutomaticBackupStartTime?: DailyAutomaticBackupStartTime;
    /**
     * The FSx for ONTAP file system deployment type to use in creating the file system.
     *
     * @default OntapDeploymentType.MULTI_AZ_2
     */
    readonly deploymentType?: OntapDeploymentType;
    /**
     * The total number of SSD IOPS provisioned for the file system.
     *
     * The minimum and maximum values for this property depend on the value of HAPairs and StorageCapacity.
     * The minimum value is calculated as StorageCapacity * 3 * HAPairs (3 IOPS per GB of StorageCapacity).
     * The maximum value is calculated as 200,000 * HAPairs.
     *
     * @default - 3 IOPS * GB of storage capacity * HAPairs
     */
    readonly diskIops?: number;
    /**
     * The IP address range in which the endpoints to access your file system will be created.
     *
     * You can have overlapping endpoint IP addresses for file systems deployed in the same VPC/route tables, as long as they don't overlap with any subnet.
     *
     * @default - an unused IP address range from the 198.19.* range
     */
    readonly endpointIpAddressRange?: string;
    /**
     * The ONTAP administrative password for the `fsxadmin` user with which you administer your file system using the NetApp ONTAP CLI and REST API.
     *
     * If you don't specify a password, Amazon FSx will not set one. In that case, the user will not be able to log in.
     *
     * You can change the admin password at any time through the management console.
     *
     * @default - do not set an admin password
     */
    readonly fsxAdminPassword?: SecretValue;
    /**
     * How many high-availability (HA) pairs of file servers will power your file system.
     *
     * First-generation file systems are powered by 1 HA pair.
     * Second-generation multi-AZ file systems are powered by 1 HA pair.
     * Second generation single-AZ file systems are powered by up to 12 HA pairs.
     *
     * The value of this property affects the values of `storageCapacity`, `iops`, and `throughputCapacity`.
     *
     * Block storage protocol support (iSCSI and NVMe over TCP) is disabled on file systems with more than 6 HA pairs.
     *
     * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/administering-file-systems.html#HA-pairs
     * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/supported-fsx-clients.html#using-block-storage
     *
     * @default 1
     */
    readonly haPairs?: number;
    /**
     * The subnet in which you want the preferred file server to be located.
     *
     * This value is required when `deploymentType` is set to `MULTI_AZ_1` or `MULTI_AZ_2`.
     *
     * @default - no default value (This value is not used for single-AZ file systems, but it is required for multi-AZ file systems)
     */
    readonly preferredSubnet?: aws_ec2.ISubnet;
    /**
     * The route tables in which Amazon FSx creates the rules for routing traffic to the correct file server.
     *
     * You should specify all virtual private cloud (VPC) route tables associated with the subnets in which your clients are located.
     *
     * Amazon FSx manages VPC route tables for Multi-AZ file systems using tag-based authentication.
     * These route tables are tagged with Key: AmazonFSx; Value: ManagedByAmazonFSx.
     *
     * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/unable-to-access.html#vpc-route-tables-not-tagged
     *
     * @default - Amazon FSx selects your VPC's default route table.
     */
    readonly routeTables?: aws_ec2.IRouteTable[];
    /**
     * The throughput capacity per HA pair for the file system.
     *
     * @see https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/managing-throughput-capacity.html
     *
     * @default - Amazon FSx determines the throughput capacity based on the storage capacity
     */
    readonly throughputCapacityPerHaPair?: ThroughputCapacityPerHaPair;
    /**
     * The preferred day and time to perform weekly maintenance.
     *
     * @default - automatically set by Amazon FSx
     */
    readonly weeklyMaintenanceStartTime?: MaintenanceTime;
}
/**
 * Properties specific to the NetApp ONTAP version of the FSx file system.
 */
export interface OntapFileSystemProps extends aws_fsx.FileSystemProps {
    /**
     * Additional configuration for FSx specific to NetApp ONTAP.
     */
    readonly ontapConfiguration: OntapConfiguration;
    /**
     * The subnet that the file system will be accessible from.
     *
     * For MULTI_AZ_1 deployment types,
     * provide exactly two subnets, one for the preferred file server and one for the standby file server.
     *
     * Specify one of these subnets as the preferred subnet using `OntapConfiguration.preferredSubnet` property for multi-AZ file system.
     */
    readonly vpcSubnets: aws_ec2.ISubnet[];
}
/**
 * The FSx for NetApp ONTAP File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
export declare class OntapFileSystem extends aws_fsx.FileSystemBase {
    /**
     * Import an existing FSx for NetApp ONTAP file system from the given properties.
     */
    static fromOntapFileSystemAttributes(scope: Construct, id: string, attrs: aws_fsx.FileSystemAttributes): aws_fsx.IFileSystem;
    /**
     * Configures a Connections object.
     */
    protected static configureConnections(securityGroup: aws_ec2.ISecurityGroup): aws_ec2.Connections;
    /**
     * The default FSx file system type used by FSx for NetApp ONTAP.
     */
    private static readonly DEFAULT_FILE_SYSTEM_TYPE;
    /**
     * The security groups/rules used to allow network connections to the file system.
     */
    readonly connections: aws_ec2.Connections;
    /**
     * The management endpoint DNS name assigned to this file system.
     */
    readonly dnsName: string;
    /**
     * The inter cluster endpoint DNS name assigned to this file system.
     */
    readonly interClusterDnsName: string;
    /**
     * The ID that AWS assigns to the file system.
     */
    readonly fileSystemId: string;
    /**
     * The encapsulated L1 file system.
     */
    private readonly fileSystem;
    /**
     * The deployment type of the file system.
     */
    private readonly deploymentType;
    constructor(scope: Construct, id: string, props: OntapFileSystemProps);
    protected createSecurityGroup(vpc: aws_ec2.IVpc): aws_ec2.SecurityGroup;
    protected createOntapFileSystem(fileSystemSecurityGroup: aws_ec2.ISecurityGroup, props: OntapFileSystemProps): aws_fsx.CfnFileSystem;
    /**
     * Validates the props provided for a new FSx for Ontap file system.
     */
    private validateProps;
    private validateAutomaticBackupRetention;
    private validateDailyAutomaticBackupStartTime;
    private validateDiskIops;
    private validateEndpointIpAddressRange;
    private validateHaPairs;
    private validateSubnets;
    private validateRouteTables;
    private validateThroughputCapacity;
    private validateStorageCapacity;
}
