import { IResource, Resource, aws_redshiftserverless, aws_ec2 } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { INamespace } from './namespace';
/**
 * A Redshift Serverless Workgroup
 */
export interface IWorkgroup extends IResource, aws_ec2.IConnectable {
    /**
     * The workgroup Arn
     *
     * @attribute
     */
    readonly workgroupArn: string;
    /**
     * The workgroup name
     *
     * @attribute
     */
    readonly workgroupName: string;
    /**
     * The workgroup id
     *
     * @attribute
     */
    readonly workgroupId: string;
    /**
     * The workgroup endpoint address
     *
     * @attribute
     */
    readonly endpointAddress: string;
    /**
     * The workgroup port
     *
     * @attribute
     */
    readonly port: number;
}
/**
 * Properties for defining a Redshift Serverless Workgroup.
 */
export interface WorkgroupProps {
    /**
     * The base compute capacity of the workgroup in Redshift Processing Units (RPUs).
     *
     * You can adjust the base capacity setting from 8 RPUs to 512 RPUs in units of 8.
     * Also you can increment or decrement RPUs in units of 32 when setting a base capacity between 512-1024.
     *
     * @default 128
     *
     * @see https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-capacity.html
     */
    readonly baseCapacity?: number;
    /**
     * A list of parameters to set for finer control over a database.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-redshiftserverless-workgroup.html#cfn-redshiftserverless-workgroup-configparameters
     * @default - no config parameters
     */
    readonly configParameters?: {
        [key: string]: string;
    };
    /**
     * The value that specifies whether to enable enhanced virtual private cloud (VPC) routing,
     * which forces Amazon Redshift Serverless to route traffic through your VPC.
     *
     * @default false
     */
    readonly enhancedVpcRouting?: boolean;
    /**
     * The namespace the workgroup is associated with.
     *
     * @default - the workgroup is not associated with any namespace
     */
    readonly namespace?: INamespace;
    /**
     * The custom port to use when connecting to a workgroup. Valid port ranges are 5431-5455 and 8191-8215.
     *
     * @default 5439
     */
    readonly port?: number;
    /**
     * A value that specifies whether the workgroup can be accessible from a public network.
     *
     * @default false
     */
    readonly publiclyAccessible?: boolean;
    /**
     * The security groups to associate with the workgroup.
     *
     * @default - a new security group is created
     */
    readonly securityGroups?: aws_ec2.ISecurityGroup[];
    /**
     * The VPC to place the workgroup in.
     *
     * `vpc` must have at least 3 subnets, and they must span across 3 Availability Zones.
     */
    readonly vpc: aws_ec2.IVpc;
    /**
     * Where to place the workgroup within the VPC.
     *
     * @default - private subnets
     */
    readonly vpcSubnets?: aws_ec2.SubnetSelection;
    /**
     * The workgroup name.
     *
     * \`workgroupName\` must be between 3 and 64 characters long, contain only lowercase letters, numbers, and hyphens.
     *
     * @default - auto generate
     */
    readonly workgroupName?: string;
}
/**
 * Attributes for importing a Redshift Serverless Workgroup.
 */
export interface WorkgroupAttributes {
    /**
     * The workgroup name
     */
    readonly workgroupName: string;
    /**
     * The workgroup id
     */
    readonly workgroupId: string;
    /**
     * The workgroup endpoint address
     */
    readonly endpointAddress: string;
    /**
     * The workgroup port
     */
    readonly port: number;
    /**
     * The security groups associated with the Redshift Serverless Workgroup.
     */
    readonly securityGroups: aws_ec2.ISecurityGroup[];
}
/**
 * Represents a Redshift Serverless Workgroup construct in AWS CDK.
 *
 * @example
 * declare const namespace: Namespace;
 * declare const vpc: aws_ec2.IVpc;
 *
 * const nameSpace = new Workgroup(
 *   stack,
 *   'Workgroup',
 *   {
 *     workgroupName: 'my-workgroup',
 *     namespace: namespace,
 *     vpc,
 *   },
 * );
 */
export declare class Workgroup extends Resource implements IWorkgroup {
    /**
     * Import an existing workgroup to the stack from its attributes.
     */
    static fromWorkgroupAttributes(scope: Construct, id: string, attrs: WorkgroupAttributes): IWorkgroup;
    /**
     * The workgroup Arn
     */
    readonly workgroupArn: string;
    /**
     * The workgroup name
     */
    readonly workgroupName: string;
    /**
     * The workgroup id
     */
    readonly workgroupId: string;
    /**
     * The workgroup endpoint address
     */
    readonly endpointAddress: string;
    /**
     * The workgroup port
     */
    readonly port: number;
    /**
     * The connection object associated with the Redshift Serverless Workgroup.
     */
    readonly connections: aws_ec2.Connections;
    private readonly props;
    private readonly securityGroups;
    private readonly vpcSubnets;
    constructor(scope: Construct, id: string, props: WorkgroupProps);
    protected createResource(scope: Construct, id: string, props: aws_redshiftserverless.CfnWorkgroupProps): aws_redshiftserverless.CfnWorkgroup;
    protected createSecurityGroup(scope: Construct, id: string, props: aws_ec2.SecurityGroupProps): aws_ec2.SecurityGroup;
    /**
     * Validates capacity settings.
     */
    private validateCapacity;
    /**
     * Validates a workgroup name.
     */
    private validateWorkgroupName;
    /**
     * Validates a port number.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-connecting.html
     */
    private validatePort;
    /**
     * Validates subnets.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-usage-considerations.html
     */
    private validateSubnet;
}
