"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const kendra = require("aws-cdk-lib/aws-kendra");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
test('Launch with minimal code and check  structure', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
        Environment: {
            Variables: {
                KENDRA_INDEX_ID: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        },
    });
    template.hasResourceProperties("AWS::Kendra::Index", {
        RoleArn: {
            "Fn::GetAtt": [
                "samplekendraindexrolesample4F9E7B66",
                "Arn",
            ],
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Allow Kendra index to write CloudWatch Logs",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "cloudwatch:PutMetricData",
                            Condition: {
                                StringEquals: {
                                    "cloudwatch:namespace": "AWS/Kendra"
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "logs:CreateLogGroup",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "logs:DescribeLogGroups",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                                "logs:DescribeLogStream"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*:log-stream:*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "AllowLogging"
            }
        ],
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Policy for Kendra S3 Data Source",
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344/*`
                        },
                        {
                            Action: "s3:ListBucket",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344`
                        },
                        {
                            Action: [
                                "kendra:BatchPutDocument",
                                "kendra:BatchDeleteDocument"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "samplekendraindexsample8A81A6C2",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "s3CrawlPolicy"
            }
        ]
    });
    // Confirm that Lambda function has QUERY access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ],
        },
        Roles: [
            {
                Ref: "sampletestfunctionname24334ServiceRole99395A01"
            }
        ]
    });
});
test('Check pattern properties on minimal launch', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const newConstruct = new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
    });
    expect(newConstruct.lambdaFunction).toBeDefined();
    expect(newConstruct.kendraDataSources).toBeDefined();
    expect(newConstruct.kendraDataSources.length).toEqual(1);
    expect(newConstruct.kendraIndex).toBeDefined();
});
test('Launch with VPC', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name12539';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        deployVpc: true,
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Check the VPC
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "10.0.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    // Is the Lambda function associated with the VPC
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "sampleReplaceDefaultSecurityGroupsecuritygroupE5725669",
                        "GroupId"
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    // Check that the Lambda function Policy has proper network access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                        "ec2:AssignPrivateIpAddresses",
                        "ec2:UnassignPrivateIpAddresses"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {},
                {},
                {}
            ],
        },
        Roles: [
            {
                Ref: "sampleLambdaFunctionServiceRole7A3C4AF5"
            }
        ]
    });
    // Check for the Kendra endpoint in the VPC
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kendra"
                ]
            ]
        },
        VpcId: {
            Ref: "Vpc8378EB38"
        },
        PrivateDnsEnabled: true,
        SecurityGroupIds: [
            {
                "Fn::GetAtt": [
                    "DefaultKENDRAsecuritygroup34536A79",
                    "GroupId"
                ]
            }
        ],
        SubnetIds: [
            {
                Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
            },
            {
                Ref: "VpcisolatedSubnet2Subnet39217055"
            }
        ],
        VpcEndpointType: "Interface"
    });
});
test('Launch with existing lambda', () => {
    const stack = new cdk.Stack();
    const testTimeout = 17;
    const testFunctionName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        timeout: cdk.Duration.seconds(testTimeout),
        functionName: testFunctionName
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingLambdaObj: existingFunction,
        kendraDataSourcesProps: [],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: testTimeout,
        FunctionName: testFunctionName,
    });
});
test('Confirm error with data source with no bucket name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', {
            lambdaFunctionProps: lambdaProps,
            kendraDataSourcesProps: [{
                    type: 'S3',
                    dataSourceConfiguration: {
                        s3Configuration: {}
                    }
                }
            ],
        });
    };
    expect(app).toThrowError(/Error - an S3 Kendra DataSource requires the DataSourceConfiguration.S3Configuration.bucketName prop/);
});
test('Launch with existing vpc', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingVpc: defaults.getTestVpc(stack),
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Launch with Read/Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["ReaD", "SubmitFeedBack", "wrITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Read permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["READ"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with SubmitFeedback permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["SUBMITFEEDBACK"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with existing kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const existingRole = new iam.Role(stack, 'existing-role', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
    });
    const existingIndex = new kendra.CfnIndex(stack, 'existing-index', {
        edition: 'ENTERPRISE',
        name: 'existingIndex',
        roleArn: existingRole.roleArn
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        existingKendraIndexObj: existingIndex,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Make sure we didn't create an index anyway
    template.resourceCountIs("AWS::Kendra::Index", 1);
    template.hasResourceProperties("AWS::Kendra::Index", {
        Edition: 'ENTERPRISE'
    });
});
test('Launch with S3 data source with overridden defaults', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name223423';
    const testInclusionPattern = 'this-folder';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        inclusionPatterns: [testInclusionPattern],
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 1);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                InclusionPatterns: [testInclusionPattern],
                BucketName: testBucketName
            },
        }
    });
});
test('Launch with S3 data source and unsupported data source', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const nonImplementedSourceType = "WEBCRAWLER";
    const nonImplementedSourceName = "test-other-source";
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    // Create a role
    const fakeRole = new iam.Role(stack, 'fakeRole', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
        roleName: 'externalFakeRole'
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                name: nonImplementedSourceName,
                roleArn: fakeRole.roleArn,
                type: nonImplementedSourceType,
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Name: nonImplementedSourceName,
        Type: nonImplementedSourceType,
    });
});
test('Launch with multiple S3 data sources', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22342';
    const secondBucketName = 'second-bucket-name22342342';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'two-sources', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: secondBucketName,
                    }
                }
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources0B8E24996", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: secondBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources164176C5E", "Arn"]
        },
    });
});
test('Test with custom environment variable name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
        indexIdEnvironmentVariableName: "MY_VAR_NAME",
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                MY_VAR_NAME: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        }
    });
});
test('Confirm CheckVpcProps is being called', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const props = {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
        deployVpc: true,
        existingVpc: vpc
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'testBucketName',
                    }
                }
            }
        ],
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLWtlbmRyYS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGFtYmRhLWtlbmRyYS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxnQ0FBNkQ7QUFDN0QsaURBQWlEO0FBQ2pELGlEQUFpRDtBQUNqRCwyQ0FBMkM7QUFDM0MsbUNBQW1DO0FBQ25DLHVEQUFrRDtBQUNsRCwyREFBMkQ7QUFFM0QsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtJQUN6RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLGdCQUFnQixHQUFHLHlCQUF5QixDQUFDO0lBQ25ELE1BQU0sY0FBYyxHQUFHLHVCQUF1QixDQUFDO0lBRS9DLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxZQUFZLEVBQUUsZ0JBQWdCO1FBQzlCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsY0FBYztxQkFDM0I7aUJBQ0Y7YUFDRjtTQUNBO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFlBQVksRUFBRSxnQkFBZ0I7UUFDOUIsV0FBVyxFQUFFO1lBQ1gsU0FBUyxFQUFFO2dCQUNULGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUUsQ0FBQyxpQ0FBaUMsRUFBRSxJQUFJLENBQUM7aUJBQ3hEO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxvQkFBb0IsRUFBRTtRQUNuRCxPQUFPLEVBQUU7WUFDUCxZQUFZLEVBQUU7Z0JBQ1oscUNBQXFDO2dCQUNyQyxLQUFLO2FBQ047U0FDRjtLQUNGLENBQUMsQ0FBQztJQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxJQUFJLEVBQUUsSUFBSTtRQUNWLHVCQUF1QixFQUFFO1lBQ3ZCLGVBQWUsRUFBRTtnQkFDZixVQUFVLEVBQUUsY0FBYzthQUMzQjtTQUNGO1FBQ0QsT0FBTyxFQUFFO1lBQ1AsWUFBWSxFQUFFLENBQUMscUNBQXFDLEVBQUUsS0FBSyxDQUFDO1NBQzdEO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsa0NBQWtDO0lBQ2xDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxnQkFBZ0IsRUFBRTtRQUMvQyxXQUFXLEVBQUUsNkNBQTZDO1FBQzFELFFBQVEsRUFBRTtZQUNSO2dCQUNFLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFLDBCQUEwQjs0QkFDbEMsU0FBUyxFQUFFO2dDQUNULFlBQVksRUFBRTtvQ0FDWixzQkFBc0IsRUFBRSxZQUFZO2lDQUNyQzs2QkFDRjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsR0FBRzt5QkFDZDt3QkFDRDs0QkFDRSxNQUFNLEVBQUUscUJBQXFCOzRCQUM3QixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsZUFBZTt3Q0FDZjs0Q0FDRSxHQUFHLEVBQUUsYUFBYTt5Q0FDbkI7d0NBQ0QsR0FBRzt3Q0FDSDs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCwwQkFBMEI7cUNBQzNCO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLE1BQU0sRUFBRSx3QkFBd0I7NEJBQ2hDLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELFFBQVE7d0NBQ1I7NENBQ0UsR0FBRyxFQUFFLGFBQWE7eUNBQ25CO3dDQUNELEdBQUc7d0NBQ0g7NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsMEJBQTBCO3FDQUMzQjtpQ0FDRjs2QkFDRjt5QkFDRjt3QkFDRDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sc0JBQXNCO2dDQUN0QixtQkFBbUI7Z0NBQ25CLHdCQUF3Qjs2QkFDekI7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLE1BQU07d0NBQ047NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsUUFBUTt3Q0FDUjs0Q0FDRSxHQUFHLEVBQUUsYUFBYTt5Q0FDbkI7d0NBQ0QsR0FBRzt3Q0FDSDs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCx1Q0FBdUM7cUNBQ3hDO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxVQUFVLEVBQUUsY0FBYzthQUMzQjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsa0NBQWtDO0lBQ2xDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxnQkFBZ0IsRUFBRTtRQUMvQyxXQUFXLEVBQUUsa0NBQWtDO1FBQy9DLHdCQUF3QixFQUFFO1lBQ3hCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsT0FBTyxFQUFFLHNCQUFzQjtxQkFDaEM7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sRUFBRSxZQUFZO1NBQ3RCO1FBQ0QsUUFBUSxFQUFFO1lBQ1I7Z0JBQ0UsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUsY0FBYzs0QkFDdEIsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFLHNDQUFzQzt5QkFDakQ7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFLGVBQWU7NEJBQ3ZCLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxvQ0FBb0M7eUJBQy9DO3dCQUNEOzRCQUNFLE1BQU0sRUFBRTtnQ0FDTix5QkFBeUI7Z0NBQ3pCLDRCQUE0Qjs2QkFDN0I7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFlBQVksRUFBRTtvQ0FDWixpQ0FBaUM7b0NBQ2pDLEtBQUs7aUNBQ047NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsT0FBTyxFQUFFLFlBQVk7aUJBQ3RCO2dCQUNELFVBQVUsRUFBRSxlQUFlO2FBQzVCO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFDSCxnREFBZ0Q7SUFDaEQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1FBQ2pELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sdUJBQXVCO3dCQUN2QiwwQkFBMEI7cUJBQzNCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRSxHQUFHO2lCQUNkO2dCQUNEO29CQUNFLE1BQU0sRUFBRTt3QkFDTixjQUFjO3dCQUNkLGtCQUFrQjt3QkFDbEIsYUFBYTt3QkFDYixrQkFBa0I7d0JBQ2xCLGNBQWM7d0JBQ2QsaUJBQWlCO3FCQUNsQjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUU7d0JBQ1IsWUFBWSxFQUFFOzRCQUNaLGlDQUFpQzs0QkFDakMsS0FBSzt5QkFDTjtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUU7d0JBQ1IsWUFBWSxFQUFFOzRCQUNaLGlDQUFpQzs0QkFDakMsS0FBSzt5QkFDTjtxQkFDRjtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxLQUFLLEVBQUU7WUFDTDtnQkFDRSxHQUFHLEVBQUUsZ0RBQWdEO2FBQ3REO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw0Q0FBNEMsRUFBRSxHQUFHLEVBQUU7SUFDdEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxXQUFXLEdBQXlCO1FBQ3hDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLE1BQU0sWUFBWSxHQUFHLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ3ZELG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsa0JBQWtCO3FCQUMvQjtpQkFDRjthQUNGO1NBQ0E7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ2xELE1BQU0sQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNyRCxNQUFNLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN6RCxNQUFNLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0FBRWpELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtJQUMzQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLGNBQWMsR0FBRyx1QkFBdUIsQ0FBQztJQUUvQyxNQUFNLFdBQVcsR0FBeUI7UUFDeEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7UUFDbEMsU0FBUyxFQUFFLElBQUk7UUFDZixtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLHNCQUFzQixFQUFFLENBQUM7Z0JBQ3ZCLElBQUksRUFBRSxJQUFJO2dCQUNWLHVCQUF1QixFQUFFO29CQUN2QixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLGNBQWM7cUJBQzNCO2lCQUNGO2FBQ0Y7U0FDQTtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLGdCQUFnQjtJQUNoQixRQUFRLENBQUMscUJBQXFCLENBQUMsZUFBZSxFQUFFO1FBQzlDLFNBQVMsRUFBRSxhQUFhO1FBQ3hCLGtCQUFrQixFQUFFLElBQUk7UUFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtRQUN0QixlQUFlLEVBQUUsU0FBUztLQUMzQixDQUFDLENBQUM7SUFFSCxpREFBaUQ7SUFDakQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFNBQVMsRUFBRTtZQUNULGdCQUFnQixFQUFFO2dCQUNoQjtvQkFDRSxZQUFZLEVBQUU7d0JBQ1osd0RBQXdEO3dCQUN4RCxTQUFTO3FCQUNWO2lCQUNGO2FBQ0Y7WUFDRCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsR0FBRyxFQUFFLGtDQUFrQztpQkFDeEM7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLGtDQUFrQztpQkFDeEM7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsa0VBQWtFO0lBQ2xFLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtRQUNqRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLDRCQUE0Qjt3QkFDNUIsK0JBQStCO3dCQUMvQiw0QkFBNEI7d0JBQzVCLDhCQUE4Qjt3QkFDOUIsZ0NBQWdDO3FCQUNqQztvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsR0FBRztpQkFDZDtnQkFDRCxFQUFFO2dCQUNGLEVBQUU7Z0JBQ0YsRUFBRTthQUNIO1NBQ0Y7UUFDRCxLQUFLLEVBQUU7WUFDTDtnQkFDRSxHQUFHLEVBQUUseUNBQXlDO2FBQy9DO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCwyQ0FBMkM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFdBQVcsRUFBRTtZQUNYLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLGdCQUFnQjtvQkFDaEI7d0JBQ0UsR0FBRyxFQUFFLGFBQWE7cUJBQ25CO29CQUNELFNBQVM7aUJBQ1Y7YUFDRjtTQUNGO1FBQ0QsS0FBSyxFQUFFO1lBQ0wsR0FBRyxFQUFFLGFBQWE7U0FDbkI7UUFDRCxpQkFBaUIsRUFBRSxJQUFJO1FBQ3ZCLGdCQUFnQixFQUFFO1lBQ2hCO2dCQUNFLFlBQVksRUFBRTtvQkFDWixvQ0FBb0M7b0JBQ3BDLFNBQVM7aUJBQ1Y7YUFDRjtTQUNGO1FBQ0QsU0FBUyxFQUFFO1lBQ1Q7Z0JBQ0UsR0FBRyxFQUFFLGtDQUFrQzthQUN4QztZQUNEO2dCQUNFLEdBQUcsRUFBRSxrQ0FBa0M7YUFDeEM7U0FDRjtRQUNELGVBQWUsRUFBRSxXQUFXO0tBQzdCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUN2QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFdBQVcsR0FBRyxFQUFFLENBQUM7SUFFdkIsTUFBTSxnQkFBZ0IsR0FBRyxXQUFXLENBQUM7SUFFckMsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQ3ZFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7UUFDeEIsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUMxQyxZQUFZLEVBQUUsZ0JBQWdCO0tBQy9CLENBQUMsQ0FBQztJQUVILElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLGlCQUFpQixFQUFFLGdCQUFnQjtRQUNuQyxzQkFBc0IsRUFBRSxFQUFFO0tBQzNCLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDckQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELE9BQU8sRUFBRSxXQUFXO1FBQ3BCLFlBQVksRUFBRSxnQkFBZ0I7S0FDL0IsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0RBQW9ELEVBQUUsR0FBRyxFQUFFO0lBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNsQyxtQkFBbUIsRUFBRSxXQUFXO1lBQ2hDLHNCQUFzQixFQUFFLENBQUM7b0JBQ3ZCLElBQUksRUFBRSxJQUFJO29CQUNWLHVCQUF1QixFQUFFO3dCQUN2QixlQUFlLEVBQUUsRUFDaEI7cUJBQ0Y7aUJBQ0Y7YUFDQTtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztJQUVGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsc0dBQXNHLENBQUMsQ0FBQztBQUNuSSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7SUFDcEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxjQUFjLEdBQUcsb0JBQW9CLENBQUM7SUFFNUMsTUFBTSxXQUFXLEdBQXlCO1FBQ3hDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLFdBQVcsRUFBRSxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztRQUN2QyxtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLHNCQUFzQixFQUFFLENBQUM7Z0JBQ3ZCLElBQUksRUFBRSxJQUFJO2dCQUNWLHVCQUF1QixFQUFFO29CQUN2QixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLGNBQWM7cUJBQzNCO2lCQUNGO2FBQ0Y7U0FDQTtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzdDLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDckQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGVBQWUsRUFBRTtRQUM5QyxrQkFBa0IsRUFBRSxJQUFJO1FBQ3hCLGdCQUFnQixFQUFFLElBQUk7S0FDdkIsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0RBQW9ELEVBQUUsR0FBRyxFQUFFO0lBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLG9CQUFvQixDQUFDO0lBRTVDLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtRQUNsQyxtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLGdCQUFnQixFQUFFLENBQUMsTUFBTSxFQUFFLGdCQUFnQixFQUFFLE9BQU8sQ0FBQztRQUNyRCxzQkFBc0IsRUFBRSxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSTtnQkFDVix1QkFBdUIsRUFBRTtvQkFDdkIsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxjQUFjO3FCQUMzQjtpQkFDRjthQUNGO1NBQ0E7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsa0JBQWtCLEVBQUU7UUFDakQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTix1QkFBdUI7d0JBQ3ZCLDBCQUEwQjtxQkFDM0I7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7Z0JBQ0Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLGNBQWM7d0JBQ2Qsa0JBQWtCO3dCQUNsQixhQUFhO3dCQUNiLGtCQUFrQjt3QkFDbEIsY0FBYzt3QkFDZCxpQkFBaUI7cUJBQ2xCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUNBQWlDOzRCQUNqQyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLE1BQU0sRUFBRSx1QkFBdUI7b0JBQy9CLE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUNBQWlDOzRCQUNqQyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLE1BQU0sRUFBRTt3QkFDTixtQkFBbUI7d0JBQ25CLGtCQUFrQjt3QkFDbEIsY0FBYzt3QkFDZCxnQkFBZ0I7d0JBQ2hCLGdCQUFnQjt3QkFDaEIsc0JBQXNCO3dCQUN0QixhQUFhO3dCQUNiLGdCQUFnQjt3QkFDaEIsZUFBZTt3QkFDZixnQkFBZ0I7d0JBQ2hCLGNBQWM7d0JBQ2Qsb0JBQW9CO3dCQUNwQixzQkFBc0I7cUJBQ3ZCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUNBQWlDOzRCQUNqQyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtJQUN6RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLGNBQWMsR0FBRyxvQkFBb0IsQ0FBQztJQUU1QyxNQUFNLFdBQVcsR0FBeUI7UUFDeEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7UUFDbEMsbUJBQW1CLEVBQUUsV0FBVztRQUNoQyxnQkFBZ0IsRUFBRSxDQUFDLE9BQU8sQ0FBQztRQUMzQixzQkFBc0IsRUFBRSxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSTtnQkFDVix1QkFBdUIsRUFBRTtvQkFDdkIsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxjQUFjO3FCQUMzQjtpQkFDRjthQUNGO1NBQ0E7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsa0JBQWtCLEVBQUU7UUFDakQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTix1QkFBdUI7d0JBQ3ZCLDBCQUEwQjtxQkFDM0I7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7Z0JBQ0Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLG1CQUFtQjt3QkFDbkIsa0JBQWtCO3dCQUNsQixjQUFjO3dCQUNkLGdCQUFnQjt3QkFDaEIsZ0JBQWdCO3dCQUNoQixzQkFBc0I7d0JBQ3RCLGFBQWE7d0JBQ2IsZ0JBQWdCO3dCQUNoQixlQUFlO3dCQUNmLGdCQUFnQjt3QkFDaEIsY0FBYzt3QkFDZCxvQkFBb0I7d0JBQ3BCLHNCQUFzQjtxQkFDdkI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWixpQ0FBaUM7NEJBQ2pDLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsOENBQThDLEVBQUUsR0FBRyxFQUFFO0lBQ3hELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLG9CQUFvQixDQUFDO0lBRTVDLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtRQUNsQyxtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLGdCQUFnQixFQUFFLENBQUMsTUFBTSxDQUFDO1FBQzFCLHNCQUFzQixFQUFFLENBQUM7Z0JBQ3ZCLElBQUksRUFBRSxJQUFJO2dCQUNWLHVCQUF1QixFQUFFO29CQUN2QixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLGNBQWM7cUJBQzNCO2lCQUNGO2FBQ0Y7U0FDQTtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtRQUNqRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLHVCQUF1Qjt3QkFDdkIsMEJBQTBCO3FCQUMzQjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsR0FBRztpQkFDZDtnQkFDRDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sY0FBYzt3QkFDZCxrQkFBa0I7d0JBQ2xCLGFBQWE7d0JBQ2Isa0JBQWtCO3dCQUNsQixjQUFjO3dCQUNkLGlCQUFpQjtxQkFDbEI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWixpQ0FBaUM7NEJBQ2pDLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLG9CQUFvQixDQUFDO0lBRTVDLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtRQUNsQyxtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLGdCQUFnQixFQUFFLENBQUMsZ0JBQWdCLENBQUM7UUFDcEMsc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsY0FBYztxQkFDM0I7aUJBQ0Y7YUFDRjtTQUNBO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1FBQ2pELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sdUJBQXVCO3dCQUN2QiwwQkFBMEI7cUJBQzNCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRSxHQUFHO2lCQUNkO2dCQUNEO29CQUNFLE1BQU0sRUFBRSx1QkFBdUI7b0JBQy9CLE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUNBQWlDOzRCQUNqQyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLGNBQWMsR0FBRyxvQkFBb0IsQ0FBQztJQUU1QyxNQUFNLFdBQVcsR0FBeUI7UUFDeEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsTUFBTSxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDeEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO0tBQzVELENBQUMsQ0FBQztJQUVILE1BQU0sYUFBYSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDakUsT0FBTyxFQUFFLFlBQVk7UUFDckIsSUFBSSxFQUFFLGVBQWU7UUFDckIsT0FBTyxFQUFFLFlBQVksQ0FBQyxPQUFPO0tBQzlCLENBQUMsQ0FBQztJQUVILElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsZ0JBQWdCLEVBQUUsQ0FBQyxPQUFPLENBQUM7UUFDM0Isc0JBQXNCLEVBQUUsYUFBYTtRQUNyQyxzQkFBc0IsRUFBRSxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSTtnQkFDVix1QkFBdUIsRUFBRTtvQkFDdkIsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxjQUFjO3FCQUMzQjtpQkFDRjthQUNGO1NBQ0E7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUzQyw2Q0FBNkM7SUFDN0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxvQkFBb0IsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNsRCxRQUFRLENBQUMscUJBQXFCLENBQUMsb0JBQW9CLEVBQUU7UUFDbkQsT0FBTyxFQUFFLFlBQVk7S0FDdEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO0lBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLHdCQUF3QixDQUFDO0lBQ2hELE1BQU0sb0JBQW9CLEdBQUcsYUFBYSxDQUFDO0lBRTNDLE1BQU0sV0FBVyxHQUF5QjtRQUN4QyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtRQUNsQyxtQkFBbUIsRUFBRSxXQUFXO1FBQ2hDLGdCQUFnQixFQUFFLENBQUMsT0FBTyxDQUFDO1FBQzNCLHNCQUFzQixFQUFFLENBQUM7Z0JBQ3ZCLElBQUksRUFBRSxJQUFJO2dCQUNWLHVCQUF1QixFQUFFO29CQUN2QixlQUFlLEVBQUU7d0JBQ2YsaUJBQWlCLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQzt3QkFDekMsVUFBVSxFQUFFLGNBQWM7cUJBQzNCO2lCQUNGO2FBQ0Y7U0FDQTtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMseUJBQXlCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDdkQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHlCQUF5QixFQUFFO1FBQ3hELElBQUksRUFBRSxJQUFJO1FBQ1YsdUJBQXVCLEVBQUU7WUFDdkIsZUFBZSxFQUFFO2dCQUNmLGlCQUFpQixFQUFFLENBQUMsb0JBQW9CLENBQUM7Z0JBQ3pDLFVBQVUsRUFBRSxjQUFjO2FBQzNCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7SUFDbEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxjQUFjLEdBQUcsb0JBQW9CLENBQUM7SUFDNUMsTUFBTSx3QkFBd0IsR0FBRyxZQUFZLENBQUM7SUFDOUMsTUFBTSx3QkFBd0IsR0FBRyxtQkFBbUIsQ0FBQztJQUVyRCxNQUFNLFdBQVcsR0FBeUI7UUFDeEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QixDQUFDO0lBRUYsZ0JBQWdCO0lBQ2hCLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQy9DLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztRQUMzRCxRQUFRLEVBQUUsa0JBQWtCO0tBQzdCLENBQUMsQ0FBQztJQUVILElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsZ0JBQWdCLEVBQUUsQ0FBQyxPQUFPLENBQUM7UUFDM0Isc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsY0FBYztxQkFDM0I7aUJBQ0Y7YUFDRjtZQUNEO2dCQUNFLElBQUksRUFBRSx3QkFBd0I7Z0JBQzlCLE9BQU8sRUFBRSxRQUFRLENBQUMsT0FBTztnQkFDekIsSUFBSSxFQUFFLHdCQUF3QjthQUMvQixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUV2RCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsSUFBSSxFQUFFLElBQUk7UUFDVix1QkFBdUIsRUFBRTtZQUN2QixlQUFlLEVBQUU7Z0JBQ2YsVUFBVSxFQUFFLGNBQWM7YUFDM0I7U0FDRjtRQUNELE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRSxDQUFDLHFDQUFxQyxFQUFFLEtBQUssQ0FBQztTQUM3RDtLQUNGLENBQUMsQ0FBQztJQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxJQUFJLEVBQUUsd0JBQXdCO1FBQzlCLElBQUksRUFBRSx3QkFBd0I7S0FDL0IsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO0lBQ2hELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sY0FBYyxHQUFHLHVCQUF1QixDQUFDO0lBQy9DLE1BQU0sZ0JBQWdCLEdBQUcsNEJBQTRCLENBQUM7SUFFdEQsTUFBTSxXQUFXLEdBQXlCO1FBQ3hDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1FBQ3ZDLG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsZ0JBQWdCLEVBQUUsQ0FBQyxPQUFPLENBQUM7UUFDM0Isc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsY0FBYztxQkFDM0I7aUJBQ0Y7YUFDRjtZQUNEO2dCQUNFLElBQUksRUFBRSxJQUFJO2dCQUNWLHVCQUF1QixFQUFFO29CQUN2QixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLGdCQUFnQjtxQkFDN0I7aUJBQ0Y7YUFDRixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN2RCxRQUFRLENBQUMscUJBQXFCLENBQUMseUJBQXlCLEVBQUU7UUFDeEQsSUFBSSxFQUFFLElBQUk7UUFDVix1QkFBdUIsRUFBRTtZQUN2QixlQUFlLEVBQUU7Z0JBQ2YsVUFBVSxFQUFFLGNBQWM7YUFDM0I7U0FDRjtRQUNELE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRSxDQUFDLDZDQUE2QyxFQUFFLEtBQUssQ0FBQztTQUNyRTtLQUNGLENBQUMsQ0FBQztJQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx5QkFBeUIsRUFBRTtRQUN4RCxJQUFJLEVBQUUsSUFBSTtRQUNWLHVCQUF1QixFQUFFO1lBQ3ZCLGVBQWUsRUFBRTtnQkFDZixVQUFVLEVBQUUsZ0JBQWdCO2FBQzdCO1NBQ0Y7UUFDRCxPQUFPLEVBQUU7WUFDUCxZQUFZLEVBQUUsQ0FBQyw2Q0FBNkMsRUFBRSxLQUFLLENBQUM7U0FDckU7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw0Q0FBNEMsRUFBRSxHQUFHLEVBQUU7SUFDdEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxXQUFXLEdBQXlCO1FBQ3hDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQztJQUVGLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ2xDLG1CQUFtQixFQUFFLFdBQVc7UUFDaEMsc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsa0JBQWtCO3FCQUMvQjtpQkFDRjthQUNGO1NBQ0E7UUFDRCw4QkFBOEIsRUFBRSxhQUFhO0tBQzlDLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxXQUFXLEVBQUU7WUFDWCxTQUFTLEVBQUU7Z0JBQ1QsV0FBVyxFQUFFO29CQUNYLFlBQVksRUFBRSxDQUFDLGlDQUFpQyxFQUFFLElBQUksQ0FBQztpQkFDeEQ7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdUNBQXVDLEVBQUUsR0FBRyxFQUFFO0lBQ2pELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sZ0JBQWdCLEdBQUcseUJBQXlCLENBQUM7SUFDbkQsTUFBTSxjQUFjLEdBQUcsdUJBQXVCLENBQUM7SUFFL0MsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7UUFDbkMsZUFBZSxFQUFFLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRTtRQUNuRCxpQkFBaUIsRUFBRTtZQUNqQixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLGdCQUFnQixFQUFFLElBQUk7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLFdBQVcsR0FBeUI7UUFDeEMsWUFBWSxFQUFFLGdCQUFnQjtRQUM5QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLEtBQUssR0FBd0I7UUFDakMsbUJBQW1CLEVBQUUsV0FBVztRQUNoQyxzQkFBc0IsRUFBRSxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSTtnQkFDVix1QkFBdUIsRUFBRTtvQkFDdkIsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxjQUFjO3FCQUMzQjtpQkFDRjthQUNGO1NBQ0E7UUFDRCxTQUFTLEVBQUUsSUFBSTtRQUNmLFdBQVcsRUFBRSxHQUFHO0tBQ2pCLENBQUM7SUFFRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3QyxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyxzR0FBc0csQ0FBQyxDQUFDO0FBQ25JLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtJQUM1QyxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDOUQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtRQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztLQUNuRCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBd0I7UUFDakMsc0JBQXNCLEVBQUUsQ0FBQztnQkFDdkIsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsdUJBQXVCLEVBQUU7b0JBQ3ZCLGVBQWUsRUFBRTt3QkFDZixVQUFVLEVBQUUsZ0JBQWdCO3FCQUM3QjtpQkFDRjthQUNGO1NBQ0E7UUFDRCxtQkFBbUIsRUFBRTtZQUNuQixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1NBQ25EO1FBQ0QsaUJBQWlCLEVBQUUsY0FBYztLQUNsQyxDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNyRCxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyxrRkFBa0YsQ0FBQyxDQUFDO0FBQy9HLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgTGFtYmRhVG9LZW5kcmEsIExhbWJkYVRvS2VuZHJhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBrZW5kcmEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWtlbmRyYSc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgeyBUZW1wbGF0ZSB9IGZyb20gJ2F3cy1jZGstbGliL2Fzc2VydGlvbnMnO1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcblxudGVzdCgnTGF1bmNoIHdpdGggbWluaW1hbCBjb2RlIGFuZCBjaGVjayAgc3RydWN0dXJlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEZ1bmN0aW9uTmFtZSA9ICd0ZXN0LWZ1bmN0aW9uLW5hbWUyNDMzNCc7XG4gIGNvbnN0IHRlc3RCdWNrZXROYW1lID0gJ3Rlc3QtYnVja2V0LW5hbWUxMjM0NCc7XG5cbiAgY29uc3QgbGFtYmRhUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGZ1bmN0aW9uTmFtZTogdGVzdEZ1bmN0aW9uTmFtZSxcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMThfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgTGFtYmRhVG9LZW5kcmEoc3RhY2ssICdzYW1wbGUnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhUHJvcHMsXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHtcbiAgICBGdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWUsXG4gICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgIFZhcmlhYmxlczoge1xuICAgICAgICBLRU5EUkFfSU5ERVhfSUQ6IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1wic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLCBcIklkXCJdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICB9KTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpLZW5kcmE6OkluZGV4XCIsIHtcbiAgICBSb2xlQXJuOiB7XG4gICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcInNhbXBsZWtlbmRyYWluZGV4cm9sZXNhbXBsZTRGOUU3QjY2XCIsXG4gICAgICAgIFwiQXJuXCIsXG4gICAgICBdLFxuICAgIH0sXG4gIH0pO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OktlbmRyYTo6RGF0YVNvdXJjZVwiLCB7XG4gICAgVHlwZTogJ1MzJyxcbiAgICBEYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgUzNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIEJ1Y2tldE5hbWU6IHRlc3RCdWNrZXROYW1lXG4gICAgICB9LFxuICAgIH0sXG4gICAgUm9sZUFybjoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcInNhbXBsZWRhdGFzb3VyY2Vyb2xlc2FtcGxlMDVBMDVGOEJEXCIsIFwiQXJuXCJdXG4gICAgfSxcbiAgfSk7XG4gIC8vIENvbmZpcm0gcG9saWN5IGZvciBLZW5kcmEgaW5kZXhcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlJvbGVcIiwge1xuICAgIERlc2NyaXB0aW9uOiBcIkFsbG93IEtlbmRyYSBpbmRleCB0byB3cml0ZSBDbG91ZFdhdGNoIExvZ3NcIixcbiAgICBQb2xpY2llczogW1xuICAgICAge1xuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBY3Rpb246IFwiY2xvdWR3YXRjaDpQdXRNZXRyaWNEYXRhXCIsXG4gICAgICAgICAgICAgIENvbmRpdGlvbjoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICAgICAgXCJjbG91ZHdhdGNoOm5hbWVzcGFjZVwiOiBcIkFXUy9LZW5kcmFcIlxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBcImxvZ3M6Q3JlYXRlTG9nR3JvdXBcIixcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICBcImFybjphd3M6bG9nczpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOlwiLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6QWNjb3VudElkXCJcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXCI6bG9nLWdyb3VwOi9hd3Mva2VuZHJhLypcIlxuICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBcImxvZ3M6RGVzY3JpYmVMb2dHcm91cHNcIixcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOmxvZ3M6XCIsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBcIjpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OkFjY291bnRJZFwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOmxvZy1ncm91cDovYXdzL2tlbmRyYS8qXCJcbiAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dTdHJlYW1cIixcbiAgICAgICAgICAgICAgICBcImxvZ3M6UHV0TG9nRXZlbnRzXCIsXG4gICAgICAgICAgICAgICAgXCJsb2dzOkRlc2NyaWJlTG9nU3RyZWFtXCJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOmxvZ3M6XCIsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBcIjpcIixcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OkFjY291bnRJZFwiXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIFwiOmxvZy1ncm91cDovYXdzL2tlbmRyYS8qOmxvZy1zdHJlYW06KlwiXG4gICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9LFxuICAgICAgICBQb2xpY3lOYW1lOiBcIkFsbG93TG9nZ2luZ1wiXG4gICAgICB9XG4gICAgXSxcbiAgfSk7XG4gIC8vIENvbmZpcm0gcG9saWN5IGZvciBLZW5kcmEgaW5kZXhcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlJvbGVcIiwge1xuICAgIERlc2NyaXB0aW9uOiBcIlBvbGljeSBmb3IgS2VuZHJhIFMzIERhdGEgU291cmNlXCIsXG4gICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgU2VydmljZTogXCJrZW5kcmEuYW1hem9uYXdzLmNvbVwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIFBvbGljaWVzOiBbXG4gICAgICB7XG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEFjdGlvbjogXCJzMzpHZXRPYmplY3RcIixcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiBgYXJuOmF3czpzMzo6OnRlc3QtYnVja2V0LW5hbWUxMjM0NC8qYFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBcInMzOkxpc3RCdWNrZXRcIixcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiBgYXJuOmF3czpzMzo6OnRlc3QtYnVja2V0LW5hbWUxMjM0NGBcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAgIFwia2VuZHJhOkJhdGNoUHV0RG9jdW1lbnRcIixcbiAgICAgICAgICAgICAgICBcImtlbmRyYTpCYXRjaERlbGV0ZURvY3VtZW50XCJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICAgIFwic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLFxuICAgICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgUG9saWN5TmFtZTogXCJzM0NyYXdsUG9saWN5XCJcbiAgICAgIH1cbiAgICBdXG4gIH0pO1xuICAvLyBDb25maXJtIHRoYXQgTGFtYmRhIGZ1bmN0aW9uIGhhcyBRVUVSWSBhY2Nlc3NcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlBvbGljeVwiLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcInhyYXk6UHV0VHJhY2VTZWdtZW50c1wiLFxuICAgICAgICAgICAgXCJ4cmF5OlB1dFRlbGVtZXRyeVJlY29yZHNcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IFwiKlwiXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgIFwia2VuZHJhOlF1ZXJ5XCIsXG4gICAgICAgICAgICBcImtlbmRyYTpEZXNjcmliZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkdldCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkJhdGNoR2V0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6TGlzdCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlJldHJpZXZlXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJzYW1wbGVrZW5kcmFpbmRleHNhbXBsZThBODFBNkMyXCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFwia2VuZHJhOlN1Ym1pdEZlZWRiYWNrXCIsXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdLFxuICAgIH0sXG4gICAgUm9sZXM6IFtcbiAgICAgIHtcbiAgICAgICAgUmVmOiBcInNhbXBsZXRlc3RmdW5jdGlvbm5hbWUyNDMzNFNlcnZpY2VSb2xlOTkzOTVBMDFcIlxuICAgICAgfVxuICAgIF1cbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgcGF0dGVybiBwcm9wZXJ0aWVzIG9uIG1pbmltYWwgbGF1bmNoJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgbmV3Q29uc3RydWN0ID0gbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYVByb3BzLFxuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFt7XG4gICAgICB0eXBlOiAnUzMnLFxuICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgczNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgYnVja2V0TmFtZTogJ3lvdXItYnVja2V0LW5hbWUnLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIF0sXG4gIH0pO1xuXG4gIGV4cGVjdChuZXdDb25zdHJ1Y3QubGFtYmRhRnVuY3Rpb24pLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChuZXdDb25zdHJ1Y3Qua2VuZHJhRGF0YVNvdXJjZXMpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChuZXdDb25zdHJ1Y3Qua2VuZHJhRGF0YVNvdXJjZXMubGVuZ3RoKS50b0VxdWFsKDEpO1xuICBleHBlY3QobmV3Q29uc3RydWN0LmtlbmRyYUluZGV4KS50b0JlRGVmaW5lZCgpO1xuXG59KTtcblxudGVzdCgnTGF1bmNoIHdpdGggVlBDJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEJ1Y2tldE5hbWUgPSAndGVzdC1idWNrZXQtbmFtZTEyNTM5JztcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGRlcGxveVZwYzogdHJ1ZSxcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBrZW5kcmFEYXRhU291cmNlc1Byb3BzOiBbe1xuICAgICAgdHlwZTogJ1MzJyxcbiAgICAgIGRhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldE5hbWU6IHRlc3RCdWNrZXROYW1lLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIF0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICAvLyBDaGVjayB0aGUgVlBDXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6RUMyOjpWUENcIiwge1xuICAgIENpZHJCbG9jazogXCIxMC4wLjAuMC8xNlwiLFxuICAgIEVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICBFbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIEluc3RhbmNlVGVuYW5jeTogXCJkZWZhdWx0XCIsXG4gIH0pO1xuXG4gIC8vIElzIHRoZSBMYW1iZGEgZnVuY3Rpb24gYXNzb2NpYXRlZCB3aXRoIHRoZSBWUENcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHtcbiAgICBWcGNDb25maWc6IHtcbiAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcInNhbXBsZVJlcGxhY2VEZWZhdWx0U2VjdXJpdHlHcm91cHNlY3VyaXR5Z3JvdXBFNTcyNTY2OVwiLFxuICAgICAgICAgICAgXCJHcm91cElkXCJcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBTdWJuZXRJZHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZjogXCJWcGNpc29sYXRlZFN1Ym5ldDFTdWJuZXRFNjJCMUI5QlwiXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBSZWY6IFwiVnBjaXNvbGF0ZWRTdWJuZXQyU3VibmV0MzkyMTcwNTVcIlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KTtcbiAgLy8gQ2hlY2sgdGhhdCB0aGUgTGFtYmRhIGZ1bmN0aW9uIFBvbGljeSBoYXMgcHJvcGVyIG5ldHdvcmsgYWNjZXNzXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZVwiLFxuICAgICAgICAgICAgXCJlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlc1wiLFxuICAgICAgICAgICAgXCJlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZVwiLFxuICAgICAgICAgICAgXCJlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzXCIsXG4gICAgICAgICAgICBcImVjMjpVbmFzc2lnblByaXZhdGVJcEFkZHJlc3Nlc1wiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZTogXCIqXCJcbiAgICAgICAgfSxcbiAgICAgICAge30sXG4gICAgICAgIHt9LFxuICAgICAgICB7fVxuICAgICAgXSxcbiAgICB9LFxuICAgIFJvbGVzOiBbXG4gICAgICB7XG4gICAgICAgIFJlZjogXCJzYW1wbGVMYW1iZGFGdW5jdGlvblNlcnZpY2VSb2xlN0EzQzRBRjVcIlxuICAgICAgfVxuICAgIF1cbiAgfSk7XG5cbiAgLy8gQ2hlY2sgZm9yIHRoZSBLZW5kcmEgZW5kcG9pbnQgaW4gdGhlIFZQQ1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDMjo6VlBDRW5kcG9pbnRcIiwge1xuICAgIFNlcnZpY2VOYW1lOiB7XG4gICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgXCJcIixcbiAgICAgICAgW1xuICAgICAgICAgIFwiY29tLmFtYXpvbmF3cy5cIixcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCIua2VuZHJhXCJcbiAgICAgICAgXVxuICAgICAgXVxuICAgIH0sXG4gICAgVnBjSWQ6IHtcbiAgICAgIFJlZjogXCJWcGM4Mzc4RUIzOFwiXG4gICAgfSxcbiAgICBQcml2YXRlRG5zRW5hYmxlZDogdHJ1ZSxcbiAgICBTZWN1cml0eUdyb3VwSWRzOiBbXG4gICAgICB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJEZWZhdWx0S0VORFJBc2VjdXJpdHlncm91cDM0NTM2QTc5XCIsXG4gICAgICAgICAgXCJHcm91cElkXCJcbiAgICAgICAgXVxuICAgICAgfVxuICAgIF0sXG4gICAgU3VibmV0SWRzOiBbXG4gICAgICB7XG4gICAgICAgIFJlZjogXCJWcGNpc29sYXRlZFN1Ym5ldDFTdWJuZXRFNjJCMUI5QlwiXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBSZWY6IFwiVnBjaXNvbGF0ZWRTdWJuZXQyU3VibmV0MzkyMTcwNTVcIlxuICAgICAgfVxuICAgIF0sXG4gICAgVnBjRW5kcG9pbnRUeXBlOiBcIkludGVyZmFjZVwiXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0xhdW5jaCB3aXRoIGV4aXN0aW5nIGxhbWJkYScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHRlc3RUaW1lb3V0ID0gMTc7XG5cbiAgY29uc3QgdGVzdEZ1bmN0aW9uTmFtZSA9ICd0ZXN0LW5hbWUnO1xuXG4gIGNvbnN0IGV4aXN0aW5nRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnZXhpc3RpbmctZnVuY3Rpb24nLCB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKHRlc3RUaW1lb3V0KSxcbiAgICBmdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWVcbiAgfSk7XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGV4aXN0aW5nTGFtYmRhT2JqOiBleGlzdGluZ0Z1bmN0aW9uLFxuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFtdLFxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLCAxKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHtcbiAgICBUaW1lb3V0OiB0ZXN0VGltZW91dCxcbiAgICBGdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWUsXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NvbmZpcm0gZXJyb3Igd2l0aCBkYXRhIHNvdXJjZSB3aXRoIG5vIGJ1Y2tldCBuYW1lJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBMYW1iZGFUb0tlbmRyYShzdGFjaywgJ3NhbXBsZScsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYVByb3BzLFxuICAgICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgICAgdHlwZTogJ1MzJyxcbiAgICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIF0sXG4gICAgfSk7XG4gIH07XG5cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKC9FcnJvciAtIGFuIFMzIEtlbmRyYSBEYXRhU291cmNlIHJlcXVpcmVzIHRoZSBEYXRhU291cmNlQ29uZmlndXJhdGlvbi5TM0NvbmZpZ3VyYXRpb24uYnVja2V0TmFtZSBwcm9wLyk7XG59KTtcblxudGVzdCgnTGF1bmNoIHdpdGggZXhpc3RpbmcgdnBjJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEJ1Y2tldE5hbWUgPSAndGVzdC1idWNrZXQtbmFtZTIyJztcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGV4aXN0aW5nVnBjOiBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKSxcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBrZW5kcmFEYXRhU291cmNlc1Byb3BzOiBbe1xuICAgICAgdHlwZTogJ1MzJyxcbiAgICAgIGRhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldE5hbWU6IHRlc3RCdWNrZXROYW1lLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIF0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpFQzI6OlZQQ1wiLCAxKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpFQzI6OlZQQ0VuZHBvaW50XCIsIDEpO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBFbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgRW5hYmxlRG5zU3VwcG9ydDogdHJ1ZSxcbiAgfSk7XG5cbn0pO1xuXG50ZXN0KCdMYXVuY2ggd2l0aCBSZWFkL1dyaXRlIHBlcm1pc3Npb25zIG9uIGtlbmRyYSBpbmRleCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHRlc3RCdWNrZXROYW1lID0gJ3Rlc3QtYnVja2V0LW5hbWUyMic7XG5cbiAgY29uc3QgbGFtYmRhUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xOF9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIG5ldyBMYW1iZGFUb0tlbmRyYShzdGFjaywgJ3NhbXBsZScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBpbmRleFBlcm1pc3Npb25zOiBbXCJSZWFEXCIsIFwiU3VibWl0RmVlZEJhY2tcIiwgXCJ3cklURVwiXSwgLy8gdGhpcyBhbHNvIGNoZWNrcyBjYXNlIHNlbnNpdGl2aXR5XG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgICAgIFwieHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcImtlbmRyYTpRdWVyeVwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpHZXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpCYXRjaEdldCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkxpc3QqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpSZXRyaWV2ZVwiLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcImtlbmRyYTpTdWJtaXRGZWVkYmFja1wiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcInNhbXBsZWtlbmRyYWluZGV4c2FtcGxlOEE4MUE2QzJcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJrZW5kcmE6QXNzb2NpYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6QmF0Y2hQdXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpDbGVhclwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGVsZXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGlzYXNzb2NpYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6UHV0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6VXBkYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6U3RhcnQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpTdWJtaXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpTdG9wKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6VGFnUmVzb3VyY2VcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlVudGFnUmVzb3VyY2VcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdMYXVuY2ggd2l0aCBXcml0ZSBwZXJtaXNzaW9ucyBvbiBrZW5kcmEgaW5kZXgnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCB0ZXN0QnVja2V0TmFtZSA9ICd0ZXN0LWJ1Y2tldC1uYW1lMjInO1xuXG4gIGNvbnN0IGxhbWJkYVByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMThfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgTGFtYmRhVG9LZW5kcmEoc3RhY2ssICdzYW1wbGUnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhUHJvcHMsXG4gICAgaW5kZXhQZXJtaXNzaW9uczogW1wiV1JJVEVcIl0sXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgICAgIFwieHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcImtlbmRyYTpBc3NvY2lhdGUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpCYXRjaFB1dCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkNsZWFyXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpDcmVhdGUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpEZWxldGUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpEaXNhc3NvY2lhdGUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpQdXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpVcGRhdGUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpTdGFydCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlN1Ym1pdCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlN0b3AqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpUYWdSZXNvdXJjZVwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6VW50YWdSZXNvdXJjZVwiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJzYW1wbGVrZW5kcmFpbmRleHNhbXBsZThBODFBNkMyXCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ0xhdW5jaCB3aXRoIFJlYWQgcGVybWlzc2lvbnMgb24ga2VuZHJhIGluZGV4JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEJ1Y2tldE5hbWUgPSAndGVzdC1idWNrZXQtbmFtZTIyJztcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYVByb3BzLFxuICAgIGluZGV4UGVybWlzc2lvbnM6IFtcIlJFQURcIl0sXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgICAgIFwieHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcImtlbmRyYTpRdWVyeVwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpHZXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpCYXRjaEdldCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkxpc3QqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpSZXRyaWV2ZVwiLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdMYXVuY2ggd2l0aCBTdWJtaXRGZWVkYmFjayBwZXJtaXNzaW9ucyBvbiBrZW5kcmEgaW5kZXgnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCB0ZXN0QnVja2V0TmFtZSA9ICd0ZXN0LWJ1Y2tldC1uYW1lMjInO1xuXG4gIGNvbnN0IGxhbWJkYVByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMThfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBuZXcgTGFtYmRhVG9LZW5kcmEoc3RhY2ssICdzYW1wbGUnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhUHJvcHMsXG4gICAgaW5kZXhQZXJtaXNzaW9uczogW1wiU1VCTUlURkVFREJBQ0tcIl0sXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICB9KTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgICAgIFwieHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcImtlbmRyYTpTdWJtaXRGZWVkYmFja1wiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcInNhbXBsZWtlbmRyYWluZGV4c2FtcGxlOEE4MUE2QzJcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnTGF1bmNoIHdpdGggZXhpc3Rpbmcga2VuZHJhIGluZGV4JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEJ1Y2tldE5hbWUgPSAndGVzdC1idWNrZXQtbmFtZTIyJztcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgZXhpc3RpbmdSb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnZXhpc3Rpbmctcm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgna2VuZHJhLmFtYXpvbmF3cy5jb20nKSxcbiAgfSk7XG5cbiAgY29uc3QgZXhpc3RpbmdJbmRleCA9IG5ldyBrZW5kcmEuQ2ZuSW5kZXgoc3RhY2ssICdleGlzdGluZy1pbmRleCcsIHtcbiAgICBlZGl0aW9uOiAnRU5URVJQUklTRScsXG4gICAgbmFtZTogJ2V4aXN0aW5nSW5kZXgnLFxuICAgIHJvbGVBcm46IGV4aXN0aW5nUm9sZS5yb2xlQXJuXG4gIH0pO1xuXG4gIG5ldyBMYW1iZGFUb0tlbmRyYShzdGFjaywgJ3NhbXBsZScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBpbmRleFBlcm1pc3Npb25zOiBbXCJXUklURVwiXSxcbiAgICBleGlzdGluZ0tlbmRyYUluZGV4T2JqOiBleGlzdGluZ0luZGV4LFxuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFt7XG4gICAgICB0eXBlOiAnUzMnLFxuICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgczNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgYnVja2V0TmFtZTogdGVzdEJ1Y2tldE5hbWUsXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgXSxcbiAgfSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuXG4gIC8vIE1ha2Ugc3VyZSB3ZSBkaWRuJ3QgY3JlYXRlIGFuIGluZGV4IGFueXdheVxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktlbmRyYTo6SW5kZXhcIiwgMSk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6S2VuZHJhOjpJbmRleFwiLCB7XG4gICAgRWRpdGlvbjogJ0VOVEVSUFJJU0UnXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0xhdW5jaCB3aXRoIFMzIGRhdGEgc291cmNlIHdpdGggb3ZlcnJpZGRlbiBkZWZhdWx0cycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHRlc3RCdWNrZXROYW1lID0gJ3Rlc3QtYnVja2V0LW5hbWUyMjM0MjMnO1xuICBjb25zdCB0ZXN0SW5jbHVzaW9uUGF0dGVybiA9ICd0aGlzLWZvbGRlcic7XG5cbiAgY29uc3QgbGFtYmRhUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xOF9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIG5ldyBMYW1iZGFUb0tlbmRyYShzdGFjaywgJ3NhbXBsZScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBpbmRleFBlcm1pc3Npb25zOiBbXCJXUklURVwiXSxcbiAgICBrZW5kcmFEYXRhU291cmNlc1Byb3BzOiBbe1xuICAgICAgdHlwZTogJ1MzJyxcbiAgICAgIGRhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGluY2x1c2lvblBhdHRlcm5zOiBbdGVzdEluY2x1c2lvblBhdHRlcm5dLFxuICAgICAgICAgIGJ1Y2tldE5hbWU6IHRlc3RCdWNrZXROYW1lLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIF0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpLZW5kcmE6OkRhdGFTb3VyY2VcIiwgMSk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6S2VuZHJhOjpEYXRhU291cmNlXCIsIHtcbiAgICBUeXBlOiAnUzMnLFxuICAgIERhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICBTM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgSW5jbHVzaW9uUGF0dGVybnM6IFt0ZXN0SW5jbHVzaW9uUGF0dGVybl0sXG4gICAgICAgIEJ1Y2tldE5hbWU6IHRlc3RCdWNrZXROYW1lXG4gICAgICB9LFxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnTGF1bmNoIHdpdGggUzMgZGF0YSBzb3VyY2UgYW5kIHVuc3VwcG9ydGVkIGRhdGEgc291cmNlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgdGVzdEJ1Y2tldE5hbWUgPSAndGVzdC1idWNrZXQtbmFtZTIyJztcbiAgY29uc3Qgbm9uSW1wbGVtZW50ZWRTb3VyY2VUeXBlID0gXCJXRUJDUkFXTEVSXCI7XG4gIGNvbnN0IG5vbkltcGxlbWVudGVkU291cmNlTmFtZSA9IFwidGVzdC1vdGhlci1zb3VyY2VcIjtcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgLy8gQ3JlYXRlIGEgcm9sZVxuICBjb25zdCBmYWtlUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ2Zha2VSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdrZW5kcmEuYW1hem9uYXdzLmNvbScpLFxuICAgIHJvbGVOYW1lOiAnZXh0ZXJuYWxGYWtlUm9sZSdcbiAgfSk7XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYVByb3BzLFxuICAgIGluZGV4UGVybWlzc2lvbnM6IFtcIldSSVRFXCJdLFxuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFt7XG4gICAgICB0eXBlOiAnUzMnLFxuICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgczNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgYnVja2V0TmFtZTogdGVzdEJ1Y2tldE5hbWUsXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIHtcbiAgICAgIG5hbWU6IG5vbkltcGxlbWVudGVkU291cmNlTmFtZSxcbiAgICAgIHJvbGVBcm46IGZha2VSb2xlLnJvbGVBcm4sXG4gICAgICB0eXBlOiBub25JbXBsZW1lbnRlZFNvdXJjZVR5cGUsXG4gICAgfV0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKFwiQVdTOjpLZW5kcmE6OkRhdGFTb3VyY2VcIiwgMik7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpLZW5kcmE6OkRhdGFTb3VyY2VcIiwge1xuICAgIFR5cGU6ICdTMycsXG4gICAgRGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgIFMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBCdWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZVxuICAgICAgfSxcbiAgICB9LFxuICAgIFJvbGVBcm46IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXCJzYW1wbGVkYXRhc291cmNlcm9sZXNhbXBsZTA1QTA1RjhCRFwiLCBcIkFyblwiXVxuICAgIH0sXG4gIH0pO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OktlbmRyYTo6RGF0YVNvdXJjZVwiLCB7XG4gICAgTmFtZTogbm9uSW1wbGVtZW50ZWRTb3VyY2VOYW1lLFxuICAgIFR5cGU6IG5vbkltcGxlbWVudGVkU291cmNlVHlwZSxcbiAgfSk7XG5cbn0pO1xuXG50ZXN0KCdMYXVuY2ggd2l0aCBtdWx0aXBsZSBTMyBkYXRhIHNvdXJjZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCB0ZXN0QnVja2V0TmFtZSA9ICd0ZXN0LWJ1Y2tldC1uYW1lMjIzNDInO1xuICBjb25zdCBzZWNvbmRCdWNrZXROYW1lID0gJ3NlY29uZC1idWNrZXQtbmFtZTIyMzQyMzQyJztcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAndHdvLXNvdXJjZXMnLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhUHJvcHMsXG4gICAgaW5kZXhQZXJtaXNzaW9uczogW1wiV1JJVEVcIl0sXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAge1xuICAgICAgdHlwZTogJ1MzJyxcbiAgICAgIGRhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIHMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldE5hbWU6IHNlY29uZEJ1Y2tldE5hbWUsXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XSxcbiAgfSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OktlbmRyYTo6RGF0YVNvdXJjZVwiLCAyKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpLZW5kcmE6OkRhdGFTb3VyY2VcIiwge1xuICAgIFR5cGU6ICdTMycsXG4gICAgRGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgIFMzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBCdWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZVxuICAgICAgfSxcbiAgICB9LFxuICAgIFJvbGVBcm46IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXCJ0d29zb3VyY2VzZGF0YXNvdXJjZXJvbGV0d29zb3VyY2VzMEI4RTI0OTk2XCIsIFwiQXJuXCJdXG4gICAgfSxcbiAgfSk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6S2VuZHJhOjpEYXRhU291cmNlXCIsIHtcbiAgICBUeXBlOiAnUzMnLFxuICAgIERhdGFTb3VyY2VDb25maWd1cmF0aW9uOiB7XG4gICAgICBTM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgQnVja2V0TmFtZTogc2Vjb25kQnVja2V0TmFtZVxuICAgICAgfSxcbiAgICB9LFxuICAgIFJvbGVBcm46IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXCJ0d29zb3VyY2VzZGF0YXNvdXJjZXJvbGV0d29zb3VyY2VzMTY0MTc2QzVFXCIsIFwiQXJuXCJdXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCB3aXRoIGN1c3RvbSBlbnZpcm9ubWVudCB2YXJpYWJsZSBuYW1lJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFQcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgbmV3IExhbWJkYVRvS2VuZHJhKHN0YWNrLCAnc2FtcGxlJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IGxhbWJkYVByb3BzLFxuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFt7XG4gICAgICB0eXBlOiAnUzMnLFxuICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgczNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgYnVja2V0TmFtZTogJ3lvdXItYnVja2V0LW5hbWUnLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIF0sXG4gICAgaW5kZXhJZEVudmlyb25tZW50VmFyaWFibGVOYW1lOiBcIk1ZX1ZBUl9OQU1FXCIsXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkxhbWJkYTo6RnVuY3Rpb25cIiwge1xuICAgIEVudmlyb25tZW50OiB7XG4gICAgICBWYXJpYWJsZXM6IHtcbiAgICAgICAgTVlfVkFSX05BTUU6IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1wic2FtcGxla2VuZHJhaW5kZXhzYW1wbGU4QTgxQTZDMlwiLCBcIklkXCJdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH0pO1xuXG59KTtcblxudGVzdCgnQ29uZmlybSBDaGVja1ZwY1Byb3BzIGlzIGJlaW5nIGNhbGxlZCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHRlc3RGdW5jdGlvbk5hbWUgPSAndGVzdC1mdW5jdGlvbi1uYW1lMjQzMzQnO1xuICBjb25zdCB0ZXN0QnVja2V0TmFtZSA9ICd0ZXN0LWJ1Y2tldC1uYW1lMTIzNDQnO1xuXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIGNvbnN0IGxhbWJkYVByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBmdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWUsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH07XG5cbiAgY29uc3QgcHJvcHM6IExhbWJkYVRvS2VuZHJhUHJvcHMgPSB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogbGFtYmRhUHJvcHMsXG4gICAga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogW3tcbiAgICAgIHR5cGU6ICdTMycsXG4gICAgICBkYXRhU291cmNlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBzM0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBidWNrZXROYW1lOiB0ZXN0QnVja2V0TmFtZSxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICAgIGRlcGxveVZwYzogdHJ1ZSxcbiAgICBleGlzdGluZ1ZwYzogdnBjXG4gIH07XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBMYW1iZGFUb0tlbmRyYShzdGFjaywgJ3NhbXBsZScsIHByb3BzKTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcignRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSBhbiBleGlzdGluZ1ZwYyBvciBzb21lIGNvbWJpbmF0aW9uIG9mIGRlcGxveVZwYyBhbmQgdnBjUHJvcHMsIGJ1dCBub3QgYm90aC5cXG4nKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIGNhbGwgdG8gQ2hlY2tMYW1iZGFQcm9wcycsICgpID0+IHtcbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgbGFtYmRhRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnYS1mdW5jdGlvbicsIHtcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBMYW1iZGFUb0tlbmRyYVByb3BzID0ge1xuICAgIGtlbmRyYURhdGFTb3VyY2VzUHJvcHM6IFt7XG4gICAgICB0eXBlOiAnUzMnLFxuICAgICAgZGF0YVNvdXJjZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgczNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgYnVja2V0TmFtZTogJ3Rlc3RCdWNrZXROYW1lJyxcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICBdLFxuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xNl9YLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgfSxcbiAgICBleGlzdGluZ0xhbWJkYU9iajogbGFtYmRhRnVuY3Rpb24sXG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgTGFtYmRhVG9LZW5kcmEoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHByb3BzKTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcignRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSBsYW1iZGFGdW5jdGlvblByb3BzIG9yIGV4aXN0aW5nTGFtYmRhT2JqLCBidXQgbm90IGJvdGguXFxuJyk7XG59KTtcbiJdfQ==