import numpy as np
from matplotlib import pyplot as plt
from pykrak import pykrak_env
from pykrak import env_pert

"""
Description:
Env factory world
Builders for commonly used environments

Date:
2/17/2021


Author: Hunter Akins

Institution: Scripps Institution of Oceanography, UC San Diego

Copyright (C) 2023  F. Hunter Akins

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
"""

class EnvFactory:
    """
    Registers environment builders  for various environments
    A builder is identified with a key
    Builder implements the __call__ method and returns an Env object
    """
    def __init__(self):
        self.builders = {}

    def register_builder(self, key, builder):
        self.builders[key] = builder

    def create(self, key, **kwargs):
        """ 
        Call the builder associated with key on kwargs
        """
        builder = self.builders[key]
        if not builder:
            raise KeyError(key)
        return builder(**kwargs)

class SwellexBuilder:
    """
    Build an waveguide with i605.prn sound speed and seabed parameters for SwellEx experiment
    """

    def __init__(self):
        self._instance = None

    def __call__(self, **kwargs):
        z_layer1 = np.array([0.0, 0.83, 1.39, 2.13, 2.95, 3.65, 4.53,\
                             5.44, 6.15, 6.8, 7.73, 8.69, 9.45, 10.12,\
                             11.0, 12.0, 12.79, 13.53, 14.29, 15.27, 16.1,\
                             16.85, 17.87, 19.03, 19.78, 20.33, 21.16, 22.17,\
                             23.25, 24.5, 25.77, 26.97, 28.12, 29.1, 29.5,\
                             29.73, 29.9, 30.27, 30.59, 30.98, 31.19, 31.31,\
                             31.44, 31.81, 31.94, 32.08, 32.33, 32.55, 32.71,\
                             32.9, 33.41, 34.17, 35.03, 35.89, 36.78, 37.82,\
                             38.9, 39.81, 40.48, 41.16, 42.24, 43.42, 44.51,\
                             45.65, 46.72, 47.72, 48.72, 49.65, 50.6, 51.08,\
                             51.48, 51.85, 51.98, 52.68, 53.81, 54.86, 55.96,\
                             57.07, 58.15, 59.0, 59.73, 60.6, 61.77, 62.85,\
                             63.85, 64.82, 66.0, 66.96, 67.83, 68.74, 69.74,\
                             70.71, 71.69, 72.65, 73.7, 74.72, 75.93, 77.04,\
                             78.0, 78.83, 79.65, 80.5, 81.32, 81.94, 82.69,\
                             83.79, 84.95, 86.11, 87.25, 88.26, 89.06, 89.88,\
                             90.7, 91.39, 92.02, 92.73, 93.35, 93.85, 94.59,\
                             95.56, 96.08, 96.4, 96.69, 96.88, 97.34, 97.79,\
                             98.42, 98.95, 99.06, 99.28, 99.59, 100.5, 101.39,\
                             102.16, 102.88, 103.66, 104.46, 104.97, 105.47, 106.24,\
                             107.04, 107.6, 108.18, 108.84, 109.53, 109.98, 110.23,\
                             110.71, 111.21, 111.44, 111.68, 112.09, 112.38, 112.75,\
                             113.2, 113.61, 113.93, 114.22, 114.8, 115.94, 117.19,\
                             118.56, 119.98, 121.38, 122.57, 123.74, 125.02, 126.39,\
                             127.67, 129.04, 130.4, 131.52, 132.62, 133.9, 135.11,\
                             136.24, 137.41, 138.69, 139.93, 140.97, 142.04, 143.16,\
                             144.22, 145.22, 146.23, 147.28, 148.25, 149.29, 150.45,\
                             151.46, 152.52, 153.76, 154.85, 155.8, 156.82, 157.94,\
                             159.05, 160.21, 161.33, 162.35, 163.46, 164.52, 164.75,\
                             168.17, 169.2, 170.18, 171.13, 172.08, 172.96, 173.84,\
                             174.68, 175.51, 176.39, 177.13, 177.9, 178.75, 179.73,\
                             180.38, 181.09, 181.78, 182.41, 183.02, 183.57, 184.15,\
                             184.9, 185.59, 186.25, 187.01, 187.95, 188.96, 189.98,\
                             191.04, 192.18, 193.13, 193.9, 194.43, 195.05, 195.71,\
                             196.17, 196.67, 197.27, 197.92, 198.47, 198.97, 199.32,\
                             199.59, 199.95, 200.16, 208.35, 216.5])
        c_layer1 = np.array([1521.94, 1521.95, 1521.94, 1521.9, 1521.84, 1521.74, 
                            1521.53, 1521.18, 1520.84, 1520.51, 1520.04, 1519.58, 1519.13,\
                             1518.59, 1517.45, 1515.25, 1512.28, 1509.48, 1507.26, 1505.05,\
                             1503.4, 1502.35, 1501.27, 1500.28, 1499.89, 1499.64, 1499.22,\
                             1498.59, 1497.96, 1497.13, 1496.52, 1496.12, 1495.77, 1495.49,\
                             1495.4, 1495.34, 1495.3, 1495.21, 1495.12, 1495.02, 1494.95,\
                             1494.91, 1494.87, 1494.7, 1494.63, 1494.56, 1494.4, 1494.27,\
                             1494.21, 1494.14, 1493.93, 1493.67, 1493.44, 1493.25, 1493.1,\
                             1492.92, 1492.68, 1492.54, 1492.45, 1492.37, 1492.27, 1492.18,\
                             1492.13, 1492.08, 1492.03, 1491.98, 1491.93, 1491.89, 1491.84,\
                             1491.82, 1491.79, 1491.77, 1491.76, 1491.69, 1491.54, 1491.37,\
                             1491.23, 1491.13, 1491.04, 1490.97, 1490.91, 1490.84, 1490.74,\
                             1490.66, 1490.59, 1490.51, 1490.42, 1490.35, 1490.3, 1490.24,\
                             1490.17, 1490.07, 1489.93, 1489.85, 1489.79, 1489.73, 1489.68,\
                             1489.66, 1489.63, 1489.61, 1489.61, 1489.62, 1489.62, 1489.63,\
                             1489.63, 1489.63, 1489.62, 1489.59, 1489.55, 1489.51, 1489.46,\
                             1489.41, 1489.33, 1489.24, 1489.16, 1489.14, 1489.13, 1489.12,\
                             1489.07, 1488.96, 1488.89, 1488.83, 1488.78, 1488.75, 1488.65,\
                             1488.59, 1488.55, 1488.51, 1488.5, 1488.49, 1488.46, 1488.33,\
                             1488.25, 1488.2, 1488.18, 1488.17, 1488.16, 1488.15, 1488.13,\
                             1488.08, 1488.05, 1488.03, 1488.0, 1487.99, 1487.97, 1487.96,\
                             1487.95, 1487.92, 1487.87, 1487.84, 1487.81, 1487.77, 1487.75,\
                             1487.73, 1487.71, 1487.7, 1487.69, 1487.69, 1487.7, 1487.73,\
                             1487.78, 1487.83, 1487.91, 1488.02, 1488.09, 1488.14, 1488.19,\
                             1488.28, 1488.4, 1488.54, 1488.67, 1488.74, 1488.81, 1488.87,\
                             1488.92, 1488.95, 1488.97, 1488.96, 1488.95, 1488.93, 1488.92,\
                             1488.91, 1488.91, 1488.91, 1488.91, 1488.89, 1488.89, 1488.89,\
                             1488.9, 1488.9, 1488.92, 1488.93, 1488.95, 1488.96, 1488.98,\
                             1488.99, 1489.0, 1489.0, 1488.99, 1488.96, 1488.92, 1488.9,\
                             1488.88, 1488.53, 1488.47, 1488.42, 1488.44, 1488.43, 1488.45,\
                             1488.46, 1488.48, 1488.48, 1488.5, 1488.51, 1488.53, 1488.57,\
                             1488.55, 1488.6, 1488.61, 1488.61, 1488.55, 1488.55, 1488.54,\
                             1488.54, 1488.55, 1488.53, 1488.5, 1488.51, 1488.53, 1488.51,\
                             1488.47, 1488.4, 1488.35, 1488.34, 1488.32, 1488.31, 1488.27,\
                             1488.25, 1488.25, 1488.26, 1488.27, 1488.29, 1488.3, 1488.3,\
                             1488.32, 1488.31, 1488.31, 1488.26, 1488.26, 1488.26])

        c_layer2 = np.array([1572.73, 1593.02])
        c_layer3 = np.array([1881.02, 3245.8])
        if 'short' in kwargs.keys():
            z_list = [z_layer1, 
                        np.array([216.5, 240.0]),
                        np.array([240.0, 400.0])]
        else:
            z_list = [z_layer1, 
                        np.array([216.5, 240.0]),
                        np.array([240.0, 1040.0])]

        c_list = [c_layer1, c_layer2, c_layer3]
        if 'dz' in kwargs.keys():
            dz = kwargs['dz']
            for i in range(len(z_list)):
                Z = z_list[i][-1] - z_list[i][0]
                N = int(Z/dz)+1
                z_fine = np.linspace(z_list[i][0], z_list[i][-1], N)
                c_layer_fine = np.interp(z_fine, z_list[i], c_list[i])
                z_list[i] = z_fine
                c_list[i] = c_layer_fine
        rho_layer1 = np.ones(z_list[0].size)
        rho_layer2 = 1.8*np.ones(z_list[1].size)
        rho_layer3 = 2.1*np.ones(z_list[2].size)
        rho_list = [rho_layer1, rho_layer2, rho_layer3]
        attn_list = [np.ones(x.size) for x in z_list]
        attn_list[0] *= 0.0
        attn_list[1] *= 0.3
        attn_list[2] *= 0.09

        c_hs = 5200.0
        rho_hs = 2.66
        attn_hs = 0.02
        attn_units = 'dbpkmhz'
        
        if 'single_layer' in kwargs.keys():
            #c_hs = 1881.
            c_hs = 1800.
            #rho_hs = 2.6
            rho_hs = 2.0
            #attn_hs = .09
            attn_hs = 0.
            z_list = z_list[:2]
            rho_list = rho_list[:2]
            c_list = c_list[:2]
            attn_list = attn_list[:2]

        elif 'hs' in kwargs.keys():
            print('halfspace')
            c_hs = 1700.
            #rho_hs = 2.6
            rho_hs = 2.0
            #attn_hs = .09
            attn_hs = 0.
            z_list = [z_list[0]]
            rho_list = [rho_list[0]]
            c_list = [c_list[0]]
            attn_list = [attn_list[0]]

        if 'pert' in kwargs.keys():
            env = env_pert.PertEnv(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        else:
            env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class PekerisBuilder:
    """
    Build an waveguide with i605.prn sound speed and seabed parameters for SwellEx experiment
    """

    def __init__(self):
        self._instance = None

    def __call__(self, **kwargs):
        z_list = [np.array([0.0, 216.5])]
        c_list = [np.array([1500.0, 1500.0])] 
        if 'dz' in kwargs.keys():
            dz = kwargs['dz']
            Z = z_list[0][-1] - z_list[0][0]
            N = int(Z/dz) + 1
            z_fine = np.linspace(z_list[0][0], z_list[0][-1], N)
            c_fine = 1500.*np.ones(N)
            z_list = [z_fine]
            c_list = [c_fine]
        rho_list = [np.ones(z_list[0].size)]
        attn_list = [np.zeros(z_list[0].size)]
        c_hs = 1600.0
        rho_hs = 5.0
        attn_hs = 0.0
        attn_units = 'dbplam'
        env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class HalfspaceBuilder:
    """ Water column ssp, fluid halfspace """
    def __init__(self):
        self._instance = None

    def __call__(self, z_w, c_w, c_hs, rho_hs, attn_hs, attn_units, pert=True, dz=None):
        if dz is not None:
            Z = z_w[-1] - z_w[0]
            N = int(Z/dz)+1
            z_fine = np.linspace(z_w[0], z_w[-1], N)
            c_layer_fine = np.interp(z_fine, z_w, c_w)
            z_w = z_fine
            c_w = c_layer_fine
        rho_w = np.ones(z_w.size)
        attn_w = np.zeros(z_w.size)
        if pert:
            env = env_pert.PertEnv([z_w], [c_w], [rho_w], [attn_w], c_hs, rho_hs, attn_hs, attn_units)
        else:
            env = pykrak_env.Env([z_w], [c_w], [rho_w], [attn_w], c_hs, rho_hs, attn_hs, attn_units)
        return env

class SingleLayerBuilder:
    """ Water column ssp, single fluid layer below, fluid halfpsace """
    def __init__(self):
        self._instance = None

    def __call__(self, z_w, c_w, z_sed, c_sed, rho_sed, attn_sed, c_hs, rho_hs, attn_hs, attn_units, pert=True, dz=None):
        if dz is not None:
            Z = z_w[-1] - z_w[0]
            N = int(Z/dz)+1
            z_fine = np.linspace(z_w[0], z_w[-1], N)
            c_layer_fine = np.interp(z_fine, z_w, c_w)
            z_w = z_fine.copy()
            c_w = c_layer_fine.copy()

            Z = z_sed[-1] - z_sed[0]
            N = int(Z/dz)+1
            z_fine = np.linspace(z_sed[0], z_sed[-1], N)
            c_layer_fine = np.interp(z_fine, z_sed, c_sed)
            z_sed = z_fine.copy()
            c_sed = c_layer_fine.copy()
            
        rho_w = np.ones(z_w.size)
        attn_w = np.zeros(z_w.size)
        rho_sed = rho_sed * np.ones(z_sed.size)
        attn_sed = attn_sed * np.ones(z_sed.size)
        if pert:
            env = env_pert.PertEnv([z_w, z_sed], [c_w, c_sed], [rho_w, rho_sed], [attn_w, attn_sed], c_hs, rho_hs, attn_hs, attn_units)
        else:
            env = pykrak_env.Env([z_w, z_sed], [c_w, c_sed], [rho_w, rho_sed], [attn_w, attn_sed], c_hs, rho_hs, attn_hs, attn_units)
        return env

class SwellexGeoBuilder:
    """
    Build an waveguide with i605.prn sound speed and model parameter m
    Geoacoustic model consists of three layers
    t1 = 
    The first layer starts at 216.5 meters, and goes to 240.0
    Second goes from 240.0 to 1040.0
    The third goes from the bottom of the second to 1040 meters (considered a prior)
    Halfspace is from 1040 meters onward, with c= 5200
    Parameter vector
    m - [c1_top, c1_bott, rho1, c2_top, c2_bott, rho2, c3_top, c3_bott, rho3]
    """

    def __init__(self):
        self._instance = None

    def __call__(self,m):
        z_layer1 = np.array([0.0, 0.83, 1.39, 2.13, 2.95, 3.65, 4.53,\
                             5.44, 6.15, 6.8, 7.73, 8.69, 9.45, 10.12,\
                             11.0, 12.0, 12.79, 13.53, 14.29, 15.27, 16.1,\
                             16.85, 17.87, 19.03, 19.78, 20.33, 21.16, 22.17,\
                             23.25, 24.5, 25.77, 26.97, 28.12, 29.1, 29.5,\
                             29.73, 29.9, 30.27, 30.59, 30.98, 31.19, 31.31,\
                             31.44, 31.81, 31.94, 32.08, 32.33, 32.55, 32.71,\
                             32.9, 33.41, 34.17, 35.03, 35.89, 36.78, 37.82,\
                             38.9, 39.81, 40.48, 41.16, 42.24, 43.42, 44.51,\
                             45.65, 46.72, 47.72, 48.72, 49.65, 50.6, 51.08,\
                             51.48, 51.85, 51.98, 52.68, 53.81, 54.86, 55.96,\
                             57.07, 58.15, 59.0, 59.73, 60.6, 61.77, 62.85,\
                             63.85, 64.82, 66.0, 66.96, 67.83, 68.74, 69.74,\
                             70.71, 71.69, 72.65, 73.7, 74.72, 75.93, 77.04,\
                             78.0, 78.83, 79.65, 80.5, 81.32, 81.94, 82.69,\
                             83.79, 84.95, 86.11, 87.25, 88.26, 89.06, 89.88,\
                             90.7, 91.39, 92.02, 92.73, 93.35, 93.85, 94.59,\
                             95.56, 96.08, 96.4, 96.69, 96.88, 97.34, 97.79,\
                             98.42, 98.95, 99.06, 99.28, 99.59, 100.5, 101.39,\
                             102.16, 102.88, 103.66, 104.46, 104.97, 105.47, 106.24,\
                             107.04, 107.6, 108.18, 108.84, 109.53, 109.98, 110.23,\
                             110.71, 111.21, 111.44, 111.68, 112.09, 112.38, 112.75,\
                             113.2, 113.61, 113.93, 114.22, 114.8, 115.94, 117.19,\
                             118.56, 119.98, 121.38, 122.57, 123.74, 125.02, 126.39,\
                             127.67, 129.04, 130.4, 131.52, 132.62, 133.9, 135.11,\
                             136.24, 137.41, 138.69, 139.93, 140.97, 142.04, 143.16,\
                             144.22, 145.22, 146.23, 147.28, 148.25, 149.29, 150.45,\
                             151.46, 152.52, 153.76, 154.85, 155.8, 156.82, 157.94,\
                             159.05, 160.21, 161.33, 162.35, 163.46, 164.52, 164.75,\
                             168.17, 169.2, 170.18, 171.13, 172.08, 172.96, 173.84,\
                             174.68, 175.51, 176.39, 177.13, 177.9, 178.75, 179.73,\
                             180.38, 181.09, 181.78, 182.41, 183.02, 183.57, 184.15,\
                             184.9, 185.59, 186.25, 187.01, 187.95, 188.96, 189.98,\
                             191.04, 192.18, 193.13, 193.9, 194.43, 195.05, 195.71,\
                             196.17, 196.67, 197.27, 197.92, 198.47, 198.97, 199.32,\
                             199.59, 199.95, 200.16, 208.35, 216.5])
        c_layer1 = np.array([1521.94, 1521.95, 1521.94, 1521.9, 1521.84, 1521.74, 
                            1521.53, 1521.18, 1520.84, 1520.51, 1520.04, 1519.58, 1519.13,\
                             1518.59, 1517.45, 1515.25, 1512.28, 1509.48, 1507.26, 1505.05,\
                             1503.4, 1502.35, 1501.27, 1500.28, 1499.89, 1499.64, 1499.22,\
                             1498.59, 1497.96, 1497.13, 1496.52, 1496.12, 1495.77, 1495.49,\
                             1495.4, 1495.34, 1495.3, 1495.21, 1495.12, 1495.02, 1494.95,\
                             1494.91, 1494.87, 1494.7, 1494.63, 1494.56, 1494.4, 1494.27,\
                             1494.21, 1494.14, 1493.93, 1493.67, 1493.44, 1493.25, 1493.1,\
                             1492.92, 1492.68, 1492.54, 1492.45, 1492.37, 1492.27, 1492.18,\
                             1492.13, 1492.08, 1492.03, 1491.98, 1491.93, 1491.89, 1491.84,\
                             1491.82, 1491.79, 1491.77, 1491.76, 1491.69, 1491.54, 1491.37,\
                             1491.23, 1491.13, 1491.04, 1490.97, 1490.91, 1490.84, 1490.74,\
                             1490.66, 1490.59, 1490.51, 1490.42, 1490.35, 1490.3, 1490.24,\
                             1490.17, 1490.07, 1489.93, 1489.85, 1489.79, 1489.73, 1489.68,\
                             1489.66, 1489.63, 1489.61, 1489.61, 1489.62, 1489.62, 1489.63,\
                             1489.63, 1489.63, 1489.62, 1489.59, 1489.55, 1489.51, 1489.46,\
                             1489.41, 1489.33, 1489.24, 1489.16, 1489.14, 1489.13, 1489.12,\
                             1489.07, 1488.96, 1488.89, 1488.83, 1488.78, 1488.75, 1488.65,\
                             1488.59, 1488.55, 1488.51, 1488.5, 1488.49, 1488.46, 1488.33,\
                             1488.25, 1488.2, 1488.18, 1488.17, 1488.16, 1488.15, 1488.13,\
                             1488.08, 1488.05, 1488.03, 1488.0, 1487.99, 1487.97, 1487.96,\
                             1487.95, 1487.92, 1487.87, 1487.84, 1487.81, 1487.77, 1487.75,\
                             1487.73, 1487.71, 1487.7, 1487.69, 1487.69, 1487.7, 1487.73,\
                             1487.78, 1487.83, 1487.91, 1488.02, 1488.09, 1488.14, 1488.19,\
                             1488.28, 1488.4, 1488.54, 1488.67, 1488.74, 1488.81, 1488.87,\
                             1488.92, 1488.95, 1488.97, 1488.96, 1488.95, 1488.93, 1488.92,\
                             1488.91, 1488.91, 1488.91, 1488.91, 1488.89, 1488.89, 1488.89,\
                             1488.9, 1488.9, 1488.92, 1488.93, 1488.95, 1488.96, 1488.98,\
                             1488.99, 1489.0, 1489.0, 1488.99, 1488.96, 1488.92, 1488.9,\
                             1488.88, 1488.53, 1488.47, 1488.42, 1488.44, 1488.43, 1488.45,\
                             1488.46, 1488.48, 1488.48, 1488.5, 1488.51, 1488.53, 1488.57,\
                             1488.55, 1488.6, 1488.61, 1488.61, 1488.55, 1488.55, 1488.54,\
                             1488.54, 1488.55, 1488.53, 1488.5, 1488.51, 1488.53, 1488.51,\
                             1488.47, 1488.4, 1488.35, 1488.34, 1488.32, 1488.31, 1488.27,\
                             1488.25, 1488.25, 1488.26, 1488.27, 1488.29, 1488.3, 1488.3,\
                             1488.32, 1488.31, 1488.31, 1488.26, 1488.26, 1488.26])

        

        c_layer2 = np.array([m[0], m[1]])
        if m.size > 5:
            c_layer3 = np.array([m[4], m[5]])
        else:
            c_layer3 = np.array([m[4], 3240.0])
        rho2 = m[2]
        
        z12 = z_layer1[-1]
        z23 = z12 + m[3]
        if m.size > 5:
            delta_z2 = m[7]
            rho3 = m[6]
        else:
            rho3 = 2.5 
            delta_z2 = 800.0

        z34 = z23 + delta_z2
        #z_34 = z_23 + np.mean(c_layer3)*tt3
    
        z_list = [z_layer1, 
                    np.array([z12, z23]),
                    np.array([z23, z34])]
        c_list = [c_layer1, c_layer2, c_layer3]
        rho_layer1 = np.ones(z_layer1.size)
        rho_layer2 = rho2*np.ones(z_list[1].size)
        rho_layer3 = rho3*np.ones(z_list[2].size)
        rho_list = [rho_layer1, rho_layer2, rho_layer3]
        attn_list = [np.ones(x.size) for x in z_list]
        attn_list[0] *= 0.0
        attn_list[1] *= 0.0
        attn_list[2] *= 0.0
        c_hs = 5200.0
        rho_hs = 2.66
        attn_hs = 0.02
        attn_units = 'dbpkmhz'
        env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class SwellPekerisGeoBuilder:
    """
    Build an waveguide with i605.prn sound speed and seabed parameters for SwellEx experiment
    """

    def __init__(self):
        self._instance = None

    def __call__(self, m):
        z_layer1 = np.array([0.0, 0.83, 1.39, 2.13, 2.95, 3.65, 4.53,\
                             5.44, 6.15, 6.8, 7.73, 8.69, 9.45, 10.12,\
                             11.0, 12.0, 12.79, 13.53, 14.29, 15.27, 16.1,\
                             16.85, 17.87, 19.03, 19.78, 20.33, 21.16, 22.17,\
                             23.25, 24.5, 25.77, 26.97, 28.12, 29.1, 29.5,\
                             29.73, 29.9, 30.27, 30.59, 30.98, 31.19, 31.31,\
                             31.44, 31.81, 31.94, 32.08, 32.33, 32.55, 32.71,\
                             32.9, 33.41, 34.17, 35.03, 35.89, 36.78, 37.82,\
                             38.9, 39.81, 40.48, 41.16, 42.24, 43.42, 44.51,\
                             45.65, 46.72, 47.72, 48.72, 49.65, 50.6, 51.08,\
                             51.48, 51.85, 51.98, 52.68, 53.81, 54.86, 55.96,\
                             57.07, 58.15, 59.0, 59.73, 60.6, 61.77, 62.85,\
                             63.85, 64.82, 66.0, 66.96, 67.83, 68.74, 69.74,\
                             70.71, 71.69, 72.65, 73.7, 74.72, 75.93, 77.04,\
                             78.0, 78.83, 79.65, 80.5, 81.32, 81.94, 82.69,\
                             83.79, 84.95, 86.11, 87.25, 88.26, 89.06, 89.88,\
                             90.7, 91.39, 92.02, 92.73, 93.35, 93.85, 94.59,\
                             95.56, 96.08, 96.4, 96.69, 96.88, 97.34, 97.79,\
                             98.42, 98.95, 99.06, 99.28, 99.59, 100.5, 101.39,\
                             102.16, 102.88, 103.66, 104.46, 104.97, 105.47, 106.24,\
                             107.04, 107.6, 108.18, 108.84, 109.53, 109.98, 110.23,\
                             110.71, 111.21, 111.44, 111.68, 112.09, 112.38, 112.75,\
                             113.2, 113.61, 113.93, 114.22, 114.8, 115.94, 117.19,\
                             118.56, 119.98, 121.38, 122.57, 123.74, 125.02, 126.39,\
                             127.67, 129.04, 130.4, 131.52, 132.62, 133.9, 135.11,\
                             136.24, 137.41, 138.69, 139.93, 140.97, 142.04, 143.16,\
                             144.22, 145.22, 146.23, 147.28, 148.25, 149.29, 150.45,\
                             151.46, 152.52, 153.76, 154.85, 155.8, 156.82, 157.94,\
                             159.05, 160.21, 161.33, 162.35, 163.46, 164.52, 164.75,\
                             168.17, 169.2, 170.18, 171.13, 172.08, 172.96, 173.84,\
                             174.68, 175.51, 176.39, 177.13, 177.9, 178.75, 179.73,\
                             180.38, 181.09, 181.78, 182.41, 183.02, 183.57, 184.15,\
                             184.9, 185.59, 186.25, 187.01, 187.95, 188.96, 189.98,\
                             191.04, 192.18, 193.13, 193.9, 194.43, 195.05, 195.71,\
                             196.17, 196.67, 197.27, 197.92, 198.47, 198.97, 199.32,\
                             199.59, 199.95, 200.16, 208.35, 216.5])
        c_layer1 = np.array([1521.94, 1521.95, 1521.94, 1521.9, 1521.84, 1521.74, 
                            1521.53, 1521.18, 1520.84, 1520.51, 1520.04, 1519.58, 1519.13,\
                             1518.59, 1517.45, 1515.25, 1512.28, 1509.48, 1507.26, 1505.05,\
                             1503.4, 1502.35, 1501.27, 1500.28, 1499.89, 1499.64, 1499.22,\
                             1498.59, 1497.96, 1497.13, 1496.52, 1496.12, 1495.77, 1495.49,\
                             1495.4, 1495.34, 1495.3, 1495.21, 1495.12, 1495.02, 1494.95,\
                             1494.91, 1494.87, 1494.7, 1494.63, 1494.56, 1494.4, 1494.27,\
                             1494.21, 1494.14, 1493.93, 1493.67, 1493.44, 1493.25, 1493.1,\
                             1492.92, 1492.68, 1492.54, 1492.45, 1492.37, 1492.27, 1492.18,\
                             1492.13, 1492.08, 1492.03, 1491.98, 1491.93, 1491.89, 1491.84,\
                             1491.82, 1491.79, 1491.77, 1491.76, 1491.69, 1491.54, 1491.37,\
                             1491.23, 1491.13, 1491.04, 1490.97, 1490.91, 1490.84, 1490.74,\
                             1490.66, 1490.59, 1490.51, 1490.42, 1490.35, 1490.3, 1490.24,\
                             1490.17, 1490.07, 1489.93, 1489.85, 1489.79, 1489.73, 1489.68,\
                             1489.66, 1489.63, 1489.61, 1489.61, 1489.62, 1489.62, 1489.63,\
                             1489.63, 1489.63, 1489.62, 1489.59, 1489.55, 1489.51, 1489.46,\
                             1489.41, 1489.33, 1489.24, 1489.16, 1489.14, 1489.13, 1489.12,\
                             1489.07, 1488.96, 1488.89, 1488.83, 1488.78, 1488.75, 1488.65,\
                             1488.59, 1488.55, 1488.51, 1488.5, 1488.49, 1488.46, 1488.33,\
                             1488.25, 1488.2, 1488.18, 1488.17, 1488.16, 1488.15, 1488.13,\
                             1488.08, 1488.05, 1488.03, 1488.0, 1487.99, 1487.97, 1487.96,\
                             1487.95, 1487.92, 1487.87, 1487.84, 1487.81, 1487.77, 1487.75,\
                             1487.73, 1487.71, 1487.7, 1487.69, 1487.69, 1487.7, 1487.73,\
                             1487.78, 1487.83, 1487.91, 1488.02, 1488.09, 1488.14, 1488.19,\
                             1488.28, 1488.4, 1488.54, 1488.67, 1488.74, 1488.81, 1488.87,\
                             1488.92, 1488.95, 1488.97, 1488.96, 1488.95, 1488.93, 1488.92,\
                             1488.91, 1488.91, 1488.91, 1488.91, 1488.89, 1488.89, 1488.89,\
                             1488.9, 1488.9, 1488.92, 1488.93, 1488.95, 1488.96, 1488.98,\
                             1488.99, 1489.0, 1489.0, 1488.99, 1488.96, 1488.92, 1488.9,\
                             1488.88, 1488.53, 1488.47, 1488.42, 1488.44, 1488.43, 1488.45,\
                             1488.46, 1488.48, 1488.48, 1488.5, 1488.51, 1488.53, 1488.57,\
                             1488.55, 1488.6, 1488.61, 1488.61, 1488.55, 1488.55, 1488.54,\
                             1488.54, 1488.55, 1488.53, 1488.5, 1488.51, 1488.53, 1488.51,\
                             1488.47, 1488.4, 1488.35, 1488.34, 1488.32, 1488.31, 1488.27,\
                             1488.25, 1488.25, 1488.26, 1488.27, 1488.29, 1488.3, 1488.3,\
                             1488.32, 1488.31, 1488.31, 1488.26, 1488.26, 1488.26])

        D = 150.
        z_layer2 = np.array([z_layer1[-1], z_layer1[-1] + D])
        if m.size == 3:
            beta = m[-1]
            if m[0] + beta*D  < 1500.:
                print('WARNING BETA IS TOO BIG')
        else:
            beta = 0.
        c_layer2 = np.array([m[0], m[0] + D*beta])
        rho2 = m[1]
        rho_layer2 = rho2 * np.ones(2)

        z_list = [z_layer1, z_layer2]
        c_list = [c_layer1, c_layer2]
        rho_list = [np.ones(z_list[0].size), rho_layer2]
        attn_list = [np.zeros(z_list[0].size), np.zeros(z_list[1].size)]
        rho_hs = 3.0
        c_hs = 5200.
        attn_hs = 0.1
        attn_units = 'dbplam'
        env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class SwellConstantLayeredBottomBuilder:
    """
    Build an waveguide with i605.prn sound speed and seabed parameters for SwellEx experiment
    Each layer has constant sound speed 
    """

    def __init__(self):
        self._instance = None

    def __call__(self, m):
        """
        m - np arr
        m[0:(m.size-1)/2] = is layer depth for each layer (meters)
        Num layers = (m.size-1)/2
        m[(m.size-1)/2], m[onward] are sound speed in layer 0, 1, ..., i-2, ...
        m[-1] is bottom halfspace speed (cb_hs)
        """
        z_layer1 = np.array([0.0, 0.83, 1.39, 2.13, 2.95, 3.65, 4.53,\
                             5.44, 6.15, 6.8, 7.73, 8.69, 9.45, 10.12,\
                             11.0, 12.0, 12.79, 13.53, 14.29, 15.27, 16.1,\
                             16.85, 17.87, 19.03, 19.78, 20.33, 21.16, 22.17,\
                             23.25, 24.5, 25.77, 26.97, 28.12, 29.1, 29.5,\
                             29.73, 29.9, 30.27, 30.59, 30.98, 31.19, 31.31,\
                             31.44, 31.81, 31.94, 32.08, 32.33, 32.55, 32.71,\
                             32.9, 33.41, 34.17, 35.03, 35.89, 36.78, 37.82,\
                             38.9, 39.81, 40.48, 41.16, 42.24, 43.42, 44.51,\
                             45.65, 46.72, 47.72, 48.72, 49.65, 50.6, 51.08,\
                             51.48, 51.85, 51.98, 52.68, 53.81, 54.86, 55.96,\
                             57.07, 58.15, 59.0, 59.73, 60.6, 61.77, 62.85,\
                             63.85, 64.82, 66.0, 66.96, 67.83, 68.74, 69.74,\
                             70.71, 71.69, 72.65, 73.7, 74.72, 75.93, 77.04,\
                             78.0, 78.83, 79.65, 80.5, 81.32, 81.94, 82.69,\
                             83.79, 84.95, 86.11, 87.25, 88.26, 89.06, 89.88,\
                             90.7, 91.39, 92.02, 92.73, 93.35, 93.85, 94.59,\
                             95.56, 96.08, 96.4, 96.69, 96.88, 97.34, 97.79,\
                             98.42, 98.95, 99.06, 99.28, 99.59, 100.5, 101.39,\
                             102.16, 102.88, 103.66, 104.46, 104.97, 105.47, 106.24,\
                             107.04, 107.6, 108.18, 108.84, 109.53, 109.98, 110.23,\
                             110.71, 111.21, 111.44, 111.68, 112.09, 112.38, 112.75,\
                             113.2, 113.61, 113.93, 114.22, 114.8, 115.94, 117.19,\
                             118.56, 119.98, 121.38, 122.57, 123.74, 125.02, 126.39,\
                             127.67, 129.04, 130.4, 131.52, 132.62, 133.9, 135.11,\
                             136.24, 137.41, 138.69, 139.93, 140.97, 142.04, 143.16,\
                             144.22, 145.22, 146.23, 147.28, 148.25, 149.29, 150.45,\
                             151.46, 152.52, 153.76, 154.85, 155.8, 156.82, 157.94,\
                             159.05, 160.21, 161.33, 162.35, 163.46, 164.52, 164.75,\
                             168.17, 169.2, 170.18, 171.13, 172.08, 172.96, 173.84,\
                             174.68, 175.51, 176.39, 177.13, 177.9, 178.75, 179.73,\
                             180.38, 181.09, 181.78, 182.41, 183.02, 183.57, 184.15,\
                             184.9, 185.59, 186.25, 187.01, 187.95, 188.96, 189.98,\
                             191.04, 192.18, 193.13, 193.9, 194.43, 195.05, 195.71,\
                             196.17, 196.67, 197.27, 197.92, 198.47, 198.97, 199.32,\
                             199.59, 199.95, 200.16, 208.35, 216.5])
        c_layer1 = np.array([1521.94, 1521.95, 1521.94, 1521.9, 1521.84, 1521.74, 
                            1521.53, 1521.18, 1520.84, 1520.51, 1520.04, 1519.58, 1519.13,\
                             1518.59, 1517.45, 1515.25, 1512.28, 1509.48, 1507.26, 1505.05,\
                             1503.4, 1502.35, 1501.27, 1500.28, 1499.89, 1499.64, 1499.22,\
                             1498.59, 1497.96, 1497.13, 1496.52, 1496.12, 1495.77, 1495.49,\
                             1495.4, 1495.34, 1495.3, 1495.21, 1495.12, 1495.02, 1494.95,\
                             1494.91, 1494.87, 1494.7, 1494.63, 1494.56, 1494.4, 1494.27,\
                             1494.21, 1494.14, 1493.93, 1493.67, 1493.44, 1493.25, 1493.1,\
                             1492.92, 1492.68, 1492.54, 1492.45, 1492.37, 1492.27, 1492.18,\
                             1492.13, 1492.08, 1492.03, 1491.98, 1491.93, 1491.89, 1491.84,\
                             1491.82, 1491.79, 1491.77, 1491.76, 1491.69, 1491.54, 1491.37,\
                             1491.23, 1491.13, 1491.04, 1490.97, 1490.91, 1490.84, 1490.74,\
                             1490.66, 1490.59, 1490.51, 1490.42, 1490.35, 1490.3, 1490.24,\
                             1490.17, 1490.07, 1489.93, 1489.85, 1489.79, 1489.73, 1489.68,\
                             1489.66, 1489.63, 1489.61, 1489.61, 1489.62, 1489.62, 1489.63,\
                             1489.63, 1489.63, 1489.62, 1489.59, 1489.55, 1489.51, 1489.46,\
                             1489.41, 1489.33, 1489.24, 1489.16, 1489.14, 1489.13, 1489.12,\
                             1489.07, 1488.96, 1488.89, 1488.83, 1488.78, 1488.75, 1488.65,\
                             1488.59, 1488.55, 1488.51, 1488.5, 1488.49, 1488.46, 1488.33,\
                             1488.25, 1488.2, 1488.18, 1488.17, 1488.16, 1488.15, 1488.13,\
                             1488.08, 1488.05, 1488.03, 1488.0, 1487.99, 1487.97, 1487.96,\
                             1487.95, 1487.92, 1487.87, 1487.84, 1487.81, 1487.77, 1487.75,\
                             1487.73, 1487.71, 1487.7, 1487.69, 1487.69, 1487.7, 1487.73,\
                             1487.78, 1487.83, 1487.91, 1488.02, 1488.09, 1488.14, 1488.19,\
                             1488.28, 1488.4, 1488.54, 1488.67, 1488.74, 1488.81, 1488.87,\
                             1488.92, 1488.95, 1488.97, 1488.96, 1488.95, 1488.93, 1488.92,\
                             1488.91, 1488.91, 1488.91, 1488.91, 1488.89, 1488.89, 1488.89,\
                             1488.9, 1488.9, 1488.92, 1488.93, 1488.95, 1488.96, 1488.98,\
                             1488.99, 1489.0, 1489.0, 1488.99, 1488.96, 1488.92, 1488.9,\
                             1488.88, 1488.53, 1488.47, 1488.42, 1488.44, 1488.43, 1488.45,\
                             1488.46, 1488.48, 1488.48, 1488.5, 1488.51, 1488.53, 1488.57,\
                             1488.55, 1488.6, 1488.61, 1488.61, 1488.55, 1488.55, 1488.54,\
                             1488.54, 1488.55, 1488.53, 1488.5, 1488.51, 1488.53, 1488.51,\
                             1488.47, 1488.4, 1488.35, 1488.34, 1488.32, 1488.31, 1488.27,\
                             1488.25, 1488.25, 1488.26, 1488.27, 1488.29, 1488.3, 1488.3,\
                             1488.32, 1488.31, 1488.31, 1488.26, 1488.26, 1488.26])

        num_layers = int((m.size-1) / 2)
        z_list = [z_layer1]
        c_list = [c_layer1]
        rho_list = [np.ones(z_list[0].size)]
        attn_list = [np.zeros(z_list[0].size)]
        for i in range(num_layers):
            z_top = z_list[-1][-1]
            layer_dz = m[i]
            layer_speed = m[num_layers + i]
            z_bott = z_top + layer_dz
            z_list.append(np.array([z_top, z_bott]))
            c_list.append(np.array([layer_speed, layer_speed]))
            rho_list.append(np.array([2.5, 2.5]))
            attn_list.append(np.array([0.0, 0.0]))
            
        rho_hs = 2.5
        c_hs = m[-1]
        attn_hs = 0.0
        attn_units = 'dbplam'
        env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class SavexBuilder:
    """
    Build an waveguide with i605.prn sound speed and seabed parameters for SwellEx experiment
    """

    def __init__(self):
        self._instance = None

    def __call__(self, **kwargs):
        z_layer1 = np.array([0.,  0.431,  0.931,  1.431,  1.931,  2.431,  2.931,  3.431,  3.931,
                4.431,  4.931,  5.431,  5.931,  6.431,  6.931,  7.431,  7.931,
                8.431,  8.931,  9.431,  9.931, 10.431, 10.931, 11.431, 11.931,
               12.431, 12.931, 13.431, 13.931, 14.431, 14.931, 15.431, 15.931,
               16.431, 16.931, 17.431, 17.931, 18.431, 18.931, 19.431, 19.931,
               20.431, 20.931, 21.431, 21.931, 22.431, 22.931, 23.431, 23.931,
               24.431, 24.931, 25.431, 25.931, 26.431, 26.931, 27.431, 27.931,
               28.431, 28.931, 29.431, 29.931, 30.431, 30.931, 31.431, 31.931,
               32.431, 32.931, 33.431, 33.931, 34.431, 34.931, 35.431, 35.931,
               36.431, 36.931, 37.431, 37.931, 38.431, 38.931, 39.431, 39.931,
               40.431, 40.931, 41.431, 41.931, 42.431, 42.931, 43.431, 43.931,
               44.431, 44.931, 45.431, 45.931, 46.431, 46.931, 47.431, 47.931,
               48.431, 48.931, 49.431, 49.931, 50.431, 50.931, 51.431, 51.931,
               52.431, 52.931, 53.431, 53.931, 54.431, 54.931, 55.431, 55.931,
               56.431, 56.931, 57.431, 57.931, 58.431, 58.931, 59.431, 59.931,
               60.431, 60.931, 61.431, 61.931, 62.431, 62.931, 63.431, 63.931,
               64.431, 64.931, 65.431, 65.931, 66.431, 66.931, 67.431, 67.931,
               68.431, 68.931, 69.431, 69.931, 70.431, 70.931, 71.431, 71.931,
               72.431, 72.931, 73.431, 73.931, 74.431, 74.931, 75.431, 75.931,
               76.431, 76.931, 77.431, 77.931, 78.431, 78.931, 79.431, 79.931,
               80.431, 80.931, 81.431, 81.931, 82.431, 82.931, 83.431, 83.931,
               84.431, 84.931, 85.431, 85.931, 86.431, 86.931, 87.431, 87.931,
               88.431, 88.931, 89.431, 89.931, 90.431, 90.931, 91.431, 91.931,
               92.431, 92.931, 93.431, 93.931, 94.431, 94.931, 95.431, 95.931,
               96.431, 96.931, 97.431, 97.931, 98.431, 100.])
        c_layer1 = np.array([1512.24436725, 1512.24436725, 1512.36070493, 1512.41669767, 1512.51150445,
            1512.70254447, 1512.79454973, 1512.73867221, 1512.73101338, 1512.66952502, 1512.74724252, 1512.72259392, 1512.6246293 ,
            1511.98613316, 1511.08746177, 1512.48452633, 1510.71318936, 1509.9839924 , 1509.49116315, 1508.94537855, 1507.514515  ,
            1506.45500553, 1504.62082978, 1505.39308437, 1502.5765917 , 1502.09673349, 1501.96078002, 1501.80431748, 1501.69388231,
            1501.84486283, 1501.23284546, 1501.1956009 , 1502.02364676, 1502.59747925, 1502.92133177, 1503.02656474, 1503.39521977,
            1503.91222944, 1504.24327261, 1504.20599509, 1503.72737646, 1503.28297996, 1502.95859435, 1502.75607422, 1502.55948419,
            1502.56201386, 1502.57053318, 1502.41561118, 1501.99709228, 1501.21754681, 1500.17649936, 1499.51330298, 1498.79009692,
            1497.9395551 , 1497.14393038, 1496.41027093, 1495.28372827, 1494.86676738, 1494.64823379, 1494.5326209 , 1494.44355041,
            1494.36327211, 1494.28085521, 1494.18194798, 1494.08747317, 1493.98361608, 1493.91446708, 1493.81847346, 1493.6255129 ,
            1493.47745112, 1493.46522826, 1493.56696865, 1493.64635371, 1493.65856595, 1493.67747235, 1493.70922288, 1493.72896385,
            1493.75014638, 1493.78569176, 1493.77925522, 1493.79131858, 1493.89494268, 1494.15734667, 1494.44903457, 1494.59693157,
            1494.66510782, 1494.76025004, 1494.8653586 , 1494.94256304, 1494.96845118, 1494.96991349, 1494.94173404, 1494.88020407,
            1494.81235049, 1494.76736161, 1494.76128017, 1494.74187641, 1494.71495518, 1494.68919474, 1494.66775471, 1494.6725757 ,
            1494.70487162, 1494.74089542, 1494.85551208, 1494.98291984, 1495.21479201, 1495.5831339 , 1496.06838971, 1496.66214021,
            1497.08845402, 1497.20920895, 1497.32267417, 1497.50160411, 1497.65561871, 1497.78529718, 1498.00537656, 1498.18412241,
            1498.27374106, 1498.305664  , 1498.35322194, 1498.556036  , 1498.80466115, 1498.92211658, 1498.98386256, 1499.02236343,
            1499.05502646, 1499.12450976, 1499.25379013, 1499.41166273, 1499.54134248, 1499.67737226, 1499.84066513, 1500.00996903,
            1500.1377689 , 1500.21162716, 1500.28598299, 1500.34752796, 1500.42306655, 1500.71301391, 1501.01541629, 1501.7307509 ,
            1502.83497943, 1503.77675649, 1504.28105578, 1504.33141386, 1504.15054031, 1504.15724114, 1504.13387322, 1504.13661596,
            1504.16826833, 1504.21533837, 1504.25519869, 1504.26777106, 1504.27542146, 1504.2859973 , 1504.29554063, 1504.30665044,
            1504.31775646, 1504.32819899, 1504.33740076, 1504.34710769, 1504.35801911, 1504.37008954, 1504.38016674, 1504.38806957,
            1504.3974699 , 1504.40585664, 1504.41546731, 1504.42522232, 1504.43516948, 1504.44437323, 1504.45302141, 1504.4624431 ,
            1504.47156331, 1504.47987001, 1504.48906411, 1504.49809728, 1504.50592205, 1504.51524246, 1504.52188999, 1504.54125342,
            1504.5596993 , 1504.5761595 , 1504.587895  , 1504.59830697, 1504.60834352, 1504.61773308, 1504.62955706, 1504.63727656,
            1504.64654781, 1504.65607063, 1504.6648576 , 1504.67420421, 1504.68325182, 1504.68992803, 1504.6986109 , 1504.70725794,
            1504.7119336 , 1504.7])

        dz = .1
        N = int(100 / dz)
        z_fine = np.linspace(0, 100., N)
        c_fine = np.interp(z_fine, z_layer1, c_layer1)

        z_layer1= z_fine    
        c_layer1= c_fine    
        
        z_list = [z_layer1]
        c_list = [c_layer1]
        rho_layer1 = np.ones(z_layer1.size)
        rho_list = [rho_layer1]
        attn_list = [np.zeros(z_layer1.size)]
        if 'single_layer' in kwargs.keys():
            h = 25. 
            N = int(h / dz)
            z_layer2 = np.linspace(z_layer1[-1], z_layer1[-1] + h, N)
            c_layer2 = 1600. + (z_layer2 - z_layer2[0])*1.
            rho_layer2 = 1.8*np.ones((N))
            attn_layer2 = np.zeros((N))
            z_list.append(z_layer2)
            c_list.append(c_layer2)
            rho_list.append(rho_layer2)
            attn_list.append(attn_layer2)

        if 'double_layer' in kwargs.keys():
            h = 25. 
            N = int(h / dz)
            z_layer2 = np.linspace(z_layer1[-1], z_layer1[-1] + h, N)
            c_layer2 = 1600. + (z_layer2 - z_layer2[0])*1.
            rho_layer2 = 1.8*np.ones((N))
            attn_layer2 = np.zeros((N))
            z_list.append(z_layer2)
            c_list.append(c_layer2)
            rho_list.append(rho_layer2)
            attn_list.append(attn_layer2)

            h = 25. 
            N = int(h / dz)
            z_layer3 = np.linspace(z_layer2[-1], z_layer2[-1] + h, N)
            c_layer3 = 1700. + (z_layer3 - z_layer3[0])*1.
            rho_layer3 = 2.0*np.ones((N))
            attn_layer3 = np.zeros((N))
            z_list.append(z_layer3)
            c_list.append(c_layer3)
            rho_list.append(rho_layer3)
            attn_list.append(attn_layer3)

        if 'triple_layer' in kwargs.keys():
            h = 25. 
            N = int(h / dz)
            z_layer2 = np.linspace(z_layer1[-1], z_layer1[-1] + h, N)
            c_layer2 = 1600. + (z_layer2 - z_layer2[0])*1.
            rho_layer2 = 1.8*np.ones((N))
            attn_layer2 = np.zeros((N))
            z_list.append(z_layer2)
            c_list.append(c_layer2)
            rho_list.append(rho_layer2)
            attn_list.append(attn_layer2)

            h = 25. 
            N = int(h / dz)
            z_layer3 = np.linspace(z_layer2[-1], z_layer2[-1] + h, N)
            c_layer3 = 1700. + (z_layer3 - z_layer3[0])*1.
            rho_layer3 = 2.0*np.ones((N))
            attn_layer3 = np.zeros((N))
            z_list.append(z_layer3)
            c_list.append(c_layer3)
            rho_list.append(rho_layer3)
            attn_list.append(attn_layer3)

            h = 25. 
            N = int(h / dz)
            z_layer4 = np.linspace(z_layer3[-1], z_layer3[-1] + h, N)
            c_layer4 = 1700. + (z_layer4 - z_layer4[0])*1.
            rho_layer4 = 2.0*np.ones((N))
            attn_layer4 = np.zeros((N))
            z_list.append(z_layer4)
            c_list.append(c_layer4)
            rho_list.append(rho_layer4)
            attn_list.append(attn_layer4)


        c_hs = 5200.
        rho_hs = 2.6
        attn_hs = 0.
        attn_units = 'dbplam'
        if 'pert' in kwargs.keys():
            env = env_pert.PertEnv(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        else:
            env = pykrak_env.Env(z_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

class IsoLayerBuilder:
    """
    Constant SSP in each layer
    """
    def __init__(self):
        self._instance = None

    def __call__(self, cw, Z, h_list, c_list, rho_list, attn_list, c_hs, rho_hs, attn_hs, attn_units, pert=True):
        """
        cw - float sound speed in water
        Z - water layer thickness
        h_list - list of layer thicknesses
        c_list - list of layer compressional speeds
        rho_list - list of layer densities
        attn_list - list of layer attenuations
        attn_units - attenuatino units
        """
        zw = np.array([0., Z])
        cw = np.array([cw, cw])
        rhow = np.ones(2)
        attnw = np.zeros(2)
        env_z_list = [zw]
        env_c_list = [cw]
        env_rho_list = [rhow]
        env_attn_list = [attnw]
        num_layers = len(h_list)
        for i in range(num_layers):
            h = h_list[i]
            z_layer = np.array([env_z_list[-1][-1], env_z_list[-1][-1] + h])
            c_layer = np.array([c_list[i], c_list[i]])
            rho_layer = np.array([rho_list[i], rho_list[i]])
            attn_layer = np.array([attn_list[i], attn_list[i]])
            env_z_list.append(z_layer)
            env_c_list.append(c_layer)
            env_rho_list.append(rho_layer)
            env_attn_list.append(attn_layer)


        if pert == True:
            env = env_pert.PertEnv(env_z_list, env_c_list, env_rho_list, env_attn_list, c_hs, rho_hs, attn_hs, attn_units)
        else:
            env = pykrak_env.Env(env_z_list, env_c_list, env_rho_list, env_attn_list, c_hs, rho_hs, attn_hs, attn_units)
        return env

factory = EnvFactory()
factory.register_builder('swellex', SwellexBuilder)
factory.register_builder('hs', HalfspaceBuilder)
factory.register_builder('single_layer', SingleLayerBuilder)
factory.register_builder('pekeris', PekerisBuilder)
factory.register_builder('swell_geo', SwellexGeoBuilder)
factory.register_builder('pekeris_geo', SwellPekerisGeoBuilder)
factory.register_builder('swell_const_layer', SwellConstantLayeredBottomBuilder)
factory.register_builder('savex', SavexBuilder)
factory.register_builder('iso_layer', IsoLayerBuilder)
