"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const defaults = require("@aws-solutions-constructs/core");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration test for alb with 2 Lambda targets';
const props = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: true,
    albLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
};
const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
const secondProps = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    ruleProps: {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    },
    existingVpc: firstConstruct.vpc,
    existingLoadBalancerObj: firstConstruct.loadBalancer,
    publicApi: true
};
new lib_1.AlbToLambda(stack, 'test-two', secondProps);
defaults.addCfnSuppressRules(firstConstruct.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = firstConstruct.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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