# Copyright (C) 2018-2025, earthobservations developers.
# Distributed under the MIT License. See LICENSE for more info.
"""Enumeration for radar periods."""

from enum import Enum


class Parameter(Enum):
    """Parameter enum with fixed names of parameters being used in the entire library.

    Groups are:
        - Clouds
        - Evapotranspiration / Evaporation
        - Fog
        - Humidity
        - Ice
        - Precipitation
        - Pressure
        - Radiation
        - Snow
        - Soil
        - Solar
        - Special
            - special parameters which may not be sorted elsewhere
        - Temperature
        - Visibility
        - Water equivalent
        - Weather / Weather phenomena
        - Wind

    Subgroups are:
        - averaged
        - aggregated
        - instant
        - duration
        - type
        - distance/length
        - probability
        - percentage/frequency
        - error (model related e.g. Mosmix)
        - count
        - time

    """

    # Clouds
    # ---- averaged ----
    # total
    CLOUD_COVER_TOTAL = "CLOUD_COVER_TOTAL"
    CLOUD_COVER_TOTAL_MIDNIGHT_TO_MIDNIGHT = "CLOUD_COVER_TOTAL_MIDNIGHT_TO_MIDNIGHT"
    CLOUD_COVER_TOTAL_MIDNIGHT_TO_MIDNIGHT_MANUAL = "CLOUD_COVER_TOTAL_MIDNIGHT_TO_MIDNIGHT_MANUAL"
    CLOUD_COVER_TOTAL_SUNRISE_TO_SUNSET = "CLOUD_COVER_TOTAL_SUNRISE_TO_SUNSET"
    CLOUD_COVER_TOTAL_SUNRISE_TO_SUNSET_MANUAL = "CLOUD_COVER_TOTAL_SUNRISE_TO_SUNSET_MANUAL"
    CLOUD_COVER_EFFECTIVE = "CLOUD_COVER_EFFECTIVE"
    # layers
    CLOUD_COVER_LAYER1 = "CLOUD_COVER_LAYER1"
    CLOUD_COVER_LAYER2 = "CLOUD_COVER_LAYER2"
    CLOUD_COVER_LAYER3 = "CLOUD_COVER_LAYER3"
    CLOUD_COVER_LAYER4 = "CLOUD_COVER_LAYER4"
    # below - above
    CLOUD_COVER_BELOW_500FT = "CLOUD_COVER_BELOW_500FT"
    CLOUD_COVER_BELOW_1000FT = "CLOUD_COVER_BELOW_1000FT"
    CLOUD_COVER_BETWEEN_2KM_TO_7KM = "CLOUD_COVER_BETWEEN_2KM_TO_7KM"
    CLOUD_COVER_BELOW_7KM = "CLOUD_COVER_BELOW_7KM"
    CLOUD_COVER_ABOVE_7KM = "CLOUD_COVER_ABOVE_7KM"
    # density
    CLOUD_DENSITY = "CLOUD_DENSITY"
    # ---- type ----
    CLOUD_TYPE_LAYER1 = "CLOUD_TYPE_LAYER1"
    CLOUD_TYPE_LAYER2 = "CLOUD_TYPE_LAYER2"
    CLOUD_TYPE_LAYER3 = "CLOUD_TYPE_LAYER3"
    CLOUD_TYPE_LAYER4 = "CLOUD_TYPE_LAYER4"
    CLOUD_COVER_TOTAL_INDEX = "CLOUD_COVER_TOTAL_INDEX"
    # ---- distance ----
    CLOUD_HEIGHT_LAYER1 = "CLOUD_HEIGHT_LAYER1"
    CLOUD_HEIGHT_LAYER2 = "CLOUD_HEIGHT_LAYER2"
    CLOUD_HEIGHT_LAYER3 = "CLOUD_HEIGHT_LAYER3"
    CLOUD_HEIGHT_LAYER4 = "CLOUD_HEIGHT_LAYER4"
    # cloud base height
    CLOUD_BASE_CONVECTIVE = "CLOUD_BASE_CONVECTIVE"
    # ---- abbreviation ----
    CLOUD_TYPE_LAYER1_ABBREVIATION = "CLOUD_TYPE_LAYER1_ABBREVIATION"
    CLOUD_TYPE_LAYER2_ABBREVIATION = "CLOUD_TYPE_LAYER2_ABBREVIATION"
    CLOUD_TYPE_LAYER3_ABBREVIATION = "CLOUD_TYPE_LAYER3_ABBREVIATION"
    CLOUD_TYPE_LAYER4_ABBREVIATION = "CLOUD_TYPE_LAYER4_ABBREVIATION"

    # Evapotranspiration
    # ---- aggregated ----
    EVAPOTRANSPIRATION_POTENTIAL_LAST_24H = "EVAPOTRANSPIRATION_POTENTIAL_LAST_24H"
    EVAPORATION_HEIGHT = "EVAPORATION_HEIGHT"
    EVAPORATION_HEIGHT_MULTIDAY = "EVAPORATION_HEIGHT_MULTIDAY"
    # ---- count ----
    # Number of days included in the multiday evaporation total
    COUNT_DAYS_MULTIDAY_EVAPORATION = "COUNT_DAYS_MULTIDAY_EVAPORATION"

    # Fog
    # ---- probability ----
    PROBABILITY_FOG_LAST_1H = "PROBABILITY_FOG_LAST_1H"
    PROBABILITY_FOG_LAST_6H = "PROBABILITY_FOG_LAST_6H"
    PROBABILITY_FOG_LAST_12H = "PROBABILITY_FOG_LAST_12H"
    PROBABILITY_FOG_LAST_24H = "PROBABILITY_FOG_LAST_24H"

    # Humidity
    # ---- averaged ----
    HUMIDITY = "HUMIDITY"
    HUMIDITY_ABSOLUTE = "HUMIDITY_ABSOLUTE"
    # ECCC special parameter
    HUMIDEX = "HUMIDEX"

    # Ice
    # ---- length ----
    ICE_ON_WATER_THICKNESS = "THICKNESS_OF_ICE_ON_WATER"

    # Precipitation
    # ---- aggregated ----
    PRECIPITATION_HEIGHT = "PRECIPITATION_HEIGHT"
    PRECIPITATION_HEIGHT_DAY = "PRECIPITATION_HEIGHT_DAY"
    PRECIPITATION_HEIGHT_NIGHT = "PRECIPITATION_HEIGHT_NIGHT"
    PRECIPITATION_HEIGHT_LIQUID = "PRECIPITATION_HEIGHT_LIQUID"
    PRECIPITATION_HEIGHT_DROPLET = "PRECIPITATION_HEIGHT_DROPLET"
    PRECIPITATION_HEIGHT_ROCKER = "PRECIPITATION_HEIGHT_ROCKER"
    PRECIPITATION_HEIGHT_LAST_1H = "PRECIPITATION_HEIGHT_LAST_1H"
    PRECIPITATION_HEIGHT_LAST_3H = "PRECIPITATION_HEIGHT_LAST_3H"
    PRECIPITATION_HEIGHT_LAST_6H = "PRECIPITATION_HEIGHT_LAST_6H"
    PRECIPITATION_HEIGHT_LAST_9H = "PRECIPITATION_HEIGHT_LAST_9H"
    PRECIPITATION_HEIGHT_LAST_12H = "PRECIPITATION_HEIGHT_LAST_12H"
    PRECIPITATION_HEIGHT_LAST_15H = "PRECIPITATION_HEIGHT_LAST_15H"
    PRECIPITATION_HEIGHT_LAST_18H = "PRECIPITATION_HEIGHT_LAST_18H"
    PRECIPITATION_HEIGHT_LAST_21H = "PRECIPITATION_HEIGHT_LAST_21H"
    PRECIPITATION_HEIGHT_LAST_24H = "PRECIPITATION_HEIGHT_LAST_24H"
    PRECIPITATION_HEIGHT_MULTIDAY = "PRECIPITATION_HEIGHT_MULTIDAY"
    # precipitation height consistent with significant weather
    PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H = "PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_1H"
    PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H = "PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_3H"
    PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H = "PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_6H"
    PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H = "PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_12H"
    PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H = "PRECIPITATION_HEIGHT_SIGNIFICANT_WEATHER_LAST_24H"
    PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H = "PRECIPITATION_HEIGHT_LIQUID_SIGNIFICANT_WEATHER_LAST_1H"
    # ---- extremes ----
    PRECIPITATION_HEIGHT_MAX = "PRECIPITATION_HEIGHT_MAX"
    PRECIPITATION_HEIGHT_LIQUID_MAX = "PRECIPITATION_HEIGHT_LIQUID_MAX"
    # ---- type ----
    PRECIPITATION_FORM = "PRECIPITATION_FORM"  # what type of precipitation, snow, ice?
    PRECIPITATION_INDEX = "PRECIPITATION_INDEX"  # True or False
    # ---- duration ----
    PRECIPITATION_DURATION = "PRECIPITATION_DURATION"
    # ---- probability ----
    # greater 0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_6H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_12H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_0MM_LAST_24H"
    # greater 0.1
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_1MM_LAST_1H"
    # greater 0.2
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_1H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_6H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_12H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_2MM_LAST_24H"
    # greater 0.3
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_3MM_LAST_1H"
    # greater 0.5
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_5MM_LAST_1H"
    # greater 0.7
    PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_0_7MM_LAST_1H"
    # greater 1.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_1H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_6H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_12H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_1MM_LAST_24H"
    # greater 2.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_2MM_LAST_1H"
    # greater 3.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_3MM_LAST_1H"
    # greater 5.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_1H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_6H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_12H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_5MM_LAST_24H"
    # greater 10.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_10MM_LAST_1H"
    # greater 15.0
    PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_15MM_LAST_1H"
    PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H = "PROBABILITY_PRECIPITATION_HEIGHT_GT_25MM_LAST_1H"

    PROBABILITY_PRECIPITATION_LAST_1H = "PROBABILITY_PRECIPITATION_LAST_1H"
    PROBABILITY_PRECIPITATION_LAST_6H = "PROBABILITY_PRECIPITATION_LAST_6H"
    PROBABILITY_PRECIPITATION_LAST_12H = "PROBABILITY_PRECIPITATION_LAST_12H"
    PROBABILITY_PRECIPITATION_LAST_24H = "PROBABILITY_PRECIPITATION_LAST_24H"

    PROBABILITY_DRIZZLE_LAST_1H = "PROBABILITY_DRIZZLE_LAST_1H"
    PROBABILITY_DRIZZLE_LAST_6H = "PROBABILITY_DRIZZLE_LAST_6H"
    PROBABILITY_DRIZZLE_LAST_12H = "PROBABILITY_DRIZZLE_LAST_12H"

    PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H = "PROBABILITY_PRECIPITATION_STRATIFORM_LAST_1H"
    PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H = "PROBABILITY_PRECIPITATION_STRATIFORM_LAST_6H"
    PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H = "PROBABILITY_PRECIPITATION_STRATIFORM_LAST_12H"

    PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H = "PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_1H"
    PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H = "PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_6H"
    PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H = "PROBABILITY_PRECIPITATION_CONVECTIVE_LAST_12H"

    PROBABILITY_PRECIPITATION_LIQUID_LAST_1H = "PROBABILITY_PRECIPITATION_LIQUID_LAST_1H"
    PROBABILITY_PRECIPITATION_LIQUID_LAST_6H = "PROBABILITY_PRECIPITATION_LIQUID_LAST_6H"
    PROBABILITY_PRECIPITATION_LIQUID_LAST_12H = "PROBABILITY_PRECIPITATION_LIQUID_LAST_12H"

    PROBABILITY_PRECIPITATION_SOLID_LAST_1H = "PROBABILITY_PRECIPITATION_SOLID_LAST_1H"
    PROBABILITY_PRECIPITATION_SOLID_LAST_6H = "PROBABILITY_PRECIPITATION_SOLID_LAST_6H"
    PROBABILITY_PRECIPITATION_SOLID_LAST_12H = "PROBABILITY_PRECIPITATION_SOLID_LAST_12H"

    PROBABILITY_PRECIPITATION_FREEZING_LAST_1H = "PROBABILITY_PRECIPITATION_FREEZING_LAST_1H"
    PROBABILITY_PRECIPITATION_FREEZING_LAST_6H = "PROBABILITY_PRECIPITATION_FREEZING_LAST_6H"
    PROBABILITY_PRECIPITATION_FREEZING_LAST_12H = "PROBABILITY_PRECIPITATION_FREEZING_LAST_12H"
    # ---- frequency ----
    PRECIPITATION_FREQUENCY = "PRECIPITATION_FREQUENCY"
    # ---- count ----
    # Number of days included in the multiday precipitation total
    COUNT_DAYS_MULTIDAY_PRECIPITATION = "COUNT_DAYS_MULTIDAY_PRECIPITATION"
    # Number of days with non-zero precipitation included in multiday precipitation total
    COUNT_DAYS_MULTIDAY_PRECIPITATION_HEIGHT_GT_0MM = "COUNT_DAYS_MULTIDAY_PRECIPITATION_HEIGHT_GT_0MM"

    # PRESSURE
    # ---- averaged ----
    # TODO: should we add _MEAN here?
    PRESSURE_AIR_SITE = "PRESSURE_AIR_SITE"  # air pressure at station height [SH]
    PRESSURE_AIR_SITE_DELTA_LAST_3H = "PRESSURE_AIR_SITE_DELTA_LAST_3H"
    PRESSURE_AIR_SITE_MAX = "PRESSURE_AIR_SITE_MAX"
    PRESSURE_AIR_SITE_MIN = "PRESSURE_AIR_SITE_MIN"
    PRESSURE_AIR_SITE_REDUCED = "PRESSURE_AIR_SITE_REDUCED"
    PRESSURE_AIR_SEA_LEVEL = "PRESSURE_AIR_SEA_LEVEL"  # air pressure at sea level [SL]
    PRESSURE_VAPOR = "PRESSURE_VAPOR"
    # ---- error ----
    ERROR_ABSOLUTE_PRESSURE_AIR_SITE = "ERROR_ABSOLUTE_PRESSURE_AIR_SITE"

    # RADIATION
    # ---- averaged ----
    RADIATION_SKY_SHORT_WAVE_DIFFUSE = "RADIATION_SKY_SHORT_WAVE_DIFFUSE"
    RADIATION_SKY_SHORT_WAVE_DIRECT = "RADIATION_SKY_SHORT_WAVE_DIRECT"
    # sum of short wave radiation a.k.a. RADIATION_SKY_SHORT_WAVE_TOTAL
    RADIATION_GLOBAL = "RADIATION_GLOBAL"
    RADIATION_GLOBAL_LAST_3H = "RADIATION_GLOBAL_LAST_3H"
    RADIATION_SKY_LONG_WAVE = "RADIATION_SKY_LONG_WAVE"
    RADIATION_SKY_LONG_WAVE_LAST_3H = "RADIATION_SKY_LONG_WAVE_LAST_3H"
    # ---- probability ----
    PROBABILITY_RADIATION_GLOBAL_LAST_1H = "PROBABILITY_RADIATION_GLOBAL_LAST_1H"

    # SNOW
    # ---- distance ----
    SNOW_DEPTH = "SNOW_DEPTH"
    SNOW_DEPTH_NEW = "SNOW_DEPTH_NEW"  # difference to previous snow depth
    SNOW_DEPTH_EXCELLED = "SNOW_DEPTH_EXCELLED"  # with spade or some device
    SNOW_DEPTH_NEW_MULTIDAY = "SNOW_DEPTH_NEW_MULTIDAY"
    SNOW_DEPTH_MANUAL = "SNOW_DEPTH_MANUAL"
    # ---- extremes ----
    SNOW_DEPTH_NEW_MAX = "SNOW_DEPTH_NEW_MAX"
    SNOW_DEPTH_MAX = "SNOW_DEPTH_MAX"
    # ---- count ----
    # Number of days included in the multiday snowfall total
    # TODO: maybe this should be COUNT_DAYS_MULTIDAY_SNOW_DEPTH instead
    COUNT_DAYS_MULTIDAY_SNOW_DEPTH_NEW = "COUNT_DAYS_MULTIDAY_SNOW_DEPTH"

    # SOIL
    FROZEN_GROUND_LAYER_BASE = "FROZEN_GROUND_LAYER_BASE"
    FROZEN_GROUND_LAYER_TOP = "FROZEN_GROUND_LAYER_TOP"
    FROZEN_GROUND_LAYER_THICKNESS = "FROZEN_GROUND_LAYER_THICKNESS"

    # SOLAR
    # ---- duration ----
    SUNSHINE_DURATION = "SUNSHINE_DURATION"
    SUNSHINE_DURATION_LAST_3H = "SUNSHINE_DURATION_LAST_3H"
    SUNSHINE_DURATION_YESTERDAY = "SUNSHINE_DURATION_YESTERDAY"
    # ---- angle ----
    SUN_ZENITH_ANGLE = "SUN_ZENITH_ANGLE"
    # ---- probability ----
    PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H = "PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_0PCT_LAST_24H"
    PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H = (
        "PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_30PCT_LAST_24H"
    )
    PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H = (
        "PROBABILITY_SUNSHINE_DURATION_RELATIVE_GT_60PCT_LAST_24H"
    )
    # ---- percentage ----
    SUNSHINE_DURATION_RELATIVE = "SUNSHINE_DURATION_RELATIVE"
    SUNSHINE_DURATION_RELATIVE_LAST_24H = "SUNSHINE_DURATION_RELATIVE_LAST_24H"
    # ---- time ----
    TRUE_LOCAL_TIME = "TRUE_LOCAL_TIME"  # alt name: mean solar time

    # SPECIAL
    # ---- time ----
    END_OF_INTERVAL = "END_OF_INTERVAL"  # time based
    # ---- distance ----
    DISTANCE_RIVER_GAUGE_HEIGHT = "DIFFERENCE_RIVER_GAUGE_HEIGHT"

    # TEMPERATURE
    # ---- instant ----
    TEMPERATURE_AIR_2M = "TEMPERATURE_AIR_2M"
    # ----- averaged ----
    TEMPERATURE_AIR_MEAN_0_05M = "TEMPERATURE_AIR_MEAN_0_05M"
    TEMPERATURE_AIR_MEAN_2M = "TEMPERATURE_AIR_MEAN_2M"
    TEMPERATURE_AIR_MEAN_2M_LAST_24H = "TEMPERATURE_AIR_MEAN_2M_LAST_24H"
    TEMPERATURE_AIR_MAX_2M_LAST_24H = "TEMPERATURE_AIR_MAX_2M_LAST_24H"
    TEMPERATURE_AIR_MIN_2M_LAST_24H = "TEMPERATURE_AIR_MIN_2M_LAST_24H"
    TEMPERATURE_AIR_MAX_2M_MEAN = "TEMPERATURE_AIR_MAX_2M_MEAN"
    TEMPERATURE_AIR_MIN_2M_MEAN = "TEMPERATURE_AIR_MIN_2M_MEAN"

    TEMPERATURE_DEW_POINT_MEAN_2M = "TEMPERATURE_DEW_POINT_MEAN_2M"
    # wind chill temperature by NWS (https://www.weather.gov/gjt/windchill)
    TEMPERATURE_WIND_CHILL = "TEMPERATURE_WIND_CHILL"

    TEMPERATURE_SOIL_MEAN_0_02M = "TEMPERATURE_SOIL_MEAN_0_02M"
    TEMPERATURE_SOIL_MEAN_0_05M = "TEMPERATURE_SOIL_MEAN_0_05M"
    TEMPERATURE_SOIL_MEAN_0_1M = "TEMPERATURE_SOIL_MEAN_0_1M"
    TEMPERATURE_SOIL_MEAN_0_2M = "TEMPERATURE_SOIL_MEAN_0_2M"
    TEMPERATURE_SOIL_MEAN_0_5M = "TEMPERATURE_SOIL_MEAN_0_5M"
    TEMPERATURE_SOIL_MEAN_1M = "TEMPERATURE_SOIL_MEAN_1M"
    TEMPERATURE_SOIL_MEAN_2M = "TEMPERATURE_SOIL_MEAN_2M"

    TEMPERATURE_WET_MEAN_2M = "TEMPERATURE_WET_MEAN_2M"
    # ---- extremes ----
    TEMPERATURE_AIR_MAX_2M = "TEMPERATURE_AIR_MAX_2M"
    TEMPERATURE_AIR_MAX_0_05M = "TEMPERATURE_AIR_MAX_0_05M"
    TEMPERATURE_AIR_MIN_2M = "TEMPERATURE_AIR_MIN_2M"
    TEMPERATURE_AIR_MIN_0_05M = "TEMPERATURE_AIR_MIN_0_05M"

    TEMPERATURE_AIR_MIN_0_05M_LAST_12H = "TEMPERATURE_AIR_MIN_0_05M_LAST_12H"
    TEMPERATURE_AIR_MIN_2M_MULTIDAY = "TEMPERATURE_AIR_MIN_2M_MULTIDAY"
    TEMPERATURE_AIR_MAX_2M_MULTIDAY = "TEMPERATURE_AIR_MAX_2M_MULTIDAY"

    TEMPERATURE_WATER_EVAPORATION_PAN_MIN = "TEMPERATURE_WATER_EVAPORATION_PAN_MIN"
    TEMPERATURE_WATER_EVAPORATION_PAN_MAX = "TEMPERATURE_WATER_EVAPORATION_PAN_MAX"

    TEMPERATURE_CONCRETE_MIN_0M = "TEMPERATURE_CONCRETE_MIN_0M"
    TEMPERATURE_CONCRETE_MEAN_0M = "TEMPERATURE_CONCRETE_MEAN_0M"
    TEMPERATURE_CONCRETE_MAX_0M = "TEMPERATURE_CONCRETE_MAX_0M"

    TEMPERATURE_SOIL_MIN_0_1M = "TEMPERATURE_SOIL_MIN_0_1M"
    TEMPERATURE_SOIL_MIN_0_2M = "TEMPERATURE_SOIL_MIN_0_2M"
    TEMPERATURE_SOIL_MIN_0_5M = "TEMPERATURE_SOIL_MIN_0_5M"
    TEMPERATURE_SOIL_MIN_1M = "TEMPERATURE_SOIL_MIN_1M"
    TEMPERATURE_SOIL_MIN_2M = "TEMPERATURE_SOIL_MIN_2M"

    TEMPERATURE_SOIL_MAX_0_1M = "TEMPERATURE_SOIL_MAX_0_1M"
    TEMPERATURE_SOIL_MAX_0_2M = "TEMPERATURE_SOIL_MAX_0_2M"
    TEMPERATURE_SOIL_MAX_0_5M = "TEMPERATURE_SOIL_MAX_0_5M"
    TEMPERATURE_SOIL_MAX_1M = "TEMPERATURE_SOIL_MAX_1M"
    TEMPERATURE_SOIL_MAX_2M = "TEMPERATURE_SOIL_MAX_2M"

    TEMPERATURE_SOIL_MIN_UNKNOWN_0_05M = "TEMPERATURE_SOIL_MIN_UNKNOWN_0_05M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_0_1M = "TEMPERATURE_SOIL_MIN_UNKNOWN_0_1M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_0_2M = "TEMPERATURE_SOIL_MIN_UNKNOWN_0_2M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_0_5M = "TEMPERATURE_SOIL_MIN_UNKNOWN_0_5M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_1M = "TEMPERATURE_SOIL_MIN_UNKNOWN_1M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_1_5M = "TEMPERATURE_SOIL_MIN_UNKNOWN_1_5M"
    TEMPERATURE_SOIL_MIN_UNKNOWN_1_8M = "TEMPERATURE_SOIL_MIN_UNKNOWN_1_8M"

    TEMPERATURE_SOIL_MAX_UNKNOWN_0_05M = "TEMPERATURE_SOIL_MAX_UNKNOWN_0_05M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_0_1M = "TEMPERATURE_SOIL_MAX_UNKNOWN_0_1M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_0_2M = "TEMPERATURE_SOIL_MAX_UNKNOWN_0_2M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_0_5M = "TEMPERATURE_SOIL_MAX_UNKNOWN_0_5M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_1M = "TEMPERATURE_SOIL_MAX_UNKNOWN_1M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_1_5M = "TEMPERATURE_SOIL_MAX_UNKNOWN_1_5M"
    TEMPERATURE_SOIL_MAX_UNKNOWN_1_8M = "TEMPERATURE_SOIL_MAX_UNKNOWN_1_8M"

    # 1 - grass
    TEMPERATURE_SOIL_MIN_GRASS_0_05M = "TEMPERATURE_SOIL_MIN_GRASS_0_05M"
    TEMPERATURE_SOIL_MIN_GRASS_0_1M = "TEMPERATURE_SOIL_MIN_GRASS_0_1M"
    TEMPERATURE_SOIL_MIN_GRASS_0_2M = "TEMPERATURE_SOIL_MIN_GRASS_0_2M"
    TEMPERATURE_SOIL_MIN_GRASS_0_5M = "TEMPERATURE_SOIL_MIN_GRASS_0_5M"
    TEMPERATURE_SOIL_MIN_GRASS_1M = "TEMPERATURE_SOIL_MIN_GRASS_1M"
    TEMPERATURE_SOIL_MIN_GRASS_1_5M = "TEMPERATURE_SOIL_MIN_GRASS_1_5M"
    TEMPERATURE_SOIL_MIN_GRASS_1_8M = "TEMPERATURE_SOIL_MIN_GRASS_1_8M"

    TEMPERATURE_SOIL_MAX_GRASS_0_05M = "TEMPERATURE_SOIL_MAX_GRASS_0_05M"
    TEMPERATURE_SOIL_MAX_GRASS_0_1M = "TEMPERATURE_SOIL_MAX_GRASS_0_1M"
    TEMPERATURE_SOIL_MAX_GRASS_0_2M = "TEMPERATURE_SOIL_MAX_GRASS_0_2M"
    TEMPERATURE_SOIL_MAX_GRASS_0_5M = "TEMPERATURE_SOIL_MAX_GRASS_0_5M"
    TEMPERATURE_SOIL_MAX_GRASS_1M = "TEMPERATURE_SOIL_MAX_GRASS_1M"
    TEMPERATURE_SOIL_MAX_GRASS_1_5M = "TEMPERATURE_SOIL_MAX_GRASS_1_5M"
    TEMPERATURE_SOIL_MAX_GRASS_1_8M = "TEMPERATURE_SOIL_MAX_GRASS_1_8M"

    # 2 - fallow
    TEMPERATURE_SOIL_MIN_FALLOW_0_05M = "TEMPERATURE_SOIL_MIN_FALLOW_0_05M"
    TEMPERATURE_SOIL_MIN_FALLOW_0_1M = "TEMPERATURE_SOIL_MIN_FALLOW_0_1M"
    TEMPERATURE_SOIL_MIN_FALLOW_0_2M = "TEMPERATURE_SOIL_MIN_FALLOW_0_2M"
    TEMPERATURE_SOIL_MIN_FALLOW_0_5M = "TEMPERATURE_SOIL_MIN_FALLOW_0_5M"
    TEMPERATURE_SOIL_MIN_FALLOW_1M = "TEMPERATURE_SOIL_MIN_FALLOW_1M"
    TEMPERATURE_SOIL_MIN_FALLOW_1_5M = "TEMPERATURE_SOIL_MIN_FALLOW_1_5M"
    TEMPERATURE_SOIL_MIN_FALLOW_1_8M = "TEMPERATURE_SOIL_MIN_FALLOW_1_8M"

    TEMPERATURE_SOIL_MAX_FALLOW_0_05M = "TEMPERATURE_SOIL_MAX_FALLOW_0_05M"
    TEMPERATURE_SOIL_MAX_FALLOW_0_1M = "TEMPERATURE_SOIL_MAX_FALLOW_0_1M"
    TEMPERATURE_SOIL_MAX_FALLOW_0_2M = "TEMPERATURE_SOIL_MAX_FALLOW_0_2M"
    TEMPERATURE_SOIL_MAX_FALLOW_0_5M = "TEMPERATURE_SOIL_MAX_FALLOW_0_5M"
    TEMPERATURE_SOIL_MAX_FALLOW_1M = "TEMPERATURE_SOIL_MAX_FALLOW_1M"
    TEMPERATURE_SOIL_MAX_FALLOW_1_5M = "TEMPERATURE_SOIL_MAX_FALLOW_1_5M"
    TEMPERATURE_SOIL_MAX_FALLOW_1_8M = "TEMPERATURE_SOIL_MAX_FALLOW_1_8M"

    # 3 - bare ground
    TEMPERATURE_SOIL_MIN_BARE_GROUND_0_05M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_0_05M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_0_1M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_0_1M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_0_2M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_0_2M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_0_5M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_0_5M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_1M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_1M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_1_5M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_1_5M"
    TEMPERATURE_SOIL_MIN_BARE_GROUND_1_8M = "TEMPERATURE_SOIL_MIN_BARE_GROUND_1_8M"

    TEMPERATURE_SOIL_MAX_BARE_GROUND_0_05M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_0_05M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_0_1M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_0_1M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_0_2M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_0_2M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_0_5M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_0_5M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_1M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_1M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_1_5M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_1_5M"
    TEMPERATURE_SOIL_MAX_BARE_GROUND_1_8M = "TEMPERATURE_SOIL_MAX_BARE_GROUND_1_8M"

    # 4 - brome grass
    TEMPERATURE_SOIL_MIN_BROME_GRASS_0_05M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_0_05M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_0_1M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_0_1M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_0_2M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_0_2M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_0_5M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_0_5M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_1M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_1M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_1_5M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_1_5M"
    TEMPERATURE_SOIL_MIN_BROME_GRASS_1_8M = "TEMPERATURE_SOIL_MIN_BROME_GRASS_1_8M"

    TEMPERATURE_SOIL_MAX_BROME_GRASS_0_05M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_0_05M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_0_1M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_0_1M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_0_2M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_0_2M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_0_5M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_0_5M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_1M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_1M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_1_5M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_1_5M"
    TEMPERATURE_SOIL_MAX_BROME_GRASS_1_8M = "TEMPERATURE_SOIL_MAX_BROME_GRASS_1_8M"

    # 5 - sod
    TEMPERATURE_SOIL_MIN_SOD_0_05M = "TEMPERATURE_SOIL_MIN_SOD_0_05M"
    TEMPERATURE_SOIL_MIN_SOD_0_1M = "TEMPERATURE_SOIL_MIN_SOD_0_1M"
    TEMPERATURE_SOIL_MIN_SOD_0_2M = "TEMPERATURE_SOIL_MIN_SOD_0_2M"
    TEMPERATURE_SOIL_MIN_SOD_0_5M = "TEMPERATURE_SOIL_MIN_SOD_0_5M"
    TEMPERATURE_SOIL_MIN_SOD_1M = "TEMPERATURE_SOIL_MIN_SOD_1M"
    TEMPERATURE_SOIL_MIN_SOD_1_5M = "TEMPERATURE_SOIL_MIN_SOD_1_5M"
    TEMPERATURE_SOIL_MIN_SOD_1_8M = "TEMPERATURE_SOIL_MIN_SOD_1_8M"

    TEMPERATURE_SOIL_MAX_SOD_0_05M = "TEMPERATURE_SOIL_MAX_SOD_0_05M"
    TEMPERATURE_SOIL_MAX_SOD_0_1M = "TEMPERATURE_SOIL_MAX_SOD_0_1M"
    TEMPERATURE_SOIL_MAX_SOD_0_2M = "TEMPERATURE_SOIL_MAX_SOD_0_2M"
    TEMPERATURE_SOIL_MAX_SOD_0_5M = "TEMPERATURE_SOIL_MAX_SOD_0_5M"
    TEMPERATURE_SOIL_MAX_SOD_1M = "TEMPERATURE_SOIL_MAX_SOD_1M"
    TEMPERATURE_SOIL_MAX_SOD_1_5M = "TEMPERATURE_SOIL_MAX_SOD_1_5M"
    TEMPERATURE_SOIL_MAX_SOD_1_8M = "TEMPERATURE_SOIL_MAX_SOD_1_8M"

    # 6 - straw mulch
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_05M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_05M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_1M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_1M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_2M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_2M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_5M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_0_5M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_1M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_1M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_1_5M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_1_5M"
    TEMPERATURE_SOIL_MIN_STRAW_MULCH_1_8M = "TEMPERATURE_SOIL_MIN_STRAW_MULCH_1_8M"

    TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_05M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_05M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_1M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_1M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_2M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_2M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_5M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_0_5M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_1M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_1M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_1_5M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_1_5M"
    TEMPERATURE_SOIL_MAX_STRAW_MULCH_1_8M = "TEMPERATURE_SOIL_MAX_STRAW_MULCH_1_8M"

    # 7 - grass muck
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_05M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_05M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_1M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_1M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_2M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_2M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_5M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_0_5M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_1M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_1M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_1_5M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_1_5M"
    TEMPERATURE_SOIL_MIN_GRASS_MUCK_1_8M = "TEMPERATURE_SOIL_MIN_GRASS_MUCK_1_8M"

    TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_05M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_05M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_1M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_1M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_2M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_2M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_5M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_0_5M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_1M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_1M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_1_5M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_1_5M"
    TEMPERATURE_SOIL_MAX_GRASS_MUCK_1_8M = "TEMPERATURE_SOIL_MAX_GRASS_MUCK_1_8M"

    # 8 - bare muck
    TEMPERATURE_SOIL_MIN_BARE_MUCK_0_05M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_0_05M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_0_1M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_0_1M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_0_2M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_0_2M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_0_5M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_0_5M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_1M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_1M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_1_5M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_1_5M"
    TEMPERATURE_SOIL_MIN_BARE_MUCK_1_8M = "TEMPERATURE_SOIL_MIN_BARE_MUCK_1_8M"

    TEMPERATURE_SOIL_MAX_BARE_MUCK_0_05M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_0_05M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_0_1M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_0_1M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_0_2M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_0_2M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_0_5M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_0_5M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_1M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_1M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_1_5M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_1_5M"
    TEMPERATURE_SOIL_MAX_BARE_MUCK_1_8M = "TEMPERATURE_SOIL_MAX_BARE_MUCK_1_8M"

    TEMPERATURE_SURFACE_MEAN = "TEMPERATURE_SURFACE_MEAN"
    # ---- count ----
    COUNT_DAYS_HEATING_DEGREE = "COUNT_DAYS_HEATING_DEGREE"
    COUNT_DAYS_COOLING_DEGREE = "COUNT_DAYS_COOLING_DEGREE"
    # Number of days included in the multiday minimum temperature
    COUNT_DAYS_MULTIDAY_TEMPERATURE_AIR_MIN_2M = "COUNT_DAYS_MULTIDAY_TEMPERATURE_AIR_MIN_2M"
    # Number of days included in the multiday maximum temperature
    COUNT_DAYS_MULTIDAY_TEMPERATURE_AIR_MAX_2M = "COUNT_DAYS_MULTIDAY_TEMPERATURE_AIR_MAX_2M"
    # ---- error ----
    ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M = "ERROR_ABSOLUTE_TEMPERATURE_AIR_MEAN_2M"
    ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M = "ERROR_ABSOLUTE_TEMPERATURE_DEW_POINT_MEAN_2M"

    # VISIBILITY
    # ---- distance ----
    VISIBILITY_RANGE_INDEX = "VISIBILITY_RANGE_INDEX"
    VISIBILITY_RANGE = "VISIBILITY_RANGE"  # through clouds, fog, etc
    # ---- probability ----
    PROBABILITY_VISIBILITY_BELOW_1000M = "PROBABILITY_VISIBILITY_BELOW_1000M"

    # WATER EQUIVALENT
    # ---- aggregated ----
    WATER_EQUIVALENT_SNOW_DEPTH = "WATER_EQUIVALENT_SNOW_DEPTH"
    WATER_EQUIVALENT_SNOW_DEPTH_NEW = "WATER_EQUIVALENT_SNOW_DEPTH_NEW"
    WATER_EQUIVALENT_SNOW_DEPTH_EXCELLED = "WATER_EQUIVALENT_SNOW_DEPTH_EXCELLED"
    WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H = "WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_1H"
    WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H = "WATER_EQUIVALENT_SNOW_DEPTH_NEW_LAST_3H"

    # WEATHER
    # ---- aggregated ----
    WEATHER_SIGNIFICANT = "WEATHER_SIGNIFICANT"
    WEATHER_SIGNIFICANT_LAST_3H = "WEATHER_SIGNIFICANT_LAST_3H"
    WEATHER_LAST_6H = "WEATHER_LAST_6H"
    WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H = "WEATHER_SIGNIFICANT_OPTIONAL_LAST_1H"
    WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H = "WEATHER_SIGNIFICANT_OPTIONAL_LAST_3H"
    WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H = "WEATHER_SIGNIFICANT_OPTIONAL_LAST_6H"
    WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H = "WEATHER_SIGNIFICANT_OPTIONAL_LAST_12H"
    WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H = "WEATHER_SIGNIFICANT_OPTIONAL_LAST_24H"
    # TODO: ECCC parameter, same as significant weather?
    WEATHER = "WEATHER"
    WEATHER_TEXT = "WEATHER_TEXT"

    # WEATHER PHENOMENA
    # ---- averaged ----
    WEATHER_TYPE_FOG = "WEATHER_TYPE_FOG"
    WEATHER_TYPE_HEAVY_FOG = "WEATHER_TYPE_HEAVY_FOG"
    WEATHER_TYPE_THUNDER = "WEATHER_TYPE_THUNDER"
    WEATHER_TYPE_ICE_SLEET_SNOW_HAIL = "WEATHER_TYPE_ICE_SLEET_SNOW_HAIL"
    WEATHER_TYPE_HAIL = "WEATHER_TYPE_HAIL"
    WEATHER_TYPE_GLAZE_RIME = "WEATHER_TYPE_GLAZE_RIME"
    WEATHER_TYPE_DUST_ASH_SAND = "WEATHER_TYPE_DUST_ASH_SAND"
    WEATHER_TYPE_SMOKE_HAZE = "WEATHER_TYPE_SMOKE_HAZE"
    WEATHER_TYPE_BLOWING_DRIFTING_SNOW = "WEATHER_TYPE_BLOWING_DRIFTING_SNOW"
    WEATHER_TYPE_TORNADO_WATERSPOUT = "WEATHER_TYPE_TORNADO_WATERSPOUT"
    WEATHER_TYPE_HIGH_DAMAGING_WINDS = "WEATHER_TYPE_HIGH_DAMAGING_WINDS"
    WEATHER_TYPE_BLOWING_SPRAY = "WEATHER_TYPE_BLOWING_SPRAY"
    WEATHER_TYPE_MIST = "WEATHER_TYPE_MIST"
    WEATHER_TYPE_DRIZZLE = "WEATHER_TYPE_DRIZZLE"
    WEATHER_TYPE_FREEZING_DRIZZLE = "WEATHER_TYPE_FREEZING_DRIZZLE"
    WEATHER_TYPE_RAIN = "WEATHER_TYPE_RAIN"
    WEATHER_TYPE_FREEZING_RAIN = "WEATHER_TYPE_FREEZING_RAIN"
    WEATHER_TYPE_SNOW_PELLETS_SNOW_GRAINS_ICE_CRYSTALS = "WEATHER_TYPE_SNOW_PELLETS_SNOW_GRAINS_ICE_CRYSTALS"
    WEATHER_TYPE_PRECIPITATION_UNKNOWN_SOURCE = "WEATHER_TYPE_PRECIPITATION_UNKNOWN_SOURCE"
    WEATHER_TYPE_GROUND_FOG = "WEATHER_TYPE_GROUND_FOG"
    WEATHER_TYPE_ICE_FOG_FREEZING_FOG = "WEATHER_TYPE_ICE_FOG_FREEZING_FOG"

    WEATHER_TYPE_VICINITY_FOG_ANY = "WEATHER_TYPE_VICINITY_FOG_ANY"
    WEATHER_TYPE_VICINITY_THUNDER = "WEATHER_TYPE_VICINITY_THUNDER"
    WEATHER_TYPE_VICINITY_DUST_ASH_SAND = "WEATHER_TYPE_VICINITY_DUST_ASH_SAND"
    WEATHER_TYPE_VICINITY_SNOW_ICE_CRYSTALS = "WEATHER_TYPE_VICINITY_SNOW_ICE_CRYSTALS"
    WEATHER_TYPE_VICINITY_RAIN_SNOW_SHOWER = "WEATHER_TYPE_VICINITY_RAIN_SNOW_SHOWER"
    # ---- count ----
    # the following are coming from DWD Observations
    COUNT_WEATHER_TYPE_FOG = "COUNT_FOG"
    COUNT_WEATHER_TYPE_THUNDER = "COUNT_THUNDER"
    COUNT_WEATHER_TYPE_STORM_STRONG_WIND = "COUNT_STORM_STRONG_WIND"
    COUNT_WEATHER_TYPE_STORM_STORMIER_WIND = "COUNT_STORM_STORMIER_WIND"
    COUNT_WEATHER_TYPE_DEW = "COUNT_DEW"
    COUNT_WEATHER_TYPE_GLAZE = "COUNT_GLAZE"
    COUNT_WEATHER_TYPE_RIPE = "COUNT_RIPE"
    COUNT_WEATHER_TYPE_SLEET = "COUNT_SLEET"
    COUNT_WEATHER_TYPE_HAIL = "COUNT_HAIL"
    # ---- probability ----
    PROBABILITY_THUNDER_LAST_1H = "PROBABILITY_THUNDER_LAST_1H"
    PROBABILITY_THUNDER_LAST_6H = "PROBABILITY_THUNDER_LAST_6H"
    PROBABILITY_THUNDER_LAST_12H = "PROBABILITY_THUNDER_LAST_12H"
    PROBABILITY_THUNDER_LAST_24H = "PROBABILITY_THUNDER_LAST_24H"

    # WIND_SPEED
    # ---- averaged ----
    WIND_SPEED = "WIND_SPEED"
    WIND_DIRECTION = "WIND_DIRECTION"
    WIND_FORCE_BEAUFORT = "WIND_FORCE_BEAUFORT"
    WIND_SPEED_ARITHMETIC = "WIND_SPEED_ARITHMETIC"
    # ---- extremes ----
    WIND_GUST_MAX = "WIND_GUST_MAX"
    WIND_GUST_MAX_LAST_1H = "WIND_GUST_MAX_LAST_1H"
    WIND_GUST_MAX_LAST_3H = "WIND_GUST_MAX_LAST_3H"
    WIND_GUST_MAX_LAST_6H = "WIND_GUST_MAX_LAST_6H"
    WIND_GUST_MAX_LAST_12H = "WIND_GUST_MAX_LAST_12H"
    WIND_SPEED_MIN = "WIND_SPEED_MIN"
    WIND_SPEED_ROLLING_MEAN_MAX = "WIND_SPEED_ROLLING_MEAN_MAX"
    WIND_DIRECTION_GUST_MAX = "WIND_DIRECTION_GUST_MAX"

    WIND_GUST_MAX_5SEC = "WIND_GUST_MAX_5SEC"
    WIND_GUST_MAX_1MIN = "WIND_GUST_MAX_1MIN"
    WIND_GUST_MAX_2MIN = "WIND_GUST_MAX_2MIN"
    WIND_GUST_MAX_INSTANT = "WIND_GUST_MAX_INSTANT"
    WIND_GUST_MAX_1MILE = "WIND_GUST_MAX_1MILE"

    WIND_DIRECTION_GUST_MAX_5SEC = "WIND_DIRECTION_GUST_MAX_5SEC"
    WIND_DIRECTION_GUST_MAX_1MIN = "WIND_DIRECTION_GUST_MAX_1MIN"
    WIND_DIRECTION_GUST_MAX_2MIN = "WIND_DIRECTION_GUST_MAX_2MIN"
    WIND_DIRECTION_GUST_MAX_INSTANT = "WIND_DIRECTION_GUST_MAX_INSTANT"
    WIND_DIRECTION_GUST_MAX_1MILE = "WIND_DIRECTION_GUST_MAX_1MILE"
    # ---- probability ----
    PROBABILITY_WIND_GUST_GE_25KN_LAST_6H = "PROBABILITY_WIND_GUST_GE_25KN_LAST_6H"
    PROBABILITY_WIND_GUST_GE_25KN_LAST_12H = "PROBABILITY_WIND_GUST_GE_25KN_LAST_12H"
    PROBABILITY_WIND_GUST_GE_40KN_LAST_6H = "PROBABILITY_WIND_GUST_GE_40KN_LAST_6H"
    PROBABILITY_WIND_GUST_GE_40KN_LAST_12H = "PROBABILITY_WIND_GUST_GE_40KN_LAST_12H"
    PROBABILITY_WIND_GUST_GE_55KN_LAST_6H = "PROBABILITY_WIND_GUST_GE_55KN_LAST_6H"
    PROBABILITY_WIND_GUST_GE_55KN_LAST_12H = "PROBABILITY_WIND_GUST_GE_55KN_LAST_12H"
    # ---- count ----
    # Number of days included in the multiday wind movement
    COUNT_DAYS_MULTIDAY_WIND_MOVEMENT = "COUNT_DAYS_MULTIDAY_WIND_MOVEMENT"
    # ---- time ----
    TIME_WIND_GUST_MAX_1MILE_OR_1MIN = "TIME_WIND_GUST_MAX_1MILE_OR_1MIN"
    TIME_WIND_GUST_MAX = "TIME_WIND_GUST_MAX"
    # ---- distance ----
    WIND_MOVEMENT_MULTIDAY = "WIND_MOVEMENT_MULTIDAY"
    WIND_MOVEMENT_24H = "WIND_MOVEMENT_24H"
    # ---- error ----
    ERROR_ABSOLUTE_WIND_SPEED = "ERROR_ABSOLUTE_WIND_SPEED"
    ERROR_ABSOLUTE_WIND_DIRECTION = "ERROR_ABSOLUTE_WIND_DIRECTION"

    # Waterways related parameters
    TEMPERATURE_WATER = "TEMPERATURE_WATER"
    TEMPERATURE_WATER_MAX = "TEMPERATURE_WATER_MAX"
    TEMPERATURE_WATER_MEAN = "TEMPERATURE_WATER_MEAN"
    TEMPERATURE_WATER_MIN = "TEMPERATURE_WATER_MIN"
    ELECTRIC_CONDUCTIVITY = "ELECTRIC_CONDUCTIVITY"
    GROUNDWATER_LEVEL = "GROUNDWATER_LEVEL"
    GROUNDWATER_LEVEL_MAX = "groundwater_level_max"
    GROUNDWATER_LEVEL_MIN = "groundwater_level_min"
    DISCHARGE = "DISCHARGE"
    DISCHARGE_MAX = "DISCHARGE_MAX"
    DISCHARGE_MEAN = "DISCHARGE_MEAN"
    DISCHARGE_MIN = "DISCHARGE_MIN"
    FLOW_SPEED = "FLOW_SPEED"
    OXYGEN_LEVEL = "OXYGEN_LEVEL"
    TURBIDITY = "TURBIDITY"
    PRECIPITATION_INTENSITY = "PRECIPITATION_INTENSITY"
    WAVE_PERIOD = "WAVE_PERIOD"
    CLEARANCE_HEIGHT = "CLEARANCE_HEIGHT"
    CURRENT = "CURRENT"
    WAVE_HEIGHT_SIGN = "WAVE_HEIGHT_SIGN"
    WAVE_HEIGHT_MAX = "WAVE_HEIGHT_MAX"
    PH_VALUE = "PH_VALUE"
    CHLORID_CONCENTRATION = "CHLORID_CONCENTRATION"
    STAGE = "STAGE"
    STAGE_MAX = "STAGE_MAX"
    STAGE_MEAN = "STAGE_MEAN"
    STAGE_MIN = "STAGE_MIN"
    WATER_FILM_THICKNESS = "WATER_FILM_THICKNESS"
    ROAD_SURFACE_CONDITION = "ROAD_SURFACE_CONDITION"
