from functools import reduce
from typing import Optional, Any

import tinycss2

from .types import Result, Specificity, Level, Position, SpecificityContributingPart

__all__ = ["calculate", "calculate_with_details"]

# Constant for universal selector.
UNIVERSAL_SELECTOR_NAME = "*"

PSEUDO_ELEMENTS_WITH_DEPRECATED_ONE_COLON = [
    "before",
    "after",
    "first-line",
    "first-letter",
]


def increment(node: dict[str, Any], result: Result, level: Level) -> Result:
    """
    Increments and updates the result with information about a specificity-contributing
    part derived from the provided node. It calculates start and end positions using
    line and column details from the node and creates a new specificity-contributing
    part for the specified level.

    Args:
        node (dict[str, Any]): The dictionary that contains information about the
            source location and details for calculating positions.
        result (Result): The current specificity result, consisting of existing
            totals and contributing parts.
        level (Level): The specificity level to update and associate the
            contributing part with.

    Returns:
        Result: The updated specificity results with the incremented total and the
        newly created specificity-contributing part added.
    """
    start_line = getattr(node.get('source_line', None), 'line', 1)
    start_column = getattr(node.get('source_column', None), 'column', 1)
    end_line = getattr(node.get('end_line', None), 'line', 1)
    end_column = getattr(node.get('end_column', None), 'column', 1)

    new_total = dict(result.total)
    new_total[level] = new_total[level] + 1

    new_part = SpecificityContributingPart(
        level=level,
        start=Position(line=start_line, column=start_column),
        end=Position(line=end_line, column=end_column)
    )

    return Result(
        total=new_total,
        contributing_parts=[*result.contributing_parts, new_part]
    )


def get_children_from_pseudo_class_node(node: dict[str, Any]) -> list[dict[str, Any]]:
    """
    Retrieves the children nodes from a pseudo-class node in a structured data tree.
    The function selectively extracts children based on specific conditions tied to
    the node type and its associated attributes.

    Args:
        node (dict[str, Any]): A dictionary representing a node in the structured data tree.

    Returns:
        list[dict[str, Any]]: A list of dictionaries representing child nodes
        extracted from the provided pseudo-class node. The selection of children
        depends on the node's properties.
    """
    if node.get('children'):
        first_child = node['children'][0]

        if first_child.get('type') == 'SelectorList':
            return first_child.get('children', [])
        elif first_child.get('type') == 'Nth' and first_child.get('selector'):
            return first_child['selector'].get('children', [])
        else:
            return node['children']
    return []


def handle_pseudo_class_selector(node: dict[str, Any], accumulating_result: Result) -> Result:
    """
    Handles specificity calculations for CSS pseudo-class selectors.

    This function processes a node representing a pseudo-class selector in a CSS
    document. Depending on the type of pseudo-class selector, it calculates and
    adjusts specificity using the provided accumulating result.

    Some selectors, such as `:is()`, `:not()`, and `:has()` replace their specificity
    with the specificity of the most specific child selector. Others like `:nth-child()`
    add specificity of their pseudo-class level plus the most specific child selector.
    Special selectors such as `:where()` always have specificity of zero.

    Args:
        node (dict[str, Any]): A dictionary representation of the pseudo-class
            selector. It may contain the name and children of the pseudo-class.
        accumulating_result (Result): The current accumulated specificity. This
            will be updated based on the specific pseudo-class and its children.

    Returns:
        Result: The updated specificity result after processing the given
        pseudo-class selector and any children it may have.
    """
    name = node.get('name', '').lower()
    children = get_children_from_pseudo_class_node(node)

    if name in ('not', 'is', 'has'):
        # The specificity of an :is(), :not(), or :has() pseudo-class is replaced by the specificity of the most
        # specific complex selector in its selector list argument.
        if children:
            results = [traverse(child, accumulating_result) for child in children]

            return sorted(results, key=lambda x: x.total, reverse=True)[0]
    elif name in ('nth-child', 'nth-last-child'):
        # The specificity of an :nth-child() or :nth-last-child() selector is the specificity of the pseudo class
        # itself (counting as one pseudo-class selector) plus the specificity of the most specific complex selector
        # in its selector list argument (if any).
        if children:
            incremented_result = increment(node, accumulating_result, Level.B)
            results = [traverse(child, incremented_result) for child in children]

            return sorted(results, key=lambda x: x.total, reverse=True)[0]
        else:
            return increment(node, accumulating_result, Level.B)
    elif name == 'where':
        # The specificity of a :where() pseudo-class is replaced by zero.
        return accumulating_result
    elif name in ('global', 'local'):
        # The specificity for :global() and :local() is replaced by the specificity of the child selector because
        # although they look like pseudo classes, they are actually an identifier for CSS Modules.
        if children:
            return reduce(
                lambda acc, child_node: traverse(child_node, acc),
                children,
                accumulating_result
            )
    elif name in PSEUDO_ELEMENTS_WITH_DEPRECATED_ONE_COLON:
        # These pseudo-elements can look like pseudo-classes
        # https://www.w3.org/TR/selectors-4/#pseudo-elements
        return increment(node, accumulating_result, Level.C)
    else:
        return increment(node, accumulating_result, Level.B)

    return accumulating_result


def traverse(node: dict[str, Any], accumulating_result: Optional[Result] = None) -> Result:
    """
    Traverses a node structure recursively, analyzing and accumulating results based
    on the node type. This function is designed to support various selector types
    and processes them accordingly to update the accumulated result.

    Args:
        node (dict[str, Any]): The current node to process, represented as a dictionary.
        accumulating_result (Optional[Result]): The result object that accumulates processing
            results across nodes. Defaults to None, which initializes a new `Result` object.

    Returns:
        Result: The accumulated result after processing the current node and its children.
    """
    if accumulating_result is None:
        accumulating_result = Result(
            total={Level.A: 0, Level.B: 0, Level.C: 0},
            contributing_parts=[]
        )

    node_type = node.get('type')

    if node_type == 'IdSelector':
        return increment(node, accumulating_result, Level.A)
    elif node_type == 'PseudoClassSelector':
        return handle_pseudo_class_selector(node, accumulating_result)
    elif node_type in ('ClassSelector', 'AttributeSelector'):
        return increment(node, accumulating_result, Level.B)
    elif node_type == 'TypeSelector':
        if node.get('name') != UNIVERSAL_SELECTOR_NAME:
            return increment(node, accumulating_result, Level.C)
    elif node_type == 'PseudoElementSelector':
        return increment(node, accumulating_result, Level.C)
    elif node_type in ('Selector', 'SelectorList') and node.get('children'):
        result = accumulating_result

        for child in node['children']:
            result = traverse(child, result)

        return result
    elif node_type == 'Raw':
        # Convertion for tinycss2
        tokens = tinycss2.parse_component_value_list(node.get('value', ''))
        parsed_node = convert_tinycss2_to_ast(tokens, node)

        return traverse(parsed_node, accumulating_result)

    return accumulating_result


def convert_tinycss2_to_ast(tokens, original_node=None):
    """
    Converts tokens from tinycss2 to a corresponding Abstract Syntax Tree (AST)
    format. This function processes given tokens and transforms them into a
    structured format compatible with AST traversal required for further usage.

    Args:
        tokens: The list of tokens to be converted into the AST format.
        original_node: Optional; The original node from the source that can be
            referenced or utilized during the conversion process. Defaults to None.

    Returns:
        dict: A dictionary representing the AST, specifically a SelectorList node
        containing children nodes derived from the provided tokens.
    """
    # This function should be completed to convert tinycss2 tokens into a format
    # compatible with AST traversal - simplified, for example
    selector_list = {'type': 'SelectorList', 'children': []}

    # TODO: Convertion logic
    #   ...

    return selector_list


def calculate(selector: str) -> Specificity:
    """
    Calculates the specificity of a given CSS selector.

    This function parses a CSS selector into its component tokens, converts them
    into an abstract syntax tree (AST), and then calculates the total specificity
    by traversing the AST.

    Args:
        selector (str): The CSS selector string for which specificity needs to be
            calculated.

    Returns:
        Specificity: An object representing the total specificity calculated.
    """
    tokens = tinycss2.parse_component_value_list(selector)
    ast = convert_tinycss2_to_ast(tokens)

    return traverse(ast).total


def calculate_with_details(selector: str) -> Result:
    """
    Parses and evaluates a CSS selector string to compute the result using a detailed
    AST-based approach. The function processes the selector into tokens, transforms it
    into an abstract syntax tree (AST), and traverses the AST to compute the final output.

    Args:
        selector (str): A CSS selector string to be parsed and evaluated.

    Returns:
        Result: The computed result obtained after parsing and evaluating the given
        selector string.
    """
    tokens = tinycss2.parse_component_value_list(selector)
    ast = convert_tinycss2_to_ast(tokens)

    return traverse(ast)
