# Contains code to transform the AST into files and C++ code, but not actually run them.
import ast
from collections import namedtuple
import os
import pickle
from typing import Iterable
from func_adl.ast import ast_hash
from func_adl.ast.func_adl_ast_utils import is_call_of
from func_adl_xAOD.backend.xAODlib.atlas_xaod_executor import atlas_xaod_executor
from func_adl_xAOD.backend.util_LINQ import find_dataset, extract_dataset_info


class CacheExeException(BaseException):
    def __init__(self, message):
        BaseException.__init__(self, message)


# Return info
HashXAODExecutorInfo = namedtuple('HashXAODExecutorInfo', 'hash main_script treename output_filename, filelist')


def _build_result(cache: tuple, url_list: Iterable[str]) -> HashXAODExecutorInfo:
    'Helper routine to build out a full result'
    return HashXAODExecutorInfo(cache[0], cache[1], cache[2], cache[3], url_list)


def use_executor_xaod_hash_cache(a: ast.AST, cache_path: str) -> HashXAODExecutorInfo:
    r'''Write out the C++ code and supporting files to a cache

    Arguments:
        a           The ast that will be transformed
        cache_path  Path the cache directory. We will write everything out in there.

    Returns:
        HashXAODExecutorInfo    Named tuple with the hash and the list of files in it.
    '''
    # We can only do this if the result is going to be a ROOT file(s). So make sure.
    if not is_call_of(a, 'ResultTTree'):
        raise CacheExeException(f'Can only cache results for a ROOT tree, not for {type(a).__name__} - {ast.dump(a)} (that should have been a call to ResultTTree).')

    # Calculate the AST hash. If this is already around then we don't need to do very much!
    hash = ast_hash.calc_ast_hash(a)

    # Next, see if the hash file is there.
    query_file_path = os.path.join(cache_path, hash)
    cache_file = os.path.join(query_file_path, 'rep_cache.pickle')
    if os.path.isfile(cache_file):
        # We have a cache hit. Look it up.
        file = find_dataset(a)
        with open(cache_file, 'rb') as f:
            result_cache = pickle.load(f)
            return _build_result(result_cache, extract_dataset_info(file))

    # Create the files to run in that location.
    if not os.path.exists(query_file_path):
        os.makedirs(query_file_path)
    exe = atlas_xaod_executor()
    f_spec = exe.write_cpp_files(exe.apply_ast_transformations(a), query_file_path)

    # Write out the basic info for the result rep and the runner into that location.
    result_cache = (hash, f_spec.main_script, f_spec.result_rep.treename, f_spec.result_rep.filename)
    with open(os.path.join(query_file_path, 'rep_cache.pickle'), 'wb') as f:
        pickle.dump(result_cache, f)

    return _build_result(result_cache, f_spec.input_urls)
