#!/usr/bin/env python3

"""
Attempt to determine the parent branch of a commit by tracking down
the branch commit history and looking for other branches that share
the same commit. Can optionally ignore feature branches and/or report
the distance to the potential parent.
"""

import argparse
import sys

import thingy.git as git
import thingy.colour as colour

################################################################################

def main():
    """ Main function """

    parser = argparse.ArgumentParser(description='Attempt to determine the parent branch for the specified branch (defaulting to the current one)')
    parser.add_argument('-a', '--all', action='store_true', help='Include feature branches as possible parents')
    parser.add_argument('-v', '--verbose', action='store_true', help='Report verbose results (includes number of commits between branch and parent)')
    parser.add_argument('branch', action='store', nargs='?', type=str, default='HEAD', help='Branch, commit or commit')

    args = parser.parse_args()

    try:
        if args.all:
            any_parents, any_distance = git.parents(args.branch)
        else:
            any_parents = []

        parents, distance = git.parents(args.branch, ignore='feature/*')

        # If we have feature and non-feature branch candidates, decide which to report
        # (one or both) based on distance.

        if parents and any_parents:
            if any_distance < distance:
                parents = any_parents
                distance = any_distance
            elif any_distance == distance:
                for more in any_parents:
                    if more not in parents:
                        parents.append(more)

    except git.GitError as exc:
        colour.error(f'[RED:ERROR:] {exc.msg}', status=exc.status)

    if parents:
        if args.verbose:
            if len(parents) == 1:
                colour.write(f'Parent branch [BLUE:{parents[0]}] is [BLUE:{distance}] commits away from [BLUE:{args.branch}]')
            else:
                colour.write(f'Parent branches [BLUE:%s] are [BLUE:{distance}] commits away from [BLUE:{args.branch}[' % (', '.join(parents)))
        else:
            print(', '.join(parents))
    else:
        colour.error('[RED:Could not determine parent branch]\n')

################################################################################

def git_parent():
    """Entry point"""

    try:
        main()
    except KeyboardInterrupt:
        sys.exit(1)
    except BrokenPipeError:
        sys.exit(2)

################################################################################

if __name__ == '__main__':
    git_parent()
