"use strict";
const os_1 = require("os");
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const s3 = require("../lib");
module.exports = {
    'default bucket'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'CFN properties are type-validated during resolution'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            bucketName: cdk.Token.asString(5),
        });
        test.throws(() => {
            assert_1.SynthUtils.synthesize(stack);
        }, /bucketName: 5 should be a string/);
        test.done();
    },
    'bucket without encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.UNENCRYPTED,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with managed encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'BucketEncryption': {
                            'ServerSideEncryptionConfiguration': [
                                {
                                    'ServerSideEncryptionByDefault': {
                                        'SSEAlgorithm': 'aws:kms',
                                    },
                                },
                            ],
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'valid bucket names'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'abc.xyz-34ab',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '124.pp--33',
        }));
        test.done();
    },
    'bucket validation skips tokenized values'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: cdk.Lazy.stringValue({ produce: () => '_BUCKET' }),
        }));
        test.done();
    },
    'fails with message on invalid bucket names'(test) {
        const stack = new cdk.Stack();
        const bucket = `-buckEt.-${new Array(65).join('$')}`;
        const expectedErrors = [
            `Invalid S3 bucket name (value: ${bucket})`,
            'Bucket name must be at least 3 and no more than 63 characters',
            'Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) (offset: 5)',
            'Bucket name must start and end with a lowercase character or number (offset: 0)',
            `Bucket name must start and end with a lowercase character or number (offset: ${bucket.length - 1})`,
            'Bucket name must not have dash next to period, or period next to dash, or consecutive periods (offset: 7)',
        ].join(os_1.EOL);
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: bucket,
        }), function (err) {
            return expectedErrors === err.message;
        });
        test.done();
    },
    'fails if bucket name has less than 3 or more than 63 characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'a',
        }), /at least 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: new Array(65).join('x'),
        }), /no more than 63/);
        test.done();
    },
    'fails if bucket name has invalid characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'b@cket',
        }), /offset: 1/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucKet',
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'bučket',
        }), /offset: 2/);
        test.done();
    },
    'fails if bucket name does not start or end with lowercase character or number'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '-ucket',
        }), /offset: 0/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucke.',
        }), /offset: 5/);
        test.done();
    },
    'fails only if bucket name has the consecutive symbols (..), (.-), (-.)'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'buc..ket',
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'buck.-et',
        }), /offset: 4/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'b-.ucket',
        }), /offset: 1/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: 'bu--cket',
        }));
        test.done();
    },
    'fails only if bucket name resembles IP address'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '1.2.3.4',
        }), /must not resemble an IP address/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '1.2.3',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: '1.2.3.a',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: '1000.2.3.4',
        }));
        test.done();
    },
    'fails if encryption key is used with managed encryption'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: myKey,
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'fails if encryption key is used with encryption set to unencrypted'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.UNENCRYPTED,
            encryptionKey: myKey,
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'encryptionKey can specify kms key'(test) {
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'MyKey', { description: 'hello, world' });
        new s3.Bucket(stack, 'MyBucket', { encryptionKey, encryption: s3.BucketEncryption.KMS });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyKey6AB29FA6': {
                    'Type': 'AWS::KMS::Key',
                    'Properties': {
                        'Description': 'hello, world',
                        'KeyPolicy': {
                            'Statement': [
                                {
                                    'Action': [
                                        'kms:Create*',
                                        'kms:Describe*',
                                        'kms:Enable*',
                                        'kms:List*',
                                        'kms:Put*',
                                        'kms:Update*',
                                        'kms:Revoke*',
                                        'kms:Disable*',
                                        'kms:Get*',
                                        'kms:Delete*',
                                        'kms:ScheduleKeyDeletion',
                                        'kms:CancelKeyDeletion',
                                        'kms:GenerateDataKey',
                                        'kms:TagResource',
                                        'kms:UntagResource',
                                    ],
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        'Ref': 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        'Ref': 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    'Resource': '*',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'BucketEncryption': {
                            'ServerSideEncryptionConfiguration': [
                                {
                                    'ServerSideEncryptionByDefault': {
                                        'KMSMasterKeyID': {
                                            'Fn::GetAtt': [
                                                'MyKey6AB29FA6',
                                                'Arn',
                                            ],
                                        },
                                        'SSEAlgorithm': 'aws:kms',
                                    },
                                },
                            ],
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with versioning turned on'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            versioned: true,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'VersioningConfiguration': {
                            'Status': 'Enabled',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with block public access set to BlockAll'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'PublicAccessBlockConfiguration': {
                            'BlockPublicAcls': true,
                            'BlockPublicPolicy': true,
                            'IgnorePublicAcls': true,
                            'RestrictPublicBuckets': true,
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with block public access set to BlockAcls'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ACLS,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'PublicAccessBlockConfiguration': {
                            'BlockPublicAcls': true,
                            'IgnorePublicAcls': true,
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with custom block public access setting'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: new s3.BlockPublicAccess({ restrictPublicBuckets: true }),
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'PublicAccessBlockConfiguration': {
                            'RestrictPublicBuckets': true,
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'bucket with custom canned access control'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            accessControl: s3.BucketAccessControl.LOG_DELIVERY_WRITE,
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'Properties': {
                        'AccessControl': 'LogDeliveryWrite',
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'permissions': {
        'addPermission creates a bucket policy'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['foo'],
                actions: ['bar:baz'],
                principals: [new iam.AnyPrincipal()],
            }));
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyBucketF68F3FF0': {
                        'Type': 'AWS::S3::Bucket',
                        'DeletionPolicy': 'Retain',
                        'UpdateReplacePolicy': 'Retain',
                    },
                    'MyBucketPolicyE7FBAC7B': {
                        'Type': 'AWS::S3::BucketPolicy',
                        'Properties': {
                            'Bucket': {
                                'Ref': 'MyBucketF68F3FF0',
                            },
                            'PolicyDocument': {
                                'Statement': [
                                    {
                                        'Action': 'bar:baz',
                                        'Effect': 'Allow',
                                        'Principal': '*',
                                        'Resource': 'foo',
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                        },
                    },
                },
            });
            test.done();
        },
        'forBucket returns a permission statement associated with the bucket\'s ARN'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const x = new iam.PolicyStatement({
                resources: [bucket.bucketArn],
                actions: ['s3:ListBucket'],
                principals: [new iam.AnyPrincipal()],
            });
            test.deepEqual(stack.resolve(x.toStatementJson()), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Principal: '*',
                Resource: { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
            });
            test.done();
        },
        'arnForObjects returns a permission statement associated with objects in the bucket'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const p = new iam.PolicyStatement({
                resources: [bucket.arnForObjects('hello/world')],
                actions: ['s3:GetObject'],
                principals: [new iam.AnyPrincipal()],
            });
            test.deepEqual(stack.resolve(p.toStatementJson()), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Principal: '*',
                Resource: {
                    'Fn::Join': [
                        '',
                        [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/hello/world'],
                    ],
                },
            });
            test.done();
        },
        'arnForObjects accepts multiple arguments and FnConcats them'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const user = new iam.User(stack, 'MyUser');
            const team = new iam.Group(stack, 'MyTeam');
            const resource = bucket.arnForObjects(`home/${team.groupName}/${user.userName}/*`);
            const p = new iam.PolicyStatement({
                resources: [resource],
                actions: ['s3:GetObject'],
                principals: [new iam.AnyPrincipal()],
            });
            test.deepEqual(stack.resolve(p.toStatementJson()), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Principal: '*',
                Resource: {
                    'Fn::Join': [
                        '',
                        [
                            { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                            '/home/',
                            { Ref: 'MyTeam01DD6685' },
                            '/',
                            { Ref: 'MyUserDC45028B' },
                            '/*',
                        ],
                    ],
                },
            });
            test.done();
        },
    },
    'removal policy can be used to specify behavior upon delete'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            encryption: s3.BucketEncryption.UNENCRYPTED,
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyBucketF68F3FF0: {
                    Type: 'AWS::S3::Bucket',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        });
        test.done();
    },
    'import/export': {
        'static import(ref) allows importing an external/existing bucket'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn });
            // this is a no-op since the bucket is external
            bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['foo'],
                actions: ['bar:baz'],
                principals: [new iam.AnyPrincipal()],
            }));
            const p = new iam.PolicyStatement({
                resources: [bucket.bucketArn],
                actions: ['s3:ListBucket'],
                principals: [new iam.AnyPrincipal()],
            });
            // it is possible to obtain a permission statement for a ref
            test.deepEqual(p.toStatementJson(), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Principal: '*',
                Resource: 'arn:aws:s3:::my-bucket',
            });
            test.deepEqual(bucket.bucketArn, bucketArn);
            test.deepEqual(stack.resolve(bucket.bucketName), 'my-bucket');
            test.deepEqual(assert_1.SynthUtils.synthesize(stack).template, {}, 'the ref is not a real resource');
            test.done();
        },
        'import does not create any resources'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['*'],
                principals: [new iam.AnyPrincipal()],
            }));
            // at this point we technically didn't create any resources in the consuming stack.
            assert_1.expect(stack).toMatch({});
            test.done();
        },
        'import can also be used to import arbitrary ARNs'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }));
            // but now we can reference the bucket
            // you can even use the bucket name, which will be extracted from the arn provided.
            const user = new iam.User(stack, 'MyUser');
            user.addToPolicy(new iam.PolicyStatement({
                resources: [bucket.arnForObjects(`my/folder/${bucket.bucketName}`)],
                actions: ['s3:*']
            }));
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyUserDC45028B': {
                        'Type': 'AWS::IAM::User',
                    },
                    'MyUserDefaultPolicy7B897426': {
                        'Type': 'AWS::IAM::Policy',
                        'Properties': {
                            'PolicyDocument': {
                                'Statement': [
                                    {
                                        'Action': 's3:*',
                                        'Effect': 'Allow',
                                        'Resource': 'arn:aws:s3:::my-bucket/my/folder/my-bucket',
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                            'PolicyName': 'MyUserDefaultPolicy7B897426',
                            'Users': [
                                {
                                    'Ref': 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
            test.done();
        },
    },
    'grantRead'(test) {
        const stack = new cdk.Stack();
        const reader = new iam.User(stack, 'Reader');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.grantRead(reader);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'ReaderF7BF189D': {
                    'Type': 'AWS::IAM::User',
                },
                'ReaderDefaultPolicy151F3818': {
                    'Type': 'AWS::IAM::Policy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': [
                                        's3:GetObject*',
                                        's3:GetBucket*',
                                        's3:List*',
                                    ],
                                    'Effect': 'Allow',
                                    'Resource': [
                                        {
                                            'Fn::GetAtt': [
                                                'MyBucketF68F3FF0',
                                                'Arn',
                                            ],
                                        },
                                        {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyBucketF68F3FF0',
                                                            'Arn',
                                                        ],
                                                    },
                                                    '/*',
                                                ],
                                            ],
                                        },
                                    ],
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'PolicyName': 'ReaderDefaultPolicy151F3818',
                        'Users': [
                            {
                                'Ref': 'ReaderF7BF189D',
                            },
                        ],
                    },
                },
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'grantReadWrite': {
        'can be used to grant reciprocal permissions to an identity'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyBucketF68F3FF0': {
                        'Type': 'AWS::S3::Bucket',
                        'DeletionPolicy': 'Retain',
                        'UpdateReplacePolicy': 'Retain',
                    },
                    'MyUserDC45028B': {
                        'Type': 'AWS::IAM::User',
                    },
                    'MyUserDefaultPolicy7B897426': {
                        'Type': 'AWS::IAM::Policy',
                        'Properties': {
                            'PolicyDocument': {
                                'Statement': [
                                    {
                                        'Action': [
                                            's3:GetObject*',
                                            's3:GetBucket*',
                                            's3:List*',
                                            's3:DeleteObject*',
                                            's3:PutObject*',
                                            's3:Abort*',
                                        ],
                                        'Effect': 'Allow',
                                        'Resource': [
                                            {
                                                'Fn::GetAtt': [
                                                    'MyBucketF68F3FF0',
                                                    'Arn',
                                                ],
                                            },
                                            {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        {
                                                            'Fn::GetAtt': [
                                                                'MyBucketF68F3FF0',
                                                                'Arn',
                                                            ],
                                                        },
                                                        '/*',
                                                    ],
                                                ],
                                            },
                                        ],
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                            'PolicyName': 'MyUserDefaultPolicy7B897426',
                            'Users': [
                                {
                                    'Ref': 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
            test.done();
        },
        'grant permissions to non-identity principal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
            // WHEN
            bucket.grantRead(new iam.OrganizationPrincipal('o-1234'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                PolicyDocument: {
                    'Version': '2012-10-17',
                    'Statement': [
                        {
                            'Action': ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                            'Condition': { 'StringEquals': { 'aws:PrincipalOrgID': 'o-1234' } },
                            'Effect': 'Allow',
                            'Principal': '*',
                            'Resource': [
                                { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                                { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/*']] },
                            ],
                        },
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                'KeyPolicy': {
                    'Statement': [
                        {
                            'Action': ['kms:Create*', 'kms:Describe*', 'kms:Enable*', 'kms:List*', 'kms:Put*', 'kms:Update*',
                                'kms:Revoke*', 'kms:Disable*', 'kms:Get*', 'kms:Delete*', 'kms:ScheduleKeyDeletion', 'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey', 'kms:TagResource', 'kms:UntagResource'],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': ['', [
                                            'arn:', { 'Ref': 'AWS::Partition' }, ':iam::', { 'Ref': 'AWS::AccountId' }, ':root',
                                        ]],
                                },
                            },
                            'Resource': '*',
                        },
                        {
                            'Action': ['kms:Decrypt', 'kms:DescribeKey'],
                            'Effect': 'Allow',
                            'Resource': '*',
                            'Principal': '*',
                            'Condition': { 'StringEquals': { 'aws:PrincipalOrgID': 'o-1234' } },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
        'if an encryption key is included, encrypt/decrypt permissions are also added both ways'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyBucketKeyC17130CF': {
                        'Type': 'AWS::KMS::Key',
                        'Properties': {
                            'Description': 'Created by Default/MyBucket',
                            'KeyPolicy': {
                                'Statement': [
                                    {
                                        'Action': [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        'Effect': 'Allow',
                                        'Principal': {
                                            'AWS': {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            'Ref': 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            'Ref': 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        'Resource': '*',
                                    },
                                    {
                                        'Action': [
                                            'kms:Decrypt',
                                            'kms:DescribeKey',
                                            'kms:Encrypt',
                                            'kms:ReEncrypt*',
                                            'kms:GenerateDataKey*',
                                        ],
                                        'Effect': 'Allow',
                                        'Principal': {
                                            'AWS': {
                                                'Fn::GetAtt': [
                                                    'MyUserDC45028B',
                                                    'Arn',
                                                ],
                                            },
                                        },
                                        'Resource': '*',
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                        },
                        'DeletionPolicy': 'Retain',
                        'UpdateReplacePolicy': 'Retain',
                    },
                    'MyBucketF68F3FF0': {
                        'Type': 'AWS::S3::Bucket',
                        'DeletionPolicy': 'Retain',
                        'UpdateReplacePolicy': 'Retain',
                        'Properties': {
                            'BucketEncryption': {
                                'ServerSideEncryptionConfiguration': [
                                    {
                                        'ServerSideEncryptionByDefault': {
                                            'KMSMasterKeyID': {
                                                'Fn::GetAtt': [
                                                    'MyBucketKeyC17130CF',
                                                    'Arn',
                                                ],
                                            },
                                            'SSEAlgorithm': 'aws:kms',
                                        },
                                    },
                                ],
                            },
                        },
                    },
                    'MyUserDC45028B': {
                        'Type': 'AWS::IAM::User',
                    },
                    'MyUserDefaultPolicy7B897426': {
                        'Type': 'AWS::IAM::Policy',
                        'Properties': {
                            'PolicyDocument': {
                                'Statement': [
                                    {
                                        'Action': [
                                            's3:GetObject*',
                                            's3:GetBucket*',
                                            's3:List*',
                                            's3:DeleteObject*',
                                            's3:PutObject*',
                                            's3:Abort*',
                                        ],
                                        'Effect': 'Allow',
                                        'Resource': [
                                            {
                                                'Fn::GetAtt': [
                                                    'MyBucketF68F3FF0',
                                                    'Arn',
                                                ],
                                            },
                                            {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        {
                                                            'Fn::GetAtt': [
                                                                'MyBucketF68F3FF0',
                                                                'Arn',
                                                            ],
                                                        },
                                                        '/*',
                                                    ],
                                                ],
                                            },
                                        ],
                                    },
                                    {
                                        'Action': [
                                            'kms:Decrypt',
                                            'kms:DescribeKey',
                                            'kms:Encrypt',
                                            'kms:ReEncrypt*',
                                            'kms:GenerateDataKey*',
                                        ],
                                        'Effect': 'Allow',
                                        'Resource': {
                                            'Fn::GetAtt': [
                                                'MyBucketKeyC17130CF',
                                                'Arn',
                                            ],
                                        },
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                            'PolicyName': 'MyUserDefaultPolicy7B897426',
                            'Users': [
                                {
                                    'Ref': 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
            test.done();
        },
    },
    'more grants'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
        const putter = new iam.User(stack, 'Putter');
        const writer = new iam.User(stack, 'Writer');
        const deleter = new iam.User(stack, 'Deleter');
        bucket.grantPut(putter);
        bucket.grantWrite(writer);
        bucket.grantDelete(deleter);
        const resources = assert_1.SynthUtils.synthesize(stack).template.Resources;
        const actions = (id) => resources[id].Properties.PolicyDocument.Statement[0].Action;
        test.deepEqual(actions('WriterDefaultPolicyDC585BCE'), ['s3:DeleteObject*', 's3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('PutterDefaultPolicyAB138DD3'), ['s3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('DeleterDefaultPolicyCD33B8A0'), 's3:DeleteObject*');
        test.done();
    },
    'grantDelete, with a KMS Key'(test) {
        // given
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'MyKey');
        const deleter = new iam.User(stack, 'Deleter');
        const bucket = new s3.Bucket(stack, 'MyBucket', {
            bucketName: 'my-bucket-physical-name',
            encryptionKey: key,
            encryption: s3.BucketEncryption.KMS,
        });
        // when
        bucket.grantDelete(deleter);
        // then
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 's3:DeleteObject*',
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'MyBucketF68F3FF0',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
        }));
        test.done();
    },
    'cross-stack permissions': {
        'in the same account and region'(test) {
            const app = new cdk.App();
            const stackA = new cdk.Stack(app, 'stackA');
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket');
            const stackB = new cdk.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            bucketFromStackA.grantRead(user);
            assert_1.expect(stackA).toMatch({
                'Resources': {
                    'MyBucketF68F3FF0': {
                        'Type': 'AWS::S3::Bucket',
                        'DeletionPolicy': 'Retain',
                        'UpdateReplacePolicy': 'Retain',
                    },
                },
                'Outputs': {
                    'ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58': {
                        'Value': {
                            'Fn::GetAtt': [
                                'MyBucketF68F3FF0',
                                'Arn',
                            ],
                        },
                        'Export': {
                            'Name': 'stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58',
                        },
                    },
                },
            });
            assert_1.expect(stackB).toMatch({
                'Resources': {
                    'UserWhoNeedsAccessF8959C3D': {
                        'Type': 'AWS::IAM::User',
                    },
                    'UserWhoNeedsAccessDefaultPolicy6A9EB530': {
                        'Type': 'AWS::IAM::Policy',
                        'Properties': {
                            'PolicyDocument': {
                                'Statement': [
                                    {
                                        'Action': [
                                            's3:GetObject*',
                                            's3:GetBucket*',
                                            's3:List*',
                                        ],
                                        'Effect': 'Allow',
                                        'Resource': [
                                            {
                                                'Fn::ImportValue': 'stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58',
                                            },
                                            {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        {
                                                            'Fn::ImportValue': 'stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58',
                                                        },
                                                        '/*',
                                                    ],
                                                ],
                                            },
                                        ],
                                    },
                                ],
                                'Version': '2012-10-17',
                            },
                            'PolicyName': 'UserWhoNeedsAccessDefaultPolicy6A9EB530',
                            'Users': [
                                {
                                    'Ref': 'UserWhoNeedsAccessF8959C3D',
                                },
                            ],
                        },
                    },
                },
            });
            test.done();
        },
        'in different accounts'(test) {
            // given
            const stackA = new cdk.Stack(undefined, 'StackA', { env: { account: '123456789012' } });
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket', {
                bucketName: 'my-bucket-physical-name',
            });
            const stackB = new cdk.Stack(undefined, 'StackB', { env: { account: '234567890123' } });
            const roleFromStackB = new iam.Role(stackB, 'MyRole', {
                assumedBy: new iam.AccountPrincipal('234567890123'),
                roleName: 'MyRolePhysicalName',
            });
            // when
            bucketFromStackA.grantRead(roleFromStackB);
            // then
            assert_1.expect(stackA).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':iam::234567890123:role/MyRolePhysicalName',
                                        ],
                                    ],
                                },
                            },
                        },
                    ],
                },
            }));
            assert_1.expect(stackB).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Resource': [
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::my-bucket-physical-name',
                                        ],
                                    ],
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::my-bucket-physical-name/*',
                                        ],
                                    ],
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'in different accounts, with a KMS Key'(test) {
            // given
            const stackA = new cdk.Stack(undefined, 'StackA', { env: { account: '123456789012' } });
            const key = new kms.Key(stackA, 'MyKey');
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket', {
                bucketName: 'my-bucket-physical-name',
                encryptionKey: key,
                encryption: s3.BucketEncryption.KMS,
            });
            const stackB = new cdk.Stack(undefined, 'StackB', { env: { account: '234567890123' } });
            const roleFromStackB = new iam.Role(stackB, 'MyRole', {
                assumedBy: new iam.AccountPrincipal('234567890123'),
                roleName: 'MyRolePhysicalName',
            });
            // when
            bucketFromStackA.grantRead(roleFromStackB);
            // then
            assert_1.expect(stackA).to(assert_1.haveResourceLike('AWS::KMS::Key', {
                'KeyPolicy': {
                    'Statement': [
                        {
                        // grant to the root of the owning account
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                            'Principal': {
                                'AWS': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':iam::234567890123:role/MyRolePhysicalName',
                                        ],
                                    ],
                                },
                            },
                        },
                    ],
                },
            }));
            assert_1.expect(stackB).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                        // Bucket grant
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                            'Resource': '*',
                        },
                    ],
                },
            }));
            test.done();
        },
    },
    'urlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketURL', { value: bucket.urlForObject() });
        new cdk.CfnOutput(stack, 'MyFileURL', { value: bucket.urlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileURL', { value: bucket.urlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
            'Outputs': {
                'BucketURL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                'https://s3.',
                                {
                                    'Ref': 'AWS::Region',
                                },
                                '.',
                                {
                                    'Ref': 'AWS::URLSuffix',
                                },
                                '/',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                            ],
                        ],
                    },
                },
                'MyFileURL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                'https://s3.',
                                {
                                    'Ref': 'AWS::Region',
                                },
                                '.',
                                {
                                    'Ref': 'AWS::URLSuffix',
                                },
                                '/',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                                '/my/file.txt',
                            ],
                        ],
                    },
                },
                'YourFileURL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                'https://s3.',
                                {
                                    'Ref': 'AWS::Region',
                                },
                                '.',
                                {
                                    'Ref': 'AWS::URLSuffix',
                                },
                                '/',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                                '/your/file.txt',
                            ],
                        ],
                    },
                },
            },
        });
        test.done();
    },
    's3UrlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketS3URL', { value: bucket.s3UrlForObject() });
        new cdk.CfnOutput(stack, 'MyFileS3URL', { value: bucket.s3UrlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileS3URL', { value: bucket.s3UrlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
            'Outputs': {
                'BucketS3URL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                            ],
                        ],
                    },
                },
                'MyFileS3URL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                                '/my/file.txt',
                            ],
                        ],
                    },
                },
                'YourFileS3URL': {
                    'Value': {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    'Ref': 'MyBucketF68F3FF0',
                                },
                                '/your/file.txt',
                            ],
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'grantPublicAccess': {
        'by default, grants s3:GetObject to all objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 's3:GetObject',
                            'Effect': 'Allow',
                            'Principal': '*',
                            'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['bC3BBCC65', 'Arn'] }, '/*']] },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
        '"keyPrefix" can be used to only grant access to certain objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('only/access/these/*');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 's3:GetObject',
                            'Effect': 'Allow',
                            'Principal': '*',
                            'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['bC3BBCC65', 'Arn'] }, '/only/access/these/*']] },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
        '"allowedActions" can be used to specify actions explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('*', 's3:GetObject', 's3:PutObject');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': ['s3:GetObject', 's3:PutObject'],
                            'Effect': 'Allow',
                            'Principal': '*',
                            'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['bC3BBCC65', 'Arn'] }, '/*']] },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
        'returns the PolicyStatement which can be then customized'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            const result = bucket.grantPublicAccess();
            result.resourceStatement.addCondition('IpAddress', { 'aws:SourceIp': '54.240.143.0/24' });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 's3:GetObject',
                            'Effect': 'Allow',
                            'Principal': '*',
                            'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['bC3BBCC65', 'Arn'] }, '/*']] },
                            'Condition': {
                                'IpAddress': { 'aws:SourceIp': '54.240.143.0/24' },
                            },
                        },
                    ],
                    'Version': '2012-10-17',
                },
            }));
            test.done();
        },
        'throws when blockPublicPolicy is set to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', {
                blockPublicAccess: new s3.BlockPublicAccess({ blockPublicPolicy: true }),
            });
            // THEN
            test.throws(() => bucket.grantPublicAccess(), /blockPublicPolicy/);
            test.done();
        },
    },
    'website configuration': {
        'only index doc'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: 'index2.html',
                },
            }));
            test.done();
        },
        'fails if only error doc is specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteErrorDocument: 'error.html',
                });
            }, /"websiteIndexDocument" is required if "websiteErrorDocument" is set/);
            test.done();
        },
        'error and index docs'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
                websiteErrorDocument: 'error.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: 'index2.html',
                    ErrorDocument: 'error.html',
                },
            }));
            test.done();
        },
        'exports the WebsiteURL'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html',
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] });
            test.done();
        },
        'exports the WebsiteDomain'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html',
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), {
                'Fn::Select': [
                    2,
                    {
                        'Fn::Split': ['/', { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] }],
                    },
                ],
            });
            test.done();
        },
        'exports the WebsiteURL for imported buckets'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketName(stack, 'Website', 'my-test-bucket');
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), {
                'Fn::Join': [
                    '',
                    [
                        'http://my-test-bucket.s3-website-',
                        { Ref: 'AWS::Region' },
                        '.',
                        { Ref: 'AWS::URLSuffix' },
                    ],
                ],
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), {
                'Fn::Join': [
                    '',
                    [
                        'my-test-bucket.s3-website-',
                        { Ref: 'AWS::Region' },
                        '.',
                        { Ref: 'AWS::URLSuffix' },
                    ],
                ],
            });
            test.done();
        },
        'exports the WebsiteURL for imported buckets with url'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Website', {
                bucketName: 'my-test-bucket',
                bucketWebsiteUrl: 'http://my-test-bucket.my-test.suffix',
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), 'http://my-test-bucket.my-test.suffix');
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), 'my-test-bucket.my-test.suffix');
            test.done();
        },
        'adds RedirectAllRequestsTo property'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteRedirect: {
                    hostName: 'www.example.com',
                    protocol: s3.RedirectProtocol.HTTPS,
                },
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    RedirectAllRequestsTo: {
                        HostName: 'www.example.com',
                        Protocol: 'https',
                    },
                },
            }));
            test.done();
        },
        'fails if websiteRedirect and websiteIndex and websiteError are specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteIndexDocument: 'index.html',
                    websiteErrorDocument: 'error.html',
                    websiteRedirect: {
                        hostName: 'www.example.com',
                    },
                });
            }, /"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used/);
            test.done();
        },
        'fails if websiteRedirect and websiteRoutingRules are specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteRoutingRules: [],
                    websiteRedirect: {
                        hostName: 'www.example.com',
                    },
                });
            }, /"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used/);
            test.done();
        },
        'adds RedirectRules property'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteRoutingRules: [{
                        hostName: 'www.example.com',
                        httpRedirectCode: '302',
                        protocol: s3.RedirectProtocol.HTTPS,
                        replaceKey: s3.ReplaceKey.prefixWith('test/'),
                        condition: {
                            httpErrorCodeReturnedEquals: '200',
                            keyPrefixEquals: 'prefix',
                        },
                    }],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    RoutingRules: [{
                            RedirectRule: {
                                HostName: 'www.example.com',
                                HttpRedirectCode: '302',
                                Protocol: 'https',
                                ReplaceKeyPrefixWith: 'test/',
                            },
                            RoutingRuleCondition: {
                                HttpErrorCodeReturnedEquals: '200',
                                KeyPrefixEquals: 'prefix',
                            },
                        }],
                },
            }));
            test.done();
        },
        'fails if routingRule condition object is empty'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteRoutingRules: [{
                            httpRedirectCode: '303',
                            condition: {},
                        }],
                });
            }, /The condition property cannot be an empty object/);
            test.done();
        },
        'isWebsite set properly with': {
            'only index doc'(test) {
                const stack = new cdk.Stack();
                const bucket = new s3.Bucket(stack, 'Website', {
                    websiteIndexDocument: 'index2.html',
                });
                test.equal(bucket.isWebsite, true);
                test.done();
            },
            'error and index docs'(test) {
                const stack = new cdk.Stack();
                const bucket = new s3.Bucket(stack, 'Website', {
                    websiteIndexDocument: 'index2.html',
                    websiteErrorDocument: 'error.html',
                });
                test.equal(bucket.isWebsite, true);
                test.done();
            },
            'redirects'(test) {
                const stack = new cdk.Stack();
                const bucket = new s3.Bucket(stack, 'Website', {
                    websiteRedirect: {
                        hostName: 'www.example.com',
                        protocol: s3.RedirectProtocol.HTTPS,
                    },
                });
                test.equal(bucket.isWebsite, true);
                test.done();
            },
            'no website properties set'(test) {
                const stack = new cdk.Stack();
                const bucket = new s3.Bucket(stack, 'Website');
                test.equal(bucket.isWebsite, false);
                test.done();
            },
            'imported website buckets'(test) {
                const stack = new cdk.Stack();
                const bucket = s3.Bucket.fromBucketAttributes(stack, 'Website', {
                    bucketArn: 'arn:aws:s3:::my-bucket',
                    isWebsite: true,
                });
                test.equal(bucket.isWebsite, true);
                test.done();
            },
            'imported buckets'(test) {
                const stack = new cdk.Stack();
                const bucket = s3.Bucket.fromBucketAttributes(stack, 'NotWebsite', {
                    bucketArn: 'arn:aws:s3:::my-bucket',
                });
                test.equal(bucket.isWebsite, false);
                test.done();
            },
        },
    },
    'Bucket.fromBucketArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const bucket = s3.Bucket.fromBucketArn(stack, 'my-bucket', 'arn:aws:s3:::my_corporate_bucket');
        // THEN
        test.deepEqual(bucket.bucketName, 'my_corporate_bucket');
        test.deepEqual(bucket.bucketArn, 'arn:aws:s3:::my_corporate_bucket');
        test.done();
    },
    'Bucket.fromBucketName'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const bucket = s3.Bucket.fromBucketName(stack, 'imported-bucket', 'my-bucket-name');
        // THEN
        test.deepEqual(bucket.bucketName, 'my-bucket-name');
        test.deepEqual(stack.resolve(bucket.bucketArn), {
            'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket-name']],
        });
        test.done();
    },
    'if a kms key is specified, it implies bucket is encrypted with kms (dah)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'k');
        // THEN
        new s3.Bucket(stack, 'b', { encryptionKey: key });
        test.done();
    },
    'Bucket with Server Access Logs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const accessLogBucket = new s3.Bucket(stack, 'AccessLogs');
        new s3.Bucket(stack, 'MyBucket', {
            serverAccessLogsBucket: accessLogBucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LoggingConfiguration: {
                DestinationBucketName: {
                    Ref: 'AccessLogs8B620ECA',
                },
            },
        }));
        test.done();
    },
    'Bucket with Server Access Logs with Prefix'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const accessLogBucket = new s3.Bucket(stack, 'AccessLogs');
        new s3.Bucket(stack, 'MyBucket', {
            serverAccessLogsBucket: accessLogBucket,
            serverAccessLogsPrefix: 'hello',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LoggingConfiguration: {
                DestinationBucketName: {
                    Ref: 'AccessLogs8B620ECA',
                },
                LogFilePrefix: 'hello',
            },
        }));
        test.done();
    },
    'Access log prefix given without bucket'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            serverAccessLogsPrefix: 'hello',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LoggingConfiguration: {
                LogFilePrefix: 'hello',
            },
        }));
        test.done();
    },
    'Bucket Allow Log delivery changes bucket Access Control should fail'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const accessLogBucket = new s3.Bucket(stack, 'AccessLogs', {
            accessControl: s3.BucketAccessControl.AUTHENTICATED_READ,
        });
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            serverAccessLogsBucket: accessLogBucket,
            serverAccessLogsPrefix: 'hello',
            accessControl: s3.BucketAccessControl.AUTHENTICATED_READ,
        }), /Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed/);
        test.done();
    },
    'Defaults for an inventory bucket'(test) {
        // Given
        const stack = new cdk.Stack();
        const inventoryBucket = new s3.Bucket(stack, 'InventoryBucket');
        new s3.Bucket(stack, 'MyBucket', {
            inventories: [
                {
                    destination: {
                        bucket: inventoryBucket,
                    },
                },
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::Bucket', {
            InventoryConfigurations: [
                {
                    Enabled: true,
                    IncludedObjectVersions: 'All',
                    ScheduleFrequency: 'Weekly',
                    Destination: {
                        Format: 'CSV',
                        BucketArn: { 'Fn::GetAtt': ['InventoryBucketA869B8CB', 'Arn'] },
                    },
                    Id: 'MyBucketInventory0',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
            Bucket: { Ref: 'InventoryBucketA869B8CB' },
            PolicyDocument: {
                Statement: assert_1.arrayWith(assert_1.objectLike({
                    Action: 's3:PutObject',
                    Principal: { Service: 's3.amazonaws.com' },
                    Resource: [
                        {
                            'Fn::GetAtt': ['InventoryBucketA869B8CB', 'Arn'],
                        },
                        {
                            'Fn::Join': ['', [{ 'Fn::GetAtt': ['InventoryBucketA869B8CB', 'Arn'] }, '/*']],
                        },
                    ],
                })),
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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