# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['confect']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'confect',
    'version': '0.2.6',
    'description': 'Python configuration library that provides pleasant configuration definition and access interface, and it reads unrestricted python configuration file.',
    'long_description': "Confect\n=======\n\n**confect** is a Python configuration library with the following features.\n\n- A readable and pleasant configuration definition and accessing interface\n- Predefined configuration and immutable conf object for reducing the\n  possibility of making errors. You shouldn't modify configuration too\n  dynamically as if they are global variables.\n- Loading configuration file from file path, module importing or even from\n  environment variable.\n- Configuration files in Python. This makes it possible to\n\n  + have complex type objects as configuration values, like Decimal, timedelta\n    or any class instance\n  + dynamically handle complicated logic, you can use conditional statements\n    like ``if`` in it.\n  + read other TOML/YMAL/JSON files or even environment variables in the\n    configuration file.\n\nInstall\n-------\n\n``confect`` is a Python package hosted on PyPI and works only with Python 3.6 up.\n\nJust like other Python package, install it by `pip\n<https://pip.pypa.io/en/stable/>`_ into a `virtualenv\n<https://hynek.me/articles/virtualenv-lives/>`_, or use `poetry\n<https://poetry.eustace.io/>`_ to manage project dependencies and virtualenv.\n\n.. code:: console\n\n   $ pip install confect\n\n\nInitialize Conf object\n---------------\n\nCalling ``conf = confect.Conf()`` creates a new configuration manager object.\nPut following lines in your application package. For example, suppose ``proj_X``\nis your top-level package name. Put the following lines into\n``proj_X.__init__.py`` or ``proj_X.core.py``.\n\n.. code:: python\n\n   import confect\n   conf = confect.Conf()\n\n   # load configuration files through importing\n   try:\n       conf.load_module('proj_X_conf')\n   except ImportError:\n       pass\n\n   # overrides configuration with environment variables\n   conf.load_envvars('proj_X')\n\nIt is possible to create multiple ``Conf`` objects, but normally we don't need\nit. In most cases, initialize only one ``Conf`` object in one module in your\npackage, then import and use it anywhere in your application.\n\nDeclare Configuration Groups and Properties\n-------------------------------------------\n\nConfiguration properties should be declared before using it. Use\n``Conf.declare_group(group_name)`` context manager to declare a configuration\ngroup and all properties under it at the same time. It's nessasery to provide\ndefault values for each properties. Default values can be any type. The group\nname should be valid attribute names.\n\nPut your configuration group declaration code in modules where you need those\nproperties. And make sure that the declaration is before all the lines that\naccess these properties. Normally, the group name is your class name, module\nname or subpackage name.\n\nSuppose that there's a ``proj_X/api.py`` module.\n\n.. code:: python\n\n   from proj_X.core import conf\n\n   with conf.declare_group('api') as cg: # `cg` stands for conf_group\n       cg.cache_expire = 60 * 60 * 24\n       cg.cache_prefix = 'proj_X_cache'\n       cg.url_base_path = 'api/v2/'\n\nAccess Configuration\n--------------------\n\nAfter declared the group and properties, they are accessable through\ngetting attribute from ``Conf`` object, like this ``conf.group_name.prop_name``.\n\nThe rest of ``proj_X/api.py`` module.\n\n.. code:: python\n\n   @routes(conf.api.url_base_path + 'add')\n   @redis_cache(key=conf.api.cache_prefix, expire=conf.api.cache_expire)\n   def add(a, b)\n       return a + b\n\n\nConfiguration properties and groups are immutable. They can only be globally\nchanged by loading configuration files. Otherwise, they are always default\nvalues.\n\n>>> conf.yummy.name = 'octopus'\nTraceback (most recent call last):\n   ...\nconfect.error.FrozenConfPropError: Configuration properties are frozen.\n\nLoading Configuration\n---------------------\n\nConfiguration properties and groups are immutable. The standard way to change it\nis to load configuration from files or environment variables.\n\nUse ``Conf.load_conf_file(path)`` or ``Conf.load_conf_module(module_name)`` to\nload configuration files, or use ``Conf.load_envvars(prefix)`` to load\nconfiguration from environment variable. No matter the loading statement is\nlocated before or after groups/properties declaration, property values in\nconfiguration file always override default values. It's possible to load \nconfiguration multiple times, the latter one would replace values from former loading.\n\nBe aware, *you should access your configuration properties after load\nconfiguration files.* If not, you might get wrong/default value. Therefore, we\nusually load configuration file right after the statement of creating the\n``Conf`` object.\n\nSometimes, it is smart to use ``PYTHONPATH`` control the source of configuration\nfile.\n\n.. code:: console\n\n   $ vi proj_X_conf.py\n   $ export PYTHONPATH=.\n   $ python your_application.py\n\nHere's an example of complex configuration loading.\n\n.. code:: python\n\n   import sys\n   import confect\n\n   conf = confect.Conf()\n\n   # load configuration file\n   if len(sys.argv) == 2:\n       conf.load_conf_file(sys.argv[1])\n   else:\n       try:\n          conf.load_conf_file('path/to/team_conf.py')\n       FileNotFoundError:\n          logger.warning('Unable to find team configuration file')\n\n       try:\n          conf.load_conf_file('path/to/personal_conf.py')\n       FileNotFoundError:\n          logger.info('Unable to find personal configuration file')\n\n   # load configuration file through importing\n   try:\n       conf.load_module('proj_X_conf')\n   except ImportError:\n       logger.warning('Unable to load find configuration module %r',\n                      'proj_x_conf')\n\n   # overrides configuration with environment variables\n   conf.load_envvars('proj_X')\n\n\nConfiguration File\n------------------\n\nThe configuration file is in Python. That makes your configuration file\nprogrammable and unrestricted. It is possible and easy to\n\n- have complex type objects as configuration values, like Decimal, timedelta or\n  any class instance\n- dynamically handle complicated logic, you can use conditional statements like\n  ``if`` in it.\n- read other TOML/YMAL/JSON files or even environment variables in the\n  configuration file.\n\nIt's not necessary and is unusual to have all configuration properties in the\nconfiguration file. *Put only those configuration properties and corresponding\nvalues that you want to override to the configuration file.*\n\nIn configuration file, import ``confect.c`` object and set all properties on it\nas if ``c`` is the conf object. Here's an example of configuration file.\n\n.. code-block:: python\n\n   from confect import c\n\n   c.yummy.kind = 'poultry'\n   c.yummy.name = 'chicken'\n   c.yummy.weight = 25\n\n   import os\n   # simple calculation or loading env var\n   c.cache.expire = 60 * 60 # one hour\n   c.cache.key = os.environ['CACHE_KEY']\n\n   # it's easy to have conditional statement\n   DEBUG = True\n   if DEBUG:\n       c.cache.disable = True\n\n   # loading some secret file and set configuration\n   import json\n   with open('secret.json') as f:\n       secret = json.load(f)\n\n   c.secret.key = secret['key']\n   c.secret.token = secret['token']\n\nThe ``c`` object only exits when loading a python configuration file, it's not\npossible to import it in your source code. You can set any property in any\nconfiguration group onto the ``c`` object. However,\n*they are only accessable if you declared it in the source code with* ``Conf.declare_group(group_name)``.\n\n\nLoad Environment Variables\n---------------------------\n\n``Conf.load_envvars(prefix: str)`` automatically searches environment variables\nin ``<prefix>__<group>__<prop>`` format. All of these three identifier are case\nsensitive. If you have a configuration property ``conf.cache.expire_time`` and\nyou call ``Conf.load_envvars('proj_X')``. It will set that ``expire_time``\nproperty to the parsed value of ``proj_X__cache__expire_time`` environment\nvariable.\n\n>>> import os\n>>> os.environ['proj_X.cache.expire'] = '3600'\n\n>>> conf = confect.Conf()\n>>> conf.load_envvars('proj_X')  # doctest: +SKIP\n\nIf ``cache.expire`` has been declared, then\n\n>>> conf.cache.expire\n3600\n\nConfect includes predefined parsers of these primitive types.\n\n- ``str``: ``s``\n- ``int``: ``int(s)``\n- ``float``: ``float(s)``\n- ``bytes``: ``s.decode()``\n- ``datetime.datetime`` : ``pendulum.parse(s)``\n- ``datetime.date`` : ``pendulum.parse(s).date()``\n- ``Decimal`` : ``decimal.Decimal(s)``\n- ``tuple`` : ``json.loads(s)``\n- ``dict``: ``json.loads(s)``\n- ``list``: ``json.loads(s)``\n\nMutable Environment\n-----------------\n\n``Conf.mutate_locally()`` context manager creates an environment that makes\n``Conf`` object temporarily mutable. All changes would be restored when it\nleaves the block. It is usaful on writing test case or testing configuration\nproperties in Python REPL.\n\n>>> conf = Conf()\n>>> conf.declare_group(  # declare group through keyword arguments\n...      'dummy',\n...      prop1=3,\n...      prop2='some string')\n...\n>>> with conf.mutate_locally():\n...      conf.dummy.prop1 = 5\n...      print(conf.dummy.prop1)\n5\n...     call_some_function_use_this_property()\n>>> print(conf.dummy.prop1)  # all configuration restored\n3\n\n\nTo-Dos\n======\n\n- A function for loading dictionary into ``conflect.c``.\n- A function that loads command line arguments and overrides configuration properties.\n- Copy-on-write mechenism in ``conf.mutate_locally()`` for better performance and memory usage.\n- API reference page\n",
    'author': '顏孜羲',
    'author_email': 'joseph.yen@gmail.com',
    'url': 'https://github.com/d2207197/confect',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
