"""Task-based source code compression for Context Engine."""

import os
from pathlib import Path
from typing import Optional
import re

from context_engine.core.task_manager import get_task
from context_engine.core.utils import (
    redact_secrets,
    compress_code,
    extract_api_docstrings,
    compress_whitespace
)
from context_engine.core import Config


def compress_for_task(task: Optional[str] = None) -> None:
    """Compress src/ directory based on current task."""
    # Get the current task if not provided
    if task is None:
        task = get_task()

    if not task:
        print("No task set for compression. Run 'context start-session --task \"your task\"' first.")
        return

    # Path to src directory in current project root
    project_root = Path.cwd()
    src_path = project_root / "src"
    context_dir = project_root / ".context"
    compressed_src_dir = context_dir / "compressed_src"

    # If src directory exists, compress it
    if src_path.exists() and src_path.is_dir():
        # Create compressed src directory if it doesn't exist
        compressed_src_dir.mkdir(parents=True, exist_ok=True)

        # Create a markdown file with the task name
        output_file = compressed_src_dir / f"src_compressed_{abs(hash(task)) % 10000:04d}.md"

        # Generate compressed markdown content
        content = _generate_compressed_content(src_path, task)

        # Write to markdown file
        output_file.write_text(content, encoding="utf-8")

        print(f"Compressed {src_path} to {output_file}")
    else:
        print("src/ directory not found. Skipping compression.")
        # Don't create compressed_src directory if no src directory exists


def _generate_compressed_content(src_path: Path, task: str) -> str:
    """Generate compressed markdown content for src directory."""
    content = []
    content.append(f"# Compressed Source Files for Task: {task}")
    content.append("")
    content.append("*Generated by Context Engine V1*")
    content.append("")

    # Walk through src directory
    for root, dirs, files in os.walk(src_path):
        # Skip hidden directories
        dirs[:] = [d for d in dirs if not d.startswith('.')]

        root_path = Path(root)
        relative_path = root_path.relative_to(src_path.parent)

        for file in files:
            file_path = root_path / file

            # Skip binary files and common non-code files
            if _should_skip_file(file):
                continue

            try:
                # Determine file type and process accordingly
                file_ext = file_path.suffix.lower()
                # Handle dotfiles specially
                if file_path.name.startswith('.') and not file_ext:
                    file_ext = file_path.name.lower()
                language = _get_language_from_ext(file_ext)

                if language:
                    # Read and compress code
                    code_content = file_path.read_text(encoding="utf-8", errors="ignore")

                    # Redact secrets first
                    code_content = redact_secrets(code_content)

                    # For CSV and gitignore files, include the content directly
                    if language in ('csv', 'gitignore'):
                        compressed_code = code_content
                    else:
                        # Extract only API docs and signatures for other languages
                        compressed_code = extract_api_docstrings(code_content, language)

                    if compressed_code and compressed_code != "(no docstrings)":
                        content.append(f"## {relative_path / file}")
                        content.append("")
                        content.append("```" + language)
                        content.append(compressed_code)
                        content.append("```")
                        content.append("")
            except (UnicodeDecodeError, IOError):
                # Skip files that can't be read as text
                continue

    return "\n".join(content)


def _should_skip_file(filename: str) -> bool:
    """Check if file should be skipped during compression."""
    try:
        config = Config()
        skip_patterns = config.get("skip_patterns", [
            # Fallback defaults if config doesn't have skip_patterns
            r'\.pyc$', r'\.pyo$', r'\.class$', r'\.jar$', r'\.war$',
            r'\.exe$', r'\.dll$', r'\.so$', r'\.dylib$',
            r'\.png$', r'\.jpg$', r'\.jpeg$', r'\.gif$', r'\.bmp$', r'\.ico$',
            r'\.pdf$', r'\.doc$', r'\.docx$', r'\.xls$', r'\.xlsx$',
            r'\.zip$', r'\.tar$', r'\.gz$', r'\.7z$', r'\.rar$',
            r'\.log$', r'\.tmp$', r'\.bak$', r'\.swp$',
            r'^__pycache__/', r'^\.git/', r'^node_modules/', r'^\.vscode/', r'^\.idea/'
        ])
    except Exception:
        # If config loading fails, use hardcoded defaults
        skip_patterns = [
            r'\.pyc$', r'\.pyo$', r'\.class$', r'\.jar$', r'\.war$',
            r'\.exe$', r'\.dll$', r'\.so$', r'\.dylib$',
            r'\.png$', r'\.jpg$', r'\.jpeg$', r'\.gif$', r'\.bmp$', r'\.ico$',
            r'\.pdf$', r'\.doc$', r'\.docx$', r'\.xls$', r'\.xlsx$',
            r'\.zip$', r'\.tar$', r'\.gz$', r'\.7z$', r'\.rar$',
            r'\.log$', r'\.tmp$', r'\.bak$', r'\.swp$',
            r'^__pycache__/', r'^\.git/', r'^node_modules/', r'^\.vscode/', r'^\.idea/'
        ]

    for pattern in skip_patterns:
        if re.search(pattern, filename, re.IGNORECASE):
            return True
    return False


def _get_language_from_ext(file_ext: str) -> str:
    """Get programming language from file extension."""
    ext_map = {
        '.py': 'python',
        '.js': 'javascript',
        '.ts': 'typescript',
        '.jsx': 'javascript',
        '.tsx': 'typescript',
        '.java': 'java',
        '.c': 'c',
        '.cpp': 'cpp',
        '.h': 'c',
        '.hpp': 'cpp',
        '.cs': 'csharp',
        '.go': 'go',
        '.rs': 'rust',
        '.rb': 'ruby',
        '.php': 'php',
        '.swift': 'swift',
        '.kt': 'kotlin',
        '.scala': 'scala',
        '.m': 'objective-c',
        '.sh': 'bash',
        '.sql': 'sql',
        '.html': 'html',
        '.css': 'css',
        '.scss': 'scss',
        '.sass': 'sass',
        '.less': 'less',
        '.vue': 'vue',
        '.svelte': 'svelte',
        '.xml': 'xml',
        '.json': 'json',
        '.yaml': 'yaml',
        '.yml': 'yaml',
        '.toml': 'toml',
        '.ini': 'ini',
        '.cfg': 'ini',
        '.csv': 'csv',
        '.md': 'markdown',
        '.txt': 'text',
        '.gitignore': 'gitignore'
    }
    return ext_map.get(file_ext, '')