"""File summarizer for Context Engine."""

import ast
import json
import re
from pathlib import Path
from typing import Dict, List, Optional, Any, Tuple
from datetime import datetime
import os

from context_engine.core.config import ContextConfig
from context_engine.core.logger import setup_logger

logger = setup_logger(__name__)

class FileSummarizer:
    """Handles file summarization with template-based approach."""
    
    def __init__(self, config: ContextConfig, project_root: Optional[Path] = None):
        self.config = config
        self.project_root = project_root or Path.cwd()
        self.summaries_dir = self.project_root / "context_engine" / "summaries"
        
        # Ensure directories exist
        self.summaries_dir.mkdir(parents=True, exist_ok=True)
    
    def _get_summary_template(self) -> str:
        """Get the standard summary template."""
        return """# File Summary: {file_path}

## Purpose
{purpose}

## Public Surface
{public_surface}

## Dependencies
{dependencies}

## Risks
{risks}

## Recent Changes
{recent_changes}

## Code Structure
{code_structure}

---
*Generated by Context Engine on {timestamp}*
"""
    
    def _analyze_python_file(self, content: str, file_path: Path) -> Dict[str, str]:
        """Analyze Python file using AST parsing."""
        try:
            tree = ast.parse(content)
        except SyntaxError as e:
            logger.warning(f"Syntax error in {file_path}: {e}")
            return self._fallback_analysis(content, file_path)
        
        # Extract information
        classes = []
        functions = []
        imports = []
        docstring = None
        
        for node in ast.walk(tree):
            if isinstance(node, ast.ClassDef):
                class_info = {
                    'name': node.name,
                    'docstring': ast.get_docstring(node),
                    'methods': [n.name for n in node.body if isinstance(n, ast.FunctionDef)],
                    'line': node.lineno
                }
                classes.append(class_info)
            
            elif isinstance(node, ast.FunctionDef):
                if node.col_offset == 0:  # Top-level function
                    func_info = {
                        'name': node.name,
                        'docstring': ast.get_docstring(node),
                        'args': [arg.arg for arg in node.args.args],
                        'line': node.lineno
                    }
                    functions.append(func_info)
            
            elif isinstance(node, (ast.Import, ast.ImportFrom)):
                if isinstance(node, ast.Import):
                    for alias in node.names:
                        imports.append(alias.name)
                else:
                    module = node.module or ''
                    for alias in node.names:
                        imports.append(f"{module}.{alias.name}" if module else alias.name)
        
        # Get module docstring
        if tree.body and isinstance(tree.body[0], ast.Expr) and isinstance(tree.body[0].value, ast.Constant):
            docstring = tree.body[0].value.value
        
        # Build analysis
        purpose = docstring or "Python module"
        if classes:
            purpose += f" containing {len(classes)} class(es)"
        if functions:
            purpose += f" and {len(functions)} function(s)"
        
        public_surface = []
        for cls in classes:
            public_methods = [m for m in cls['methods'] if not m.startswith('_')]
            public_surface.append(f"Class {cls['name']}: {', '.join(public_methods) if public_methods else 'no public methods'}")
        
        for func in functions:
            if not func['name'].startswith('_'):
                args_str = ', '.join(func['args']) if func['args'] else 'no args'
                public_surface.append(f"Function {func['name']}({args_str})")
        
        dependencies = list(set(imports)) if imports else ["No external dependencies"]
        
        risks = []
        if any('eval' in content or 'exec' in content for content in [content]):
            risks.append("Uses eval/exec - potential security risk")
        if 'subprocess' in imports or 'os.system' in content:
            risks.append("Executes system commands")
        if not risks:
            risks = ["No obvious risks detected"]
        
        code_structure = []
        if classes:
            code_structure.append(f"Classes: {', '.join([c['name'] for c in classes])}")
        if functions:
            code_structure.append(f"Functions: {', '.join([f['name'] for f in functions])}")
        if not code_structure:
            code_structure = ["Simple script or module"]
        
        return {
            'purpose': purpose,
            'public_surface': '\n'.join(public_surface) or 'No public interface',
            'dependencies': '\n'.join(dependencies),
            'risks': '\n'.join(risks),
            'code_structure': '\n'.join(code_structure)
        }
    
    def _analyze_javascript_file(self, content: str, file_path: Path) -> Dict[str, str]:
        """Analyze JavaScript/TypeScript file using regex patterns."""
        # Extract functions
        function_patterns = [
            r'function\s+(\w+)\s*\(',
            r'const\s+(\w+)\s*=\s*\([^)]*\)\s*=>',
            r'let\s+(\w+)\s*=\s*\([^)]*\)\s*=>',
            r'var\s+(\w+)\s*=\s*function',
            r'(\w+)\s*:\s*function',
            r'async\s+function\s+(\w+)',
        ]
        
        functions = set()
        for pattern in function_patterns:
            matches = re.findall(pattern, content, re.MULTILINE)
            functions.update(matches)
        
        # Extract classes
        class_matches = re.findall(r'class\s+(\w+)', content)
        
        # Extract imports/requires
        import_patterns = [
            r'import.*?from\s+"([^"]*)"',
            r'import.*?from\s+\'([^\']*)\'',
            r'require\("([^"]*)"\)',
            r'require\(\'([^\']*)\'\')',
            r'import\("([^"]*)"\)',
            r'import\(\'([^\']*)\'\')'  
        ]
        
        imports = set()
        for pattern in import_patterns:
            matches = re.findall(pattern, content)
            imports.update(matches)
        
        # Extract exports
        export_patterns = [
            r'export\s+(?:default\s+)?(?:function\s+)?(\w+)',
            r'export\s*\{([^}]+)\}',
            r'module\.exports\s*=\s*(\w+)'
        ]
        
        exports = set()
        for pattern in export_patterns:
            matches = re.findall(pattern, content)
            for match in matches:
                if ',' in match:  # Handle export { a, b, c }
                    exports.update([name.strip() for name in match.split(',')])
                else:
                    exports.add(match)
        
        # Build analysis
        file_type = "TypeScript" if file_path.suffix in ['.ts', '.tsx'] else "JavaScript"
        purpose = f"{file_type} module"
        if class_matches:
            purpose += f" with {len(class_matches)} class(es)"
        if functions:
            purpose += f" and {len(functions)} function(s)"
        
        public_surface = []
        if exports:
            public_surface.extend([f"Export: {exp}" for exp in exports])
        if class_matches:
            public_surface.extend([f"Class: {cls}" for cls in class_matches])
        
        dependencies = list(imports) if imports else ["No external dependencies"]
        
        risks = []
        if 'eval(' in content:
            risks.append("Uses eval() - potential security risk")
        if 'innerHTML' in content:
            risks.append("Manipulates innerHTML - XSS risk")
        if 'document.write' in content:
            risks.append("Uses document.write - potential XSS")
        if not risks:
            risks = ["No obvious risks detected"]
        
        code_structure = []
        if class_matches:
            code_structure.append(f"Classes: {', '.join(class_matches)}")
        if functions:
            code_structure.append(f"Functions: {', '.join(list(functions)[:10])}{'...' if len(functions) > 10 else ''}")
        
        return {
            'purpose': purpose,
            'public_surface': '\n'.join(public_surface) or 'No public interface detected',
            'dependencies': '\n'.join(dependencies),
            'risks': '\n'.join(risks),
            'code_structure': '\n'.join(code_structure) or 'Simple script'
        }
    
    def _analyze_java_file(self, content: str, file_path: Path) -> Dict[str, str]:
        """Analyze Java file using regex patterns."""
        # Extract package
        package_match = re.search(r'package\s+([^;]+);', content)
        package = package_match.group(1) if package_match else None
        
        # Extract classes
        class_matches = re.findall(r'(?:public\s+)?(?:abstract\s+)?class\s+(\w+)', content)
        interface_matches = re.findall(r'(?:public\s+)?interface\s+(\w+)', content)
        
        # Extract methods
        method_pattern = r'(?:public|private|protected)\s+(?:static\s+)?(?:\w+\s+)+(\w+)\s*\('
        methods = re.findall(method_pattern, content)
        
        # Extract imports
        import_matches = re.findall(r'import\s+([^;]+);', content)
        
        purpose = "Java class"
        if package:
            purpose += f" in package {package}"
        
        public_surface = []
        if class_matches:
            public_surface.extend([f"Class: {cls}" for cls in class_matches])
        if interface_matches:
            public_surface.extend([f"Interface: {iface}" for iface in interface_matches])
        
        public_methods = [m for m in methods if not m.startswith('_')]
        if public_methods:
            public_surface.append(f"Methods: {', '.join(public_methods[:5])}{'...' if len(public_methods) > 5 else ''}")
        
        dependencies = import_matches if import_matches else ["No external dependencies"]
        
        risks = []
        if 'Runtime.exec' in content or 'ProcessBuilder' in content:
            risks.append("Executes system processes")
        if 'reflection' in content.lower() or 'Class.forName' in content:
            risks.append("Uses reflection - potential security risk")
        if not risks:
            risks = ["No obvious risks detected"]
        
        code_structure = []
        if class_matches or interface_matches:
            all_types = class_matches + interface_matches
            code_structure.append(f"Types: {', '.join(all_types)}")
        if methods:
            code_structure.append(f"Methods: {len(methods)} total")
        
        return {
            'purpose': purpose,
            'public_surface': '\n'.join(public_surface) or 'No public interface detected',
            'dependencies': '\n'.join(dependencies),
            'risks': '\n'.join(risks),
            'code_structure': '\n'.join(code_structure) or 'Simple Java file'
        }
    
    def _fallback_analysis(self, content: str, file_path: Path) -> Dict[str, str]:
        """Fallback analysis for unknown file types."""
        lines = content.split('\n')
        non_empty_lines = [line for line in lines if line.strip()]
        
        # Basic statistics
        total_lines = len(lines)
        code_lines = len(non_empty_lines)
        comment_lines = len([line for line in lines if line.strip().startswith(('#', '//', '/*', '*', '--'))])
        
        # Try to detect file type from extension
        ext = file_path.suffix.lower()
        file_type_map = {
            '.md': 'Markdown document',
            '.txt': 'Text file',
            '.json': 'JSON configuration',
            '.yaml': 'YAML configuration',
            '.yml': 'YAML configuration',
            '.xml': 'XML document',
            '.html': 'HTML document',
            '.css': 'CSS stylesheet',
            '.sql': 'SQL script',
            '.sh': 'Shell script',
            '.bat': 'Batch script',
            '.ps1': 'PowerShell script'
        }
        
        file_type = file_type_map.get(ext, f"File ({ext or 'no extension'})")
        
        purpose = f"{file_type} with {total_lines} lines"
        
        # Extract some key information
        first_lines = '\n'.join(lines[:3]).strip()
        if first_lines:
            purpose += f". Starts with: {first_lines[:100]}{'...' if len(first_lines) > 100 else ''}"
        
        public_surface = f"File contains {code_lines} lines of content"
        
        # Look for common patterns
        dependencies = []
        if ext in ['.json', '.yaml', '.yml']:
            dependencies.append("Configuration file - no code dependencies")
        elif ext == '.md':
            dependencies.append("Documentation file - no code dependencies")
        else:
            dependencies.append("Unknown dependencies")
        
        risks = ["Unknown file type - manual review recommended"]
        
        code_structure = [
            f"Total lines: {total_lines}",
            f"Non-empty lines: {code_lines}",
            f"Comment lines: {comment_lines}"
        ]
        
        return {
            'purpose': purpose,
            'public_surface': public_surface,
            'dependencies': '\n'.join(dependencies),
            'risks': '\n'.join(risks),
            'code_structure': '\n'.join(code_structure)
        }
    
    def analyze_file(self, file_path: Path, content: str) -> Dict[str, str]:
        """Analyze a file and extract structured information."""
        ext = file_path.suffix.lower()
        
        if ext == '.py':
            return self._analyze_python_file(content, file_path)
        elif ext in ['.js', '.jsx', '.ts', '.tsx']:
            return self._analyze_javascript_file(content, file_path)
        elif ext == '.java':
            return self._analyze_java_file(content, file_path)
        else:
            return self._fallback_analysis(content, file_path)
    
    def generate_summary(self, file_path: Path, content: str, recent_changes: str = "") -> str:
        """Generate a summary for a file."""
        analysis = self.analyze_file(file_path, content)
        
        # Add recent changes info
        if not recent_changes:
            recent_changes = "No recent changes tracked"
        
        # Fill template
        template = self._get_summary_template()
        summary = template.format(
            file_path=file_path.relative_to(self.project_root),
            purpose=analysis['purpose'],
            public_surface=analysis['public_surface'],
            dependencies=analysis['dependencies'],
            risks=analysis['risks'],
            recent_changes=recent_changes,
            code_structure=analysis['code_structure'],
            timestamp=datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        )
        
        return summary
    
    def save_summary(self, file_path: Path, summary: str) -> None:
        """Save summary to file."""
        relative_path = file_path.relative_to(self.project_root)
        summary_path = self.summaries_dir / f"{relative_path}.md"
        
        # Create directory structure
        summary_path.parent.mkdir(parents=True, exist_ok=True)
        
        try:
            with open(summary_path, 'w', encoding='utf-8') as f:
                f.write(summary)
            logger.debug(f"Saved summary for {relative_path}")
        except (OSError, PermissionError) as e:
            logger.error(f"Could not save summary for {file_path}: {e}")
    
    def summarize_file(self, file_path: Path, content: str, recent_changes: str = "") -> bool:
        """Generate and save summary for a file."""
        try:
            summary = self.generate_summary(file_path, content, recent_changes)
            self.save_summary(file_path, summary)
            return True
        except Exception as e:
            logger.error(f"Failed to summarize {file_path}: {e}")
            return False

def main():
    """Main entry point for summarizer script."""
    import argparse
    
    parser = argparse.ArgumentParser(description='Context Engine File Summarizer')
    parser.add_argument('file_path', help='Path to file to summarize')
    parser.add_argument('--output', help='Output summary file path')
    
    args = parser.parse_args()
    
    try:
        config = ContextConfig.load_or_create()
        summarizer = FileSummarizer(config)
        
        file_path = Path(args.file_path)
        if not file_path.exists():
            logger.error(f"File not found: {file_path}")
            return 1
        
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        summary = summarizer.generate_summary(file_path, content)
        
        if args.output:
            with open(args.output, 'w', encoding='utf-8') as f:
                f.write(summary)
            logger.info(f"Summary saved to {args.output}")
        else:
            print(summary)
        
        return 0
        
    except Exception as e:
        logger.error(f"Summarizer failed: {e}")
        return 1

if __name__ == '__main__':
    import sys
    sys.exit(main())