__version__ = "0.1.2"

import os
import cv2
import pandas as pd
import numpy as np
from sklearn.cluster import KMeans, AgglomerativeClustering
from tensorflow.keras.applications import VGG16
from tensorflow.keras.applications.vgg16 import preprocess_input

# Load pre-trained VGG16 model
model = VGG16(weights='imagenet', include_top=False, pooling='avg')

def load_and_preprocess_image(image_path):
    img = cv2.imread(image_path)
    img = cv2.resize(img, (224, 224))
    img = preprocess_input(img)
    return img

def extract_features(image_paths):
    features = []
    for image_path in image_paths:
        img = load_and_preprocess_image(image_path)
        img = np.expand_dims(img, axis=0)
        feature = model.predict(img)
        features.append(feature.flatten())
    return np.array(features)

def extract_visual_feature(image_path, output_excel_path, num_clusters=30, clustering_algorithm="kmeans"):
    image_paths = [os.path.join(image_path, img) for img in os.listdir(image_path) if img.endswith('.png')]
    features = extract_features(image_paths)

    if clustering_algorithm == "kmeans":
        cluster_model = KMeans(n_clusters=num_clusters, random_state=42)
    elif clustering_algorithm == "hierarchical":
        cluster_model = AgglomerativeClustering(n_clusters=num_clusters)
    else:
        raise ValueError("Invalid clustering algorithm. Choose 'kmeans' or 'hierarchical'.")

    clusters = cluster_model.fit_predict(features)

    df = pd.DataFrame({
        "ISIN": [os.path.basename(image_path) for image_path in image_paths],
        "Cluster": clusters
    })

    df.to_excel(output_excel_path, index=False)
    print(f"Results saved to {output_excel_path}")

if __name__ == "__main__":
    main()
