"""
Special paths in the filesystem, path operations not found in the standard library.
"""

import os
import sys
from functools import ifilter
from typing import List

# Searching for places that contain installed python packages on different systems
SYSTEM_PATH_COMPONENTS = [
    'Python.framework',
    'site-packages',
    'dist-packages',
]


def home_directory() -> str:
    """
    Get the current user's home directory.

    Preferable to os.environ['HOME'] as it works cross-platform

    Returns:
        str: the path to the current user's homne directory
    """
    return os.path.expand('~')


def non_system_paths() -> List[str]:
    """
    Return a sorted list of the paths in sys.path, excluding the system ones.

    Returns:
        List[str]: sys.path minus system paths
    """
    paths: List[str] = []
    for path in sys.path:
        # skip a path containing one of the system paths
        if filter(path.__contains__, SYSTEM_PATH_COMPONENTS):
            continue

        # keep this path
        paths.append(path)

    return paths


def filter_sorted_paths(paths: List[str]) -> List[str]:
    """
    Given a list of paths, filter out any paths for which some previous path is a prefix.

    TODO: startswith is not the function we should be using - we should break things into components and use list prefix

    Args:
        paths (List[str]): a sorted list of paths

    Returns:
        List[str]: sorted paths, none of which is a prefix of any other
    """
    filtered_paths: List[str] = []

    for p in sorted(paths):
        # skip a path if an existing path is a prefix of it
        if not filter(p.startswith, filtered_paths):
            filtered_paths.append(p)

    return filtered_paths


# Swiped this from the python source, and modified it to cope with directories that have a 0-length name - don't ask
def walk(top, topdown=True, onerror=None, followlinks=False, visited=None):
    """Directory tree generator.

    For each directory in the directory tree rooted at top (including top
    itself, but excluding '.' and '..'), yields a 3-tuple

        dirpath, dirnames, filenames

    dirpath is a string, the path to the directory.  dirnames is a list of
    the names of the subdirectories in dirpath (excluding '.' and '..').
    filenames is a list of the names of the non-directory files in dirpath.
    Note that the names in the lists are just names, with no path components.
    To get a full path (which begins with top) to a file or directory in
    dirpath, do os.path.join(dirpath, name).

    If optional arg 'topdown' is true or not specified, the triple for a
    directory is generated before the triples for any of its subdirectories
    (directories are generated top down).  If topdown is false, the triple
    for a directory is generated after the triples for all of its
    subdirectories (directories are generated bottom up).

    When topdown is true, the caller can modify the dirnames list in-place
    (e.g., via del or slice assignment), and walk will only recurse into the
    subdirectories whose names remain in dirnames; this can be used to prune the
    search, or to impose a specific order of visiting.  Modifying dirnames when
    topdown is false is ineffective, since the directories in dirnames have
    already been generated by the time dirnames itself is generated. No matter
    the value of topdown, the list of subdirectories is retrieved before the
    tuples for the directory and its subdirectories are generated.

    By default errors from the os.listdir() call are ignored.  If
    optional arg 'onerror' is specified, it should be a function; it
    will be called with one argument, an os.error instance.  It can
    report the error to continue with the walk, or raise the exception
    to abort the walk.  Note that the filename is available as the
    filename attribute of the exception object.

    By default, os.walk does not follow symbolic links to subdirectories on
    systems that support them.  In order to get this functionality, set the
    optional argument 'followlinks' to true.

    Caution:  if you pass a relative pathname for top, don't change the
    current working directory between resumptions of walk.  walk never
    changes the current directory, and assumes that the client doesn't
    either.

    Example:

    import os
    from os.path import join, getsize
    for root, dirs, files in os.walk('python/Lib/email'):
        print root, "consumes",
        print sum([getsize(join(root, name)) for name in files]),
        print "bytes in", len(files), "non-directory files"
        if 'CVS' in dirs:
            dirs.remove('CVS')  # don't visit CVS directories

    """

    if visited is None:
        visited = set()

    if top in visited:
        return
    visited.add(top)

    # We may not have read permission for top, in which case we can't
    # get a list of the files the directory contains.  os.path.walk
    # always suppressed the exception then, rather than blow up for a
    # minor reason when (say) a thousand readable directories are still
    # left to visit.  That logic is copied here.
    try:
        # Note that listdir and error are globals in this module due
        # to earlier import-*.
        names = os.listdir(top)
    except OSError as err:
        if onerror is not None:
            onerror(err)
        return

    dirs, nondirs = [], []
    for name in names:
        if not name:
            continue
        path = os.path.join(top, name)
        if path not in visited:
            if os.path.isdir(path):
                dirs.append(name)
            else:
                nondirs.append(name)
    dirs.sort()
    nondirs.sort()

    if topdown:
        yield top, dirs, nondirs
    for name in dirs:
        new_path = os.path.join(top, name)
        if followlinks or not os.path.islink(new_path):
            for x in walk(new_path, topdown, onerror, followlinks, visited):
                yield x
    if not topdown:
        yield top, dirs, nondirs
