"""
The theory here is to spell a phrase with chemical element abbreviations
"""

from collections import defaultdict
from typing import Dict, List, NamedTuple, Set

from ..utils.key_functions import len_then_content
from .alphabet import ALPHABET_LOWER


# A table of the elements scraped from Wikipedia
from_wikipedia: str = '''
|-
| 1 || style="background:{{element color|diatomic nonmetal}}" | H || [[Hydrogen]] || composed of the [[Greek language|Greek]] elements ''hydro-'' and ''-gen'' meaning 'water-forming' || 1 || 1 || {{sort|001|1.008}}{{ref|2|2}} {{ref|3|3}} {{ref|4|4}} {{ref|9|9}} || 0.00008988 || 14.01 || 20.28 || 14.304 || 2.20 || 1400
|-
| 2 || style="background:{{element color|noble gas}}" | He || [[Helium]] || the Greek ''helios'', 'sun' ||  18 || 1 ||  {{sort|004|4.002602(2)}}{{ref|2|2}} {{ref|4|4}} || 0.0001785 || 0.95{{ref|6|6}} || 4.22 || 5.193 || – || 0.008
|-
| 3 || style="background:{{element color|alkali metal}}" | Li || [[Lithium]] || the Greek ''lithos'', 'stone'||  1 || 2 ||  {{sort|007|6.94}}{{ref|2|2}} {{ref|3|3}} {{ref|4|4}} {{ref|5|5}} {{ref|9|9}} || 0.534 || 453.69 || 1560 || 3.582 || 0.98 || 20
|-
| 4 || style="background:{{element color|alkaline earth metal}}" | Be || [[Beryllium]] || [[beryl]], a mineral ||  2 || 2 || {{sort|009|9.0121831(5)}} || 1.85 || 1560 || 2742 || 1.825 ||  1.57 || 2.8
|-
| 5 || style="background:{{element color|metalloid}}" | B || [[Boron]] || [[borax]], a mineral ||  13 || 2 || {{sort|011|10.81{{ref|2|2}} {{ref|3|3}} {{ref|4|4}}}} {{ref|9|9}} || 2.34 || 2349 || 4200 || 1.026 ||  2.04 || 10
|-
| 6 || style="background:{{element color|polyatomic nonmetal}}" | C || [[Carbon]] || the [[Latin]] ''carbo'', 'coal'  || 14 || 2 || {{sort|012|12.011{{ref|2|2}} {{ref|4|4}}}} {{ref|9|9}} || 2.267 || 3800 || 4300 || 0.709 ||  2.55 || 200
|-
| 7 || style="background:{{element color|diatomic nonmetal}}" | N || [[Nitrogen]] || the Greek ''nitron'' and '-gen' meaning '[[niter]]-forming' || 15 ||  2 || {{sort|014|14.007{{ref|2|2}} {{ref|4|4}}}} {{ref|9|9}} || 0.0012506 || 63.15 || 77.36 || 1.04 ||  3.04 || 19
|-
| 8 || style="background:{{element color|diatomic nonmetal}}" | O || [[Oxygen]] || from the Greek ''oxy-'', both 'sharp' and 'acid', and ''-gen'', meaning 'acid-forming' || 16 || 2 || {{sort|016|15.999{{ref|2|2}} {{ref|4|4}}}} {{ref|9|9}} || 0.001429 || 54.36 || 90.20 || 0.918 ||  3.44 || 461000
|-
| 9 || style="background:{{element color|diatomic nonmetal}}" | F || [[Fluorine]] || the Latin ''fluere'', 'to flow' || 17 || 2 || {{sort|019|18.998403163(6)}} || 0.001696 || 53.53 || 85.03 || 0.824 ||  3.98 || 585
|-
| 10 || style="background:{{element color|noble gas}}" | Ne || [[Neon]] || the Greek ''neos'', meaning 'new' || 18 || 2 || {{sort|020|20.1797(6){{ref|2|2}} {{ref|3|3}}}} || 0.0008999 || 24.56 ||  27.07 || 1.03 || – || 0.005
|-
| 11 || style="background:{{element color|alkali metal}}" | Na || [[Sodium]] || the [[English language|English]] word ''soda'' (''natrium'' in Latin)<ref name=innvista>[http://www.innvista.com/chemistry/elements/etymology-of-elements/]</ref> || 1 || 3 || {{sort|023|22.98976928(2)}} || 0.971 || 370.87 || 1156 || 1.228 || 0.93 || 23600
|-
| 12 || style="background:{{element color|alkaline earth metal}}" | Mg || [[Magnesium]] || [[Magnesia (regional unit)|Magnesia]], a district of Eastern [[Thessaly]] in [[Greece]] || 2 || 3 || {{sort|024|24.305}}{{ref|9|9}} || 1.738 || 923 || 1363 || 1.023 || 1.31 || 23300
|-
| 13 || style="background:{{element color|post-transition metal}}" | Al || [[Aluminium]] || from [[aluminium oxide|''alumina'']], a compound (originally ''aluminum'') || 13 || 3 || {{sort|027|26.9815385(7)}} || 2.698 || 933.47<ref name=autm/> || 2792 || 0.897 || 1.61 || 82300
|-
| 14 || style="background:{{element color|metalloid}}" | Si || [[Silicon]] || from the Latin ''silex'', 'flint' (originally ''silicium'') || 14 || 3 || {{sort|028|28.085}}{{ref|4|4}} {{ref|9|9}} || 2.3296 || 1687 || 3538 || 0.705 || 1.9 || 282000
|-
| 15 || style="background:{{element color|polyatomic nonmetal}}" | P || [[Phosphorus]] || the Greek ''phoosphoros'', 'carrying light' || 15 || 3 || {{sort|031|30.973761998(5)}} || 1.82 || 317.30 || 550 || 0.769 || 2.19 || 1050
|-
| 16 || style="background:{{element color|polyatomic nonmetal}}" | S || [[Sulfur]] || the Latin ''sulphur'', 'fire and brimstone'<ref>{{Cite web|url=http://www.etymonline.com/index.php?term=sulfur|title=Online Etymology Dictionary|website=www.etymonline.com|access-date=2016-06-18}}</ref> || 16 || 3 || {{sort|032|32.06}}{{ref|2|2}} {{ref|4|4}} {{ref|9|9}} || 2.067 || 388.36 || 717.87 || 0.71 || 2.58 || 350
|-
| 17 || style="background:{{element color|diatomic nonmetal}}" | Cl || [[Chlorine]] || the Greek ''chloros'', 'greenish yellow' || 17 || 3 || {{sort|035|35.45}}{{ref|2|2}} {{ref|3|3}} {{ref|4|4}} {{ref|9|9}} || 0.003214 || 171.6 || 239.11 || 0.479 || 3.16 || 145
|-
| 18 || style="background:{{element color|noble gas}}" | Ar || [[Argon]] || the Greek ''argos'', 'idle' || 18 || 3 || {{sort|040|39.948(1)}}{{ref|2|2}} {{ref|4|4}} || 0.0017837 || 83.80 || 87.30 || 0.52 || – || 3.5
|-
| 19 || style="background:{{element color|alkali metal}}" | K || [[Potassium]] || [[New Latin]] ''potassa'', 'potash' (''kalium'' in Latin)<ref name=innvista/> || 1 || 4 || {{sort|039|39.0983(1)}} || 0.862 || 336.53 || 1032 || 0.757 || 0.82 || 20900
|-
| 20 || style="background:{{element color|alkaline earth metal}}" | Ca || [[Calcium]] || the Latin ''calx'', 'lime' || 2 || 4 || {{sort|041|40.078(4)}}{{ref|2|2}} || 1.54 || 1115 || 1757 || 0.647 || 1 || 41500
|-
| 21 || style="background:{{element color|transition metal}}" | Sc || [[Scandium]] || ''Scandia'', the Latin name for [[Scandinavia]] || 3 || 4 || {{sort|045|44.955908(5)}} || 2.989 || 1814 || 3109 || 0.568 || 1.36 || 22
|-
| 22 || style="background:{{element color|transition metal}}" | Ti || [[Titanium]] || [[Titan (mythology)|Titans]], the sons of the Earth goddess of Greek mythology || 4 || 4 || {{sort|048|47.867(1)}} || 4.54 || 1941 || 3560 || 0.523 || 1.54 || 5650
|-
| 23 || style="background:{{element color|transition metal}}" | V || [[Vanadium]] || [[List of names of Freyja|Vanadis]], an [[Old Norse]] name for the Scandinavian goddess [[Freyja]] || 5 || 4 || {{sort|051|50.9415(1)}} || 6.11 || 2183 || 3680 || 0.489 || 1.63 || 120
|-
| 24 || style="background:{{element color|transition metal}}" | Cr || [[Chromium]] || the Greek ''chroma'', 'color' || 6 || 4 || {{sort|052|51.9961(6)}} || 7.15 || 2180 || 2944 || 0.449 || 1.66 || 102
|-
| 25 || style="background:{{element color|transition metal}}" | Mn || [[Manganese]] || corrupted from ''magnesia negra'', see Magnesium ||7 || 4 || {{sort|055|54.938044(3)}} || 7.44 || 1519 || 2334 || 0.479 || 1.55 || 950
|-
| 26 || style="background:{{element color|transition metal}}" | Fe || [[Iron]] || English word (''ferrum'' in Latin) || 8 || 4 || {{sort|056|55.845(2)}} || 7.874 || 1811 || 3134 || 0.449 || 1.83 || 56300
|-
| 27 || style="background:{{element color|transition metal}}" | Co || [[Cobalt]] || the [[German language|German]] word ''Kobold'', 'goblin' || 9 || 4 || {{sort|059|58.933194(4)}} || 8.86 || 1768 || 3200 || 0.421 || 1.88 || 25
|-
| 28 || style="background:{{element color|transition metal}}" | Ni || [[Nickel]] || from [[Swedish language|Swedish]] ''kopparnickel'', containing the [[German language|German]] word ''Nickel'', 'goblin' || 10 || 4 ||  {{sort|058|58.6934(4)}} || 8.912 || 1728 || 3186 || 0.444 || 1.91 || 84
|-
| 29 || style="background:{{element color|transition metal}}" | Cu || [[Copper]] || English word (Latin ''cuprum'') || 11 || 4 || {{sort|064|63.546(3)}}{{ref|4|4}} || 8.96 || 1357.77<ref name=autm/> || 2835 || 0.385 || 1.9 || 60
|-
| 30 || style="background:{{element color|transition metal}}" | Zn || [[Zinc]] || the German ''Zink'' ||  12 || 4 || {{sort|065|65.38(2)}} || 7.134 || 692.88 || 1180 || 0.388 || 1.65 || 70
|-
| 31 || style="background:{{element color|post-transition metal}}" | Ga || [[Gallium]] || ''Gallia'', the Latin name for [[France]] ||13 || 4 || {{sort|070|69.723(1)}} || 5.907 || 302.9146 || 2673 || 0.371 || 1.81 || 19
|-
| 32 || style="background:{{element color|metalloid}}" | Ge || [[Germanium]] || ''Germania'', the Latin name for [[Germany]] || 14 || 4 || {{sort|073|72.630(8)}} || 5.323 || 1211.40 || 3106 || 0.32 || 2.01 || 1.5
|-
| 33 || style="background:{{element color|metalloid}}" | As || [[Arsenic]] || English word (Latin ''arsenicum'') || 15 || 4 || {{sort|075|74.921595(6)}} || 5.776 || {{sort|1090|1090 {{ref|7|7}}}} || 887 || 0.329 || 2.18  || 1.8
|-
| 34 || style="background:{{element color|polyatomic nonmetal}}" | Se || [[Selenium]] || the Greek ''selene'', 'moon' || 16 || 4 || {{sort|079|78.971(8)}}{{ref|4|4}} || 4.809 || 453 || 958 || 0.321 || 2.55 || 0.05
|-
| 35 || style="background:{{element color|diatomic nonmetal}}" | Br || [[Bromine]] || the Greek ''bromos'', 'stench' ||  17 || 4 || {{sort|080|79.904}}{{ref|9|9}} || 3.122 || 265.8 || 332.0 || 0.474 || 2.96  || 2.4
|-
| 36 || style="background:{{element color|noble gas}}" | Kr || [[Krypton]] || the Greek ''kryptos'', 'hidden' || 18 || 4 || {{sort|084|83.798(2)}}{{ref|2|2}} {{ref|3|3}} || 0.003733 || 115.79 || 119.93 || 0.248 || 3 || {{sort|.0001|1×10<sup>−4</sup>}}
|-
| 37 || style="background:{{element color|alkali metal}}" | Rb || [[Rubidium]] || the Latin ''rubidus'', 'deep red' || 1 || 5 || {{sort|085|85.4678(3)}}{{ref|2|2}} || 1.532 || 312.46 || 961 || 0.363 || 0.82  || 90
|-
| 38 || style="background:{{element color|alkaline earth metal}}" | Sr || [[Strontium]] || [[Strontian]], a small town in [[Scotland]] ||  2 || 5 || {{sort|087|87.62(1)}}{{ref|2|2}} {{ref|4|4}} || 2.64 || 1050 || 1655 || 0.301 ||  0.95 || 370
|-
| 39 || style="background:{{element color|transition metal}}" | Y || [[Yttrium]] || [[Ytterby]], [[Sweden]] || 3 || 5 || {{sort|089|88.90584(2)}} || 4.469 || 1799 || 3609 || 0.298 ||  1.22 || 33
|-
| 40 || style="background:{{element color|transition metal}}" | Zr || [[Zirconium]] || Persian ''Zargun'', 'gold-colored'; German ''Zirkoon'', '[[jargoon]]' ||  4 || 5 || {{sort|091|91.224(2)}}{{ref|2|2}} || 6.506 || 2128 || 4682 || 0.278 || 1.33  ||165
|-
| 41 || style="background:{{element color|transition metal}}" | Nb || [[Niobium]] || [[Niobe]], daughter of king [[Tantalus]] from Greek mythology || 5 || 5 || {{sort|093|92.90637(2)}} || 8.57 || 2750 || 5017 || 0.265 || 1.6 || 20
|-
| 42 || style="background:{{element color|transition metal}}" | Mo || [[Molybdenum]] || the Greek ''molybdos'' meaning 'lead' || 6 || 5 || {{sort|096|95.95(1)}}{{ref|2|2}} || 10.22 || 2896 || 4912 || 0.251 || 2.16 || 1.2
|-
| 43 || style="background:{{element color|transition metal}}" | Tc || [[Technetium]] || the Greek ''tekhnètos'' meaning 'artificial' || 7 || 5 || {{sort|098|[98]{{ref|1|1}}}} || 11.5 || 2430 || 4538 || – || 1.9 || {{sort|.000000003|~&nbsp;3×10<sup>−9</sup>}}
|-
| 44 || style="background:{{element color|transition metal}}" | Ru || [[Ruthenium]] || ''Ruthenia'', the [[New Latin]] name for [[Russia]] || 8 || 5 || 101.07(2){{ref|2|2}} || 12.37 || 2607 || 4423 || 0.238 || 2.2 || 0.001
|-
| 45 || style="background:{{element color|transition metal}}" | Rh || [[Rhodium]] || the Greek ''rhodos'', meaning 'rose coloured' ||  9 || 5 || 102.90550(2) || 12.41 || 2237 || 3968 || 0.243 || 2.28 || 0.001
|-
| 46 || style="background:{{element color|transition metal}}" | Pd || [[Palladium]] || the then recently discovered asteroid [[2 Pallas|Pallas]], considered a planet at the time || 10 || 5 || 106.42(1){{ref|2|2}} || 12.02 || 1828.05 || 3236 || 0.244 || 2.2 || 0.015
|-
| 47 || style="background:{{element color|transition metal}}" | Ag || [[Silver]] || English word (''argentum'' in Latin)<ref name=innvista/> || 11 || 5 || 107.8682(2){{ref|2|2}} || 10.501 || 1234.93<ref name=autm/> || 2435 || 0.235 || 1.93 || 0.075
|-
| 48 || style="background:{{element color|transition metal}}" | Cd || [[Cadmium]] || the [[New Latin]] ''cadmia'', from King [[Kadmos]] || 12 || 5 || 112.414(4){{ref|2|2}} || 8.69 || 594.22 || 1040 || 0.232 || 1.69 || 0.159
|-
| 49 || style="background:{{element color|post-transition metal}}" | In || [[Indium]] || ''[[indigo]]'' || 13 || 5 || 114.818(1) || 7.31 || 429.75 || 2345 || 0.233 || 1.78 || 0.25
|-
| 50 || style="background:{{element color|post-transition metal}}" | Sn || [[Tin]] || English word (''stannum'' in Latin) || 14 || 5 || 118.710(7){{ref|2|2}} || 7.287 || 505.08 || 2875 || 0.228 || 1.96 || 2.3
|-
| 51 || style="background:{{element color|metalloid}}" | Sb || [[Antimony]] || composed from the Greek ''anti'', 'against', and ''monos'', 'alone' (''stibium'' in Latin) || 15 || 5 || 121.760(1){{ref|2|2}} || 6.685 || 903.78 || 1860 || 0.207 || 2.05 || 0.2
|-
| 52 || style="background:{{element color|metalloid}}" | Te || [[Tellurium]] || Latin ''tellus'', 'earth' || 16 || 5 || 127.60(3){{ref|2|2}} || 6.232 || 722.66 || 1261 || 0.202 || 2.1 || 0.001
|-
| 53 || style="background:{{element color|diatomic nonmetal}}" | I || [[Iodine]] || French ''iode'' (after the Greek ''ioeides'', 'violet') || 17 || 5 || 126.90447(3) || 4.93 || 386.85 || 457.4 || 0.214 || 2.66 || 0.45
|-
| 54 || style="background:{{element color|noble gas}}" | Xe || [[Xenon]] || the Greek ''xenos'', 'strange' || 18 || 5 || 131.293(6){{ref|2|2}} {{ref|3|3}} || 0.005887 || 161.4 || 165.03 || 0.158 || 2.6 || {{sort|.00003|3×10<sup>−5</sup>}}
|-
| 55 || style="background:{{element color|alkali metal}}" | Cs || [[Caesium]] || the Latin ''caesius'', 'sky blue' || 1 || 6 || 132.90545196(6) || 1.873 || 301.59 || 944 || 0.242 || 0.79 || 3
|-
| 56 || style="background:{{element color|alkaline earth metal}}" | Ba || [[Barium]] || the Greek ''barys'', 'heavy' || 2 || 6 || 137.327(7) || 3.594 || 1000 || 2170 || 0.204 || 0.89 || 425
|-
| 57 || style="background:{{element color|lanthanide}}" | La || [[Lanthanum]] || the Greek ''lanthanein'', 'to lie hidden' || 3 || 6 || 138.90547(7){{ref|2|2}} || 6.145 || 1193 || 3737 || 0.195 || 1.1 || 39
|-
| 58 || style="background:{{element color|lanthanide}}" | Ce || [[Cerium]] || the then recently discovered asteroid [[Ceres (dwarf planet)|Ceres]], considered a planet at the time ||  || 6 || 140.116(1){{ref|2|2}} || 6.77 || 1068 || 3716 || 0.192 || 1.12 || 66.5
|-
| 59 || style="background:{{element color|lanthanide}}" | Pr || [[Praseodymium]] || the Greek ''praseios didymos'' meaning 'green twin' || || 6 || 140.90766(2) || 6.773 || 1208 || 3793 || 0.193 || 1.13 || 9.2
|-
| 60 || style="background:{{element color|lanthanide}}" | Nd || [[Neodymium]] || the Greek ''neos didymos'' meaning 'new twin' ||  || 6 || 144.242(3){{ref|2|2}} || 7.007 || 1297 || 3347 || 0.19 || 1.14 || 41.5
|-
| 61 || style="background:{{element color|lanthanide}}" | Pm || [[Promethium]] || [[Prometheus]] of Greek mythology who stole fire from the Gods and gave it to humans ||  || 6 || {{sort|145|[145]{{ref|1|1}}}} || 7.26 || 1315 || 3273 || – || 1.13 || {{sort|.0000000000000000002|2×10<sup>−19</sup>}}
|-
| 62 || style="background:{{element color|lanthanide}}" | Sm || [[Samarium]] || [[Samarskite]], the name of the mineral from which it was first isolated ||  || 6 || 150.36(2){{ref|2|2}} || 7.52 || 1345 || 2067 || 0.197 ||  1.17 || 7.05
|-
| 63 || style="background:{{element color|lanthanide}}" | Eu || [[Europium]] || [[Europe]] || || 6 || 151.964(1){{ref|2|2}} || 5.243 || 1099 || 1802 || 0.182 || 1.2 || 2
|-
| 64 || style="background:{{element color|lanthanide}}" | Gd || [[Gadolinium]] || [[Johan Gadolin]], chemist, physicist and mineralogist || || 6 || 157.25(3){{ref|2|2}} || 7.895 || 1585 || 3546 || 0.236 ||  1.2 || 6.2
|-
| 65 || style="background:{{element color|lanthanide}}" | Tb || [[Terbium]] || [[Ytterby]], Sweden || || 6 || 158.92535(2) || 8.229 || 1629 || 3503 || 0.182 ||  1.2 || 1.2
|-
| 66 || style="background:{{element color|lanthanide}}" | Dy || [[Dysprosium]] || the Greek ''dysprositos'', 'hard to get' ||  || 6 || 162.500(1){{ref|2|2}} || 8.55 || 1680 || 2840 || 0.17 || 1.22 || 5.2
|-
| 67 || style="background:{{element color|lanthanide}}" | Ho || [[Holmium]] || ''Holmia'', the [[New Latin]] name for [[Stockholm]] || ||6 || 164.93033(2) || 8.795 || 1734 || 2993 || 0.165 ||  1.23 || 1.3
|-
| 68 || style="background:{{element color|lanthanide}}" | Er || [[Erbium]] || [[Ytterby]], Sweden || ||6 || 167.259(3){{ref|2|2}} || 9.066 || 1802 || 3141 || 0.168 ||  1.24 || 3.5
|-
| 69 || style="background:{{element color|lanthanide}}" | Tm || [[Thulium]] || [[Thule]], the ancient name for Scandinavia || ||6 || 168.93422(2) || 9.321 || 1818 || 2223 || 0.16 ||  1.25 || 0.52
|-
| 70 || style="background:{{element color|lanthanide}}" | Yb || [[Ytterbium]] || [[Ytterby]], Sweden || ||6 || 173.045(10){{ref|2|2}} || 6.965 || 1097 || 1469 || 0.155 ||  1.1 || 3.2
|-
| 71 || style="background:{{element color|lanthanide}}" | Lu || [[Lutetium]] || ''Lutetia'', the Latin name for [[Paris]] || || 6 || 174.9668(1){{ref|2|2}} || 9.84 || 1925 || 3675 || 0.154 ||  1.27 || 0.8
|-
| 72 || style="background:{{element color|transition metal}}" | Hf || [[Hafnium]] || ''Hafnia'', the New Latin name for [[Copenhagen]] || 4 || 6 || 178.49(2) || 13.31 || 2506 || 4876 || 0.144 ||  1.3 || 3
|-
| 73 || style="background:{{element color|transition metal}}" | Ta || [[Tantalum]] || King [[Tantalus]], father of Niobe from Greek mythology || 5 || 6 || 180.94788(2) || 16.654 || 3290 || 5731 || 0.14 ||  1.5 || 2
|-
| 74 || style="background:{{element color|transition metal}}" | W || [[Tungsten]] || the Swedish ''tung sten'', 'heavy stone' (W is ''wolfram'', the old name of the tungsten mineral wolframite)<ref name=innvista/> ||  6 || 6 || 183.84(1) || 19.25 || 3695 || 5828 || 0.132 ||  2.36 || 1.3
|-
| 75 || style="background:{{element color|transition metal}}" | Re || [[Rhenium]] || ''Rhenus'', the Latin name for the river [[Rhine]] || 7 || 6 || 186.207(1) || 21.02 || 3459 || 5869 || 0.137 ||  1.9 || {{sort|.0007|7×10<sup>−4</sup>}}
|-
| 76 || style="background:{{element color|transition metal}}" | Os || [[Osmium]] || the Greek ''osmè'', meaning 'smell'||  8 || 6 || 190.23(3){{ref|2|2}} || 22.61 || 3306 || 5285 || 0.13 ||  2.2 || 0.002
|-
| 77 || style="background:{{element color|transition metal}}" | Ir || [[Iridium]] || [[Iris (mythology)|Iris]], the Greek goddess of the rainbow || 9 || 6 || 192.217(3) || 22.56 || 2719 || 4701 || 0.131 || 2.2 || 0.001
|-
| 78 || style="background:{{element color|transition metal}}" | Pt || [[Platinum]] || the [[Spanish language|Spanish]] ''platina'', meaning 'little silver'|| 10 || 6 || 195.084(9) || 21.46 || 2041.4<ref name=autm/> || 4098 || 0.133 || 2.28 || 0.005
|-
| 79 || style="background:{{element color|transition metal}}" | Au || [[Gold]] || English word (''aurum'' in Latin) || 11 || 6 || 196.966569(5) || 19.282 || 1337.33<ref name=autm>[http://www.jstor.org/stable/20020628 Holman, Lawrence and Barr]</ref> || 3129 || 0.129 || 2.54 || 0.004
|-
| 80 || style="background:{{element color|transition metal}}" | Hg || [[Mercury (element)|Mercury]] || the [[New Latin]] name ''mercurius'', named after the [[Mercury_(mythology)|Roman god]] (Hg from former name ''hydrargyrum'', from Greek ''hydr-'', 'water', and ''argyros'', 'silver')|| 12 || 6 || 200.592(3) || 13.5336 || 234.43 || 629.88 || 0.14 || 2 || 0.085
|-
| 81 || style="background:{{element color|post-transition metal}}" | Tl || [[Thallium]] || the Greek ''thallos'', 'green twig' || 13 || 6 || 204.38{{ref|9|9}} || 11.85 || 577 || 1746 || 0.129 ||  1.62 || 0.85
|-
| 82 || style="background:{{element color|post-transition metal}}" | Pb || [[Lead]] || English word (''plumbum'' in Latin)<ref name=innvista/>|| 14 || 6 || 207.2(1){{ref|2|2}} {{ref|4|4}} || 11.342 || 600.61 || 2022 || 0.129 ||  1.87 || 14
|-
| 83 || style="background:{{element color|post-transition metal}}" | Bi || [[Bismuth]] || German word, now obsolete || 15 || 6 || 208.98040(1){{ref|1|1}} || 9.807 || 544.7 || 1837 || 0.122 ||  2.02 || 0.009
|-
| 84 || style="background:{{element color|post-transition metal}}" | Po || [[Polonium]] || Named after the home country of [[Marie Curie]]([[Poland]]), who is also the discoverer of [[Radium]] ||  16 || 6 || {{sort|209|[209]}}{{ref|1|1}} || 9.32 || 527 || 1235 || – ||  2.0 || {{sort|.0000000002|2×10<sup>−10</sup>}}
|-
| 85 || style="background:{{element color|metalloid}}" | At || [[Astatine]] || the Greek ''astatos'', 'unstable' ||  17 || 6 || {{sort|210|[210]}}{{ref|1|1}} || 7 || 575 || 610 || – ||  2.2 || {{sort|0.000000000000000000003|3×10<sup>−20</sup>}}
|-
| 86 || style="background:{{element color|noble gas}}" | Rn || [[Radon]] || From ''radium'', as it was first detected as an emission from radium during radioactive decay || 18 || 6 || {{sort|222|[222]}}{{ref|1|1}} || 0.00973 || 202 || 211.3 || 0.094 || 2.2 || {{sort|.0000000000004|4×10<sup>−13</sup>}}
|-
| 87 || style="background:{{element color|alkali metal}}" | Fr || [[Francium]] || ''Francia'', the [[New Latin]] name for [[France]] || 1 || 7 || {{sort|223|[223]{{ref|1|1}}}}  || 1.87 || 300 || 950 || – ||  0.7 || {{sort|0.000000000000000001|~&nbsp;1×10<sup>−18</sup>}}
|-
| 88 || style="background:{{element color|alkaline earth metal}}" | Ra || [[Radium]] || the Latin ''radius'', 'ray' || 2 || 7 || {{sort|223|[226]}}{{ref|1|1}} || 5.5 || 973 || 2010 || 0.094 ||  0.9 || {{sort|.0000009|9×10<sup>−7</sup>}}
|-
| 89 || style="background:{{element color|actinide}}" | Ac || [[Actinium]] || the Greek ''aktis'', 'ray'|| 3 || 7 || {{sort|227|[227]}}{{ref|1|1}} || 10.07 || 1323 || 3471 || 0.12 ||  1.1 || {{sort|.00000000055|5.5×10<sup>−10</sup>}}
|-
| 90 || style="background:{{element color|actinide}}" | Th || [[Thorium]] || [[Thor]], the Scandinavian god of thunder ||   || 7 || 232.0377(4){{ref|1|1}} {{ref|2|2}} || 11.72 || 2115 || 5061 || 0.113 ||  1.3 || 9.6
|-
| 91 || style="background:{{element color|actinide}}" | Pa || [[Protactinium]] || the Greek ''protos'', 'first', and [[actinium]], which is produced through the radioactive decay of protactinium ||   || 7 || 231.03588(2){{ref|1|1}} || 15.37 || 1841 || 4300 || – || 1.5 || {{sort|.0000014|1.4×10<sup>−6</sup>}}
|-
| 92 || style="background:{{element color|actinide}}" | U || [[Uranium]] || [[Uranus]], the seventh planet in the Solar System || || 7 || 238.02891(3){{ref|1|1}} || 18.95 || 1405.3 || 4404 || 0.116 || 1.38 || 2.7
|-
| 93 || style="background:{{element color|actinide}}" | Np || [[Neptunium]] || [[Neptune]], the eighth planet in the Solar System || || 7 || {{sort|237|[237]}}{{ref|1|1}} || 20.45 || 917 || 4273 || – || 1.36 || {{sort|0.000000000003|≤&nbsp;3×10<sup>−12</sup>}}
|-
| 94 || style="background:{{element color|actinide}}" | Pu || [[Plutonium]] || [[Pluto]], a dwarf planet in the Solar System (then considered the ninth planet) || || 7 || {{sort|244|[244]}}{{ref|1|1}} || 19.84 || 912.5 || 3501 || – || 1.28 || {{sort|0.00000000003|≤&nbsp;3×10<sup>−11</sup>}}
|-
| 95 || style="background:{{element color|actinide}}" | Am || [[Americium]] || [[The Americas]], as the element was first synthesized on the continent, by analogy with europium ||  || 7 || {{sort|243|[243]}}{{ref|1|1}} || 13.69 || 1449 || 2880 || – || 1.13 || {{sort|0|0}} {{ref|8|8}}
|-
| 96 || style="background:{{element color|actinide}}" | Cm || [[Curium]] || [[Pierre Curie]], a physicist, and [[Marie Curie]], a physicist and chemist, named after great scientists by analogy with gadolinium || || 7 || {{sort|247|[247]}}{{ref|1|1}} || 13.51 || 1613 || 3383 || – || 1.28 || {{sort|0|0}} {{ref|8|8}}
|-
| 97 || style="background:{{element color|actinide}}" | Bk || [[Berkelium]] || [[Berkeley, California]], where the element was first synthesized, by analogy with terbium || || 7 || {{sort|247|[247]}}{{ref|1|1}} || 14.79 || 1259 || 2900 || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 98 || style="background:{{element color|actinide}}" | Cf || [[Californium]] || [[State of California|California]], where the element was first synthesized || || 7 || {{sort|251|[251]}}{{ref|1|1}} || 15.1 || 1173 || {{sort|1743|(1743)}}{{ref|11|11}} || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 99 || style="background:{{element color|actinide}}" | Es || [[Einsteinium]] || [[Albert Einstein]], physicist || || 7 || {{sort|252|[252]}}{{ref|1|1}} || 8.84 || 1133 || {{sort|1269|(1269)}}{{ref|11|11}} || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 100 || style="background:{{element color|actinide}}" | Fm || [[Fermium]] || [[Enrico Fermi]], physicist || || 7 || {{sort|257|[257]}}{{ref|1|1}} || {{sort|9.7|(9.7)}}{{ref|11|11}} || {{sort|1125|(1125)}}{{ref|11|11}} || – || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 101 || style="background:{{element color|actinide}}" | Md || [[Mendelevium]] || [[Dmitri Mendeleev]], chemist and inventor || ||7 || {{sort|258|[258]}}{{ref|1|1}} || {{sort|10.3|(10.3)}}{{ref|11|11}} || {{sort|1100|(1100)}}{{ref|11|11}} || – || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 102 || style="background:{{element color|actinide}}" | No || [[Nobelium]] || [[Alfred Nobel]], chemist, engineer, innovator, and armaments manufacturer || ||7 || {{sort|259|[259]}}{{ref|1|1}} || {{sort|9.9|(9.9)}}{{ref|11|11}} || {{sort|1100|(1100)}}{{ref|11|11}} || – || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 103 || style="background:{{element color|actinide}}" | Lr || [[Lawrencium]] || [[Ernest O. Lawrence]], physicist  || || 7 || {{sort|266|[266]}}{{ref|1|1}} || {{sort|15.6|(15.6)}}{{ref|11|11}} || {{sort|1900|(1900)}}{{ref|11|11}} || – || – || 1.3 || {{sort|0|0}} {{ref|8|8}}
|-
| 104 || style="background:{{element color|transition metal}}" | Rf || [[Rutherfordium]] || [[Ernest Rutherford]], chemist and physicist || 4 || 7 || {{sort|267|[267]}}{{ref|1|1}} || {{sort|23.2|(23.2)}}{{ref|11|11}} || {{sort|2400|(2400)}}{{ref|11|11}} || {{sort|5800|(5800)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 105 || style="background:{{element color|transition metal}}" | Db || [[Dubnium]] || [[Dubna]], Russia || 5 || 7 || {{sort|268|[268]}}{{ref|1|1}} || {{sort|29.3|(29.3)}}{{ref|11|11}} || –|| – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 106 || style="background:{{element color|transition metal}}" | Sg || [[Seaborgium]] || [[Glenn T. Seaborg]], scientist || 6 || 7 || {{sort|269|[269]}}{{ref|1|1}} || {{sort|35.0|(35.0)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 107 || style="background:{{element color|transition metal}}" | Bh || [[Bohrium]] || [[Niels Bohr]], physicist || 7 || 7 || {{sort|270|[270]}}{{ref|1|1}} || {{sort|37.1|(37.1)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 108 || style="background:{{element color|transition metal}}" | Hs || [[Hassium]] || [[Hesse]], Germany, where the element was first synthesized || 8 || 7 || {{sort|277|[277]}}{{ref|1|1}} || {{sort|40.7|(40.7)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 109 || style="background:{{element color|unknown chemical properties}}" | Mt || [[Meitnerium]] || [[Lise Meitner]], physicist || 9 || 7 || {{sort|278|[278]}}{{ref|1|1}} || {{sort|37.4|(37.4)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 110 || style="background:{{element color|unknown chemical properties}}" | Ds || [[Darmstadtium]] || [[Darmstadt]], Germany, where the element was first synthesized || 10 || 7 || {{sort|281|[281]}}{{ref|1|1}} || {{sort|34.8|(34.8)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 111 || style="background:{{element color|unknown chemical properties}}" | Rg || [[Roentgenium]] || [[Wilhelm Conrad Röntgen]], physicist || 11 || 7 || {{sort|282|[282]}}{{ref|1|1}} || {{sort|28.7|(28.7)}}{{ref|11|11}} || – || – || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 112 || style="background:{{element color|transition metal}}" | Cn || [[Copernicium]] || [[Nicolaus Copernicus]], astronomer || 12 || 7 || {{sort|285|[285]}}{{ref|1|1}} || {{sort|23.7|(23.7)}}{{ref|11|11}} || – || {{sort|357|357}} {{ref|12|12}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 113 || style="background:{{element color|unknown chemical properties}}" | Nh || [[Nihonium]] || the [[Japanese language|Japanese]] name for [[Japan]], ''Nihon'', where the element was first synthesized || 13 || 7 || {{sort|286|[286]}}{{ref|1|1}} || {{sort|16|(16)}}{{ref|11|11}} || {{sort|700|(700)}}{{ref|11|11}} || {{sort|1400|(1400)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 114 || style="background:{{element color|post-transition metal}}" | Fl || [[Flerovium]] || [[Flerov Laboratory of Nuclear Reactions]], part of [[Joint Institute for Nuclear Research|JINR]] where the element was synthesized; itself named for [[Georgy Flyorov]], physicist || 14 || 7 || {{sort|289|[289]}}{{ref|1|1}} || {{sort|14|(14)}}{{ref|11|11}} || {{sort|340|(340)}}{{ref|11|11}} || {{sort|420|(420)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 115 || style="background:{{element color|unknown chemical properties}}" | Mc || [[Moscovium]] || [[Moscow Oblast]], Russia, where the element was first synthesized || 15 || 7 || {{sort|290|[290]}}{{ref|1|1}} || {{sort|13.5|(13.5)}}{{ref|11|11}} || {{sort|700|(700)}}{{ref|11|11}} || {{sort|1400|(1400)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 116 || style="background:{{element color|unknown chemical properties}}" | Lv || [[Livermorium]] || [[Lawrence Livermore National Laboratory]] (in [[Livermore, California]]) which collaborated with [[Joint Institute for Nuclear Research|JINR]] on its synthesis || 16 || 7 || {{sort|293|[293]}}{{ref|1|1}} || {{sort|12.9|(12.9)}}{{ref|11|11}} || {{sort|709|(709)}}{{ref|11|11}} || {{sort|1085|(1085)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 117 || style="background:{{element color|unknown chemical properties}}" | Ts || [[Tennessine]] || [[Tennessee]], United States || 17 || 7 || {{sort|294|[294]}}{{ref|1|1}} || {{sort|7.2|(7.2)}}{{ref|11|11}} || {{sort|723|(723)}}{{ref|11|11}} || {{sort|883|(883)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
|-
| 118 || style="background:{{element color|unknown chemical properties}}" | Og || [[Oganesson]] || [[Yuri Oganessian]], physicist || 18 || 7 || {{sort|294|[294]}}{{ref|1|1}} || {{sort|5.0|(5.0)}}{{ref|11|11}} {{ref|13|13}} || {{sort|258|(258)}}{{ref|11|11}} || {{sort|263|(263)}}{{ref|11|11}} || – || – || {{sort|0|0}} {{ref|8|8}}
'''


class Element(NamedTuple):
    """
    A chemical element
    """
    atomic_number: int
    symbol: str
    long_name: str


BY_ATOMIC_NUMBER: Dict[int, Element] = {}
BY_SYMBOL = defaultdict(dict)


def parse_elements(src):
    """
    Turn the chemical element table excerpted from Wikipedia's entry on the periodic table into tuples and index them

    Args:
        src (str): as described above
    """
    for line in src.split('\n'):
        line = line.strip()

        # skip blank lines and separators
        if not line or line.startswith('|-'):
            continue

        fields = line.split('||')
        long_name = fields[2].strip()[2:-2]
        symbol = fields[1].split('|')[2].strip()
        atomic_number = int(fields[0].split('|')[1].strip())

        element = Element(atomic_number, symbol, long_name)
        BY_ATOMIC_NUMBER[atomic_number] = element

        lower_sym = symbol.lower()
        BY_SYMBOL[lower_sym[0]][lower_sym[1:]] = element


def spell_it(msg: str, spellings: List[List], working: List[Element]):
    """
    Spell a message with chemical element symbols

    Args:
        msg (str): suffix of message
        spellings List[List]: list of lists of alternate spellings
        working (List): prefix of atomic numbers spelling the msg prefix
    """
    if not working:
        # extract the alphabetic characters from msg and make them lower-case
        msg = ''.join([char for char in msg.lower() if char in ALPHABET_LOWER])

    if not msg:
        # we got all the way through
        spellings.append(working)
        return

    if msg[0] in BY_SYMBOL:
        for char2, element in BY_SYMBOL[msg[0]].items():
            prefix = msg[0] + char2
            if msg.startswith(prefix):
                spell_it(msg[len(prefix):], spellings, working + [element])


if __name__ == '__main__':
    parse_elements(from_wikipedia)

    if False:
        test_cases: List[str] = [
            'hancock',
            'youth',
            'zoinks',
            'fruitcake',
            'nautilus',
            'nasty',
        ]

        for case in test_cases:
            print('-' * 40)
            print(case)
            spellings = []
            spell_it(case, spellings)
            for index, spelling in enumerate(spellings):
                if index > 0:
                    print('-' * 20)
                for piece in spelling:
                    print(f'    {piece}')

    with open('/usr/share/dict/words', 'r', encoding='utf-8') as f:
        for line in f.readlines():
            word: str = line.strip().lower()
            spellings: List = []
            working: List[Element] = []

            spell_it(word, spellings, working)

            if spellings:
                print('-' * 40)
                print(word)
                for index, spelling in enumerate(sorted(spellings, key=len_then_content)):
                    if index > 0:
                        print('-' * 20)
                    for piece in spelling:
                        print(f'    {piece}')
