"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnGuardValidator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const check_1 = require("./check");
const utils_1 = require("./utils");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const packageJson = require('../package.json');
/**
 * A validation plugin using CFN Guard
 */
class CfnGuardValidator {
    constructor(props = {}) {
        this.rulesPaths = [];
        this.name = 'cdk-validator-cfnguard';
        this.disabledRules = props.disabledRules ?? [];
        const defaultRulesPath = path.join(__dirname, '..', 'rules/control-tower/cfn-guard');
        if (props.controlTowerRulesEnabled ?? true) {
            this.rulesPaths.push(defaultRulesPath);
        }
        this.rulesPaths.push(...props.rules ?? []);
        const osPlatform = os.platform();
        // guard calls it ubuntu but seems to apply to all linux
        // https://github.com/aws-cloudformation/cloudformation-guard/blob/184002cdfc0ae9e29c61995aae41b7d1f1d3b26c/install-guard.sh#L43-L46
        const platform = osPlatform === 'linux'
            ? 'ubuntu'
            : osPlatform === 'darwin' ? 'macos' : undefined;
        if (!platform) {
            throw new Error(`${os.platform()} not supported, must be either 'darwin' or 'linux'`);
        }
        this.guard = path.join(__dirname, '..', 'bin', platform, 'cfn-guard');
        this.ruleIds = this.rulesPaths.flatMap(rule => {
            const parsed = path.parse(rule);
            if (rule === defaultRulesPath || parsed.dir.startsWith(defaultRulesPath)) {
                return this.getRuleIds(rule);
            }
            return [];
        });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        this.version = packageJson.version;
    }
    getRuleIds(rulesPath) {
        const stat = fs.statSync(rulesPath);
        if (stat.isFile()) {
            const parsed = path.parse(rulesPath);
            if (!this.disabledRules.includes(parsed.name)) {
                return [path.parse(rulesPath).name.replace('ct-', '').replace(/-/g, '')];
            }
            return [];
        }
        else {
            return fs.readdirSync(rulesPath).flatMap(rule => this.getRuleIds(path.join(rulesPath, rule)));
        }
    }
    validate(context) {
        const report = this.execGuard({
            templatePaths: context.templatePaths,
            rulePaths: this.rulesPaths,
        });
        return report;
    }
    /**
     * Get the rules to execute. We can return directories as long as none of the rules in the
     * directory have been disabled
     */
    getRules(filePaths) {
        return filePaths.flatMap(file => {
            const stat = fs.statSync(file);
            if (stat.isDirectory()) {
                const dir = fs.readdirSync(file);
                const rules = dir.flatMap(d => this.getRules([path.join(file, d)]));
                if (rules.length === dir.length)
                    return [file];
                return rules;
            }
            else {
                if (!this.disabledRules.includes(path.parse(file).name)) {
                    return [file];
                }
                else {
                    return [];
                }
            }
        });
    }
    execGuard(config) {
        const flags = [
            'validate',
            ...config.rulePaths.flatMap(rule => ['--rules', rule]),
            ...config.templatePaths.flatMap(template => ['--data', template]),
            '--output-format', 'json',
            '--show-summary', 'none',
            '--structured',
        ];
        const violations = [];
        let success = true;
        try {
            const result = (0, utils_1.exec)([this.guard, ...flags], {
                json: true,
            });
            const guardResults = JSON.parse(JSON.stringify(result), reviver);
            for (const guardResult of guardResults) {
                if (!guardResult.not_compliant || guardResult.not_compliant.length === 0) {
                    continue;
                }
                success = false;
                guardResult.not_compliant.forEach((check) => {
                    const violationCheck = new check_1.ViolationCheck(check, guardResult.name);
                    const violation = violationCheck.processCheck();
                    violations.push(...violation);
                });
            }
        }
        catch (e) {
            success = false;
            throw new Error(`
        CfnGuardValidator plugin failed processing cfn-guard results.
        Please create an issue https://github.com/cdklabs/cdk-validator-cfnguard/issues/new
        Error: ${e}`);
        }
        return {
            success,
            violations: violations,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
CfnGuardValidator[_a] = { fqn: "@cdklabs/cdk-validator-cfnguard.CfnGuardValidator", version: "0.0.52" };
exports.CfnGuardValidator = CfnGuardValidator;
/**
 * Guard does not have a standard JSON schema and the schema
 * that is returned can be dependent on the type of rule or type
 * of check that was executed. The results are very much an attempt to
 * display the internals of guard to the user. Trying to make sense of that
 * can be difficult.
 *
 * The result structure can depend on the way that the rule was written. For example
 * I could write a rule like this:
 *
 *     rule MY_RULE {
 *       # This is a "check" and is a `Clause` type check
 *       Properties.SomeProp == true
 *     }
 *
 * Or I could write a rule like this:
 *
 *     rule MY_RULE {
 *       #  This is a "check" and is a `Rule` type check
 *       check(Properties)
 *     }
 *     rule check(properties) {
 *       properties.SomeProp == true
 *     }
 *
 * Both of the above examples are checking the same thing
 * but the schema that is returned is different because the
 * way the rule was written is different
 *
 * This reviver function is executed bottom up and is essentially
 * creating a new object with a well known schema that the rest of the
 * plugin can work with. It looks for certain fields that always appear in
 * the guard results, but appear in different locations. It finds those fields
 * and then pulls them up the object, dropping any fields that we don't
 * care about. For example guard may return
 *
 * {
 *   Clause: {
 *     Unary: {
 *       check: {
 *         UnResolved: {
 *           value: {
 *             traversed_to: {...} // we only care about this!!!
 *           }
 *         }
 *       }
 *     }
 *   }
 * }
 *
 * Or it may return
 *
 * {
 *   Rule: {
 *     checks: [{
 *       Block: {
 *         unresolved: {
 *           traversed_to: {...} // we only care about this!!!
 *         }
 *       }
 *     }]
 *   }
 * }
 *
 * In the above example we only care about the 'traversed_to' field,
 * so this reviver function will grab that field and pull it up the object, dropping
 * the fields we don't care about, ending with something like
 * {
 *   checks: [{
 *     resolved: false,
 *     traversed: {...}
 *   }]
 * }
 *
 */
function reviver(key, value) {
    if (key === 'not_compliant') {
        // not_compliant can sometimes be an empty object (but not an Array), so we
        // process this value before diving into other object values to ensure this
        // one is always made into an Array
        return Object.values(value).map((v) => v.Rule);
    }
    else if (value !== null && typeof value === 'object' && !Array.isArray(value)) {
        return extractNestedObject(value);
    }
    else if (key === 'checks' && Array.isArray(value)) {
        return extractNestedChecks(value.flatMap(v => v));
    }
    return value;
}
/**
 * Extract a nested 'checks' object. This also handles checks
 * nested within checks. It will grab the checks at the level below
 * and pull it up to the next level.
 */
function extractNestedChecks(checks) {
    const containsNestedChecks = checks.some(check => Object.values(check).some((value) => {
        return typeof value === 'object' && value.hasOwnProperty('checks');
    }));
    if (containsNestedChecks) {
        return checks.flatMap(check => {
            if (Object.keys(check).includes('traversed')) {
                return check;
            }
            return Object.values(check).flatMap((checkValue) => {
                return Object.values(checkValue.checks ?? checkValue).flatMap((nestedCheckValue) => {
                    return {
                        ...nestedCheckValue,
                        name: checkValue.name,
                        messages: checkValue.messages ?? nestedCheckValue.messages,
                    };
                });
            });
        });
    }
    return checks.flatMap(check => {
        if (Object.keys(check).includes('traversed')) {
            return check;
        }
        return Object.values(check);
    });
}
/**
 * Extract a nested object and pull it up a level
 */
function extractNestedObject(object) {
    let newObject = object;
    Object.entries(object).forEach(([level1NestedKey, level1NestedValue]) => {
        const nestedValue = level1NestedValue;
        switch (level1NestedKey.toLowerCase()) {
            // this should always be found earlier than the rest since it appears
            // within the 'unresolved' and 'resolved' objects. The object
            // is slightly different for each case so here we create
            // a new object with the key 'traversed' with a consistent value
            case 'traversed_to':
                newObject = {
                    traversed: {
                        to: {
                            path: nestedValue.path,
                            value: nestedValue.value,
                        },
                        from: nestedValue.from ? {
                            path: nestedValue.from.path,
                            value: undefined,
                        } : undefined,
                    },
                    messages: nestedValue.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            case 'unresolved':
                newObject = {
                    resolved: false,
                    traversed: nestedValue.traversed,
                    messages: nestedValue.messages ?? object.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            // A check can either be resolved or unresolved
            case 'resolved':
            case 'inresolved':
                if ('from' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.from,
                            to: {
                                path: nestedValue.from.path,
                                value: nestedValue.to.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                else if ('value' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.value,
                            to: {
                                path: nestedValue.value.path,
                                value: nestedValue.value.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                break;
        }
        // this check will be evaluated _after_ the 'traversed_to' check and _before_ the 'resolved'
        // and 'unresolved' checks above. There may be a case where 'traversed' is nested 2 (or 3 or 4) below
        // 'unresolved' or 'resolved' and this will keep pulling it up until it is just one below
        // and the above checks can work
        if (level1NestedValue !== null && typeof level1NestedValue === 'object' && !Array.isArray(level1NestedValue)) {
            Object.entries(level1NestedValue).forEach(([level2NestedKey, level2NestedValue]) => {
                switch (level2NestedKey.toLowerCase()) {
                    case 'traversed':
                        newObject = {
                            traversed: nestedValue.traversed,
                            resolved: nestedValue.resolved,
                            messages: nestedValue.messages ?? level2NestedValue.messages ?? object.messages,
                        };
                        break;
                }
            });
        }
    });
    return newObject;
}
//# sourceMappingURL=data:application/json;base64,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