#include "../io_helpers.h"
#include <string>
#include <fstream>
#include <iostream>
#include <unordered_map>
#include "../matrix_exponential/matrix_exponential.hpp"

inline int get_char_index(char c, const unordered_map<char, int>& alphabet) {
    auto it = alphabet.find(c);
    return (it != alphabet.end()) ? it->second : -1;
}
std::vector<int> string_to_ints(const std::string& x, const unordered_map<char, int>& alphabet) {
    std::vector<int> res;
    res.reserve(x.length());
    for(int i = 0; i < x.length(); i++) {
        res.push_back(get_char_index(x[i], alphabet));
    }
    return res;
}

void test_read_msa() {
    msa_type msa = read_msa(TEST_MSA_PATH, default_alphabet);
    std::vector<std::string> expected_names = {"1433E_SHEEP_4_239", "BMH1_YEAST_4_240", "RAD24_SCHPO_6_241", "RAD25_SCHPO_5_240", "1433B_VICFA_7_242", "14333_LYCES_9_246", "14334_LYCES_6_243", "14336_ARATH_7_240", "14332_ENTHI_4_238", "14331_ENTHI_4_239", "1433_XENLA_1_227", "1433T_HUMAN_3_236", "1433Z_DROME_6_239", "14331_CAEEL_5_237", "1433F_RAT_3_240", "1433S_HUMAN_3_238"};
    std::unordered_map<std::string, std::vector<int> > expected_map = {
        {"1433E_SHEEP_4_239", string_to_ints("VYQAKLAEQAERYDEMVESMKKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEENKGGEDKLKMIREYRQMVETELKLICCDILDVLDKHLIPAAESKVFYYKMKGDYHRYLAEFATGNDRKEAAENSLVAYKAASDIAMTELPPTHPIRLGLALNFSVFYYEILNSPDRACRLAKAAFDDAIAELDTLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"BMH1_YEAST_4_240", string_to_ints("VYLAKLAEQAERYEEMVENMKTVELSVEERNLLSVAYKNVIGARRASWRIVSSIEQKEESKESEHQVELICSYRSKIETELTKISDDILSVLDSHLIPSAESKVFYYKMKGDYHRYLAEFSSGDAREKATNASLEAYKTASEIATTELPPTHPIRLGLALNFSVFYYEIQNSPDKACHLAKQAFDDAIAELDTLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"RAD24_SCHPO_6_241", string_to_ints("VYLAKLAEQAERYEGMVENMKSVELTVEERNLLSVAYKNVIGARRASWRIVSSIEQKEESKGNTAQVELIKEYRQKIEQELDTICQDILTVLEKHLIPNAESKVFYYKMKGDYYRYLAEFAVGEKRQHSADQSLEGYKAASEIATAELAPTHPIRLGLALNFSVFYYEILNSPDRACYLAKQAFDEAISELDSLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"RAD25_SCHPO_5_240", string_to_ints("VYLAKLAEQAERYEEMVENMKKVKLSVEERNLLSVAYKNIIGARRASWRIISSIEQKEESRGNTRQAALIKEYRKKIEDELSDICHDVLSVLEKHLIPAAESKVFYYKMKGDYYRYLAEFTVGEVCKEAADSSLEAYKAASDIAVAELPPTDPMRLGLALNFSVFYYEILDSPESACHLAKQVFDEAISELDSLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"1433B_VICFA_7_242", string_to_ints("VYIAKLAEQAERYEEMVDSMKNVELTIEERNLLSVGYKNVIGARRASWRILSSIEQKEESKGNDVNAKRIKEYRHKVETELSNICIDVMRVIDEHLIPSAESTVFYYKMKGDYYRYLAEFKTGNEKKEAGDQSMKAYESATTAAEAELPPTHPIRLGLALNFSVFYYEILNSPERACHLAKQAFDEAISELDTLNEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14333_LYCES_9_246", string_to_ints("VYMAKLADRAESDEEMVEFMEKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEESRGNEEHVNSIREYRSKIENELSKICDGILKLLDSKLIPSADSKVFYLKMKGDYHRYLAEFKTGAERKEAAESTLTAYKAAQDIASAELAPTHPIRLGLALNFSVFYYEILNSPDRACNLAKQAFDEAIAELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14334_LYCES_6_243", string_to_ints("VYLAKLAEQAERYEEMIEFMEKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEESRGNEDHVNTIKEYRSKIEADLSKICDGILSLLESNLIPSAESKVFHLKMKGDYHRYLAEFKTGTERKEAAENTLLAYKSAQDIALAELAPTHPIRLGLALNFSVFYYEILNSPDRACNLAKQAFDEAISELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14336_ARATH_7_240", string_to_ints("VYMAKLAEQAERYEEMVQFMEQLELTVEERNLLSVAYKNVIGSLRAAWRIVSSIEQKEESRKNDEHVSLVKDYRSKVESELSSVCSGILKLLDSHLIPSAESKVFYLKMKGDYHRYMAEFKSGDERKTAAEDTMLAYKAAQDIAAADMAPTHPIRLGLALNFSVFYYEILNSSDKACNMAKQAFEEAIAELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14332_ENTHI_4_238", string_to_ints("VYLSKLAEQSERYEEMVQYMKQVELSVEERNLISVAYKNVVGSRRASWRIISSLEQKEQAKGNTQRVELIKTYRAKIEQELSQKCDDVLKIITEFLLKNSESKVFFKKMEGDYYRYYAEFTVDEKRKEVADKSLAAYQEATDTA-ASLVPTHPIRLGLALNFSVFYYQIMNDADKACQLAKEAFDEAIQKLDEVPEESYKESTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14331_ENTHI_4_239", string_to_ints("VYTAKLAEQSERYDEMVQCMKQVELSIEERNLLSVAYKNVIGAKRASWRIISSLEQKEQAKGNDKHVEIIKGYRAKIEKELSTCCDDVLKVIQENLLPKAESKVFFKKMEGDYYRYFAEFTVDEKRKEVADKSLAAYTEATEISNAELAPTHPIRLGLALNFSVFYFEIMNDADKACQLAKQAFDDAIAKLDEVPENMYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"1433_XENLA_1_227", string_to_ints("---AKLSEQAERYDDMAASMKAVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEG--NDKRQQMAREYREKVETELQDICKDVLDLLDRFLVPNAESKVFYLKMKGDYYRYLSEVASGDSKQETVASSQQAYQEAFEISKSEMQPTHPIRLGLALNFSVFYYEILNSPEKACSLAKSAFDEAIRELDTLNEESYKDSTLIMQLLRDNLTLWTSE", default_alphabet)},
        {"1433T_HUMAN_3_236", string_to_ints("IQKAKLAEQAERYDDMATCMKAVELSNEERNLLSVAYKNVVGGRRSAWRVISSIEQKTDT--SDKKLQLIKDYREKVESELRSICTTVLELLDKYLIANAESKVFYLKMKGDYFRYLAEVACGDDRKQTIDNSQGAYQEAFDISKKEMQPTHPIRLGLALNFSVFYYEILNNPELACTLAKTAFDEAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"1433Z_DROME_6_239", string_to_ints("VQKAKLAEQSERYDDMAQAMKSVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEA--SARKQQLAREYRERVEKELREICYEVLGLLDKYLIPKAESKVFYLKMKGDYYRYLAEVATGDARNTVVDDSQTAYQDAFDISKGKMQPTHPIRLGLALNFSVFYYEILNSPDKACQLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"14331_CAEEL_5_237", string_to_ints("VQRAKLAEQAERYDDMAAAMKKVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEG--SEKKQQLAKEYRVKVEQELNDICQDVLKLLDEFLIVKAESKAFYLKMKGDYYRYLAEVAS-EDRAAVVEKSQKAYQEALDIAKDKMQPTHPIRLGLALNFSVFYYEILNTPEHACQLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"1433F_RAT_3_240", string_to_ints("LQRARLAEQAERYDDMASAMKAVPLSNEDRNLLSVAYKNVVGARRSSWRVISSIEQKTMADGNEKKLEKVKAYREKIEKELETVCNDVLALLDKFLIKNCESKVFYLKMKGDYYRYLAEVASGEKKNSVVEASEAAYKEAFEISKEHMQPTHPIRLGLALNFSVFYYEIQNAPEQACLLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet)},
        {"1433S_HUMAN_3_238", string_to_ints("IQKAKLAEQAERYEDMAAFMKGAELSCEERNLLSVAYKNVVGGQRAAWRVLSSIEQKSNEEGSEEKGPEVREYREKVETELQGVCDTVLGLLDSHLIKEAESRVFYLKMKGDYYRYLAEVATGDDKKRIIDSARSAYQEAMDISKKEMPPTNPIRLGLALNFSVFHYEIANSPEEAISLAKTTFDEAMADLHTLSEDSYKDSTLIMQLLRDNLTLWTAD", default_alphabet)}
    };
    std::vector<std::vector<int> > expected_sequences = {
        string_to_ints("VYQAKLAEQAERYDEMVESMKKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEENKGGEDKLKMIREYRQMVETELKLICCDILDVLDKHLIPAAESKVFYYKMKGDYHRYLAEFATGNDRKEAAENSLVAYKAASDIAMTELPPTHPIRLGLALNFSVFYYEILNSPDRACRLAKAAFDDAIAELDTLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYLAKLAEQAERYEEMVENMKTVELSVEERNLLSVAYKNVIGARRASWRIVSSIEQKEESKESEHQVELICSYRSKIETELTKISDDILSVLDSHLIPSAESKVFYYKMKGDYHRYLAEFSSGDAREKATNASLEAYKTASEIATTELPPTHPIRLGLALNFSVFYYEIQNSPDKACHLAKQAFDDAIAELDTLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYLAKLAEQAERYEGMVENMKSVELTVEERNLLSVAYKNVIGARRASWRIVSSIEQKEESKGNTAQVELIKEYRQKIEQELDTICQDILTVLEKHLIPNAESKVFYYKMKGDYYRYLAEFAVGEKRQHSADQSLEGYKAASEIATAELAPTHPIRLGLALNFSVFYYEILNSPDRACYLAKQAFDEAISELDSLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYLAKLAEQAERYEEMVENMKKVKLSVEERNLLSVAYKNIIGARRASWRIISSIEQKEESRGNTRQAALIKEYRKKIEDELSDICHDVLSVLEKHLIPAAESKVFYYKMKGDYYRYLAEFTVGEVCKEAADSSLEAYKAASDIAVAELPPTDPMRLGLALNFSVFYYEILDSPESACHLAKQVFDEAISELDSLSEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYIAKLAEQAERYEEMVDSMKNVELTIEERNLLSVGYKNVIGARRASWRILSSIEQKEESKGNDVNAKRIKEYRHKVETELSNICIDVMRVIDEHLIPSAESTVFYYKMKGDYYRYLAEFKTGNEKKEAGDQSMKAYESATTAAEAELPPTHPIRLGLALNFSVFYYEILNSPERACHLAKQAFDEAISELDTLNEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYMAKLADRAESDEEMVEFMEKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEESRGNEEHVNSIREYRSKIENELSKICDGILKLLDSKLIPSADSKVFYLKMKGDYHRYLAEFKTGAERKEAAESTLTAYKAAQDIASAELAPTHPIRLGLALNFSVFYYEILNSPDRACNLAKQAFDEAIAELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYLAKLAEQAERYEEMIEFMEKVELTVEERNLLSVAYKNVIGARRASWRIISSIEQKEESRGNEDHVNTIKEYRSKIEADLSKICDGILSLLESNLIPSAESKVFHLKMKGDYHRYLAEFKTGTERKEAAENTLLAYKSAQDIALAELAPTHPIRLGLALNFSVFYYEILNSPDRACNLAKQAFDEAISELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYMAKLAEQAERYEEMVQFMEQLELTVEERNLLSVAYKNVIGSLRAAWRIVSSIEQKEESRKNDEHVSLVKDYRSKVESELSSVCSGILKLLDSHLIPSAESKVFYLKMKGDYHRYMAEFKSGDERKTAAEDTMLAYKAAQDIAAADMAPTHPIRLGLALNFSVFYYEILNSSDKACNMAKQAFEEAIAELDTLGEESYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYLSKLAEQSERYEEMVQYMKQVELSVEERNLISVAYKNVVGSRRASWRIISSLEQKEQAKGNTQRVELIKTYRAKIEQELSQKCDDVLKIITEFLLKNSESKVFFKKMEGDYYRYYAEFTVDEKRKEVADKSLAAYQEATDTA-ASLVPTHPIRLGLALNFSVFYYQIMNDADKACQLAKEAFDEAIQKLDEVPEESYKESTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VYTAKLAEQSERYDEMVQCMKQVELSIEERNLLSVAYKNVIGAKRASWRIISSLEQKEQAKGNDKHVEIIKGYRAKIEKELSTCCDDVLKVIQENLLPKAESKVFFKKMEGDYYRYFAEFTVDEKRKEVADKSLAAYTEATEISNAELAPTHPIRLGLALNFSVFYFEIMNDADKACQLAKQAFDDAIAKLDEVPENMYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("---AKLSEQAERYDDMAASMKAVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEG--NDKRQQMAREYREKVETELQDICKDVLDLLDRFLVPNAESKVFYLKMKGDYYRYLSEVASGDSKQETVASSQQAYQEAFEISKSEMQPTHPIRLGLALNFSVFYYEILNSPEKACSLAKSAFDEAIRELDTLNEESYKDSTLIMQLLRDNLTLWTSE", default_alphabet),
        string_to_ints("IQKAKLAEQAERYDDMATCMKAVELSNEERNLLSVAYKNVVGGRRSAWRVISSIEQKTDT--SDKKLQLIKDYREKVESELRSICTTVLELLDKYLIANAESKVFYLKMKGDYFRYLAEVACGDDRKQTIDNSQGAYQEAFDISKKEMQPTHPIRLGLALNFSVFYYEILNNPELACTLAKTAFDEAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VQKAKLAEQSERYDDMAQAMKSVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEA--SARKQQLAREYRERVEKELREICYEVLGLLDKYLIPKAESKVFYLKMKGDYYRYLAEVATGDARNTVVDDSQTAYQDAFDISKGKMQPTHPIRLGLALNFSVFYYEILNSPDKACQLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("VQRAKLAEQAERYDDMAAAMKKVELSNEERNLLSVAYKNVVGARRSSWRVISSIEQKTEG--SEKKQQLAKEYRVKVEQELNDICQDVLKLLDEFLIVKAESKAFYLKMKGDYYRYLAEVAS-EDRAAVVEKSQKAYQEALDIAKDKMQPTHPIRLGLALNFSVFYYEILNTPEHACQLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("LQRARLAEQAERYDDMASAMKAVPLSNEDRNLLSVAYKNVVGARRSSWRVISSIEQKTMADGNEKKLEKVKAYREKIEKELETVCNDVLALLDKFLIKNCESKVFYLKMKGDYYRYLAEVASGEKKNSVVEASEAAYKEAFEISKEHMQPTHPIRLGLALNFSVFYYEIQNAPEQACLLAKQAFDDAIAELDTLNEDSYKDSTLIMQLLRDNLTLWTSD", default_alphabet),
        string_to_ints("IQKAKLAEQAERYEDMAAFMKGAELSCEERNLLSVAYKNVVGGQRAAWRVLSSIEQKSNEEGSEEKGPEVREYREKVETELQGVCDTVLGLLDSHLIKEAESRVFYLKMKGDYYRYLAEVATGDDKKRIIDSARSAYQEAMDISKKEMPPTNPIRLGLALNFSVFHYEIANSPEEAISLAKTTFDEAMADLHTLSEDSYKDSTLIMQLLRDNLTLWTAD", default_alphabet)
    };
    assert(msa.all_names.size() == expected_names.size());
    for(int i = 0; i < msa.all_names.size(); i++) {
        assert(msa.all_names[i] == expected_names[i]);
        assert(msa.names_to_sequence[expected_names[i]] == expected_map[expected_names[i]]); 
        assert(msa.all_sequences[i] == expected_sequences[i]); 
    }
    std::cout << "read_msa sanity passes" << std::endl;
}

void test_write_cherries_and_distances() {
    std::vector<std::pair<std::string, std::string> > cherries = {
        {"1433E_SHEEP_4_239", "BMH1_YEAST_4_240"},
        {"RAD24_SCHPO_6_241", "RAD25_SCHPO_5_240"},
        {"1433B_VICFA_7_242", "14333_LYCES_9_246"},
        {"14334_LYCES_6_243", "14336_ARATH_7_240"},
        {"14332_ENTHI_4_238", "14331_ENTHI_4_239"},
        {"1433_XENLA_1_227", "1433T_HUMAN_3_236"},
        {"1433Z_DROME_6_239", "14331_CAEEL_5_237"},
        {"1433F_RAT_3_240", "1433S_HUMAN_3_238"}
    };
    std::vector<double> distances = {1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0};
    write_cherries_and_distances(cherries, distances, "test_write.txt");
}

void test_read_rate_matrix(){
    std::vector<double> res = read_rate_matrix_from_file("tests/lg.txt", 20);
    for(int i = 0; i < 10; i++) {
        std::cout << res[i] << " ";
    }
    std::cout << std::endl;
    std::cout << "test_read_rate_matrix sanity passes" << std::endl;
}

void test_read_rate_compute_log_transition_matrices() {
    std::vector<double> quantization_points = {6.729602379904665e-05, 7.402562617895133e-05, 8.142818879684645e-05, 8.95710076765311e-05, 9.852810844418422e-05, 0.00010838091928860265, 0.00011921901121746292, 0.00013114091233920925, 0.0001442550035731302, 0.0001586805039304432, 0.00017454855432348754, 0.0001920034097558363, 0.00021120375073141996, 0.00023232412580456198, 0.00025555653838501814, 0.00028111219222352006, 0.000309223411445872, 0.0003401457525904593, 0.00037416032784950526, 0.0004115763606344558, 0.00045273399669790143, 0.0004980073963676917, 0.0005478081360044608, 0.0006025889496049069, 0.0006628478445653976, 0.0007291326290219376, 0.0008020458919241313, 0.0008822504811165445, 0.0009704755292281991, 0.0010675230821510192, 0.001174275390366121, 0.0012917029294027332, 0.0014208732223430067, 0.0015629605445773075, 0.0017192565990350385, 0.0018911822589385424, 0.0020803004848323967, 0.0022883305333156363, 0.0025171635866472, 0.0027688799453119205, 0.003045767939843113, 0.0033503447338274245, 0.0036853792072101673, 0.004053917127931184, 0.004459308840724303, 0.004905239724796734, 0.005395763697276408, 0.005935340067004049, 0.0065288740737044545, 0.0071817614810749004, 0.007899937629182391, 0.00868993139210063, 0.009558924531310695, 0.010514816984441766, 0.011566298682885941, 0.012722928551174538, 0.013995221406291992, 0.015394743546921193, 0.016934217901613313, 0.018627639691774646, 0.020490403660952113, 0.022539444027047325, 0.024793388429752063, 0.02727272727272727, 0.03, 0.033, 0.036300000000000006, 0.039930000000000014, 0.04392300000000001, 0.04831530000000001, 0.05314683000000002, 0.058461513000000034, 0.06430766430000004, 0.07073843073000005, 0.07781227380300007, 0.08559350118330007, 0.09415285130163009, 0.1035681364317931, 0.11392495007497243, 0.12531744508246967, 0.13784918959071665, 0.15163410854978834, 0.1667975194047672, 0.18347727134524391, 0.2018249984797683, 0.22200749832774516, 0.24420824816051973, 0.2686290729765717, 0.2954919802742289, 0.3250411783016518, 0.357545296131817, 0.3932998257449988, 0.43262980831949865, 0.47589278915144856, 0.5234820680665935, 0.5758302748732529, 0.6334133023605781, 0.6967546325966361, 0.7664300958562997, 0.8430731054419298, 0.9273804159861229, 1.020118457584735, 1.1221303033432088, 1.2343433336775298, 1.357777667045283, 1.4935554337498114, 1.6429109771247925, 1.807202074837272, 1.9879222823209994, 2.1867145105530996, 2.4053859616084097, 2.6459245577692507, 2.910517013546176, 3.201568714900794, 3.5217255863908736, 3.8738981450299614, 4.261287959532957, 4.687416755486254, 5.15615843103488, 5.671774274138368, 6.238951701552206, 6.8628468717074265, 7.54913155887817, 8.304044714765988, 9.134449186242586, 10.047894104866845, 11.052683515353532, 12.157951866888887, 13.373747053577777};
    transition_matrices log_transition_matrices = read_rate_compute_log_transition_matrices("tests/lg.txt", quantization_points, {1.0}, 20);
    for(int i = 0; i < 10; i++) {
        std::cout << log_transition_matrices(0,0,0,i) << " ";
    }
    std::cout << "test_read_rate_compute_log_transition_matrices sanity passes" << std::endl;
}

void test_compute_quantization_points() {
    double quantization_grid_center = 0.03;
    double quantization_grid_step = 1.1;
    double quantization_grid_num_steps = 64;

    std::vector<double> quantization_points = compute_quantization_points(
        quantization_grid_center, 
        quantization_grid_step, 
        quantization_grid_num_steps
    );
    std::cout << quantization_points.size() << std::endl;
    for(double pt:quantization_points) {
        std::cout << pt << " ";
    }
    assert(quantization_points.size() == 129);
    std::cout << "\n" << "test_compute_quantization_points sanity passes" << std::endl;
}

int main() {
    test_read_msa();
    test_write_cherries_and_distances();
    test_read_rate_matrix();
    test_read_rate_compute_log_transition_matrices();
    test_compute_quantization_points();
    return 0;
};
