import subprocess
import sys
from pathlib import Path
from typing import Optional

from loguru import logger

from secretflow_doctools.cmd.util import (
    SphinxPaths,
    SphinxPreconditions,
    fatal_on_invalid_sphinx_conf,
)
from secretflow_doctools.l10n import gettext as _
from secretflow_doctools.options import GettextConfig, parse_config
from secretflow_doctools.utils.subprocess import fatal_on_subprocess_error


def update_translations(
    config_dir: Optional[str],
    source_dir: Optional[str],
    output_dir: Optional[str],
    lang: str,
    with_swagger: bool = False,
    sphinx_args: tuple[str, ...] = (),
):
    paths = SphinxPaths.check(
        config_dir=config_dir,
        source_dir=source_dir,
        output_dir=output_dir,
    )

    del config_dir
    del source_dir
    del output_dir

    options = SphinxPreconditions(paths=paths, args=sphinx_args).check()
    assert options.config

    with fatal_on_invalid_sphinx_conf():
        gettext = parse_config(options.config, GettextConfig)

    locale = lang.replace("-", "_")
    locale_dir = paths.source_dir.joinpath(gettext.locale_dirs[0])
    catalog_dir = locale_dir.joinpath(locale).joinpath("LC_MESSAGES")

    cwd = Path.cwd().resolve()

    logger.info(_("source dir: {dir}"), dir=paths.source_dir.relative_to(cwd))
    logger.info(_("output dir: {dir}"), dir=catalog_dir.relative_to(cwd))

    logger.info(
        _("cleaning up autogenerated *.mo files in {dir}"),
        dir=catalog_dir.relative_to(cwd),
    )

    for mo in catalog_dir.glob("**/*.mo"):
        logger.debug(_("removing {mo}"), mo=mo)
        with logger.catch(
            OSError,
            level="WARNING",
            message=_("failed to remove {mo}").format(mo=mo),
        ):
            mo.unlink(missing_ok=True)

    with fatal_on_subprocess_error(
        sys.executable,
        "-m",
        "sphinx",
        "-b",
        "gettext",
        "-D",
        f"language={locale}",
        "-c",
        paths.config_dir,
        paths.source_dir,
        paths.output_dir.joinpath("gettext"),
    ) as cmd:
        subprocess.run(cmd, stdout=None, stderr=None, text=True).check_returncode()

    if with_swagger:
        with fatal_on_subprocess_error(
            sys.executable,
            "-m",
            "sphinx",
            "-b",
            "gettext-swagger",
            "-D",
            f"language={locale}",
            "-c",
            paths.config_dir,
            paths.source_dir,
            paths.output_dir.joinpath("gettext-swagger"),
        ) as cmd:
            subprocess.run(cmd, stdout=None, stderr=None, text=True).check_returncode()

    with fatal_on_subprocess_error(
        sys.executable,
        "-m",
        "sphinx_intl",
        "update",
        "-p",
        paths.output_dir.joinpath("gettext"),
        "-l",
        locale,
        "-d",
        locale_dir,
    ) as cmd:
        subprocess.run(cmd, stdout=None, stderr=None, text=True).check_returncode()

    logger.success(_("finished updating translation files"))
