"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const got_1 = require("got");
const secretsmanagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const slackClient = got_1.default.extend({
    prefixUrl: 'https://slack.com/api',
});
async function handler(event) {
    console.log('Event: %j', event);
    const data = await secretsmanagerClient.send(new client_secrets_manager_1.GetSecretValueCommand({
        SecretId: event.ResourceProperties.configurationTokenSecretArn,
    }));
    if (!data.SecretString) {
        throw new Error('No secret string found in configuration token secret');
    }
    const secret = JSON.parse(data.SecretString);
    let accessToken = secret.accessToken;
    if (!accessToken || isExpired(secret.exp ?? 0)) {
        if (!secret.refreshToken) {
            throw new Error('No refresh token found in configuration token secret');
        }
        console.log('Refreshing access token');
        const rotate = await slackClient.get('tooling.tokens.rotate', {
            searchParams: { refresh_token: secret.refreshToken },
        }).json();
        if (!rotate.ok) {
            throw new Error(`Failed to refresh access token: ${rotate.error}`);
        }
        console.log('Access token refreshed');
        accessToken = rotate.token;
        console.log('Saving access token');
        const putSecretValue = await secretsmanagerClient.send(new client_secrets_manager_1.PutSecretValueCommand({
            SecretId: event.ResourceProperties.configurationTokenSecretArn,
            SecretString: JSON.stringify({
                accessToken,
                refreshToken: rotate.refresh_token,
                exp: rotate.exp,
            }),
        }));
        console.log(`Successfully saved access token in secret ${putSecretValue.ARN}`);
    }
    const operation = event.RequestType.toLowerCase();
    const request = getManifestRequest(event);
    console.log(`Calling ${operation} manifest API: %j`, request);
    const response = await slackClient.post(`apps.manifest.${operation}`, {
        headers: { Authorization: `Bearer ${accessToken}` },
        json: request,
    }).json();
    if (!response.ok) {
        const errors = response.errors
            ? response.errors.map((err) => `${err.message} at ${err.pointer}`).join('\n')
            : '';
        throw new Error(`Failed to ${operation} manifest: ${response.error}.${errors ? `\n${errors}}` : ''}`);
    }
    console.log(`Successfully ${operation}d Slack app ${event.PhysicalResourceId ?? response.app_id}`);
    if (event.RequestType === 'Create' && response.credentials) {
        console.log('Saving app credentials');
        const putSecretValue = await secretsmanagerClient.send(new client_secrets_manager_1.PutSecretValueCommand({
            SecretId: event.ResourceProperties.credentialsSecretArn,
            SecretString: JSON.stringify({
                appId: response.app_id,
                clientId: response.credentials.client_id,
                clientSecret: response.credentials.client_secret,
                verificationToken: response.credentials.verification_token,
                signingSecret: response.credentials.signing_secret,
            }),
        }));
        console.log(`Successfully saved app credentials in secret ${putSecretValue.ARN}`);
    }
    return {
        PhysicalResourceId: response.app_id,
        Data: {
            appId: response.app_id,
        },
    };
}
exports.handler = handler;
function isExpired(iat) {
    return (iat - (Date.now() / 1000)) < 0;
}
function getManifestRequest(event) {
    switch (event.RequestType) {
        case 'Create':
            return {
                manifest: event.ResourceProperties.manifest,
            };
        case 'Update':
            return {
                app_id: event.PhysicalResourceId,
                manifest: event.ResourceProperties.manifest,
            };
        case 'Delete':
            return {
                app_id: event.PhysicalResourceId,
            };
    }
}
//# sourceMappingURL=data:application/json;base64,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