/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as apigateway from 'aws-cdk-lib/aws-apigateway';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { Duration } from "aws-cdk-lib";
import { Construct } from 'constructs';
/**
 * The ApiIntegration interface is used to correlate a user-specified id with either a existing lambda function or set of lambda props.
 *
 * See the 'Overview of how the OpenAPI file transformation works' section of the README.md for more details on its usage.
 */
export interface ApiIntegration {
    /**
     * Id of the ApiIntegration, used to correlate this lambda function to the api integration in the open api definition.
     *
     * Note this is not a CDK Construct ID, and is instead a client defined string used to map the resolved lambda resource with the OpenAPI definition.
     */
    readonly id: string;
    /**
     * The Lambda function to associate with the API method in the OpenAPI file matched by id.
     *
     * One and only one of existingLambdaObj or lambdaFunctionProps must be specified, any other combination will cause an error.
     */
    readonly existingLambdaObj?: lambda.Function;
    /**
     * Properties for the Lambda function to create and associate with the API method in the OpenAPI file matched by id.
     *
     * One and only one of existingLambdaObj or lambdaFunctionProps must be specified, any other combination will cause an error.
     */
    readonly lambdaFunctionProps?: lambda.FunctionProps;
}
/**
 * Helper object to map an ApiIntegration id to its resolved lambda.Function. This type is exposed as a property on the instantiated construct.
 */
export interface ApiLambdaFunction {
    /**
     * Id of the ApiIntegration, used to correlate this lambda function to the api integration in the open api definition.
     */
    readonly id: string;
    /**
     * The instantiated lambda.Function.
     */
    readonly lambdaFunction: lambda.Function;
}
export interface OpenApiProps {
    readonly apiDefinitionAsset?: Asset;
    readonly apiDefinitionJson?: any;
    readonly apiDefinitionBucket?: s3.IBucket;
    readonly apiDefinitionKey?: string;
    readonly apiIntegrations: ApiIntegration[];
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
export declare function CheckOpenApiProps(props: OpenApiProps): void;
export interface ObtainApiDefinitionProps {
    readonly tokenToFunctionMap: ApiLambdaFunction[];
    readonly apiDefinitionBucket?: s3.IBucket;
    readonly apiDefinitionKey?: string;
    readonly apiDefinitionAsset?: Asset;
    readonly apiJsonDefinition?: any;
    readonly internalTransformTimeout?: Duration;
    readonly internalTransformMemorySize?: number;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
export declare function ObtainApiDefinition(scope: Construct, props: ObtainApiDefinitionProps): apigateway.ApiDefinition;
