"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObtainApiDefinition = exports.CheckOpenApiProps = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
/*
 * This file is core openapi functionality and should ideally be in the core library. Since
 * that causes a circular reference with the resources library we have left it here for now
 * in the interest of getting these updates out faster
 */
const apigateway = require("aws-cdk-lib/aws-apigateway");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const resources = require("@aws-solutions-constructs/resources");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckOpenApiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if ((props.apiDefinitionBucket && !props.apiDefinitionKey) || (!props.apiDefinitionBucket && props.apiDefinitionKey)) {
        errorMessages += 'apiDefinitionBucket and apiDefinitionKey must be specified together.\n';
        errorFound = true;
    }
    const definitionCount = (props.apiDefinitionAsset ? 1 : 0) +
        (props.apiDefinitionBucket ? 1 : 0) +
        (props.apiDefinitionJson ? 1 : 0);
    if (definitionCount !== 1) {
        errorMessages += 'Exactly one of apiDefinitionAsset, apiDefinitionJson or (apiDefinitionBucket/apiDefinitionKey) must be provided\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckOpenApiProps = CheckOpenApiProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function ObtainApiDefinition(scope, props) {
    const apiRawInlineSpec = props.apiJsonDefinition;
    const meldedDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const meldedDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    // Map each id and lambda function pair to the required format for the template writer custom resource
    const apiIntegrationUris = props.tokenToFunctionMap.map(apiLambdaFunction => {
        // the placeholder string that will be replaced in the OpenAPI Definition
        const uriPlaceholderString = apiLambdaFunction.id;
        // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
        return {
            id: uriPlaceholderString,
            value: uriResolvedValue
        };
    });
    let apiDefinitionWriter;
    let newApiDefinition;
    if (props.apiDefinitionAsset || props.apiDefinitionBucket) {
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        apiDefinitionWriter = resources.createTemplateWriterCustomResource(scope, 'Api', {
            // CheckOpenapiProps() has confirmed the existence of these values
            templateBucket: meldedDefinitionBucket,
            templateKey: meldedDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? aws_cdk_lib_1.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        newApiDefinition = apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key);
    }
    else if (apiRawInlineSpec) {
        newApiDefinition = InlineTemplateWriter(apiRawInlineSpec, apiIntegrationUris);
    }
    else {
        throw new Error("No definition provided (this code should be unreachable)");
    }
    return newApiDefinition;
}
exports.ObtainApiDefinition = ObtainApiDefinition;
function InlineTemplateWriter(rawInlineSpec, templateValues) {
    let template = JSON.stringify(rawInlineSpec);
    // This replicates logic in the template writer custom resource (resources/lib/template-writer-custom-resource/index.ts),
    // any logic changes should be made to both locations every time
    templateValues.forEach((templateValue) => {
        template = template?.replace(new RegExp(templateValue.id, 'g'), templateValue.value);
    });
    return new apigateway.InlineApiDefinition(JSON.parse(template));
}
//# sourceMappingURL=data:application/json;base64,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