"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CidrContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const util_1 = require("../util");
const assign_on_launch_1 = require("./assign-on-launch");
/**
 * Allocates IPv6 CIDRs and routes for subnets in a VPC.
 *
 * @see {@link https://github.com/aws/aws-cdk/issues/894}
 * @see {@link https://github.com/aws/aws-cdk/issues/5927}
 */
class CidrContext extends constructs_1.Construct {
    /**
     * Creates a new BetterVpc.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param options - The constructor options.
     */
    constructor(scope, id, options) {
        super(scope, id);
        this.assignAddressOnLaunch = options.assignAddressOnLaunch !== false;
        const vpc = util_1.assertType(options.vpc, aws_ec2_1.Vpc);
        this.vpc = vpc;
        // Create the VPC CIDR block.
        const cidrBlock = new aws_ec2_1.CfnVPCCidrBlock(this, "Ipv6CidrBlock", {
            vpcId: vpc.vpcId,
            amazonProvidedIpv6CidrBlock: options.addressPool === undefined,
            ipv6Pool: options.addressPool,
            ipv6CidrBlock: options.cidrBlock,
        });
        // Create a placeholder for CloudFormation to provide a CIDR array.
        const cidrs = aws_cdk_lib_1.Fn.cidr(aws_cdk_lib_1.Fn.select(0, vpc.vpcIpv6CidrBlocks), this.validateCidrCount(vpc, options.cidrCount), "64");
        if (vpc.publicSubnets.length > 0) {
            this.assignPublicSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.privateSubnets.length > 0) {
            this.assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.isolatedSubnets.length > 0) {
            this.assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (this.assignAddressOnLaunch) {
            const vpcSubnets = {
                subnets: [
                    ...vpc.publicSubnets,
                    ...vpc.privateSubnets,
                    ...vpc.isolatedSubnets,
                ],
            };
            new assign_on_launch_1.AssignOnLaunch(this, "AssignOnLaunch", { vpc, vpcSubnets });
        }
    }
    /**
     * Figure out the minimun required CIDR subnets and the number desired.
     *
     * @param vpc - The VPC.
     * @param cidrCount - Optional. Divide the VPC CIDR into this many subsets.
     */
    validateCidrCount(vpc, cidrCount) {
        const leastCidrBlocksNeeded = vpc.publicSubnets.length +
            vpc.privateSubnets.length +
            vpc.isolatedSubnets.length;
        const cidrBlockCount = cidrCount !== null && cidrCount !== void 0 ? cidrCount : leastCidrBlocksNeeded;
        if (cidrBlockCount < leastCidrBlocksNeeded) {
            throw new Error(`You must create at least ${leastCidrBlocksNeeded} CIDR blocks in this VPC`);
        }
        return cidrBlockCount;
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPublicSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = util_1.findInstanceOf(vpc.node.children, aws_ec2_1.CfnInternetGateway);
        for (const [index, subnet] of vpc.publicSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Internet gateway for Internet-bound IPv6 traffic.
            const concreteSubnet = util_1.assertType(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: gateway.ref,
                routerType: aws_ec2_1.RouterType.GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for private subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = new aws_ec2_1.CfnEgressOnlyInternetGateway(this, "EgressOnlyIgw", {
            vpcId: vpc.vpcId,
        });
        for (const [index, subnet] of vpc.privateSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Egress-only Internet Gateway.
            const concreteSubnet = util_1.assertType(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: `${gateway.ref}`,
                routerType: aws_ec2_1.RouterType.EGRESS_ONLY_INTERNET_GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock) {
        for (const [index, subnet] of vpc.isolatedSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length + vpc.privateSubnets.length, cidrs);
        }
    }
}
exports.CidrContext = CidrContext;
_a = JSII_RTTI_SYMBOL_1;
CidrContext[_a] = { fqn: "shady-island.CidrContext", version: "0.0.1-alpha.29" };
//# sourceMappingURL=data:application/json;base64,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