"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoToApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const api = require("aws-cdk-lib/aws-apigateway");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class CognitoToApiGatewayToLambda extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CognitoToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckLambdaProps(props);
        // This Construct requires that the auth type be COGNITO regardless of what is specified in the props
        if (props.apiGatewayProps) {
            if (props.apiGatewayProps.defaultMethodOptions === undefined) {
                props.apiGatewayProps.defaultMethodOptions = {
                    authorizationType: api.AuthorizationType.COGNITO,
                };
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType === undefined) {
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType !== 'COGNITO_USER_POOLS') {
                defaults.printWarning('Overriding Authorization type to be AuthorizationType.COGNITO');
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
        }
        if (props.apiGatewayProps && (typeof props.apiGatewayProps.proxy !== 'undefined') && (props.apiGatewayProps.proxy === false)) {
            defaults.printWarning('For non-proxy API, addAuthorizers() method must be called after all the resources and methods for API are fully defined. Not calling addAuthorizers() will result in API methods NOT protected by Cognito.');
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        const globalLambdaRestApiResponse = defaults.GlobalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway = globalLambdaRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalLambdaRestApiResponse.role;
        this.apiGatewayLogGroup = globalLambdaRestApiResponse.group;
        this.userPool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userPoolClient = defaults.buildUserPoolClient(this, this.userPool, props.cognitoUserPoolClientProps);
        this.apiGatewayAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.apiGateway.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userPool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.addAuthorizers();
    }
    addAuthorizers() {
        this.apiGateway.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method to support CORS, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: this.apiGatewayAuthorizer.logicalId });
            }
        });
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
CognitoToApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-cognito-apigateway-lambda.CognitoToApiGatewayToLambda", version: "2.78.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBRUgsa0RBQWtEO0FBS2xELDJEQUEyRDtBQUMzRCx3RkFBd0Y7QUFDeEYsMkNBQXVDO0FBNEN2QyxNQUFhLDJCQUE0QixTQUFRLHNCQUFTO0lBU3hEOzs7Ozs7O09BT0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVDO1FBQy9FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRWpDLHFHQUFxRztRQUNyRyxJQUFJLEtBQUssQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUMxQixJQUFJLEtBQUssQ0FBQyxlQUFlLENBQUMsb0JBQW9CLEtBQUssU0FBUyxFQUFFLENBQUM7Z0JBQzdELEtBQUssQ0FBQyxlQUFlLENBQUMsb0JBQW9CLEdBQUc7b0JBQzNDLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPO2lCQUNqRCxDQUFDO1lBQ0osQ0FBQztpQkFBTSxJQUFJLEtBQUssQ0FBQyxlQUFlLEVBQUUsb0JBQW9CLENBQUMsaUJBQWlCLEtBQUssU0FBUyxFQUFFLENBQUM7Z0JBQ3ZGLEtBQUssQ0FBQyxlQUFlLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQztZQUMvRixDQUFDO2lCQUFNLElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRSxvQkFBb0IsQ0FBQyxpQkFBaUIsS0FBSyxvQkFBb0IsRUFBRSxDQUFDO2dCQUNsRyxRQUFRLENBQUMsWUFBWSxDQUFDLCtEQUErRCxDQUFDLENBQUM7Z0JBQ3ZGLEtBQUssQ0FBQyxlQUFlLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQztZQUMvRixDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksS0FBSyxDQUFDLGVBQWUsSUFBSSxDQUFDLE9BQU8sS0FBSyxDQUFDLGVBQWUsQ0FBQyxLQUFLLEtBQUssV0FBVyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLEtBQUssS0FBSyxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzdILFFBQVEsQ0FBQyxZQUFZLENBQUMsNE1BQTRNLENBQUMsQ0FBQztRQUN0TyxDQUFDO1FBRUQsSUFBSSxDQUFDLGNBQWMsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFO1lBQ3ZELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtTQUMvQyxDQUFDLENBQUM7UUFDSCxNQUFNLDJCQUEyQixHQUFHLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxLQUFLLENBQUMsZUFBZSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4SSxJQUFJLENBQUMsVUFBVSxHQUFHLDJCQUEyQixDQUFDLEdBQUcsQ0FBQztRQUNsRCxJQUFJLENBQUMsd0JBQXdCLEdBQUcsMkJBQTJCLENBQUMsSUFBSSxDQUFDO1FBQ2pFLElBQUksQ0FBQyxrQkFBa0IsR0FBRywyQkFBMkIsQ0FBQyxLQUFLLENBQUM7UUFFNUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUN6RSxJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUUxRyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUMzRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTO1lBQ3BDLElBQUksRUFBRSxvQkFBb0I7WUFDMUIsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUM7WUFDekMsY0FBYyxFQUFFLHFDQUFxQztZQUNyRCxJQUFJLEVBQUUsWUFBWTtTQUNuQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7SUFDeEIsQ0FBQztJQUVNLGNBQWM7UUFDbkIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUU7WUFDNUMsb0dBQW9HO1lBQ3BHLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBa0IsQ0FBQztZQUNwRSxJQUFJLFNBQVMsQ0FBQyxVQUFVLEtBQUssU0FBUyxFQUFFLENBQUM7Z0JBQ3ZDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxtQkFBbUIsRUFBRSxNQUFNLENBQUMsQ0FBQztZQUN6RCxDQUFDO2lCQUFNLENBQUM7Z0JBQ04sS0FBSyxDQUFDLG1CQUFtQixDQUFDLG1CQUFtQixFQUFFLG9CQUFvQixDQUFDLENBQUM7Z0JBQ3JFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7WUFDMUYsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUF6RUgsa0VBMEVDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgYXBpIGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNvZ25pdG8gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNvZ25pdG8nO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuLy8gTm90ZTogVG8gZW5zdXJlIENES3YyIGNvbXBhdGliaWxpdHksIGtlZXAgdGhlIGltcG9ydCBzdGF0ZW1lbnQgZm9yIENvbnN0cnVjdCBzZXBhcmF0ZVxuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbi8qKlxuICogQHN1bW1hcnkgVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBDb2duaXRvVG9BcGlHYXRld2F5VG9MYW1iZGEgQ29uc3RydWN0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhUHJvcHMge1xuICAvKipcbiAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgTGFtYmRhIEZ1bmN0aW9uIG9iamVjdCwgcHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGxhbWJkYUZ1bmN0aW9uUHJvcHNgIHdpbGwgY2F1c2UgYW4gZXJyb3IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdMYW1iZGFPYmo/OiBsYW1iZGEuRnVuY3Rpb24sXG4gIC8qKlxuICAgKiBVc2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uUHJvcHM/OiBsYW1iZGEuRnVuY3Rpb25Qcm9wc1xuICAvKipcbiAgICogT3B0aW9uYWwgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIEFQSSBHYXRld2F5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGFwaUdhdGV3YXlQcm9wcz86IGFwaS5MYW1iZGFSZXN0QXBpUHJvcHMgfCBhbnlcbiAgLyoqXG4gICAqIE9wdGlvbmFsIHVzZXIgcHJvdmlkZWQgcHJvcHMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcHNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBjb2duaXRvVXNlclBvb2xQcm9wcz86IGNvZ25pdG8uVXNlclBvb2xQcm9wc1xuICAvKipcbiAgICogT3B0aW9uYWwgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGNvZ25pdG9Vc2VyUG9vbENsaWVudFByb3BzPzogY29nbml0by5Vc2VyUG9vbENsaWVudFByb3BzIHwgYW55LFxuICAvKipcbiAgICogVXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIENsb3VkV2F0Y2hMb2dzIExvZ0dyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGxvZ0dyb3VwUHJvcHM/OiBsb2dzLkxvZ0dyb3VwUHJvcHNcbn1cblxuZXhwb3J0IGNsYXNzIENvZ25pdG9Ub0FwaUdhdGV3YXlUb0xhbWJkYSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbDogY29nbml0by5Vc2VyUG9vbDtcbiAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sQ2xpZW50OiBjb2duaXRvLlVzZXJQb29sQ2xpZW50O1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpR2F0ZXdheTogYXBpLlJlc3RBcGk7XG4gIHB1YmxpYyByZWFkb25seSBhcGlHYXRld2F5Q2xvdWRXYXRjaFJvbGU/OiBpYW0uUm9sZTtcbiAgcHVibGljIHJlYWRvbmx5IGFwaUdhdGV3YXlMb2dHcm91cDogbG9ncy5Mb2dHcm91cDtcbiAgcHVibGljIHJlYWRvbmx5IGFwaUdhdGV3YXlBdXRob3JpemVyOiBhcGkuQ2ZuQXV0aG9yaXplcjtcbiAgcHVibGljIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uOiBsYW1iZGEuRnVuY3Rpb247XG5cbiAgLyoqXG4gICAqIEBzdW1tYXJ5IENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIENvZ25pdG9Ub0FwaUdhdGV3YXlUb0xhbWJkYSBjbGFzcy5cbiAgICogQHBhcmFtIHtjZGsuQXBwfSBzY29wZSAtIHJlcHJlc2VudHMgdGhlIHNjb3BlIGZvciBhbGwgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIC0gdGhpcyBpcyBhIGEgc2NvcGUtdW5pcXVlIGlkLlxuICAgKiBAcGFyYW0ge0NvZ25pdG9Ub0FwaUdhdGV3YXlUb0xhbWJkYVByb3BzfSBwcm9wcyAtIHVzZXIgcHJvdmlkZWQgcHJvcHMgZm9yIHRoZSBjb25zdHJ1Y3RcbiAgICogQHNpbmNlIDAuOC4wXG4gICAqIEBhY2Nlc3MgcHVibGljXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ29nbml0b1RvQXBpR2F0ZXdheVRvTGFtYmRhUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIGRlZmF1bHRzLkNoZWNrTGFtYmRhUHJvcHMocHJvcHMpO1xuXG4gICAgLy8gVGhpcyBDb25zdHJ1Y3QgcmVxdWlyZXMgdGhhdCB0aGUgYXV0aCB0eXBlIGJlIENPR05JVE8gcmVnYXJkbGVzcyBvZiB3aGF0IGlzIHNwZWNpZmllZCBpbiB0aGUgcHJvcHNcbiAgICBpZiAocHJvcHMuYXBpR2F0ZXdheVByb3BzKSB7XG4gICAgICBpZiAocHJvcHMuYXBpR2F0ZXdheVByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcHJvcHMuYXBpR2F0ZXdheVByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zID0ge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcGkuQXV0aG9yaXphdGlvblR5cGUuQ09HTklUTyxcbiAgICAgICAgfTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuYXBpR2F0ZXdheVByb3BzPy5kZWZhdWx0TWV0aG9kT3B0aW9ucy5hdXRob3JpemF0aW9uVHlwZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHByb3BzLmFwaUdhdGV3YXlQcm9wcy5kZWZhdWx0TWV0aG9kT3B0aW9ucy5hdXRob3JpemF0aW9uVHlwZSA9IGFwaS5BdXRob3JpemF0aW9uVHlwZS5DT0dOSVRPO1xuICAgICAgfSBlbHNlIGlmIChwcm9wcy5hcGlHYXRld2F5UHJvcHM/LmRlZmF1bHRNZXRob2RPcHRpb25zLmF1dGhvcml6YXRpb25UeXBlICE9PSAnQ09HTklUT19VU0VSX1BPT0xTJykge1xuICAgICAgICBkZWZhdWx0cy5wcmludFdhcm5pbmcoJ092ZXJyaWRpbmcgQXV0aG9yaXphdGlvbiB0eXBlIHRvIGJlIEF1dGhvcml6YXRpb25UeXBlLkNPR05JVE8nKTtcbiAgICAgICAgcHJvcHMuYXBpR2F0ZXdheVByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zLmF1dGhvcml6YXRpb25UeXBlID0gYXBpLkF1dGhvcml6YXRpb25UeXBlLkNPR05JVE87XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmFwaUdhdGV3YXlQcm9wcyAmJiAodHlwZW9mIHByb3BzLmFwaUdhdGV3YXlQcm9wcy5wcm94eSAhPT0gJ3VuZGVmaW5lZCcpICYmIChwcm9wcy5hcGlHYXRld2F5UHJvcHMucHJveHkgPT09IGZhbHNlKSkge1xuICAgICAgZGVmYXVsdHMucHJpbnRXYXJuaW5nKCdGb3Igbm9uLXByb3h5IEFQSSwgYWRkQXV0aG9yaXplcnMoKSBtZXRob2QgbXVzdCBiZSBjYWxsZWQgYWZ0ZXIgYWxsIHRoZSByZXNvdXJjZXMgYW5kIG1ldGhvZHMgZm9yIEFQSSBhcmUgZnVsbHkgZGVmaW5lZC4gTm90IGNhbGxpbmcgYWRkQXV0aG9yaXplcnMoKSB3aWxsIHJlc3VsdCBpbiBBUEkgbWV0aG9kcyBOT1QgcHJvdGVjdGVkIGJ5IENvZ25pdG8uJyk7XG4gICAgfVxuXG4gICAgdGhpcy5sYW1iZGFGdW5jdGlvbiA9IGRlZmF1bHRzLmJ1aWxkTGFtYmRhRnVuY3Rpb24odGhpcywge1xuICAgICAgZXhpc3RpbmdMYW1iZGFPYmo6IHByb3BzLmV4aXN0aW5nTGFtYmRhT2JqLFxuICAgICAgbGFtYmRhRnVuY3Rpb25Qcm9wczogcHJvcHMubGFtYmRhRnVuY3Rpb25Qcm9wc1xuICAgIH0pO1xuICAgIGNvbnN0IGdsb2JhbExhbWJkYVJlc3RBcGlSZXNwb25zZSA9IGRlZmF1bHRzLkdsb2JhbExhbWJkYVJlc3RBcGkodGhpcywgdGhpcy5sYW1iZGFGdW5jdGlvbiwgcHJvcHMuYXBpR2F0ZXdheVByb3BzLCBwcm9wcy5sb2dHcm91cFByb3BzKTtcbiAgICB0aGlzLmFwaUdhdGV3YXkgPSBnbG9iYWxMYW1iZGFSZXN0QXBpUmVzcG9uc2UuYXBpO1xuICAgIHRoaXMuYXBpR2F0ZXdheUNsb3VkV2F0Y2hSb2xlID0gZ2xvYmFsTGFtYmRhUmVzdEFwaVJlc3BvbnNlLnJvbGU7XG4gICAgdGhpcy5hcGlHYXRld2F5TG9nR3JvdXAgPSBnbG9iYWxMYW1iZGFSZXN0QXBpUmVzcG9uc2UuZ3JvdXA7XG5cbiAgICB0aGlzLnVzZXJQb29sID0gZGVmYXVsdHMuYnVpbGRVc2VyUG9vbCh0aGlzLCBwcm9wcy5jb2duaXRvVXNlclBvb2xQcm9wcyk7XG4gICAgdGhpcy51c2VyUG9vbENsaWVudCA9IGRlZmF1bHRzLmJ1aWxkVXNlclBvb2xDbGllbnQodGhpcywgdGhpcy51c2VyUG9vbCwgcHJvcHMuY29nbml0b1VzZXJQb29sQ2xpZW50UHJvcHMpO1xuXG4gICAgdGhpcy5hcGlHYXRld2F5QXV0aG9yaXplciA9IG5ldyBhcGkuQ2ZuQXV0aG9yaXplcih0aGlzLCAnQ29nbml0b0F1dGhvcml6ZXInLCB7XG4gICAgICByZXN0QXBpSWQ6IHRoaXMuYXBpR2F0ZXdheS5yZXN0QXBpSWQsXG4gICAgICB0eXBlOiAnQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAgIHByb3ZpZGVyQXJuczogW3RoaXMudXNlclBvb2wudXNlclBvb2xBcm5dLFxuICAgICAgaWRlbnRpdHlTb3VyY2U6IFwibWV0aG9kLnJlcXVlc3QuaGVhZGVyLkF1dGhvcml6YXRpb25cIixcbiAgICAgIG5hbWU6IFwiYXV0aG9yaXplclwiXG4gICAgfSk7XG5cbiAgICB0aGlzLmFkZEF1dGhvcml6ZXJzKCk7XG4gIH1cblxuICBwdWJsaWMgYWRkQXV0aG9yaXplcnMoKSB7XG4gICAgdGhpcy5hcGlHYXRld2F5Lm1ldGhvZHMuZm9yRWFjaCgoYXBpTWV0aG9kKSA9PiB7XG4gICAgICAvLyBMZWF2ZSB0aGUgYXV0aG9yaXplciBOT05FIGZvciBIVFRQIE9QVElPTlMgbWV0aG9kIHRvIHN1cHBvcnQgQ09SUywgZm9yIHRoZSByZXN0IHNldCBpdCB0byBDT0dOSVRPXG4gICAgICBjb25zdCBjaGlsZCA9IGFwaU1ldGhvZC5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBhcGkuQ2ZuTWV0aG9kO1xuICAgICAgaWYgKGFwaU1ldGhvZC5odHRwTWV0aG9kID09PSAnT1BUSU9OUycpIHtcbiAgICAgICAgY2hpbGQuYWRkUHJvcGVydHlPdmVycmlkZSgnQXV0aG9yaXphdGlvblR5cGUnLCAnTk9ORScpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY2hpbGQuYWRkUHJvcGVydHlPdmVycmlkZSgnQXV0aG9yaXphdGlvblR5cGUnLCAnQ09HTklUT19VU0VSX1BPT0xTJyk7XG4gICAgICAgIGNoaWxkLmFkZFByb3BlcnR5T3ZlcnJpZGUoJ0F1dGhvcml6ZXJJZCcsIHsgUmVmOiB0aGlzLmFwaUdhdGV3YXlBdXRob3JpemVyLmxvZ2ljYWxJZCB9KTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxufVxuIl19