#!/bin/bash
#SBATCH --partition=PARTITION
#SBATCH --nodes=1
#SBATCH --ntasks=1
#SBATCH --cpus-per-task=THREADS_VALUE
#SBATCH --mem=MEMORY
#SBATCH --time=TIME
#SBATCH --job-name=emtax
#SBATCH --output=WORKDIR_PATH/emtax_%j.out
#SBATCH --error=WORKDIR_PATH/emtax_%j.err

# emtax workflow script
# Generated by emtax

echo "Starting emtax workflow at $(date)"

# Set variables
export WORKDIR="WORKDIR_PATH"
export RAWDATA_DIR="RAWDATA_DIR_PATH"
export RESULTS_DIR="RESULTS_DIR_PATH"
export KRAKEN_DB="KRAKEN_DB_PATH"
export CORN_DB="CORN_DB_PATH"
export THREADS=THREADS_VALUE

# Check if conda is available
if command -v conda &> /dev/null; then
    echo "Conda is available"
    
    # Initialize conda for bash
    CONDA_BASE=$(conda info --base)
    source "${CONDA_BASE}/etc/profile.d/conda.sh"
    
    # Activate conda environment
    echo "Activating conda environment: emtax_env"
    conda activate emtax_env
else
    echo "WARNING: Conda not available, proceeding without environment activation"
fi

# Check network connectivity
if ping -c 1 8.8.8.8 &> /dev/null; then
    echo "Network is reachable, using online mode"
else
    echo "WARNING: Network is unreachable, using offline mode"
fi

# Change to workflow directory
cd "${WORKDIR}" || { echo "ERROR: Failed to change to workflow directory"; exit 1; }

echo "Processing samples from config file: ${WORKDIR}/config.yaml"

# Create output directories
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2"
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/2_DNA_Bracken"
mkdir -p "${RESULTS_DIR}/Taxonomic_Profiling/5_DNA_Relative_Abundance_Matrix_Python"

# Create workflow subdirectories
mkdir -p "${WORKDIR}/Preprocessed_Data"
mkdir -p "${WORKDIR}/Logs"

# Process each sample
for SAMPLE in SAMPLES; do
    echo "Processing sample: $SAMPLE"
    
    # Step 1: Check if files are already lane-combined, single-end, or need combining
    mkdir -p "${WORKDIR}/Preprocessed_Data/combined"
    
    # Check for different file patterns
    NO_LANE_R1=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_R1.fastq.gz" | sort)
    NO_LANE_R2=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_R2.fastq.gz" | sort)
    SINGLE_END_FILE=$(find "${RAWDATA_DIR}" -name "${SAMPLE}.fastq.gz" | sort)
    
    # Check if this is a single-end file (no R1/R2, no lane info)
    if [ -n "$SINGLE_END_FILE" ]; then
        echo "Found single-end file for ${SAMPLE}: $SINGLE_END_FILE"
        # Copy the single-end file with its original name
        mkdir -p "${WORKDIR}/Preprocessed_Data/combined"
        cp $SINGLE_END_FILE "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}.fastq.gz"
        # Mark this sample as single-end for downstream processing
        SINGLE_END_SAMPLES="${SINGLE_END_SAMPLES} ${SAMPLE}"
        echo "Marked ${SAMPLE} as a single-end sample"
    # Check if files are already lane-combined
    elif [ -n "$NO_LANE_R1" ] || [ -n "$NO_LANE_R2" ]; then
        echo "Found files without lane information for ${SAMPLE}, skipping lane combining..."
        # Copy the files directly to the combined directory
        if [ -n "$NO_LANE_R1" ]; then
            cp $NO_LANE_R1 "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz"
        else
            # Create empty R1 file if not found
            touch "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz"
        fi
        if [ -n "$NO_LANE_R2" ]; then
            cp $NO_LANE_R2 "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz"
        else
            # Create empty R2 file if not found
            touch "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz"
        fi
    else
        echo "Combining lanes for ${SAMPLE}..."
        # Find all R1 and R2 files for this sample with lane information
        R1_FILES=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_*_R1.fastq.gz" | sort)
        R2_FILES=$(find "${RAWDATA_DIR}" -name "${SAMPLE}_*_R2.fastq.gz" | sort)
        
        # Combine lanes
        if [ -n "$R1_FILES" ]; then
            cat $R1_FILES > "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz"
        else
            touch "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz"
        fi
        
        if [ -n "$R2_FILES" ]; then
            cat $R2_FILES > "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz"
        else
            touch "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz"
        fi
    fi
    
    # Step 2: Run fastp on combined files
    echo "Running fastp for ${SAMPLE}..."
    if command -v fastp &> /dev/null; then
        mkdir -p "${WORKDIR}/Preprocessed_Data/fastp"
        mkdir -p "${WORKDIR}/Logs/fastp"
        
        # Check if this is a single-end sample
        IS_SINGLE_END=0
        if [[ " $SINGLE_END_SAMPLES " =~ " $SAMPLE " ]]; then
            IS_SINGLE_END=1
        fi
        
        if [ $IS_SINGLE_END -eq 1 ]; then
            echo "Running fastp in single-end mode for ${SAMPLE}"
            # Use original filename format for single-end samples
            fastp -i "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}.fastq.gz" \
                  -o "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.fastq.gz" \
                  --json "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" \
                  --html "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html" \
                  --thread "${THREADS}" \
                  --report_title "${SAMPLE} Quality Report" \
                  > "${WORKDIR}/Logs/fastp/${SAMPLE}.log" 2>&1 || echo "WARNING: fastp failed for ${SAMPLE}"
        elif [ -s "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz" ]; then
            echo "Running fastp in paired-end mode for ${SAMPLE}"
            fastp -i "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz" \
                  -I "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R2.fastq.gz" \
                  -o "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
                  -O "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" \
                  --json "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" \
                  --html "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html" \
                  --thread "${THREADS}" \
                  --report_title "${SAMPLE} Quality Report" \
                  > "${WORKDIR}/Logs/fastp/${SAMPLE}.log" 2>&1 || echo "WARNING: fastp failed for ${SAMPLE}"
        else
            echo "R2 file is empty but not marked as single-end, running fastp in single-end mode for ${SAMPLE}"
            fastp -i "${WORKDIR}/Preprocessed_Data/combined/${SAMPLE}_R1.fastq.gz" \
                  -o "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
                  --json "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" \
                  --html "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html" \
                  --thread "${THREADS}" \
                  --report_title "${SAMPLE} Quality Report" \
                  > "${WORKDIR}/Logs/fastp/${SAMPLE}.log" 2>&1 || echo "WARNING: fastp failed for ${SAMPLE}"
            
            # Create an empty R2 file for downstream compatibility
            touch "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz"
        fi
        
        echo "Fastp reports generated for ${SAMPLE}:"
        ls -la "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.json" "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.html"
    else
        echo "WARNING: fastp not available, skipping quality trimming for ${SAMPLE}"
    fi
    
    # Step 3: Run dehost on fastp output
    echo "Running dehost for ${SAMPLE}..."
    if command -v bowtie2 &> /dev/null; then
        
        # Check if Bowtie2 index exists (either as direct path or in a subdirectory)
        if [ -e "${CORN_DB}.1.bt2" ] || [ -e "${CORN_DB}.1.bt2l" ] || [ -e "${CORN_DB}/corn_db.1.bt2" ] || [ -e "${CORN_DB}/corn_db.1.bt2l" ] || [ -e "${CORN_DB}/Zm-B73-REFERENCE-NAM-5.0.1.bt2" ] || [ -e "${CORN_DB}/Zm-B73-REFERENCE-NAM-5.0.1.bt2l" ]; then
            echo "Bowtie2 index found, proceeding with dehosting..."
            
            # Determine the correct index path
            if [ -e "${CORN_DB}.1.bt2" ] || [ -e "${CORN_DB}.1.bt2l" ]; then
                INDEX_PATH="${CORN_DB}"
            elif [ -e "${CORN_DB}/corn_db.1.bt2" ] || [ -e "${CORN_DB}/corn_db.1.bt2l" ]; then
                INDEX_PATH="${CORN_DB}/corn_db"
            elif [ -e "${CORN_DB}/Zm-B73-REFERENCE-NAM-5.0.1.bt2" ] || [ -e "${CORN_DB}/Zm-B73-REFERENCE-NAM-5.0.1.bt2l" ]; then
                INDEX_PATH="${CORN_DB}/Zm-B73-REFERENCE-NAM-5.0"
            fi
            echo "Using Bowtie2 index at: ${INDEX_PATH}"
            
            # Check if this is a single-end sample
            IS_SINGLE_END=0
            if [[ " $SINGLE_END_SAMPLES " =~ " $SAMPLE " ]]; then
                IS_SINGLE_END=1
            fi
            
            if [ $IS_SINGLE_END -eq 1 ]; then
                echo "Running bowtie2 in single-end mode for ${SAMPLE}"
                # Run bowtie2 with single-end mode using original filename format
                bowtie2 -p "${THREADS}" -x "${INDEX_PATH}" \
                        -U "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.fastq.gz" \
                        -S /dev/null \
                        --un-gz "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost.fastq.gz"
                
                echo "Dehosting completed for single-end sample ${SAMPLE}"
            elif [ -s "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" ]; then
                echo "Running bowtie2 in paired-end mode for ${SAMPLE}"
                # Run bowtie2 with paired-end mode
                bowtie2 -p "${THREADS}" -x "${INDEX_PATH}" \
                        -1 "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
                        -2 "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" \
                        -S /dev/null \
                        --un-conc-gz "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R%.fastq.gz"
            else
                echo "R2 file is empty but not marked as single-end, running bowtie2 in single-end mode for ${SAMPLE}"
                # Run bowtie2 with single-end mode
                bowtie2 -p "${THREADS}" -x "${INDEX_PATH}" \
                        -U "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" \
                        -S /dev/null \
                        --un-gz "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
                
                # Create an empty R2 file for downstream compatibility
                touch "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
            fi
            
            # Check if dehosting was successful
            if [ $IS_SINGLE_END -eq 1 ]; then
                # Check if single-end dehosting was successful
                if [ ! -f "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost.fastq.gz" ]; then
                    # Fallback: use fastp output as dehosted files
                    cp "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost.fastq.gz"
                    echo "Using fastp output as dehosted file for ${SAMPLE}"
                fi
            else
                # Check if paired-end dehosting was successful
                if [ ! -f "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz" ]; then
                    # Fallback: use fastp output as dehosted files
                    cp "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
                fi
                
                if [ ! -f "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz" ]; then
                    if [ -s "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" ]; then
                        # If original R2 exists and has content, copy it
                        cp "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
                    else
                        # Create empty R2 file for other cases
                        touch "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
                    fi
                fi
            fi
            
            echo "Dehosting completed for ${SAMPLE}"
        else
            echo "WARNING: Bowtie2 index not found at ${CORN_DB}, using fastp output instead"
            # Use fastp output as dehosted files
            ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
            
            # Check if this is a single-end sample
            if [[ " $SINGLE_END_SAMPLES " =~ " $SAMPLE " ]] || grep -q "SINGLE_END_MARKER" "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" 2>/dev/null; then
                echo "SINGLE_END_MARKER" > "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
            elif [ -s "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" ]; then
                ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
            else
                touch "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
            fi
        fi
    else
        echo "WARNING: bowtie2 not available, using fastp output instead"
        # Use fastp output as dehosted files
        ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R1.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz"
        
        # Check if this is a single-end sample
        if [[ " $SINGLE_END_SAMPLES " =~ " $SAMPLE " ]] || grep -q "SINGLE_END_MARKER" "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" 2>/dev/null; then
            echo "SINGLE_END_MARKER" > "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
        elif [ -s "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" ]; then
            ln -sf "${WORKDIR}/Preprocessed_Data/fastp/${SAMPLE}_R2.fastq.gz" "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
        else
            touch "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz"
        fi
    fi
    
    # Step 4: Run kraken2 if available
    if command -v kraken2 &> /dev/null; then
        echo "Running kraken2 for taxonomic classification for ${SAMPLE}..."
        
        # Check if this is a single-end sample
        IS_SINGLE_END=0
        if [[ " $SINGLE_END_SAMPLES " =~ " $SAMPLE " ]]; then
            IS_SINGLE_END=1
        fi
        
        if [ $IS_SINGLE_END -eq 1 ]; then
            echo "Running kraken2 in single-end mode for ${SAMPLE}"
            kraken2 --db "${KRAKEN_DB}" \
                    --threads "${THREADS}" \
                    --output "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken" \
                    --report "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                    "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost.fastq.gz" || echo "WARNING: kraken2 failed for ${SAMPLE}"
        elif [ -s "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz" ]; then
            echo "Running kraken2 in paired-end mode for ${SAMPLE}"
            kraken2 --db "${KRAKEN_DB}" \
                    --threads "${THREADS}" \
                    --paired \
                    --output "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken" \
                    --report "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                    "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz" \
                    "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R2.fastq.gz" || echo "WARNING: kraken2 failed for ${SAMPLE}"
        else
            echo "R2 file is empty but not marked as single-end, running kraken2 in single-end mode for ${SAMPLE}"
            kraken2 --db "${KRAKEN_DB}" \
                    --threads "${THREADS}" \
                    --output "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken" \
                    --report "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                    "${WORKDIR}/Preprocessed_Data/${SAMPLE}_dehost_R1.fastq.gz" || echo "WARNING: kraken2 failed for ${SAMPLE}"
        fi
        
        # Ensure output files exist (to prevent workflow failures)
        if [ ! -f "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken" ]; then
            echo "WARNING: Creating empty kraken output file" 
            touch "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.kraken"
        fi
        
        if [ ! -f "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" ]; then
            echo "WARNING: Creating empty kraken report file"
            touch "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report"
        fi
    else
        echo "WARNING: kraken2 not available, skipping taxonomic classification for ${SAMPLE}"
    fi
    
    # Run bracken if available
    if command -v bracken &> /dev/null && [ -f "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" ]; then
        echo "Running bracken for abundance estimation for ${SAMPLE}..."
        bracken -d "${KRAKEN_DB}" \
                -i "${RESULTS_DIR}/Taxonomic_Profiling/1_DNA_Kraken2/${SAMPLE}.report" \
                -o "${RESULTS_DIR}/Taxonomic_Profiling/2_DNA_Bracken/${SAMPLE}.bracken" \
                -r 150 -l S -t 10 || echo "WARNING: bracken failed for ${SAMPLE}"
    else
        echo "WARNING: bracken not available or kraken2 report missing, skipping abundance estimation for ${SAMPLE}"
    fi
done

# Generate abundance matrix
echo "Generating abundance matrix..."
python "${WORKDIR}/scripts/create_abundance_matrix.py" \
    --input_files "${RESULTS_DIR}"/Taxonomic_Profiling/2_DNA_Bracken/*.bracken \
    --output "${RESULTS_DIR}/Taxonomic_Profiling/5_DNA_Relative_Abundance_Matrix_Python/abundance_matrix.csv" || \
    echo "WARNING: Failed to generate abundance matrix"

echo "emtax workflow finished at $(date)"
