# -*- coding: utf-8 -*-
"""pig_audio_processing.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/github/xu-maxwell/long-lab-audio-processing/blob/main/pig_audio_processing.ipynb
"""


import os
import numpy as np
import scipy.io.wavfile as wavfile
from pydub import AudioSegment
from scipy.signal import find_peaks
import csv
import tempfile
from boxsdk import Client, OAuth2
import scipy.signal

def list_audio_files_recursive(folder_id):
    audio_extensions = ['.mp3', '.wav', '.flac', '.ogg', '.aac', '.m4a']  # Common audio formats
    audio_files = []

    # Get items from the current folder
    folder_items = client.folder(folder_id).get_items()

    for item in folder_items:
        if item.type == 'file':
            # Check if the file has an audio extension
            if any(item.name.lower().endswith(ext) for ext in audio_extensions):
                audio_files.append(item.id)
        elif item.type == 'folder':
            # Recursively check inside the subfolder
            # print(f"Checking inside folder: {item.name}")
            audio_files.extend(list_audio_files_recursive(item.id))

    return audio_files

def download_box_file(file_id):
    """Download a file from Box using its file ID and return the local temporary file path."""
    box_file = client.file(file_id).get()

    # Create a temporary file to store the downloaded content
    with tempfile.NamedTemporaryFile(delete=False, suffix=os.path.splitext(box_file.name)[1]) as temp_file:
        temp_file.write(client.file(file_id).content())
        temp_file_path = temp_file.name

    return temp_file_path, box_file.name


def find_timepoints(audio_file_id, timepoints_folder_id):
    # Download the file from Box using the file ID
    temp_file_path, file_name = download_box_file(audio_file_id)

    # Parameters for sound detection
    frame_duration = 0.02  # 20 ms
    overlap_duration = 0.01  # 10 ms
    threshold = 0.05  # Sound threshold
    min_sound_duration = 1  # Minimum duration of sound in seconds
    padding_duration = 0.8  # Duration of audio to include before and after the sound event in seconds

    # Use pydub to read the downloaded audio file
    try:
        audio_segment = AudioSegment.from_file(temp_file_path)
        fs = audio_segment.frame_rate
        audio_in = np.array(audio_segment.get_array_of_samples()).astype(np.float32) / (2 ** 15)

        # If stereo, convert to mono by averaging the channels
        if audio_segment.channels > 1:
            audio_in = audio_in.reshape((-1, audio_segment.channels))
            audio_in = np.mean(audio_in, axis=1)
    except Exception as e:
        print(f"Error reading the audio file {file_name}: {e}")
        return None

    # Calculate frame length and overlap length
    frame_length = int(frame_duration * fs)
    overlap_length = int(overlap_duration * fs)
    min_sound_frames = int(min_sound_duration / frame_duration)
    padding_samples = int(padding_duration * fs)

    # Divide the audio into frames
    total_samples = len(audio_in)
    step_size = frame_length - overlap_length
    frames = np.lib.stride_tricks.sliding_window_view(audio_in, frame_length)[::step_size]

    # Calculate RMS amplitude of each frame
    rms_amplitudes = np.array([np.sqrt(np.mean(np.square(frame))) for frame in frames])

    # Detect frames with amplitude above the threshold
    is_sound = rms_amplitudes > threshold

    # Identify start and end indices of sound segments
    sound_start_indices = np.where(np.diff(np.concatenate(([0], is_sound))) == 1)[0]
    sound_end_indices = np.where(np.diff(np.concatenate((is_sound, [0]))) == -1)[0]

    # Ensure segments meet the minimum duration requirement
    valid_segments = (sound_end_indices - sound_start_indices) >= min_sound_frames
    sound_start_indices = sound_start_indices[valid_segments]
    sound_end_indices = sound_end_indices[valid_segments]

    # Prepare the output matrix and CSV data
    timepoints_matrix = []
    csv_data = []
    for start_idx, end_idx in zip(sound_start_indices, sound_end_indices):
        start_time = max(0, start_idx * step_size - padding_samples) / fs
        end_time = min(total_samples, (end_idx + 1) * step_size + padding_samples) / fs
        timepoints_matrix.append([start_time, end_time])
        csv_data.append([file_name, start_time, end_time])

    timepoints_matrix = np.array(timepoints_matrix)

    # Generate CSV output
    csv_filename = f"{os.path.splitext(file_name)[0]}_timepoints.csv"
    with open(csv_filename, mode='w', newline='') as file:
        writer = csv.writer(file)
        writer.writerow(["audio_file_name", "start_time", "end_time"])
        writer.writerows(csv_data)

    print(f"CSV file created: {csv_filename}")

    # Upload the CSV file to the specified Box folder
    upload_file_to_box(timepoints_folder_id, csv_filename, csv_filename)

    # Clean up the temporary audio file and CSV file
    os.remove(temp_file_path)
    os.remove(csv_filename)

    # Return the timepoints matrix (start_time, end_time)
    return timepoints_matrix


def snippet_extraction(audio_file_id, snippets_folder_id, timepoints):
    # Download the audio file from Box using the file ID
    temp_file_path, file_name = download_box_file(audio_file_id)

    # Load the audio file using pydub
    try:
        audio_segment = AudioSegment.from_file(temp_file_path)
    except Exception as e:
        print(f"Error reading the audio file {file_name}: {e}")
        return None

    # Iterate over the timepoints matrix and extract each snippet
    for idx, (start_time, end_time) in enumerate(timepoints):
        # Convert time from seconds to milliseconds for pydub
        start_ms = int(start_time * 1000)  # start time in milliseconds
        end_ms = int(end_time * 1000)      # end time in milliseconds

        # Extract the snippet from the audio segment
        snippet = audio_segment[start_ms:end_ms]

        # Create a temporary file for the snippet
        snippet_filename = f"{os.path.splitext(file_name)[0]}_snippet_{idx+1}.wav"
        snippet_file_path = os.path.join(tempfile.gettempdir(), snippet_filename)

        # Export the snippet as a WAV file
        snippet.export(snippet_file_path, format="wav")

        # Upload the snippet to the specified Box folder
        upload_file_to_box(snippets_folder_id, snippet_file_path, snippet_filename)

        # Clean up the temporary snippet file
        os.remove(snippet_file_path)

    # Clean up the downloaded audio file
    os.remove(temp_file_path)

    print("All snippets have been uploaded.")

def upload_file_to_box(folder_id, file_path, file_name):
    """Upload a file to the specified Box folder."""
    folder = client.folder(folder_id).get()
    uploaded_file = folder.upload(file_path, file_name=file_name)
    print(f"File '{file_name}' uploaded to folder '{folder.name}' (ID: {folder_id}).")
    return uploaded_file

def start(audio_folder_id, timepoints_folder_id, snippets_folder_id):
    all_audio_files = list_audio_files_recursive(audio_folder_id)
    print("Audio files in " + client.folder(audio_folder_id).get().name + " recursively found.")

    for file_id in all_audio_files:
        file_path = None  # Initialize file_path to None for error handling

        try:
            # Retrieve the file object using the file ID
            file = client.file(file_id).get()

            # Retrieve the parent folder name
            parent_folder_id = file.parent['id']
            parent_folder_name = client.folder(parent_folder_id).get().name

            # Construct the file path
            file_path = parent_folder_name + "/" + file.name

            # Proceed with processing
            timepoints = find_timepoints(file_id, timepoints_folder_id)
            print("Timepoints found for " + file_path)

            snippet_extraction(file_id, snippets_folder_id, timepoints)
            print("Snippets extracted for " + file_path)

        except Exception as e:
            # If file_path is not constructed, fall back to file_id in the error message
            error_message = file_path if file_path else file_id
            print(f"An error occurred while processing {error_message}: {e}")
            continue

def begin():
  global client
  # Ask user to input the developer token
  developer_token = input("Please enter your developer token: ")

  # Set up OAuth2 and Box Client
  oauth2 = OAuth2(None, None, access_token=developer_token)
  client = Client(oauth2)

  # Test the connection by fetching your user info
  user = client.user().get()
  print(f"Connected to Box account: {user.name}")

  # Ask user to input the folder IDs
  audio_folder_id = input("Please enter the audio folder ID: ")
  timepoints_folder_id = input("Please enter the timepoints folder ID: ")
  snippets_folder_id = input("Please enter the snippets folder ID: ")

  # Convert folder IDs to integers (if necessary)
  audio_folder_id = int(audio_folder_id)
  timepoints_folder_id = int(timepoints_folder_id)
  snippets_folder_id = int(snippets_folder_id)

  # Call the start function with the provided folder IDs
  start(audio_folder_id, timepoints_folder_id, snippets_folder_id)

