import logging
import os
from datetime import datetime
from typing import cast

from docx import Document
from docx.enum.table import WD_ALIGN_VERTICAL, WD_TABLE_ALIGNMENT
from docx.enum.text import WD_ALIGN_PARAGRAPH
from docx.opc.constants import RELATIONSHIP_TYPE
from docx.oxml import OxmlElement
from docx.oxml.ns import qn
from docx.shared import Inches, Pt
from docx.styles.style import ParagraphStyle
from docx.text.paragraph import Paragraph

from orbis.data.models import Figure, PodStats, ReportMetadata, WorkerQueueStats

logger = logging.getLogger("root")


class DocxReportGenerator:
    def __init__(self, metadata: ReportMetadata, progress_callback=None):
        self.progress_callback = progress_callback
        self.metadata = metadata
        self.doc = Document()
        # Set default font to Calibri
        style = cast(ParagraphStyle, self.doc.styles["Normal"])
        style.font.name = "Calibri"
        self.output_folder = f"output/{self.metadata.organization_name}"

    def setup_document(self):
        self.set_margins()
        self.add_header_footer()
        self.add_title_page()

    def set_margins(self):
        for section in self.doc.sections:
            section.top_margin = Inches(0.5)
            section.bottom_margin = Inches(0.5)
            section.left_margin = Inches(0.5)
            section.right_margin = Inches(0.5)

    def add_header_footer(self):
        section = self.doc.sections[0]
        footer = section.footer
        footer_para = footer.paragraphs[0]
        footer_para.text = "Generated by Orbis | Astronomer"
        footer_style = cast(ParagraphStyle, footer_para.style)
        footer_style.font.size = Pt(8)
        footer_style.font.name = "Calibri"
        footer_para.alignment = WD_ALIGN_PARAGRAPH.RIGHT

    def add_title_page(self):
        self.add_blank_lines(2)
        report_title = self.doc.add_heading("Deployment Compute Report", level=1)
        self.add_blank_lines(3)

        org_name = self.doc.add_heading("Organization Name: " + self.metadata.organization_name, level=2)
        self.add_blank_lines(1)
        # Since start_date and end_date are strings, we'll display them as-is
        report_time_frame = self.doc.add_heading(f"Report Time Frame: {self.metadata.start_date} - {self.metadata.end_date}", level=2)

        self.add_blank_lines(3)
        # Add divider line (simplified - border_bottom not supported in this way)
        divider = self.doc.add_paragraph()
        divider.add_run("_" * 50)  # Simple text divider

        self.add_blank_lines(2)
        namespaces_header = self.doc.add_heading("Namespaces:", level=2)
        namespaces_header.alignment = WD_ALIGN_PARAGRAPH.LEFT

        # Add namespaces as bullet points
        for namespace in self.metadata.namespaces:
            formatted_namespace = namespace.replace("-", "\u2011")  # Use non-breaking hyphen
            bullet_para = self.doc.add_paragraph(style="List Bullet")
            bullet_para.paragraph_format.left_indent = Inches(0.5)
            bullet_para.add_run(formatted_namespace)

        self.add_blank_lines(4)
        created_on = self.doc.add_paragraph("Created on: " + datetime.now().strftime("%m/%d/%Y, %H:%M:%S"))
        created_on.paragraph_format.space_after = Pt(30)

        # Style formatting
        title_style = cast(ParagraphStyle, report_title.style)
        title_style.font.size = Pt(40)
        title_style.font.bold = True
        title_style.font.italic = False
        title_style.font.name = "Calibri"

        org_style = cast(ParagraphStyle, org_name.style)
        org_style.font.size = Pt(30)
        org_style.font.name = "Calibri"

        timeframe_style = cast(ParagraphStyle, report_time_frame.style)
        timeframe_style.font.size = Pt(18)
        timeframe_style.font.name = "Calibri"

        header_style = cast(ParagraphStyle, namespaces_header.style)
        header_style.font.size = Pt(18)
        header_style.font.name = "Calibri"
        for para in self.doc.paragraphs:
            para_cast = cast(Paragraph, para)
            if para_cast.style and para_cast.style.name == "List Bullet":
                bullet_style = cast(ParagraphStyle, para_cast.style)
                bullet_style.font.size = Pt(16)
                bullet_style.font.name = "Calibri"

        created_style = cast(ParagraphStyle, created_on.style)
        created_style.font.size = Pt(12)
        created_style.font.name = "Calibri"

        # Alignment
        report_title.alignment = WD_ALIGN_PARAGRAPH.CENTER
        org_name.alignment = WD_ALIGN_PARAGRAPH.CENTER
        report_time_frame.alignment = WD_ALIGN_PARAGRAPH.CENTER
        created_on.alignment = WD_ALIGN_PARAGRAPH.RIGHT

    def add_figure(self, figure: Figure):
        if figure.metric.include_in and not figure.metric.include_in.get("docx", True):
            return

        table = self.doc.add_table(rows=2, cols=1)
        table.alignment = WD_TABLE_ALIGNMENT.CENTER

        cell = table.cell(0, 0)
        cell.text = f"NAMESPACE: {figure.namespace}      METRIC: {figure.metric.metric_name}"
        cell.paragraphs[0].alignment = WD_ALIGN_PARAGRAPH.CENTER

        cell = table.cell(1, 0)
        cell.paragraphs[0].add_run().add_picture(figure.image_path, width=Inches(7.5))
        cell.vertical_alignment = WD_ALIGN_VERTICAL.CENTER

    def add_visualization_url(self, figure: Figure):
        if not figure.url or (figure.metric.include_in and not figure.metric.include_in.get("docx", True)):
            return
        paragraph = self.doc.add_paragraph()
        self.add_hyperlink(paragraph, figure.url, "View in Prometheus")

    def _format_cell(self, cell, text):
        """Helper function to format cell consistently"""
        cell.vertical_alignment = WD_ALIGN_VERTICAL.CENTER
        paragraph = cell.paragraphs[0]
        paragraph.alignment = WD_ALIGN_PARAGRAPH.CENTER
        run = paragraph.add_run(str(text))
        run.font.name = "Calibri"

    def _format_header_row(self, table, headers):
        """Format the header row of the table"""
        header_row = table.rows[0]
        for j, text in enumerate(headers):
            cell = header_row.cells[j]
            cell.vertical_alignment = WD_ALIGN_VERTICAL.CENTER
            paragraph = cell.paragraphs[0]
            paragraph.alignment = WD_ALIGN_PARAGRAPH.CENTER
            run = paragraph.add_run(text)
            run.font.bold = True
            run.font.name = "Calibri"

    def _create_table(self, figure: Figure):
        """Create and configure the appropriate table based on figure data"""
        if figure.pod_stats:
            table = self.doc.add_table(rows=len(figure.pod_stats) + 1, cols=6)
            headers = ["Pod Type", "Mean", "Median", "Max", "Min", "P90"]
        elif figure.worker_queue_stats:
            table = self.doc.add_table(rows=len(figure.worker_queue_stats) + 1, cols=5)
            headers = ["Worker Queue", "Mean", "Median", "Max", "Min"]
        else:
            table = self.doc.add_table(rows=2, cols=5)
            headers = ["Metric", "Mean", "Median", "Max", "Min"]

        table.style = "Table Grid"
        table.alignment = WD_TABLE_ALIGNMENT.CENTER
        self._format_header_row(table, headers)
        return table

    def _populate_pod_stats(self, table, pod_stats: list[PodStats]):
        """Populate table with pod statistics"""
        for idx, pod_stat in enumerate(pod_stats):
            row = table.rows[idx + 1].cells
            row_data = [
                pod_stat.pod_type,
                f"{pod_stat.mean_value:.2f}",
                f"{pod_stat.median_value:.2f}",
                f"{getattr(pod_stat, 'max_value', getattr(pod_stat, 'max_value', None)):.2f}",
                f"{getattr(pod_stat, 'min_value', getattr(pod_stat, 'min_value', None)):.2f}",
                f"{pod_stat.p90_value:.2f}",
            ]
            for j, text in enumerate(row_data):
                self._format_cell(row[j], text)

    def _populate_queue_stats(self, table, queue_stats: list[WorkerQueueStats]):
        """Populate table with worker queue statistics"""
        for idx, queue_stat in enumerate(queue_stats):
            row = table.rows[idx + 1].cells
            row_data = [
                queue_stat.queue_name,
                f"{queue_stat.mean_value:.2f}",
                f"{queue_stat.median_value:.2f}",
                f"{queue_stat.max_value:.2f}",
                f"{queue_stat.min_value:.2f}",
            ]
            for j, text in enumerate(row_data):
                self._format_cell(row[j], text)

    def _populate_basic_stats(self, table, statistics):
        """Populate table with basic statistics"""
        row = table.rows[1].cells
        self._format_cell(row[0], "Values")
        self._format_cell(row[1], f"{statistics['mean_value']:.2f}")
        self._format_cell(row[2], f"{statistics['median_value']:.2f}")
        self._format_cell(row[3], f"{statistics['max_value']:.2f}")
        self._format_cell(row[4], f"{statistics['min_value']:.2f}")

    def add_statistics_table(self, figure: Figure):
        """Add statistics table to document based on figure data"""
        if figure.metric.include_in and not figure.metric.include_in.get("docx", True):
            return

        table = self._create_table(figure)

        if figure.pod_stats:
            self._populate_pod_stats(table, figure.pod_stats)
        elif figure.worker_queue_stats:
            self._populate_queue_stats(table, figure.worker_queue_stats)
        else:
            self._populate_basic_stats(table, figure.statistics)

    def add_hyperlink(self, paragraph, url, text):
        part = paragraph.part
        r_id = part.relate_to(url, RELATIONSHIP_TYPE.HYPERLINK, is_external=True)
        hyperlink = OxmlElement("w:hyperlink")
        hyperlink.set(qn("r:id"), r_id)
        new_run = OxmlElement("w:r")
        rPr = OxmlElement("w:rPr")
        rStyle = OxmlElement("w:rStyle")
        rStyle.set(qn("w:val"), "Hyperlink")
        rPr.append(rStyle)
        new_run.append(rPr)
        # Create text element instead of setting text directly
        t = OxmlElement("w:t")
        t.text = text  # type: ignore  # OxmlElement text assignment works at runtime
        new_run.append(t)
        hyperlink.append(new_run)
        paragraph._p.append(hyperlink)

    def add_blank_lines(self, number_of_lines: int):
        for _ in range(number_of_lines):
            self.doc.add_paragraph()

    def save_document(self):
        os.makedirs(self.output_folder, exist_ok=True)
        self.doc.save(f"{self.output_folder}/report.docx")
        if self.progress_callback:
            self.progress_callback()
