/**
 * (experimental) Bundling options.
 *
 * @experimental
 */
export interface BundlingOptions {
    /**
     * (experimental) The Docker image where the command will run.
     *
     * @experimental
     */
    readonly image: BundlingDockerImage;
    /**
     * (experimental) The command to run in the Docker container.
     *
     * @default - run the command defined in the image
     * @see https://docs.docker.com/engine/reference/run/
     * @experimental
     * @example
     *
     * ['npm', 'install']
     */
    readonly command?: string[];
    /**
     * (experimental) Additional Docker volumes to mount.
     *
     * @default - no additional volumes are mounted
     * @experimental
     */
    readonly volumes?: DockerVolume[];
    /**
     * (experimental) The environment variables to pass to the Docker container.
     *
     * @default - no environment variables.
     * @experimental
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * (experimental) Working directory inside the Docker container.
     *
     * @default /asset-input
     * @experimental
     */
    readonly workingDirectory?: string;
    /**
     * (experimental) The user to use when running the Docker container.
     *
     * user | user:group | uid | uid:gid | user:gid | uid:group
     *
     * @default - uid:gid of the current user or 1000:1000 on Windows
     * @see https://docs.docker.com/engine/reference/run/#user
     * @experimental
     */
    readonly user?: string;
    /**
     * (experimental) Local bundling provider.
     *
     * The provider implements a method `tryBundle()` which should return `true`
     * if local bundling was performed. If `false` is returned, docker bundling
     * will be done.
     *
     * @default - bundling will only be performed in a Docker container
     * @experimental
     */
    readonly local?: ILocalBundling;
}
/**
 * (experimental) Local bundling.
 *
 * @experimental
 */
export interface ILocalBundling {
    /**
     * (experimental) This method is called before attempting docker bundling to allow the bundler to be executed locally.
     *
     * If the local bundler exists, and bundling
     * was performed locally, return `true`. Otherwise, return `false`.
     *
     * @param outputDir the directory where the bundled asset should be output.
     * @param options bundling options for this asset.
     * @experimental
     */
    tryBundle(outputDir: string, options: BundlingOptions): boolean;
}
/**
 * A Docker image used for asset bundling.
 */
export declare class BundlingDockerImage {
    readonly image: string;
    private readonly _imageHash?;
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     */
    static fromRegistry(image: string): BundlingDockerImage;
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     */
    static fromAsset(path: string, options?: DockerBuildOptions): BundlingDockerImage;
    /** @param image The Docker image */
    private constructor();
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     */
    toJSON(): string;
    /**
     * Runs a Docker image.
     */
    run(options?: DockerRunOptions): void;
    /**
     * Copies a file or directory out of the Docker image to the local filesystem.
     */
    cp(imagePath: string, outputPath: string): void;
}
/**
 * A Docker volume.
 */
export interface DockerVolume {
    /**
     * The path to the file or directory on the host machine.
     */
    readonly hostPath: string;
    /**
     * The path where the file or directory is mounted in the container.
     */
    readonly containerPath: string;
    /**
     * Mount consistency.
     *
     * Only applicable for macOS
     *
     * @default DockerConsistency.DELEGATED
     * @see https://docs.docker.com/storage/bind-mounts/#configure-mount-consistency-for-macos
     */
    readonly consistency?: DockerVolumeConsistency;
}
/**
 * Supported Docker volume consistency types.
 *
 * Only valid on macOS due to the way file storage works on Mac
 */
export declare enum DockerVolumeConsistency {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes.
     */
    CONSISTENT = "consistent",
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine.
     */
    DELEGATED = "delegated",
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container.
     */
    CACHED = "cached"
}
/**
 * Docker run options.
 */
export interface DockerRunOptions {
    /**
     * The command to run in the container.
     *
     * @default - run the command defined in the image
     */
    readonly command?: string[];
    /**
     * Docker volumes to mount.
     *
     * @default - no volumes are mounted
     */
    readonly volumes?: DockerVolume[];
    /**
     * The environment variables to pass to the container.
     *
     * @default - no environment variables.
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Working directory inside the container.
     *
     * @default - image default
     */
    readonly workingDirectory?: string;
    /**
     * The user to use when running the container.
     *
     * @default - root or image default
     */
    readonly user?: string;
}
/**
 * Docker build options.
 */
export interface DockerBuildOptions {
    /**
     * Build args.
     *
     * @default - no build args
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
    /**
     * Name of the Dockerfile.
     *
     * @default - The Dockerfile immediately within the build context path
     */
    readonly file?: string;
}
