"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostgresqlDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const triggers_1 = require("aws-cdk-lib/triggers");
const base_database_1 = require("./base-database");
const HANDLER_PATH = path.join(__dirname, "../../assets/rds/triggers/pgsql");
/**
 * A PostgreSQL database.
 */
class PostgresqlDatabase extends base_database_1.BaseDatabase {
    /**
     * Creates a new PostgresqlDatabase.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param props - The configuration properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.ownerSecrets = [];
        this.readerSecrets = [];
        this.unprivilegedSecrets = [];
        const { adminSecret, vpc, ownerSecret, schemaName, encoding = "UTF8", } = props;
        const environment = {
            AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
            ADMIN_SECRET_ARN: adminSecret.secretArn,
            OWNER_SECRET_ARN: ownerSecret.secretArn,
            OWNER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.ownerSecrets.map((s) => s.secretArn)),
            }),
            READER_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.readerSecrets.map((s) => s.secretArn)),
            }),
            UNPRIVILEGED_SECRETS: aws_cdk_lib_1.Lazy.string({
                produce: () => JSON.stringify(this.unprivilegedSecrets.map((s) => s.secretArn)),
            }),
            DB_NAME: this.databaseName,
            SCHEMA_NAME: schemaName || "",
            DB_ENCODING: encoding,
        };
        if (props.locale) {
            environment.DB_LOCALE = props.locale;
        }
        if (props.certificateAuthoritiesUrl) {
            environment.CA_CERTS_URL = props.certificateAuthoritiesUrl;
        }
        this.lambdaFunction = new aws_lambda_1.Function(this, "Function", {
            runtime: aws_lambda_1.Runtime.NODEJS_20_X,
            code: aws_lambda_1.Code.fromAsset(HANDLER_PATH),
            handler: "index.handler",
            description: "Creates a schema and possibly some secondary users",
            vpc,
            vpcSubnets: this.subnetSelection,
            securityGroups: [this.securityGroup],
            environment,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
        });
        adminSecret.grantRead(this.lambdaFunction);
        ownerSecret.grantRead(this.lambdaFunction);
        this.trigger = new triggers_1.Trigger(this, "Trigger", {
            handler: this.lambdaFunction,
        });
        this.trigger.executeAfter(adminSecret, ownerSecret, this.lambdaFunction);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forCluster(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            vpc: cluster.vpc,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forClusterFromSnapshot(scope, id, cluster, options) {
        // The DatabaseClusterFromSnapshot type is equivalent to DatabaseCluster.
        return PostgresqlDatabase.forCluster(scope, id, cluster, options);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseCluster.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessCluster(scope, id, cluster, options) {
        // The ServerlessClusterFromSnapshot type is a subset of ServerlessCluster.
        return PostgresqlDatabase.forServerlessClusterFromSnapshot(scope, id, cluster, options);
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseClusterFromSnapshot.
     *
     * This method automatically adds the cluster to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param cluster - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forServerlessClusterFromSnapshot(scope, id, cluster, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || cluster.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: cluster,
            endpoint: cluster.clusterEndpoint,
            adminSecret: clusterSecret,
            ...props,
        });
        database.trigger.executeAfter(cluster);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseInstance.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstance(scope, id, instance, options) {
        const props = { ...options };
        let clusterSecret = props.adminSecret || instance.secret;
        if (clusterSecret === undefined) {
            throw new Error("You must provide either the adminSecret property or a cluster with an attached secret");
        }
        delete props.adminSecret;
        const database = new PostgresqlDatabase(scope, id, {
            target: instance,
            endpoint: instance.instanceEndpoint,
            adminSecret: clusterSecret,
            vpc: instance.vpc,
            ...props,
        });
        database.trigger.executeAfter(instance);
        return database;
    }
    /**
     * Create a new PostgresqlDatabase inside a DatabaseInstanceFromSnapshot.
     *
     * This method automatically adds the instance to the CloudFormation
     * dependencies of the CDK Trigger.
     *
     * @param scope - The Construct that contains this one.
     * @param id - The identifier of this construct.
     * @param instance - The database cluster construct.
     * @param options - The configuration properties for this construct.
     */
    static forInstanceFromSnapshot(scope, id, instance, options) {
        // The DatabaseInstanceFromSnapshot type is equivalent to DatabaseInstance.
        return PostgresqlDatabase.forInstance(scope, id, instance, options);
    }
    addUserAsOwner(secret) {
        this.ownerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsReader(secret) {
        this.readerSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
    addUserAsUnprivileged(secret) {
        this.unprivilegedSecrets.push(secret);
        secret.grantRead(this.lambdaFunction);
        this.trigger.executeAfter(secret);
    }
}
exports.PostgresqlDatabase = PostgresqlDatabase;
_a = JSII_RTTI_SYMBOL_1;
PostgresqlDatabase[_a] = { fqn: "shady-island.PostgresqlDatabase", version: "0.1.17" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGdzcWwtZGF0YWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvcmRzL3Bnc3FsLWRhdGFiYXNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLDZDQUE2QztBQUU3Qyx1REFBaUU7QUFVakUsbURBQXlEO0FBRXpELG1EQUl5QjtBQUV6QixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxpQ0FBaUMsQ0FBQyxDQUFDO0FBeUU3RTs7R0FFRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsNEJBQVk7SUF1TGxEOzs7Ozs7T0FNRztJQUNILFlBQ0UsS0FBaUIsRUFDakIsRUFBVSxFQUNWLEtBQThCO1FBRTlCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBaEJQLGlCQUFZLEdBQWMsRUFBRSxDQUFDO1FBQzdCLGtCQUFhLEdBQWMsRUFBRSxDQUFDO1FBQzlCLHdCQUFtQixHQUFjLEVBQUUsQ0FBQztRQWdCckQsTUFBTSxFQUNKLFdBQVcsRUFDWCxHQUFHLEVBQ0gsV0FBVyxFQUNYLFVBQVUsRUFDVixRQUFRLEdBQUcsTUFBTSxHQUNsQixHQUFHLEtBQUssQ0FBQztRQUVWLE1BQU0sV0FBVyxHQUEyQjtZQUMxQyxtQ0FBbUMsRUFBRSxHQUFHO1lBQ3hDLGdCQUFnQixFQUFFLFdBQVcsQ0FBQyxTQUFTO1lBQ3ZDLGdCQUFnQixFQUFFLFdBQVcsQ0FBQyxTQUFTO1lBQ3ZDLGFBQWEsRUFBRSxrQkFBSSxDQUFDLE1BQU0sQ0FBQztnQkFDekIsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUNaLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQzthQUM1RCxDQUFDO1lBQ0YsY0FBYyxFQUFFLGtCQUFJLENBQUMsTUFBTSxDQUFDO2dCQUMxQixPQUFPLEVBQUUsR0FBRyxFQUFFLENBQ1osSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO2FBQzdELENBQUM7WUFDRixvQkFBb0IsRUFBRSxrQkFBSSxDQUFDLE1BQU0sQ0FBQztnQkFDaEMsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUNaLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO2FBQ25FLENBQUM7WUFDRixPQUFPLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDMUIsV0FBVyxFQUFFLFVBQVUsSUFBSSxFQUFFO1lBQzdCLFdBQVcsRUFBRSxRQUFRO1NBQ3RCLENBQUM7UUFDRixJQUFJLEtBQUssQ0FBQyxNQUFNLEVBQUU7WUFDaEIsV0FBVyxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQ3RDO1FBQ0QsSUFBSSxLQUFLLENBQUMseUJBQXlCLEVBQUU7WUFDbkMsV0FBVyxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMseUJBQXlCLENBQUM7U0FDNUQ7UUFFRCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ25ELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7WUFDNUIsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQztZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixXQUFXLEVBQUUsb0RBQW9EO1lBQ2pFLEdBQUc7WUFDSCxVQUFVLEVBQUUsSUFBSSxDQUFDLGVBQWU7WUFDaEMsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQztZQUNwQyxXQUFXO1lBQ1gsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUM3QixDQUFDLENBQUM7UUFDSCxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUMzQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzQyxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksa0JBQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzFDLE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYztTQUM3QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUMzRSxDQUFDO0lBelBEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUN0QixLQUFnQixFQUNoQixFQUFVLEVBQ1YsT0FBd0IsRUFDeEIsT0FBNEM7UUFFNUMsTUFBTSxLQUFLLEdBQUcsRUFBRSxHQUFHLE9BQU8sRUFBRSxDQUFDO1FBQzdCLElBQUksYUFBYSxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUN4RCxJQUFJLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FDYix1RkFBdUYsQ0FDeEYsQ0FBQztTQUNIO1FBQ0QsT0FBTyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3pCLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQWtCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNqRCxNQUFNLEVBQUUsT0FBTztZQUNmLFFBQVEsRUFBRSxPQUFPLENBQUMsZUFBZTtZQUNqQyxXQUFXLEVBQUUsYUFBYTtZQUMxQixHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDaEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO1FBQ0gsUUFBUSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDdkMsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsc0JBQXNCLENBQ2xDLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixPQUFvQyxFQUNwQyxPQUE0QztRQUU1Qyx5RUFBeUU7UUFDekUsT0FBTyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsb0JBQW9CLENBQ2hDLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixPQUEwQixFQUMxQixPQUFzRDtRQUV0RCwyRUFBMkU7UUFDM0UsT0FBTyxrQkFBa0IsQ0FBQyxnQ0FBZ0MsQ0FDeEQsS0FBSyxFQUNMLEVBQUUsRUFDRixPQUFPLEVBQ1AsT0FBTyxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE1BQU0sQ0FBQyxnQ0FBZ0MsQ0FDNUMsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLE9BQXNDLEVBQ3RDLE9BQXNEO1FBRXRELE1BQU0sS0FBSyxHQUFHLEVBQUUsR0FBRyxPQUFPLEVBQUUsQ0FBQztRQUM3QixJQUFJLGFBQWEsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDeEQsSUFBSSxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQ2IsdUZBQXVGLENBQ3hGLENBQUM7U0FDSDtRQUNELE9BQU8sS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUN6QixNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFrQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDakQsTUFBTSxFQUFFLE9BQU87WUFDZixRQUFRLEVBQUUsT0FBTyxDQUFDLGVBQWU7WUFDakMsV0FBVyxFQUFFLGFBQWE7WUFDMUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO1FBQ0gsUUFBUSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDdkMsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUN2QixLQUFnQixFQUNoQixFQUFVLEVBQ1YsUUFBMEIsRUFDMUIsT0FBNEM7UUFFNUMsTUFBTSxLQUFLLEdBQUcsRUFBRSxHQUFHLE9BQU8sRUFBRSxDQUFDO1FBQzdCLElBQUksYUFBYSxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksUUFBUSxDQUFDLE1BQU0sQ0FBQztRQUN6RCxJQUFJLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FDYix1RkFBdUYsQ0FDeEYsQ0FBQztTQUNIO1FBQ0QsT0FBTyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3pCLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQWtCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNqRCxNQUFNLEVBQUUsUUFBUTtZQUNoQixRQUFRLEVBQUUsUUFBUSxDQUFDLGdCQUFnQjtZQUNuQyxXQUFXLEVBQUUsYUFBYTtZQUMxQixHQUFHLEVBQUUsUUFBUSxDQUFDLEdBQUc7WUFDakIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO1FBQ0gsUUFBUSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDeEMsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsdUJBQXVCLENBQ25DLEtBQWdCLEVBQ2hCLEVBQVUsRUFDVixRQUFzQyxFQUN0QyxPQUE0QztRQUU1QywyRUFBMkU7UUFDM0UsT0FBTyxrQkFBa0IsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdEUsQ0FBQztJQThFTSxjQUFjLENBQUMsTUFBZTtRQUNuQyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMvQixNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRU0sZUFBZSxDQUFDLE1BQWU7UUFDcEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDaEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVNLHFCQUFxQixDQUFDLE1BQWU7UUFDMUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN0QyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNwQyxDQUFDOztBQTVRSCxnREE2UUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBEdXJhdGlvbiwgTGF6eSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgSVZwYyB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtZWMyXCI7XG5pbXBvcnQgeyBDb2RlLCBGdW5jdGlvbiwgUnVudGltZSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtbGFtYmRhXCI7XG5pbXBvcnQge1xuICBEYXRhYmFzZUNsdXN0ZXIsXG4gIERhdGFiYXNlQ2x1c3RlckZyb21TbmFwc2hvdCxcbiAgRGF0YWJhc2VJbnN0YW5jZSxcbiAgRGF0YWJhc2VJbnN0YW5jZUZyb21TbmFwc2hvdCxcbiAgU2VydmVybGVzc0NsdXN0ZXIsXG4gIFNlcnZlcmxlc3NDbHVzdGVyRnJvbVNuYXBzaG90LFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXJkc1wiO1xuaW1wb3J0IHsgSVNlY3JldCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mtc2VjcmV0c21hbmFnZXJcIjtcbmltcG9ydCB7IElUcmlnZ2VyLCBUcmlnZ2VyIH0gZnJvbSBcImF3cy1jZGstbGliL3RyaWdnZXJzXCI7XG5pbXBvcnQgeyBJQ29uc3RydWN0LCBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHtcbiAgQmFzZURhdGFiYXNlLFxuICBCYXNlRGF0YWJhc2VQcm9wcyxcbiAgQmFzZURhdGFiYXNlT3B0aW9ucyxcbn0gZnJvbSBcIi4vYmFzZS1kYXRhYmFzZVwiO1xuXG5jb25zdCBIQU5ETEVSX1BBVEggPSBwYXRoLmpvaW4oX19kaXJuYW1lLCBcIi4uLy4uL2Fzc2V0cy9yZHMvdHJpZ2dlcnMvcGdzcWxcIik7XG5cbi8qKlxuICogUG9zdGdyZVNRTC1zcGVjaWZpYyBvcHRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBvc3RncmVzcWxEYXRhYmFzZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIFNlY3JldHMgTWFuYWdlciBzZWNyZXQgZm9yIHRoZSBvd25lciBvZiB0aGUgc2NoZW1hLlxuICAgKi9cbiAgcmVhZG9ubHkgb3duZXJTZWNyZXQ6IElTZWNyZXQ7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgc2NoZW1hIHRvIGNyZWF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgdXNlcm5hbWUgb2YgdGhlIG93bmVyU2VjcmV0LlxuICAgKi9cbiAgcmVhZG9ubHkgc2NoZW1hTmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBkZWZhdWx0IGVuY29kaW5nIHNldCB0byB1c2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gXCJVVEY4XCJcbiAgICovXG4gIHJlYWRvbmx5IGVuY29kaW5nPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGRhdGFiYXNlIGRlZmF1bHQgbG9jYWxlIHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSByZWx5IG9uIFBvc3RncmVTUUwgdG8gY2hvb3NlIHRoZSBkZWZhdWx0IGxvY2FsZS5cbiAgICovXG4gIHJlYWRvbmx5IGxvY2FsZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBVUkwgdG8gdGhlIFBFTS1lbmNvZGVkIENlcnRpZmljYXRlIEF1dGhvcml0eSBmaWxlLlxuICAgKlxuICAgKiBOb3JtYWxseSwgd2Ugd291bGQganVzdCBhc3N1bWUgdGhlIExhbWJkYSBydW50aW1lIGhhcyB0aGUgY2VydGlmaWNhdGVzIHRvXG4gICAqIHRydXN0IGFscmVhZHkgaW5zdGFsbGVkLiBTaW5jZSB0aGUgY3VycmVudCBMYW1iZGEgcnVudGltZSBlbnZpcm9ubWVudHMgbGFja1xuICAgKiB0aGUgbmV3ZXIgUkRTIGNlcnRpZmljYXRlIGF1dGhvcml0eSBjZXJ0aWZpY2F0ZXMsIHRoaXMgb3B0aW9uIGNhbiBiZSB1c2VkXG4gICAqIHRvIHNwZWNpZnkgYSBVUkwgdG8gYSByZW1vdGUgZmlsZSBjb250YWluaW5nIHRoZSBDQXMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtbGFtYmRhLWJhc2UtaW1hZ2VzL2lzc3Vlcy8xMjNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBodHRwczovL3RydXN0c3RvcmUucGtpLnJkcy5hbWF6b25hd3MuY29tL1JFR0lPTi9SRUdJT04tYnVuZGxlLnBlbVxuICAgKi9cbiAgcmVhZG9ubHkgY2VydGlmaWNhdGVBdXRob3JpdGllc1VybD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3RvciBwcm9wZXJ0aWVzIGZvciBQb3N0Z3Jlc3FsRGF0YWJhc2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUG9zdGdyZXNxbERhdGFiYXNlUHJvcHNcbiAgZXh0ZW5kcyBCYXNlRGF0YWJhc2VQcm9wcyxcbiAgICBQb3N0Z3Jlc3FsRGF0YWJhc2VPcHRpb25zIHt9XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBzcGVjaWZ5IHdoZW4gdXNpbmcgUG9zdGdyZXNxbERhdGFiYXNlLmZvckNsdXN0ZXIoKS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb3N0Z3Jlc3FsRGF0YWJhc2VGb3JDbHVzdGVyT3B0aW9uc1xuICBleHRlbmRzIFBvc3RncmVzcWxEYXRhYmFzZU9wdGlvbnMsXG4gICAgQmFzZURhdGFiYXNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBBIFNlY3JldHMgTWFuYWdlciBzZWNyZXQgdGhhdCBjb250YWlucyBhZG1pbmlzdHJhdGl2ZSBjcmVkZW50aWFscy5cbiAgICovXG4gIHJlYWRvbmx5IGFkbWluU2VjcmV0PzogSVNlY3JldDtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIHNwZWNpZnkgd2hlbiB1c2luZyBQb3N0Z3Jlc3FsRGF0YWJhc2UuZm9yU2VydmVybGVzc0NsdXN0ZXIoKS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb3N0Z3Jlc3FsRGF0YWJhc2VGb3JTZXJ2ZXJsZXNzQ2x1c3Rlck9wdGlvbnNcbiAgZXh0ZW5kcyBQb3N0Z3Jlc3FsRGF0YWJhc2VGb3JDbHVzdGVyT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoZXJlIHRoZSBMYW1iZGEgZnVuY3Rpb24gd2lsbCBydW4uXG4gICAqL1xuICByZWFkb25seSB2cGM6IElWcGM7XG59XG5cbi8qKlxuICogQSBQb3N0Z3JlU1FMIGRhdGFiYXNlLlxuICovXG5leHBvcnQgY2xhc3MgUG9zdGdyZXNxbERhdGFiYXNlIGV4dGVuZHMgQmFzZURhdGFiYXNlIHtcbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBQb3N0Z3Jlc3FsRGF0YWJhc2UgaW5zaWRlIGEgRGF0YWJhc2VDbHVzdGVyLlxuICAgKlxuICAgKiBUaGlzIG1ldGhvZCBhdXRvbWF0aWNhbGx5IGFkZHMgdGhlIGNsdXN0ZXIgdG8gdGhlIENsb3VkRm9ybWF0aW9uXG4gICAqIGRlcGVuZGVuY2llcyBvZiB0aGUgQ0RLIFRyaWdnZXIuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIFRoZSBDb25zdHJ1Y3QgdGhhdCBjb250YWlucyB0aGlzIG9uZS5cbiAgICogQHBhcmFtIGlkIC0gVGhlIGlkZW50aWZpZXIgb2YgdGhpcyBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBjbHVzdGVyIC0gVGhlIGRhdGFiYXNlIGNsdXN0ZXIgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIFRoZSBjb25maWd1cmF0aW9uIHByb3BlcnRpZXMgZm9yIHRoaXMgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmb3JDbHVzdGVyKFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBjbHVzdGVyOiBEYXRhYmFzZUNsdXN0ZXIsXG4gICAgb3B0aW9uczogUG9zdGdyZXNxbERhdGFiYXNlRm9yQ2x1c3Rlck9wdGlvbnNcbiAgKTogUG9zdGdyZXNxbERhdGFiYXNlIHtcbiAgICBjb25zdCBwcm9wcyA9IHsgLi4ub3B0aW9ucyB9O1xuICAgIGxldCBjbHVzdGVyU2VjcmV0ID0gcHJvcHMuYWRtaW5TZWNyZXQgfHwgY2x1c3Rlci5zZWNyZXQ7XG4gICAgaWYgKGNsdXN0ZXJTZWNyZXQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIllvdSBtdXN0IHByb3ZpZGUgZWl0aGVyIHRoZSBhZG1pblNlY3JldCBwcm9wZXJ0eSBvciBhIGNsdXN0ZXIgd2l0aCBhbiBhdHRhY2hlZCBzZWNyZXRcIlxuICAgICAgKTtcbiAgICB9XG4gICAgZGVsZXRlIHByb3BzLmFkbWluU2VjcmV0O1xuICAgIGNvbnN0IGRhdGFiYXNlID0gbmV3IFBvc3RncmVzcWxEYXRhYmFzZShzY29wZSwgaWQsIHtcbiAgICAgIHRhcmdldDogY2x1c3RlcixcbiAgICAgIGVuZHBvaW50OiBjbHVzdGVyLmNsdXN0ZXJFbmRwb2ludCxcbiAgICAgIGFkbWluU2VjcmV0OiBjbHVzdGVyU2VjcmV0LFxuICAgICAgdnBjOiBjbHVzdGVyLnZwYyxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICAgIGRhdGFiYXNlLnRyaWdnZXIuZXhlY3V0ZUFmdGVyKGNsdXN0ZXIpO1xuICAgIHJldHVybiBkYXRhYmFzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgUG9zdGdyZXNxbERhdGFiYXNlIGluc2lkZSBhIERhdGFiYXNlQ2x1c3RlckZyb21TbmFwc2hvdC5cbiAgICpcbiAgICogVGhpcyBtZXRob2QgYXV0b21hdGljYWxseSBhZGRzIHRoZSBjbHVzdGVyIHRvIHRoZSBDbG91ZEZvcm1hdGlvblxuICAgKiBkZXBlbmRlbmNpZXMgb2YgdGhlIENESyBUcmlnZ2VyLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBUaGUgQ29uc3RydWN0IHRoYXQgY29udGFpbnMgdGhpcyBvbmUuXG4gICAqIEBwYXJhbSBpZCAtIFRoZSBpZGVudGlmaWVyIG9mIHRoaXMgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gY2x1c3RlciAtIFRoZSBkYXRhYmFzZSBjbHVzdGVyIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIG9wdGlvbnMgLSBUaGUgY29uZmlndXJhdGlvbiBwcm9wZXJ0aWVzIGZvciB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZm9yQ2x1c3RlckZyb21TbmFwc2hvdChcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgY2x1c3RlcjogRGF0YWJhc2VDbHVzdGVyRnJvbVNuYXBzaG90LFxuICAgIG9wdGlvbnM6IFBvc3RncmVzcWxEYXRhYmFzZUZvckNsdXN0ZXJPcHRpb25zXG4gICk6IFBvc3RncmVzcWxEYXRhYmFzZSB7XG4gICAgLy8gVGhlIERhdGFiYXNlQ2x1c3RlckZyb21TbmFwc2hvdCB0eXBlIGlzIGVxdWl2YWxlbnQgdG8gRGF0YWJhc2VDbHVzdGVyLlxuICAgIHJldHVybiBQb3N0Z3Jlc3FsRGF0YWJhc2UuZm9yQ2x1c3RlcihzY29wZSwgaWQsIGNsdXN0ZXIsIG9wdGlvbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBQb3N0Z3Jlc3FsRGF0YWJhc2UgaW5zaWRlIGEgRGF0YWJhc2VDbHVzdGVyLlxuICAgKlxuICAgKiBUaGlzIG1ldGhvZCBhdXRvbWF0aWNhbGx5IGFkZHMgdGhlIGNsdXN0ZXIgdG8gdGhlIENsb3VkRm9ybWF0aW9uXG4gICAqIGRlcGVuZGVuY2llcyBvZiB0aGUgQ0RLIFRyaWdnZXIuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIFRoZSBDb25zdHJ1Y3QgdGhhdCBjb250YWlucyB0aGlzIG9uZS5cbiAgICogQHBhcmFtIGlkIC0gVGhlIGlkZW50aWZpZXIgb2YgdGhpcyBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBjbHVzdGVyIC0gVGhlIGRhdGFiYXNlIGNsdXN0ZXIgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIFRoZSBjb25maWd1cmF0aW9uIHByb3BlcnRpZXMgZm9yIHRoaXMgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmb3JTZXJ2ZXJsZXNzQ2x1c3RlcihcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgY2x1c3RlcjogU2VydmVybGVzc0NsdXN0ZXIsXG4gICAgb3B0aW9uczogUG9zdGdyZXNxbERhdGFiYXNlRm9yU2VydmVybGVzc0NsdXN0ZXJPcHRpb25zXG4gICk6IFBvc3RncmVzcWxEYXRhYmFzZSB7XG4gICAgLy8gVGhlIFNlcnZlcmxlc3NDbHVzdGVyRnJvbVNuYXBzaG90IHR5cGUgaXMgYSBzdWJzZXQgb2YgU2VydmVybGVzc0NsdXN0ZXIuXG4gICAgcmV0dXJuIFBvc3RncmVzcWxEYXRhYmFzZS5mb3JTZXJ2ZXJsZXNzQ2x1c3RlckZyb21TbmFwc2hvdChcbiAgICAgIHNjb3BlLFxuICAgICAgaWQsXG4gICAgICBjbHVzdGVyLFxuICAgICAgb3B0aW9uc1xuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgbmV3IFBvc3RncmVzcWxEYXRhYmFzZSBpbnNpZGUgYSBEYXRhYmFzZUNsdXN0ZXJGcm9tU25hcHNob3QuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIGF1dG9tYXRpY2FsbHkgYWRkcyB0aGUgY2x1c3RlciB0byB0aGUgQ2xvdWRGb3JtYXRpb25cbiAgICogZGVwZW5kZW5jaWVzIG9mIHRoZSBDREsgVHJpZ2dlci5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIC0gVGhlIENvbnN0cnVjdCB0aGF0IGNvbnRhaW5zIHRoaXMgb25lLlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgaWRlbnRpZmllciBvZiB0aGlzIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIGNsdXN0ZXIgLSBUaGUgZGF0YWJhc2UgY2x1c3RlciBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBvcHRpb25zIC0gVGhlIGNvbmZpZ3VyYXRpb24gcHJvcGVydGllcyBmb3IgdGhpcyBjb25zdHJ1Y3QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZvclNlcnZlcmxlc3NDbHVzdGVyRnJvbVNuYXBzaG90KFxuICAgIHNjb3BlOiBDb25zdHJ1Y3QsXG4gICAgaWQ6IHN0cmluZyxcbiAgICBjbHVzdGVyOiBTZXJ2ZXJsZXNzQ2x1c3RlckZyb21TbmFwc2hvdCxcbiAgICBvcHRpb25zOiBQb3N0Z3Jlc3FsRGF0YWJhc2VGb3JTZXJ2ZXJsZXNzQ2x1c3Rlck9wdGlvbnNcbiAgKTogUG9zdGdyZXNxbERhdGFiYXNlIHtcbiAgICBjb25zdCBwcm9wcyA9IHsgLi4ub3B0aW9ucyB9O1xuICAgIGxldCBjbHVzdGVyU2VjcmV0ID0gcHJvcHMuYWRtaW5TZWNyZXQgfHwgY2x1c3Rlci5zZWNyZXQ7XG4gICAgaWYgKGNsdXN0ZXJTZWNyZXQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIllvdSBtdXN0IHByb3ZpZGUgZWl0aGVyIHRoZSBhZG1pblNlY3JldCBwcm9wZXJ0eSBvciBhIGNsdXN0ZXIgd2l0aCBhbiBhdHRhY2hlZCBzZWNyZXRcIlxuICAgICAgKTtcbiAgICB9XG4gICAgZGVsZXRlIHByb3BzLmFkbWluU2VjcmV0O1xuICAgIGNvbnN0IGRhdGFiYXNlID0gbmV3IFBvc3RncmVzcWxEYXRhYmFzZShzY29wZSwgaWQsIHtcbiAgICAgIHRhcmdldDogY2x1c3RlcixcbiAgICAgIGVuZHBvaW50OiBjbHVzdGVyLmNsdXN0ZXJFbmRwb2ludCxcbiAgICAgIGFkbWluU2VjcmV0OiBjbHVzdGVyU2VjcmV0LFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gICAgZGF0YWJhc2UudHJpZ2dlci5leGVjdXRlQWZ0ZXIoY2x1c3Rlcik7XG4gICAgcmV0dXJuIGRhdGFiYXNlO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIG5ldyBQb3N0Z3Jlc3FsRGF0YWJhc2UgaW5zaWRlIGEgRGF0YWJhc2VJbnN0YW5jZS5cbiAgICpcbiAgICogVGhpcyBtZXRob2QgYXV0b21hdGljYWxseSBhZGRzIHRoZSBpbnN0YW5jZSB0byB0aGUgQ2xvdWRGb3JtYXRpb25cbiAgICogZGVwZW5kZW5jaWVzIG9mIHRoZSBDREsgVHJpZ2dlci5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIC0gVGhlIENvbnN0cnVjdCB0aGF0IGNvbnRhaW5zIHRoaXMgb25lLlxuICAgKiBAcGFyYW0gaWQgLSBUaGUgaWRlbnRpZmllciBvZiB0aGlzIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIGluc3RhbmNlIC0gVGhlIGRhdGFiYXNlIGNsdXN0ZXIgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIFRoZSBjb25maWd1cmF0aW9uIHByb3BlcnRpZXMgZm9yIHRoaXMgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmb3JJbnN0YW5jZShcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgaW5zdGFuY2U6IERhdGFiYXNlSW5zdGFuY2UsXG4gICAgb3B0aW9uczogUG9zdGdyZXNxbERhdGFiYXNlRm9yQ2x1c3Rlck9wdGlvbnNcbiAgKTogUG9zdGdyZXNxbERhdGFiYXNlIHtcbiAgICBjb25zdCBwcm9wcyA9IHsgLi4ub3B0aW9ucyB9O1xuICAgIGxldCBjbHVzdGVyU2VjcmV0ID0gcHJvcHMuYWRtaW5TZWNyZXQgfHwgaW5zdGFuY2Uuc2VjcmV0O1xuICAgIGlmIChjbHVzdGVyU2VjcmV0ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJZb3UgbXVzdCBwcm92aWRlIGVpdGhlciB0aGUgYWRtaW5TZWNyZXQgcHJvcGVydHkgb3IgYSBjbHVzdGVyIHdpdGggYW4gYXR0YWNoZWQgc2VjcmV0XCJcbiAgICAgICk7XG4gICAgfVxuICAgIGRlbGV0ZSBwcm9wcy5hZG1pblNlY3JldDtcbiAgICBjb25zdCBkYXRhYmFzZSA9IG5ldyBQb3N0Z3Jlc3FsRGF0YWJhc2Uoc2NvcGUsIGlkLCB7XG4gICAgICB0YXJnZXQ6IGluc3RhbmNlLFxuICAgICAgZW5kcG9pbnQ6IGluc3RhbmNlLmluc3RhbmNlRW5kcG9pbnQsXG4gICAgICBhZG1pblNlY3JldDogY2x1c3RlclNlY3JldCxcbiAgICAgIHZwYzogaW5zdGFuY2UudnBjLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gICAgZGF0YWJhc2UudHJpZ2dlci5leGVjdXRlQWZ0ZXIoaW5zdGFuY2UpO1xuICAgIHJldHVybiBkYXRhYmFzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgUG9zdGdyZXNxbERhdGFiYXNlIGluc2lkZSBhIERhdGFiYXNlSW5zdGFuY2VGcm9tU25hcHNob3QuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIGF1dG9tYXRpY2FsbHkgYWRkcyB0aGUgaW5zdGFuY2UgdG8gdGhlIENsb3VkRm9ybWF0aW9uXG4gICAqIGRlcGVuZGVuY2llcyBvZiB0aGUgQ0RLIFRyaWdnZXIuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIFRoZSBDb25zdHJ1Y3QgdGhhdCBjb250YWlucyB0aGlzIG9uZS5cbiAgICogQHBhcmFtIGlkIC0gVGhlIGlkZW50aWZpZXIgb2YgdGhpcyBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBpbnN0YW5jZSAtIFRoZSBkYXRhYmFzZSBjbHVzdGVyIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIG9wdGlvbnMgLSBUaGUgY29uZmlndXJhdGlvbiBwcm9wZXJ0aWVzIGZvciB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZm9ySW5zdGFuY2VGcm9tU25hcHNob3QoXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIGluc3RhbmNlOiBEYXRhYmFzZUluc3RhbmNlRnJvbVNuYXBzaG90LFxuICAgIG9wdGlvbnM6IFBvc3RncmVzcWxEYXRhYmFzZUZvckNsdXN0ZXJPcHRpb25zXG4gICk6IFBvc3RncmVzcWxEYXRhYmFzZSB7XG4gICAgLy8gVGhlIERhdGFiYXNlSW5zdGFuY2VGcm9tU25hcHNob3QgdHlwZSBpcyBlcXVpdmFsZW50IHRvIERhdGFiYXNlSW5zdGFuY2UuXG4gICAgcmV0dXJuIFBvc3RncmVzcWxEYXRhYmFzZS5mb3JJbnN0YW5jZShzY29wZSwgaWQsIGluc3RhbmNlLCBvcHRpb25zKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSB0cmlnZ2VyOiBJVHJpZ2dlcjtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IEZ1bmN0aW9uO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgb3duZXJTZWNyZXRzOiBJU2VjcmV0W10gPSBbXTtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHJlYWRlclNlY3JldHM6IElTZWNyZXRbXSA9IFtdO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdW5wcml2aWxlZ2VkU2VjcmV0czogSVNlY3JldFtdID0gW107XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgUG9zdGdyZXNxbERhdGFiYXNlLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBUaGUgQ29uc3RydWN0IHRoYXQgY29udGFpbnMgdGhpcyBvbmUuXG4gICAqIEBwYXJhbSBpZCAtIFRoZSBpZGVudGlmaWVyIG9mIHRoaXMgY29uc3RydWN0LlxuICAgKiBAcGFyYW0gcHJvcHMgLSBUaGUgY29uZmlndXJhdGlvbiBwcm9wZXJ0aWVzIGZvciB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihcbiAgICBzY29wZTogSUNvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIHByb3BzOiBQb3N0Z3Jlc3FsRGF0YWJhc2VQcm9wc1xuICApIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIGNvbnN0IHtcbiAgICAgIGFkbWluU2VjcmV0LFxuICAgICAgdnBjLFxuICAgICAgb3duZXJTZWNyZXQsXG4gICAgICBzY2hlbWFOYW1lLFxuICAgICAgZW5jb2RpbmcgPSBcIlVURjhcIixcbiAgICB9ID0gcHJvcHM7XG5cbiAgICBjb25zdCBlbnZpcm9ubWVudDogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHtcbiAgICAgIEFXU19OT0RFSlNfQ09OTkVDVElPTl9SRVVTRV9FTkFCTEVEOiBcIjFcIixcbiAgICAgIEFETUlOX1NFQ1JFVF9BUk46IGFkbWluU2VjcmV0LnNlY3JldEFybixcbiAgICAgIE9XTkVSX1NFQ1JFVF9BUk46IG93bmVyU2VjcmV0LnNlY3JldEFybixcbiAgICAgIE9XTkVSX1NFQ1JFVFM6IExhenkuc3RyaW5nKHtcbiAgICAgICAgcHJvZHVjZTogKCkgPT5cbiAgICAgICAgICBKU09OLnN0cmluZ2lmeSh0aGlzLm93bmVyU2VjcmV0cy5tYXAoKHMpID0+IHMuc2VjcmV0QXJuKSksXG4gICAgICB9KSxcbiAgICAgIFJFQURFUl9TRUNSRVRTOiBMYXp5LnN0cmluZyh7XG4gICAgICAgIHByb2R1Y2U6ICgpID0+XG4gICAgICAgICAgSlNPTi5zdHJpbmdpZnkodGhpcy5yZWFkZXJTZWNyZXRzLm1hcCgocykgPT4gcy5zZWNyZXRBcm4pKSxcbiAgICAgIH0pLFxuICAgICAgVU5QUklWSUxFR0VEX1NFQ1JFVFM6IExhenkuc3RyaW5nKHtcbiAgICAgICAgcHJvZHVjZTogKCkgPT5cbiAgICAgICAgICBKU09OLnN0cmluZ2lmeSh0aGlzLnVucHJpdmlsZWdlZFNlY3JldHMubWFwKChzKSA9PiBzLnNlY3JldEFybikpLFxuICAgICAgfSksXG4gICAgICBEQl9OQU1FOiB0aGlzLmRhdGFiYXNlTmFtZSxcbiAgICAgIFNDSEVNQV9OQU1FOiBzY2hlbWFOYW1lIHx8IFwiXCIsXG4gICAgICBEQl9FTkNPRElORzogZW5jb2RpbmcsXG4gICAgfTtcbiAgICBpZiAocHJvcHMubG9jYWxlKSB7XG4gICAgICBlbnZpcm9ubWVudC5EQl9MT0NBTEUgPSBwcm9wcy5sb2NhbGU7XG4gICAgfVxuICAgIGlmIChwcm9wcy5jZXJ0aWZpY2F0ZUF1dGhvcml0aWVzVXJsKSB7XG4gICAgICBlbnZpcm9ubWVudC5DQV9DRVJUU19VUkwgPSBwcm9wcy5jZXJ0aWZpY2F0ZUF1dGhvcml0aWVzVXJsO1xuICAgIH1cblxuICAgIHRoaXMubGFtYmRhRnVuY3Rpb24gPSBuZXcgRnVuY3Rpb24odGhpcywgXCJGdW5jdGlvblwiLCB7XG4gICAgICBydW50aW1lOiBSdW50aW1lLk5PREVKU18yMF9YLFxuICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQoSEFORExFUl9QQVRIKSxcbiAgICAgIGhhbmRsZXI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgZGVzY3JpcHRpb246IFwiQ3JlYXRlcyBhIHNjaGVtYSBhbmQgcG9zc2libHkgc29tZSBzZWNvbmRhcnkgdXNlcnNcIixcbiAgICAgIHZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHRoaXMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgc2VjdXJpdHlHcm91cHM6IFt0aGlzLnNlY3VyaXR5R3JvdXBdLFxuICAgICAgZW52aXJvbm1lbnQsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDIpLFxuICAgIH0pO1xuICAgIGFkbWluU2VjcmV0LmdyYW50UmVhZCh0aGlzLmxhbWJkYUZ1bmN0aW9uKTtcbiAgICBvd25lclNlY3JldC5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jdGlvbik7XG5cbiAgICB0aGlzLnRyaWdnZXIgPSBuZXcgVHJpZ2dlcih0aGlzLCBcIlRyaWdnZXJcIiwge1xuICAgICAgaGFuZGxlcjogdGhpcy5sYW1iZGFGdW5jdGlvbixcbiAgICB9KTtcbiAgICB0aGlzLnRyaWdnZXIuZXhlY3V0ZUFmdGVyKGFkbWluU2VjcmV0LCBvd25lclNlY3JldCwgdGhpcy5sYW1iZGFGdW5jdGlvbik7XG4gIH1cblxuICBwdWJsaWMgYWRkVXNlckFzT3duZXIoc2VjcmV0OiBJU2VjcmV0KSB7XG4gICAgdGhpcy5vd25lclNlY3JldHMucHVzaChzZWNyZXQpO1xuICAgIHNlY3JldC5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jdGlvbik7XG4gICAgdGhpcy50cmlnZ2VyLmV4ZWN1dGVBZnRlcihzZWNyZXQpO1xuICB9XG5cbiAgcHVibGljIGFkZFVzZXJBc1JlYWRlcihzZWNyZXQ6IElTZWNyZXQpIHtcbiAgICB0aGlzLnJlYWRlclNlY3JldHMucHVzaChzZWNyZXQpO1xuICAgIHNlY3JldC5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jdGlvbik7XG4gICAgdGhpcy50cmlnZ2VyLmV4ZWN1dGVBZnRlcihzZWNyZXQpO1xuICB9XG5cbiAgcHVibGljIGFkZFVzZXJBc1VucHJpdmlsZWdlZChzZWNyZXQ6IElTZWNyZXQpIHtcbiAgICB0aGlzLnVucHJpdmlsZWdlZFNlY3JldHMucHVzaChzZWNyZXQpO1xuICAgIHNlY3JldC5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jdGlvbik7XG4gICAgdGhpcy50cmlnZ2VyLmV4ZWN1dGVBZnRlcihzZWNyZXQpO1xuICB9XG59XG4iXX0=