"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CidrContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const assign_on_launch_1 = require("./assign-on-launch");
const util_1 = require("../util");
/**
 * Allocates IPv6 CIDRs and routes for subnets in a VPC.
 *
 * @see {@link https://github.com/aws/aws-cdk/issues/894}
 * @see {@link https://github.com/aws/aws-cdk/issues/5927}
 */
class CidrContext extends constructs_1.Construct {
    /**
     * Creates a new BetterVpc.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param options - The constructor options.
     */
    constructor(scope, id, options) {
        super(scope, id);
        this.assignAddressOnLaunch = options.assignAddressOnLaunch !== false;
        const vpc = util_1.assertType(options.vpc, aws_ec2_1.Vpc);
        this.vpc = vpc;
        // Create the VPC CIDR block.
        const cidrBlock = new aws_ec2_1.CfnVPCCidrBlock(this, "Ipv6CidrBlock", {
            vpcId: vpc.vpcId,
            amazonProvidedIpv6CidrBlock: options.addressPool === undefined,
            ipv6Pool: options.addressPool,
            ipv6CidrBlock: options.cidrBlock,
        });
        // Create a placeholder for CloudFormation to provide a CIDR array.
        const cidrs = aws_cdk_lib_1.Fn.cidr(aws_cdk_lib_1.Fn.select(0, vpc.vpcIpv6CidrBlocks), this.validateCidrCount(vpc, options.cidrCount), "64");
        if (vpc.publicSubnets.length > 0) {
            this.assignPublicSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.privateSubnets.length > 0) {
            this.assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.isolatedSubnets.length > 0) {
            this.assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (this.assignAddressOnLaunch) {
            const vpcSubnets = {
                subnets: [
                    ...vpc.publicSubnets,
                    ...vpc.privateSubnets,
                    ...vpc.isolatedSubnets,
                ],
            };
            new assign_on_launch_1.AssignOnLaunch(this, "AssignOnLaunch", { vpc, vpcSubnets });
        }
    }
    /**
     * Figure out the minimun required CIDR subnets and the number desired.
     *
     * @param vpc - The VPC.
     * @param cidrCount - Optional. Divide the VPC CIDR into this many subsets.
     */
    validateCidrCount(vpc, cidrCount) {
        const leastCidrBlocksNeeded = vpc.publicSubnets.length +
            vpc.privateSubnets.length +
            vpc.isolatedSubnets.length;
        const cidrBlockCount = cidrCount ?? leastCidrBlocksNeeded;
        if (cidrBlockCount < leastCidrBlocksNeeded) {
            throw new Error(`You must create at least ${leastCidrBlocksNeeded} CIDR blocks in this VPC`);
        }
        return cidrBlockCount;
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPublicSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = util_1.findInstanceOf(vpc.node.children, aws_ec2_1.CfnInternetGateway);
        for (const [index, subnet] of vpc.publicSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Internet gateway for Internet-bound IPv6 traffic.
            const concreteSubnet = util_1.assertType(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: gateway.ref,
                routerType: aws_ec2_1.RouterType.GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for private subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = new aws_ec2_1.CfnEgressOnlyInternetGateway(this, "EgressOnlyIgw", {
            vpcId: vpc.vpcId,
        });
        for (const [index, subnet] of vpc.privateSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Egress-only Internet Gateway.
            const concreteSubnet = util_1.assertType(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: `${gateway.ref}`,
                routerType: aws_ec2_1.RouterType.EGRESS_ONLY_INTERNET_GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock) {
        for (const [index, subnet] of vpc.isolatedSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = util_1.assertType(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length + vpc.privateSubnets.length, cidrs);
        }
    }
}
exports.CidrContext = CidrContext;
_a = JSII_RTTI_SYMBOL_1;
CidrContext[_a] = { fqn: "shady-island.CidrContext", version: "0.1.17" };
//# sourceMappingURL=data:application/json;base64,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