# coding: utf-8

from deployv_addon_gitlab_tools.common import check_env_vars
from deployv.helpers import utils
from docker import Client
from os import environ
import subprocess
import shlex
import sys
import click
import logging


_logger = logging.getLogger('deployv.' + __name__)
_cli = Client(timeout=600)


def build_image():
    format_values = {
        'url': environ['CI_BUILD_REPO'],
        'version': environ['CI_BUILD_REF_NAME'],
        'base': environ['BASE_IMAGE'],
        'odoo_repo': environ['ODOO_REPO'],
        'odoo_branch': environ['ODOO_BRANCH'],
        'name': environ['_IMAGE_NAME'],
    }
    _logger.info('Bulding image %s', environ['_IMAGE_NAME'])
    cmd = (
        'deployvcmd build -u {url} -v {version} -i {base} -O {odoo_repo}#{odoo_branch} -T {name}'
        .format(**format_values)
    )
    subprocess.check_call(shlex.split(cmd))


def push_image():
    for tag in ['latest', environ['_IMAGE_TAG']]:
        _logger.info('Pushing image %s to %s:%s', environ['_IMAGE_NAME'], environ['_IMAGE_REPO'], tag)
        _cli.tag(environ['_IMAGE_NAME'], environ['_IMAGE_REPO'], tag=tag)
        _cli.push(environ['_IMAGE_REPO'], tag=tag)


@click.command()
@click.option('--ci_project_name', default=environ.get('CI_PROJECT_NAME'),
              help=("The project name that is currently being built."
                    " Env var: CI_PROJECT_NAME."))
@click.option('--ci_build_ref', default=environ.get('CI_BUILD_REF'),
              help=("The commit revision for which project is built."
                    " Env var: CI_BUILD_REF."))
@click.option('--ci_build_ref_name', default=environ.get('CI_BUILD_REF_NAME'),
              help=("The branch or tag name for which project is built."
                    " Env var: CI_BUILD_REF_NAME."))
@click.option('--ci_build_repo', default=environ.get('CI_BUILD_REPO'),
              help=("The URL to clone the Git repository."
                    " Env var: CI_BUILD_REPO."))
@click.option('--base_image', default=environ.get('BASE_IMAGE'),
              help=("Env var: BASE_IMAGE."))
@click.option('--odoo_repo', default=environ.get('ODOO_REPO'),
              help=("Env var: ODOO_REPO."))
@click.option('--odoo_branch', default=environ.get('ODOO_BRANCH'),
              help=("Env var: ODOO_BRANCH."))
def upload_image(**kwargs):
    check_env_vars(**kwargs)
    image_name = '{customer}{ver}'.format(
        customer=environ.get('CUSTOMER', environ.get('CI_PROJECT_NAME')),
        ver=environ.get('CI_BUILD_REF_NAME').replace('.', '')
    )
    image_repo = 'quay.io/vauxoo/{image}'.format(image=image_name)
    image_tag = environ['CI_BUILD_REF'][:7].lower()
    environ.update({
        '_IMAGE_NAME': image_name,
        '_IMAGE_REPO': image_repo,
        '_IMAGE_TAG': image_tag,
    })
    build_image()
    push_image()
    sys.exit(0)

