"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
const ACTIONS = [
    'kms:Create*',
    'kms:Describe*',
    'kms:Enable*',
    'kms:List*',
    'kms:Put*',
    'kms:Update*',
    'kms:Revoke*',
    'kms:Disable*',
    'kms:Get*',
    'kms:Delete*',
    'kms:ScheduleKeyDeletion',
    'kms:CancelKeyDeletion',
    'kms:GenerateDataKey',
    'kms:TagResource',
    'kms:UntagResource',
];
module.exports = {
    'default key'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { removalPolicy: core_1.RemovalPolicy.DESTROY });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new iam.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                                {
                                    Action: 'kms:encrypt',
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: 'arn',
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new iam.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.applyAspect(new core_1.Tag('tag1', 'value1'));
        key.node.applyAspect(new core_1.Tag('tag2', 'value2'));
        key.node.applyAspect(new core_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                                {
                                    Action: 'kms:encrypt',
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: 'arn',
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: 'tag1',
                                Value: 'value1',
                            },
                            {
                                Key: 'tag2',
                                Value: 'value2',
                            },
                            {
                                Key: 'tag3',
                                Value: '',
                            },
                        ],
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        assert_1.expect(stack).to(assert_1.countResources('AWS::KMS::Alias', 1));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/xoo',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'can run multiple addAlias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const alias1 = key.addAlias('alias/alias1');
        const alias2 = key.addAlias('alias/alias2');
        test.ok(alias1.aliasName);
        test.ok(alias2.aliasName);
        assert_1.expect(stack).to(assert_1.countResources('AWS::KMS::Alias', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/alias1',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/alias2',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new iam.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        Action: ACTIONS,
                        Effect: 'Allow',
                        Principal: { AWS: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] } },
                        Resource: '*',
                    },
                    // This is the interesting one
                    {
                        Action: 'kms:Decrypt',
                        Effect: 'Allow',
                        Principal: { AWS: { 'Fn::GetAtt': ['User00B015A1', 'Arn'] } },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'kms:Decrypt',
                        Effect: 'Allow',
                        Resource: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grant for a principal in a dependent stack works correctly'(test) {
        const app = new core_1.App();
        const principalStack = new core_1.Stack(app, 'PrincipalStack');
        const principal = new iam.Role(principalStack, 'Role', {
            assumedBy: new iam.AnyPrincipal(),
        });
        const keyStack = new core_1.Stack(app, 'KeyStack');
        const key = new lib_1.Key(keyStack, 'Key');
        principalStack.addDependency(keyStack);
        key.grantEncrypt(principal);
        assert_1.expect(keyStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            'KeyPolicy': {
                'Statement': [
                    {
                    // owning account management permissions - we don't care about them in this test
                    },
                    {
                        'Action': [
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        'Effect': 'Allow',
                        'Principal': {
                            'AWS': {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { 'Ref': 'AWS::Partition' },
                                        ':iam::',
                                        { 'Ref': 'AWS::AccountId' },
                                        ':root',
                                    ]],
                            },
                        },
                        'Resource': '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'keyId resolves to a Ref'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey');
        new core_1.CfnOutput(stack, 'Out', {
            value: key.keyId,
        });
        const template = assert_1.SynthUtils.synthesize(stack).template.Outputs;
        test.deepEqual(template, {
            'Out': {
                'Value': {
                    'Ref': 'MyKey6AB29FA6',
                },
            },
        });
        test.done();
    },
    'enablePolicyControl changes key policy to allow IAM control'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey', { trustAccountIdentities: true });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            'KeyPolicy': {
                'Statement': [
                    {
                        'Action': 'kms:*',
                        'Effect': 'Allow',
                        'Principal': {
                            'AWS': {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { 'Ref': 'AWS::Partition' },
                                        ':iam::',
                                        { 'Ref': 'AWS::AccountId' },
                                        ':root',
                                    ]],
                            },
                        },
                        'Resource': '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'imported keys': {
        'throw an error when providing something that is not a valid key ARN'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key');
            }, /KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key\/<keyId>', got: 'arn:aws:kms:us-east-1:123456789012:key'/);
            test.done();
        },
        'can have aliases added to them'(test) {
            const stack2 = new core_1.Stack();
            const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
            // addAlias can be called on imported keys.
            myKeyImported.addAlias('alias/hello');
            test.equal(myKeyImported.keyId, '12345678-1234-1234-1234-123456789012');
            assert_1.expect(stack2).toMatch({
                Resources: {
                    MyKeyImportedAliasB1C5269F: {
                        Type: 'AWS::KMS::Alias',
                        Properties: {
                            AliasName: 'alias/hello',
                            TargetKeyId: 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012',
                        },
                    },
                },
            });
            test.done();
        },
        'addToResourcePolicy allowNoOp and there is no policy': {
            'succeed if set to true (default)'(test) {
                const stack = new core_1.Stack();
                const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
                key.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }));
                test.done();
            },
            'fails if set to false'(test) {
                const stack = new core_1.Stack();
                const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
                test.throws(() => {
                    key.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }), /* allowNoOp */ false);
                }, 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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