"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.ObjectOwnership = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const path = require("path");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
const AUTO_DELETE_OBJECTS_RESOURCE_TYPE = 'Custom::S3AutoDeleteObjects';
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id, options = {}) {
        var _d, _e;
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: (_e = (_d = options.paths) === null || _d === void 0 ? void 0 : _d.map(p => this.arnForObjects(p))) !== null && _e !== void 0 ? _e : [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailPutObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailWriteObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @stability stable
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policy) === null || _d === void 0 ? void 0 : _d.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * The https URL of an S3 object.
     *
     * Specify `regional: false` at the options
     * for non-regional URLs. For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${this.env.region}.${stack.urlSuffix}/`;
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key, options) {
        var _d;
        const domainName = ((_d = options === null || options === void 0 ? void 0 : options.regional) !== null && _d !== void 0 ? _d : true) ? this.bucketRegionalDomainName : this.bucketDomainName;
        const prefix = `https://${domainName}`;
        if (typeof key !== 'string') {
            return prefix;
        }
        return this.urlJoin(prefix, key);
    }
    /**
     * The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key) {
        const prefix = 's3://';
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * If you need to specify a keyPattern with multiple components, concatenate them into a single string, e.g.:
     *
     *    arnForObjects(`home/${team}/${user}/*`)
     *
     * @stability stable
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @stability stable
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, this.writeActions, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, this.putActions, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant the given IAM identity permissions to modify the ACLs of objects in the given Bucket.
     *
     * If your application has the '@aws-cdk/aws-s3:grantWriteWithoutAcl' feature flag set,
     * calling {@link grantWrite} or {@link grantReadWrite} no longer grants permissions to modify the ACLs of the objects;
     * in this case, if you need to modify object ACLs, call this method explicitly.
     *
     * @stability stable
     */
    grantPutAcl(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACL_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM principal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @stability stable
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(this.writeActions);
        // we need unique permissions because some permissions are common between read and write key actions
        const keyActions = [...new Set([...perms.KEY_READ_ACTIONS, ...perms.KEY_WRITE_ACTIONS])];
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @stability stable
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.AnyPrincipal(),
            resource: this,
        });
    }
    get writeActions() {
        return [
            ...perms.BUCKET_DELETE_ACTIONS,
            ...this.putActions,
        ];
    }
    get putActions() {
        return core_1.FeatureFlags.of(this).isEnabled(cxapi.S3_GRANT_WRITE_WITHOUT_ACL)
            ? perms.BUCKET_PUT_ACTIONS
            : perms.LEGACY_BUCKET_PUT_ACTIONS;
    }
    urlJoin(...components) {
        return components.reduce((result, component) => {
            if (result.endsWith('/')) {
                result = result.slice(0, -1);
            }
            if (component.startsWith('/')) {
                component = component.slice(1);
            }
            return `${result}/${component}`;
        });
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
/**
 * @stability stable
 */
class BlockPublicAccess {
    /**
     * @stability stable
     */
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
exports.BlockPublicAccess = BlockPublicAccess;
_a = JSII_RTTI_SYMBOL_1;
BlockPublicAccess[_a] = { fqn: "@aws-cdk/aws-s3.BlockPublicAccess", version: "1.106.0" };
/**
 * @stability stable
 */
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
/**
 * @stability stable
 */
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
/**
 * All http request methods.
 *
 * @stability stable
 */
var HttpMethods;
(function (HttpMethods) {
    HttpMethods["GET"] = "GET";
    HttpMethods["PUT"] = "PUT";
    HttpMethods["HEAD"] = "HEAD";
    HttpMethods["POST"] = "POST";
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods.
 *
 * @stability stable
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 *
 * @stability stable
 */
var InventoryFormat;
(function (InventoryFormat) {
    InventoryFormat["CSV"] = "CSV";
    InventoryFormat["PARQUET"] = "Parquet";
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 *
 * @stability stable
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    InventoryFrequency["DAILY"] = "Daily";
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 *
 * @stability stable
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    InventoryObjectVersion["ALL"] = "All";
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
 * The ObjectOwnership of the bucket.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html
 * @stability stable
 */
var ObjectOwnership;
(function (ObjectOwnership) {
    ObjectOwnership["BUCKET_OWNER_PREFERRED"] = "BucketOwnerPreferred";
    ObjectOwnership["OBJECT_WRITER"] = "ObjectWriter";
})(ObjectOwnership = exports.ObjectOwnership || (exports.ObjectOwnership = {}));
/**
 * An S3 bucket with associated policy objects.
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 *
 * @stability stable
 */
class Bucket extends BucketBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _d;
        super(scope, id, {
            physicalName: props.bucketName,
        });
        /**
         * Indicates if a bucket resource policy should automatically created upon the first call to `addToResourcePolicy`.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        this.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.any({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.any({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.any({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.string({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.any({ produce: () => this.parseInventoryConfiguration() }),
            ownershipControls: this.parseOwnershipControls(props),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        // Enforce AWS Foundational Security Best Practice
        if (props.enforceSSL) {
            this.enforceSSLStatement();
        }
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery();
        }
        for (const inventory of (_d = props.inventories) !== null && _d !== void 0 ? _d : []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
        if (props.autoDeleteObjects) {
            if (props.removalPolicy !== core_1.RemovalPolicy.DESTROY) {
                throw new Error('Cannot use \'autoDeleteObjects\' property on a bucket without setting removal policy to \'DESTROY\'.');
            }
            this.enableAutoDeleteObjects();
        }
    }
    /**
     * @stability stable
     */
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    /**
     * @stability stable
     */
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object.
     * @stability stable
     */
    static fromBucketAttributes(scope, id, attrs) {
        var _d;
        const stack = core_1.Stack.of(scope);
        const region = (_d = attrs.region) !== null && _d !== void 0 ? _d : stack.region;
        const urlSuffix = stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteDomain = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                var _d;
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = (_d = attrs.isWebsite) !== null && _d !== void 0 ? _d : false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id, {
            account: attrs.account,
            region: attrs.region,
        });
    }
    /**
     * Add a lifecycle rule to the bucket.
     *
     * @param rule The rule to add.
     * @stability stable
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpiration !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add.
     * @stability stable
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket.
     *
     * @param rule The CORS configuration rule to add.
     * @stability stable
     */
    addCorsRule(rule) {
        this.cors.push(rule);
    }
    /**
     * Adds a bucket notification event destination.
     *
     * @param event The event to trigger the notification.
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue).
     * @param filters S3 object key filter rules to determine which objects trigger this event.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     * @stability stable
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notifications when an object is created in the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notifications when an object is removed from the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add.
     * @stability stable
     */
    addInventory(inventory) {
        this.inventories.push(inventory);
    }
    /**
     * Adds an iam statement to enforce SSL requests only.
     */
    enforceSSLStatement() {
        const statement = new iam.PolicyStatement({
            actions: ['s3:*'],
            conditions: {
                Bool: { 'aws:SecureTransport': 'false' },
            },
            effect: iam.Effect.DENY,
            resources: [
                this.bucketArn,
                this.arnForObjects('*'),
            ],
            principals: [new iam.AnyPrincipal()],
        });
        this.addToResourcePolicy(statement);
    }
    validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        // if bucketKeyEnabled is set, encryption must be set to KMS.
        if (props.bucketKeyEnabled && encryptionType !== BucketEncryption.KMS) {
            throw new Error(`bucketKeyEnabled is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        bucketKeyEnabled: props.bucketKeyEnabled,
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the bucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            var _d, _e;
            const enabled = (_d = rule.enabled) !== null && _d !== void 0 ? _d : true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: (_e = rule.expiration) === null || _e === void 0 ? void 0 : _e.toDays(),
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpiration && rule.noncurrentVersionExpiration.toDays(),
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                tagFilters: self.parseTagFilters(rule.tagFilters),
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        var _d;
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        return {
            destinationBucketName: (_d = props.serverAccessLogsBucket) === null || _d === void 0 ? void 0 : _d.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    parseOwnershipControls({ objectOwnership }) {
        if (!objectOwnership) {
            return undefined;
        }
        return {
            rules: [{
                    objectOwnership,
                }],
        };
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    allowLogDelivery() {
        if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            var _d, _e, _f, _g, _h, _j;
            const format = (_d = inventory.format) !== null && _d !== void 0 ? _d : InventoryFormat.CSV;
            const frequency = (_e = inventory.frequency) !== null && _e !== void 0 ? _e : InventoryFrequency.WEEKLY;
            const id = (_f = inventory.inventoryId) !== null && _f !== void 0 ? _f : `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${(_g = inventory.destination.prefix) !== null && _g !== void 0 ? _g : ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: (_h = inventory.enabled) !== null && _h !== void 0 ? _h : true,
                includedObjectVersions: (_j = inventory.includeObjectVersions) !== null && _j !== void 0 ? _j : InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
    enableAutoDeleteObjects() {
        const provider = core_1.CustomResourceProvider.getOrCreateProvider(this, AUTO_DELETE_OBJECTS_RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'auto-delete-objects-handler'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            description: `Lambda function for auto-deleting objects in ${this.bucketName} S3 bucket.`,
        });
        // Use a bucket policy to allow the custom resource to delete
        // objects in the bucket
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: [
                ...perms.BUCKET_READ_ACTIONS,
                ...perms.BUCKET_DELETE_ACTIONS,
            ],
            resources: [
                this.bucketArn,
                this.arnForObjects('*'),
            ],
            principals: [new iam.ArnPrincipal(provider.roleArn)],
        }));
        const customResource = new core_1.CustomResource(this, 'AutoDeleteObjectsCustomResource', {
            resourceType: AUTO_DELETE_OBJECTS_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                BucketName: this.bucketName,
            },
        });
        // Ensure bucket policy is deleted AFTER the custom resource otherwise
        // we don't have permissions to list and delete in the bucket.
        // (add a `if` to make TS happy)
        if (this.policy) {
            customResource.node.addDependency(this.policy);
        }
    }
}
exports.Bucket = Bucket;
_b = JSII_RTTI_SYMBOL_1;
Bucket[_b] = { fqn: "@aws-cdk/aws-s3.Bucket", version: "1.106.0" };
/**
 * What kind of server-side encryption to apply to this bucket.
 *
 * @stability stable
 */
var BucketEncryption;
(function (BucketEncryption) {
    BucketEncryption["UNENCRYPTED"] = "NONE";
    BucketEncryption["KMS_MANAGED"] = "MANAGED";
    BucketEncryption["S3_MANAGED"] = "S3MANAGED";
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 *
 * @stability stable
 */
var EventType;
(function (EventType) {
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    EventType["OBJECT_RESTORE_POST"] = "s3:ObjectRestore:Post";
    EventType["OBJECT_RESTORE_COMPLETED"] = "s3:ObjectRestore:Completed";
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
    EventType["REPLICATION_OPERATION_FAILED_REPLICATION"] = "s3:Replication:OperationFailedReplication";
    EventType["REPLICATION_OPERATION_MISSED_THRESHOLD"] = "s3:Replication:OperationMissedThreshold";
    EventType["REPLICATION_OPERATION_REPLICATED_AFTER_THRESHOLD"] = "s3:Replication:OperationReplicatedAfterThreshold";
    EventType["REPLICATION_OPERATION_NOT_TRACKED"] = "s3:Replication:OperationNotTracked";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 * @stability stable
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    BucketAccessControl["PRIVATE"] = "Private";
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
/**
 * @stability stable
 */
class ReplaceKey {
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
    /**
     * The specific object key to use in the redirect request.
     *
     * @stability stable
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request.
     *
     * @stability stable
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
}
exports.ReplaceKey = ReplaceKey;
_c = JSII_RTTI_SYMBOL_1;
ReplaceKey[_c] = { fqn: "@aws-cdk/aws-s3.ReplaceKey", version: "1.106.0" };
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,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