"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const STAGING_TMP = '.cdk.staging';
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
    }
    synthesize(session) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Move bundling directory to staging directory
            fs.moveSync(this.bundleDir, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Temp staging directory in the working directory
        const stagingTmp = path.join('.', STAGING_TMP);
        fs.ensureDirSync(stagingTmp);
        // Create temp directory for bundling inside the temp staging directory
        const bundleDir = path.resolve(fs.mkdtempSync(path.join(stagingTmp, 'asset-bundle-')));
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            options.image._run({
                command: options.command,
                user,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : AssetStaging.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${AssetStaging.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                // Hash the hash to make sure we can use it in a file/directory name.
                // The resulting hash will also have the same length as for the other hash types.
                return crypto.createHash('sha256').update(props.assetHash).digest('hex');
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
//# sourceMappingURL=data:application/json;base64,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