import enum

from .win32_filetime import dt_to_filetime, filetime_to_dt, filetime_now

# Security descriptor conversion
# see https://docs.microsoft.com/en-us/windows/desktop/api/sddl/nf-sddl-convertstringsecuritydescriptortosecuritydescriptorw
from .bindings import ffi, lib


# TODO: should use `lib.LocalFree` to free generated security descriptor at one point
def security_descriptor_factory(str_security_descriptor):
    psd = ffi.new("SECURITY_DESCRIPTOR**")
    psd_size = ffi.new("ULONG*")
    if not lib.ConvertStringSecurityDescriptorToSecurityDescriptorW(
        str_security_descriptor,
        lib.WFSPY_STRING_SECURITY_DESCRIPTOR_REVISION,
        psd,
        psd_size,
    ):
        raise RuntimeError(f"error: {cook_ntstatus(lib.GetLastError())}")
    # assert lib.IsValidSecurityDescriptor(psd[0])
    return psd[0], psd_size[0]


class FILE_ATTRIBUTE(enum.IntEnum):
    """
    NT File attributes
    see https://docs.microsoft.com/en-us/dotnet/api/system.io.fileattributes
    """

    FILE_ATTRIBUTE_ARCHIVE = 0x20
    FILE_ATTRIBUTE_COMPRESSED = 0x800
    FILE_ATTRIBUTE_DEVICE = 0x40
    FILE_ATTRIBUTE_DIRECTORY = 0x10
    FILE_ATTRIBUTE_ENCRYPTED = 0x4000
    FILE_ATTRIBUTE_HIDDEN = 0x2
    FILE_ATTRIBUTE_INTEGRITY_STREAM = 0x8000
    FILE_ATTRIBUTE_NORMAL = 0x80
    FILE_ATTRIBUTE_NOT_CONTENT_INDEXED = 0x2000
    FILE_ATTRIBUTE_NO_SCRUB_DATA = 0x20000
    FILE_ATTRIBUTE_OFFLINE = 0x1000
    FILE_ATTRIBUTE_READONLY = 0x1
    FILE_ATTRIBUTE_RECALL_ON_DATA_ACCESS = 0x400000
    FILE_ATTRIBUTE_RECALL_ON_OPEN = 0x40000
    FILE_ATTRIBUTE_REPARSE_POINT = 0x400
    FILE_ATTRIBUTE_SPARSE_FILE = 0x200
    FILE_ATTRIBUTE_SYSTEM = 0x4
    FILE_ATTRIBUTE_TEMPORARY = 0x100
    FILE_ATTRIBUTE_VIRTUAL = 0x10000

    INVALID_FILE_ATTRIBUTES = lib.WFSPY_FILE_ATTRIBUTE_INVALID_FILE_ATTRIBUTES


class CREATE_FILE_CREATE_OPTIONS(enum.IntEnum):
    """
    `NtCreateFile`'s CreateOptions flags
    see https://docs.microsoft.com/en-us/windows/desktop/api/winternl/nf-winternl-ntcreatefile
    """

    FILE_DIRECTORY_FILE = lib.WFSPY_FILE_DIRECTORY_FILE
    FILE_NON_DIRECTORY_FILE = lib.WFSPY_FILE_NON_DIRECTORY_FILE
    FILE_WRITE_THROUGH = lib.WFSPY_FILE_WRITE_THROUGH
    FILE_SEQUENTIAL_ONLY = lib.WFSPY_FILE_SEQUENTIAL_ONLY
    FILE_RANDOM_ACCESS = lib.WFSPY_FILE_RANDOM_ACCESS
    FILE_NO_INTERMEDIATE_BUFFERING = lib.WFSPY_FILE_NO_INTERMEDIATE_BUFFERING
    FILE_SYNCHRONOUS_IO_ALERT = lib.WFSPY_FILE_SYNCHRONOUS_IO_ALERT
    FILE_SYNCHRONOUS_IO_NONALERT = lib.WFSPY_FILE_SYNCHRONOUS_IO_NONALERT
    FILE_CREATE_TREE_CONNECTION = lib.WFSPY_FILE_CREATE_TREE_CONNECTION
    FILE_NO_EA_KNOWLEDGE = lib.WFSPY_FILE_NO_EA_KNOWLEDGE
    FILE_OPEN_REPARSE_POINT = lib.WFSPY_FILE_OPEN_REPARSE_POINT
    FILE_DELETE_ON_CLOSE = lib.WFSPY_FILE_DELETE_ON_CLOSE
    FILE_OPEN_BY_FILE_ID = lib.WFSPY_FILE_OPEN_BY_FILE_ID
    FILE_OPEN_FOR_BACKUP_INTENT = lib.WFSPY_FILE_OPEN_FOR_BACKUP_INTENT
    FILE_RESERVE_OPFILTER = lib.WFSPY_FILE_RESERVE_OPFILTER
    FILE_OPEN_REQUIRING_OPLOCK = lib.WFSPY_FILE_OPEN_REQUIRING_OPLOCK
    FILE_COMPLETE_IF_OPLOCKED = lib.WFSPY_FILE_COMPLETE_IF_OPLOCKED


# NT status codes, legend says there is one for each moment in life ;-)
# see https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-erref/596a1078-e883-4972-9bbc-49e60bebca55


def nt_success(status):
    """
    Evaluates to TRUE if the return value specified by Status is a success
    type (0 − 0x3FFFFFFF) or an informational type (0x40000000 − 0x7FFFFFFF).
    """
    return 0 <= status <= 0x3FFFFFFF or 0x40000000 <= status <= 0x7FFFFFFF


def nt_information(status):
    """
    Evaluates to TRUE if the return value specified by Status is an
    informational type (0x40000000 − 0x7FFFFFFF).
    """
    return 0x40000000 <= status <= 0x7FFFFFFF


def nt_warning(status):
    """
    Evaluates to TRUE if the return value specified by Status is a warning
    type (0x80000000 − 0xBFFFFFFF).
    """
    return 0x80000000 <= status <= 0xBFFFFFFF


def nt_error(status):
    """
    Evaluates to TRUE if the return value specified by Status is an error
    type (0xC0000000 - 0xFFFFFFFF).
    """
    return 0xC0000000 <= status <= 0xFFFFFFFF


def cook_ntstatus(status):
    for candidate in NTSTATUS:
        if candidate.value == status:
            return candidate


class NTSTATUS(enum.IntEnum):
    STATUS_SUCCESS = 0x0
    STATUS_WAIT_0 = 0x0
    STATUS_WAIT_1 = 0x1
    STATUS_WAIT_2 = 0x2
    STATUS_WAIT_3 = 0x3
    STATUS_WAIT_63 = 0x3F
    STATUS_ABANDONED = 0x80
    STATUS_ABANDONED_WAIT_0 = 0x80
    STATUS_ABANDONED_WAIT_63 = 0xBF
    STATUS_USER_APC = 0xC0
    STATUS_ALERTED = 0x101
    STATUS_TIMEOUT = 0x102
    STATUS_PENDING = 0x103
    STATUS_REPARSE = 0x104
    STATUS_MORE_ENTRIES = 0x105
    STATUS_NOT_ALL_ASSIGNED = 0x106
    STATUS_SOME_NOT_MAPPED = 0x107
    STATUS_OPLOCK_BREAK_IN_PROGRESS = 0x108
    STATUS_VOLUME_MOUNTED = 0x109
    STATUS_RXACT_COMMITTED = 0x10A
    STATUS_NOTIFY_CLEANUP = 0x10B
    STATUS_NOTIFY_ENUM_DIR = 0x10C
    STATUS_NO_QUOTAS_FOR_ACCOUNT = 0x10D
    STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED = 0x10E
    STATUS_PAGE_FAULT_TRANSITION = 0x110
    STATUS_PAGE_FAULT_DEMAND_ZERO = 0x111
    STATUS_PAGE_FAULT_COPY_ON_WRITE = 0x112
    STATUS_PAGE_FAULT_GUARD_PAGE = 0x113
    STATUS_PAGE_FAULT_PAGING_FILE = 0x114
    STATUS_CACHE_PAGE_LOCKED = 0x115
    STATUS_CRASH_DUMP = 0x116
    STATUS_BUFFER_ALL_ZEROS = 0x117
    STATUS_REPARSE_OBJECT = 0x118
    STATUS_RESOURCE_REQUIREMENTS_CHANGED = 0x119
    STATUS_TRANSLATION_COMPLETE = 0x120
    STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY = 0x121
    STATUS_NOTHING_TO_TERMINATE = 0x122
    STATUS_PROCESS_NOT_IN_JOB = 0x123
    STATUS_PROCESS_IN_JOB = 0x124
    STATUS_VOLSNAP_HIBERNATE_READY = 0x125
    STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY = 0x126
    STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED = 0x127
    STATUS_INTERRUPT_STILL_CONNECTED = 0x128
    STATUS_PROCESS_CLONED = 0x129
    STATUS_FILE_LOCKED_WITH_ONLY_READERS = 0x12A
    STATUS_FILE_LOCKED_WITH_WRITERS = 0x12B
    STATUS_RESOURCEMANAGER_READ_ONLY = 0x202
    STATUS_WAIT_FOR_OPLOCK = 0x367
    DBG_EXCEPTION_HANDLED = 0x10001
    DBG_CONTINUE = 0x10002
    STATUS_FLT_IO_COMPLETE = 0x1C0001
    STATUS_FILE_NOT_AVAILABLE = 0xC0000467
    STATUS_CALLBACK_RETURNED_THREAD_AFFINITY = 0xC0000721
    STATUS_OBJECT_NAME_EXISTS = 0x40000000
    STATUS_THREAD_WAS_SUSPENDED = 0x40000001
    STATUS_WORKING_SET_LIMIT_RANGE = 0x40000002
    STATUS_IMAGE_NOT_AT_BASE = 0x40000003
    STATUS_RXACT_STATE_CREATED = 0x40000004
    STATUS_SEGMENT_NOTIFICATION = 0x40000005
    STATUS_LOCAL_USER_SESSION_KEY = 0x40000006
    STATUS_BAD_CURRENT_DIRECTORY = 0x40000007
    STATUS_SERIAL_MORE_WRITES = 0x40000008
    STATUS_REGISTRY_RECOVERED = 0x40000009
    STATUS_FT_READ_RECOVERY_FROM_BACKUP = 0x4000000A
    STATUS_FT_WRITE_RECOVERY = 0x4000000B
    STATUS_SERIAL_COUNTER_TIMEOUT = 0x4000000C
    STATUS_NULL_LM_PASSWORD = 0x4000000D
    STATUS_IMAGE_MACHINE_TYPE_MISMATCH = 0x4000000E
    STATUS_RECEIVE_PARTIAL = 0x4000000F
    STATUS_RECEIVE_EXPEDITED = 0x40000010
    STATUS_RECEIVE_PARTIAL_EXPEDITED = 0x40000011
    STATUS_EVENT_DONE = 0x40000012
    STATUS_EVENT_PENDING = 0x40000013
    STATUS_CHECKING_FILE_SYSTEM = 0x40000014
    STATUS_FATAL_APP_EXIT = 0x40000015
    STATUS_PREDEFINED_HANDLE = 0x40000016
    STATUS_WAS_UNLOCKED = 0x40000017
    STATUS_SERVICE_NOTIFICATION = 0x40000018
    STATUS_WAS_LOCKED = 0x40000019
    STATUS_LOG_HARD_ERROR = 0x4000001A
    STATUS_ALREADY_WIN32 = 0x4000001B
    STATUS_WX86_UNSIMULATE = 0x4000001C
    STATUS_WX86_CONTINUE = 0x4000001D
    STATUS_WX86_SINGLE_STEP = 0x4000001E
    STATUS_WX86_BREAKPOINT = 0x4000001F
    STATUS_WX86_EXCEPTION_CONTINUE = 0x40000020
    STATUS_WX86_EXCEPTION_LASTCHANCE = 0x40000021
    STATUS_WX86_EXCEPTION_CHAIN = 0x40000022
    STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE = 0x40000023
    STATUS_NO_YIELD_PERFORMED = 0x40000024
    STATUS_TIMER_RESUME_IGNORED = 0x40000025
    STATUS_ARBITRATION_UNHANDLED = 0x40000026
    STATUS_CARDBUS_NOT_SUPPORTED = 0x40000027
    STATUS_WX86_CREATEWX86TIB = 0x40000028
    STATUS_MP_PROCESSOR_MISMATCH = 0x40000029
    STATUS_HIBERNATED = 0x4000002A
    STATUS_RESUME_HIBERNATION = 0x4000002B
    STATUS_FIRMWARE_UPDATED = 0x4000002C
    STATUS_DRIVERS_LEAKING_LOCKED_PAGES = 0x4000002D
    STATUS_MESSAGE_RETRIEVED = 0x4000002E
    STATUS_SYSTEM_POWERSTATE_TRANSITION = 0x4000002F
    STATUS_ALPC_CHECK_COMPLETION_LIST = 0x40000030
    STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION = 0x40000031
    STATUS_ACCESS_AUDIT_BY_POLICY = 0x40000032
    STATUS_ABANDON_HIBERFILE = 0x40000033
    STATUS_BIZRULES_NOT_ENABLED = 0x40000034
    STATUS_WAKE_SYSTEM = 0x40000294
    STATUS_DS_SHUTTING_DOWN = 0x40000370
    DBG_REPLY_LATER = 0x40010001
    DBG_UNABLE_TO_PROVIDE_HANDLE = 0x40010002
    DBG_TERMINATE_THREAD = 0x40010003
    DBG_TERMINATE_PROCESS = 0x40010004
    DBG_CONTROL_C = 0x40010005
    DBG_PRINTEXCEPTION_C = 0x40010006
    DBG_RIPEXCEPTION = 0x40010007
    DBG_CONTROL_BREAK = 0x40010008
    DBG_COMMAND_EXCEPTION = 0x40010009
    RPC_NT_UUID_LOCAL_ONLY = 0x40020056
    RPC_NT_SEND_INCOMPLETE = 0x400200AF
    STATUS_CTX_CDM_CONNECT = 0x400A0004
    STATUS_CTX_CDM_DISCONNECT = 0x400A0005
    STATUS_SXS_RELEASE_ACTIVATION_CONTEXT = 0x4015000D
    STATUS_RECOVERY_NOT_NEEDED = 0x40190034
    STATUS_RM_ALREADY_STARTED = 0x40190035
    STATUS_LOG_NO_RESTART = 0x401A000C
    STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST = 0x401B00EC
    STATUS_GRAPHICS_PARTIAL_DATA_POPULATED = 0x401E000A
    STATUS_GRAPHICS_DRIVER_MISMATCH = 0x401E0117
    STATUS_GRAPHICS_MODE_NOT_PINNED = 0x401E0307
    STATUS_GRAPHICS_NO_PREFERRED_MODE = 0x401E031E
    STATUS_GRAPHICS_DATASET_IS_EMPTY = 0x401E034B
    STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET = 0x401E034C
    STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED = 0x401E0351
    STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS = 0x401E042F
    STATUS_GRAPHICS_LEADLINK_START_DEFERRED = 0x401E0437
    STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY = 0x401E0439
    STATUS_GRAPHICS_START_DEFERRED = 0x401E043A
    STATUS_NDIS_INDICATION_REQUIRED = 0x40230001
    STATUS_GUARD_PAGE_VIOLATION = 0x80000001
    STATUS_DATATYPE_MISALIGNMENT = 0x80000002
    STATUS_BREAKPOINT = 0x80000003
    STATUS_SINGLE_STEP = 0x80000004
    STATUS_BUFFER_OVERFLOW = 0x80000005
    STATUS_NO_MORE_FILES = 0x80000006
    STATUS_WAKE_SYSTEM_DEBUGGER = 0x80000007
    STATUS_HANDLES_CLOSED = 0x8000000A
    STATUS_NO_INHERITANCE = 0x8000000B
    STATUS_GUID_SUBSTITUTION_MADE = 0x8000000C
    STATUS_PARTIAL_COPY = 0x8000000D
    STATUS_DEVICE_PAPER_EMPTY = 0x8000000E
    STATUS_DEVICE_POWERED_OFF = 0x8000000F
    STATUS_DEVICE_OFF_LINE = 0x80000010
    STATUS_DEVICE_BUSY = 0x80000011
    STATUS_NO_MORE_EAS = 0x80000012
    STATUS_INVALID_EA_NAME = 0x80000013
    STATUS_EA_LIST_INCONSISTENT = 0x80000014
    STATUS_INVALID_EA_FLAG = 0x80000015
    STATUS_VERIFY_REQUIRED = 0x80000016
    STATUS_EXTRANEOUS_INFORMATION = 0x80000017
    STATUS_RXACT_COMMIT_NECESSARY = 0x80000018
    STATUS_NO_MORE_ENTRIES = 0x8000001A
    STATUS_FILEMARK_DETECTED = 0x8000001B
    STATUS_MEDIA_CHANGED = 0x8000001C
    STATUS_BUS_RESET = 0x8000001D
    STATUS_END_OF_MEDIA = 0x8000001E
    STATUS_BEGINNING_OF_MEDIA = 0x8000001F
    STATUS_MEDIA_CHECK = 0x80000020
    STATUS_SETMARK_DETECTED = 0x80000021
    STATUS_NO_DATA_DETECTED = 0x80000022
    STATUS_REDIRECTOR_HAS_OPEN_HANDLES = 0x80000023
    STATUS_SERVER_HAS_OPEN_HANDLES = 0x80000024
    STATUS_ALREADY_DISCONNECTED = 0x80000025
    STATUS_LONGJUMP = 0x80000026
    STATUS_CLEANER_CARTRIDGE_INSTALLED = 0x80000027
    STATUS_PLUGPLAY_QUERY_VETOED = 0x80000028
    STATUS_UNWIND_CONSOLIDATE = 0x80000029
    STATUS_REGISTRY_HIVE_RECOVERED = 0x8000002A
    STATUS_DLL_MIGHT_BE_INSECURE = 0x8000002B
    STATUS_DLL_MIGHT_BE_INCOMPATIBLE = 0x8000002C
    STATUS_STOPPED_ON_SYMLINK = 0x8000002D
    STATUS_DEVICE_REQUIRES_CLEANING = 0x80000288
    STATUS_DEVICE_DOOR_OPEN = 0x80000289
    STATUS_DATA_LOST_REPAIR = 0x80000803
    DBG_EXCEPTION_NOT_HANDLED = 0x80010001
    STATUS_CLUSTER_NODE_ALREADY_UP = 0x80130001
    STATUS_CLUSTER_NODE_ALREADY_DOWN = 0x80130002
    STATUS_CLUSTER_NETWORK_ALREADY_ONLINE = 0x80130003
    STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE = 0x80130004
    STATUS_CLUSTER_NODE_ALREADY_MEMBER = 0x80130005
    STATUS_COULD_NOT_RESIZE_LOG = 0x80190009
    STATUS_NO_TXF_METADATA = 0x80190029
    STATUS_CANT_RECOVER_WITH_HANDLE_OPEN = 0x80190031
    STATUS_TXF_METADATA_ALREADY_PRESENT = 0x80190041
    STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET = 0x80190042
    STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED = 0x801B00EB
    STATUS_FLT_BUFFER_TOO_SMALL = 0x801C0001
    STATUS_FVE_PARTIAL_METADATA = 0x80210001
    STATUS_FVE_TRANSIENT_STATE = 0x80210002
    STATUS_UNSUCCESSFUL = 0xC0000001
    STATUS_NOT_IMPLEMENTED = 0xC0000002
    STATUS_INVALID_INFO_CLASS = 0xC0000003
    STATUS_INFO_LENGTH_MISMATCH = 0xC0000004
    STATUS_ACCESS_VIOLATION = 0xC0000005
    STATUS_IN_PAGE_ERROR = 0xC0000006
    STATUS_PAGEFILE_QUOTA = 0xC0000007
    STATUS_INVALID_HANDLE = 0xC0000008
    STATUS_BAD_INITIAL_STACK = 0xC0000009
    STATUS_BAD_INITIAL_PC = 0xC000000A
    STATUS_INVALID_CID = 0xC000000B
    STATUS_TIMER_NOT_CANCELED = 0xC000000C
    STATUS_INVALID_PARAMETER = 0xC000000D
    STATUS_NO_SUCH_DEVICE = 0xC000000E
    STATUS_NO_SUCH_FILE = 0xC000000F
    STATUS_INVALID_DEVICE_REQUEST = 0xC0000010
    STATUS_END_OF_FILE = 0xC0000011
    STATUS_WRONG_VOLUME = 0xC0000012
    STATUS_NO_MEDIA_IN_DEVICE = 0xC0000013
    STATUS_UNRECOGNIZED_MEDIA = 0xC0000014
    STATUS_NONEXISTENT_SECTOR = 0xC0000015
    STATUS_MORE_PROCESSING_REQUIRED = 0xC0000016
    STATUS_NO_MEMORY = 0xC0000017
    STATUS_CONFLICTING_ADDRESSES = 0xC0000018
    STATUS_NOT_MAPPED_VIEW = 0xC0000019
    STATUS_UNABLE_TO_FREE_VM = 0xC000001A
    STATUS_UNABLE_TO_DELETE_SECTION = 0xC000001B
    STATUS_INVALID_SYSTEM_SERVICE = 0xC000001C
    STATUS_ILLEGAL_INSTRUCTION = 0xC000001D
    STATUS_INVALID_LOCK_SEQUENCE = 0xC000001E
    STATUS_INVALID_VIEW_SIZE = 0xC000001F
    STATUS_INVALID_FILE_FOR_SECTION = 0xC0000020
    STATUS_ALREADY_COMMITTED = 0xC0000021
    STATUS_ACCESS_DENIED = 0xC0000022
    STATUS_BUFFER_TOO_SMALL = 0xC0000023
    STATUS_OBJECT_TYPE_MISMATCH = 0xC0000024
    STATUS_NONCONTINUABLE_EXCEPTION = 0xC0000025
    STATUS_INVALID_DISPOSITION = 0xC0000026
    STATUS_UNWIND = 0xC0000027
    STATUS_BAD_STACK = 0xC0000028
    STATUS_INVALID_UNWIND_TARGET = 0xC0000029
    STATUS_NOT_LOCKED = 0xC000002A
    STATUS_PARITY_ERROR = 0xC000002B
    STATUS_UNABLE_TO_DECOMMIT_VM = 0xC000002C
    STATUS_NOT_COMMITTED = 0xC000002D
    STATUS_INVALID_PORT_ATTRIBUTES = 0xC000002E
    STATUS_PORT_MESSAGE_TOO_LONG = 0xC000002F
    STATUS_INVALID_PARAMETER_MIX = 0xC0000030
    STATUS_INVALID_QUOTA_LOWER = 0xC0000031
    STATUS_DISK_CORRUPT_ERROR = 0xC0000032
    STATUS_OBJECT_NAME_INVALID = 0xC0000033
    STATUS_OBJECT_NAME_NOT_FOUND = 0xC0000034
    STATUS_OBJECT_NAME_COLLISION = 0xC0000035
    STATUS_PORT_DISCONNECTED = 0xC0000037
    STATUS_DEVICE_ALREADY_ATTACHED = 0xC0000038
    STATUS_OBJECT_PATH_INVALID = 0xC0000039
    STATUS_OBJECT_PATH_NOT_FOUND = 0xC000003A
    STATUS_OBJECT_PATH_SYNTAX_BAD = 0xC000003B
    STATUS_DATA_OVERRUN = 0xC000003C
    STATUS_DATA_LATE_ERROR = 0xC000003D
    STATUS_DATA_ERROR = 0xC000003E
    STATUS_CRC_ERROR = 0xC000003F
    STATUS_SECTION_TOO_BIG = 0xC0000040
    STATUS_PORT_CONNECTION_REFUSED = 0xC0000041
    STATUS_INVALID_PORT_HANDLE = 0xC0000042
    STATUS_SHARING_VIOLATION = 0xC0000043
    STATUS_QUOTA_EXCEEDED = 0xC0000044
    STATUS_INVALID_PAGE_PROTECTION = 0xC0000045
    STATUS_MUTANT_NOT_OWNED = 0xC0000046
    STATUS_SEMAPHORE_LIMIT_EXCEEDED = 0xC0000047
    STATUS_PORT_ALREADY_SET = 0xC0000048
    STATUS_SECTION_NOT_IMAGE = 0xC0000049
    STATUS_SUSPEND_COUNT_EXCEEDED = 0xC000004A
    STATUS_THREAD_IS_TERMINATING = 0xC000004B
    STATUS_BAD_WORKING_SET_LIMIT = 0xC000004C
    STATUS_INCOMPATIBLE_FILE_MAP = 0xC000004D
    STATUS_SECTION_PROTECTION = 0xC000004E
    STATUS_EAS_NOT_SUPPORTED = 0xC000004F
    STATUS_EA_TOO_LARGE = 0xC0000050
    STATUS_NONEXISTENT_EA_ENTRY = 0xC0000051
    STATUS_NO_EAS_ON_FILE = 0xC0000052
    STATUS_EA_CORRUPT_ERROR = 0xC0000053
    STATUS_FILE_LOCK_CONFLICT = 0xC0000054
    STATUS_LOCK_NOT_GRANTED = 0xC0000055
    STATUS_DELETE_PENDING = 0xC0000056
    STATUS_CTL_FILE_NOT_SUPPORTED = 0xC0000057
    STATUS_UNKNOWN_REVISION = 0xC0000058
    STATUS_REVISION_MISMATCH = 0xC0000059
    STATUS_INVALID_OWNER = 0xC000005A
    STATUS_INVALID_PRIMARY_GROUP = 0xC000005B
    STATUS_NO_IMPERSONATION_TOKEN = 0xC000005C
    STATUS_CANT_DISABLE_MANDATORY = 0xC000005D
    STATUS_NO_LOGON_SERVERS = 0xC000005E
    STATUS_NO_SUCH_LOGON_SESSION = 0xC000005F
    STATUS_NO_SUCH_PRIVILEGE = 0xC0000060
    STATUS_PRIVILEGE_NOT_HELD = 0xC0000061
    STATUS_INVALID_ACCOUNT_NAME = 0xC0000062
    STATUS_USER_EXISTS = 0xC0000063
    STATUS_NO_SUCH_USER = 0xC0000064
    STATUS_GROUP_EXISTS = 0xC0000065
    STATUS_NO_SUCH_GROUP = 0xC0000066
    STATUS_MEMBER_IN_GROUP = 0xC0000067
    STATUS_MEMBER_NOT_IN_GROUP = 0xC0000068
    STATUS_LAST_ADMIN = 0xC0000069
    STATUS_WRONG_PASSWORD = 0xC000006A
    STATUS_ILL_FORMED_PASSWORD = 0xC000006B
    STATUS_PASSWORD_RESTRICTION = 0xC000006C
    STATUS_LOGON_FAILURE = 0xC000006D
    STATUS_ACCOUNT_RESTRICTION = 0xC000006E
    STATUS_INVALID_LOGON_HOURS = 0xC000006F
    STATUS_INVALID_WORKSTATION = 0xC0000070
    STATUS_PASSWORD_EXPIRED = 0xC0000071
    STATUS_ACCOUNT_DISABLED = 0xC0000072
    STATUS_NONE_MAPPED = 0xC0000073
    STATUS_TOO_MANY_LUIDS_REQUESTED = 0xC0000074
    STATUS_LUIDS_EXHAUSTED = 0xC0000075
    STATUS_INVALID_SUB_AUTHORITY = 0xC0000076
    STATUS_INVALID_ACL = 0xC0000077
    STATUS_INVALID_SID = 0xC0000078
    STATUS_INVALID_SECURITY_DESCR = 0xC0000079
    STATUS_PROCEDURE_NOT_FOUND = 0xC000007A
    STATUS_INVALID_IMAGE_FORMAT = 0xC000007B
    STATUS_NO_TOKEN = 0xC000007C
    STATUS_BAD_INHERITANCE_ACL = 0xC000007D
    STATUS_RANGE_NOT_LOCKED = 0xC000007E
    STATUS_DISK_FULL = 0xC000007F
    STATUS_SERVER_DISABLED = 0xC0000080
    STATUS_SERVER_NOT_DISABLED = 0xC0000081
    STATUS_TOO_MANY_GUIDS_REQUESTED = 0xC0000082
    STATUS_GUIDS_EXHAUSTED = 0xC0000083
    STATUS_INVALID_ID_AUTHORITY = 0xC0000084
    STATUS_AGENTS_EXHAUSTED = 0xC0000085
    STATUS_INVALID_VOLUME_LABEL = 0xC0000086
    STATUS_SECTION_NOT_EXTENDED = 0xC0000087
    STATUS_NOT_MAPPED_DATA = 0xC0000088
    STATUS_RESOURCE_DATA_NOT_FOUND = 0xC0000089
    STATUS_RESOURCE_TYPE_NOT_FOUND = 0xC000008A
    STATUS_RESOURCE_NAME_NOT_FOUND = 0xC000008B
    STATUS_ARRAY_BOUNDS_EXCEEDED = 0xC000008C
    STATUS_FLOAT_DENORMAL_OPERAND = 0xC000008D
    STATUS_FLOAT_DIVIDE_BY_ZERO = 0xC000008E
    STATUS_FLOAT_INEXACT_RESULT = 0xC000008F
    STATUS_FLOAT_INVALID_OPERATION = 0xC0000090
    STATUS_FLOAT_OVERFLOW = 0xC0000091
    STATUS_FLOAT_STACK_CHECK = 0xC0000092
    STATUS_FLOAT_UNDERFLOW = 0xC0000093
    STATUS_INTEGER_DIVIDE_BY_ZERO = 0xC0000094
    STATUS_INTEGER_OVERFLOW = 0xC0000095
    STATUS_PRIVILEGED_INSTRUCTION = 0xC0000096
    STATUS_TOO_MANY_PAGING_FILES = 0xC0000097
    STATUS_FILE_INVALID = 0xC0000098
    STATUS_ALLOTTED_SPACE_EXCEEDED = 0xC0000099
    STATUS_INSUFFICIENT_RESOURCES = 0xC000009A
    STATUS_DFS_EXIT_PATH_FOUND = 0xC000009B
    STATUS_DEVICE_DATA_ERROR = 0xC000009C
    STATUS_DEVICE_NOT_CONNECTED = 0xC000009D
    STATUS_FREE_VM_NOT_AT_BASE = 0xC000009F
    STATUS_MEMORY_NOT_ALLOCATED = 0xC00000A0
    STATUS_WORKING_SET_QUOTA = 0xC00000A1
    STATUS_MEDIA_WRITE_PROTECTED = 0xC00000A2
    STATUS_DEVICE_NOT_READY = 0xC00000A3
    STATUS_INVALID_GROUP_ATTRIBUTES = 0xC00000A4
    STATUS_BAD_IMPERSONATION_LEVEL = 0xC00000A5
    STATUS_CANT_OPEN_ANONYMOUS = 0xC00000A6
    STATUS_BAD_VALIDATION_CLASS = 0xC00000A7
    STATUS_BAD_TOKEN_TYPE = 0xC00000A8
    STATUS_BAD_MASTER_BOOT_RECORD = 0xC00000A9
    STATUS_INSTRUCTION_MISALIGNMENT = 0xC00000AA
    STATUS_INSTANCE_NOT_AVAILABLE = 0xC00000AB
    STATUS_PIPE_NOT_AVAILABLE = 0xC00000AC
    STATUS_INVALID_PIPE_STATE = 0xC00000AD
    STATUS_PIPE_BUSY = 0xC00000AE
    STATUS_ILLEGAL_FUNCTION = 0xC00000AF
    STATUS_PIPE_DISCONNECTED = 0xC00000B0
    STATUS_PIPE_CLOSING = 0xC00000B1
    STATUS_PIPE_CONNECTED = 0xC00000B2
    STATUS_PIPE_LISTENING = 0xC00000B3
    STATUS_INVALID_READ_MODE = 0xC00000B4
    STATUS_IO_TIMEOUT = 0xC00000B5
    STATUS_FILE_FORCED_CLOSED = 0xC00000B6
    STATUS_PROFILING_NOT_STARTED = 0xC00000B7
    STATUS_PROFILING_NOT_STOPPED = 0xC00000B8
    STATUS_COULD_NOT_INTERPRET = 0xC00000B9
    STATUS_FILE_IS_A_DIRECTORY = 0xC00000BA
    STATUS_NOT_SUPPORTED = 0xC00000BB
    STATUS_REMOTE_NOT_LISTENING = 0xC00000BC
    STATUS_DUPLICATE_NAME = 0xC00000BD
    STATUS_BAD_NETWORK_PATH = 0xC00000BE
    STATUS_NETWORK_BUSY = 0xC00000BF
    STATUS_DEVICE_DOES_NOT_EXIST = 0xC00000C0
    STATUS_TOO_MANY_COMMANDS = 0xC00000C1
    STATUS_ADAPTER_HARDWARE_ERROR = 0xC00000C2
    STATUS_INVALID_NETWORK_RESPONSE = 0xC00000C3
    STATUS_UNEXPECTED_NETWORK_ERROR = 0xC00000C4
    STATUS_BAD_REMOTE_ADAPTER = 0xC00000C5
    STATUS_PRINT_QUEUE_FULL = 0xC00000C6
    STATUS_NO_SPOOL_SPACE = 0xC00000C7
    STATUS_PRINT_CANCELLED = 0xC00000C8
    STATUS_NETWORK_NAME_DELETED = 0xC00000C9
    STATUS_NETWORK_ACCESS_DENIED = 0xC00000CA
    STATUS_BAD_DEVICE_TYPE = 0xC00000CB
    STATUS_BAD_NETWORK_NAME = 0xC00000CC
    STATUS_TOO_MANY_NAMES = 0xC00000CD
    STATUS_TOO_MANY_SESSIONS = 0xC00000CE
    STATUS_SHARING_PAUSED = 0xC00000CF
    STATUS_REQUEST_NOT_ACCEPTED = 0xC00000D0
    STATUS_REDIRECTOR_PAUSED = 0xC00000D1
    STATUS_NET_WRITE_FAULT = 0xC00000D2
    STATUS_PROFILING_AT_LIMIT = 0xC00000D3
    STATUS_NOT_SAME_DEVICE = 0xC00000D4
    STATUS_FILE_RENAMED = 0xC00000D5
    STATUS_VIRTUAL_CIRCUIT_CLOSED = 0xC00000D6
    STATUS_NO_SECURITY_ON_OBJECT = 0xC00000D7
    STATUS_CANT_WAIT = 0xC00000D8
    STATUS_PIPE_EMPTY = 0xC00000D9
    STATUS_CANT_ACCESS_DOMAIN_INFO = 0xC00000DA
    STATUS_CANT_TERMINATE_SELF = 0xC00000DB
    STATUS_INVALID_SERVER_STATE = 0xC00000DC
    STATUS_INVALID_DOMAIN_STATE = 0xC00000DD
    STATUS_INVALID_DOMAIN_ROLE = 0xC00000DE
    STATUS_NO_SUCH_DOMAIN = 0xC00000DF
    STATUS_DOMAIN_EXISTS = 0xC00000E0
    STATUS_DOMAIN_LIMIT_EXCEEDED = 0xC00000E1
    STATUS_OPLOCK_NOT_GRANTED = 0xC00000E2
    STATUS_INVALID_OPLOCK_PROTOCOL = 0xC00000E3
    STATUS_INTERNAL_DB_CORRUPTION = 0xC00000E4
    STATUS_INTERNAL_ERROR = 0xC00000E5
    STATUS_GENERIC_NOT_MAPPED = 0xC00000E6
    STATUS_BAD_DESCRIPTOR_FORMAT = 0xC00000E7
    STATUS_INVALID_USER_BUFFER = 0xC00000E8
    STATUS_UNEXPECTED_IO_ERROR = 0xC00000E9
    STATUS_UNEXPECTED_MM_CREATE_ERR = 0xC00000EA
    STATUS_UNEXPECTED_MM_MAP_ERROR = 0xC00000EB
    STATUS_UNEXPECTED_MM_EXTEND_ERR = 0xC00000EC
    STATUS_NOT_LOGON_PROCESS = 0xC00000ED
    STATUS_LOGON_SESSION_EXISTS = 0xC00000EE
    STATUS_INVALID_PARAMETER_1 = 0xC00000EF
    STATUS_INVALID_PARAMETER_2 = 0xC00000F0
    STATUS_INVALID_PARAMETER_3 = 0xC00000F1
    STATUS_INVALID_PARAMETER_4 = 0xC00000F2
    STATUS_INVALID_PARAMETER_5 = 0xC00000F3
    STATUS_INVALID_PARAMETER_6 = 0xC00000F4
    STATUS_INVALID_PARAMETER_7 = 0xC00000F5
    STATUS_INVALID_PARAMETER_8 = 0xC00000F6
    STATUS_INVALID_PARAMETER_9 = 0xC00000F7
    STATUS_INVALID_PARAMETER_10 = 0xC00000F8
    STATUS_INVALID_PARAMETER_11 = 0xC00000F9
    STATUS_INVALID_PARAMETER_12 = 0xC00000FA
    STATUS_REDIRECTOR_NOT_STARTED = 0xC00000FB
    STATUS_REDIRECTOR_STARTED = 0xC00000FC
    STATUS_STACK_OVERFLOW = 0xC00000FD
    STATUS_NO_SUCH_PACKAGE = 0xC00000FE
    STATUS_BAD_FUNCTION_TABLE = 0xC00000FF
    STATUS_VARIABLE_NOT_FOUND = 0xC0000100
    STATUS_DIRECTORY_NOT_EMPTY = 0xC0000101
    STATUS_FILE_CORRUPT_ERROR = 0xC0000102
    STATUS_NOT_A_DIRECTORY = 0xC0000103
    STATUS_BAD_LOGON_SESSION_STATE = 0xC0000104
    STATUS_LOGON_SESSION_COLLISION = 0xC0000105
    STATUS_NAME_TOO_LONG = 0xC0000106
    STATUS_FILES_OPEN = 0xC0000107
    STATUS_CONNECTION_IN_USE = 0xC0000108
    STATUS_MESSAGE_NOT_FOUND = 0xC0000109
    STATUS_PROCESS_IS_TERMINATING = 0xC000010A
    STATUS_INVALID_LOGON_TYPE = 0xC000010B
    STATUS_NO_GUID_TRANSLATION = 0xC000010C
    STATUS_CANNOT_IMPERSONATE = 0xC000010D
    STATUS_IMAGE_ALREADY_LOADED = 0xC000010E
    STATUS_NO_LDT = 0xC0000117
    STATUS_INVALID_LDT_SIZE = 0xC0000118
    STATUS_INVALID_LDT_OFFSET = 0xC0000119
    STATUS_INVALID_LDT_DESCRIPTOR = 0xC000011A
    STATUS_INVALID_IMAGE_NE_FORMAT = 0xC000011B
    STATUS_RXACT_INVALID_STATE = 0xC000011C
    STATUS_RXACT_COMMIT_FAILURE = 0xC000011D
    STATUS_MAPPED_FILE_SIZE_ZERO = 0xC000011E
    STATUS_TOO_MANY_OPENED_FILES = 0xC000011F
    STATUS_CANCELLED = 0xC0000120
    STATUS_CANNOT_DELETE = 0xC0000121
    STATUS_INVALID_COMPUTER_NAME = 0xC0000122
    STATUS_FILE_DELETED = 0xC0000123
    STATUS_SPECIAL_ACCOUNT = 0xC0000124
    STATUS_SPECIAL_GROUP = 0xC0000125
    STATUS_SPECIAL_USER = 0xC0000126
    STATUS_MEMBERS_PRIMARY_GROUP = 0xC0000127
    STATUS_FILE_CLOSED = 0xC0000128
    STATUS_TOO_MANY_THREADS = 0xC0000129
    STATUS_THREAD_NOT_IN_PROCESS = 0xC000012A
    STATUS_TOKEN_ALREADY_IN_USE = 0xC000012B
    STATUS_PAGEFILE_QUOTA_EXCEEDED = 0xC000012C
    STATUS_COMMITMENT_LIMIT = 0xC000012D
    STATUS_INVALID_IMAGE_LE_FORMAT = 0xC000012E
    STATUS_INVALID_IMAGE_NOT_MZ = 0xC000012F
    STATUS_INVALID_IMAGE_PROTECT = 0xC0000130
    STATUS_INVALID_IMAGE_WIN_16 = 0xC0000131
    STATUS_LOGON_SERVER_CONFLICT = 0xC0000132
    STATUS_TIME_DIFFERENCE_AT_DC = 0xC0000133
    STATUS_SYNCHRONIZATION_REQUIRED = 0xC0000134
    STATUS_DLL_NOT_FOUND = 0xC0000135
    STATUS_OPEN_FAILED = 0xC0000136
    STATUS_IO_PRIVILEGE_FAILED = 0xC0000137
    STATUS_ORDINAL_NOT_FOUND = 0xC0000138
    STATUS_ENTRYPOINT_NOT_FOUND = 0xC0000139
    STATUS_CONTROL_C_EXIT = 0xC000013A
    STATUS_LOCAL_DISCONNECT = 0xC000013B
    STATUS_REMOTE_DISCONNECT = 0xC000013C
    STATUS_REMOTE_RESOURCES = 0xC000013D
    STATUS_LINK_FAILED = 0xC000013E
    STATUS_LINK_TIMEOUT = 0xC000013F
    STATUS_INVALID_CONNECTION = 0xC0000140
    STATUS_INVALID_ADDRESS = 0xC0000141
    STATUS_DLL_INIT_FAILED = 0xC0000142
    STATUS_MISSING_SYSTEMFILE = 0xC0000143
    STATUS_UNHANDLED_EXCEPTION = 0xC0000144
    STATUS_APP_INIT_FAILURE = 0xC0000145
    STATUS_PAGEFILE_CREATE_FAILED = 0xC0000146
    STATUS_NO_PAGEFILE = 0xC0000147
    STATUS_INVALID_LEVEL = 0xC0000148
    STATUS_WRONG_PASSWORD_CORE = 0xC0000149
    STATUS_ILLEGAL_FLOAT_CONTEXT = 0xC000014A
    STATUS_PIPE_BROKEN = 0xC000014B
    STATUS_REGISTRY_CORRUPT = 0xC000014C
    STATUS_REGISTRY_IO_FAILED = 0xC000014D
    STATUS_NO_EVENT_PAIR = 0xC000014E
    STATUS_UNRECOGNIZED_VOLUME = 0xC000014F
    STATUS_SERIAL_NO_DEVICE_INITED = 0xC0000150
    STATUS_NO_SUCH_ALIAS = 0xC0000151
    STATUS_MEMBER_NOT_IN_ALIAS = 0xC0000152
    STATUS_MEMBER_IN_ALIAS = 0xC0000153
    STATUS_ALIAS_EXISTS = 0xC0000154
    STATUS_LOGON_NOT_GRANTED = 0xC0000155
    STATUS_TOO_MANY_SECRETS = 0xC0000156
    STATUS_SECRET_TOO_LONG = 0xC0000157
    STATUS_INTERNAL_DB_ERROR = 0xC0000158
    STATUS_FULLSCREEN_MODE = 0xC0000159
    STATUS_TOO_MANY_CONTEXT_IDS = 0xC000015A
    STATUS_LOGON_TYPE_NOT_GRANTED = 0xC000015B
    STATUS_NOT_REGISTRY_FILE = 0xC000015C
    STATUS_NT_CROSS_ENCRYPTION_REQUIRED = 0xC000015D
    STATUS_DOMAIN_CTRLR_CONFIG_ERROR = 0xC000015E
    STATUS_FT_MISSING_MEMBER = 0xC000015F
    STATUS_ILL_FORMED_SERVICE_ENTRY = 0xC0000160
    STATUS_ILLEGAL_CHARACTER = 0xC0000161
    STATUS_UNMAPPABLE_CHARACTER = 0xC0000162
    STATUS_UNDEFINED_CHARACTER = 0xC0000163
    STATUS_FLOPPY_VOLUME = 0xC0000164
    STATUS_FLOPPY_ID_MARK_NOT_FOUND = 0xC0000165
    STATUS_FLOPPY_WRONG_CYLINDER = 0xC0000166
    STATUS_FLOPPY_UNKNOWN_ERROR = 0xC0000167
    STATUS_FLOPPY_BAD_REGISTERS = 0xC0000168
    STATUS_DISK_RECALIBRATE_FAILED = 0xC0000169
    STATUS_DISK_OPERATION_FAILED = 0xC000016A
    STATUS_DISK_RESET_FAILED = 0xC000016B
    STATUS_SHARED_IRQ_BUSY = 0xC000016C
    STATUS_FT_ORPHANING = 0xC000016D
    STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT = 0xC000016E
    STATUS_PARTITION_FAILURE = 0xC0000172
    STATUS_INVALID_BLOCK_LENGTH = 0xC0000173
    STATUS_DEVICE_NOT_PARTITIONED = 0xC0000174
    STATUS_UNABLE_TO_LOCK_MEDIA = 0xC0000175
    STATUS_UNABLE_TO_UNLOAD_MEDIA = 0xC0000176
    STATUS_EOM_OVERFLOW = 0xC0000177
    STATUS_NO_MEDIA = 0xC0000178
    STATUS_NO_SUCH_MEMBER = 0xC000017A
    STATUS_INVALID_MEMBER = 0xC000017B
    STATUS_KEY_DELETED = 0xC000017C
    STATUS_NO_LOG_SPACE = 0xC000017D
    STATUS_TOO_MANY_SIDS = 0xC000017E
    STATUS_LM_CROSS_ENCRYPTION_REQUIRED = 0xC000017F
    STATUS_KEY_HAS_CHILDREN = 0xC0000180
    STATUS_CHILD_MUST_BE_VOLATILE = 0xC0000181
    STATUS_DEVICE_CONFIGURATION_ERROR = 0xC0000182
    STATUS_DRIVER_INTERNAL_ERROR = 0xC0000183
    STATUS_INVALID_DEVICE_STATE = 0xC0000184
    STATUS_IO_DEVICE_ERROR = 0xC0000185
    STATUS_DEVICE_PROTOCOL_ERROR = 0xC0000186
    STATUS_BACKUP_CONTROLLER = 0xC0000187
    STATUS_LOG_FILE_FULL = 0xC0000188
    STATUS_TOO_LATE = 0xC0000189
    STATUS_NO_TRUST_LSA_SECRET = 0xC000018A
    STATUS_NO_TRUST_SAM_ACCOUNT = 0xC000018B
    STATUS_TRUSTED_DOMAIN_FAILURE = 0xC000018C
    STATUS_TRUSTED_RELATIONSHIP_FAILURE = 0xC000018D
    STATUS_EVENTLOG_FILE_CORRUPT = 0xC000018E
    STATUS_EVENTLOG_CANT_START = 0xC000018F
    STATUS_TRUST_FAILURE = 0xC0000190
    STATUS_MUTANT_LIMIT_EXCEEDED = 0xC0000191
    STATUS_NETLOGON_NOT_STARTED = 0xC0000192
    STATUS_ACCOUNT_EXPIRED = 0xC0000193
    STATUS_POSSIBLE_DEADLOCK = 0xC0000194
    STATUS_NETWORK_CREDENTIAL_CONFLICT = 0xC0000195
    STATUS_REMOTE_SESSION_LIMIT = 0xC0000196
    STATUS_EVENTLOG_FILE_CHANGED = 0xC0000197
    STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT = 0xC0000198
    STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT = 0xC0000199
    STATUS_NOLOGON_SERVER_TRUST_ACCOUNT = 0xC000019A
    STATUS_DOMAIN_TRUST_INCONSISTENT = 0xC000019B
    STATUS_FS_DRIVER_REQUIRED = 0xC000019C
    STATUS_IMAGE_ALREADY_LOADED_AS_DLL = 0xC000019D
    STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING = 0xC000019E
    STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME = 0xC000019F
    STATUS_SECURITY_STREAM_IS_INCONSISTENT = 0xC00001A0
    STATUS_INVALID_LOCK_RANGE = 0xC00001A1
    STATUS_INVALID_ACE_CONDITION = 0xC00001A2
    STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT = 0xC00001A3
    STATUS_NOTIFICATION_GUID_ALREADY_DEFINED = 0xC00001A4
    STATUS_NETWORK_OPEN_RESTRICTION = 0xC0000201
    STATUS_NO_USER_SESSION_KEY = 0xC0000202
    STATUS_USER_SESSION_DELETED = 0xC0000203
    STATUS_RESOURCE_LANG_NOT_FOUND = 0xC0000204
    STATUS_INSUFF_SERVER_RESOURCES = 0xC0000205
    STATUS_INVALID_BUFFER_SIZE = 0xC0000206
    STATUS_INVALID_ADDRESS_COMPONENT = 0xC0000207
    STATUS_INVALID_ADDRESS_WILDCARD = 0xC0000208
    STATUS_TOO_MANY_ADDRESSES = 0xC0000209
    STATUS_ADDRESS_ALREADY_EXISTS = 0xC000020A
    STATUS_ADDRESS_CLOSED = 0xC000020B
    STATUS_CONNECTION_DISCONNECTED = 0xC000020C
    STATUS_CONNECTION_RESET = 0xC000020D
    STATUS_TOO_MANY_NODES = 0xC000020E
    STATUS_TRANSACTION_ABORTED = 0xC000020F
    STATUS_TRANSACTION_TIMED_OUT = 0xC0000210
    STATUS_TRANSACTION_NO_RELEASE = 0xC0000211
    STATUS_TRANSACTION_NO_MATCH = 0xC0000212
    STATUS_TRANSACTION_RESPONDED = 0xC0000213
    STATUS_TRANSACTION_INVALID_ID = 0xC0000214
    STATUS_TRANSACTION_INVALID_TYPE = 0xC0000215
    STATUS_NOT_SERVER_SESSION = 0xC0000216
    STATUS_NOT_CLIENT_SESSION = 0xC0000217
    STATUS_CANNOT_LOAD_REGISTRY_FILE = 0xC0000218
    STATUS_DEBUG_ATTACH_FAILED = 0xC0000219
    STATUS_SYSTEM_PROCESS_TERMINATED = 0xC000021A
    STATUS_DATA_NOT_ACCEPTED = 0xC000021B
    STATUS_NO_BROWSER_SERVERS_FOUND = 0xC000021C
    STATUS_VDM_HARD_ERROR = 0xC000021D
    STATUS_DRIVER_CANCEL_TIMEOUT = 0xC000021E
    STATUS_REPLY_MESSAGE_MISMATCH = 0xC000021F
    STATUS_MAPPED_ALIGNMENT = 0xC0000220
    STATUS_IMAGE_CHECKSUM_MISMATCH = 0xC0000221
    STATUS_LOST_WRITEBEHIND_DATA = 0xC0000222
    STATUS_CLIENT_SERVER_PARAMETERS_INVALID = 0xC0000223
    STATUS_PASSWORD_MUST_CHANGE = 0xC0000224
    STATUS_NOT_FOUND = 0xC0000225
    STATUS_NOT_TINY_STREAM = 0xC0000226
    STATUS_RECOVERY_FAILURE = 0xC0000227
    STATUS_STACK_OVERFLOW_READ = 0xC0000228
    STATUS_FAIL_CHECK = 0xC0000229
    STATUS_DUPLICATE_OBJECTID = 0xC000022A
    STATUS_OBJECTID_EXISTS = 0xC000022B
    STATUS_CONVERT_TO_LARGE = 0xC000022C
    STATUS_RETRY = 0xC000022D
    STATUS_FOUND_OUT_OF_SCOPE = 0xC000022E
    STATUS_ALLOCATE_BUCKET = 0xC000022F
    STATUS_PROPSET_NOT_FOUND = 0xC0000230
    STATUS_MARSHALL_OVERFLOW = 0xC0000231
    STATUS_INVALID_VARIANT = 0xC0000232
    STATUS_DOMAIN_CONTROLLER_NOT_FOUND = 0xC0000233
    STATUS_ACCOUNT_LOCKED_OUT = 0xC0000234
    STATUS_HANDLE_NOT_CLOSABLE = 0xC0000235
    STATUS_CONNECTION_REFUSED = 0xC0000236
    STATUS_GRACEFUL_DISCONNECT = 0xC0000237
    STATUS_ADDRESS_ALREADY_ASSOCIATED = 0xC0000238
    STATUS_ADDRESS_NOT_ASSOCIATED = 0xC0000239
    STATUS_CONNECTION_INVALID = 0xC000023A
    STATUS_CONNECTION_ACTIVE = 0xC000023B
    STATUS_NETWORK_UNREACHABLE = 0xC000023C
    STATUS_HOST_UNREACHABLE = 0xC000023D
    STATUS_PROTOCOL_UNREACHABLE = 0xC000023E
    STATUS_PORT_UNREACHABLE = 0xC000023F
    STATUS_REQUEST_ABORTED = 0xC0000240
    STATUS_CONNECTION_ABORTED = 0xC0000241
    STATUS_BAD_COMPRESSION_BUFFER = 0xC0000242
    STATUS_USER_MAPPED_FILE = 0xC0000243
    STATUS_AUDIT_FAILED = 0xC0000244
    STATUS_TIMER_RESOLUTION_NOT_SET = 0xC0000245
    STATUS_CONNECTION_COUNT_LIMIT = 0xC0000246
    STATUS_LOGIN_TIME_RESTRICTION = 0xC0000247
    STATUS_LOGIN_WKSTA_RESTRICTION = 0xC0000248
    STATUS_IMAGE_MP_UP_MISMATCH = 0xC0000249
    STATUS_INSUFFICIENT_LOGON_INFO = 0xC0000250
    STATUS_BAD_DLL_ENTRYPOINT = 0xC0000251
    STATUS_BAD_SERVICE_ENTRYPOINT = 0xC0000252
    STATUS_LPC_REPLY_LOST = 0xC0000253
    STATUS_IP_ADDRESS_CONFLICT1 = 0xC0000254
    STATUS_IP_ADDRESS_CONFLICT2 = 0xC0000255
    STATUS_REGISTRY_QUOTA_LIMIT = 0xC0000256
    STATUS_PATH_NOT_COVERED = 0xC0000257
    STATUS_NO_CALLBACK_ACTIVE = 0xC0000258
    STATUS_LICENSE_QUOTA_EXCEEDED = 0xC0000259
    STATUS_PWD_TOO_SHORT = 0xC000025A
    STATUS_PWD_TOO_RECENT = 0xC000025B
    STATUS_PWD_HISTORY_CONFLICT = 0xC000025C
    STATUS_PLUGPLAY_NO_DEVICE = 0xC000025E
    STATUS_UNSUPPORTED_COMPRESSION = 0xC000025F
    STATUS_INVALID_HW_PROFILE = 0xC0000260
    STATUS_INVALID_PLUGPLAY_DEVICE_PATH = 0xC0000261
    STATUS_DRIVER_ORDINAL_NOT_FOUND = 0xC0000262
    STATUS_DRIVER_ENTRYPOINT_NOT_FOUND = 0xC0000263
    STATUS_RESOURCE_NOT_OWNED = 0xC0000264
    STATUS_TOO_MANY_LINKS = 0xC0000265
    STATUS_QUOTA_LIST_INCONSISTENT = 0xC0000266
    STATUS_FILE_IS_OFFLINE = 0xC0000267
    STATUS_EVALUATION_EXPIRATION = 0xC0000268
    STATUS_ILLEGAL_DLL_RELOCATION = 0xC0000269
    STATUS_LICENSE_VIOLATION = 0xC000026A
    STATUS_DLL_INIT_FAILED_LOGOFF = 0xC000026B
    STATUS_DRIVER_UNABLE_TO_LOAD = 0xC000026C
    STATUS_DFS_UNAVAILABLE = 0xC000026D
    STATUS_VOLUME_DISMOUNTED = 0xC000026E
    STATUS_WX86_INTERNAL_ERROR = 0xC000026F
    STATUS_WX86_FLOAT_STACK_CHECK = 0xC0000270
    STATUS_VALIDATE_CONTINUE = 0xC0000271
    STATUS_NO_MATCH = 0xC0000272
    STATUS_NO_MORE_MATCHES = 0xC0000273
    STATUS_NOT_A_REPARSE_POINT = 0xC0000275
    STATUS_IO_REPARSE_TAG_INVALID = 0xC0000276
    STATUS_IO_REPARSE_TAG_MISMATCH = 0xC0000277
    STATUS_IO_REPARSE_DATA_INVALID = 0xC0000278
    STATUS_IO_REPARSE_TAG_NOT_HANDLED = 0xC0000279
    STATUS_REPARSE_POINT_NOT_RESOLVED = 0xC0000280
    STATUS_DIRECTORY_IS_A_REPARSE_POINT = 0xC0000281
    STATUS_RANGE_LIST_CONFLICT = 0xC0000282
    STATUS_SOURCE_ELEMENT_EMPTY = 0xC0000283
    STATUS_DESTINATION_ELEMENT_FULL = 0xC0000284
    STATUS_ILLEGAL_ELEMENT_ADDRESS = 0xC0000285
    STATUS_MAGAZINE_NOT_PRESENT = 0xC0000286
    STATUS_REINITIALIZATION_NEEDED = 0xC0000287
    STATUS_ENCRYPTION_FAILED = 0xC000028A
    STATUS_DECRYPTION_FAILED = 0xC000028B
    STATUS_RANGE_NOT_FOUND = 0xC000028C
    STATUS_NO_RECOVERY_POLICY = 0xC000028D
    STATUS_NO_EFS = 0xC000028E
    STATUS_WRONG_EFS = 0xC000028F
    STATUS_NO_USER_KEYS = 0xC0000290
    STATUS_FILE_NOT_ENCRYPTED = 0xC0000291
    STATUS_NOT_EXPORT_FORMAT = 0xC0000292
    STATUS_FILE_ENCRYPTED = 0xC0000293
    STATUS_WMI_GUID_NOT_FOUND = 0xC0000295
    STATUS_WMI_INSTANCE_NOT_FOUND = 0xC0000296
    STATUS_WMI_ITEMID_NOT_FOUND = 0xC0000297
    STATUS_WMI_TRY_AGAIN = 0xC0000298
    STATUS_SHARED_POLICY = 0xC0000299
    STATUS_POLICY_OBJECT_NOT_FOUND = 0xC000029A
    STATUS_POLICY_ONLY_IN_DS = 0xC000029B
    STATUS_VOLUME_NOT_UPGRADED = 0xC000029C
    STATUS_REMOTE_STORAGE_NOT_ACTIVE = 0xC000029D
    STATUS_REMOTE_STORAGE_MEDIA_ERROR = 0xC000029E
    STATUS_NO_TRACKING_SERVICE = 0xC000029F
    STATUS_SERVER_SID_MISMATCH = 0xC00002A0
    STATUS_DS_NO_ATTRIBUTE_OR_VALUE = 0xC00002A1
    STATUS_DS_INVALID_ATTRIBUTE_SYNTAX = 0xC00002A2
    STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED = 0xC00002A3
    STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS = 0xC00002A4
    STATUS_DS_BUSY = 0xC00002A5
    STATUS_DS_UNAVAILABLE = 0xC00002A6
    STATUS_DS_NO_RIDS_ALLOCATED = 0xC00002A7
    STATUS_DS_NO_MORE_RIDS = 0xC00002A8
    STATUS_DS_INCORRECT_ROLE_OWNER = 0xC00002A9
    STATUS_DS_RIDMGR_INIT_ERROR = 0xC00002AA
    STATUS_DS_OBJ_CLASS_VIOLATION = 0xC00002AB
    STATUS_DS_CANT_ON_NON_LEAF = 0xC00002AC
    STATUS_DS_CANT_ON_RDN = 0xC00002AD
    STATUS_DS_CANT_MOD_OBJ_CLASS = 0xC00002AE
    STATUS_DS_CROSS_DOM_MOVE_FAILED = 0xC00002AF
    STATUS_DS_GC_NOT_AVAILABLE = 0xC00002B0
    STATUS_DIRECTORY_SERVICE_REQUIRED = 0xC00002B1
    STATUS_REPARSE_ATTRIBUTE_CONFLICT = 0xC00002B2
    STATUS_CANT_ENABLE_DENY_ONLY = 0xC00002B3
    STATUS_FLOAT_MULTIPLE_FAULTS = 0xC00002B4
    STATUS_FLOAT_MULTIPLE_TRAPS = 0xC00002B5
    STATUS_DEVICE_REMOVED = 0xC00002B6
    STATUS_JOURNAL_DELETE_IN_PROGRESS = 0xC00002B7
    STATUS_JOURNAL_NOT_ACTIVE = 0xC00002B8
    STATUS_NOINTERFACE = 0xC00002B9
    STATUS_DS_ADMIN_LIMIT_EXCEEDED = 0xC00002C1
    STATUS_DRIVER_FAILED_SLEEP = 0xC00002C2
    STATUS_MUTUAL_AUTHENTICATION_FAILED = 0xC00002C3
    STATUS_CORRUPT_SYSTEM_FILE = 0xC00002C4
    STATUS_DATATYPE_MISALIGNMENT_ERROR = 0xC00002C5
    STATUS_WMI_READ_ONLY = 0xC00002C6
    STATUS_WMI_SET_FAILURE = 0xC00002C7
    STATUS_COMMITMENT_MINIMUM = 0xC00002C8
    STATUS_REG_NAT_CONSUMPTION = 0xC00002C9
    STATUS_TRANSPORT_FULL = 0xC00002CA
    STATUS_DS_SAM_INIT_FAILURE = 0xC00002CB
    STATUS_ONLY_IF_CONNECTED = 0xC00002CC
    STATUS_DS_SENSITIVE_GROUP_VIOLATION = 0xC00002CD
    STATUS_PNP_RESTART_ENUMERATION = 0xC00002CE
    STATUS_JOURNAL_ENTRY_DELETED = 0xC00002CF
    STATUS_DS_CANT_MOD_PRIMARYGROUPID = 0xC00002D0
    STATUS_SYSTEM_IMAGE_BAD_SIGNATURE = 0xC00002D1
    STATUS_PNP_REBOOT_REQUIRED = 0xC00002D2
    STATUS_POWER_STATE_INVALID = 0xC00002D3
    STATUS_DS_INVALID_GROUP_TYPE = 0xC00002D4
    STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN = 0xC00002D5
    STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN = 0xC00002D6
    STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER = 0xC00002D7
    STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER = 0xC00002D8
    STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER = 0xC00002D9
    STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER = 0xC00002DA
    STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER = 0xC00002DB
    STATUS_DS_HAVE_PRIMARY_MEMBERS = 0xC00002DC
    STATUS_WMI_NOT_SUPPORTED = 0xC00002DD
    STATUS_INSUFFICIENT_POWER = 0xC00002DE
    STATUS_SAM_NEED_BOOTKEY_PASSWORD = 0xC00002DF
    STATUS_SAM_NEED_BOOTKEY_FLOPPY = 0xC00002E0
    STATUS_DS_CANT_START = 0xC00002E1
    STATUS_DS_INIT_FAILURE = 0xC00002E2
    STATUS_SAM_INIT_FAILURE = 0xC00002E3
    STATUS_DS_GC_REQUIRED = 0xC00002E4
    STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY = 0xC00002E5
    STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS = 0xC00002E6
    STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED = 0xC00002E7
    STATUS_CURRENT_DOMAIN_NOT_ALLOWED = 0xC00002E9
    STATUS_CANNOT_MAKE = 0xC00002EA
    STATUS_SYSTEM_SHUTDOWN = 0xC00002EB
    STATUS_DS_INIT_FAILURE_CONSOLE = 0xC00002EC
    STATUS_DS_SAM_INIT_FAILURE_CONSOLE = 0xC00002ED
    STATUS_UNFINISHED_CONTEXT_DELETED = 0xC00002EE
    STATUS_NO_TGT_REPLY = 0xC00002EF
    STATUS_OBJECTID_NOT_FOUND = 0xC00002F0
    STATUS_NO_IP_ADDRESSES = 0xC00002F1
    STATUS_WRONG_CREDENTIAL_HANDLE = 0xC00002F2
    STATUS_CRYPTO_SYSTEM_INVALID = 0xC00002F3
    STATUS_MAX_REFERRALS_EXCEEDED = 0xC00002F4
    STATUS_MUST_BE_KDC = 0xC00002F5
    STATUS_STRONG_CRYPTO_NOT_SUPPORTED = 0xC00002F6
    STATUS_TOO_MANY_PRINCIPALS = 0xC00002F7
    STATUS_NO_PA_DATA = 0xC00002F8
    STATUS_PKINIT_NAME_MISMATCH = 0xC00002F9
    STATUS_SMARTCARD_LOGON_REQUIRED = 0xC00002FA
    STATUS_KDC_INVALID_REQUEST = 0xC00002FB
    STATUS_KDC_UNABLE_TO_REFER = 0xC00002FC
    STATUS_KDC_UNKNOWN_ETYPE = 0xC00002FD
    STATUS_SHUTDOWN_IN_PROGRESS = 0xC00002FE
    STATUS_SERVER_SHUTDOWN_IN_PROGRESS = 0xC00002FF
    STATUS_NOT_SUPPORTED_ON_SBS = 0xC0000300
    STATUS_WMI_GUID_DISCONNECTED = 0xC0000301
    STATUS_WMI_ALREADY_DISABLED = 0xC0000302
    STATUS_WMI_ALREADY_ENABLED = 0xC0000303
    STATUS_MFT_TOO_FRAGMENTED = 0xC0000304
    STATUS_COPY_PROTECTION_FAILURE = 0xC0000305
    STATUS_CSS_AUTHENTICATION_FAILURE = 0xC0000306
    STATUS_CSS_KEY_NOT_PRESENT = 0xC0000307
    STATUS_CSS_KEY_NOT_ESTABLISHED = 0xC0000308
    STATUS_CSS_SCRAMBLED_SECTOR = 0xC0000309
    STATUS_CSS_REGION_MISMATCH = 0xC000030A
    STATUS_CSS_RESETS_EXHAUSTED = 0xC000030B
    STATUS_PKINIT_FAILURE = 0xC0000320
    STATUS_SMARTCARD_SUBSYSTEM_FAILURE = 0xC0000321
    STATUS_NO_KERB_KEY = 0xC0000322
    STATUS_HOST_DOWN = 0xC0000350
    STATUS_UNSUPPORTED_PREAUTH = 0xC0000351
    STATUS_EFS_ALG_BLOB_TOO_BIG = 0xC0000352
    STATUS_PORT_NOT_SET = 0xC0000353
    STATUS_DEBUGGER_INACTIVE = 0xC0000354
    STATUS_DS_VERSION_CHECK_FAILURE = 0xC0000355
    STATUS_AUDITING_DISABLED = 0xC0000356
    STATUS_PRENT4_MACHINE_ACCOUNT = 0xC0000357
    STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER = 0xC0000358
    STATUS_INVALID_IMAGE_WIN_32 = 0xC0000359
    STATUS_INVALID_IMAGE_WIN_64 = 0xC000035A
    STATUS_BAD_BINDINGS = 0xC000035B
    STATUS_NETWORK_SESSION_EXPIRED = 0xC000035C
    STATUS_APPHELP_BLOCK = 0xC000035D
    STATUS_ALL_SIDS_FILTERED = 0xC000035E
    STATUS_NOT_SAFE_MODE_DRIVER = 0xC000035F
    STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT = 0xC0000361
    STATUS_ACCESS_DISABLED_BY_POLICY_PATH = 0xC0000362
    STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER = 0xC0000363
    STATUS_ACCESS_DISABLED_BY_POLICY_OTHER = 0xC0000364
    STATUS_FAILED_DRIVER_ENTRY = 0xC0000365
    STATUS_DEVICE_ENUMERATION_ERROR = 0xC0000366
    STATUS_MOUNT_POINT_NOT_RESOLVED = 0xC0000368
    STATUS_INVALID_DEVICE_OBJECT_PARAMETER = 0xC0000369
    STATUS_MCA_OCCURED = 0xC000036A
    STATUS_DRIVER_BLOCKED_CRITICAL = 0xC000036B
    STATUS_DRIVER_BLOCKED = 0xC000036C
    STATUS_DRIVER_DATABASE_ERROR = 0xC000036D
    STATUS_SYSTEM_HIVE_TOO_LARGE = 0xC000036E
    STATUS_INVALID_IMPORT_OF_NON_DLL = 0xC000036F
    STATUS_NO_SECRETS = 0xC0000371
    STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY = 0xC0000372
    STATUS_FAILED_STACK_SWITCH = 0xC0000373
    STATUS_HEAP_CORRUPTION = 0xC0000374
    STATUS_SMARTCARD_WRONG_PIN = 0xC0000380
    STATUS_SMARTCARD_CARD_BLOCKED = 0xC0000381
    STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED = 0xC0000382
    STATUS_SMARTCARD_NO_CARD = 0xC0000383
    STATUS_SMARTCARD_NO_KEY_CONTAINER = 0xC0000384
    STATUS_SMARTCARD_NO_CERTIFICATE = 0xC0000385
    STATUS_SMARTCARD_NO_KEYSET = 0xC0000386
    STATUS_SMARTCARD_IO_ERROR = 0xC0000387
    STATUS_DOWNGRADE_DETECTED = 0xC0000388
    STATUS_SMARTCARD_CERT_REVOKED = 0xC0000389
    STATUS_ISSUING_CA_UNTRUSTED = 0xC000038A
    STATUS_REVOCATION_OFFLINE_C = 0xC000038B
    STATUS_PKINIT_CLIENT_FAILURE = 0xC000038C
    STATUS_SMARTCARD_CERT_EXPIRED = 0xC000038D
    STATUS_DRIVER_FAILED_PRIOR_UNLOAD = 0xC000038E
    STATUS_SMARTCARD_SILENT_CONTEXT = 0xC000038F
    STATUS_PER_USER_TRUST_QUOTA_EXCEEDED = 0xC0000401
    STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED = 0xC0000402
    STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED = 0xC0000403
    STATUS_DS_NAME_NOT_UNIQUE = 0xC0000404
    STATUS_DS_DUPLICATE_ID_FOUND = 0xC0000405
    STATUS_DS_GROUP_CONVERSION_ERROR = 0xC0000406
    STATUS_VOLSNAP_PREPARE_HIBERNATE = 0xC0000407
    STATUS_USER2USER_REQUIRED = 0xC0000408
    STATUS_STACK_BUFFER_OVERRUN = 0xC0000409
    STATUS_NO_S4U_PROT_SUPPORT = 0xC000040A
    STATUS_CROSSREALM_DELEGATION_FAILURE = 0xC000040B
    STATUS_REVOCATION_OFFLINE_KDC = 0xC000040C
    STATUS_ISSUING_CA_UNTRUSTED_KDC = 0xC000040D
    STATUS_KDC_CERT_EXPIRED = 0xC000040E
    STATUS_KDC_CERT_REVOKED = 0xC000040F
    STATUS_PARAMETER_QUOTA_EXCEEDED = 0xC0000410
    STATUS_HIBERNATION_FAILURE = 0xC0000411
    STATUS_DELAY_LOAD_FAILED = 0xC0000412
    STATUS_AUTHENTICATION_FIREWALL_FAILED = 0xC0000413
    STATUS_VDM_DISALLOWED = 0xC0000414
    STATUS_HUNG_DISPLAY_DRIVER_THREAD = 0xC0000415
    STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE = 0xC0000416
    STATUS_INVALID_CRUNTIME_PARAMETER = 0xC0000417
    STATUS_NTLM_BLOCKED = 0xC0000418
    STATUS_DS_SRC_SID_EXISTS_IN_FOREST = 0xC0000419
    STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST = 0xC000041A
    STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST = 0xC000041B
    STATUS_INVALID_USER_PRINCIPAL_NAME = 0xC000041C
    STATUS_ASSERTION_FAILURE = 0xC0000420
    STATUS_VERIFIER_STOP = 0xC0000421
    STATUS_CALLBACK_POP_STACK = 0xC0000423
    STATUS_INCOMPATIBLE_DRIVER_BLOCKED = 0xC0000424
    STATUS_HIVE_UNLOADED = 0xC0000425
    STATUS_COMPRESSION_DISABLED = 0xC0000426
    STATUS_FILE_SYSTEM_LIMITATION = 0xC0000427
    STATUS_INVALID_IMAGE_HASH = 0xC0000428
    STATUS_NOT_CAPABLE = 0xC0000429
    STATUS_REQUEST_OUT_OF_SEQUENCE = 0xC000042A
    STATUS_IMPLEMENTATION_LIMIT = 0xC000042B
    STATUS_ELEVATION_REQUIRED = 0xC000042C
    STATUS_NO_SECURITY_CONTEXT = 0xC000042D
    STATUS_PKU2U_CERT_FAILURE = 0xC000042E
    STATUS_BEYOND_VDL = 0xC0000432
    STATUS_ENCOUNTERED_WRITE_IN_PROGRESS = 0xC0000433
    STATUS_PTE_CHANGED = 0xC0000434
    STATUS_PURGE_FAILED = 0xC0000435
    STATUS_CRED_REQUIRES_CONFIRMATION = 0xC0000440
    STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE = 0xC0000441
    STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER = 0xC0000442
    STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE = 0xC0000443
    STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE = 0xC0000444
    STATUS_CS_ENCRYPTION_FILE_NOT_CSE = 0xC0000445
    STATUS_INVALID_LABEL = 0xC0000446
    STATUS_DRIVER_PROCESS_TERMINATED = 0xC0000450
    STATUS_AMBIGUOUS_SYSTEM_DEVICE = 0xC0000451
    STATUS_SYSTEM_DEVICE_NOT_FOUND = 0xC0000452
    STATUS_RESTART_BOOT_APPLICATION = 0xC0000453
    STATUS_INSUFFICIENT_NVRAM_RESOURCES = 0xC0000454
    STATUS_NO_RANGES_PROCESSED = 0xC0000460
    STATUS_DEVICE_FEATURE_NOT_SUPPORTED = 0xC0000463
    STATUS_DEVICE_UNREACHABLE = 0xC0000464
    STATUS_INVALID_TOKEN = 0xC0000465
    STATUS_SERVER_UNAVAILABLE = 0xC0000466
    STATUS_INVALID_TASK_NAME = 0xC0000500
    STATUS_INVALID_TASK_INDEX = 0xC0000501
    STATUS_THREAD_ALREADY_IN_TASK = 0xC0000502
    STATUS_CALLBACK_BYPASS = 0xC0000503
    STATUS_FAIL_FAST_EXCEPTION = 0xC0000602
    STATUS_IMAGE_CERT_REVOKED = 0xC0000603
    STATUS_PORT_CLOSED = 0xC0000700
    STATUS_MESSAGE_LOST = 0xC0000701
    STATUS_INVALID_MESSAGE = 0xC0000702
    STATUS_REQUEST_CANCELED = 0xC0000703
    STATUS_RECURSIVE_DISPATCH = 0xC0000704
    STATUS_LPC_RECEIVE_BUFFER_EXPECTED = 0xC0000705
    STATUS_LPC_INVALID_CONNECTION_USAGE = 0xC0000706
    STATUS_LPC_REQUESTS_NOT_ALLOWED = 0xC0000707
    STATUS_RESOURCE_IN_USE = 0xC0000708
    STATUS_HARDWARE_MEMORY_ERROR = 0xC0000709
    STATUS_THREADPOOL_HANDLE_EXCEPTION = 0xC000070A
    STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED = 0xC000070B
    STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED = 0xC000070C
    STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED = 0xC000070D
    STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED = 0xC000070E
    STATUS_THREADPOOL_RELEASED_DURING_OPERATION = 0xC000070F
    STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING = 0xC0000710
    STATUS_APC_RETURNED_WHILE_IMPERSONATING = 0xC0000711
    STATUS_PROCESS_IS_PROTECTED = 0xC0000712
    STATUS_MCA_EXCEPTION = 0xC0000713
    STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE = 0xC0000714
    STATUS_SYMLINK_CLASS_DISABLED = 0xC0000715
    STATUS_INVALID_IDN_NORMALIZATION = 0xC0000716
    STATUS_NO_UNICODE_TRANSLATION = 0xC0000717
    STATUS_ALREADY_REGISTERED = 0xC0000718
    STATUS_CONTEXT_MISMATCH = 0xC0000719
    STATUS_PORT_ALREADY_HAS_COMPLETION_LIST = 0xC000071A
    STATUS_CALLBACK_RETURNED_THREAD_PRIORITY = 0xC000071B
    STATUS_INVALID_THREAD = 0xC000071C
    STATUS_CALLBACK_RETURNED_TRANSACTION = 0xC000071D
    STATUS_CALLBACK_RETURNED_LDR_LOCK = 0xC000071E
    STATUS_CALLBACK_RETURNED_LANG = 0xC000071F
    STATUS_CALLBACK_RETURNED_PRI_BACK = 0xC0000720
    STATUS_DISK_REPAIR_DISABLED = 0xC0000800
    STATUS_DS_DOMAIN_RENAME_IN_PROGRESS = 0xC0000801
    STATUS_DISK_QUOTA_EXCEEDED = 0xC0000802
    STATUS_CONTENT_BLOCKED = 0xC0000804
    STATUS_BAD_CLUSTERS = 0xC0000805
    STATUS_VOLUME_DIRTY = 0xC0000806
    STATUS_FILE_CHECKED_OUT = 0xC0000901
    STATUS_CHECKOUT_REQUIRED = 0xC0000902
    STATUS_BAD_FILE_TYPE = 0xC0000903
    STATUS_FILE_TOO_LARGE = 0xC0000904
    STATUS_FORMS_AUTH_REQUIRED = 0xC0000905
    STATUS_VIRUS_INFECTED = 0xC0000906
    STATUS_VIRUS_DELETED = 0xC0000907
    STATUS_BAD_MCFG_TABLE = 0xC0000908
    STATUS_CANNOT_BREAK_OPLOCK = 0xC0000909
    STATUS_WOW_ASSERTION = 0xC0009898
    STATUS_INVALID_SIGNATURE = 0xC000A000
    STATUS_HMAC_NOT_SUPPORTED = 0xC000A001
    STATUS_IPSEC_QUEUE_OVERFLOW = 0xC000A010
    STATUS_ND_QUEUE_OVERFLOW = 0xC000A011
    STATUS_HOPLIMIT_EXCEEDED = 0xC000A012
    STATUS_PROTOCOL_NOT_SUPPORTED = 0xC000A013
    STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED = 0xC000A080
    STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR = 0xC000A081
    STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR = 0xC000A082
    STATUS_XML_PARSE_ERROR = 0xC000A083
    STATUS_XMLDSIG_ERROR = 0xC000A084
    STATUS_WRONG_COMPARTMENT = 0xC000A085
    STATUS_AUTHIP_FAILURE = 0xC000A086
    STATUS_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS = 0xC000A087
    STATUS_DS_OID_NOT_FOUND = 0xC000A088
    STATUS_HASH_NOT_SUPPORTED = 0xC000A100
    STATUS_HASH_NOT_PRESENT = 0xC000A101
    STATUS_OFFLOAD_READ_FLT_NOT_SUPPORTED = 0xC000A2A1
    STATUS_OFFLOAD_WRITE_FLT_NOT_SUPPORTED = 0xC000A2A2
    STATUS_OFFLOAD_READ_FILE_NOT_SUPPORTED = 0xC000A2A3
    STATUS_OFFLOAD_WRITE_FILE_NOT_SUPPORTED = 0xC000A2A4
    DBG_NO_STATE_CHANGE = 0xC0010001
    DBG_APP_NOT_IDLE = 0xC0010002
    RPC_NT_INVALID_STRING_BINDING = 0xC0020001
    RPC_NT_WRONG_KIND_OF_BINDING = 0xC0020002
    RPC_NT_INVALID_BINDING = 0xC0020003
    RPC_NT_PROTSEQ_NOT_SUPPORTED = 0xC0020004
    RPC_NT_INVALID_RPC_PROTSEQ = 0xC0020005
    RPC_NT_INVALID_STRING_UUID = 0xC0020006
    RPC_NT_INVALID_ENDPOINT_FORMAT = 0xC0020007
    RPC_NT_INVALID_NET_ADDR = 0xC0020008
    RPC_NT_NO_ENDPOINT_FOUND = 0xC0020009
    RPC_NT_INVALID_TIMEOUT = 0xC002000A
    RPC_NT_OBJECT_NOT_FOUND = 0xC002000B
    RPC_NT_ALREADY_REGISTERED = 0xC002000C
    RPC_NT_TYPE_ALREADY_REGISTERED = 0xC002000D
    RPC_NT_ALREADY_LISTENING = 0xC002000E
    RPC_NT_NO_PROTSEQS_REGISTERED = 0xC002000F
    RPC_NT_NOT_LISTENING = 0xC0020010
    RPC_NT_UNKNOWN_MGR_TYPE = 0xC0020011
    RPC_NT_UNKNOWN_IF = 0xC0020012
    RPC_NT_NO_BINDINGS = 0xC0020013
    RPC_NT_NO_PROTSEQS = 0xC0020014
    RPC_NT_CANT_CREATE_ENDPOINT = 0xC0020015
    RPC_NT_OUT_OF_RESOURCES = 0xC0020016
    RPC_NT_SERVER_UNAVAILABLE = 0xC0020017
    RPC_NT_SERVER_TOO_BUSY = 0xC0020018
    RPC_NT_INVALID_NETWORK_OPTIONS = 0xC0020019
    RPC_NT_NO_CALL_ACTIVE = 0xC002001A
    RPC_NT_CALL_FAILED = 0xC002001B
    RPC_NT_CALL_FAILED_DNE = 0xC002001C
    RPC_NT_PROTOCOL_ERROR = 0xC002001D
    RPC_NT_UNSUPPORTED_TRANS_SYN = 0xC002001F
    RPC_NT_UNSUPPORTED_TYPE = 0xC0020021
    RPC_NT_INVALID_TAG = 0xC0020022
    RPC_NT_INVALID_BOUND = 0xC0020023
    RPC_NT_NO_ENTRY_NAME = 0xC0020024
    RPC_NT_INVALID_NAME_SYNTAX = 0xC0020025
    RPC_NT_UNSUPPORTED_NAME_SYNTAX = 0xC0020026
    RPC_NT_UUID_NO_ADDRESS = 0xC0020028
    RPC_NT_DUPLICATE_ENDPOINT = 0xC0020029
    RPC_NT_UNKNOWN_AUTHN_TYPE = 0xC002002A
    RPC_NT_MAX_CALLS_TOO_SMALL = 0xC002002B
    RPC_NT_STRING_TOO_LONG = 0xC002002C
    RPC_NT_PROTSEQ_NOT_FOUND = 0xC002002D
    RPC_NT_PROCNUM_OUT_OF_RANGE = 0xC002002E
    RPC_NT_BINDING_HAS_NO_AUTH = 0xC002002F
    RPC_NT_UNKNOWN_AUTHN_SERVICE = 0xC0020030
    RPC_NT_UNKNOWN_AUTHN_LEVEL = 0xC0020031
    RPC_NT_INVALID_AUTH_IDENTITY = 0xC0020032
    RPC_NT_UNKNOWN_AUTHZ_SERVICE = 0xC0020033
    EPT_NT_INVALID_ENTRY = 0xC0020034
    EPT_NT_CANT_PERFORM_OP = 0xC0020035
    EPT_NT_NOT_REGISTERED = 0xC0020036
    RPC_NT_NOTHING_TO_EXPORT = 0xC0020037
    RPC_NT_INCOMPLETE_NAME = 0xC0020038
    RPC_NT_INVALID_VERS_OPTION = 0xC0020039
    RPC_NT_NO_MORE_MEMBERS = 0xC002003A
    RPC_NT_NOT_ALL_OBJS_UNEXPORTED = 0xC002003B
    RPC_NT_INTERFACE_NOT_FOUND = 0xC002003C
    RPC_NT_ENTRY_ALREADY_EXISTS = 0xC002003D
    RPC_NT_ENTRY_NOT_FOUND = 0xC002003E
    RPC_NT_NAME_SERVICE_UNAVAILABLE = 0xC002003F
    RPC_NT_INVALID_NAF_ID = 0xC0020040
    RPC_NT_CANNOT_SUPPORT = 0xC0020041
    RPC_NT_NO_CONTEXT_AVAILABLE = 0xC0020042
    RPC_NT_INTERNAL_ERROR = 0xC0020043
    RPC_NT_ZERO_DIVIDE = 0xC0020044
    RPC_NT_ADDRESS_ERROR = 0xC0020045
    RPC_NT_FP_DIV_ZERO = 0xC0020046
    RPC_NT_FP_UNDERFLOW = 0xC0020047
    RPC_NT_FP_OVERFLOW = 0xC0020048
    RPC_NT_CALL_IN_PROGRESS = 0xC0020049
    RPC_NT_NO_MORE_BINDINGS = 0xC002004A
    RPC_NT_GROUP_MEMBER_NOT_FOUND = 0xC002004B
    EPT_NT_CANT_CREATE = 0xC002004C
    RPC_NT_INVALID_OBJECT = 0xC002004D
    RPC_NT_NO_INTERFACES = 0xC002004F
    RPC_NT_CALL_CANCELLED = 0xC0020050
    RPC_NT_BINDING_INCOMPLETE = 0xC0020051
    RPC_NT_COMM_FAILURE = 0xC0020052
    RPC_NT_UNSUPPORTED_AUTHN_LEVEL = 0xC0020053
    RPC_NT_NO_PRINC_NAME = 0xC0020054
    RPC_NT_NOT_RPC_ERROR = 0xC0020055
    RPC_NT_SEC_PKG_ERROR = 0xC0020057
    RPC_NT_NOT_CANCELLED = 0xC0020058
    RPC_NT_INVALID_ASYNC_HANDLE = 0xC0020062
    RPC_NT_INVALID_ASYNC_CALL = 0xC0020063
    RPC_NT_PROXY_ACCESS_DENIED = 0xC0020064
    RPC_NT_NO_MORE_ENTRIES = 0xC0030001
    RPC_NT_SS_CHAR_TRANS_OPEN_FAIL = 0xC0030002
    RPC_NT_SS_CHAR_TRANS_SHORT_FILE = 0xC0030003
    RPC_NT_SS_IN_NULL_CONTEXT = 0xC0030004
    RPC_NT_SS_CONTEXT_MISMATCH = 0xC0030005
    RPC_NT_SS_CONTEXT_DAMAGED = 0xC0030006
    RPC_NT_SS_HANDLES_MISMATCH = 0xC0030007
    RPC_NT_SS_CANNOT_GET_CALL_HANDLE = 0xC0030008
    RPC_NT_NULL_REF_POINTER = 0xC0030009
    RPC_NT_ENUM_VALUE_OUT_OF_RANGE = 0xC003000A
    RPC_NT_BYTE_COUNT_TOO_SMALL = 0xC003000B
    RPC_NT_BAD_STUB_DATA = 0xC003000C
    RPC_NT_INVALID_ES_ACTION = 0xC0030059
    RPC_NT_WRONG_ES_VERSION = 0xC003005A
    RPC_NT_WRONG_STUB_VERSION = 0xC003005B
    RPC_NT_INVALID_PIPE_OBJECT = 0xC003005C
    RPC_NT_INVALID_PIPE_OPERATION = 0xC003005D
    RPC_NT_WRONG_PIPE_VERSION = 0xC003005E
    RPC_NT_PIPE_CLOSED = 0xC003005F
    RPC_NT_PIPE_DISCIPLINE_ERROR = 0xC0030060
    RPC_NT_PIPE_EMPTY = 0xC0030061
    STATUS_PNP_BAD_MPS_TABLE = 0xC0040035
    STATUS_PNP_TRANSLATION_FAILED = 0xC0040036
    STATUS_PNP_IRQ_TRANSLATION_FAILED = 0xC0040037
    STATUS_PNP_INVALID_ID = 0xC0040038
    STATUS_IO_REISSUE_AS_CACHED = 0xC0040039
    STATUS_CTX_WINSTATION_NAME_INVALID = 0xC00A0001
    STATUS_CTX_INVALID_PD = 0xC00A0002
    STATUS_CTX_PD_NOT_FOUND = 0xC00A0003
    STATUS_CTX_CLOSE_PENDING = 0xC00A0006
    STATUS_CTX_NO_OUTBUF = 0xC00A0007
    STATUS_CTX_MODEM_INF_NOT_FOUND = 0xC00A0008
    STATUS_CTX_INVALID_MODEMNAME = 0xC00A0009
    STATUS_CTX_RESPONSE_ERROR = 0xC00A000A
    STATUS_CTX_MODEM_RESPONSE_TIMEOUT = 0xC00A000B
    STATUS_CTX_MODEM_RESPONSE_NO_CARRIER = 0xC00A000C
    STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE = 0xC00A000D
    STATUS_CTX_MODEM_RESPONSE_BUSY = 0xC00A000E
    STATUS_CTX_MODEM_RESPONSE_VOICE = 0xC00A000F
    STATUS_CTX_TD_ERROR = 0xC00A0010
    STATUS_CTX_LICENSE_CLIENT_INVALID = 0xC00A0012
    STATUS_CTX_LICENSE_NOT_AVAILABLE = 0xC00A0013
    STATUS_CTX_LICENSE_EXPIRED = 0xC00A0014
    STATUS_CTX_WINSTATION_NOT_FOUND = 0xC00A0015
    STATUS_CTX_WINSTATION_NAME_COLLISION = 0xC00A0016
    STATUS_CTX_WINSTATION_BUSY = 0xC00A0017
    STATUS_CTX_BAD_VIDEO_MODE = 0xC00A0018
    STATUS_CTX_GRAPHICS_INVALID = 0xC00A0022
    STATUS_CTX_NOT_CONSOLE = 0xC00A0024
    STATUS_CTX_CLIENT_QUERY_TIMEOUT = 0xC00A0026
    STATUS_CTX_CONSOLE_DISCONNECT = 0xC00A0027
    STATUS_CTX_CONSOLE_CONNECT = 0xC00A0028
    STATUS_CTX_SHADOW_DENIED = 0xC00A002A
    STATUS_CTX_WINSTATION_ACCESS_DENIED = 0xC00A002B
    STATUS_CTX_INVALID_WD = 0xC00A002E
    STATUS_CTX_WD_NOT_FOUND = 0xC00A002F
    STATUS_CTX_SHADOW_INVALID = 0xC00A0030
    STATUS_CTX_SHADOW_DISABLED = 0xC00A0031
    STATUS_RDP_PROTOCOL_ERROR = 0xC00A0032
    STATUS_CTX_CLIENT_LICENSE_NOT_SET = 0xC00A0033
    STATUS_CTX_CLIENT_LICENSE_IN_USE = 0xC00A0034
    STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE = 0xC00A0035
    STATUS_CTX_SHADOW_NOT_RUNNING = 0xC00A0036
    STATUS_CTX_LOGON_DISABLED = 0xC00A0037
    STATUS_CTX_SECURITY_LAYER_ERROR = 0xC00A0038
    STATUS_TS_INCOMPATIBLE_SESSIONS = 0xC00A0039
    STATUS_MUI_FILE_NOT_FOUND = 0xC00B0001
    STATUS_MUI_INVALID_FILE = 0xC00B0002
    STATUS_MUI_INVALID_RC_CONFIG = 0xC00B0003
    STATUS_MUI_INVALID_LOCALE_NAME = 0xC00B0004
    STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME = 0xC00B0005
    STATUS_MUI_FILE_NOT_LOADED = 0xC00B0006
    STATUS_RESOURCE_ENUM_USER_STOP = 0xC00B0007
    STATUS_CLUSTER_INVALID_NODE = 0xC0130001
    STATUS_CLUSTER_NODE_EXISTS = 0xC0130002
    STATUS_CLUSTER_JOIN_IN_PROGRESS = 0xC0130003
    STATUS_CLUSTER_NODE_NOT_FOUND = 0xC0130004
    STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND = 0xC0130005
    STATUS_CLUSTER_NETWORK_EXISTS = 0xC0130006
    STATUS_CLUSTER_NETWORK_NOT_FOUND = 0xC0130007
    STATUS_CLUSTER_NETINTERFACE_EXISTS = 0xC0130008
    STATUS_CLUSTER_NETINTERFACE_NOT_FOUND = 0xC0130009
    STATUS_CLUSTER_INVALID_REQUEST = 0xC013000A
    STATUS_CLUSTER_INVALID_NETWORK_PROVIDER = 0xC013000B
    STATUS_CLUSTER_NODE_DOWN = 0xC013000C
    STATUS_CLUSTER_NODE_UNREACHABLE = 0xC013000D
    STATUS_CLUSTER_NODE_NOT_MEMBER = 0xC013000E
    STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS = 0xC013000F
    STATUS_CLUSTER_INVALID_NETWORK = 0xC0130010
    STATUS_CLUSTER_NO_NET_ADAPTERS = 0xC0130011
    STATUS_CLUSTER_NODE_UP = 0xC0130012
    STATUS_CLUSTER_NODE_PAUSED = 0xC0130013
    STATUS_CLUSTER_NODE_NOT_PAUSED = 0xC0130014
    STATUS_CLUSTER_NO_SECURITY_CONTEXT = 0xC0130015
    STATUS_CLUSTER_NETWORK_NOT_INTERNAL = 0xC0130016
    STATUS_CLUSTER_POISONED = 0xC0130017
    STATUS_ACPI_INVALID_OPCODE = 0xC0140001
    STATUS_ACPI_STACK_OVERFLOW = 0xC0140002
    STATUS_ACPI_ASSERT_FAILED = 0xC0140003
    STATUS_ACPI_INVALID_INDEX = 0xC0140004
    STATUS_ACPI_INVALID_ARGUMENT = 0xC0140005
    STATUS_ACPI_FATAL = 0xC0140006
    STATUS_ACPI_INVALID_SUPERNAME = 0xC0140007
    STATUS_ACPI_INVALID_ARGTYPE = 0xC0140008
    STATUS_ACPI_INVALID_OBJTYPE = 0xC0140009
    STATUS_ACPI_INVALID_TARGETTYPE = 0xC014000A
    STATUS_ACPI_INCORRECT_ARGUMENT_COUNT = 0xC014000B
    STATUS_ACPI_ADDRESS_NOT_MAPPED = 0xC014000C
    STATUS_ACPI_INVALID_EVENTTYPE = 0xC014000D
    STATUS_ACPI_HANDLER_COLLISION = 0xC014000E
    STATUS_ACPI_INVALID_DATA = 0xC014000F
    STATUS_ACPI_INVALID_REGION = 0xC0140010
    STATUS_ACPI_INVALID_ACCESS_SIZE = 0xC0140011
    STATUS_ACPI_ACQUIRE_GLOBAL_LOCK = 0xC0140012
    STATUS_ACPI_ALREADY_INITIALIZED = 0xC0140013
    STATUS_ACPI_NOT_INITIALIZED = 0xC0140014
    STATUS_ACPI_INVALID_MUTEX_LEVEL = 0xC0140015
    STATUS_ACPI_MUTEX_NOT_OWNED = 0xC0140016
    STATUS_ACPI_MUTEX_NOT_OWNER = 0xC0140017
    STATUS_ACPI_RS_ACCESS = 0xC0140018
    STATUS_ACPI_INVALID_TABLE = 0xC0140019
    STATUS_ACPI_REG_HANDLER_FAILED = 0xC0140020
    STATUS_ACPI_POWER_REQUEST_FAILED = 0xC0140021
    STATUS_SXS_SECTION_NOT_FOUND = 0xC0150001
    STATUS_SXS_CANT_GEN_ACTCTX = 0xC0150002
    STATUS_SXS_INVALID_ACTCTXDATA_FORMAT = 0xC0150003
    STATUS_SXS_ASSEMBLY_NOT_FOUND = 0xC0150004
    STATUS_SXS_MANIFEST_FORMAT_ERROR = 0xC0150005
    STATUS_SXS_MANIFEST_PARSE_ERROR = 0xC0150006
    STATUS_SXS_ACTIVATION_CONTEXT_DISABLED = 0xC0150007
    STATUS_SXS_KEY_NOT_FOUND = 0xC0150008
    STATUS_SXS_VERSION_CONFLICT = 0xC0150009
    STATUS_SXS_WRONG_SECTION_TYPE = 0xC015000A
    STATUS_SXS_THREAD_QUERIES_DISABLED = 0xC015000B
    STATUS_SXS_ASSEMBLY_MISSING = 0xC015000C
    STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET = 0xC015000E
    STATUS_SXS_EARLY_DEACTIVATION = 0xC015000F
    STATUS_SXS_INVALID_DEACTIVATION = 0xC0150010
    STATUS_SXS_MULTIPLE_DEACTIVATION = 0xC0150011
    STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY = 0xC0150012
    STATUS_SXS_PROCESS_TERMINATION_REQUESTED = 0xC0150013
    STATUS_SXS_CORRUPT_ACTIVATION_STACK = 0xC0150014
    STATUS_SXS_CORRUPTION = 0xC0150015
    STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE = 0xC0150016
    STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME = 0xC0150017
    STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE = 0xC0150018
    STATUS_SXS_IDENTITY_PARSE_ERROR = 0xC0150019
    STATUS_SXS_COMPONENT_STORE_CORRUPT = 0xC015001A
    STATUS_SXS_FILE_HASH_MISMATCH = 0xC015001B
    STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT = 0xC015001C
    STATUS_SXS_IDENTITIES_DIFFERENT = 0xC015001D
    STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT = 0xC015001E
    STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY = 0xC015001F
    STATUS_ADVANCED_INSTALLER_FAILED = 0xC0150020
    STATUS_XML_ENCODING_MISMATCH = 0xC0150021
    STATUS_SXS_MANIFEST_TOO_BIG = 0xC0150022
    STATUS_SXS_SETTING_NOT_REGISTERED = 0xC0150023
    STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE = 0xC0150024
    STATUS_SMI_PRIMITIVE_INSTALLER_FAILED = 0xC0150025
    STATUS_GENERIC_COMMAND_FAILED = 0xC0150026
    STATUS_SXS_FILE_HASH_MISSING = 0xC0150027
    STATUS_TRANSACTIONAL_CONFLICT = 0xC0190001
    STATUS_INVALID_TRANSACTION = 0xC0190002
    STATUS_TRANSACTION_NOT_ACTIVE = 0xC0190003
    STATUS_TM_INITIALIZATION_FAILED = 0xC0190004
    STATUS_RM_NOT_ACTIVE = 0xC0190005
    STATUS_RM_METADATA_CORRUPT = 0xC0190006
    STATUS_TRANSACTION_NOT_JOINED = 0xC0190007
    STATUS_DIRECTORY_NOT_RM = 0xC0190008
    STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE = 0xC019000A
    STATUS_LOG_RESIZE_INVALID_SIZE = 0xC019000B
    STATUS_REMOTE_FILE_VERSION_MISMATCH = 0xC019000C
    STATUS_CRM_PROTOCOL_ALREADY_EXISTS = 0xC019000F
    STATUS_TRANSACTION_PROPAGATION_FAILED = 0xC0190010
    STATUS_CRM_PROTOCOL_NOT_FOUND = 0xC0190011
    STATUS_TRANSACTION_SUPERIOR_EXISTS = 0xC0190012
    STATUS_TRANSACTION_REQUEST_NOT_VALID = 0xC0190013
    STATUS_TRANSACTION_NOT_REQUESTED = 0xC0190014
    STATUS_TRANSACTION_ALREADY_ABORTED = 0xC0190015
    STATUS_TRANSACTION_ALREADY_COMMITTED = 0xC0190016
    STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER = 0xC0190017
    STATUS_CURRENT_TRANSACTION_NOT_VALID = 0xC0190018
    STATUS_LOG_GROWTH_FAILED = 0xC0190019
    STATUS_OBJECT_NO_LONGER_EXISTS = 0xC0190021
    STATUS_STREAM_MINIVERSION_NOT_FOUND = 0xC0190022
    STATUS_STREAM_MINIVERSION_NOT_VALID = 0xC0190023
    STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION = 0xC0190024
    STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT = 0xC0190025
    STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS = 0xC0190026
    STATUS_HANDLE_NO_LONGER_VALID = 0xC0190028
    STATUS_LOG_CORRUPTION_DETECTED = 0xC0190030
    STATUS_RM_DISCONNECTED = 0xC0190032
    STATUS_ENLISTMENT_NOT_SUPERIOR = 0xC0190033
    STATUS_FILE_IDENTITY_NOT_PERSISTENT = 0xC0190036
    STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY = 0xC0190037
    STATUS_CANT_CROSS_RM_BOUNDARY = 0xC0190038
    STATUS_TXF_DIR_NOT_EMPTY = 0xC0190039
    STATUS_INDOUBT_TRANSACTIONS_EXIST = 0xC019003A
    STATUS_TM_VOLATILE = 0xC019003B
    STATUS_ROLLBACK_TIMER_EXPIRED = 0xC019003C
    STATUS_TXF_ATTRIBUTE_CORRUPT = 0xC019003D
    STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION = 0xC019003E
    STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED = 0xC019003F
    STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE = 0xC0190040
    STATUS_TRANSACTION_REQUIRED_PROMOTION = 0xC0190043
    STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION = 0xC0190044
    STATUS_TRANSACTIONS_NOT_FROZEN = 0xC0190045
    STATUS_TRANSACTION_FREEZE_IN_PROGRESS = 0xC0190046
    STATUS_NOT_SNAPSHOT_VOLUME = 0xC0190047
    STATUS_NO_SAVEPOINT_WITH_OPEN_FILES = 0xC0190048
    STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION = 0xC0190049
    STATUS_TM_IDENTITY_MISMATCH = 0xC019004A
    STATUS_FLOATED_SECTION = 0xC019004B
    STATUS_CANNOT_ACCEPT_TRANSACTED_WORK = 0xC019004C
    STATUS_CANNOT_ABORT_TRANSACTIONS = 0xC019004D
    STATUS_TRANSACTION_NOT_FOUND = 0xC019004E
    STATUS_RESOURCEMANAGER_NOT_FOUND = 0xC019004F
    STATUS_ENLISTMENT_NOT_FOUND = 0xC0190050
    STATUS_TRANSACTIONMANAGER_NOT_FOUND = 0xC0190051
    STATUS_TRANSACTIONMANAGER_NOT_ONLINE = 0xC0190052
    STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION = 0xC0190053
    STATUS_TRANSACTION_NOT_ROOT = 0xC0190054
    STATUS_TRANSACTION_OBJECT_EXPIRED = 0xC0190055
    STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION = 0xC0190056
    STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED = 0xC0190057
    STATUS_TRANSACTION_RECORD_TOO_LONG = 0xC0190058
    STATUS_NO_LINK_TRACKING_IN_TRANSACTION = 0xC0190059
    STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION = 0xC019005A
    STATUS_TRANSACTION_INTEGRITY_VIOLATED = 0xC019005B
    STATUS_EXPIRED_HANDLE = 0xC0190060
    STATUS_TRANSACTION_NOT_ENLISTED = 0xC0190061
    STATUS_LOG_SECTOR_INVALID = 0xC01A0001
    STATUS_LOG_SECTOR_PARITY_INVALID = 0xC01A0002
    STATUS_LOG_SECTOR_REMAPPED = 0xC01A0003
    STATUS_LOG_BLOCK_INCOMPLETE = 0xC01A0004
    STATUS_LOG_INVALID_RANGE = 0xC01A0005
    STATUS_LOG_BLOCKS_EXHAUSTED = 0xC01A0006
    STATUS_LOG_READ_CONTEXT_INVALID = 0xC01A0007
    STATUS_LOG_RESTART_INVALID = 0xC01A0008
    STATUS_LOG_BLOCK_VERSION = 0xC01A0009
    STATUS_LOG_BLOCK_INVALID = 0xC01A000A
    STATUS_LOG_READ_MODE_INVALID = 0xC01A000B
    STATUS_LOG_METADATA_CORRUPT = 0xC01A000D
    STATUS_LOG_METADATA_INVALID = 0xC01A000E
    STATUS_LOG_METADATA_INCONSISTENT = 0xC01A000F
    STATUS_LOG_RESERVATION_INVALID = 0xC01A0010
    STATUS_LOG_CANT_DELETE = 0xC01A0011
    STATUS_LOG_CONTAINER_LIMIT_EXCEEDED = 0xC01A0012
    STATUS_LOG_START_OF_LOG = 0xC01A0013
    STATUS_LOG_POLICY_ALREADY_INSTALLED = 0xC01A0014
    STATUS_LOG_POLICY_NOT_INSTALLED = 0xC01A0015
    STATUS_LOG_POLICY_INVALID = 0xC01A0016
    STATUS_LOG_POLICY_CONFLICT = 0xC01A0017
    STATUS_LOG_PINNED_ARCHIVE_TAIL = 0xC01A0018
    STATUS_LOG_RECORD_NONEXISTENT = 0xC01A0019
    STATUS_LOG_RECORDS_RESERVED_INVALID = 0xC01A001A
    STATUS_LOG_SPACE_RESERVED_INVALID = 0xC01A001B
    STATUS_LOG_TAIL_INVALID = 0xC01A001C
    STATUS_LOG_FULL = 0xC01A001D
    STATUS_LOG_MULTIPLEXED = 0xC01A001E
    STATUS_LOG_DEDICATED = 0xC01A001F
    STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS = 0xC01A0020
    STATUS_LOG_ARCHIVE_IN_PROGRESS = 0xC01A0021
    STATUS_LOG_EPHEMERAL = 0xC01A0022
    STATUS_LOG_NOT_ENOUGH_CONTAINERS = 0xC01A0023
    STATUS_LOG_CLIENT_ALREADY_REGISTERED = 0xC01A0024
    STATUS_LOG_CLIENT_NOT_REGISTERED = 0xC01A0025
    STATUS_LOG_FULL_HANDLER_IN_PROGRESS = 0xC01A0026
    STATUS_LOG_CONTAINER_READ_FAILED = 0xC01A0027
    STATUS_LOG_CONTAINER_WRITE_FAILED = 0xC01A0028
    STATUS_LOG_CONTAINER_OPEN_FAILED = 0xC01A0029
    STATUS_LOG_CONTAINER_STATE_INVALID = 0xC01A002A
    STATUS_LOG_STATE_INVALID = 0xC01A002B
    STATUS_LOG_PINNED = 0xC01A002C
    STATUS_LOG_METADATA_FLUSH_FAILED = 0xC01A002D
    STATUS_LOG_INCONSISTENT_SECURITY = 0xC01A002E
    STATUS_LOG_APPENDED_FLUSH_FAILED = 0xC01A002F
    STATUS_LOG_PINNED_RESERVATION = 0xC01A0030
    STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD = 0xC01B00EA
    STATUS_FLT_NO_HANDLER_DEFINED = 0xC01C0001
    STATUS_FLT_CONTEXT_ALREADY_DEFINED = 0xC01C0002
    STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST = 0xC01C0003
    STATUS_FLT_DISALLOW_FAST_IO = 0xC01C0004
    STATUS_FLT_INVALID_NAME_REQUEST = 0xC01C0005
    STATUS_FLT_NOT_SAFE_TO_POST_OPERATION = 0xC01C0006
    STATUS_FLT_NOT_INITIALIZED = 0xC01C0007
    STATUS_FLT_FILTER_NOT_READY = 0xC01C0008
    STATUS_FLT_POST_OPERATION_CLEANUP = 0xC01C0009
    STATUS_FLT_INTERNAL_ERROR = 0xC01C000A
    STATUS_FLT_DELETING_OBJECT = 0xC01C000B
    STATUS_FLT_MUST_BE_NONPAGED_POOL = 0xC01C000C
    STATUS_FLT_DUPLICATE_ENTRY = 0xC01C000D
    STATUS_FLT_CBDQ_DISABLED = 0xC01C000E
    STATUS_FLT_DO_NOT_ATTACH = 0xC01C000F
    STATUS_FLT_DO_NOT_DETACH = 0xC01C0010
    STATUS_FLT_INSTANCE_ALTITUDE_COLLISION = 0xC01C0011
    STATUS_FLT_INSTANCE_NAME_COLLISION = 0xC01C0012
    STATUS_FLT_FILTER_NOT_FOUND = 0xC01C0013
    STATUS_FLT_VOLUME_NOT_FOUND = 0xC01C0014
    STATUS_FLT_INSTANCE_NOT_FOUND = 0xC01C0015
    STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND = 0xC01C0016
    STATUS_FLT_INVALID_CONTEXT_REGISTRATION = 0xC01C0017
    STATUS_FLT_NAME_CACHE_MISS = 0xC01C0018
    STATUS_FLT_NO_DEVICE_OBJECT = 0xC01C0019
    STATUS_FLT_VOLUME_ALREADY_MOUNTED = 0xC01C001A
    STATUS_FLT_ALREADY_ENLISTED = 0xC01C001B
    STATUS_FLT_CONTEXT_ALREADY_LINKED = 0xC01C001C
    STATUS_FLT_NO_WAITER_FOR_REPLY = 0xC01C0020
    STATUS_MONITOR_NO_DESCRIPTOR = 0xC01D0001
    STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT = 0xC01D0002
    STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM = 0xC01D0003
    STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK = 0xC01D0004
    STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED = 0xC01D0005
    STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK = 0xC01D0006
    STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK = 0xC01D0007
    STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA = 0xC01D0008
    STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK = 0xC01D0009
    STATUS_MONITOR_INVALID_MANUFACTURE_DATE = 0xC01D000A
    STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER = 0xC01E0000
    STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER = 0xC01E0001
    STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER = 0xC01E0002
    STATUS_GRAPHICS_ADAPTER_WAS_RESET = 0xC01E0003
    STATUS_GRAPHICS_INVALID_DRIVER_MODEL = 0xC01E0004
    STATUS_GRAPHICS_PRESENT_MODE_CHANGED = 0xC01E0005
    STATUS_GRAPHICS_PRESENT_OCCLUDED = 0xC01E0006
    STATUS_GRAPHICS_PRESENT_DENIED = 0xC01E0007
    STATUS_GRAPHICS_CANNOTCOLORCONVERT = 0xC01E0008
    STATUS_GRAPHICS_PRESENT_REDIRECTION_DISABLED = 0xC01E000B
    STATUS_GRAPHICS_PRESENT_UNOCCLUDED = 0xC01E000C
    STATUS_GRAPHICS_NO_VIDEO_MEMORY = 0xC01E0100
    STATUS_GRAPHICS_CANT_LOCK_MEMORY = 0xC01E0101
    STATUS_GRAPHICS_ALLOCATION_BUSY = 0xC01E0102
    STATUS_GRAPHICS_TOO_MANY_REFERENCES = 0xC01E0103
    STATUS_GRAPHICS_TRY_AGAIN_LATER = 0xC01E0104
    STATUS_GRAPHICS_TRY_AGAIN_NOW = 0xC01E0105
    STATUS_GRAPHICS_ALLOCATION_INVALID = 0xC01E0106
    STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE = 0xC01E0107
    STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED = 0xC01E0108
    STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION = 0xC01E0109
    STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE = 0xC01E0110
    STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION = 0xC01E0111
    STATUS_GRAPHICS_ALLOCATION_CLOSED = 0xC01E0112
    STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE = 0xC01E0113
    STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE = 0xC01E0114
    STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE = 0xC01E0115
    STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST = 0xC01E0116
    STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE = 0xC01E0200
    STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY = 0xC01E0300
    STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED = 0xC01E0301
    STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED = 0xC01E0302
    STATUS_GRAPHICS_INVALID_VIDPN = 0xC01E0303
    STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE = 0xC01E0304
    STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET = 0xC01E0305
    STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED = 0xC01E0306
    STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET = 0xC01E0308
    STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET = 0xC01E0309
    STATUS_GRAPHICS_INVALID_FREQUENCY = 0xC01E030A
    STATUS_GRAPHICS_INVALID_ACTIVE_REGION = 0xC01E030B
    STATUS_GRAPHICS_INVALID_TOTAL_REGION = 0xC01E030C
    STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE = 0xC01E0310
    STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE = 0xC01E0311
    STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET = 0xC01E0312
    STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY = 0xC01E0313
    STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET = 0xC01E0314
    STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET = 0xC01E0315
    STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET = 0xC01E0316
    STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET = 0xC01E0317
    STATUS_GRAPHICS_TARGET_ALREADY_IN_SET = 0xC01E0318
    STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH = 0xC01E0319
    STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY = 0xC01E031A
    STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET = 0xC01E031B
    STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE = 0xC01E031C
    STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET = 0xC01E031D
    STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET = 0xC01E031F
    STATUS_GRAPHICS_STALE_MODESET = 0xC01E0320
    STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET = 0xC01E0321
    STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE = 0xC01E0322
    STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN = 0xC01E0323
    STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE = 0xC01E0324
    STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION = 0xC01E0325
    STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES = 0xC01E0326
    STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY = 0xC01E0327
    STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE = 0xC01E0328
    STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET = 0xC01E0329
    STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET = 0xC01E032A
    STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR = 0xC01E032B
    STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET = 0xC01E032C
    STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET = 0xC01E032D
    STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE = 0xC01E032E
    STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE = 0xC01E032F
    STATUS_GRAPHICS_RESOURCES_NOT_RELATED = 0xC01E0330
    STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE = 0xC01E0331
    STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE = 0xC01E0332
    STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET = 0xC01E0333
    STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER = 0xC01E0334
    STATUS_GRAPHICS_NO_VIDPNMGR = 0xC01E0335
    STATUS_GRAPHICS_NO_ACTIVE_VIDPN = 0xC01E0336
    STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY = 0xC01E0337
    STATUS_GRAPHICS_MONITOR_NOT_CONNECTED = 0xC01E0338
    STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY = 0xC01E0339
    STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE = 0xC01E033A
    STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE = 0xC01E033B
    STATUS_GRAPHICS_INVALID_STRIDE = 0xC01E033C
    STATUS_GRAPHICS_INVALID_PIXELFORMAT = 0xC01E033D
    STATUS_GRAPHICS_INVALID_COLORBASIS = 0xC01E033E
    STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE = 0xC01E033F
    STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY = 0xC01E0340
    STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT = 0xC01E0341
    STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE = 0xC01E0342
    STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN = 0xC01E0343
    STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL = 0xC01E0344
    STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION = 0xC01E0345
    STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED = 0xC01E0346
    STATUS_GRAPHICS_INVALID_GAMMA_RAMP = 0xC01E0347
    STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED = 0xC01E0348
    STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED = 0xC01E0349
    STATUS_GRAPHICS_MODE_NOT_IN_MODESET = 0xC01E034A
    STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON = 0xC01E034D
    STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE = 0xC01E034E
    STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE = 0xC01E034F
    STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS = 0xC01E0350
    STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING = 0xC01E0352
    STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED = 0xC01E0353
    STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS = 0xC01E0354
    STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT = 0xC01E0355
    STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM = 0xC01E0356
    STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN = 0xC01E0357
    STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT = 0xC01E0358
    STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED = 0xC01E0359
    STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION = 0xC01E035A
    STATUS_GRAPHICS_INVALID_CLIENT_TYPE = 0xC01E035B
    STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET = 0xC01E035C
    STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED = 0xC01E0400
    STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED = 0xC01E0401
    STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER = 0xC01E0430
    STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED = 0xC01E0431
    STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED = 0xC01E0432
    STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY = 0xC01E0433
    STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED = 0xC01E0434
    STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON = 0xC01E0435
    STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE = 0xC01E0436
    STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER = 0xC01E0438
    STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED = 0xC01E043B
    STATUS_GRAPHICS_OPM_NOT_SUPPORTED = 0xC01E0500
    STATUS_GRAPHICS_COPP_NOT_SUPPORTED = 0xC01E0501
    STATUS_GRAPHICS_UAB_NOT_SUPPORTED = 0xC01E0502
    STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS = 0xC01E0503
    STATUS_GRAPHICS_OPM_PARAMETER_ARRAY_TOO_SMALL = 0xC01E0504
    STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST = 0xC01E0505
    STATUS_GRAPHICS_PVP_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME = 0xC01E0506
    STATUS_GRAPHICS_PVP_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP = 0xC01E0507
    STATUS_GRAPHICS_PVP_MIRRORING_DEVICES_NOT_SUPPORTED = 0xC01E0508
    STATUS_GRAPHICS_OPM_INVALID_POINTER = 0xC01E050A
    STATUS_GRAPHICS_OPM_INTERNAL_ERROR = 0xC01E050B
    STATUS_GRAPHICS_OPM_INVALID_HANDLE = 0xC01E050C
    STATUS_GRAPHICS_PVP_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE = 0xC01E050D
    STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH = 0xC01E050E
    STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED = 0xC01E050F
    STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED = 0xC01E0510
    STATUS_GRAPHICS_PVP_HFS_FAILED = 0xC01E0511
    STATUS_GRAPHICS_OPM_INVALID_SRM = 0xC01E0512
    STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP = 0xC01E0513
    STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP = 0xC01E0514
    STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA = 0xC01E0515
    STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET = 0xC01E0516
    STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH = 0xC01E0517
    STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE = 0xC01E0518
    STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS = 0xC01E051A
    STATUS_GRAPHICS_OPM_SESSION_TYPE_CHANGE_IN_PROGRESS = 0xC01E051B
    STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS = 0xC01E051C
    STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST = 0xC01E051D
    STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR = 0xC01E051E
    STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS = 0xC01E051F
    STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED = 0xC01E0520
    STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST = 0xC01E0521
    STATUS_GRAPHICS_I2C_NOT_SUPPORTED = 0xC01E0580
    STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST = 0xC01E0581
    STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA = 0xC01E0582
    STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA = 0xC01E0583
    STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED = 0xC01E0584
    STATUS_GRAPHICS_DDCCI_INVALID_DATA = 0xC01E0585
    STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE = 0xC01E0586
    STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING = 0xC01E0587
    STATUS_GRAPHICS_MCA_INTERNAL_ERROR = 0xC01E0588
    STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND = 0xC01E0589
    STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH = 0xC01E058A
    STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM = 0xC01E058B
    STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE = 0xC01E058C
    STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS = 0xC01E058D
    STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED = 0xC01E05E0
    STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME = 0xC01E05E1
    STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP = 0xC01E05E2
    STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED = 0xC01E05E3
    STATUS_GRAPHICS_INVALID_POINTER = 0xC01E05E4
    STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE = 0xC01E05E5
    STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL = 0xC01E05E6
    STATUS_GRAPHICS_INTERNAL_ERROR = 0xC01E05E7
    STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS = 0xC01E05E8
    STATUS_FVE_LOCKED_VOLUME = 0xC0210000
    STATUS_FVE_NOT_ENCRYPTED = 0xC0210001
    STATUS_FVE_BAD_INFORMATION = 0xC0210002
    STATUS_FVE_TOO_SMALL = 0xC0210003
    STATUS_FVE_FAILED_WRONG_FS = 0xC0210004
    STATUS_FVE_FAILED_BAD_FS = 0xC0210005
    STATUS_FVE_FS_NOT_EXTENDED = 0xC0210006
    STATUS_FVE_FS_MOUNTED = 0xC0210007
    STATUS_FVE_NO_LICENSE = 0xC0210008
    STATUS_FVE_ACTION_NOT_ALLOWED = 0xC0210009
    STATUS_FVE_BAD_DATA = 0xC021000A
    STATUS_FVE_VOLUME_NOT_BOUND = 0xC021000B
    STATUS_FVE_NOT_DATA_VOLUME = 0xC021000C
    STATUS_FVE_CONV_READ_ERROR = 0xC021000D
    STATUS_FVE_CONV_WRITE_ERROR = 0xC021000E
    STATUS_FVE_OVERLAPPED_UPDATE = 0xC021000F
    STATUS_FVE_FAILED_SECTOR_SIZE = 0xC0210010
    STATUS_FVE_FAILED_AUTHENTICATION = 0xC0210011
    STATUS_FVE_NOT_OS_VOLUME = 0xC0210012
    STATUS_FVE_KEYFILE_NOT_FOUND = 0xC0210013
    STATUS_FVE_KEYFILE_INVALID = 0xC0210014
    STATUS_FVE_KEYFILE_NO_VMK = 0xC0210015
    STATUS_FVE_TPM_DISABLED = 0xC0210016
    STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO = 0xC0210017
    STATUS_FVE_TPM_INVALID_PCR = 0xC0210018
    STATUS_FVE_TPM_NO_VMK = 0xC0210019
    STATUS_FVE_PIN_INVALID = 0xC021001A
    STATUS_FVE_AUTH_INVALID_APPLICATION = 0xC021001B
    STATUS_FVE_AUTH_INVALID_CONFIG = 0xC021001C
    STATUS_FVE_DEBUGGER_ENABLED = 0xC021001D
    STATUS_FVE_DRY_RUN_FAILED = 0xC021001E
    STATUS_FVE_BAD_METADATA_POINTER = 0xC021001F
    STATUS_FVE_OLD_METADATA_COPY = 0xC0210020
    STATUS_FVE_REBOOT_REQUIRED = 0xC0210021
    STATUS_FVE_RAW_ACCESS = 0xC0210022
    STATUS_FVE_RAW_BLOCKED = 0xC0210023
    STATUS_FVE_NO_FEATURE_LICENSE = 0xC0210026
    STATUS_FVE_POLICY_USER_DISABLE_RDV_NOT_ALLOWED = 0xC0210027
    STATUS_FVE_CONV_RECOVERY_FAILED = 0xC0210028
    STATUS_FVE_VIRTUALIZED_SPACE_TOO_BIG = 0xC0210029
    STATUS_FVE_VOLUME_TOO_SMALL = 0xC0210030
    STATUS_FWP_CALLOUT_NOT_FOUND = 0xC0220001
    STATUS_FWP_CONDITION_NOT_FOUND = 0xC0220002
    STATUS_FWP_FILTER_NOT_FOUND = 0xC0220003
    STATUS_FWP_LAYER_NOT_FOUND = 0xC0220004
    STATUS_FWP_PROVIDER_NOT_FOUND = 0xC0220005
    STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND = 0xC0220006
    STATUS_FWP_SUBLAYER_NOT_FOUND = 0xC0220007
    STATUS_FWP_NOT_FOUND = 0xC0220008
    STATUS_FWP_ALREADY_EXISTS = 0xC0220009
    STATUS_FWP_IN_USE = 0xC022000A
    STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS = 0xC022000B
    STATUS_FWP_WRONG_SESSION = 0xC022000C
    STATUS_FWP_NO_TXN_IN_PROGRESS = 0xC022000D
    STATUS_FWP_TXN_IN_PROGRESS = 0xC022000E
    STATUS_FWP_TXN_ABORTED = 0xC022000F
    STATUS_FWP_SESSION_ABORTED = 0xC0220010
    STATUS_FWP_INCOMPATIBLE_TXN = 0xC0220011
    STATUS_FWP_TIMEOUT = 0xC0220012
    STATUS_FWP_NET_EVENTS_DISABLED = 0xC0220013
    STATUS_FWP_INCOMPATIBLE_LAYER = 0xC0220014
    STATUS_FWP_KM_CLIENTS_ONLY = 0xC0220015
    STATUS_FWP_LIFETIME_MISMATCH = 0xC0220016
    STATUS_FWP_BUILTIN_OBJECT = 0xC0220017
    STATUS_FWP_TOO_MANY_BOOTTIME_FILTERS = 0xC0220018
    STATUS_FWP_TOO_MANY_CALLOUTS = 0xC0220018
    STATUS_FWP_NOTIFICATION_DROPPED = 0xC0220019
    STATUS_FWP_TRAFFIC_MISMATCH = 0xC022001A
    STATUS_FWP_INCOMPATIBLE_SA_STATE = 0xC022001B
    STATUS_FWP_NULL_POINTER = 0xC022001C
    STATUS_FWP_INVALID_ENUMERATOR = 0xC022001D
    STATUS_FWP_INVALID_FLAGS = 0xC022001E
    STATUS_FWP_INVALID_NET_MASK = 0xC022001F
    STATUS_FWP_INVALID_RANGE = 0xC0220020
    STATUS_FWP_INVALID_INTERVAL = 0xC0220021
    STATUS_FWP_ZERO_LENGTH_ARRAY = 0xC0220022
    STATUS_FWP_NULL_DISPLAY_NAME = 0xC0220023
    STATUS_FWP_INVALID_ACTION_TYPE = 0xC0220024
    STATUS_FWP_INVALID_WEIGHT = 0xC0220025
    STATUS_FWP_MATCH_TYPE_MISMATCH = 0xC0220026
    STATUS_FWP_TYPE_MISMATCH = 0xC0220027
    STATUS_FWP_OUT_OF_BOUNDS = 0xC0220028
    STATUS_FWP_RESERVED = 0xC0220029
    STATUS_FWP_DUPLICATE_CONDITION = 0xC022002A
    STATUS_FWP_DUPLICATE_KEYMOD = 0xC022002B
    STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER = 0xC022002C
    STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER = 0xC022002D
    STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER = 0xC022002E
    STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT = 0xC022002F
    STATUS_FWP_INCOMPATIBLE_AUTH_METHOD = 0xC0220030
    STATUS_FWP_INCOMPATIBLE_DH_GROUP = 0xC0220031
    STATUS_FWP_EM_NOT_SUPPORTED = 0xC0220032
    STATUS_FWP_NEVER_MATCH = 0xC0220033
    STATUS_FWP_PROVIDER_CONTEXT_MISMATCH = 0xC0220034
    STATUS_FWP_INVALID_PARAMETER = 0xC0220035
    STATUS_FWP_TOO_MANY_SUBLAYERS = 0xC0220036
    STATUS_FWP_CALLOUT_NOTIFICATION_FAILED = 0xC0220037
    STATUS_FWP_INCOMPATIBLE_AUTH_CONFIG = 0xC0220038
    STATUS_FWP_INCOMPATIBLE_CIPHER_CONFIG = 0xC0220039
    STATUS_FWP_DUPLICATE_AUTH_METHOD = 0xC022003C
    STATUS_FWP_TCPIP_NOT_READY = 0xC0220100
    STATUS_FWP_INJECT_HANDLE_CLOSING = 0xC0220101
    STATUS_FWP_INJECT_HANDLE_STALE = 0xC0220102
    STATUS_FWP_CANNOT_PEND = 0xC0220103
    STATUS_NDIS_CLOSING = 0xC0230002
    STATUS_NDIS_BAD_VERSION = 0xC0230004
    STATUS_NDIS_BAD_CHARACTERISTICS = 0xC0230005
    STATUS_NDIS_ADAPTER_NOT_FOUND = 0xC0230006
    STATUS_NDIS_OPEN_FAILED = 0xC0230007
    STATUS_NDIS_DEVICE_FAILED = 0xC0230008
    STATUS_NDIS_MULTICAST_FULL = 0xC0230009
    STATUS_NDIS_MULTICAST_EXISTS = 0xC023000A
    STATUS_NDIS_MULTICAST_NOT_FOUND = 0xC023000B
    STATUS_NDIS_REQUEST_ABORTED = 0xC023000C
    STATUS_NDIS_RESET_IN_PROGRESS = 0xC023000D
    STATUS_NDIS_INVALID_PACKET = 0xC023000F
    STATUS_NDIS_INVALID_DEVICE_REQUEST = 0xC0230010
    STATUS_NDIS_ADAPTER_NOT_READY = 0xC0230011
    STATUS_NDIS_INVALID_LENGTH = 0xC0230014
    STATUS_NDIS_INVALID_DATA = 0xC0230015
    STATUS_NDIS_BUFFER_TOO_SHORT = 0xC0230016
    STATUS_NDIS_INVALID_OID = 0xC0230017
    STATUS_NDIS_ADAPTER_REMOVED = 0xC0230018
    STATUS_NDIS_UNSUPPORTED_MEDIA = 0xC0230019
    STATUS_NDIS_GROUP_ADDRESS_IN_USE = 0xC023001A
    STATUS_NDIS_FILE_NOT_FOUND = 0xC023001B
    STATUS_NDIS_ERROR_READING_FILE = 0xC023001C
    STATUS_NDIS_ALREADY_MAPPED = 0xC023001D
    STATUS_NDIS_RESOURCE_CONFLICT = 0xC023001E
    STATUS_NDIS_MEDIA_DISCONNECTED = 0xC023001F
    STATUS_NDIS_INVALID_ADDRESS = 0xC0230022
    STATUS_NDIS_PAUSED = 0xC023002A
    STATUS_NDIS_INTERFACE_NOT_FOUND = 0xC023002B
    STATUS_NDIS_UNSUPPORTED_REVISION = 0xC023002C
    STATUS_NDIS_INVALID_PORT = 0xC023002D
    STATUS_NDIS_INVALID_PORT_STATE = 0xC023002E
    STATUS_NDIS_LOW_POWER_STATE = 0xC023002F
    STATUS_NDIS_NOT_SUPPORTED = 0xC02300BB
    STATUS_NDIS_OFFLOAD_POLICY = 0xC023100F
    STATUS_NDIS_OFFLOAD_CONNECTION_REJECTED = 0xC0231012
    STATUS_NDIS_OFFLOAD_PATH_REJECTED = 0xC0231013
    STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED = 0xC0232000
    STATUS_NDIS_DOT11_MEDIA_IN_USE = 0xC0232001
    STATUS_NDIS_DOT11_POWER_STATE_INVALID = 0xC0232002
    STATUS_NDIS_PM_WOL_PATTERN_LIST_FULL = 0xC0232003
    STATUS_NDIS_PM_PROTOCOL_OFFLOAD_LIST_FULL = 0xC0232004
    STATUS_IPSEC_BAD_SPI = 0xC0360001
    STATUS_IPSEC_SA_LIFETIME_EXPIRED = 0xC0360002
    STATUS_IPSEC_WRONG_SA = 0xC0360003
    STATUS_IPSEC_REPLAY_CHECK_FAILED = 0xC0360004
    STATUS_IPSEC_INVALID_PACKET = 0xC0360005
    STATUS_IPSEC_INTEGRITY_CHECK_FAILED = 0xC0360006
    STATUS_IPSEC_CLEAR_TEXT_DROP = 0xC0360007
    STATUS_IPSEC_AUTH_FIREWALL_DROP = 0xC0360008
    STATUS_IPSEC_THROTTLE_DROP = 0xC0360009
    STATUS_IPSEC_DOSP_BLOCK = 0xC0368000
    STATUS_IPSEC_DOSP_RECEIVED_MULTICAST = 0xC0368001
    STATUS_IPSEC_DOSP_INVALID_PACKET = 0xC0368002
    STATUS_IPSEC_DOSP_STATE_LOOKUP_FAILED = 0xC0368003
    STATUS_IPSEC_DOSP_MAX_ENTRIES = 0xC0368004
    STATUS_IPSEC_DOSP_KEYMOD_NOT_ALLOWED = 0xC0368005
    STATUS_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES = 0xC0368006
    STATUS_VOLMGR_MIRROR_NOT_SUPPORTED = 0xC038005B
    STATUS_VOLMGR_RAID5_NOT_SUPPORTED = 0xC038005C
    STATUS_VIRTDISK_PROVIDER_NOT_FOUND = 0xC03A0014
    STATUS_VIRTDISK_NOT_VIRTUAL_DISK = 0xC03A0015
    STATUS_VHD_PARENT_VHD_ACCESS_DENIED = 0xC03A0016
    STATUS_VHD_CHILD_PARENT_SIZE_MISMATCH = 0xC03A0017
    STATUS_VHD_DIFFERENCING_CHAIN_CYCLE_DETECTED = 0xC03A0018
    STATUS_VHD_DIFFERENCING_CHAIN_ERROR_IN_PARENT = 0xC03A0019


# Taken from Winfsp/scr/dll/fuse/errno.i
_POSIX_NTSTATUS_CONVERSIONS = (
    (0, NTSTATUS.STATUS_SUCCESS),
    (1, NTSTATUS.STATUS_ACCESS_DENIED),
    (2, NTSTATUS.STATUS_OBJECT_NAME_NOT_FOUND),
    (3, NTSTATUS.STATUS_PROCEDURE_NOT_FOUND),
    (4, NTSTATUS.STATUS_CANCELLED),
    (5, NTSTATUS.STATUS_IO_DEVICE_ERROR),
    (6, NTSTATUS.STATUS_FILE_INVALID),
    (7, NTSTATUS.STATUS_INSUFFICIENT_RESOURCES),
    (8, NTSTATUS.STATUS_INVALID_IMAGE_FORMAT),
    (9, NTSTATUS.STATUS_INVALID_HANDLE),
    (12, NTSTATUS.STATUS_INSUFFICIENT_RESOURCES),
    (13, NTSTATUS.STATUS_ACCESS_DENIED),
    (14, NTSTATUS.STATUS_ACCESS_VIOLATION),
    (16, NTSTATUS.STATUS_DEVICE_BUSY),
    (17, NTSTATUS.STATUS_OBJECT_NAME_COLLISION),
    (18, NTSTATUS.STATUS_NOT_SAME_DEVICE),
    (19, NTSTATUS.STATUS_NO_SUCH_DEVICE),
    (20, NTSTATUS.STATUS_NOT_A_DIRECTORY),
    (21, NTSTATUS.STATUS_FILE_IS_A_DIRECTORY),
    (22, NTSTATUS.STATUS_INVALID_PARAMETER),
    (23, NTSTATUS.STATUS_TOO_MANY_OPENED_FILES),
    (24, NTSTATUS.STATUS_TOO_MANY_OPENED_FILES),
    (27, NTSTATUS.STATUS_DISK_FULL),
    (28, NTSTATUS.STATUS_DISK_FULL),
    (29, NTSTATUS.STATUS_INVALID_PARAMETER),
    (30, NTSTATUS.STATUS_MEDIA_WRITE_PROTECTED),
    (31, NTSTATUS.STATUS_TOO_MANY_LINKS),
    (32, NTSTATUS.STATUS_PIPE_BROKEN),
    (33, NTSTATUS.STATUS_INVALID_PARAMETER),
    (34, NTSTATUS.STATUS_INVALID_PARAMETER),
    (36, NTSTATUS.STATUS_POSSIBLE_DEADLOCK),
    (38, NTSTATUS.STATUS_NAME_TOO_LONG),
    (39, NTSTATUS.STATUS_LOCK_NOT_GRANTED),
    (40, NTSTATUS.STATUS_INVALID_DEVICE_REQUEST),
    (41, NTSTATUS.STATUS_DIRECTORY_NOT_EMPTY),
    (42, NTSTATUS.STATUS_INVALID_PARAMETER),
    (100, NTSTATUS.STATUS_ADDRESS_ALREADY_ASSOCIATED),
    (103, NTSTATUS.STATUS_CONNECTION_ACTIVE),
    (105, NTSTATUS.STATUS_CANCELLED),
    (106, NTSTATUS.STATUS_CONNECTION_ABORTED),
    (107, NTSTATUS.STATUS_CONNECTION_REFUSED),
    (108, NTSTATUS.STATUS_CONNECTION_RESET),
    (110, NTSTATUS.STATUS_HOST_UNREACHABLE),
    (113, NTSTATUS.STATUS_CONNECTION_ACTIVE),
    (114, NTSTATUS.STATUS_REPARSE_POINT_NOT_RESOLVED),
    (116, NTSTATUS.STATUS_HOST_DOWN),
    (117, NTSTATUS.STATUS_CONNECTION_RESET),
    (118, NTSTATUS.STATUS_NETWORK_UNREACHABLE),
    (119, NTSTATUS.STATUS_INSUFFICIENT_RESOURCES),
    (120, NTSTATUS.STATUS_END_OF_FILE),
    (121, NTSTATUS.STATUS_CONNECTION_INVALID),
    (126, NTSTATUS.STATUS_CONNECTION_INVALID),
    (128, NTSTATUS.STATUS_INVALID_HANDLE),
    (138, NTSTATUS.STATUS_TRANSACTION_TIMED_OUT),
)
_POSIX_TO_NT_LOOKUP = {p: nt for p, nt in _POSIX_NTSTATUS_CONVERSIONS}
_NT_TO_POSIX_LOOKUP = {nt: p for p, nt in _POSIX_NTSTATUS_CONVERSIONS}


def posix_to_ntstatus(code):
    return _POSIX_TO_NT_LOOKUP[code]


def ntstatus_to_posix(code):
    return _NT_TO_POSIX_LOOKUP[code]
