"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
function deployStack(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
}
test('check lambda function properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackLambdaToDynamoDBLambdaFunctionServiceRole31915E05",
                "Arn"
            ]
        },
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                DDB_TABLE_NAME: {
                    Ref: "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733"
                }
            }
        }
    });
});
test('check lambda function permission', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackLambdaToDynamoDBLambdaFunction5165A7EE",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaIotTopic74F5E3BB",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Lambda: {
                        FunctionArn: {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackLambdaToDynamoDBLambdaFunction5165A7EE",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'connectedcar/dtc/#'"
        }
    });
});
test('check dynamo table properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('check lambda function policy ', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem",
                        "dynamodb:DescribeTable"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const props = {
        deployVpc: true,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    const construct = new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.dynamoTable).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.vpc).toBeDefined();
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Check incorrect table permission', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        },
        tablePermissions: 'Reed'
    };
    const app = () => {
        new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
    };
    // Assertion
    expect(app).toThrowError(/Invalid table permission submitted - Reed/);
});
test('check lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        tableEnvironmentVariableName: 'CUSTOM_DYNAMODB_TABLE',
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToLambdaToDynamoDB(stack, 'test-lambda-dynamodb-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DYNAMODB_TABLE: {
                    Ref: 'testlambdadynamodbstackLambdaToDynamoDBDynamoTable7E730A23'
                }
            }
        }
    });
});
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.IotToLambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackLambdaToDynamoDBReplaceDefaultSecurityGroupsecuritygroup04A024BF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatodynamodbstackVpcisolatedSubnet1Subnet90CC3593",
                },
                {
                    Ref: "lambdatodynamodbstackVpcisolatedSubnet2Subnet4693DAE3",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.IotToLambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackLambdaToDynamoDBReplaceDefaultSecurityGroupsecuritygroup04A024BF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatodynamodbstackVpcisolatedSubnet1Subnet90CC3593",
                },
                {
                    Ref: "lambdatodynamodbstackVpcisolatedSubnet2Subnet4693DAE3",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new cdk.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.IotToLambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: testVpc,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatodynamodbstackLambdaToDynamoDBReplaceDefaultSecurityGroupsecuritygroup04A024BF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new cdk.Stack();
    // buildLambdaFunction should throw an error if the Lambda function is not
    // attached to a VPC
    const testLambdaFunction = new lambda.Function(stack, 'test-lambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.IotToLambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
            iotTopicRuleProps: {
                topicRulePayload: {
                    ruleDisabled: false,
                    description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                    sql: "SELECT * FROM 'connectedcar/dtc/#'",
                    actions: []
                }
            }
        });
    };
    // Assertion
    expect(app).toThrowError();
});
test("Confirm CheckVpcProps is called", () => {
    // Stack
    const stack = new cdk.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.IotToLambdaToDynamoDB(stack, "lambda-to-dynamodb-stack", {
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            existingVpc: testVpc,
            deployVpc: true,
            iotTopicRuleProps: {
                topicRulePayload: {
                    ruleDisabled: false,
                    description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                    sql: "SELECT * FROM 'connectedcar/dtc/#'",
                    actions: []
                }
            }
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm CheckLambdaProps is being called', () => {
    const stack = new cdk.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        },
        existingLambdaObj,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        }
    };
    const app = () => {
        new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-ddb', props);
    };
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
// NOTE: existingTableObj was omitted from the interface for this construct,
// so this test cannot be run. Leaving it here so it can be used if/when existingTableObj
// is added to the interface
//
// test("Confirm CheckDynamoDBProps is getting called", () => {
//   const stack = new cdk.Stack();
//   const tableName = 'table-name';
//   const existingTable = new dynamodb.Table(stack, 'MyTablet', {
//     tableName,
//     partitionKey: {
//       name: 'id',
//       type: dynamodb.AttributeType.STRING
//     }
//   });
//   const props: IotToLambdaToDynamoDBProps = {
//     lambdaFunctionProps: {
//       code: lambda.Code.fromAsset(`${__dirname}/lambda`),
//       runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
//       handler: 'index.handler'
//     },
//     iotTopicRuleProps: {
//       topicRulePayload: {
//         ruleDisabled: false,
//         description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
//         sql: "SELECT * FROM 'connectedcar/dtc/#'",
//         actions: []
//       }
//     },
//     existingTableObj: existingTable,
//     dynamoTableProps: {
//       tableName,
//       partitionKey: {
//         name: 'id',
//         type: dynamodb.AttributeType.STRING
//       },
//     },
// };
//   const app = () => {
//     new IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
//   };
//   expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
// });
//# sourceMappingURL=data:application/json;base64,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