from hestia_earth.schema import EmissionMethodTier, EmissionStatsDefinition, TermTermType
from hestia_earth.utils.model import filter_list_term_type
from hestia_earth.utils.tools import safe_parse_float

from hestia_earth.models.log import logRequirements, logShouldRun, debugValues, log_as_table
from hestia_earth.models.utils.constant import Units, get_atomic_conversion
from hestia_earth.models.utils.completeness import _is_term_type_complete
from hestia_earth.models.utils.emission import _new_emission
from hestia_earth.models.utils.cycle import get_ecoClimateZone, get_crop_residue_decomposition_N_total
from hestia_earth.models.utils.ecoClimateZone import get_ecoClimateZone_lookup_value
from hestia_earth.models.utils.term import get_lookup_value
from hestia_earth.models.utils.product import has_flooded_rice
from . import MODEL

REQUIREMENTS = {
    "Cycle": {
        "completeness.cropResidue": "True",
        "products": [{
            "@type": "Product",
            "value": "",
            "term.termType": "cropResidue",
            "properties": [{"@type": "Property", "value": "", "term.@id": "nitrogenContent"}]
        }],
        "optional": {
            "endDate": "",
            "site": {
                "@type": "Site",
                "measurements": [
                    {"@type": "Measurement", "value": "", "term.@id": "ecoClimateZone"}
                ]
            },
            "products": [{"@type": "Product", "term.@id": "riceGrainInHuskFlooded"}],
            "practices": [{"@type": "Practice", "term.termType": "waterRegime"}]
        }
    }
}
RETURNS = {
    "Emission": [{
        "value": "",
        "min": "",
        "max": "",
        "sd": "",
        "methodTier": "tier 1",
        "statsDefinition": "modelled",
        "methodModelDescription": ["Aggregated version", "Disaggregated version"]
    }]
}
LOOKUPS = {
    "cropResidue": "decomposesOnField",
    "waterRegime": ["IPCC_2019_N2O_rice", "IPCC_2019_N2O_rice-min", "IPCC_2019_N2O_rice-max"]
}
TERM_ID = 'n2OToAirCropResidueDecompositionDirect'
TIER = EmissionMethodTier.TIER_1.value
FACTORS = {
    'dry': {
        'value': 0.005,
        'min': 0,
        'max': 0.011
    },
    'wet': {
        'value': 0.006,
        'min': 0.001,
        'max': 0.011
    },
    'default': {
        'value': 0.01,
        'min': 0.001,
        'max': 0.018
    },
    'flooded_rice': {
        'value': 0.004,
        'min': 0,
        'max': 0.029
    }
}


def _emission(value: float, min: float, max: float, sd: float, aggregated: bool = False):
    emission = _new_emission(TERM_ID, MODEL)
    emission['value'] = [value]
    emission['min'] = [min]
    emission['max'] = [max]
    emission['sd'] = [sd]
    emission['methodTier'] = TIER
    emission['statsDefinition'] = EmissionStatsDefinition.MODELLED.value
    emission['methodModelDescription'] = 'Aggregated version' if aggregated else 'Disaggregated version'
    return emission


def _get_waterRegime_lookup(practice: dict, col: str):
    return safe_parse_float(get_lookup_value(practice.get('term', {}), col, model=MODEL, term=TERM_ID), None)


def _flooded_rice_factors(cycle: dict):
    practices = filter_list_term_type(cycle.get('practices', []), TermTermType.WATERREGIME)
    practice = next((p for p in practices if _get_waterRegime_lookup(p, LOOKUPS['waterRegime'][0]) is not None), None)

    factors = {
        'value': _get_waterRegime_lookup(practice, LOOKUPS['waterRegime'][0]),
        'min': _get_waterRegime_lookup(practice, LOOKUPS['waterRegime'][1]),
        'max': _get_waterRegime_lookup(practice, LOOKUPS['waterRegime'][2])
    } if practice else FACTORS['flooded_rice']

    return (factors, practice is None)


def _is_wet(ecoClimateZone: str = None):
    return get_ecoClimateZone_lookup_value(ecoClimateZone, 'wet') == 1 if ecoClimateZone else None


def _run(cycle: dict, N_total: float, ecoClimateZone: str = None, flooded_rice: bool = False):
    converted_N_total = N_total * get_atomic_conversion(Units.KG_N2O, Units.TO_N)
    is_wet = _is_wet(ecoClimateZone)
    factors_key = 'default' if is_wet is None else 'wet' if is_wet else 'dry'

    factors, aggregated = _flooded_rice_factors(cycle) if flooded_rice else (FACTORS[factors_key], is_wet is None)

    debugValues(cycle, model=MODEL, term=TERM_ID,
                factors_used=log_as_table(factors),
                aggregated=aggregated)

    value = converted_N_total * factors['value']
    min = converted_N_total * factors['min']
    max = converted_N_total * factors['max']
    sd = converted_N_total * (factors['max'] - factors['min'])/4
    return [_emission(value, min, max, sd, aggregated=aggregated)]


def _should_run(cycle: dict):
    term_type_complete = _is_term_type_complete(cycle, TermTermType.CROPRESIDUE)
    N_crop_residue = get_crop_residue_decomposition_N_total(cycle)
    ecoClimateZone = get_ecoClimateZone(cycle)

    flooded_rice = has_flooded_rice(cycle.get('products', []))

    logRequirements(cycle, model=MODEL, term=TERM_ID,
                    term_type_cropResidue_complete=term_type_complete,
                    N_crop_residue=N_crop_residue,
                    has_flooded_rice=flooded_rice,
                    ecoClimateZone=ecoClimateZone)

    should_run = all([term_type_complete, N_crop_residue])
    logShouldRun(cycle, MODEL, TERM_ID, should_run, methodTier=TIER)
    return should_run, N_crop_residue, ecoClimateZone, flooded_rice


def run(cycle: dict):
    should_run, N_total, ecoClimateZone, flooded_rice = _should_run(cycle)
    return _run(cycle, N_total, ecoClimateZone, flooded_rice) if should_run else []
