"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_1 = require("./construct");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = physical_name_generator_1.generatePhysicalName(this);
        }
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            }
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            }
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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