#  drizzle-interface: Interface Wrappers for Drizzle
#
#  Copyright (c) 2010 Max Goodman
#
#  All rights reserved.
# 
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
# 
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#  3. The name of the author may not be used to endorse or promote products
#     derived from this software without specific prior written permission.
# 
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
#  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
#  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
#  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
#  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
#  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import exceptions
import types

# Standard DBAPI Exceptions
class Error(exceptions.StandardError):
    """Base class of all other error exceptions"""
    pass

class CodedError(Error):
    """Base class of all other error exceptions"""
    def __init__(self, code, msg):
        self.code = code
        self.msg = msg
        
    def __str__(self):
        return "{msg} (#{code})".format(msg=self.msg, code=self.code)

class Warning(exceptions.StandardError):
    """Exception raised for important warnings."""
    pass

class InterfaceError(CodedError):
    """Exception raised for errors that are related to the database interface 
    rather than the database itself.
    
    """
    pass
    
class DatabaseError(CodedError):
    """Exception raised for errors that are related to the database."""
    pass

class InternalError(DatabaseError):
    """Exception raised when the database encounters an internal error."""
    pass

class OperationalError(DatabaseError):
    """Exception raised for errors that are related to the database's operation
    and not necessarily under the control of the programmer.
    
    """
    pass

class ProgrammingError(DatabaseError):
    """Exception raised for programming errors."""
    pass

class IntegrityError(DatabaseError):
    """Exception raised when the database encounters an internal error."""
    pass

class DataError(DatabaseError):
    """Exception raised for errors that are due to problems with the processed
    data.
    
    """
    pass

class NotSupportedError(DatabaseError):
    """Exception raised in case a method or database API was used which is not 
    supported by the database.
    
    """
    pass

# Drizzle Return Code Exceptions
class AddressError(InterfaceError):
    """Exception raised when a hostname or IP is invalid or cannot be resolved."""
    pass

class AuthFailedError(InterfaceError):
    """Exception raised when authentication with the server failed."""
    pass

class LostConnectionError(InterfaceError):
    """Exception raised when the connection to a database is lost."""
    pass

class CouldNotConnectError(InterfaceError):
    """Exception raised when a connection could not be established."""
    pass
