"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/cdk");
const s3 = require("../lib");
module.exports = {
    'default bucket'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket without encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with managed encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'fails if encryption key is used with managed encryption'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.EncryptionKey(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'fails if encryption key is used with encryption set to unencrypted'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.EncryptionKey(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'encryptionKey can specify kms key'(test) {
        const stack = new cdk.Stack();
        const encryptionKey = new kms.EncryptionKey(stack, 'MyKey', { description: 'hello, world' });
        new s3.Bucket(stack, 'MyBucket', { encryptionKey, encryption: s3.BucketEncryption.Kms });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyKey6AB29FA6": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "hello, world",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain"
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": [
                                                "MyKey6AB29FA6",
                                                "Arn"
                                            ]
                                        },
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with versioning turned on'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            versioned: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "VersioningConfiguration": {
                            "Status": "Enabled"
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAll'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAll,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "BlockPublicPolicy": true,
                            "IgnorePublicAcls": true,
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAcls'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAcls,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "IgnorePublicAcls": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with custom block public access setting'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: new s3.BlockPublicAccess({ restrictPublicBuckets: true })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'permissions': {
        'addPermission creates a bucket policy'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyBucketPolicyE7FBAC7B": {
                        "Type": "AWS::S3::BucketPolicy",
                        "Properties": {
                            "Bucket": {
                                "Ref": "MyBucketF68F3FF0"
                            },
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "bar",
                                        "Effect": "Allow",
                                        "Resource": "foo"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                    }
                }
            });
            test.done();
        },
        'forBucket returns a permission statement associated with the bucket\'s ARN'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const x = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            test.deepEqual(bucket.node.resolve(x), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }
            });
            test.done();
        },
        'arnForObjects returns a permission statement associated with objects in the bucket'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const p = new iam.PolicyStatement().addResource(bucket.arnForObjects('hello/world')).addAction('s3:GetObject');
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/hello/world']
                    ]
                }
            });
            test.done();
        },
        'arnForObjects accepts multiple arguments and FnConcats them'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const user = new iam.User(stack, 'MyUser');
            const team = new iam.Group(stack, 'MyTeam');
            const resource = bucket.arnForObjects('home/', team.groupName, '/', user.userName, '/*');
            const p = new iam.PolicyStatement().addResource(resource).addAction('s3:GetObject');
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [
                            { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                            '/home/',
                            { Ref: 'MyTeam01DD6685' },
                            '/',
                            { Ref: 'MyUserDC45028B' },
                            '/*'
                        ]
                    ]
                }
            });
            test.done();
        }
    },
    'removal policy can be used to specify behavior upon delete'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', { removalPolicy: cdk.RemovalPolicy.Orphan, encryption: s3.BucketEncryption.Unencrypted });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyBucketF68F3FF0: {
                    Type: 'AWS::S3::Bucket',
                    DeletionPolicy: 'Retain'
                }
            }
        });
        test.done();
    },
    'import/export': {
        'export creates outputs for the bucket attributes and returns a ref object'(test) {
            const stack = new cdk.Stack(undefined, 'MyStack');
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const bucketRef = bucket.export();
            test.deepEqual(bucket.node.resolve(bucketRef), {
                bucketArn: { 'Fn::ImportValue': 'MyStack:MyBucketBucketArnE260558C' },
                bucketName: { 'Fn::ImportValue': 'MyStack:MyBucketBucketName8A027014' },
                bucketDomainName: { 'Fn::ImportValue': 'MyStack:MyBucketDomainNameF76B9A7A' },
                bucketWebsiteUrl: { 'Fn::ImportValue': 'MyStack:MyBucketWebsiteURL9C222788' }
            });
            test.done();
        },
        'refs will include the bucket\'s encryption key if defined'(test) {
            const stack = new cdk.Stack(undefined, 'MyStack');
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            const bucketRef = bucket.export();
            test.deepEqual(bucket.node.resolve(bucketRef), {
                bucketArn: { 'Fn::ImportValue': 'MyStack:MyBucketBucketArnE260558C' },
                bucketName: { 'Fn::ImportValue': 'MyStack:MyBucketBucketName8A027014' },
                bucketDomainName: { 'Fn::ImportValue': 'MyStack:MyBucketDomainNameF76B9A7A' },
                bucketWebsiteUrl: { 'Fn::ImportValue': 'MyStack:MyBucketWebsiteURL9C222788' }
            });
            test.done();
        },
        'static import(ref) allows importing an external/existing bucket'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            const bucket = s3.Bucket.import(stack, 'ImportedBucket', { bucketArn });
            // this is a no-op since the bucket is external
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            const p = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            // it is possible to obtain a permission statement for a ref
            test.deepEqual(bucket.node.resolve(p), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: 'arn:aws:s3:::my-bucket'
            });
            test.deepEqual(bucket.bucketArn, bucketArn);
            test.deepEqual(bucket.node.resolve(bucket.bucketName), 'my-bucket');
            test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack), {}, 'the ref is not a real resource');
            test.done();
        },
        'import can also be used to import arbitrary ARNs'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.import(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addAllResources().addAction('*'));
            // at this point we technically didn't create any resources in the consuming stack.
            assert_1.expect(stack).toMatch({});
            // but now we can reference the bucket
            // you can even use the bucket name, which will be extracted from the arn provided.
            const user = new iam.User(stack, 'MyUser');
            user.addToPolicy(new iam.PolicyStatement()
                .addResource(bucket.arnForObjects('my/folder/', bucket.bucketName))
                .addAction('s3:*'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "s3:*",
                                        "Effect": "Allow",
                                        "Resource": "arn:aws:s3:::my-bucket/my/folder/my-bucket"
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        },
                    }
                }
            });
            test.done();
        },
        'this is how export/import work together'(test) {
            const stack1 = new cdk.Stack(undefined, 'S1');
            const bucket = new s3.Bucket(stack1, 'MyBucket');
            const bucketRef = bucket.export();
            assert_1.expect(stack1).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    }
                },
                "Outputs": {
                    "MyBucketBucketArnE260558C": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyBucketF68F3FF0",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "S1:MyBucketBucketArnE260558C"
                        }
                    },
                    "MyBucketBucketName8A027014": {
                        "Value": {
                            "Ref": "MyBucketF68F3FF0"
                        },
                        "Export": {
                            "Name": "S1:MyBucketBucketName8A027014"
                        }
                    },
                    "MyBucketDomainNameF76B9A7A": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyBucketF68F3FF0",
                                "DomainName"
                            ]
                        },
                        "Export": {
                            "Name": "S1:MyBucketDomainNameF76B9A7A"
                        }
                    },
                    "MyBucketWebsiteURL9C222788": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyBucketF68F3FF0",
                                "WebsiteURL"
                            ]
                        },
                        "Export": {
                            "Name": "S1:MyBucketWebsiteURL9C222788"
                        }
                    }
                }
            });
            const stack2 = new cdk.Stack(undefined, 'S2');
            const importedBucket = s3.Bucket.import(stack2, 'ImportedBucket', bucketRef);
            const user = new iam.User(stack2, 'MyUser');
            importedBucket.grantRead(user);
            assert_1.expect(stack2).toMatch({
                "Resources": {
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::ImportValue": "S1:MyBucketBucketArnE260558C"
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        { "Fn::ImportValue": "S1:MyBucketBucketArnE260558C" },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        }
    },
    'grantRead'(test) {
        const stack = new cdk.Stack();
        const reader = new iam.User(stack, 'Reader');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.grantRead(reader);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ReaderF7BF189D": {
                    "Type": "AWS::IAM::User"
                },
                "ReaderDefaultPolicy151F3818": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::GetAtt": [
                                                "MyBucketF68F3FF0",
                                                "Arn"
                                            ]
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::GetAtt": [
                                                            "MyBucketF68F3FF0",
                                                            "Arn"
                                                        ]
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "ReaderDefaultPolicy151F3818",
                        "Users": [
                            {
                                "Ref": "ReaderF7BF189D"
                            }
                        ]
                    }
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                },
            }
        });
        test.done();
    },
    'grantReadWrite': {
        'can be used to grant reciprocal permissions to an identity'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'if an encryption key is included, encrypt/decrypt permissions are also added both ways'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketKeyC17130CF": {
                        "Type": "AWS::KMS::Key",
                        "Properties": {
                            "Description": "Created by MyBucket",
                            "KeyPolicy": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kms:Create*",
                                            "kms:Describe*",
                                            "kms:Enable*",
                                            "kms:List*",
                                            "kms:Put*",
                                            "kms:Update*",
                                            "kms:Revoke*",
                                            "kms:Disable*",
                                            "kms:Get*",
                                            "kms:Delete*",
                                            "kms:ScheduleKeyDeletion",
                                            "kms:CancelKeyDeletion"
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":iam::",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":root"
                                                    ]
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::GetAtt": [
                                                    "MyUserDC45028B",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                        "DeletionPolicy": "Retain"
                    },
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "Properties": {
                            "BucketEncryption": {
                                "ServerSideEncryptionConfiguration": [
                                    {
                                        "ServerSideEncryptionByDefault": {
                                            "KMSMasterKeyID": {
                                                "Fn::GetAtt": [
                                                    "MyBucketKeyC17130CF",
                                                    "Arn"
                                                ]
                                            },
                                            "SSEAlgorithm": "aws:kms"
                                        }
                                    }
                                ]
                            }
                        }
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyBucketKeyC17130CF",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
    },
    'more grants'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
        const putter = new iam.User(stack, 'Putter');
        const writer = new iam.User(stack, 'Writer');
        const deleter = new iam.User(stack, 'Deleter');
        bucket.grantPut(putter);
        bucket.grantWrite(writer);
        bucket.grantDelete(deleter);
        const resources = assert_1.SynthUtils.toCloudFormation(stack).Resources;
        const actions = (id) => resources[id].Properties.PolicyDocument.Statement[0].Action;
        test.deepEqual(actions('WriterDefaultPolicyDC585BCE'), ['s3:DeleteObject*', 's3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('PutterDefaultPolicyAB138DD3'), ['s3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('DeleterDefaultPolicyCD33B8A0'), 's3:DeleteObject*');
        test.done();
    },
    'cross-stack permissions'(test) {
        const stackA = new cdk.Stack();
        const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket');
        const refToBucketFromStackA = bucketFromStackA.export();
        const stackB = new cdk.Stack();
        const user = new iam.User(stackB, 'UserWhoNeedsAccess');
        const theBucketFromStackAAsARefInStackB = s3.Bucket.import(stackB, 'RefToBucketFromStackA', refToBucketFromStackA);
        theBucketFromStackAAsARefInStackB.grantRead(user);
        assert_1.expect(stackA).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            },
            "Outputs": {
                "MyBucketBucketArnE260558C": {
                    "Value": {
                        "Fn::GetAtt": [
                            "MyBucketF68F3FF0",
                            "Arn"
                        ]
                    },
                    "Export": {
                        "Name": "Stack:MyBucketBucketArnE260558C"
                    }
                },
                "MyBucketBucketName8A027014": {
                    "Value": {
                        "Ref": "MyBucketF68F3FF0"
                    },
                    "Export": {
                        "Name": "Stack:MyBucketBucketName8A027014"
                    }
                },
                "MyBucketDomainNameF76B9A7A": {
                    "Value": {
                        "Fn::GetAtt": [
                            "MyBucketF68F3FF0",
                            "DomainName"
                        ]
                    },
                    "Export": {
                        "Name": "Stack:MyBucketDomainNameF76B9A7A"
                    }
                },
                "MyBucketWebsiteURL9C222788": {
                    "Value": {
                        "Fn::GetAtt": [
                            "MyBucketF68F3FF0",
                            "WebsiteURL"
                        ]
                    },
                    "Export": { "Name": "Stack:MyBucketWebsiteURL9C222788" }
                }
            }
        });
        assert_1.expect(stackB).toMatch({
            "Resources": {
                "UserWhoNeedsAccessF8959C3D": {
                    "Type": "AWS::IAM::User"
                },
                "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::ImportValue": "Stack:MyBucketBucketArnE260558C"
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::ImportValue": "Stack:MyBucketBucketArnE260558C"
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                        "Users": [
                            {
                                "Ref": "UserWhoNeedsAccessF8959C3D"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'urlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketURL', { value: bucket.bucketUrl });
        new cdk.CfnOutput(stack, 'MyFileURL', { value: bucket.urlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileURL', { value: bucket.urlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            },
            "Outputs": {
                "BucketURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                }
                            ]
                        ]
                    },
                },
                "MyFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/my/file.txt"
                            ]
                        ]
                    }
                },
                "YourFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/your/file.txt"
                            ]
                        ]
                    },
                }
            }
        });
        test.done();
    },
    'grantPublicAccess': {
        'by default, grants s3:GetObject to all objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"keyPrefix" can be used to only grant access to certain objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('only/access/these/*');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/only/access/these/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"allowedActions" can be used to specify actions explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('*', 's3:GetObject', 's3:PutObject');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": ["s3:GetObject", "s3:PutObject"],
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'returns the PolicyStatement which can be then customized'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            const statement = bucket.grantPublicAccess();
            statement.addCondition('IpAddress', { "aws:SourceIp": "54.240.143.0/24" });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] },
                            "Condition": {
                                "IpAddress": { "aws:SourceIp": "54.240.143.0/24" }
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'throws when blockPublicPolicy is set to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', {
                blockPublicAccess: new s3.BlockPublicAccess({ blockPublicPolicy: true })
            });
            // THEN
            test.throws(() => bucket.grantPublicAccess(), /blockPublicPolicy/);
            test.done();
        }
    },
    'website configuration': {
        'only index doc'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html'
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html"
                }
            }));
            test.done();
        },
        'fails if only error doc is specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteErrorDocument: 'error.html'
                });
            }, /"websiteIndexDocument" is required if "websiteErrorDocument" is set/);
            test.done();
        },
        'error and index docs'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
                websiteErrorDocument: 'error.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html",
                    ErrorDocument: "error.html"
                }
            }));
            test.done();
        },
        'exports the WebsiteURL'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html'
            });
            test.deepEqual(bucket.node.resolve(bucket.bucketWebsiteUrl), { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] });
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5idWNrZXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmJ1Y2tldC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQW1FO0FBQ25FLHdDQUF5QztBQUN6Qyx3Q0FBeUM7QUFDekMsb0NBQXFDO0FBRXJDLDZCQUE4QjtBQUs5QixpQkFBUztJQUNQLGdCQUFnQixDQUFDLElBQVU7UUFDekIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVqQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDcEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsZ0JBQWdCLEVBQUUsUUFBUTtpQkFDekI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyQkFBMkIsQ0FBQyxJQUFVO1FBQ3BDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9CLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztTQUM1QyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDcEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsZ0JBQWdCLEVBQUUsUUFBUTtpQkFDekI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnQ0FBZ0MsQ0FBQyxJQUFVO1FBQ3pDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9CLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtTQUMzQyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDcEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsWUFBWSxFQUFFO3dCQUNaLGtCQUFrQixFQUFFOzRCQUNwQixtQ0FBbUMsRUFBRTtnQ0FDbkM7b0NBQ0EsK0JBQStCLEVBQUU7d0NBQy9CLGNBQWMsRUFBRSxTQUFTO3FDQUMxQjtpQ0FDQTs2QkFDRjt5QkFDQTtxQkFDRjtvQkFDRCxnQkFBZ0IsRUFBRSxRQUFRO2lCQUN6QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlEQUF5RCxDQUFDLElBQVU7UUFDbEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVwRCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ2pELFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtZQUMxQyxhQUFhLEVBQUUsS0FBSztTQUNyQixDQUFDLEVBQUUsZ0VBQWdFLENBQUMsQ0FBQztRQUV0RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0VBQW9FLENBQUMsSUFBVTtRQUM3RSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXBELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDakQsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXO1lBQzNDLGFBQWEsRUFBRSxLQUFLO1NBQ3JCLENBQUMsRUFBRSxnRUFBZ0UsQ0FBQyxDQUFDO1FBRXRFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsV0FBVyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFFN0YsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxhQUFhLEVBQUUsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBRXpGLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLGVBQWUsRUFBRTtvQkFDakIsTUFBTSxFQUFFLGVBQWU7b0JBQ3ZCLFlBQVksRUFBRTt3QkFDWixhQUFhLEVBQUUsY0FBYzt3QkFDN0IsV0FBVyxFQUFFOzRCQUNiLFdBQVcsRUFBRTtnQ0FDWDtvQ0FDQSxRQUFRLEVBQUU7d0NBQ1IsYUFBYTt3Q0FDYixlQUFlO3dDQUNmLGFBQWE7d0NBQ2IsV0FBVzt3Q0FDWCxVQUFVO3dDQUNWLGFBQWE7d0NBQ2IsYUFBYTt3Q0FDYixjQUFjO3dDQUNkLFVBQVU7d0NBQ1YsYUFBYTt3Q0FDYix5QkFBeUI7d0NBQ3pCLHVCQUF1QjtxQ0FDeEI7b0NBQ0QsUUFBUSxFQUFFLE9BQU87b0NBQ2pCLFdBQVcsRUFBRTt3Q0FDWCxLQUFLLEVBQUU7NENBQ1AsVUFBVSxFQUFFO2dEQUNWLEVBQUU7Z0RBQ0Y7b0RBQ0EsTUFBTTtvREFDTjt3REFDRSxLQUFLLEVBQUUsZ0JBQWdCO3FEQUN4QjtvREFDRCxRQUFRO29EQUNSO3dEQUNFLEtBQUssRUFBRSxnQkFBZ0I7cURBQ3hCO29EQUNELE9BQU87aURBQ047NkNBQ0Y7eUNBQ0E7cUNBQ0Y7b0NBQ0QsVUFBVSxFQUFFLEdBQUc7aUNBQ2Q7NkJBQ0Y7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3RCO3FCQUNGO29CQUNELGdCQUFnQixFQUFFLFFBQVE7aUJBQ3pCO2dCQUNELGtCQUFrQixFQUFFO29CQUNwQixNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osa0JBQWtCLEVBQUU7NEJBQ3BCLG1DQUFtQyxFQUFFO2dDQUNuQztvQ0FDQSwrQkFBK0IsRUFBRTt3Q0FDL0IsZ0JBQWdCLEVBQUU7NENBQ2xCLFlBQVksRUFBRTtnREFDWixlQUFlO2dEQUNmLEtBQUs7NkNBQ047eUNBQ0E7d0NBQ0QsY0FBYyxFQUFFLFNBQVM7cUNBQzFCO2lDQUNBOzZCQUNGO3lCQUNBO3FCQUNGO29CQUNELGdCQUFnQixFQUFFLFFBQVE7aUJBQ3pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0NBQWtDLENBQUMsSUFBVTtRQUMzQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMvQixTQUFTLEVBQUUsSUFBSTtTQUNoQixDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDbEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsWUFBWSxFQUFFO3dCQUNaLHlCQUF5QixFQUFFOzRCQUN6QixRQUFRLEVBQUUsU0FBUzt5QkFDcEI7cUJBQ0Y7b0JBQ0gsZ0JBQWdCLEVBQUUsUUFBUTtpQkFDekI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpREFBaUQsQ0FBQyxJQUFVO1FBQzFELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9CLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRO1NBQ2pELENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLGtCQUFrQixFQUFFO29CQUNsQixNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osZ0NBQWdDLEVBQUU7NEJBQ2hDLGlCQUFpQixFQUFFLElBQUk7NEJBQ3ZCLG1CQUFtQixFQUFFLElBQUk7NEJBQ3pCLGtCQUFrQixFQUFFLElBQUk7NEJBQ3hCLHVCQUF1QixFQUFFLElBQUk7eUJBQzlCO3FCQUNGO29CQUNILGdCQUFnQixFQUFFLFFBQVE7aUJBQ3pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMvQixpQkFBaUIsRUFBRSxFQUFFLENBQUMsaUJBQWlCLENBQUMsU0FBUztTQUNsRCxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDbEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsWUFBWSxFQUFFO3dCQUNaLGdDQUFnQyxFQUFFOzRCQUNoQyxpQkFBaUIsRUFBRSxJQUFJOzRCQUN2QixrQkFBa0IsRUFBRSxJQUFJO3lCQUN6QjtxQkFDRjtvQkFDSCxnQkFBZ0IsRUFBRSxRQUFRO2lCQUN6QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdEQUFnRCxDQUFDLElBQVU7UUFDekQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0IsaUJBQWlCLEVBQUUsSUFBSSxFQUFFLENBQUMsaUJBQWlCLENBQUMsRUFBRSxxQkFBcUIsRUFBRSxJQUFJLEVBQUUsQ0FBQztTQUM3RSxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxrQkFBa0IsRUFBRTtvQkFDbEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsWUFBWSxFQUFFO3dCQUNaLGdDQUFnQyxFQUFFOzRCQUNoQyx1QkFBdUIsRUFBRSxJQUFJO3lCQUM5QjtxQkFDRjtvQkFDSCxnQkFBZ0IsRUFBRSxRQUFRO2lCQUN6QjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGFBQWEsRUFBRTtRQUViLHVDQUF1QyxDQUFDLElBQVU7WUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7WUFFakcsTUFBTSxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUUxRixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixXQUFXLEVBQUU7b0JBQ1gsa0JBQWtCLEVBQUU7d0JBQ3BCLE1BQU0sRUFBRSxpQkFBaUI7d0JBQzNCLGdCQUFnQixFQUFFLFFBQVE7cUJBQ3ZCO29CQUNELHdCQUF3QixFQUFFO3dCQUMxQixNQUFNLEVBQUUsdUJBQXVCO3dCQUMvQixZQUFZLEVBQUU7NEJBQ1osUUFBUSxFQUFFO2dDQUNWLEtBQUssRUFBRSxrQkFBa0I7NkJBQ3hCOzRCQUNELGdCQUFnQixFQUFFO2dDQUNsQixXQUFXLEVBQUU7b0NBQ1g7d0NBQ0EsUUFBUSxFQUFFLEtBQUs7d0NBQ2YsUUFBUSxFQUFFLE9BQU87d0NBQ2pCLFVBQVUsRUFBRSxLQUFLO3FDQUNoQjtpQ0FDRjtnQ0FDRCxTQUFTLEVBQUUsWUFBWTs2QkFDdEI7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNEVBQTRFLENBQUMsSUFBVTtZQUNyRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztZQUVqRyxNQUFNLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUU3RixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNyQyxNQUFNLEVBQUUsZUFBZTtnQkFDdkIsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFFLEVBQUU7YUFDMUQsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG9GQUFvRixDQUFDLElBQVU7WUFDN0YsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7WUFFakcsTUFBTSxDQUFDLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxFQUFFLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUM7WUFFL0csSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDckMsTUFBTSxFQUFFLGNBQWM7Z0JBQ3RCLE1BQU0sRUFBRSxPQUFPO2dCQUNmLFFBQVEsRUFBRTtvQkFDUixVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRixDQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxjQUFjLENBQUU7cUJBQ3BFO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDZEQUE2RCxDQUFDLElBQVU7WUFFdEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7WUFFakcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUMzQyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBRTVDLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDekYsTUFBTSxDQUFDLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxFQUFFLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUVwRixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUNyQyxNQUFNLEVBQUUsY0FBYztnQkFDdEIsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFO29CQUNSLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLEVBQUUsWUFBWSxFQUFFLENBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFFLEVBQUU7NEJBQy9DLFFBQVE7NEJBQ1IsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7NEJBQ3pCLEdBQUc7NEJBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7NEJBQ3pCLElBQUk7eUJBQ0w7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUUzSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFNBQVMsRUFBRTtnQkFDVCxnQkFBZ0IsRUFBRTtvQkFDaEIsSUFBSSxFQUFFLGlCQUFpQjtvQkFDdkIsY0FBYyxFQUFFLFFBQVE7aUJBQ3pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZUFBZSxFQUFFO1FBQ2YsMkVBQTJFLENBQUMsSUFBVTtZQUNwRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQ2xELE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDaEQsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2xDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUU7Z0JBQzdDLFNBQVMsRUFBRSxFQUFFLGlCQUFpQixFQUFFLG1DQUFtQyxFQUFFO2dCQUNyRSxVQUFVLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxvQ0FBb0MsRUFBRTtnQkFDdkUsZ0JBQWdCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxvQ0FBb0MsRUFBRTtnQkFDN0UsZ0JBQWdCLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxvQ0FBb0MsRUFBRTthQUM5RSxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsMkRBQTJELENBQUMsSUFBVTtZQUNwRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQ2xELE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQ3pGLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFO2dCQUM3QyxTQUFTLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxtQ0FBbUMsRUFBRTtnQkFDckUsVUFBVSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsb0NBQW9DLEVBQUU7Z0JBQ3ZFLGdCQUFnQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsb0NBQW9DLEVBQUU7Z0JBQzdFLGdCQUFnQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsb0NBQW9DLEVBQUU7YUFDOUUsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGlFQUFpRSxDQUFDLElBQVU7WUFDMUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsTUFBTSxTQUFTLEdBQUcsd0JBQXdCLENBQUM7WUFDM0MsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztZQUV4RSwrQ0FBK0M7WUFDL0MsTUFBTSxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUUxRixNQUFNLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUU3Riw0REFBNEQ7WUFDNUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDckMsTUFBTSxFQUFFLGVBQWU7Z0JBQ3ZCLE1BQU0sRUFBRSxPQUFPO2dCQUNmLFFBQVEsRUFBRSx3QkFBd0I7YUFDbkMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQzVDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBRXBFLElBQUksQ0FBQyxTQUFTLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFFLEVBQUUsZ0NBQWdDLENBQUMsQ0FBQztZQUN6RixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsa0RBQWtELENBQUMsSUFBVTtZQUMzRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxTQUFTLEVBQUUsd0JBQXdCLEVBQUUsQ0FBQyxDQUFDO1lBQ2xHLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUV2RixtRkFBbUY7WUFDbkYsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUUxQixzQ0FBc0M7WUFDdEMsbUZBQW1GO1lBQ25GLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDM0MsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUU7aUJBQ3ZDLFdBQVcsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUM7aUJBQ2xFLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1lBRXRCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7Z0JBQ3BCLFdBQVcsRUFBRTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsTUFBTSxFQUFFLGdCQUFnQjtxQkFDekI7b0JBQ0QsNkJBQTZCLEVBQUU7d0JBQzdCLE1BQU0sRUFBRSxrQkFBa0I7d0JBQzFCLFlBQVksRUFBRTs0QkFDZCxnQkFBZ0IsRUFBRTtnQ0FDaEIsV0FBVyxFQUFFO29DQUNiO3dDQUNFLFFBQVEsRUFBRSxNQUFNO3dDQUNoQixRQUFRLEVBQUUsT0FBTzt3Q0FDakIsVUFBVSxFQUFFLDRDQUE0QztxQ0FDekQ7aUNBQ0E7Z0NBQ0QsU0FBUyxFQUFFLFlBQVk7NkJBQ3hCOzRCQUNELFlBQVksRUFBRSw2QkFBNkI7NEJBQzNDLE9BQU8sRUFBRTtnQ0FDUDtvQ0FDQSxLQUFLLEVBQUUsZ0JBQWdCO2lDQUN0Qjs2QkFDRjt5QkFDQTtxQkFDRjtpQkFDQTthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx5Q0FBeUMsQ0FBQyxJQUFVO1lBQ2xELE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDOUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztZQUNqRCxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUM7WUFFbEMsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQztnQkFDckIsV0FBVyxFQUFFO29CQUNiLGtCQUFrQixFQUFFO3dCQUNsQixNQUFNLEVBQUUsaUJBQWlCO3dCQUMzQixnQkFBZ0IsRUFBRSxRQUFRO3FCQUN6QjtpQkFDQTtnQkFDRCxTQUFTLEVBQUU7b0JBQ1gsMkJBQTJCLEVBQUU7d0JBQzNCLE9BQU8sRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osa0JBQWtCO2dDQUNsQixLQUFLOzZCQUNOO3lCQUNBO3dCQUNELFFBQVEsRUFBRTs0QkFDVixNQUFNLEVBQUUsOEJBQThCO3lCQUNyQztxQkFDRjtvQkFDRCw0QkFBNEIsRUFBRTt3QkFDNUIsT0FBTyxFQUFFOzRCQUNULEtBQUssRUFBRSxrQkFBa0I7eUJBQ3hCO3dCQUNELFFBQVEsRUFBRTs0QkFDVixNQUFNLEVBQUUsK0JBQStCO3lCQUN0QztxQkFDRjtvQkFDRCw0QkFBNEIsRUFBRTt3QkFDNUIsT0FBTyxFQUFFOzRCQUNQLFlBQVksRUFBRTtnQ0FDWixrQkFBa0I7Z0NBQ2xCLFlBQVk7NkJBQ2I7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLE1BQU0sRUFBRSwrQkFBK0I7eUJBQ3hDO3FCQUNGO29CQUNELDRCQUE0QixFQUFFO3dCQUM1QixPQUFPLEVBQUU7NEJBQ1AsWUFBWSxFQUFFO2dDQUNaLGtCQUFrQjtnQ0FDbEIsWUFBWTs2QkFDYjt5QkFDRjt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsTUFBTSxFQUFFLCtCQUErQjt5QkFDeEM7cUJBQ0Y7aUJBQ0E7YUFDRixDQUFDLENBQUM7WUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzlDLE1BQU0sY0FBYyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUM3RSxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQzVDLGNBQWMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFL0IsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQztnQkFDckIsV0FBVyxFQUFFO29CQUNiLGdCQUFnQixFQUFFO3dCQUNoQixNQUFNLEVBQUUsZ0JBQWdCO3FCQUN6QjtvQkFDRCw2QkFBNkIsRUFBRTt3QkFDN0IsTUFBTSxFQUFFLGtCQUFrQjt3QkFDMUIsWUFBWSxFQUFFOzRCQUNkLGdCQUFnQixFQUFFO2dDQUNoQixXQUFXLEVBQUU7b0NBQ2I7d0NBQ0UsUUFBUSxFQUFFOzRDQUNWLGVBQWU7NENBQ2YsZUFBZTs0Q0FDZixVQUFVO3lDQUNUO3dDQUNELFFBQVEsRUFBRSxPQUFPO3dDQUNqQixVQUFVLEVBQUU7NENBQ1o7Z0RBQ0UsaUJBQWlCLEVBQUUsOEJBQThCOzZDQUNsRDs0Q0FDRDtnREFDRSxVQUFVLEVBQUU7b0RBQ1osRUFBRTtvREFDRjt3REFDRSxFQUFFLGlCQUFpQixFQUFFLDhCQUE4QixFQUFFO3dEQUNyRCxJQUFJO3FEQUNMO2lEQUNBOzZDQUNGO3lDQUNBO3FDQUNGO2lDQUNBO2dDQUNELFNBQVMsRUFBRSxZQUFZOzZCQUN4Qjs0QkFDRCxZQUFZLEVBQUUsNkJBQTZCOzRCQUMzQyxPQUFPLEVBQUU7Z0NBQ1A7b0NBQ0EsS0FBSyxFQUFFLGdCQUFnQjtpQ0FDdEI7NkJBQ0Y7eUJBQ0E7cUJBQ0Y7aUJBQ0E7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxXQUFXLENBQUMsSUFBVTtRQUNwQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDaEQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDYixnQkFBZ0IsRUFBRTtvQkFDaEIsTUFBTSxFQUFFLGdCQUFnQjtpQkFDekI7Z0JBQ0QsNkJBQTZCLEVBQUU7b0JBQzdCLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDZCxnQkFBZ0IsRUFBRTs0QkFDaEIsV0FBVyxFQUFFO2dDQUNiO29DQUNFLFFBQVEsRUFBRTt3Q0FDVixlQUFlO3dDQUNmLGVBQWU7d0NBQ2YsVUFBVTtxQ0FDVDtvQ0FDRCxRQUFRLEVBQUUsT0FBTztvQ0FDakIsVUFBVSxFQUFFO3dDQUNaOzRDQUNFLFlBQVksRUFBRTtnREFDZCxrQkFBa0I7Z0RBQ2xCLEtBQUs7NkNBQ0o7eUNBQ0Y7d0NBQ0Q7NENBQ0UsVUFBVSxFQUFFO2dEQUNaLEVBQUU7Z0RBQ0Y7b0RBQ0U7d0RBQ0EsWUFBWSxFQUFFOzREQUNaLGtCQUFrQjs0REFDbEIsS0FBSzt5REFDTjtxREFDQTtvREFDRCxJQUFJO2lEQUNMOzZDQUNBO3lDQUNGO3FDQUNBO2lDQUNGOzZCQUNBOzRCQUNELFNBQVMsRUFBRSxZQUFZO3lCQUN4Qjt3QkFDRCxZQUFZLEVBQUUsNkJBQTZCO3dCQUMzQyxPQUFPLEVBQUU7NEJBQ1A7Z0NBQ0EsS0FBSyxFQUFFLGdCQUFnQjs2QkFDdEI7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLE1BQU0sRUFBRSxpQkFBaUI7b0JBQ3pCLGdCQUFnQixFQUFFLFFBQVE7aUJBQzNCO2FBQ0E7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0JBQWdCLEVBQUU7UUFDaEIsNERBQTRELENBQUMsSUFBVTtZQUNyRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBQ2hELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDM0MsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUU1QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixXQUFXLEVBQUU7b0JBQ1gsa0JBQWtCLEVBQUU7d0JBQ3BCLE1BQU0sRUFBRSxpQkFBaUI7d0JBQzNCLGdCQUFnQixFQUFFLFFBQVE7cUJBQ3ZCO29CQUNELGdCQUFnQixFQUFFO3dCQUNsQixNQUFNLEVBQUUsZ0JBQWdCO3FCQUN2QjtvQkFDRCw2QkFBNkIsRUFBRTt3QkFDL0IsTUFBTSxFQUFFLGtCQUFrQjt3QkFDMUIsWUFBWSxFQUFFOzRCQUNaLGdCQUFnQixFQUFFO2dDQUNsQixXQUFXLEVBQUU7b0NBQ1g7d0NBQ0EsUUFBUSxFQUFFOzRDQUNSLGVBQWU7NENBQ2YsZUFBZTs0Q0FDZixVQUFVOzRDQUNWLGtCQUFrQjs0Q0FDbEIsZUFBZTs0Q0FDZixXQUFXO3lDQUNaO3dDQUNELFFBQVEsRUFBRSxPQUFPO3dDQUNqQixVQUFVLEVBQUU7NENBQ1Y7Z0RBQ0EsWUFBWSxFQUFFO29EQUNaLGtCQUFrQjtvREFDbEIsS0FBSztpREFDTjs2Q0FDQTs0Q0FDRDtnREFDQSxVQUFVLEVBQUU7b0RBQ1YsRUFBRTtvREFDRjt3REFDQTs0REFDRSxZQUFZLEVBQUU7Z0VBQ2Qsa0JBQWtCO2dFQUNsQixLQUFLOzZEQUNKO3lEQUNGO3dEQUNELElBQUk7cURBQ0g7aURBQ0Y7NkNBQ0E7eUNBQ0Y7cUNBQ0E7aUNBQ0Y7Z0NBQ0QsU0FBUyxFQUFFLFlBQVk7NkJBQ3RCOzRCQUNELFlBQVksRUFBRSw2QkFBNkI7NEJBQzNDLE9BQU8sRUFBRTtnQ0FDVDtvQ0FDRSxLQUFLLEVBQUUsZ0JBQWdCO2lDQUN4Qjs2QkFDQTt5QkFDRjtxQkFDQTtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx3RkFBd0YsQ0FBQyxJQUFVO1lBQ2pHLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQ3pGLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDM0MsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUU1QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixXQUFXLEVBQUU7b0JBQ1gscUJBQXFCLEVBQUU7d0JBQ3ZCLE1BQU0sRUFBRSxlQUFlO3dCQUN2QixZQUFZLEVBQUU7NEJBQ1osYUFBYSxFQUFFLHFCQUFxQjs0QkFDcEMsV0FBVyxFQUFFO2dDQUNiLFdBQVcsRUFBRTtvQ0FDWDt3Q0FDQSxRQUFRLEVBQUU7NENBQ1IsYUFBYTs0Q0FDYixlQUFlOzRDQUNmLGFBQWE7NENBQ2IsV0FBVzs0Q0FDWCxVQUFVOzRDQUNWLGFBQWE7NENBQ2IsYUFBYTs0Q0FDYixjQUFjOzRDQUNkLFVBQVU7NENBQ1YsYUFBYTs0Q0FDYix5QkFBeUI7NENBQ3pCLHVCQUF1Qjt5Q0FDeEI7d0NBQ0QsUUFBUSxFQUFFLE9BQU87d0NBQ2pCLFdBQVcsRUFBRTs0Q0FDWCxLQUFLLEVBQUU7Z0RBQ1AsVUFBVSxFQUFFO29EQUNWLEVBQUU7b0RBQ0Y7d0RBQ0EsTUFBTTt3REFDTjs0REFDRSxLQUFLLEVBQUUsZ0JBQWdCO3lEQUN4Qjt3REFDRCxRQUFRO3dEQUNSOzREQUNFLEtBQUssRUFBRSxnQkFBZ0I7eURBQ3hCO3dEQUNELE9BQU87cURBQ047aURBQ0Y7NkNBQ0E7eUNBQ0Y7d0NBQ0QsVUFBVSxFQUFFLEdBQUc7cUNBQ2Q7b0NBQ0Q7d0NBQ0EsUUFBUSxFQUFFOzRDQUNSLGFBQWE7NENBQ2IsaUJBQWlCOzRDQUNqQixhQUFhOzRDQUNiLGdCQUFnQjs0Q0FDaEIsc0JBQXNCO3lDQUN2Qjt3Q0FDRCxRQUFRLEVBQUUsT0FBTzt3Q0FDakIsV0FBVyxFQUFFOzRDQUNYLEtBQUssRUFBRTtnREFDUCxZQUFZLEVBQUU7b0RBQ1osZ0JBQWdCO29EQUNoQixLQUFLO2lEQUNOOzZDQUNBO3lDQUNGO3dDQUNELFVBQVUsRUFBRSxHQUFHO3FDQUNkO2lDQUNGO2dDQUNELFNBQVMsRUFBRSxZQUFZOzZCQUN0Qjt5QkFDRjt3QkFDRCxnQkFBZ0IsRUFBRSxRQUFRO3FCQUN6QjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDcEIsTUFBTSxFQUFFLGlCQUFpQjt3QkFDekIsZ0JBQWdCLEVBQUUsUUFBUTt3QkFDMUIsWUFBWSxFQUFFOzRCQUNaLGtCQUFrQixFQUFFO2dDQUNwQixtQ0FBbUMsRUFBRTtvQ0FDbkM7d0NBQ0EsK0JBQStCLEVBQUU7NENBQy9CLGdCQUFnQixFQUFFO2dEQUNsQixZQUFZLEVBQUU7b0RBQ1oscUJBQXFCO29EQUNyQixLQUFLO2lEQUNOOzZDQUNBOzRDQUNELGNBQWMsRUFBRSxTQUFTO3lDQUMxQjtxQ0FDQTtpQ0FDRjs2QkFDQTt5QkFDRjtxQkFDQTtvQkFDRCxnQkFBZ0IsRUFBRTt3QkFDbEIsTUFBTSxFQUFFLGdCQUFnQjtxQkFDdkI7b0JBQ0QsNkJBQTZCLEVBQUU7d0JBQy9CLE1BQU0sRUFBRSxrQkFBa0I7d0JBQzFCLFlBQVksRUFBRTs0QkFDWixnQkFBZ0IsRUFBRTtnQ0FDbEIsV0FBVyxFQUFFO29DQUNYO3dDQUNBLFFBQVEsRUFBRTs0Q0FDUixlQUFlOzRDQUNmLGVBQWU7NENBQ2YsVUFBVTs0Q0FDVixrQkFBa0I7NENBQ2xCLGVBQWU7NENBQ2YsV0FBVzt5Q0FDWjt3Q0FDRCxRQUFRLEVBQUUsT0FBTzt3Q0FDakIsVUFBVSxFQUFFOzRDQUNWO2dEQUNBLFlBQVksRUFBRTtvREFDWixrQkFBa0I7b0RBQ2xCLEtBQUs7aURBQ047NkNBQ0E7NENBQ0Q7Z0RBQ0EsVUFBVSxFQUFFO29EQUNWLEVBQUU7b0RBQ0Y7d0RBQ0E7NERBQ0UsWUFBWSxFQUFFO2dFQUNkLGtCQUFrQjtnRUFDbEIsS0FBSzs2REFDSjt5REFDRjt3REFDRCxJQUFJO3FEQUNIO2lEQUNGOzZDQUNBO3lDQUNGO3FDQUNBO29DQUNEO3dDQUNBLFFBQVEsRUFBRTs0Q0FDUixhQUFhOzRDQUNiLGlCQUFpQjs0Q0FDakIsYUFBYTs0Q0FDYixnQkFBZ0I7NENBQ2hCLHNCQUFzQjt5Q0FDdkI7d0NBQ0QsUUFBUSxFQUFFLE9BQU87d0NBQ2pCLFVBQVUsRUFBRTs0Q0FDVixZQUFZLEVBQUU7Z0RBQ2QscUJBQXFCO2dEQUNyQixLQUFLOzZDQUNKO3lDQUNGO3FDQUNBO2lDQUNGO2dDQUNELFNBQVMsRUFBRSxZQUFZOzZCQUN0Qjs0QkFDRCxZQUFZLEVBQUUsNkJBQTZCOzRCQUMzQyxPQUFPLEVBQUU7Z0NBQ1Q7b0NBQ0UsS0FBSyxFQUFFLGdCQUFnQjtpQ0FDeEI7NkJBQ0E7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCxhQUFhLENBQUMsSUFBVTtRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUN6RixNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDN0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUUvQyxNQUFNLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3hCLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDMUIsTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUU1QixNQUFNLFNBQVMsR0FBRyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUMvRCxNQUFNLE9BQU8sR0FBRyxDQUFDLEVBQVUsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUU1RixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyw2QkFBNkIsQ0FBQyxFQUFFLENBQUUsa0JBQWtCLEVBQUUsZUFBZSxFQUFFLFdBQVcsQ0FBRSxDQUFDLENBQUM7UUFDN0csSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsNkJBQTZCLENBQUMsRUFBRSxDQUFFLGVBQWUsRUFBRSxXQUFXLENBQUUsQ0FBQyxDQUFDO1FBQ3pGLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLDhCQUE4QixDQUFDLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUM1RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUJBQXlCLENBQUMsSUFBVTtRQUNsQyxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMvQixNQUFNLGdCQUFnQixHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDM0QsTUFBTSxxQkFBcUIsR0FBRyxnQkFBZ0IsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUV4RCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMvQixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDeEQsTUFBTSxpQ0FBaUMsR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsdUJBQXVCLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUNuSCxpQ0FBaUMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFbEQsZUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNyQixXQUFXLEVBQUU7Z0JBQ2Isa0JBQWtCLEVBQUU7b0JBQ2xCLE1BQU0sRUFBRSxpQkFBaUI7b0JBQ3ZCLGdCQUFnQixFQUFFLFFBQVE7aUJBQzdCO2FBQ0E7WUFDRCxTQUFTLEVBQUU7Z0JBQ1gsMkJBQTJCLEVBQUU7b0JBQzNCLE9BQU8sRUFBRTt3QkFDVCxZQUFZLEVBQUU7NEJBQ1osa0JBQWtCOzRCQUNsQixLQUFLO3lCQUNOO3FCQUNBO29CQUNELFFBQVEsRUFBRTt3QkFDVixNQUFNLEVBQUUsaUNBQWlDO3FCQUN4QztpQkFDRjtnQkFDRCw0QkFBNEIsRUFBRTtvQkFDNUIsT0FBTyxFQUFFO3dCQUNULEtBQUssRUFBRSxrQkFBa0I7cUJBQ3hCO29CQUNELFFBQVEsRUFBRTt3QkFDVixNQUFNLEVBQUUsa0NBQWtDO3FCQUN6QztpQkFDRjtnQkFDRCw0QkFBNEIsRUFBRTtvQkFDNUIsT0FBTyxFQUFFO3dCQUNQLFlBQVksRUFBRTs0QkFDWixrQkFBa0I7NEJBQ2xCLFlBQVk7eUJBQ2I7cUJBQ0Y7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLE1BQU0sRUFBRSxrQ0FBa0M7cUJBQzNDO2lCQUNGO2dCQUNELDRCQUE0QixFQUFFO29CQUM1QixPQUFPLEVBQUU7d0JBQ1AsWUFBWSxFQUFFOzRCQUNaLGtCQUFrQjs0QkFDbEIsWUFBWTt5QkFDYjtxQkFDRjtvQkFDRCxRQUFRLEVBQUUsRUFBQyxNQUFNLEVBQUUsa0NBQWtDLEVBQUM7aUJBQ3ZEO2FBQ0E7U0FDRixDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3JCLFdBQVcsRUFBRTtnQkFDYiw0QkFBNEIsRUFBRTtvQkFDNUIsTUFBTSxFQUFFLGdCQUFnQjtpQkFDekI7Z0JBQ0QseUNBQXlDLEVBQUU7b0JBQ3pDLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDZCxnQkFBZ0IsRUFBRTs0QkFDaEIsV0FBVyxFQUFFO2dDQUNiO29DQUNFLFFBQVEsRUFBRTt3Q0FDVixlQUFlO3dDQUNmLGVBQWU7d0NBQ2YsVUFBVTtxQ0FDVDtvQ0FDRCxRQUFRLEVBQUUsT0FBTztvQ0FDakIsVUFBVSxFQUFFO3dDQUNaOzRDQUNFLGlCQUFpQixFQUFFLGlDQUFpQzt5Q0FDckQ7d0NBQ0Q7NENBQ0UsVUFBVSxFQUFFO2dEQUNaLEVBQUU7Z0RBQ0Y7b0RBQ0U7d0RBQ0EsaUJBQWlCLEVBQUUsaUNBQWlDO3FEQUNuRDtvREFDRCxJQUFJO2lEQUNMOzZDQUNBO3lDQUNGO3FDQUNBO2lDQUNGOzZCQUNBOzRCQUNELFNBQVMsRUFBRSxZQUFZO3lCQUN4Qjt3QkFDRCxZQUFZLEVBQUUseUNBQXlDO3dCQUN2RCxPQUFPLEVBQUU7NEJBQ1A7Z0NBQ0EsS0FBSyxFQUFFLDRCQUE0Qjs2QkFDbEM7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7YUFDQTtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyREFBMkQsQ0FBQyxJQUFVO1FBQ3BFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFaEQsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDbkUsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDckYsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGtCQUFrQjtRQUU3RyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDYixrQkFBa0IsRUFBRTtvQkFDbEIsTUFBTSxFQUFFLGlCQUFpQjtvQkFDekIsZ0JBQWdCLEVBQUUsUUFBUTtpQkFDM0I7YUFDQTtZQUNELFNBQVMsRUFBRTtnQkFDWCxXQUFXLEVBQUU7b0JBQ1gsT0FBTyxFQUFFO3dCQUNULFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNBLGFBQWE7Z0NBQ2I7b0NBQ0UsS0FBSyxFQUFFLGFBQWE7aUNBQ3JCO2dDQUNELEdBQUc7Z0NBQ0g7b0NBQ0UsS0FBSyxFQUFFLGdCQUFnQjtpQ0FDeEI7Z0NBQ0QsR0FBRztnQ0FDSDtvQ0FDRSxLQUFLLEVBQUUsa0JBQWtCO2lDQUMxQjs2QkFDQTt5QkFDRjtxQkFDQTtpQkFDRjtnQkFDRCxXQUFXLEVBQUU7b0JBQ1gsT0FBTyxFQUFFO3dCQUNULFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNBLGFBQWE7Z0NBQ2I7b0NBQ0UsS0FBSyxFQUFFLGFBQWE7aUNBQ3JCO2dDQUNELEdBQUc7Z0NBQ0g7b0NBQ0UsS0FBSyxFQUFFLGdCQUFnQjtpQ0FDeEI7Z0NBQ0QsR0FBRztnQ0FDSDtvQ0FDRSxLQUFLLEVBQUUsa0JBQWtCO2lDQUMxQjtnQ0FDRCxjQUFjOzZCQUNiO3lCQUNGO3FCQUNBO2lCQUNGO2dCQUNELGFBQWEsRUFBRTtvQkFDYixPQUFPLEVBQUU7d0JBQ1QsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0EsYUFBYTtnQ0FDYjtvQ0FDRSxLQUFLLEVBQUUsYUFBYTtpQ0FDckI7Z0NBQ0QsR0FBRztnQ0FDSDtvQ0FDRSxLQUFLLEVBQUUsZ0JBQWdCO2lDQUN4QjtnQ0FDRCxHQUFHO2dDQUNIO29DQUNFLEtBQUssRUFBRSxrQkFBa0I7aUNBQzFCO2dDQUNELGdCQUFnQjs2QkFDZjt5QkFDRjtxQkFDQTtpQkFDRjthQUNBO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1CQUFtQixFQUFFO1FBQ25CLGdEQUFnRCxDQUFDLElBQVU7WUFDekQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFFekMsT0FBTztZQUNQLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1lBRTNCLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3JELGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFLGNBQWM7NEJBQ3hCLFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUUsR0FBRzs0QkFDaEIsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFLENBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxXQUFXLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxJQUFJLENBQUUsQ0FBRSxFQUFFO3lCQUN2RjtxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxpRUFBaUUsQ0FBQyxJQUFVO1lBQzFFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxNQUFNLENBQUMsaUJBQWlCLENBQUMscUJBQXFCLENBQUMsQ0FBQztZQUVoRCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRSxjQUFjOzRCQUN4QixRQUFRLEVBQUUsT0FBTzs0QkFDakIsV0FBVyxFQUFFLEdBQUc7NEJBQ2hCLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFFLEVBQUUsRUFBRSxDQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsV0FBVyxFQUFFLEtBQUssQ0FBRSxFQUFFLEVBQUUsc0JBQXNCLENBQUUsQ0FBRSxFQUFFO3lCQUN6RztxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw0REFBNEQsQ0FBQyxJQUFVO1lBQ3JFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxNQUFNLENBQUMsaUJBQWlCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRSxjQUFjLENBQUMsQ0FBQztZQUU5RCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO2dCQUNyRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRSxDQUFFLGNBQWMsRUFBRSxjQUFjLENBQUU7NEJBQzVDLFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUUsR0FBRzs0QkFDaEIsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFLENBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxXQUFXLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxJQUFJLENBQUUsQ0FBRSxFQUFFO3lCQUN2RjtxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwwREFBMEQsQ0FBQyxJQUFVO1lBQ25FLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBRXpDLE9BQU87WUFDUCxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztZQUM3QyxTQUFTLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRSxFQUFFLGNBQWMsRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7WUFFM0UsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtnQkFDckQsZ0JBQWdCLEVBQUU7b0JBQ2hCLFdBQVcsRUFBRTt3QkFDWDs0QkFDRSxRQUFRLEVBQUUsY0FBYzs0QkFDeEIsUUFBUSxFQUFFLE9BQU87NEJBQ2pCLFdBQVcsRUFBRSxHQUFHOzRCQUNoQixVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBRSxFQUFFLEVBQUUsQ0FBRSxFQUFFLFlBQVksRUFBRSxDQUFFLFdBQVcsRUFBRSxLQUFLLENBQUUsRUFBRSxFQUFFLElBQUksQ0FBRSxDQUFFLEVBQUU7NEJBQ3RGLFdBQVcsRUFBRTtnQ0FDWCxXQUFXLEVBQUUsRUFBRSxjQUFjLEVBQUUsaUJBQWlCLEVBQUU7NkJBQ25EO3lCQUNGO3FCQUNGO29CQUNELFNBQVMsRUFBRSxZQUFZO2lCQUN4QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDhDQUE4QyxDQUFDLElBQVU7WUFDdkQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUM5QyxpQkFBaUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBRSxDQUFDO2FBQ3pFLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLG1CQUFtQixDQUFDLENBQUM7WUFFbkUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsdUJBQXVCLEVBQUU7UUFDdkIsZ0JBQWdCLENBQUMsSUFBVTtZQUN6QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtnQkFDOUIsb0JBQW9CLEVBQUUsYUFBYTthQUNwQyxDQUFDLENBQUM7WUFDSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9DLG9CQUFvQixFQUFFO29CQUNwQixhQUFhLEVBQUUsYUFBYTtpQkFDN0I7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxzQ0FBc0MsQ0FBQyxJQUFVO1lBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUM5QixvQkFBb0IsRUFBRSxZQUFZO2lCQUNuQyxDQUFDLENBQUM7WUFDTCxDQUFDLEVBQUUscUVBQXFFLENBQUMsQ0FBQztZQUMxRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsc0JBQXNCLENBQUMsSUFBVTtZQUMvQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtnQkFDOUIsb0JBQW9CLEVBQUUsYUFBYTtnQkFDbkMsb0JBQW9CLEVBQUUsWUFBWTthQUNuQyxDQUFDLENBQUM7WUFDSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9DLG9CQUFvQixFQUFFO29CQUNwQixhQUFhLEVBQUUsYUFBYTtvQkFDNUIsYUFBYSxFQUFFLFlBQVk7aUJBQzVCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsd0JBQXdCLENBQUMsSUFBVTtZQUNqQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtnQkFDN0Msb0JBQW9CLEVBQUUsWUFBWTthQUNuQyxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsaUJBQWlCLEVBQUUsWUFBWSxDQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3BILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSwgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IGttcyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1rbXMnKTtcbmltcG9ydCBjZGsgPSByZXF1aXJlKCdAYXdzLWNkay9jZGsnKTtcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgczMgPSByZXF1aXJlKCcuLi9saWInKTtcblxuLy8gdG8gbWFrZSBpdCBlYXN5IHRvIGNvcHkgJiBwYXN0ZSBmcm9tIG91dHB1dDpcbi8vIHRzbGludDpkaXNhYmxlOm9iamVjdC1saXRlcmFsLWtleS1xdW90ZXNcblxuZXhwb3J0ID0ge1xuICAnZGVmYXVsdCBidWNrZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCI6IHtcbiAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2J1Y2tldCB3aXRob3V0IGVuY3J5cHRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7XG4gICAgICBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlVuZW5jcnlwdGVkXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlMzOjpCdWNrZXRcIixcbiAgICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYnVja2V0IHdpdGggbWFuYWdlZCBlbmNyeXB0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015QnVja2V0Jywge1xuICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5LbXNNYW5hZ2VkXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlMzOjpCdWNrZXRcIixcbiAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIkJ1Y2tldEVuY3J5cHRpb25cIjoge1xuICAgICAgICAgIFwiU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiU2VydmVyU2lkZUVuY3J5cHRpb25CeURlZmF1bHRcIjoge1xuICAgICAgICAgICAgICBcIlNTRUFsZ29yaXRobVwiOiBcImF3czprbXNcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmYWlscyBpZiBlbmNyeXB0aW9uIGtleSBpcyB1c2VkIHdpdGggbWFuYWdlZCBlbmNyeXB0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgbXlLZXkgPSBuZXcga21zLkVuY3J5cHRpb25LZXkoc3RhY2ssICdNeUtleScpO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015QnVja2V0Jywge1xuICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5LbXNNYW5hZ2VkLFxuICAgICAgZW5jcnlwdGlvbktleTogbXlLZXlcbiAgICB9KSwgL2VuY3J5cHRpb25LZXkgaXMgc3BlY2lmaWVkLCBzbyAnZW5jcnlwdGlvbicgbXVzdCBiZSBzZXQgdG8gS01TLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZmFpbHMgaWYgZW5jcnlwdGlvbiBrZXkgaXMgdXNlZCB3aXRoIGVuY3J5cHRpb24gc2V0IHRvIHVuZW5jcnlwdGVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgbXlLZXkgPSBuZXcga21zLkVuY3J5cHRpb25LZXkoc3RhY2ssICdNeUtleScpO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015QnVja2V0Jywge1xuICAgICAgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5VbmVuY3J5cHRlZCxcbiAgICAgIGVuY3J5cHRpb25LZXk6IG15S2V5XG4gICAgfSksIC9lbmNyeXB0aW9uS2V5IGlzIHNwZWNpZmllZCwgc28gJ2VuY3J5cHRpb24nIG11c3QgYmUgc2V0IHRvIEtNUy8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2VuY3J5cHRpb25LZXkgY2FuIHNwZWNpZnkga21zIGtleScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgY29uc3QgZW5jcnlwdGlvbktleSA9IG5ldyBrbXMuRW5jcnlwdGlvbktleShzdGFjaywgJ015S2V5JywgeyBkZXNjcmlwdGlvbjogJ2hlbGxvLCB3b3JsZCcgfSk7XG5cbiAgICBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7IGVuY3J5cHRpb25LZXksIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uS21zIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgXCJNeUtleTZBQjI5RkE2XCI6IHtcbiAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpLTVM6OktleVwiLFxuICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiRGVzY3JpcHRpb25cIjogXCJoZWxsbywgd29ybGRcIixcbiAgICAgICAgICBcIktleVBvbGljeVwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgXCJBY3Rpb25cIjogW1xuICAgICAgICAgICAgICBcImttczpDcmVhdGUqXCIsXG4gICAgICAgICAgICAgIFwia21zOkRlc2NyaWJlKlwiLFxuICAgICAgICAgICAgICBcImttczpFbmFibGUqXCIsXG4gICAgICAgICAgICAgIFwia21zOkxpc3QqXCIsXG4gICAgICAgICAgICAgIFwia21zOlB1dCpcIixcbiAgICAgICAgICAgICAgXCJrbXM6VXBkYXRlKlwiLFxuICAgICAgICAgICAgICBcImttczpSZXZva2UqXCIsXG4gICAgICAgICAgICAgIFwia21zOkRpc2FibGUqXCIsXG4gICAgICAgICAgICAgIFwia21zOkdldCpcIixcbiAgICAgICAgICAgICAgXCJrbXM6RGVsZXRlKlwiLFxuICAgICAgICAgICAgICBcImttczpTY2hlZHVsZUtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICAgIFwia21zOkNhbmNlbEtleURlbGV0aW9uXCJcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICBcIlByaW5jaXBhbFwiOiB7XG4gICAgICAgICAgICAgIFwiQVdTXCI6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgXCJSZWZcIjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBcIjppYW06OlwiLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXCI6cm9vdFwiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIlJlc291cmNlXCI6IFwiKlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIlxuICAgICAgICB9LFxuICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlMzOjpCdWNrZXRcIixcbiAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIkJ1Y2tldEVuY3J5cHRpb25cIjoge1xuICAgICAgICAgIFwiU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiU2VydmVyU2lkZUVuY3J5cHRpb25CeURlZmF1bHRcIjoge1xuICAgICAgICAgICAgICBcIktNU01hc3RlcktleUlEXCI6IHtcbiAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcIk15S2V5NkFCMjlGQTZcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXCJTU0VBbGdvcml0aG1cIjogXCJhd3M6a21zXCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBcIkRlbGV0aW9uUG9saWN5XCI6IFwiUmV0YWluXCIsXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYnVja2V0IHdpdGggdmVyc2lvbmluZyB0dXJuZWQgb24nKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7XG4gICAgICB2ZXJzaW9uZWQ6IHRydWVcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiVmVyc2lvbmluZ0NvbmZpZ3VyYXRpb25cIjoge1xuICAgICAgICAgICAgICBcIlN0YXR1c1wiOiBcIkVuYWJsZWRcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdidWNrZXQgd2l0aCBibG9jayBwdWJsaWMgYWNjZXNzIHNldCB0byBCbG9ja0FsbCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHtcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBzMy5CbG9ja1B1YmxpY0FjY2Vzcy5CbG9ja0FsbCxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiUHVibGljQWNjZXNzQmxvY2tDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgICAgICAgICAgXCJCbG9ja1B1YmxpY0FjbHNcIjogdHJ1ZSxcbiAgICAgICAgICAgICAgXCJCbG9ja1B1YmxpY1BvbGljeVwiOiB0cnVlLFxuICAgICAgICAgICAgICBcIklnbm9yZVB1YmxpY0FjbHNcIjogdHJ1ZSxcbiAgICAgICAgICAgICAgXCJSZXN0cmljdFB1YmxpY0J1Y2tldHNcIjogdHJ1ZSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICBcIkRlbGV0aW9uUG9saWN5XCI6IFwiUmV0YWluXCIsXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYnVja2V0IHdpdGggYmxvY2sgcHVibGljIGFjY2VzcyBzZXQgdG8gQmxvY2tBY2xzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015QnVja2V0Jywge1xuICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHMzLkJsb2NrUHVibGljQWNjZXNzLkJsb2NrQWNscyxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiUHVibGljQWNjZXNzQmxvY2tDb25maWd1cmF0aW9uXCI6IHtcbiAgICAgICAgICAgICAgXCJCbG9ja1B1YmxpY0FjbHNcIjogdHJ1ZSxcbiAgICAgICAgICAgICAgXCJJZ25vcmVQdWJsaWNBY2xzXCI6IHRydWUsXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2J1Y2tldCB3aXRoIGN1c3RvbSBibG9jayBwdWJsaWMgYWNjZXNzIHNldHRpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7XG4gICAgICBibG9ja1B1YmxpY0FjY2VzczogbmV3IHMzLkJsb2NrUHVibGljQWNjZXNzKHsgcmVzdHJpY3RQdWJsaWNCdWNrZXRzOiB0cnVlIH0pXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6UzM6OkJ1Y2tldFwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgICBcIlB1YmxpY0FjY2Vzc0Jsb2NrQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICAgICAgICAgIFwiUmVzdHJpY3RQdWJsaWNCdWNrZXRzXCI6IHRydWUsXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiLFxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Blcm1pc3Npb25zJzoge1xuXG4gICAgJ2FkZFBlcm1pc3Npb24gY3JlYXRlcyBhIGJ1Y2tldCBwb2xpY3knKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015QnVja2V0JywgeyBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlVuZW5jcnlwdGVkIH0pO1xuXG4gICAgICBidWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZFJlc291cmNlKCdmb28nKS5hZGRBY3Rpb24oJ2JhcicpKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTXlCdWNrZXRQb2xpY3lFN0ZCQUM3QlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0UG9saWN5XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiQnVja2V0XCI6IHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlCdWNrZXRGNjhGM0ZGMFwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJiYXJcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IFwiZm9vXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdmb3JCdWNrZXQgcmV0dXJucyBhIHBlcm1pc3Npb24gc3RhdGVtZW50IGFzc29jaWF0ZWQgd2l0aCB0aGUgYnVja2V0XFwncyBBUk4nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7IGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uVW5lbmNyeXB0ZWQgfSk7XG5cbiAgICAgIGNvbnN0IHggPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZFJlc291cmNlKGJ1Y2tldC5idWNrZXRBcm4pLmFkZEFjdGlvbignczM6TGlzdEJ1Y2tldCcpO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChidWNrZXQubm9kZS5yZXNvbHZlKHgpLCB7XG4gICAgICAgIEFjdGlvbjogJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgIFJlc291cmNlOiB7ICdGbjo6R2V0QXR0JzogWyAnTXlCdWNrZXRGNjhGM0ZGMCcsICdBcm4nIF0gfVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYXJuRm9yT2JqZWN0cyByZXR1cm5zIGEgcGVybWlzc2lvbiBzdGF0ZW1lbnQgYXNzb2NpYXRlZCB3aXRoIG9iamVjdHMgaW4gdGhlIGJ1Y2tldCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHsgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5VbmVuY3J5cHRlZCB9KTtcblxuICAgICAgY29uc3QgcCA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KCkuYWRkUmVzb3VyY2UoYnVja2V0LmFybkZvck9iamVjdHMoJ2hlbGxvL3dvcmxkJykpLmFkZEFjdGlvbignczM6R2V0T2JqZWN0Jyk7XG5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGJ1Y2tldC5ub2RlLnJlc29sdmUocCksIHtcbiAgICAgICAgQWN0aW9uOiAnczM6R2V0T2JqZWN0JyxcbiAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgWyB7ICdGbjo6R2V0QXR0JzogWyAnTXlCdWNrZXRGNjhGM0ZGMCcsICdBcm4nIF0gfSwgJy9oZWxsby93b3JsZCcgXVxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYXJuRm9yT2JqZWN0cyBhY2NlcHRzIG11bHRpcGxlIGFyZ3VtZW50cyBhbmQgRm5Db25jYXRzIHRoZW0nKHRlc3Q6IFRlc3QpIHtcblxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHsgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5VbmVuY3J5cHRlZCB9KTtcblxuICAgICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ015VXNlcicpO1xuICAgICAgY29uc3QgdGVhbSA9IG5ldyBpYW0uR3JvdXAoc3RhY2ssICdNeVRlYW0nKTtcblxuICAgICAgY29uc3QgcmVzb3VyY2UgPSBidWNrZXQuYXJuRm9yT2JqZWN0cygnaG9tZS8nLCB0ZWFtLmdyb3VwTmFtZSwgJy8nLCB1c2VyLnVzZXJOYW1lLCAnLyonKTtcbiAgICAgIGNvbnN0IHAgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZFJlc291cmNlKHJlc291cmNlKS5hZGRBY3Rpb24oJ3MzOkdldE9iamVjdCcpO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChidWNrZXQubm9kZS5yZXNvbHZlKHApLCB7XG4gICAgICAgIEFjdGlvbjogJ3MzOkdldE9iamVjdCcsXG4gICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAnJyxcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgeyAnRm46OkdldEF0dCc6IFsgJ015QnVja2V0RjY4RjNGRjAnLCAnQXJuJyBdIH0sXG4gICAgICAgICAgICAgICcvaG9tZS8nLFxuICAgICAgICAgICAgICB7IFJlZjogJ015VGVhbTAxREQ2Njg1JyB9LFxuICAgICAgICAgICAgICAnLycsXG4gICAgICAgICAgICAgIHsgUmVmOiAnTXlVc2VyREM0NTAyOEInIH0sXG4gICAgICAgICAgICAgICcvKidcbiAgICAgICAgICAgIF1cbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ3JlbW92YWwgcG9saWN5IGNhbiBiZSB1c2VkIHRvIHNwZWNpZnkgYmVoYXZpb3IgdXBvbiBkZWxldGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7IHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5Lk9ycGhhbiwgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5VbmVuY3J5cHRlZCB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgTXlCdWNrZXRGNjhGM0ZGMDoge1xuICAgICAgICAgIFR5cGU6ICdBV1M6OlMzOjpCdWNrZXQnLFxuICAgICAgICAgIERlbGV0aW9uUG9saWN5OiAnUmV0YWluJ1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW1wb3J0L2V4cG9ydCc6IHtcbiAgICAnZXhwb3J0IGNyZWF0ZXMgb3V0cHV0cyBmb3IgdGhlIGJ1Y2tldCBhdHRyaWJ1dGVzIGFuZCByZXR1cm5zIGEgcmVmIG9iamVjdCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgJ015U3RhY2snKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcpO1xuICAgICAgY29uc3QgYnVja2V0UmVmID0gYnVja2V0LmV4cG9ydCgpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoYnVja2V0Lm5vZGUucmVzb2x2ZShidWNrZXRSZWYpLCB7XG4gICAgICAgIGJ1Y2tldEFybjogeyAnRm46OkltcG9ydFZhbHVlJzogJ015U3RhY2s6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4QycgfSxcbiAgICAgICAgYnVja2V0TmFtZTogeyAnRm46OkltcG9ydFZhbHVlJzogJ015U3RhY2s6TXlCdWNrZXRCdWNrZXROYW1lOEEwMjcwMTQnIH0sXG4gICAgICAgIGJ1Y2tldERvbWFpbk5hbWU6IHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdNeVN0YWNrOk15QnVja2V0RG9tYWluTmFtZUY3NkI5QTdBJyB9LFxuICAgICAgICBidWNrZXRXZWJzaXRlVXJsOiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnTXlTdGFjazpNeUJ1Y2tldFdlYnNpdGVVUkw5QzIyMjc4OCcgfVxuICAgICAgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3JlZnMgd2lsbCBpbmNsdWRlIHRoZSBidWNrZXRcXCdzIGVuY3J5cHRpb24ga2V5IGlmIGRlZmluZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsICdNeVN0YWNrJyk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnLCB7IGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uS21zIH0pO1xuICAgICAgY29uc3QgYnVja2V0UmVmID0gYnVja2V0LmV4cG9ydCgpO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoYnVja2V0Lm5vZGUucmVzb2x2ZShidWNrZXRSZWYpLCB7XG4gICAgICAgIGJ1Y2tldEFybjogeyAnRm46OkltcG9ydFZhbHVlJzogJ015U3RhY2s6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4QycgfSxcbiAgICAgICAgYnVja2V0TmFtZTogeyAnRm46OkltcG9ydFZhbHVlJzogJ015U3RhY2s6TXlCdWNrZXRCdWNrZXROYW1lOEEwMjcwMTQnIH0sXG4gICAgICAgIGJ1Y2tldERvbWFpbk5hbWU6IHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdNeVN0YWNrOk15QnVja2V0RG9tYWluTmFtZUY3NkI5QTdBJyB9LFxuICAgICAgICBidWNrZXRXZWJzaXRlVXJsOiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnTXlTdGFjazpNeUJ1Y2tldFdlYnNpdGVVUkw5QzIyMjc4OCcgfVxuICAgICAgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3N0YXRpYyBpbXBvcnQocmVmKSBhbGxvd3MgaW1wb3J0aW5nIGFuIGV4dGVybmFsL2V4aXN0aW5nIGJ1Y2tldCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGJ1Y2tldEFybiA9ICdhcm46YXdzOnMzOjo6bXktYnVja2V0JztcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IHMzLkJ1Y2tldC5pbXBvcnQoc3RhY2ssICdJbXBvcnRlZEJ1Y2tldCcsIHsgYnVja2V0QXJuIH0pO1xuXG4gICAgICAvLyB0aGlzIGlzIGEgbm8tb3Agc2luY2UgdGhlIGJ1Y2tldCBpcyBleHRlcm5hbFxuICAgICAgYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoKS5hZGRSZXNvdXJjZSgnZm9vJykuYWRkQWN0aW9uKCdiYXInKSk7XG5cbiAgICAgIGNvbnN0IHAgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZFJlc291cmNlKGJ1Y2tldC5idWNrZXRBcm4pLmFkZEFjdGlvbignczM6TGlzdEJ1Y2tldCcpO1xuXG4gICAgICAvLyBpdCBpcyBwb3NzaWJsZSB0byBvYnRhaW4gYSBwZXJtaXNzaW9uIHN0YXRlbWVudCBmb3IgYSByZWZcbiAgICAgIHRlc3QuZGVlcEVxdWFsKGJ1Y2tldC5ub2RlLnJlc29sdmUocCksIHtcbiAgICAgICAgQWN0aW9uOiAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgUmVzb3VyY2U6ICdhcm46YXdzOnMzOjo6bXktYnVja2V0J1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKGJ1Y2tldC5idWNrZXRBcm4sIGJ1Y2tldEFybik7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChidWNrZXQubm9kZS5yZXNvbHZlKGJ1Y2tldC5idWNrZXROYW1lKSwgJ215LWJ1Y2tldCcpO1xuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLCB7fSwgJ3RoZSByZWYgaXMgbm90IGEgcmVhbCByZXNvdXJjZScpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpbXBvcnQgY2FuIGFsc28gYmUgdXNlZCB0byBpbXBvcnQgYXJiaXRyYXJ5IEFSTnMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYnVja2V0ID0gczMuQnVja2V0LmltcG9ydChzdGFjaywgJ0ltcG9ydGVkQnVja2V0JywgeyBidWNrZXRBcm46ICdhcm46YXdzOnMzOjo6bXktYnVja2V0JyB9KTtcbiAgICAgIGJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KCkuYWRkQWxsUmVzb3VyY2VzKCkuYWRkQWN0aW9uKCcqJykpO1xuXG4gICAgICAvLyBhdCB0aGlzIHBvaW50IHdlIHRlY2huaWNhbGx5IGRpZG4ndCBjcmVhdGUgYW55IHJlc291cmNlcyBpbiB0aGUgY29uc3VtaW5nIHN0YWNrLlxuICAgICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHt9KTtcblxuICAgICAgLy8gYnV0IG5vdyB3ZSBjYW4gcmVmZXJlbmNlIHRoZSBidWNrZXRcbiAgICAgIC8vIHlvdSBjYW4gZXZlbiB1c2UgdGhlIGJ1Y2tldCBuYW1lLCB3aGljaCB3aWxsIGJlIGV4dHJhY3RlZCBmcm9tIHRoZSBhcm4gcHJvdmlkZWQuXG4gICAgICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnTXlVc2VyJyk7XG4gICAgICB1c2VyLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KClcbiAgICAgICAgLmFkZFJlc291cmNlKGJ1Y2tldC5hcm5Gb3JPYmplY3RzKCdteS9mb2xkZXIvJywgYnVja2V0LmJ1Y2tldE5hbWUpKVxuICAgICAgICAuYWRkQWN0aW9uKCdzMzoqJykpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlVc2VyREM0NTAyOEJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpVc2VyXCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJNeVVzZXJEZWZhdWx0UG9saWN5N0I4OTc0MjZcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpQb2xpY3lcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcInMzOipcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IFwiYXJuOmF3czpzMzo6Om15LWJ1Y2tldC9teS9mb2xkZXIvbXktYnVja2V0XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15VXNlckRlZmF1bHRQb2xpY3k3Qjg5NzQyNlwiLFxuICAgICAgICAgIFwiVXNlcnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJNeVVzZXJEQzQ1MDI4QlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhpcyBpcyBob3cgZXhwb3J0L2ltcG9ydCB3b3JrIHRvZ2V0aGVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjazEgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgJ1MxJyk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrMSwgJ015QnVja2V0Jyk7XG4gICAgICBjb25zdCBidWNrZXRSZWYgPSBidWNrZXQuZXhwb3J0KCk7XG5cbiAgICAgIGV4cGVjdChzdGFjazEpLnRvTWF0Y2goe1xuICAgICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBcIk91dHB1dHNcIjoge1xuICAgICAgICBcIk15QnVja2V0QnVja2V0QXJuRTI2MDU1OENcIjoge1xuICAgICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkV4cG9ydFwiOiB7XG4gICAgICAgICAgXCJOYW1lXCI6IFwiUzE6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4Q1wiXG4gICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBcIk15QnVja2V0QnVja2V0TmFtZThBMDI3MDE0XCI6IHtcbiAgICAgICAgICBcIlZhbHVlXCI6IHtcbiAgICAgICAgICBcIlJlZlwiOiBcIk15QnVja2V0RjY4RjNGRjBcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgIFwiTmFtZVwiOiBcIlMxOk15QnVja2V0QnVja2V0TmFtZThBMDI3MDE0XCJcbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiTXlCdWNrZXREb21haW5OYW1lRjc2QjlBN0FcIjoge1xuICAgICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCIsXG4gICAgICAgICAgICAgIFwiRG9tYWluTmFtZVwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkV4cG9ydFwiOiB7XG4gICAgICAgICAgICBcIk5hbWVcIjogXCJTMTpNeUJ1Y2tldERvbWFpbk5hbWVGNzZCOUE3QVwiXG4gICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBcIk15QnVja2V0V2Vic2l0ZVVSTDlDMjIyNzg4XCI6IHtcbiAgICAgICAgICBcIlZhbHVlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiLFxuICAgICAgICAgICAgICBcIldlYnNpdGVVUkxcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiUzE6TXlCdWNrZXRXZWJzaXRlVVJMOUMyMjI3ODhcIlxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgY29uc3Qgc3RhY2syID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsICdTMicpO1xuICAgICAgY29uc3QgaW1wb3J0ZWRCdWNrZXQgPSBzMy5CdWNrZXQuaW1wb3J0KHN0YWNrMiwgJ0ltcG9ydGVkQnVja2V0JywgYnVja2V0UmVmKTtcbiAgICAgIGNvbnN0IHVzZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2syLCAnTXlVc2VyJyk7XG4gICAgICBpbXBvcnRlZEJ1Y2tldC5ncmFudFJlYWQodXNlcik7XG5cbiAgICAgIGV4cGVjdChzdGFjazIpLnRvTWF0Y2goe1xuICAgICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlVc2VyREM0NTAyOEJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpVc2VyXCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJNeVVzZXJEZWZhdWx0UG9saWN5N0I4OTc0MjZcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpQb2xpY3lcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgIFwiczM6R2V0T2JqZWN0KlwiLFxuICAgICAgICAgICAgICBcInMzOkdldEJ1Y2tldCpcIixcbiAgICAgICAgICAgICAgXCJzMzpMaXN0KlwiXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIkZuOjpJbXBvcnRWYWx1ZVwiOiBcIlMxOk15QnVja2V0QnVja2V0QXJuRTI2MDU1OENcIlxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICB7IFwiRm46OkltcG9ydFZhbHVlXCI6IFwiUzE6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4Q1wiIH0sXG4gICAgICAgICAgICAgICAgICBcIi8qXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15VXNlckRlZmF1bHRQb2xpY3k3Qjg5NzQyNlwiLFxuICAgICAgICAgIFwiVXNlcnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJNeVVzZXJEQzQ1MDI4QlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuICB9LFxuXG4gICdncmFudFJlYWQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCByZWFkZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2ssICdSZWFkZXInKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnKTtcbiAgICBidWNrZXQuZ3JhbnRSZWFkKHJlYWRlcik7XG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgIFwiUmVhZGVyRjdCRjE4OURcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6VXNlclwiXG4gICAgICB9LFxuICAgICAgXCJSZWFkZXJEZWZhdWx0UG9saWN5MTUxRjM4MThcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICBcInMzOkdldE9iamVjdCpcIixcbiAgICAgICAgICAgIFwiczM6R2V0QnVja2V0KlwiLFxuICAgICAgICAgICAgXCJzMzpMaXN0KlwiXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBcIi8qXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH0sXG4gICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIlJlYWRlckRlZmF1bHRQb2xpY3kxNTFGMzgxOFwiLFxuICAgICAgICBcIlVzZXJzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgXCJSZWZcIjogXCJSZWFkZXJGN0JGMTg5RFwiXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlMzOjpCdWNrZXRcIixcbiAgICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiXG4gICAgICB9LFxuICAgICAgfVxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdncmFudFJlYWRXcml0ZSc6IHtcbiAgICAnY2FuIGJlIHVzZWQgdG8gZ3JhbnQgcmVjaXByb2NhbCBwZXJtaXNzaW9ucyB0byBhbiBpZGVudGl0eScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnKTtcbiAgICAgIGNvbnN0IHVzZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2ssICdNeVVzZXInKTtcbiAgICAgIGJ1Y2tldC5ncmFudFJlYWRXcml0ZSh1c2VyKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTXlVc2VyREM0NTAyOEJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpVc2VyXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTXlVc2VyRGVmYXVsdFBvbGljeTdCODk3NDI2XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgICAgICBcInMzOkdldE9iamVjdCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkdldEJ1Y2tldCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkxpc3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpEZWxldGVPYmplY3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpQdXRPYmplY3QqXCIsXG4gICAgICAgICAgICAgICAgXCJzMzpBYm9ydCpcIlxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCIsXG4gICAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCIsXG4gICAgICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiLypcIlxuICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15VXNlckRlZmF1bHRQb2xpY3k3Qjg5NzQyNlwiLFxuICAgICAgICAgICAgXCJVc2Vyc1wiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlVc2VyREM0NTAyOEJcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2lmIGFuIGVuY3J5cHRpb24ga2V5IGlzIGluY2x1ZGVkLCBlbmNyeXB0L2RlY3J5cHQgcGVybWlzc2lvbnMgYXJlIGFsc28gYWRkZWQgYm90aCB3YXlzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHsgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5LbXMgfSk7XG4gICAgICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnTXlVc2VyJyk7XG4gICAgICBidWNrZXQuZ3JhbnRSZWFkV3JpdGUodXNlcik7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgICBcIk15QnVja2V0S2V5QzE3MTMwQ0ZcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6S01TOjpLZXlcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJEZXNjcmlwdGlvblwiOiBcIkNyZWF0ZWQgYnkgTXlCdWNrZXRcIixcbiAgICAgICAgICAgIFwiS2V5UG9saWN5XCI6IHtcbiAgICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgICAgXCJrbXM6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgICAgIFwia21zOkRlc2NyaWJlKlwiLFxuICAgICAgICAgICAgICAgIFwia21zOkVuYWJsZSpcIixcbiAgICAgICAgICAgICAgICBcImttczpMaXN0KlwiLFxuICAgICAgICAgICAgICAgIFwia21zOlB1dCpcIixcbiAgICAgICAgICAgICAgICBcImttczpVcGRhdGUqXCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6UmV2b2tlKlwiLFxuICAgICAgICAgICAgICAgIFwia21zOkRpc2FibGUqXCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6R2V0KlwiLFxuICAgICAgICAgICAgICAgIFwia21zOkRlbGV0ZSpcIixcbiAgICAgICAgICAgICAgICBcImttczpTY2hlZHVsZUtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6Q2FuY2VsS2V5RGVsZXRpb25cIlxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IHtcbiAgICAgICAgICAgICAgICBcIkFXU1wiOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpQYXJ0aXRpb25cIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiOmlhbTo6XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiOnJvb3RcIlxuICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogXCIqXCJcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBbXG4gICAgICAgICAgICAgICAgXCJrbXM6RGVjcnlwdFwiLFxuICAgICAgICAgICAgICAgIFwia21zOkRlc2NyaWJlS2V5XCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6RW5jcnlwdFwiLFxuICAgICAgICAgICAgICAgIFwia21zOlJlRW5jcnlwdCpcIixcbiAgICAgICAgICAgICAgICBcImttczpHZW5lcmF0ZURhdGFLZXkqXCIsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJQcmluY2lwYWxcIjoge1xuICAgICAgICAgICAgICAgIFwiQVdTXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgXCJNeVVzZXJEQzQ1MDI4QlwiLFxuICAgICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogXCIqXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJEZWxldGlvblBvbGljeVwiOiBcIlJldGFpblwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6UzM6OkJ1Y2tldFwiLFxuICAgICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJCdWNrZXRFbmNyeXB0aW9uXCI6IHtcbiAgICAgICAgICAgIFwiU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uXCI6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIlNlcnZlclNpZGVFbmNyeXB0aW9uQnlEZWZhdWx0XCI6IHtcbiAgICAgICAgICAgICAgICBcIktNU01hc3RlcktleUlEXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgXCJNeUJ1Y2tldEtleUMxNzEzMENGXCIsXG4gICAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXCJTU0VBbGdvcml0aG1cIjogXCJhd3M6a21zXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeVVzZXJEQzQ1MDI4QlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpJQU06OlVzZXJcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeVVzZXJEZWZhdWx0UG9saWN5N0I4OTc0MjZcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpQb2xpY3lcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogW1xuICAgICAgICAgICAgICAgIFwiczM6R2V0T2JqZWN0KlwiLFxuICAgICAgICAgICAgICAgIFwiczM6R2V0QnVja2V0KlwiLFxuICAgICAgICAgICAgICAgIFwiczM6TGlzdCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkRlbGV0ZU9iamVjdCpcIixcbiAgICAgICAgICAgICAgICBcInMzOlB1dE9iamVjdCpcIixcbiAgICAgICAgICAgICAgICBcInMzOkFib3J0KlwiXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCIvKlwiXG4gICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgICAgICBcImttczpEZWNyeXB0XCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6RGVzY3JpYmVLZXlcIixcbiAgICAgICAgICAgICAgICBcImttczpFbmNyeXB0XCIsXG4gICAgICAgICAgICAgICAgXCJrbXM6UmVFbmNyeXB0KlwiLFxuICAgICAgICAgICAgICAgIFwia21zOkdlbmVyYXRlRGF0YUtleSpcIixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgIFwiTXlCdWNrZXRLZXlDMTcxMzBDRlwiLFxuICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15VXNlckRlZmF1bHRQb2xpY3k3Qjg5NzQyNlwiLFxuICAgICAgICAgICAgXCJVc2Vyc1wiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlVc2VyREM0NTAyOEJcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdtb3JlIGdyYW50cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHsgZW5jcnlwdGlvbjogczMuQnVja2V0RW5jcnlwdGlvbi5LbXMgfSk7XG4gICAgY29uc3QgcHV0dGVyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnUHV0dGVyJyk7XG4gICAgY29uc3Qgd3JpdGVyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnV3JpdGVyJyk7XG4gICAgY29uc3QgZGVsZXRlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ0RlbGV0ZXInKTtcblxuICAgIGJ1Y2tldC5ncmFudFB1dChwdXR0ZXIpO1xuICAgIGJ1Y2tldC5ncmFudFdyaXRlKHdyaXRlcik7XG4gICAgYnVja2V0LmdyYW50RGVsZXRlKGRlbGV0ZXIpO1xuXG4gICAgY29uc3QgcmVzb3VyY2VzID0gU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKS5SZXNvdXJjZXM7XG4gICAgY29uc3QgYWN0aW9ucyA9IChpZDogc3RyaW5nKSA9PiByZXNvdXJjZXNbaWRdLlByb3BlcnRpZXMuUG9saWN5RG9jdW1lbnQuU3RhdGVtZW50WzBdLkFjdGlvbjtcblxuICAgIHRlc3QuZGVlcEVxdWFsKGFjdGlvbnMoJ1dyaXRlckRlZmF1bHRQb2xpY3lEQzU4NUJDRScpLCBbICdzMzpEZWxldGVPYmplY3QqJywgJ3MzOlB1dE9iamVjdConLCAnczM6QWJvcnQqJyBdKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhY3Rpb25zKCdQdXR0ZXJEZWZhdWx0UG9saWN5QUIxMzhERDMnKSwgWyAnczM6UHV0T2JqZWN0KicsICdzMzpBYm9ydConIF0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFjdGlvbnMoJ0RlbGV0ZXJEZWZhdWx0UG9saWN5Q0QzM0I4QTAnKSwgJ3MzOkRlbGV0ZU9iamVjdConKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY3Jvc3Mtc3RhY2sgcGVybWlzc2lvbnMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFja0EgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0RnJvbVN0YWNrQSA9IG5ldyBzMy5CdWNrZXQoc3RhY2tBLCAnTXlCdWNrZXQnKTtcbiAgICBjb25zdCByZWZUb0J1Y2tldEZyb21TdGFja0EgPSBidWNrZXRGcm9tU3RhY2tBLmV4cG9ydCgpO1xuXG4gICAgY29uc3Qgc3RhY2tCID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHVzZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2tCLCAnVXNlcldob05lZWRzQWNjZXNzJyk7XG4gICAgY29uc3QgdGhlQnVja2V0RnJvbVN0YWNrQUFzQVJlZkluU3RhY2tCID0gczMuQnVja2V0LmltcG9ydChzdGFja0IsICdSZWZUb0J1Y2tldEZyb21TdGFja0EnLCByZWZUb0J1Y2tldEZyb21TdGFja0EpO1xuICAgIHRoZUJ1Y2tldEZyb21TdGFja0FBc0FSZWZJblN0YWNrQi5ncmFudFJlYWQodXNlcik7XG5cbiAgICBleHBlY3Qoc3RhY2tBKS50b01hdGNoKHtcbiAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgIFwiTXlCdWNrZXRGNjhGM0ZGMFwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6UzM6OkJ1Y2tldFwiLFxuICAgICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIixcbiAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcIk91dHB1dHNcIjoge1xuICAgICAgXCJNeUJ1Y2tldEJ1Y2tldEFybkUyNjA1NThDXCI6IHtcbiAgICAgICAgXCJWYWx1ZVwiOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCIsXG4gICAgICAgICAgXCJBcm5cIlxuICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIFwiRXhwb3J0XCI6IHtcbiAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4Q1wiXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcIk15QnVja2V0QnVja2V0TmFtZThBMDI3MDE0XCI6IHtcbiAgICAgICAgXCJWYWx1ZVwiOiB7XG4gICAgICAgIFwiUmVmXCI6IFwiTXlCdWNrZXRGNjhGM0ZGMFwiXG4gICAgICAgIH0sXG4gICAgICAgIFwiRXhwb3J0XCI6IHtcbiAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6TXlCdWNrZXRCdWNrZXROYW1lOEEwMjcwMTRcIlxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJNeUJ1Y2tldERvbWFpbk5hbWVGNzZCOUE3QVwiOiB7XG4gICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgIFwiRG9tYWluTmFtZVwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBcIkV4cG9ydFwiOiB7XG4gICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6TXlCdWNrZXREb21haW5OYW1lRjc2QjlBN0FcIlxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJNeUJ1Y2tldFdlYnNpdGVVUkw5QzIyMjc4OFwiOiB7XG4gICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15QnVja2V0RjY4RjNGRjBcIixcbiAgICAgICAgICAgIFwiV2Vic2l0ZVVSTFwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBcIkV4cG9ydFwiOiB7XCJOYW1lXCI6IFwiU3RhY2s6TXlCdWNrZXRXZWJzaXRlVVJMOUMyMjI3ODhcIn1cbiAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFja0IpLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgXCJVc2VyV2hvTmVlZHNBY2Nlc3NGODk1OUMzRFwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpVc2VyXCJcbiAgICAgIH0sXG4gICAgICBcIlVzZXJXaG9OZWVkc0FjY2Vzc0RlZmF1bHRQb2xpY3k2QTlFQjUzMFwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpQb2xpY3lcIixcbiAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgIFwiczM6R2V0T2JqZWN0KlwiLFxuICAgICAgICAgICAgXCJzMzpHZXRCdWNrZXQqXCIsXG4gICAgICAgICAgICBcInMzOkxpc3QqXCJcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICBcIlJlc291cmNlXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJGbjo6SW1wb3J0VmFsdWVcIjogXCJTdGFjazpNeUJ1Y2tldEJ1Y2tldEFybkUyNjA1NThDXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiRm46OkltcG9ydFZhbHVlXCI6IFwiU3RhY2s6TXlCdWNrZXRCdWNrZXRBcm5FMjYwNTU4Q1wiXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBcIi8qXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH0sXG4gICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIlVzZXJXaG9OZWVkc0FjY2Vzc0RlZmF1bHRQb2xpY3k2QTlFQjUzMFwiLFxuICAgICAgICBcIlVzZXJzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgXCJSZWZcIjogXCJVc2VyV2hvTmVlZHNBY2Nlc3NGODk1OUMzRFwiXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd1cmxGb3JPYmplY3QgcmV0dXJucyBhIHRva2VuIHdpdGggdGhlIFMzIFVSTCBvZiB0aGUgdG9rZW4nKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlCdWNrZXQnKTtcblxuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHN0YWNrLCAnQnVja2V0VVJMJywgeyB2YWx1ZTogYnVja2V0LmJ1Y2tldFVybCB9KTtcbiAgICBuZXcgY2RrLkNmbk91dHB1dChzdGFjaywgJ015RmlsZVVSTCcsIHsgdmFsdWU6IGJ1Y2tldC51cmxGb3JPYmplY3QoJ215L2ZpbGUudHh0JykgfSk7XG4gICAgbmV3IGNkay5DZm5PdXRwdXQoc3RhY2ssICdZb3VyRmlsZVVSTCcsIHsgdmFsdWU6IGJ1Y2tldC51cmxGb3JPYmplY3QoJy95b3VyL2ZpbGUudHh0JykgfSk7IC8vIFwiL1wiIGlzIG9wdGlvbmFsXG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgXCJNeUJ1Y2tldEY2OEYzRkYwXCI6IHtcbiAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTMzo6QnVja2V0XCIsXG4gICAgICAgIFwiRGVsZXRpb25Qb2xpY3lcIjogXCJSZXRhaW5cIlxuICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiT3V0cHV0c1wiOiB7XG4gICAgICBcIkJ1Y2tldFVSTFwiOiB7XG4gICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICBcIlwiLFxuICAgICAgICAgIFtcbiAgICAgICAgICBcImh0dHBzOi8vczMuXCIsXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIi5cIixcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6VVJMU3VmZml4XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiL1wiLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlCdWNrZXRGNjhGM0ZGMFwiXG4gICAgICAgICAgfVxuICAgICAgICAgIF1cbiAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIFwiTXlGaWxlVVJMXCI6IHtcbiAgICAgICAgXCJWYWx1ZVwiOiB7XG4gICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgW1xuICAgICAgICAgIFwiaHR0cHM6Ly9zMy5cIixcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UmVnaW9uXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiLlwiLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpVUkxTdWZmaXhcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCIvXCIsXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJNeUJ1Y2tldEY2OEYzRkYwXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiL215L2ZpbGUudHh0XCJcbiAgICAgICAgICBdXG4gICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiWW91ckZpbGVVUkxcIjoge1xuICAgICAgICBcIlZhbHVlXCI6IHtcbiAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgXCJcIixcbiAgICAgICAgICBbXG4gICAgICAgICAgXCJodHRwczovL3MzLlwiLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCIuXCIsXG4gICAgICAgICAge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIi9cIixcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIlJlZlwiOiBcIk15QnVja2V0RjY4RjNGRjBcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCIveW91ci9maWxlLnR4dFwiXG4gICAgICAgICAgXVxuICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZ3JhbnRQdWJsaWNBY2Nlc3MnOiB7XG4gICAgJ2J5IGRlZmF1bHQsIGdyYW50cyBzMzpHZXRPYmplY3QgdG8gYWxsIG9iamVjdHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdiJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGJ1Y2tldC5ncmFudFB1YmxpY0FjY2VzcygpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTMzo6QnVja2V0UG9saWN5Jywge1xuICAgICAgICBcIlBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwiczM6R2V0T2JqZWN0XCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJQcmluY2lwYWxcIjogXCIqXCIsXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogeyBcIkZuOjpKb2luXCI6IFsgXCJcIiwgWyB7IFwiRm46OkdldEF0dFwiOiBbIFwiYkMzQkJDQzY1XCIsIFwiQXJuXCIgXSB9LCBcIi8qXCIgXSBdIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9XG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ1wia2V5UHJlZml4XCIgY2FuIGJlIHVzZWQgdG8gb25seSBncmFudCBhY2Nlc3MgdG8gY2VydGFpbiBvYmplY3RzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnYicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBidWNrZXQuZ3JhbnRQdWJsaWNBY2Nlc3MoJ29ubHkvYWNjZXNzL3RoZXNlLyonKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6UzM6OkJ1Y2tldFBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcInMzOkdldE9iamVjdFwiLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IFwiKlwiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbIFwiXCIsIFsgeyBcIkZuOjpHZXRBdHRcIjogWyBcImJDM0JCQ0M2NVwiLCBcIkFyblwiIF0gfSwgXCIvb25seS9hY2Nlc3MvdGhlc2UvKlwiIF0gXSB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfVxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdcImFsbG93ZWRBY3Rpb25zXCIgY2FuIGJlIHVzZWQgdG8gc3BlY2lmeSBhY3Rpb25zIGV4cGxpY2l0bHknKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdiJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGJ1Y2tldC5ncmFudFB1YmxpY0FjY2VzcygnKicsICdzMzpHZXRPYmplY3QnLCAnczM6UHV0T2JqZWN0Jyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OlMzOjpCdWNrZXRQb2xpY3knLCB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogWyBcInMzOkdldE9iamVjdFwiLCBcInMzOlB1dE9iamVjdFwiIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJQcmluY2lwYWxcIjogXCIqXCIsXG4gICAgICAgICAgICAgIFwiUmVzb3VyY2VcIjogeyBcIkZuOjpKb2luXCI6IFsgXCJcIiwgWyB7IFwiRm46OkdldEF0dFwiOiBbIFwiYkMzQkJDQzY1XCIsIFwiQXJuXCIgXSB9LCBcIi8qXCIgXSBdIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9XG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ3JldHVybnMgdGhlIFBvbGljeVN0YXRlbWVudCB3aGljaCBjYW4gYmUgdGhlbiBjdXN0b21pemVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnYicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBzdGF0ZW1lbnQgPSBidWNrZXQuZ3JhbnRQdWJsaWNBY2Nlc3MoKTtcbiAgICAgIHN0YXRlbWVudC5hZGRDb25kaXRpb24oJ0lwQWRkcmVzcycsIHsgXCJhd3M6U291cmNlSXBcIjogXCI1NC4yNDAuMTQzLjAvMjRcIiB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6UzM6OkJ1Y2tldFBvbGljeScsIHtcbiAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcInMzOkdldE9iamVjdFwiLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IFwiKlwiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbIFwiXCIsIFsgeyBcIkZuOjpHZXRBdHRcIjogWyBcImJDM0JCQ0M2NVwiLCBcIkFyblwiIF0gfSwgXCIvKlwiIF0gXSB9LFxuICAgICAgICAgICAgICBcIkNvbmRpdGlvblwiOiB7XG4gICAgICAgICAgICAgICAgXCJJcEFkZHJlc3NcIjogeyBcImF3czpTb3VyY2VJcFwiOiBcIjU0LjI0MC4xNDMuMC8yNFwiIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH1cbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhyb3dzIHdoZW4gYmxvY2tQdWJsaWNQb2xpY3kgaXMgc2V0IHRvIHRydWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcsIHtcbiAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IG5ldyBzMy5CbG9ja1B1YmxpY0FjY2Vzcyh7IGJsb2NrUHVibGljUG9saWN5OiB0cnVlIH0pXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gYnVja2V0LmdyYW50UHVibGljQWNjZXNzKCksIC9ibG9ja1B1YmxpY1BvbGljeS8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ3dlYnNpdGUgY29uZmlndXJhdGlvbic6IHtcbiAgICAnb25seSBpbmRleCBkb2MnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgbmV3IHMzLkJ1Y2tldChzdGFjaywgJ1dlYnNpdGUnLCB7XG4gICAgICAgIHdlYnNpdGVJbmRleERvY3VtZW50OiAnaW5kZXgyLmh0bWwnXG4gICAgICB9KTtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OlMzOjpCdWNrZXQnLCB7XG4gICAgICAgIFdlYnNpdGVDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgSW5kZXhEb2N1bWVudDogXCJpbmRleDIuaHRtbFwiXG4gICAgICAgIH1cbiAgICAgIH0pKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2ZhaWxzIGlmIG9ubHkgZXJyb3IgZG9jIGlzIHNwZWNpZmllZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIG5ldyBzMy5CdWNrZXQoc3RhY2ssICdXZWJzaXRlJywge1xuICAgICAgICAgIHdlYnNpdGVFcnJvckRvY3VtZW50OiAnZXJyb3IuaHRtbCdcbiAgICAgICAgfSk7XG4gICAgICB9LCAvXCJ3ZWJzaXRlSW5kZXhEb2N1bWVudFwiIGlzIHJlcXVpcmVkIGlmIFwid2Vic2l0ZUVycm9yRG9jdW1lbnRcIiBpcyBzZXQvKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgJ2Vycm9yIGFuZCBpbmRleCBkb2NzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIG5ldyBzMy5CdWNrZXQoc3RhY2ssICdXZWJzaXRlJywge1xuICAgICAgICB3ZWJzaXRlSW5kZXhEb2N1bWVudDogJ2luZGV4Mi5odG1sJyxcbiAgICAgICAgd2Vic2l0ZUVycm9yRG9jdW1lbnQ6ICdlcnJvci5odG1sJyxcbiAgICAgIH0pO1xuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6UzM6OkJ1Y2tldCcsIHtcbiAgICAgICAgV2Vic2l0ZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBJbmRleERvY3VtZW50OiBcImluZGV4Mi5odG1sXCIsXG4gICAgICAgICAgRXJyb3JEb2N1bWVudDogXCJlcnJvci5odG1sXCJcbiAgICAgICAgfVxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAnZXhwb3J0cyB0aGUgV2Vic2l0ZVVSTCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnV2Vic2l0ZScsIHtcbiAgICAgICAgd2Vic2l0ZUluZGV4RG9jdW1lbnQ6ICdpbmRleC5odG1sJ1xuICAgICAgfSk7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChidWNrZXQubm9kZS5yZXNvbHZlKGJ1Y2tldC5idWNrZXRXZWJzaXRlVXJsKSwgeyAnRm46OkdldEF0dCc6IFsgJ1dlYnNpdGUzMjk2MkQwQicsICdXZWJzaXRlVVJMJyBdIH0pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuICB9XG59O1xuIl19