from unittest.mock import Mock

from dbt_platform_helper.constants import DEFAULT_TERRAFORM_PLATFORM_MODULES_VERSION
from dbt_platform_helper.domain.terraform_environment import (
    PlatformTerraformManifestGenerator,
)
from dbt_platform_helper.providers.files import FileProvider


class TestPlatformTerraformManifestGenerator:

    def test_generator_generates_expected_manifest_content_with_version_override(self):
        test_environment_config = {
            "vpc": "vpc3",
            "accounts": {
                "deploy": {"name": "non-prod-acc", "id": "1122334455"},
                "dns": {"name": "non-prod-dns-acc", "id": "6677889900"},
            },
            "versions": {"terraform-platform-modules": 3},
        }
        expected_header = "# WARNING: This is an autogenerated file, not for manual editing."
        expected_modules = "git::https://github.com/uktrade/terraform-platform-modules.git//extensions?depth=1&ref=123456"
        expected_moved_block = (
            "moved {\n  from = module.extensions-tf\n  to   = module.extensions\n}\n"
        )

        result = PlatformTerraformManifestGenerator(Mock()).generate_manifest(
            "test", "test-app", test_environment_config, 123456
        )

        assert expected_header in result
        assert expected_modules in result
        assert expected_moved_block in result
        assert 'environment = "test"' in result
        assert 'application    = "test-app"' in result
        assert 'vpc_name    = "vpc3"'

    def test_generator_generates_expected_manifest_content_with_tpm_version_set_in_config(self):
        test_environment_config = {
            "vpc": "vpc3",
            "accounts": {
                "deploy": {"name": "non-prod-acc", "id": "1122334455"},
                "dns": {"name": "non-prod-dns-acc", "id": "6677889900"},
            },
            "versions": {"terraform-platform-modules": 3},
        }
        expected_header = "# WARNING: This is an autogenerated file, not for manual editing."
        expected_modules = "git::https://github.com/uktrade/terraform-platform-modules.git//extensions?depth=1&ref=3"
        expected_moved_block = (
            "moved {\n  from = module.extensions-tf\n  to   = module.extensions\n}\n"
        )

        result = PlatformTerraformManifestGenerator(Mock()).generate_manifest(
            "test", "test-app", test_environment_config
        )

        assert expected_header in result
        assert expected_modules in result
        assert expected_moved_block in result
        assert 'environment = "test"' in result
        assert 'application    = "test-app"' in result
        assert 'vpc_name    = "vpc3"'

    def test_generator_generates_expected_manifest_content_with_default_version(self):
        test_environment_config = {
            "vpc": "vpc3",
            "accounts": {
                "deploy": {"name": "non-prod-acc", "id": "1122334455"},
                "dns": {"name": "non-prod-dns-acc", "id": "6677889900"},
            },
        }
        expected_header = "# WARNING: This is an autogenerated file, not for manual editing."
        expected_modules = f"git::https://github.com/uktrade/terraform-platform-modules.git//extensions?depth=1&ref={DEFAULT_TERRAFORM_PLATFORM_MODULES_VERSION}"
        expected_moved_block = (
            "moved {\n  from = module.extensions-tf\n  to   = module.extensions\n}\n"
        )

        result = PlatformTerraformManifestGenerator(Mock()).generate_manifest(
            "test", "test-app", test_environment_config
        )

        assert expected_header in result
        assert expected_modules in result
        assert expected_moved_block in result
        assert 'environment = "test"' in result
        assert 'application    = "test-app"' in result
        assert 'vpc_name    = "vpc3"'

    def test_generator_write_manifest_makes_the_expected_manifest_file(self):
        mock_file_provider = Mock(spec=FileProvider)
        PlatformTerraformManifestGenerator(mock_file_provider).write_manifest(
            "test-environment", "test-manifest-content"
        )

        mock_file_provider.mkfile.assert_called_once_with(
            ".",
            f"terraform/environments/test-environment/main.tf",
            "test-manifest-content",
            overwrite=True,
        )
