/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hipparchus.analysis.differentiation.DSFactory;
import org.hipparchus.analysis.differentiation.DerivativeStructure;
import org.hipparchus.analysis.interpolation.HermiteInterpolator;
import org.hipparchus.exception.MathIllegalStateException;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.FieldKeplerianOrbit;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class KeplerianOrbit
extends Orbit {
    private static final long serialVersionUID = 20170414L;
    private static final DSFactory FACTORY = new DSFactory(1, 1);
    private static final double A;
    private static final double B;
    private final double a;
    private final double e;
    private final double i;
    private final double pa;
    private final double raan;
    private final double v;
    private final double aDot;
    private final double eDot;
    private final double iDot;
    private final double paDot;
    private final double raanDot;
    private final double vDot;
    private transient PVCoordinates partialPV;

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, e, i, pa, raan, anomaly, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, type, frame, date, mu);
    }

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, double aDot, double eDot, double iDot, double paDot, double raanDot, double anomalyDot, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (a * (1.0 - e) < 0.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.ORBIT_A_E_MISMATCH_WITH_CONIC_TYPE, a, e);
        }
        this.a = a;
        this.aDot = aDot;
        this.e = e;
        this.eDot = eDot;
        this.i = i;
        this.iDot = iDot;
        this.pa = pa;
        this.paDot = paDot;
        this.raan = raan;
        this.raanDot = raanDot;
        if (this.hasDerivatives()) {
            DerivativeStructure vDS;
            DerivativeStructure eDS = FACTORY.build(e, eDot);
            DerivativeStructure anomalyDS = FACTORY.build(anomaly, anomalyDot);
            switch (type) {
                case MEAN: {
                    vDS = a < 0.0 ? FieldKeplerianOrbit.hyperbolicEccentricToTrue(FieldKeplerianOrbit.meanToHyperbolicEccentric(anomalyDS, eDS), eDS) : FieldKeplerianOrbit.ellipticEccentricToTrue(FieldKeplerianOrbit.meanToEllipticEccentric(anomalyDS, eDS), eDS);
                    break;
                }
                case ECCENTRIC: {
                    vDS = a < 0.0 ? FieldKeplerianOrbit.hyperbolicEccentricToTrue(anomalyDS, eDS) : FieldKeplerianOrbit.ellipticEccentricToTrue(anomalyDS, eDS);
                    break;
                }
                case TRUE: {
                    vDS = anomalyDS;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.v = vDS.getValue();
            this.vDot = vDS.getPartialDerivative(1);
        } else {
            switch (type) {
                case MEAN: {
                    this.v = a < 0.0 ? KeplerianOrbit.hyperbolicEccentricToTrue(KeplerianOrbit.meanToHyperbolicEccentric(anomaly, e), e) : KeplerianOrbit.ellipticEccentricToTrue(KeplerianOrbit.meanToEllipticEccentric(anomaly, e), e);
                    break;
                }
                case ECCENTRIC: {
                    this.v = a < 0.0 ? KeplerianOrbit.hyperbolicEccentricToTrue(anomaly, e) : KeplerianOrbit.ellipticEccentricToTrue(anomaly, e);
                    break;
                }
                case TRUE: {
                    this.v = anomaly;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.vDot = Double.NaN;
        }
        if (1.0 + e * FastMath.cos(this.v) <= 0.0) {
            double vMax = FastMath.acos(-1.0 / e);
            throw new OrekitIllegalArgumentException(OrekitMessages.ORBIT_ANOMALY_OUT_OF_HYPERBOLIC_RANGE, this.v, e, -vMax, vMax);
        }
        this.partialPV = null;
    }

    public KeplerianOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        this(pvCoordinates, frame, mu, KeplerianOrbit.hasNonKeplerianAcceleration(pvCoordinates, mu));
    }

    private KeplerianOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu, boolean reliableAcceleration) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D momentum = pvCoordinates.getMomentum();
        double m2 = momentum.getNormSq();
        this.i = Vector3D.angle(momentum, Vector3D.PLUS_K);
        this.raan = Vector3D.crossProduct(Vector3D.PLUS_K, momentum).getAlpha();
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        Vector3D pvA = pvCoordinates.getAcceleration();
        double r2 = pvP.getNormSq();
        double r = FastMath.sqrt(r2);
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        this.a = r / (2.0 - rV2OnMu);
        double muA = mu * this.a;
        if (this.a > 0.0) {
            double eSE = Vector3D.dotProduct(pvP, pvV) / FastMath.sqrt(muA);
            double eCE = rV2OnMu - 1.0;
            this.e = FastMath.sqrt(eSE * eSE + eCE * eCE);
            this.v = KeplerianOrbit.ellipticEccentricToTrue(FastMath.atan2(eSE, eCE), this.e);
        } else {
            double eSH = Vector3D.dotProduct(pvP, pvV) / FastMath.sqrt(-muA);
            double eCH = rV2OnMu - 1.0;
            this.e = FastMath.sqrt(1.0 - m2 / muA);
            this.v = KeplerianOrbit.hyperbolicEccentricToTrue(FastMath.log((eCH + eSH) / (eCH - eSH)) / 2.0, this.e);
        }
        Vector3D node = new Vector3D(this.raan, 0.0);
        double px = Vector3D.dotProduct(pvP, node);
        double py = Vector3D.dotProduct(pvP, Vector3D.crossProduct(momentum, node)) / FastMath.sqrt(m2);
        this.pa = FastMath.atan2(py, px) - this.v;
        this.partialPV = pvCoordinates;
        if (reliableAcceleration) {
            double[][] jacobian = new double[6][6];
            this.getJacobianWrtCartesian(PositionAngle.MEAN, jacobian);
            Vector3D keplerianAcceleration = new Vector3D(-mu / (r * r2), pvP);
            Vector nonKeplerianAcceleration = pvA.subtract((Vector)keplerianAcceleration);
            double aX = ((Vector3D)nonKeplerianAcceleration).getX();
            double aY = ((Vector3D)nonKeplerianAcceleration).getY();
            double aZ = ((Vector3D)nonKeplerianAcceleration).getZ();
            this.aDot = jacobian[0][3] * aX + jacobian[0][4] * aY + jacobian[0][5] * aZ;
            this.eDot = jacobian[1][3] * aX + jacobian[1][4] * aY + jacobian[1][5] * aZ;
            this.iDot = jacobian[2][3] * aX + jacobian[2][4] * aY + jacobian[2][5] * aZ;
            this.paDot = jacobian[3][3] * aX + jacobian[3][4] * aY + jacobian[3][5] * aZ;
            this.raanDot = jacobian[4][3] * aX + jacobian[4][4] * aY + jacobian[4][5] * aZ;
            double MDot = this.getKeplerianMeanMotion() + jacobian[5][3] * aX + jacobian[5][4] * aY + jacobian[5][5] * aZ;
            DerivativeStructure eDS = FACTORY.build(this.e, this.eDot);
            DerivativeStructure MDS = FACTORY.build(this.getMeanAnomaly(), MDot);
            DerivativeStructure vDS = this.a < 0.0 ? FieldKeplerianOrbit.hyperbolicEccentricToTrue(FieldKeplerianOrbit.meanToHyperbolicEccentric(MDS, eDS), eDS) : FieldKeplerianOrbit.ellipticEccentricToTrue(FieldKeplerianOrbit.meanToEllipticEccentric(MDS, eDS), eDS);
            this.vDot = vDS.getPartialDerivative(1);
        } else {
            this.aDot = Double.NaN;
            this.eDot = Double.NaN;
            this.iDot = Double.NaN;
            this.paDot = Double.NaN;
            this.raanDot = Double.NaN;
            this.vDot = Double.NaN;
        }
    }

    public KeplerianOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public KeplerianOrbit(Orbit op) {
        this(op.getPVCoordinates(), op.getFrame(), op.getMu(), op.hasDerivatives());
    }

    @Override
    public OrbitType getType() {
        return OrbitType.KEPLERIAN;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getADot() {
        return this.aDot;
    }

    @Override
    public double getE() {
        return this.e;
    }

    @Override
    public double getEDot() {
        return this.eDot;
    }

    @Override
    public double getI() {
        return this.i;
    }

    @Override
    public double getIDot() {
        return this.iDot;
    }

    public double getPerigeeArgument() {
        return this.pa;
    }

    public double getPerigeeArgumentDot() {
        return this.paDot;
    }

    public double getRightAscensionOfAscendingNode() {
        return this.raan;
    }

    public double getRightAscensionOfAscendingNodeDot() {
        return this.raanDot;
    }

    public double getTrueAnomaly() {
        return this.v;
    }

    public double getTrueAnomalyDot() {
        return this.vDot;
    }

    public double getEccentricAnomaly() {
        return this.a < 0.0 ? KeplerianOrbit.trueToHyperbolicEccentric(this.v, this.e) : KeplerianOrbit.trueToEllipticEccentric(this.v, this.e);
    }

    public double getEccentricAnomalyDot() {
        DerivativeStructure eDS = FACTORY.build(this.e, this.eDot);
        DerivativeStructure vDS = FACTORY.build(this.v, this.vDot);
        DerivativeStructure EDS = this.a < 0.0 ? FieldKeplerianOrbit.trueToHyperbolicEccentric(vDS, eDS) : FieldKeplerianOrbit.trueToEllipticEccentric(vDS, eDS);
        return EDS.getPartialDerivative(1);
    }

    public double getMeanAnomaly() {
        return this.a < 0.0 ? KeplerianOrbit.hyperbolicEccentricToMean(KeplerianOrbit.trueToHyperbolicEccentric(this.v, this.e), this.e) : KeplerianOrbit.ellipticEccentricToMean(KeplerianOrbit.trueToEllipticEccentric(this.v, this.e), this.e);
    }

    public double getMeanAnomalyDot() {
        DerivativeStructure eDS = FACTORY.build(this.e, this.eDot);
        DerivativeStructure vDS = FACTORY.build(this.v, this.vDot);
        DerivativeStructure MDS = this.a < 0.0 ? FieldKeplerianOrbit.hyperbolicEccentricToMean(FieldKeplerianOrbit.trueToHyperbolicEccentric(vDS, eDS), eDS) : FieldKeplerianOrbit.ellipticEccentricToMean(FieldKeplerianOrbit.trueToEllipticEccentric(vDS, eDS), eDS);
        return MDS.getPartialDerivative(1);
    }

    public double getAnomaly(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getMeanAnomaly() : (type == PositionAngle.ECCENTRIC ? this.getEccentricAnomaly() : this.getTrueAnomaly());
    }

    public double getAnomalyDot(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getMeanAnomalyDot() : (type == PositionAngle.ECCENTRIC ? this.getEccentricAnomalyDot() : this.getTrueAnomalyDot());
    }

    public static double ellipticEccentricToTrue(double E, double e) {
        double beta = e / (1.0 + FastMath.sqrt((1.0 - e) * (1.0 + e)));
        return E + 2.0 * FastMath.atan(beta * FastMath.sin(E) / (1.0 - beta * FastMath.cos(E)));
    }

    public static double trueToEllipticEccentric(double v, double e) {
        double beta = e / (1.0 + FastMath.sqrt(1.0 - e * e));
        return v - 2.0 * FastMath.atan(beta * FastMath.sin(v) / (1.0 + beta * FastMath.cos(v)));
    }

    public static double hyperbolicEccentricToTrue(double H, double e) {
        return 2.0 * FastMath.atan(FastMath.sqrt((e + 1.0) / (e - 1.0)) * FastMath.tanh(H / 2.0));
    }

    public static double trueToHyperbolicEccentric(double v, double e) {
        double sinhH = FastMath.sqrt(e * e - 1.0) * FastMath.sin(v) / (1.0 + e * FastMath.cos(v));
        return FastMath.asinh(sinhH);
    }

    public static double meanToEllipticEccentric(double M, double e) {
        double w;
        double E;
        double reducedM = MathUtils.normalizeAngle(M, 0.0);
        if (FastMath.abs(reducedM) < 0.16666666666666666) {
            E = reducedM + e * (FastMath.cbrt(6.0 * reducedM) - reducedM);
        } else if (reducedM < 0.0) {
            w = Math.PI + reducedM;
            E = reducedM + e * (A * w / (B - w) - Math.PI - reducedM);
        } else {
            w = Math.PI - reducedM;
            E = reducedM + e * (Math.PI - A * w / (B - w) - reducedM);
        }
        double e1 = 1.0 - e;
        boolean noCancellationRisk = e1 + E * E / 6.0 >= 0.1;
        for (int j = 0; j < 2; ++j) {
            double fd;
            double f;
            double fdd = e * FastMath.sin(E);
            double fddd = e * FastMath.cos(E);
            if (noCancellationRisk) {
                f = E - fdd - reducedM;
                fd = 1.0 - fddd;
            } else {
                f = KeplerianOrbit.eMeSinE(E, e) - reducedM;
                double s = FastMath.sin(0.5 * E);
                fd = e1 + 2.0 * e * s * s;
            }
            double dee = f * fd / (0.5 * f * fdd - fd * fd);
            double w2 = fd + 0.5 * dee * (fdd + dee * fddd / 3.0);
            E -= (f - dee * ((fd += dee * (fdd + 0.5 * dee * fddd)) - w2)) / fd;
        }
        return E += M - reducedM;
    }

    private static double eMeSinE(double E, double e) {
        double x = (1.0 - e) * FastMath.sin(E);
        double mE2 = -E * E;
        double term = E;
        double d = 0.0;
        double x0 = Double.NaN;
        while (!Double.valueOf(x).equals(x0)) {
            x0 = x;
            x -= (term *= mE2 / ((d += 2.0) * (d + 1.0)));
        }
        return x;
    }

    public static double meanToHyperbolicEccentric(double M, double ecc) {
        double H = ecc < 1.6 ? (-Math.PI < M && M < 0.0 || M > Math.PI ? M - ecc : M + ecc) : (ecc < 3.6 && FastMath.abs(M) > Math.PI ? M - FastMath.copySign(ecc, M) : M / (ecc - 1.0));
        int iter = 0;
        do {
            double f3 = ecc * FastMath.cosh(H);
            double f2 = ecc * FastMath.sinh(H);
            double f1 = f3 - 1.0;
            double f0 = f2 - H - M;
            double f12 = 2.0 * f1;
            double d = f0 / f12;
            double fdf = f1 - d * f2;
            double ds = f0 / fdf;
            double shift = f0 / (fdf + ds * ds * f3 / 6.0);
            H -= shift;
            if (!(FastMath.abs(shift) <= 1.0E-12)) continue;
            return H;
        } while (++iter < 50);
        throw new MathIllegalStateException(OrekitMessages.UNABLE_TO_COMPUTE_HYPERBOLIC_ECCENTRIC_ANOMALY, iter);
    }

    public static double ellipticEccentricToMean(double E, double e) {
        return E - e * FastMath.sin(E);
    }

    public static double hyperbolicEccentricToMean(double H, double e) {
        return e * FastMath.sinh(H) - H;
    }

    @Override
    public double getEquinoctialEx() {
        return this.e * FastMath.cos(this.pa + this.raan);
    }

    @Override
    public double getEquinoctialExDot() {
        double paPraan = this.pa + this.raan;
        double cos = FastMath.cos(paPraan);
        double sin = FastMath.sin(paPraan);
        return this.eDot * cos - this.e * sin * (this.paDot + this.raanDot);
    }

    @Override
    public double getEquinoctialEy() {
        return this.e * FastMath.sin(this.pa + this.raan);
    }

    @Override
    public double getEquinoctialEyDot() {
        double paPraan = this.pa + this.raan;
        double cos = FastMath.cos(paPraan);
        double sin = FastMath.sin(paPraan);
        return this.eDot * sin + this.e * cos * (this.paDot + this.raanDot);
    }

    @Override
    public double getHx() {
        if (FastMath.abs(this.i - Math.PI) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.cos(this.raan) * FastMath.tan(0.5 * this.i);
    }

    @Override
    public double getHxDot() {
        if (FastMath.abs(this.i - Math.PI) < 1.0E-10) {
            return Double.NaN;
        }
        double cosRaan = FastMath.cos(this.raan);
        double sinRaan = FastMath.sin(this.raan);
        double tan = FastMath.tan(0.5 * this.i);
        return 0.5 * (1.0 + tan * tan) * cosRaan * this.iDot - tan * sinRaan * this.raanDot;
    }

    @Override
    public double getHy() {
        if (FastMath.abs(this.i - Math.PI) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.sin(this.raan) * FastMath.tan(0.5 * this.i);
    }

    @Override
    public double getHyDot() {
        if (FastMath.abs(this.i - Math.PI) < 1.0E-10) {
            return Double.NaN;
        }
        double cosRaan = FastMath.cos(this.raan);
        double sinRaan = FastMath.sin(this.raan);
        double tan = FastMath.tan(0.5 * this.i);
        return 0.5 * (1.0 + tan * tan) * sinRaan * this.iDot + tan * cosRaan * this.raanDot;
    }

    @Override
    public double getLv() {
        return this.pa + this.raan + this.v;
    }

    @Override
    public double getLvDot() {
        return this.paDot + this.raanDot + this.vDot;
    }

    @Override
    public double getLE() {
        return this.pa + this.raan + this.getEccentricAnomaly();
    }

    @Override
    public double getLEDot() {
        return this.paDot + this.raanDot + this.getEccentricAnomalyDot();
    }

    @Override
    public double getLM() {
        return this.pa + this.raan + this.getMeanAnomaly();
    }

    @Override
    public double getLMDot() {
        return this.paDot + this.raanDot + this.getMeanAnomalyDot();
    }

    private void computePVWithoutA() {
        if (this.partialPV != null) {
            return;
        }
        double cosRaan = FastMath.cos(this.raan);
        double sinRaan = FastMath.sin(this.raan);
        double cosPa = FastMath.cos(this.pa);
        double sinPa = FastMath.sin(this.pa);
        double cosI = FastMath.cos(this.i);
        double sinI = FastMath.sin(this.i);
        double crcp = cosRaan * cosPa;
        double crsp = cosRaan * sinPa;
        double srcp = sinRaan * cosPa;
        double srsp = sinRaan * sinPa;
        Vector3D p = new Vector3D(crcp - cosI * srsp, srcp + cosI * crsp, sinI * sinPa);
        Vector3D q = new Vector3D(-crsp - cosI * srcp, -srsp + cosI * crcp, sinI * cosPa);
        if (this.a > 0.0) {
            double uME2 = (1.0 - this.e) * (1.0 + this.e);
            double s1Me2 = FastMath.sqrt(uME2);
            double E = this.getEccentricAnomaly();
            double cosE = FastMath.cos(E);
            double sinE = FastMath.sin(E);
            double x = this.a * (cosE - this.e);
            double y = this.a * sinE * s1Me2;
            double factor = FastMath.sqrt(this.getMu() / this.a) / (1.0 - this.e * cosE);
            double xDot = -sinE * factor;
            double yDot = cosE * s1Me2 * factor;
            Vector3D position = new Vector3D(x, p, y, q);
            Vector3D velocity = new Vector3D(xDot, p, yDot, q);
            this.partialPV = new PVCoordinates(position, velocity);
        } else {
            double sinV = FastMath.sin(this.v);
            double cosV = FastMath.cos(this.v);
            double f = this.a * (1.0 - this.e * this.e);
            double posFactor = f / (1.0 + this.e * cosV);
            double velFactor = FastMath.sqrt(this.getMu() / f);
            double x = posFactor * cosV;
            double y = posFactor * sinV;
            double xDot = -velFactor * sinV;
            double yDot = velFactor * (this.e + cosV);
            Vector3D position = new Vector3D(x, p, y, q);
            Vector3D velocity = new Vector3D(xDot, p, yDot, q);
            this.partialPV = new PVCoordinates(position, velocity);
        }
    }

    private Vector3D nonKeplerianAcceleration() {
        double[][] dCdP = new double[6][6];
        this.getJacobianWrtParameters(PositionAngle.MEAN, dCdP);
        double nonKeplerianMeanMotion = this.getMeanAnomalyDot() - this.getKeplerianMeanMotion();
        double nonKeplerianAx = dCdP[3][0] * this.aDot + dCdP[3][1] * this.eDot + dCdP[3][2] * this.iDot + dCdP[3][3] * this.paDot + dCdP[3][4] * this.raanDot + dCdP[3][5] * nonKeplerianMeanMotion;
        double nonKeplerianAy = dCdP[4][0] * this.aDot + dCdP[4][1] * this.eDot + dCdP[4][2] * this.iDot + dCdP[4][3] * this.paDot + dCdP[4][4] * this.raanDot + dCdP[4][5] * nonKeplerianMeanMotion;
        double nonKeplerianAz = dCdP[5][0] * this.aDot + dCdP[5][1] * this.eDot + dCdP[5][2] * this.iDot + dCdP[5][3] * this.paDot + dCdP[5][4] * this.raanDot + dCdP[5][5] * nonKeplerianMeanMotion;
        return new Vector3D(nonKeplerianAx, nonKeplerianAy, nonKeplerianAz);
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        this.computePVWithoutA();
        double r2 = this.partialPV.getPosition().getNormSq();
        Vector3D keplerianAcceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt(r2)), this.partialPV.getPosition());
        Vector3D acceleration = this.hasDerivatives() ? keplerianAcceleration.add((Vector)this.nonKeplerianAcceleration()) : keplerianAcceleration;
        return new TimeStampedPVCoordinates(this.getDate(), this.partialPV.getPosition(), this.partialPV.getVelocity(), acceleration);
    }

    @Override
    public KeplerianOrbit shiftedBy(double dt) {
        KeplerianOrbit keplerianShifted = new KeplerianOrbit(this.a, this.e, this.i, this.pa, this.raan, this.getMeanAnomaly() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
        if (this.hasDerivatives()) {
            Vector3D nonKeplerianAcceleration = this.nonKeplerianAcceleration();
            keplerianShifted.computePVWithoutA();
            Vector3D fixedP = new Vector3D(1.0, keplerianShifted.partialPV.getPosition(), 0.5 * dt * dt, nonKeplerianAcceleration);
            double fixedR2 = fixedP.getNormSq();
            double fixedR = FastMath.sqrt(fixedR2);
            Vector3D fixedV = new Vector3D(1.0, keplerianShifted.partialPV.getVelocity(), dt, nonKeplerianAcceleration);
            Vector3D fixedA = new Vector3D(-this.getMu() / (fixedR2 * fixedR), keplerianShifted.partialPV.getPosition(), 1.0, nonKeplerianAcceleration);
            return new KeplerianOrbit(new TimeStampedPVCoordinates(keplerianShifted.getDate(), fixedP, fixedV, fixedA), keplerianShifted.getFrame(), keplerianShifted.getMu());
        }
        return keplerianShifted;
    }

    @Override
    public KeplerianOrbit interpolate(AbsoluteDate date, Stream<Orbit> sample) {
        List list = sample.collect(Collectors.toList());
        boolean useDerivatives = true;
        for (Orbit orbit : list) {
            useDerivatives = useDerivatives && orbit.hasDerivatives();
        }
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousPA = Double.NaN;
        double previousRAAN = Double.NaN;
        double previousM = Double.NaN;
        for (Orbit orbit : list) {
            double continuousM;
            double continuousRAAN;
            double continuousPA;
            KeplerianOrbit kep = (KeplerianOrbit)OrbitType.KEPLERIAN.convertType(orbit);
            if (previousDate == null) {
                continuousPA = kep.getPerigeeArgument();
                continuousRAAN = kep.getRightAscensionOfAscendingNode();
                continuousM = kep.getMeanAnomaly();
            } else {
                double dt = kep.getDate().durationFrom(previousDate);
                double keplerM = previousM + kep.getKeplerianMeanMotion() * dt;
                continuousPA = MathUtils.normalizeAngle(kep.getPerigeeArgument(), previousPA);
                continuousRAAN = MathUtils.normalizeAngle(kep.getRightAscensionOfAscendingNode(), previousRAAN);
                continuousM = MathUtils.normalizeAngle(kep.getMeanAnomaly(), keplerM);
            }
            previousDate = kep.getDate();
            previousPA = continuousPA;
            previousRAAN = continuousRAAN;
            previousM = continuousM;
            if (useDerivatives) {
                interpolator.addSamplePoint(kep.getDate().durationFrom(date), {kep.getA(), kep.getE(), kep.getI(), continuousPA, continuousRAAN, continuousM}, {kep.getADot(), kep.getEDot(), kep.getIDot(), kep.getPerigeeArgumentDot(), kep.getRightAscensionOfAscendingNodeDot(), kep.getMeanAnomalyDot()});
                continue;
            }
            interpolator.addSamplePoint(kep.getDate().durationFrom(date), new double[][]{{kep.getA(), kep.getE(), kep.getI(), continuousPA, continuousRAAN, continuousM}});
        }
        double[][] interpolated = interpolator.derivatives(0.0, 1);
        return new KeplerianOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianMeanWrtCartesianElliptical();
        }
        return this.computeJacobianMeanWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianMeanWrtCartesianElliptical() {
        double factorI1;
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        Vector3D momentum = this.partialPV.getMomentum();
        double v2 = velocity.getNormSq();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt(r2);
        double r3 = r * r2;
        double px = position.getX();
        double py = position.getY();
        double pz = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt(this.a * mu);
        double sqrtAoMu = FastMath.sqrt(this.a / mu);
        double a2 = this.a * this.a;
        double twoA = 2.0 * this.a;
        double rOnA = r / this.a;
        double oMe2 = 1.0 - this.e * this.e;
        double epsilon = FastMath.sqrt(oMe2);
        double sqrtRec = 1.0 / epsilon;
        double cosI = FastMath.cos(this.i);
        double sinI = FastMath.sin(this.i);
        double cosPA = FastMath.cos(this.pa);
        double sinPA = FastMath.sin(this.pa);
        double pv = Vector3D.dotProduct(position, velocity);
        double cosE = (this.a - r) / (this.a * this.e);
        double sinE = pv / (this.e * sqrtMuA);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double factorER3 = pv / twoA;
        Vector3D vectorER = new Vector3D(cosE * v2 / (r * mu), position, sinE / sqrtMuA, velocity, -factorER3 * sinE / sqrtMuA, vectorAR);
        Vector3D vectorERDot = new Vector3D(sinE / sqrtMuA, position, cosE * 2.0 * r / mu, velocity, -factorER3 * sinE / sqrtMuA, vectorARDot);
        KeplerianOrbit.fillHalfRow(1.0, vectorER, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorERDot, jacobian[1], 3);
        double coefE = cosE / (this.e * sqrtMuA);
        Vector3D vectorEAnR = new Vector3D(-sinE * v2 / (this.e * r * mu), position, coefE, velocity, -factorER3 * coefE, vectorAR);
        Vector3D vectorEAnRDot = new Vector3D(-sinE * 2.0 * r / (this.e * mu), velocity, coefE, position, -factorER3 * coefE, vectorARDot);
        double s1 = -sinE * pz / r - cosE * vz * sqrtAoMu;
        double s2 = -cosE * pz / r3;
        double s3 = -sinE * vz / (2.0 * sqrtMuA);
        double t1 = sqrtRec * (cosE * pz / r - sinE * vz * sqrtAoMu);
        double t2 = sqrtRec * (-sinE * pz / r3);
        double t3 = sqrtRec * (cosE - this.e) * vz / (2.0 * sqrtMuA);
        double t4 = sqrtRec * (this.e * sinI * cosPA * sqrtRec - vz * sqrtAoMu);
        Vector3D s = new Vector3D(cosE / r, Vector3D.PLUS_K, s1, vectorEAnR, s2, position, s3, vectorAR);
        Vector3D sDot = new Vector3D(-sinE * sqrtAoMu, Vector3D.PLUS_K, s1, vectorEAnRDot, s3, vectorARDot);
        Vector t = new Vector3D(sqrtRec * sinE / r, Vector3D.PLUS_K).add((Vector)new Vector3D(t1, vectorEAnR, t2, position, t3, vectorAR, t4, vectorER));
        Vector3D tDot = new Vector3D(sqrtRec * (cosE - this.e) * sqrtAoMu, Vector3D.PLUS_K, t1, vectorEAnRDot, t3, vectorARDot, t4, vectorERDot);
        double i1 = factorI1 = -sinI * sqrtRec / sqrtMuA;
        double i2 = -factorI1 * mz / twoA;
        double i3 = factorI1 * mz * this.e / oMe2;
        double i4 = cosI * sinPA;
        double i5 = cosI * cosPA;
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(vy, -vx, 0.0), i2, vectorAR, i3, vectorER, i4, s, i5, (Vector3D)t, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(-py, px, 0.0), i2, vectorARDot, i3, vectorERDot, i4, sDot, i5, tDot, jacobian[2], 3);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, s, -sinPA / sinI, (Vector3D)t, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, sDot, -sinPA / sinI, tDot, jacobian[3], 3);
        double factorRaanR = 1.0 / (mu * this.a * oMe2 * sinI * sinI);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, vz, -vy), factorRaanR * mx, new Vector3D(-vz, 0.0, vx), jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, -pz, py), factorRaanR * mx, new Vector3D(pz, 0.0, -px), jacobian[4], 3);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnR, -sinE, vectorER, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnRDot, -sinE, vectorERDot, jacobian[5], 3);
        return jacobian;
    }

    private double[][] computeJacobianMeanWrtCartesianHyperbolic() {
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        Vector3D momentum = this.partialPV.getMomentum();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt(r2);
        double r3 = r * r2;
        double x = position.getX();
        double y = position.getY();
        double z = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double absA = -this.a;
        double sqrtMuA = FastMath.sqrt(absA * mu);
        double a2 = this.a * this.a;
        double rOa = r / absA;
        double cosI = FastMath.cos(this.i);
        double sinI = FastMath.sin(this.i);
        double pv = Vector3D.dotProduct(position, velocity);
        Vector3D vectorAR = new Vector3D(-2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(-2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(-1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(-1.0, vectorARDot, jacobian[0], 3);
        double m = momentum.getNorm();
        double oOm = 1.0 / m;
        Vector3D dcXP = new Vector3D(0.0, vz, -vy);
        Vector3D dcYP = new Vector3D(-vz, 0.0, vx);
        Vector3D dcZP = new Vector3D(vy, -vx, 0.0);
        Vector3D dcXV = new Vector3D(0.0, -z, y);
        Vector3D dcYV = new Vector3D(z, 0.0, -x);
        Vector3D dcZV = new Vector3D(-y, x, 0.0);
        Vector3D dCP = new Vector3D(mx * oOm, dcXP, my * oOm, dcYP, mz * oOm, dcZP);
        Vector3D dCV = new Vector3D(mx * oOm, dcXV, my * oOm, dcYV, mz * oOm, dcZV);
        double mOMu = m / mu;
        Vector3D dpP = new Vector3D(2.0 * mOMu, dCP);
        Vector3D dpV = new Vector3D(2.0 * mOMu, dCV);
        double p = m * mOMu;
        double moO2ae = 1.0 / (2.0 * absA * this.e);
        double m2OaMu = -p / absA;
        KeplerianOrbit.fillHalfRow(moO2ae, dpP, m2OaMu * moO2ae, vectorAR, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(moO2ae, dpV, m2OaMu * moO2ae, vectorARDot, jacobian[1], 3);
        double cI1 = 1.0 / (m * sinI);
        double cI2 = cosI * cI1;
        KeplerianOrbit.fillHalfRow(cI2, dCP, -cI1, dcZP, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(cI2, dCV, -cI1, dcZV, jacobian[2], 3);
        double cP1 = y * oOm;
        double cP2 = -x * oOm;
        double cP3 = -(mx * cP1 + my * cP2);
        double cP4 = cP3 * oOm;
        double cP5 = -1.0 / (r2 * sinI * sinI);
        double cP6 = z * cP5;
        double cP7 = cP3 * cP5;
        Vector3D dacP = new Vector3D(cP1, dcXP, cP2, dcYP, cP4, dCP, oOm, new Vector3D(-my, mx, 0.0));
        Vector3D dacV = new Vector3D(cP1, dcXV, cP2, dcYV, cP4, dCV);
        Vector3D dpoP = new Vector3D(cP6, dacP, cP7, Vector3D.PLUS_K);
        Vector3D dpoV = new Vector3D(cP6, dacV);
        double re2 = r2 * this.e * this.e;
        double recOre2 = (p - r) / re2;
        double resOre2 = pv * mOMu / re2;
        Vector3D dreP = new Vector3D(mOMu, velocity, pv / mu, dCP);
        Vector3D dreV = new Vector3D(mOMu, position, pv / mu, dCV);
        Vector3D davP = new Vector3D(-resOre2, dpP, recOre2, dreP, resOre2 / r, position);
        Vector3D davV = new Vector3D(-resOre2, dpV, recOre2, dreV);
        KeplerianOrbit.fillHalfRow(1.0, dpoP, -1.0, davP, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(1.0, dpoV, -1.0, davV, jacobian[3], 3);
        double cO0 = cI1 * cI1;
        double cO1 = mx * cO0;
        double cO2 = -my * cO0;
        KeplerianOrbit.fillHalfRow(cO1, dcYP, cO2, dcXP, jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(cO1, dcYV, cO2, dcXV, jacobian[4], 3);
        double s2a = pv / (2.0 * absA);
        double oObux = 1.0 / FastMath.sqrt(m * m + mu * absA);
        double scasbu = pv * oObux;
        Vector3D dauP = new Vector3D(1.0 / sqrtMuA, velocity, -s2a / sqrtMuA, vectorAR);
        Vector3D dauV = new Vector3D(1.0 / sqrtMuA, position, -s2a / sqrtMuA, vectorARDot);
        Vector3D dbuP = new Vector3D(oObux * mu / 2.0, vectorAR, m * oObux, dCP);
        Vector3D dbuV = new Vector3D(oObux * mu / 2.0, vectorARDot, m * oObux, dCV);
        Vector3D dcuP = new Vector3D(oObux, velocity, -scasbu * oObux, dbuP);
        Vector3D dcuV = new Vector3D(oObux, position, -scasbu * oObux, dbuV);
        KeplerianOrbit.fillHalfRow(1.0, dauP, -this.e / (1.0 + rOa), dcuP, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(1.0, dauV, -this.e / (1.0 + rOa), dcuV, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianEccentricWrtCartesianElliptical();
        }
        return this.computeJacobianEccentricWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianEccentricWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianElliptical();
        double eccentricAnomaly = this.getEccentricAnomaly();
        double cosE = FastMath.cos(eccentricAnomaly);
        double sinE = FastMath.sin(eccentricAnomaly);
        double aOr = 1.0 / (1.0 - this.e * cosE);
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aOr * (anomalyRow[j] + sinE * eRow[j]);
        }
        return jacobian;
    }

    private double[][] computeJacobianEccentricWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianHyperbolic();
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh(H);
        double sinhH = FastMath.sinh(H);
        double absaOr = 1.0 / (this.e * coshH - 1.0);
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = absaOr * (anomalyRow[j] - sinhH * eRow[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        if (this.a > 0.0) {
            return this.computeJacobianTrueWrtCartesianElliptical();
        }
        return this.computeJacobianTrueWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianTrueWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianElliptical();
        double e2 = this.e * this.e;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt(oMe2);
        double eccentricAnomaly = this.getEccentricAnomaly();
        double cosE = FastMath.cos(eccentricAnomaly);
        double sinE = FastMath.sin(eccentricAnomaly);
        double aOr = 1.0 / (1.0 - this.e * cosE);
        double aFactor = epsilon * aOr;
        double eFactor = sinE * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] + eFactor * eRow[j];
        }
        return jacobian;
    }

    private double[][] computeJacobianTrueWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianHyperbolic();
        double e2 = this.e * this.e;
        double e2Mo = e2 - 1.0;
        double epsilon = FastMath.sqrt(e2Mo);
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh(H);
        double sinhH = FastMath.sinh(H);
        double aOr = 1.0 / (this.e * coshH - 1.0);
        double aFactor = epsilon * aOr;
        double eFactor = sinhH * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] - eFactor * eRow[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double absA = FastMath.abs(this.a);
        double n = FastMath.sqrt(gm / absA) / absA;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = FastMath.abs(1.0 - this.e * this.e);
                double ksi = 1.0 + this.e * FastMath.cos(this.v);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = FastMath.abs(1.0 - this.e * this.e);
                double ksi = 1.0 + this.e * FastMath.cos(this.v);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt(oMe2));
                break;
            }
            default: {
                throw new OrekitInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("Keplerian parameters: ").append('{').append("a: ").append(this.a).append("; e: ").append(this.e).append("; i: ").append(FastMath.toDegrees(this.i)).append("; pa: ").append(FastMath.toDegrees(this.pa)).append("; raan: ").append(FastMath.toDegrees(this.raan)).append("; v: ").append(FastMath.toDegrees(this.v)).append(";}").toString();
    }

    @DefaultDataContext
    private Object writeReplace() {
        return new DTO(this);
    }

    static {
        double k1 = 11.42477796076938;
        double k2 = 2.141592653589793;
        double k3 = 17.84955592153876;
        A = 1.2043347651023166;
        B = 4.64788969626918;
    }

    @DefaultDataContext
    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20170414L;
        private double[] d;
        private final Frame frame;

        private DTO(KeplerianOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            double epoch = FastMath.floor(pv.getDate().durationFrom(j2000Epoch));
            double offset = pv.getDate().durationFrom(j2000Epoch.shiftedBy(epoch));
            this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.e, orbit.i, orbit.pa, orbit.raan, orbit.v, orbit.aDot, orbit.eDot, orbit.iDot, orbit.paDot, orbit.raanDot, orbit.vDot} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.e, orbit.i, orbit.pa, orbit.raan, orbit.v};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            if (this.d.length >= 15) {
                return new KeplerianOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], PositionAngle.TRUE, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
            }
            return new KeplerianOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

