/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.propagation.numerical;

import java.util.Arrays;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Euclidean3D;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.ode.ODEIntegrator;
import org.hipparchus.ode.nonstiff.ClassicalRungeKuttaIntegrator;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.attitudes.Attitude;
import org.orekit.attitudes.AttitudeProvider;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.gnss.GLONASSEphemeris;
import org.orekit.orbits.CartesianOrbit;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.propagation.PropagationType;
import org.orekit.propagation.Propagator;
import org.orekit.propagation.SpacecraftState;
import org.orekit.propagation.analytical.gnss.GLONASSOrbitalElements;
import org.orekit.propagation.integration.AbstractIntegratedPropagator;
import org.orekit.propagation.integration.StateMapper;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.GLONASSDate;
import org.orekit.utils.AbsolutePVCoordinates;
import org.orekit.utils.IERSConventions;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class GLONASSNumericalPropagator
extends AbstractIntegratedPropagator {
    private static final double GLONASS_J20 = 0.00108262575;
    private static final double GLONASS_EARTH_EQUATORIAL_RADIUS = 6378136.0;
    private static final double GLONASS_AV = 7.2921151467E-5;
    private static final double A;
    private static final double B;
    private final GLONASSOrbitalElements glonassOrbit;
    private final GLONASSDate initDate;
    private final double mass;
    private final Frame eci;
    private double[] moonElements;
    private double[] sunElements;
    private final boolean isAccAvailable;
    private final DataContext dataContext;

    public GLONASSNumericalPropagator(Builder builder) {
        super(builder.integrator, PropagationType.MEAN);
        this.dataContext = builder.dataContext;
        this.isAccAvailable = builder.isAccAvailable;
        this.glonassOrbit = builder.orbit;
        this.eci = builder.eci;
        this.mass = builder.mass;
        this.initDate = new GLONASSDate(this.glonassOrbit.getDate(), this.dataContext.getTimeScales().getGLONASS());
        this.initMapper();
        this.setInitialState();
        this.setAttitudeProvider(builder.attitudeProvider);
        this.setOrbitType(OrbitType.CARTESIAN);
        this.setPositionAngleType(PositionAngle.TRUE);
        this.setMu(3.986004418E14);
        if (!this.isAccAvailable) {
            this.computeMoonElements(this.initDate);
            this.computeSunElements(this.initDate);
        }
    }

    public GLONASSOrbitalElements getGLONASSOrbitalElements() {
        return this.glonassOrbit;
    }

    @Override
    public SpacecraftState propagate(AbsoluteDate date) {
        SpacecraftState stateInInertial = super.propagate(date);
        PVCoordinates pvInPZ90 = this.getPVInPZ90(stateInInertial);
        AbsolutePVCoordinates absPV = new AbsolutePVCoordinates((Frame)this.dataContext.getFrames().getPZ9011(IERSConventions.IERS_2010, true), stateInInertial.getDate(), pvInPZ90);
        TimeStampedPVCoordinates pvInInertial = absPV.getPVCoordinates(this.eci);
        SpacecraftState transformedState = new SpacecraftState(new CartesianOrbit(pvInInertial, this.eci, pvInInertial.getDate(), 3.986004418E14), stateInInertial.getAttitude(), stateInInertial.getMass(), stateInInertial.getAdditionalStates());
        return transformedState;
    }

    private void setInitialState() {
        PVCoordinates pvInInertial = this.getPVInInertial(this.initDate);
        CartesianOrbit orbit = new CartesianOrbit(pvInInertial, this.eci, this.initDate.getDate(), 3.986004418E14);
        this.resetInitialState(new SpacecraftState((Orbit)orbit, this.mass));
    }

    private void computeMoonElements(GLONASSDate date) {
        this.moonElements = new double[4];
        double am = 3.84385243E8;
        double em = 0.054900489;
        double im = 0.089804108;
        double dtoJD = (this.glonassOrbit.getTime() - 10800.0) / 86400.0;
        double t = (date.getJD0() + dtoJD - 2451545.0) / 36525.0;
        double t2 = t * t;
        double eps = 0.4090926006 - 2.270711E-4 * t;
        double gammaM = 1.4547885346 + 71.0176852437 * t - 1.801481E-4 * t2;
        double omegaM = 2.1824391966 - 33.7570459536 * t + 3.62262E-5 * t2;
        double qm = 2.3555557435 + 8328.691425719 * t + 1.545547E-4 * t2;
        double cosOm = FastMath.cos(omegaM);
        double sinOm = FastMath.sin(omegaM);
        double cosIm = FastMath.cos(0.089804108);
        double sinIm = FastMath.sin(0.089804108);
        double cosEs = FastMath.cos(eps);
        double sinEs = FastMath.sin(eps);
        double cosGm = FastMath.cos(gammaM);
        double sinGm = FastMath.sin(gammaM);
        double psiStar = cosOm * sinIm;
        double etaStar = sinOm * sinIm;
        double epsStar = 1.0 - cosOm * cosOm * (1.0 - cosIm);
        double eps11 = sinOm * cosOm * (1.0 - cosIm);
        double eps12 = 1.0 - sinOm * sinOm * (1.0 - cosIm);
        double eta11 = epsStar * cosEs - psiStar * sinEs;
        double eta12 = eps11 * cosEs + etaStar * sinEs;
        double psi11 = epsStar * sinEs + psiStar * cosEs;
        double psi12 = eps11 * sinEs - etaStar * cosEs;
        double ek = this.getEccentricAnomaly(qm, 0.054900489);
        double vk = this.getTrueAnomaly(ek, 0.054900489);
        double sinVk = FastMath.sin(vk);
        double cosVk = FastMath.cos(vk);
        double epsM = eps11 * (sinVk * cosGm + cosVk * sinGm) + eps12 * (cosVk * cosGm - sinVk * sinGm);
        double etaM = eta11 * (sinVk * cosGm + cosVk * sinGm) + eta12 * (cosVk * cosGm - sinVk * sinGm);
        double psiM = psi11 * (sinVk * cosGm + cosVk * sinGm) + psi12 * (cosVk * cosGm - sinVk * sinGm);
        double rm = 3.84385243E8 * (1.0 - 0.054900489 * FastMath.cos(ek));
        this.moonElements[0] = epsM;
        this.moonElements[1] = etaM;
        this.moonElements[2] = psiM;
        this.moonElements[3] = rm;
    }

    private void computeSunElements(GLONASSDate date) {
        this.sunElements = new double[4];
        double as = 1.49598E11;
        double es = 0.016719;
        double dtoJD = (this.glonassOrbit.getTime() - 10800.0) / 86400.0;
        double t = (date.getJD0() + dtoJD - 2451545.0) / 36525.0;
        double t2 = t * t;
        double eps = 0.4090926006 - 2.270711E-4 * t;
        double ws = -7.6281824375 + 0.0300101976 * t + 7.9741E-6 * t2;
        double qs = 6.2400601269 + 628.3019551714 * t - 2.682E-6 * t2;
        double ek = this.getEccentricAnomaly(qs, 0.016719);
        double vk = this.getTrueAnomaly(ek, 0.016719);
        double sinVk = FastMath.sin(vk);
        double cosVk = FastMath.cos(vk);
        double cosWs = FastMath.cos(ws);
        double sinWs = FastMath.sin(ws);
        double cosEs = FastMath.cos(eps);
        double sinEs = FastMath.sin(eps);
        double epsS = cosVk * cosWs - sinVk * sinWs;
        double etaS = cosEs * (sinVk * cosWs + cosVk * sinWs);
        double psiS = sinEs * (sinVk * cosWs + cosVk * sinWs);
        double rs = 1.49598E11 * (1.0 - 0.016719 * FastMath.cos(ek));
        this.sunElements[0] = epsS;
        this.sunElements[1] = etaS;
        this.sunElements[2] = psiS;
        this.sunElements[3] = rs;
    }

    private double getEccentricAnomaly(double M, double e) {
        double w;
        double E;
        double reducedM = MathUtils.normalizeAngle(M, 0.0);
        if (FastMath.abs(reducedM) < 0.16666666666666666) {
            E = reducedM + e * (FastMath.cbrt(6.0 * reducedM) - reducedM);
        } else if (reducedM < 0.0) {
            w = Math.PI + reducedM;
            E = reducedM + e * (A * w / (B - w) - Math.PI - reducedM);
        } else {
            w = Math.PI - reducedM;
            E = reducedM + e * (Math.PI - A * w / (B - w) - reducedM);
        }
        double e1 = 1.0 - e;
        boolean noCancellationRisk = e1 + E * E / 6.0 >= 0.1;
        for (int j = 0; j < 2; ++j) {
            double fd;
            double f;
            double fdd = e * FastMath.sin(E);
            double fddd = e * FastMath.cos(E);
            if (noCancellationRisk) {
                f = E - fdd - reducedM;
                fd = 1.0 - fddd;
            } else {
                f = GLONASSNumericalPropagator.eMeSinE(E, e) - reducedM;
                double s = FastMath.sin(0.5 * E);
                fd = e1 + 2.0 * e * s * s;
            }
            double dee = f * fd / (0.5 * f * fdd - fd * fd);
            double w2 = fd + 0.5 * dee * (fdd + dee * fddd / 3.0);
            E -= (f - dee * ((fd += dee * (fdd + 0.5 * dee * fddd)) - w2)) / fd;
        }
        return E += M - reducedM;
    }

    private static double eMeSinE(double E, double e) {
        double x = (1.0 - e) * FastMath.sin(E);
        double mE2 = -E * E;
        double term = E;
        double d = 0.0;
        double x0 = Double.NaN;
        while (!Double.valueOf(x).equals(x0)) {
            x0 = x;
            x -= (term *= mE2 / ((d += 2.0) * (d + 1.0)));
        }
        return x;
    }

    private double getTrueAnomaly(double ek, double ecc) {
        double svk = FastMath.sqrt(1.0 - ecc * ecc) * FastMath.sin(ek);
        double cvk = FastMath.cos(ek) - ecc;
        return FastMath.atan2(svk, cvk);
    }

    public PVCoordinates getPVInPZ90(SpacecraftState state) {
        double dt = state.getDate().durationFrom(this.initDate.getDate());
        TimeStampedPVCoordinates pv = state.getPVCoordinates();
        Vector3D pos = pv.getPosition();
        Vector3D vel = pv.getVelocity();
        double x0 = pos.getX();
        double y0 = pos.getY();
        double z0 = pos.getZ();
        double vx0 = vel.getX();
        double vy0 = vel.getY();
        double vz0 = vel.getZ();
        GLONASSDate gloDate = new GLONASSDate(state.getDate(), this.dataContext.getTimeScales().getGLONASS());
        double gmst = gloDate.getGMST();
        double ti = this.glonassOrbit.getTime() + dt;
        double s = gmst + 7.2921151467E-5 * (ti - 10800.0);
        double cosS = FastMath.cos(s);
        double sinS = FastMath.sin(s);
        double x = x0 * cosS + y0 * sinS;
        double y = -x0 * sinS + y0 * cosS;
        double z = z0;
        double vx = vx0 * cosS + vy0 * sinS + 7.2921151467E-5 * y;
        double vy = -vx0 * sinS + vy0 * cosS - 7.2921151467E-5 * x;
        double vz = vz0;
        return new PVCoordinates(new Vector3D(x, y, z), new Vector3D(vx, vy, vz));
    }

    private PVCoordinates getPVInInertial(GLONASSDate date) {
        double gmst = date.getGMST();
        double time = this.glonassOrbit.getTime();
        double dt = time - 10800.0;
        double s = gmst + 7.2921151467E-5 * dt;
        double cosS = FastMath.cos(s);
        double sinS = FastMath.sin(s);
        double x0 = this.glonassOrbit.getX() * cosS - this.glonassOrbit.getY() * sinS;
        double y0 = this.glonassOrbit.getX() * sinS + this.glonassOrbit.getY() * cosS;
        double z0 = this.glonassOrbit.getZ();
        double vx0 = this.glonassOrbit.getXDot() * cosS - this.glonassOrbit.getYDot() * sinS - 7.2921151467E-5 * y0;
        double vy0 = this.glonassOrbit.getXDot() * sinS + this.glonassOrbit.getYDot() * cosS + 7.2921151467E-5 * x0;
        double vz0 = this.glonassOrbit.getZDot();
        return new PVCoordinates(new Vector3D(x0, y0, z0), new Vector3D(vx0, vy0, vz0));
    }

    @Override
    protected StateMapper createMapper(AbsoluteDate referenceDate, double mu, OrbitType orbitType, PositionAngle positionAngleType, AttitudeProvider attitudeProvider, Frame frame) {
        return new Mapper(referenceDate, mu, orbitType, positionAngleType, attitudeProvider, frame);
    }

    @Override
    protected AbstractIntegratedPropagator.MainStateEquations getMainStateEquations(ODEIntegrator integ) {
        return new Main();
    }

    static {
        double k1 = 11.42477796076938;
        double k2 = 2.141592653589793;
        double k3 = 17.84955592153876;
        A = 1.2043347651023166;
        B = 4.64788969626918;
    }

    private class Main
    implements AbstractIntegratedPropagator.MainStateEquations {
        private final double[] yDot = new double[7];

        Main() {
        }

        @Override
        public double[] computeDerivatives(SpacecraftState state) {
            Vector<Euclidean3D> acc;
            GLONASSDate gloDate = new GLONASSDate(state.getDate(), GLONASSNumericalPropagator.this.dataContext.getTimeScales().getGLONASS());
            Vector3D vel = state.getPVCoordinates().getVelocity();
            Vector3D pos = state.getPVCoordinates().getPosition();
            Arrays.fill(this.yDot, 0.0);
            this.yDot[0] = this.yDot[0] + vel.getX();
            this.yDot[1] = this.yDot[1] + vel.getY();
            this.yDot[2] = this.yDot[2] + vel.getZ();
            double x0 = pos.getX();
            double y0 = pos.getY();
            double z0 = pos.getZ();
            double r = pos.getNorm();
            double r2 = r * r;
            double oor = 1.0 / r;
            double oor2 = 1.0 / r2;
            double x = x0 * oor;
            double y = y0 * oor;
            double z = z0 * oor;
            double g = 3.986004418E14 * oor2;
            double ro = 6378136.0 * oor;
            this.yDot[3] = this.yDot[3] + x * (-g + -0.0016239386250000002 * g * ro * ro * (1.0 - 5.0 * z * z));
            this.yDot[4] = this.yDot[4] + y * (-g + -0.0016239386250000002 * g * ro * ro * (1.0 - 5.0 * z * z));
            this.yDot[5] = this.yDot[5] + z * (-g + -0.0016239386250000002 * g * ro * ro * (3.0 - 5.0 * z * z));
            if (GLONASSNumericalPropagator.this.isAccAvailable) {
                acc = this.getLuniSolarPerturbations(gloDate);
            } else {
                Vector3D accMoon = this.computeLuniSolarPerturbations(state, GLONASSNumericalPropagator.this.moonElements[0], GLONASSNumericalPropagator.this.moonElements[1], GLONASSNumericalPropagator.this.moonElements[2], GLONASSNumericalPropagator.this.moonElements[3], GLONASSNumericalPropagator.this.dataContext.getCelestialBodies().getMoon().getGM());
                Vector3D accSun = this.computeLuniSolarPerturbations(state, GLONASSNumericalPropagator.this.sunElements[0], GLONASSNumericalPropagator.this.sunElements[1], GLONASSNumericalPropagator.this.sunElements[2], GLONASSNumericalPropagator.this.sunElements[3], GLONASSNumericalPropagator.this.dataContext.getCelestialBodies().getSun().getGM());
                acc = accMoon.add((Vector)accSun);
            }
            this.yDot[3] = this.yDot[3] + acc.getX();
            this.yDot[4] = this.yDot[4] + acc.getY();
            this.yDot[5] = this.yDot[5] + acc.getZ();
            return (double[])this.yDot.clone();
        }

        private Vector3D computeLuniSolarPerturbations(SpacecraftState state, double eps, double eta, double psi, double r, double g) {
            TimeStampedPVCoordinates pv = state.getPVCoordinates();
            double oor = 1.0 / r;
            double oor2 = oor * oor;
            double x = pv.getPosition().getX() * oor;
            double y = pv.getPosition().getY() * oor;
            double z = pv.getPosition().getZ() * oor;
            double gm = g * oor2;
            double epsmX = eps - x;
            double etamY = eta - y;
            double psimZ = psi - z;
            Vector3D vector = new Vector3D(epsmX, etamY, psimZ);
            double d2 = vector.getNormSq();
            double deltaM = FastMath.sqrt(d2) * d2;
            double accX = gm * (epsmX / deltaM - eps);
            double accY = gm * (etamY / deltaM - eta);
            double accZ = gm * (psimZ / deltaM - psi);
            return new Vector3D(accX, accY, accZ);
        }

        private Vector3D getLuniSolarPerturbations(GLONASSDate date) {
            double gmst = date.getGMST();
            double time = GLONASSNumericalPropagator.this.glonassOrbit.getTime();
            double dt = time - 10800.0;
            double s = gmst + 7.2921151467E-5 * dt;
            double cosS = FastMath.cos(s);
            double sinS = FastMath.sin(s);
            double accX = GLONASSNumericalPropagator.this.glonassOrbit.getXDotDot() * cosS - GLONASSNumericalPropagator.this.glonassOrbit.getYDotDot() * sinS;
            double accY = GLONASSNumericalPropagator.this.glonassOrbit.getXDotDot() * sinS + GLONASSNumericalPropagator.this.glonassOrbit.getYDotDot() * cosS;
            double accZ = GLONASSNumericalPropagator.this.glonassOrbit.getZDotDot();
            return new Vector3D(accX, accY, accZ);
        }
    }

    private static class Mapper
    extends StateMapper {
        Mapper(AbsoluteDate referenceDate, double mu, OrbitType orbitType, PositionAngle positionAngleType, AttitudeProvider attitudeProvider, Frame frame) {
            super(referenceDate, mu, orbitType, positionAngleType, attitudeProvider, frame);
        }

        @Override
        public SpacecraftState mapArrayToState(AbsoluteDate date, double[] y, double[] yDot, PropagationType type) {
            double mass = y[6];
            if (mass <= 0.0) {
                throw new OrekitException((Localizable)OrekitMessages.SPACECRAFT_MASS_BECOMES_NEGATIVE, mass);
            }
            Orbit orbit = this.getOrbitType().mapArrayToOrbit(y, yDot, this.getPositionAngleType(), date, this.getMu(), this.getFrame());
            Attitude attitude = this.getAttitudeProvider().getAttitude(orbit, date, this.getFrame());
            return new SpacecraftState(orbit, attitude, mass);
        }

        @Override
        public void mapStateToArray(SpacecraftState state, double[] y, double[] yDot) {
            this.getOrbitType().mapOrbitToArray(state.getOrbit(), this.getPositionAngleType(), y, yDot);
            y[6] = state.getMass();
        }
    }

    public static class Builder {
        private final GLONASSEphemeris orbit;
        private final ClassicalRungeKuttaIntegrator integrator;
        private final boolean isAccAvailable;
        private AttitudeProvider attitudeProvider;
        private double mass = 1000.0;
        private Frame eci = null;
        private DataContext dataContext;

        @DefaultDataContext
        public Builder(ClassicalRungeKuttaIntegrator integrator, GLONASSEphemeris glonassOrbElt, boolean isAccAvailable) {
            this.isAccAvailable = isAccAvailable;
            this.integrator = integrator;
            this.orbit = glonassOrbElt;
            this.dataContext = DataContext.getDefault();
            this.eci = this.dataContext.getFrames().getEME2000();
            this.attitudeProvider = Propagator.getDefaultLaw(this.dataContext.getFrames());
        }

        public Builder attitudeProvider(AttitudeProvider userProvider) {
            this.attitudeProvider = userProvider;
            return this;
        }

        public Builder mass(double userMass) {
            this.mass = userMass;
            return this;
        }

        public Builder eci(Frame inertial) {
            this.eci = inertial;
            return this;
        }

        public Builder dataContext(DataContext context) {
            this.dataContext = context;
            return this;
        }

        public GLONASSNumericalPropagator build() {
            return new GLONASSNumericalPropagator(this);
        }
    }
}

