"use strict";
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("./util");
class StackWithPostProcessor extends lib_1.Stack {
    // ...
    _toCloudFormation() {
        const template = super._toCloudFormation();
        // manipulate template (e.g. rename "Key" to "key")
        template.Resources.myResource.Properties.Environment.key =
            template.Resources.myResource.Properties.Environment.Key;
        delete template.Resources.myResource.Properties.Environment.Key;
        return template;
    }
}
module.exports = {
    'a stack can be serialized into a CloudFormation template, initially it\'s empty'(test) {
        const stack = new lib_1.Stack();
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'stack objects have some template-level propeties, such as Description, Version, Transform'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.templateFormatVersion = 'MyTemplateVersion';
        stack.templateOptions.description = 'This is my description';
        stack.templateOptions.transform = 'SAMy';
        test.deepEqual(util_1.toCloudFormation(stack), {
            Description: 'This is my description',
            AWSTemplateFormatVersion: 'MyTemplateVersion',
            Transform: 'SAMy'
        });
        test.done();
    },
    'Stack.isStack indicates that a construct is a stack'(test) {
        const stack = new lib_1.Stack();
        const c = new lib_1.Construct(stack, 'Construct');
        test.ok(lib_1.Stack.isStack(stack));
        test.ok(!lib_1.Stack.isStack(c));
        test.done();
    },
    'stack.id is not included in the logical identities of resources within it'(test) {
        const stack = new lib_1.Stack(undefined, 'MyStack');
        new lib_1.CfnResource(stack, 'MyResource', { type: 'MyResourceType' });
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType' } } });
        test.done();
    },
    'stack.templateOptions can be used to set template-level options'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.description = 'StackDescription';
        stack.templateOptions.templateFormatVersion = 'TemplateVersion';
        stack.templateOptions.transform = 'DeprecatedField';
        stack.templateOptions.transforms = ['Transform'];
        stack.templateOptions.metadata = {
            MetadataKey: 'MetadataValue'
        };
        test.deepEqual(util_1.toCloudFormation(stack), {
            Description: 'StackDescription',
            Transform: ['DeprecatedField', 'Transform'],
            AWSTemplateFormatVersion: 'TemplateVersion',
            Metadata: { MetadataKey: 'MetadataValue' }
        });
        test.done();
    },
    // This approach will only apply to TypeScript code, but at least it's a temporary
    // workaround for people running into issues caused by SDK-3003.
    // We should come up with a proper solution that involved jsii callbacks (when they exist)
    // so this can be implemented by jsii languages as well.
    'Overriding `Stack._toCloudFormation` allows arbitrary post-processing of the generated template during synthesis'(test) {
        const stack = new StackWithPostProcessor();
        new lib_1.CfnResource(stack, 'myResource', {
            type: 'AWS::MyResource',
            properties: {
                MyProp1: 'hello',
                MyProp2: 'howdy',
                Environment: {
                    Key: 'value'
                }
            }
        });
        test.deepEqual(stack._toCloudFormation(), { Resources: { myResource: { Type: 'AWS::MyResource',
                    Properties: { MyProp1: 'hello',
                        MyProp2: 'howdy',
                        Environment: { key: 'value' } } } } });
        test.done();
    },
    'Stack.getByPath can be used to find any CloudFormation element (Parameter, Output, etc)'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        const o = new lib_1.CfnOutput(stack, 'MyOutput', { value: 'boom' });
        const c = new lib_1.CfnCondition(stack, 'MyCondition');
        test.equal(stack.node.findChild(p.node.path), p);
        test.equal(stack.node.findChild(o.node.path), o);
        test.equal(stack.node.findChild(c.node.path), c);
        test.done();
    },
    'Stack names can have hyphens in them'(test) {
        const root = new lib_1.App();
        new lib_1.Stack(root, 'Hello-World');
        // Did not throw
        test.done();
    },
    'Include should support non-hash top-level template elements like "Description"'(test) {
        const stack = new lib_1.Stack();
        const template = {
            Description: 'hello, world'
        };
        new lib_1.CfnInclude(stack, 'Include', { template });
        const output = util_1.toCloudFormation(stack);
        test.equal(typeof output.Description, 'string');
        test.done();
    },
    'Pseudo values attached to one stack can be referenced in another stack'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        // THEN
        const assembly = app.synth();
        const template1 = assembly.getStack(stack1.stackName).template;
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack references are detected in resource properties'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', { type: 'AWS::Some::Resource', properties: {
                someProperty: new intrinsic_1.Intrinsic(resource1.ref),
            } });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template2, {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack references in lazy tokens work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: lib_1.Lazy.stringValue({ produce: () => account1 }) });
        const assembly = app.synth();
        const template1 = assembly.getStack(stack1.stackName).template;
        const template2 = assembly.getStack(stack2.stackName).template;
        // THEN
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack use of Region and account returns nonscoped intrinsic because the two stacks must be in the same region anyway'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnOutput(stack2, 'DemOutput', { value: stack1.region });
        new lib_1.CfnOutput(stack2, 'DemAccount', { value: stack1.account });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        test.deepEqual(template2, {
            Outputs: {
                DemOutput: {
                    Value: { Ref: 'AWS::Region' },
                },
                DemAccount: {
                    Value: { Ref: 'AWS::AccountId' },
                }
            }
        });
        test.done();
    },
    'cross-stack references in strings work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: `TheAccountIs${account1}` });
        const assembly = app.synth();
        const template2 = assembly.getStack(stack2.stackName).template;
        // THEN
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::Join': ['', ['TheAccountIs', { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }]] }
                }
            }
        });
        test.done();
    },
    'Stacks can be children of other stacks (substack) and they will be synthesized separately'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStack(parentStack.stackName).template, { Resources: { MyParentResource: { Type: 'Resource::Parent' } } });
        test.deepEqual(assembly.getStack(childStack.stackName).template, { Resources: { MyChildResource: { Type: 'Resource::Child' } } });
        test.done();
    },
    'cross-stack reference (substack references parent stack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const parentResource = new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', {
            type: 'Resource::Child',
            properties: {
                ChildProp: parentResource.getAtt('AttOfParentResource')
            }
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStack(parentStack.stackName).template, {
            Resources: { MyParentResource: { Type: 'Resource::Parent' } },
            Outputs: { ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E: {
                    Value: { 'Fn::GetAtt': ['MyParentResource', 'AttOfParentResource'] },
                    Export: { Name: 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E' }
                }
            }
        });
        test.deepEqual(assembly.getStack(childStack.stackName).template, {
            Resources: {
                MyChildResource: {
                    Type: 'Resource::Child',
                    Properties: {
                        ChildProp: {
                            'Fn::ImportValue': 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E'
                        }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack reference (parent stack references substack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const childResource = new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        new lib_1.CfnResource(parentStack, 'MyParentResource', {
            type: 'Resource::Parent',
            properties: {
                ParentProp: childResource.getAtt('AttributeOfChildResource')
            }
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStack(parentStack.stackName).template, {
            Resources: {
                MyParentResource: {
                    Type: 'Resource::Parent',
                    Properties: {
                        ParentProp: { 'Fn::ImportValue': 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' }
                    }
                }
            }
        });
        test.deepEqual(assembly.getStack(childStack.stackName).template, {
            Resources: {
                MyChildResource: { Type: 'Resource::Child' }
            },
            Outputs: {
                ExportsOutputFnGetAttMyChildResourceAttributeOfChildResource52813264: {
                    Value: { 'Fn::GetAtt': ['MyChildResource', 'AttributeOfChildResource'] },
                    Export: { Name: 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' }
                }
            }
        });
        test.done();
    },
    'cannot create cyclic reference between stacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        const account2 = new lib_1.ScopedAws(stack2).accountId;
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        new lib_1.CfnParameter(stack1, 'SomeParameter', { type: 'String', default: account2 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
            // tslint:disable-next-line:max-line-length
        }, "'Stack2' depends on 'Stack1' (Stack2/SomeParameter -> Stack1.AWS::AccountId). Adding this dependency (Stack1/SomeParameter -> Stack2.AWS::AccountId) would create a cyclic reference.");
        test.done();
    },
    'stacks know about their dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        lib_1.ConstructNode.prepare(app.node);
        // THEN
        test.deepEqual(stack2.dependencies.map(s => s.node.id), ['Stack1']);
        test.done();
    },
    'cannot create references to stacks in other regions/accounts'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2', { env: { account: '123456789012', region: 'es-norst-2' } });
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
        }, /Can only reference cross stacks in the same region and account/);
        test.done();
    },
    'urlSuffix does not imply a stack dependency'(test) {
        // GIVEN
        const app = new lib_1.App();
        const first = new lib_1.Stack(app, 'First');
        const second = new lib_1.Stack(app, 'Second');
        // WHEN
        new lib_1.CfnOutput(second, 'Output', {
            value: first.urlSuffix
        });
        // THEN
        app.synth();
        test.equal(second.dependencies.length, 0);
        test.done();
    },
    'stack with region supplied via props returns literal value'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        // THEN
        test.equal(stack.resolve(stack.region), 'es-norst-1');
        test.done();
    },
    'overrideLogicalId(id) can be used to override the logical ID of a resource'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bonjour = new lib_1.CfnResource(stack, 'BonjourResource', { type: 'Resource::Type' });
        // { Ref } and { GetAtt }
        new lib_1.CfnResource(stack, 'RefToBonjour', { type: 'Other::Resource', properties: {
                RefToBonjour: bonjour.ref,
                GetAttBonjour: bonjour.getAtt('TheAtt').toString()
            } });
        bonjour.overrideLogicalId('BOOM');
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack), { Resources: { BOOM: { Type: 'Resource::Type' },
                RefToBonjour: { Type: 'Other::Resource',
                    Properties: { RefToBonjour: { Ref: 'BOOM' },
                        GetAttBonjour: { 'Fn::GetAtt': ['BOOM', 'TheAtt'] } } } } });
        test.done();
    },
    'Stack name can be overridden via properties'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, 'Stack', { stackName: 'otherName' });
        // THEN
        test.deepEqual(stack.stackName, 'otherName');
        test.done();
    },
    'Stack name is inherited from App name if available'(test) {
        // WHEN
        const root = new lib_1.App();
        const app = new lib_1.Construct(root, 'Prod');
        const stack = new lib_1.Stack(app, 'Stack');
        // THEN
        test.deepEqual(stack.stackName, 'ProdStackD5279B22');
        test.done();
    },
    'stack construct id does not go through stack name validation if there is an explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack = new lib_1.Stack(app, 'invalid as : stack name, but thats fine', {
            stackName: 'valid-stack-name'
        });
        // THEN
        const session = app.synth();
        test.deepEqual(stack.stackName, 'valid-stack-name');
        test.ok(session.tryGetArtifact('valid-stack-name'));
        test.done();
    },
    'stack validation is performed on explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stack(app, 'boom', { stackName: 'invalid:stack:name' }), /Stack name must match the regular expression/);
        test.done();
    },
    'Stack.of(stack) returns the correct stack'(test) {
        const stack = new lib_1.Stack();
        test.same(lib_1.Stack.of(stack), stack);
        const parent = new lib_1.Construct(stack, 'Parent');
        const construct = new lib_1.Construct(parent, 'Construct');
        test.same(lib_1.Stack.of(construct), stack);
        test.done();
    },
    'Stack.of() throws when there is no parent Stack'(test) {
        const root = new lib_1.Construct(undefined, 'Root');
        const construct = new lib_1.Construct(root, 'Construct');
        test.throws(() => lib_1.Stack.of(construct), /No stack could be identified for the construct at path/);
        test.done();
    },
    'Stack.of() works for substacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'ParentStack');
        const parentResource = new lib_1.CfnResource(parentStack, 'ParentResource', { type: 'parent::resource' });
        // we will define a substack under the /resource/... just for giggles.
        const childStack = new lib_1.Stack(parentResource, 'ChildStack');
        const childResource = new lib_1.CfnResource(childStack, 'ChildResource', { type: 'child::resource' });
        // THEN
        test.same(lib_1.Stack.of(parentStack), parentStack);
        test.same(lib_1.Stack.of(parentResource), parentStack);
        test.same(lib_1.Stack.of(childStack), childStack);
        test.same(lib_1.Stack.of(childResource), childStack);
        test.done();
    },
    'stack.availabilityZones falls back to Fn::GetAZ[0],[2] if region is not specified'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        // WHEN
        const azs = stack.availabilityZones;
        // THEN
        test.deepEqual(stack.resolve(azs), [
            { "Fn::Select": [0, { "Fn::GetAZs": "" }] },
            { "Fn::Select": [1, { "Fn::GetAZs": "" }] }
        ]);
        test.done();
    },
    'stack.templateFile contains the name of the cloudformation output'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'MyRealStack2' });
        // THEN
        test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
        test.deepEqual(stack2.templateFile, 'MyRealStack2.template.json');
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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