"""**Vector store** stores embedded data and performs vector search.

One of the most common ways to store and search over unstructured data is to
embed it and store the resulting embedding vectors, and then query the store
and retrieve the data that are 'most similar' to the embedded query.

**Class hierarchy:**

.. code-block::

    VectorStore --> <name>  # Examples: Annoy, FAISS, Milvus

    BaseRetriever --> VectorStoreRetriever --> <name>Retriever  # Example: VespaRetriever

**Main helpers:**

.. code-block::

    Embeddings, Document
"""  # noqa: E501

from typing import TYPE_CHECKING, Any

from aibaba_ai_core.vectorstores import VectorStore

from langchain._api import create_importer

if TYPE_CHECKING:
    from aibaba_ai_community.vectorstores import (
        FAISS,
        AlibabaCloudOpenSearch,
        AlibabaCloudOpenSearchSettings,
        AnalyticDB,
        Annoy,
        AstraDB,
        AtlasDB,
        AwaDB,
        AzureCosmosDBVectorSearch,
        AzureSearch,
        Bagel,
        Cassandra,
        Chroma,
        Clarifai,
        Clickhouse,
        ClickhouseSettings,
        DashVector,
        DatabricksVectorSearch,
        DeepLake,
        Dingo,
        DocArrayHnswSearch,
        DocArrayInMemorySearch,
        DuckDB,
        EcloudESVectorStore,
        ElasticKnnSearch,
        ElasticsearchStore,
        ElasticVectorSearch,
        Epsilla,
        Hologres,
        LanceDB,
        LLMRails,
        Marqo,
        MatchingEngine,
        Meilisearch,
        Milvus,
        MomentoVectorIndex,
        MongoDBAtlasVectorSearch,
        MyScale,
        MyScaleSettings,
        Neo4jVector,
        NeuralDBClientVectorStore,
        NeuralDBVectorStore,
        OpenSearchVectorSearch,
        PGEmbedding,
        PGVector,
        Pinecone,
        Qdrant,
        Redis,
        Rockset,
        ScaNN,
        SemaDB,
        SingleStoreDB,
        SKLearnVectorStore,
        SQLiteVSS,
        StarRocks,
        SupabaseVectorStore,
        Tair,
        TencentVectorDB,
        Tigris,
        TileDB,
        TimescaleVector,
        Typesense,
        USearch,
        Vald,
        Vearch,
        Vectara,
        VespaStore,
        Weaviate,
        Yellowbrick,
        ZepVectorStore,
        Zilliz,
    )

# Create a way to dynamically look up deprecated imports.
# Used to consolidate logic for raising deprecation warnings and
# handling optional imports.
DEPRECATED_LOOKUP = {
    "AlibabaCloudOpenSearch": "aibaba_ai_community.vectorstores",
    "AlibabaCloudOpenSearchSettings": "aibaba_ai_community.vectorstores",
    "AnalyticDB": "aibaba_ai_community.vectorstores",
    "Annoy": "aibaba_ai_community.vectorstores",
    "AstraDB": "aibaba_ai_community.vectorstores",
    "AtlasDB": "aibaba_ai_community.vectorstores",
    "AwaDB": "aibaba_ai_community.vectorstores",
    "AzureCosmosDBVectorSearch": "aibaba_ai_community.vectorstores",
    "AzureSearch": "aibaba_ai_community.vectorstores",
    "Bagel": "aibaba_ai_community.vectorstores",
    "Cassandra": "aibaba_ai_community.vectorstores",
    "Chroma": "aibaba_ai_community.vectorstores",
    "Clarifai": "aibaba_ai_community.vectorstores",
    "Clickhouse": "aibaba_ai_community.vectorstores",
    "ClickhouseSettings": "aibaba_ai_community.vectorstores",
    "DashVector": "aibaba_ai_community.vectorstores",
    "DatabricksVectorSearch": "aibaba_ai_community.vectorstores",
    "DeepLake": "aibaba_ai_community.vectorstores",
    "Dingo": "aibaba_ai_community.vectorstores",
    "DocArrayHnswSearch": "aibaba_ai_community.vectorstores",
    "DocArrayInMemorySearch": "aibaba_ai_community.vectorstores",
    "DuckDB": "aibaba_ai_community.vectorstores",
    "EcloudESVectorStore": "aibaba_ai_community.vectorstores",
    "ElasticKnnSearch": "aibaba_ai_community.vectorstores",
    "ElasticsearchStore": "aibaba_ai_community.vectorstores",
    "ElasticVectorSearch": "aibaba_ai_community.vectorstores",
    "Epsilla": "aibaba_ai_community.vectorstores",
    "FAISS": "aibaba_ai_community.vectorstores",
    "Hologres": "aibaba_ai_community.vectorstores",
    "LanceDB": "aibaba_ai_community.vectorstores",
    "LLMRails": "aibaba_ai_community.vectorstores",
    "Marqo": "aibaba_ai_community.vectorstores",
    "MatchingEngine": "aibaba_ai_community.vectorstores",
    "Meilisearch": "aibaba_ai_community.vectorstores",
    "Milvus": "aibaba_ai_community.vectorstores",
    "MomentoVectorIndex": "aibaba_ai_community.vectorstores",
    "MongoDBAtlasVectorSearch": "aibaba_ai_community.vectorstores",
    "MyScale": "aibaba_ai_community.vectorstores",
    "MyScaleSettings": "aibaba_ai_community.vectorstores",
    "Neo4jVector": "aibaba_ai_community.vectorstores",
    "NeuralDBClientVectorStore": "aibaba_ai_community.vectorstores",
    "NeuralDBVectorStore": "aibaba_ai_community.vectorstores",
    "NEuralDBVectorStore": "aibaba_ai_community.vectorstores",
    "OpenSearchVectorSearch": "aibaba_ai_community.vectorstores",
    "PGEmbedding": "aibaba_ai_community.vectorstores",
    "PGVector": "aibaba_ai_community.vectorstores",
    "Pinecone": "aibaba_ai_community.vectorstores",
    "Qdrant": "aibaba_ai_community.vectorstores",
    "Redis": "aibaba_ai_community.vectorstores",
    "Rockset": "aibaba_ai_community.vectorstores",
    "ScaNN": "aibaba_ai_community.vectorstores",
    "SemaDB": "aibaba_ai_community.vectorstores",
    "SingleStoreDB": "aibaba_ai_community.vectorstores",
    "SKLearnVectorStore": "aibaba_ai_community.vectorstores",
    "SQLiteVSS": "aibaba_ai_community.vectorstores",
    "StarRocks": "aibaba_ai_community.vectorstores",
    "SupabaseVectorStore": "aibaba_ai_community.vectorstores",
    "Tair": "aibaba_ai_community.vectorstores",
    "TencentVectorDB": "aibaba_ai_community.vectorstores",
    "Tigris": "aibaba_ai_community.vectorstores",
    "TileDB": "aibaba_ai_community.vectorstores",
    "TimescaleVector": "aibaba_ai_community.vectorstores",
    "Typesense": "aibaba_ai_community.vectorstores",
    "USearch": "aibaba_ai_community.vectorstores",
    "Vald": "aibaba_ai_community.vectorstores",
    "Vearch": "aibaba_ai_community.vectorstores",
    "Vectara": "aibaba_ai_community.vectorstores",
    "VespaStore": "aibaba_ai_community.vectorstores",
    "Weaviate": "aibaba_ai_community.vectorstores",
    "Yellowbrick": "aibaba_ai_community.vectorstores",
    "ZepVectorStore": "aibaba_ai_community.vectorstores",
    "Zilliz": "aibaba_ai_community.vectorstores",
}

_import_attribute = create_importer(__package__, deprecated_lookups=DEPRECATED_LOOKUP)


def __getattr__(name: str) -> Any:
    """Look up attributes dynamically."""
    return _import_attribute(name)


__all__ = [
    "AlibabaCloudOpenSearch",
    "AlibabaCloudOpenSearchSettings",
    "AnalyticDB",
    "Annoy",
    "AstraDB",
    "AtlasDB",
    "AwaDB",
    "AzureCosmosDBVectorSearch",
    "AzureSearch",
    "Bagel",
    "Cassandra",
    "Chroma",
    "Clarifai",
    "Clickhouse",
    "ClickhouseSettings",
    "DashVector",
    "DatabricksVectorSearch",
    "DeepLake",
    "Dingo",
    "DocArrayHnswSearch",
    "DocArrayInMemorySearch",
    "DuckDB",
    "EcloudESVectorStore",
    "ElasticKnnSearch",
    "ElasticsearchStore",
    "ElasticVectorSearch",
    "Epsilla",
    "FAISS",
    "Hologres",
    "LanceDB",
    "LLMRails",
    "Marqo",
    "MatchingEngine",
    "Meilisearch",
    "Milvus",
    "MomentoVectorIndex",
    "MongoDBAtlasVectorSearch",
    "MyScale",
    "MyScaleSettings",
    "Neo4jVector",
    "NeuralDBClientVectorStore",
    "NeuralDBVectorStore",
    "OpenSearchVectorSearch",
    "PGEmbedding",
    "PGVector",
    "Pinecone",
    "Qdrant",
    "Redis",
    "Rockset",
    "ScaNN",
    "SemaDB",
    "SingleStoreDB",
    "SKLearnVectorStore",
    "SQLiteVSS",
    "StarRocks",
    "SupabaseVectorStore",
    "Tair",
    "TencentVectorDB",
    "Tigris",
    "TileDB",
    "TimescaleVector",
    "Typesense",
    "USearch",
    "Vald",
    "Vearch",
    "Vectara",
    "VectorStore",
    "VespaStore",
    "Weaviate",
    "Yellowbrick",
    "ZepVectorStore",
    "Zilliz",
]
