# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
Version 2 of the L-Land model is a slight modification of |lland_v1|.
While |lland_v1| implements the Turc-Wendling approach for calculating the
reference evapotranspiration (|ET0|) for each hydrological response unit.
In contrast, |lland_v2| expects subbasin-wide potential evaporation values
(|PET|) to be calculated externally and adjusts them to the different
response units of the subbasin.  We created |lland_v2| for the flood
forecasting system of the German Free State of Saxony.

|lland_v1| should only be applied on daily step sized due to calculating
reference evaporation with the Turc-Wendling equation. As |lland_v2| does
not rely on Turc-Wendling, we can apply it on arbitrary simulation step sizes.

Integration tests
=================

.. how_to_understand_integration_tests::

The following integration tests are recalculations of the tests performed
for |lland_v1| (you can click on the captions to go to the related tests
of |lland_v1|). Their purpose is to show that both models function in an
equal manner.   Hence, most configurations are identical.  One exception
is that |lland_v2| requires no global radiation input (|Glob|).  Instead,
potential evaporation needs (|PET|) to be defined, which we take from the
integration test results of model |lland_v1| to achieve comparability.
Another exception is that |lland_v1| allows for smoothing the calculated
|ET0| values over time, which we discuss below.

The following general setup is identical with the one of |lland_v1|:

>>> from hydpy import pub
>>> pub.timegrids = '2000-01-01', '2000-01-05', '1h'
>>> from hydpy.models.lland_v2 import *
>>> parameterstep('1h')
>>> from hydpy import Node, Element
>>> outlet = Node('outlet')
>>> inlet = Node('inlet')
>>> land = Element('land', inlets=inlet, outlets=outlet)
>>> land.model = model
>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)
>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(land)

.. _lland_v2_acker_summer:

:ref:`acre (summer) <lland_v1_acker_summer>`
____________________________________________

The following parameter and initial condition values are identical with
the ones of |lland_v1|, except that the Turc-Wendling related parameter
|KF| is missing:

>>> lnk(ACKER)
>>> kg(1.2)
>>> kt(0.8)
>>> ke(0.4)
>>> wfet0(1.0)
>>> fln(0.5)
>>> hinz(0.2)
>>> lai(4.0)
>>> treft(0.0)
>>> trefn(0.0)
>>> tgr(1.0)
>>> tsp(2.0)
>>> gtf(0.5)
>>> pwmax(1.4)
>>> grasref_r(5.0)
>>> wmax(200.0)
>>> fk(relative=0.5)
>>> pwp(relative=0.05)
>>> kapgrenz(option='0_WMax/10')
>>> kapmax(0.08)
>>> beta(0.005)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(1.0)
>>> dmin(0.1)
>>> bsf(0.4)
>>> a1(1.0)
>>> a2(1.0)
>>> tind(1.0)
>>> eqb(100.0)
>>> eqi1(20.0)
>>> eqi2(10.0)
>>> eqd1(5.0)
>>> eqd2(2.0)
>>> test.inits = ((states.inzp, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.bowa, 20.0),
...               (states.qdgz1, 0.0),
...               (states.qdgz2, 0.0),
...               (states.qigz1, 0.0),
...               (states.qigz2, 0.0),
...               (states.qbgz, 0.0),
...               (states.qdga1, 0.0),
...               (states.qdga2, 0.0),
...               (states.qiga1, 0.0),
...               (states.qiga2, 0.0),
...               (states.qbga, 0.0),
...               (logs.wet0, 0.0))

The values of the input sequences of |Nied| (precipitation) and |TemL| (air
temperature) and from the inflow provided by node `inlet` also stem from the
input data of the example on |lland_v1|.  But we take the values of |PET|
(potential evaporation) from the the output data of |lland_v1| and divide the
original values by 0.4 to account for the selected value of the evaporation
adjustment factor |KE|:

>>> inputs.nied.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1,
...     10.4, 3.5, 3.4, 1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7,
...     37.9, 8.2, 3.6, 7.5, 18.5, 15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5,
...     2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6, 0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.3, 0.0, 0.0, 0.0, 0.7, 0.4,
...     0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0)
>>> inputs.teml.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2,
...     35.2, 37.1, 31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8,
...     19.6, 19.2, 19.2, 19.2, 18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8,
...     19.0, 19.2, 19.3, 19.0, 18.8, 18.7, 17.8, 17.4, 17.3, 16.8, 16.5,
...     16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3, 14.9, 15.7, 16.0,
...     16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3, 18.1,
...     16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1,
...     11.9, 12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8,
...     22.2, 20.1, 17.8, 15.2, 14.5, 12.4, 11.7, 11.9)
>>> inputs.pet.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908,
...     1.932875, 4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503,
...     7.397619, 2.39151, 1.829834, 1.136569, 0.750986, 0.223895, 0.099425,
...     0.098454, 0.098128, 0.097474, 0.097474, 0.097474, 0.096981, 0.096652,
...     0.096321, 0.09599, 0.187298, 1.264612, 3.045538, 1.930758, 2.461001,
...     6.215945, 3.374783, 8.821555, 4.046025, 2.110757, 2.239257, 2.877848,
...     1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683, 0.089858,
...     0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594,
...     4.851567, 5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771,
...     0.084317, 0.083215, 0.082289, 0.0845, 0.084864, 0.083584, 0.0834,
...     0.084864, 0.310229, 1.391958, 3.195876, 5.191651, 7.155036, 8.391432,
...     8.391286, 10.715238, 9.383394, 7.861915, 6.298329, 2.948416, 1.309232,
...     0.32955, 0.089508, 0.085771, 0.0845, 0.084864)
>>> inlet.sequences.sim.series = 0.0

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_acker_summer` for arable land (|ACKER|) with zero
inflow exactly:

.. integration-test::

    >>> test('lland_v2_acker_summer',
    ...      axis1=(inputs.nied, fluxes.qah),
    ...      axis2=states.bowa)
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000495 | 0.000138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.0 | 0.000138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 |  0.03912 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 0.001462 | 0.000406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.0 | 0.000406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 0.002394 | 0.000665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.0 | 0.000665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 0.003291 | 0.000914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.0 | 0.000914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 0.004156 | 0.001154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.0 | 0.001154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041104 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  0.00499 | 0.001386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.0 | 0.001386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 0.005793 | 0.001609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.0 | 0.001609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 0.006566 | 0.001824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.0 | 0.001824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 0.007306 |  0.00203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.0 |  0.00203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 0.008003 | 0.002223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.0 | 0.002223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 0.008643 | 0.002401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.0 | 0.002401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 0.009219 | 0.002561 |      0.0 |  0.0 |  0.0 |  17.837971 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.0 | 0.002561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 0.009727 | 0.002702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.0 | 0.002702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  0.01018 | 0.002828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.0 | 0.002828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 0.010586 | 0.002941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.0 | 0.002941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 0.010942 | 0.003039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.0 | 0.003039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731934 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 0.011263 | 0.003129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.0 | 0.003129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 0.011564 | 0.003212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.0 | 0.003212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  0.01363 | 0.003786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.0 | 0.003786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 0.034046 | 0.009457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.0 | 0.009457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 0.060472 | 0.016798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.0 | 0.016798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039382 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 0.088924 | 0.024701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.0 | 0.024701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  0.17003 | 0.047231 | 0.780374 |  0.0 |  0.0 |  44.225069 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.0 | 0.047231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 0.210341 | 0.058428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.0 | 0.058428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 0.183571 | 0.050992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.0 | 0.050992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 0.185428 | 0.051508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.0 | 0.051508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038792 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 0.198953 | 0.055265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.0 | 0.055265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 0.291718 | 0.081033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.0 | 0.081033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 0.384279 | 0.106744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.0 | 0.106744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 0.401551 | 0.111542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.0 | 0.111542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 0.391634 | 0.108787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.0 | 0.108787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  0.34564 | 0.096011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.0 | 0.096011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   0.2946 | 0.081833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.0 | 0.081833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 0.252616 | 0.070171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.0 | 0.070171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 |   0.9844 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 0.218815 | 0.060782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.0 | 0.060782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 0.191684 | 0.053246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.0 | 0.053246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 0.206673 | 0.057409 | 0.125043 |  0.0 |  0.0 |  68.927692 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.0 | 0.057409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 0.288705 | 0.080196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.0 | 0.080196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 0.323726 | 0.089924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.0 | 0.089924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 0.821088 |  0.22808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.0 |  0.22808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 2.868463 | 0.796795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.0 | 0.796795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.504942 | 3.791851 | 1.053292 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.600789 | 0.904153 | 0.064494 | 0.140463 | 0.594939 | 0.837653 | 2.869278 |  0.02348 | 0.006795 | 0.054646 |   0.0 | 1.053292 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.045681 |  2.93515 | 0.815319 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.043686 | 0.001996 | 0.067538 | 0.186956 | 0.625375 | 0.923811 | 1.903847 | 0.025555 | 0.021764 | 0.060173 |   0.0 | 0.815319 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.527094 | 2.481423 | 0.689284 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.604289 | 0.922805 | 0.068447 | 0.201685 | 0.634469 | 0.998043 | 1.351716 | 0.027625 | 0.038197 | 0.065843 |   0.0 | 0.689284 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.227705 | 3.418899 | 0.949694 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.861643 | 5.366061 | 0.071072 | 0.246232 |  0.66072 | 1.132039 | 2.129639 |  0.02968 |  0.05591 | 0.071631 |   0.0 | 0.949694 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  6.97553 | 4.804338 | 1.334538 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.856642 | 5.118889 |  0.07804 | 0.377966 | 0.730399 | 1.263826 | 3.350409 |  0.03187 | 0.080394 |  0.07784 |   0.0 | 1.334538 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 |  3.09949 |  4.84592 | 1.346089 |  0.78175 |  0.0 |  0.0 | 169.469597 | 1.677366 | 1.422124 |  0.08319 | 0.486735 | 0.781897 | 1.354495 | 3.258614 | 0.034249 | 0.113973 |  0.08459 |   0.0 | 1.346089 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 0.948023 | 3.857116 | 1.071421 | 0.782063 |  0.0 |  0.0 | 169.380125 | 0.948023 |      0.0 | 0.084735 | 0.521116 | 0.797348 | 1.344716 | 2.233012 | 0.036674 | 0.151109 | 0.091605 |   0.0 | 1.071421 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 2.488688 | 3.201733 |  0.88937 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.598182 | 0.890506 |  0.08469 |  0.52011 | 0.796901 | 1.333698 | 1.544122 | 0.039016 | 0.186271 | 0.098625 |   0.0 |  0.88937 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 2.838425 | 0.788451 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.351893 | 1.120521 | 0.041268 |  0.21912 | 0.105622 |   0.0 | 0.788451 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 0.249877 | 2.347956 |  0.65221 | 0.782239 |  0.0 |  0.0 | 170.639599 | 0.249877 |      0.0 | 0.085875 | 0.546974 | 0.808745 | 1.242434 | 0.699367 | 0.043439 | 0.250108 | 0.112609 |   0.0 |  0.65221 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 |  1.91484 |   0.5319 | 0.782169 |  0.0 |  0.0 | 169.345118 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.047903 | 0.424187 | 0.045495 | 0.277747 | 0.119508 |   0.0 |   0.5319 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 0.242047 | 1.621283 | 0.450356 | 0.781959 |  0.0 |  0.0 | 168.284125 | 0.242047 |      0.0 | 0.084673 | 0.519716 | 0.796726 | 0.888843 | 0.257283 | 0.047421 | 0.301457 | 0.126278 |   0.0 | 0.450356 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 1.523064 | 0.423073 | 0.781681 |  0.0 |  0.0 | 168.578821 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 0.857129 | 0.162134 | 0.049225 | 0.321653 | 0.132923 |   0.0 | 0.423073 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061944 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.189947 | 1.454205 | 0.403946 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.189947 |      0.0 | 0.084289 | 0.511125 | 0.792894 | 0.820771 | 0.103491 | 0.050931 | 0.339529 | 0.139482 |   0.0 | 0.403946 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 1.313091 | 0.364748 | 0.705926 |  0.0 |  0.0 | 166.216859 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.696563 | 0.062771 | 0.052545 | 0.355257 | 0.145956 |   0.0 | 0.364748 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 |      0.0 | 1.190388 | 0.330663 | 0.471181 |  0.0 |  0.0 | 164.867718 |      0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 | 0.577717 | 0.038072 |  0.05405 | 0.368242 | 0.152307 |   0.0 | 0.330663 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 1.088702 | 0.302417 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.472994 | 0.023092 | 0.055451 | 0.378635 | 0.158529 |   0.0 | 0.302417 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 |      0.0 | 1.009088 | 0.280302 |      0.0 |  0.0 |  0.0 | 161.377452 |      0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 | 0.387255 | 0.014006 | 0.056746 | 0.386466 | 0.164615 |   0.0 | 0.280302 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 1.028002 | 0.285556 |      0.0 |  0.0 |  0.0 | 161.294055 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.399248 | 0.008495 | 0.057936 |  0.39177 | 0.170553 |   0.0 | 0.285556 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.014983 | 1.041382 | 0.289273 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.014983 |      0.0 | 0.080647 | 0.431887 |  0.75647 | 0.405172 | 0.005153 | 0.059044 | 0.395629 | 0.176385 |   0.0 | 0.289273 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 0.976673 | 0.271298 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.333039 | 0.003125 | 0.060079 | 0.398311 | 0.182119 |   0.0 | 0.271298 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 |      0.0 | 0.922391 |  0.25622 |      0.0 |  0.0 |  0.0 | 155.593339 |      0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 | 0.272669 | 0.001896 | 0.061023 |  0.39909 | 0.187714 |   0.0 |  0.25622 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 0.877158 | 0.243655 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.223243 |  0.00115 |  0.06187 | 0.397747 | 0.193148 |   0.0 | 0.243655 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 |      0.0 | 0.838885 | 0.233024 |      0.0 |  0.0 |  0.0 | 151.294476 |      0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 | 0.182776 | 0.000697 | 0.062621 | 0.394374 | 0.198417 |   0.0 | 0.233024 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 0.806179 | 0.223939 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.149644 | 0.000423 | 0.063282 | 0.389303 | 0.203526 |   0.0 | 0.223939 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 |      0.0 | 0.778035 | 0.216121 |      0.0 |  0.0 |  0.0 | 148.167947 |      0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 | 0.122518 | 0.000257 | 0.063864 | 0.382909 | 0.208487 |   0.0 | 0.216121 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 0.753875 |  0.20941 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.100309 | 0.000156 | 0.064379 |  0.37571 | 0.213321 |   0.0 |  0.20941 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 |      0.0 | 0.733188 | 0.203663 |      0.0 |  0.0 |  0.0 | 145.983055 |      0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 | 0.082126 | 0.000094 | 0.064839 | 0.368084 | 0.218044 |   0.0 | 0.203663 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 0.715434 | 0.198732 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.067239 | 0.000057 | 0.065249 | 0.360222 | 0.222667 |   0.0 | 0.198732 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 |      0.0 | 0.700075 | 0.194465 |      0.0 |  0.0 |  0.0 | 143.898607 |      0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 | 0.055051 | 0.000035 | 0.065614 | 0.352185 |  0.22719 |   0.0 | 0.194465 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 | 0.686662 | 0.190739 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 | 0.045072 | 0.000021 | 0.065935 | 0.344017 | 0.231617 |   0.0 | 0.190739 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |      0.0 |  0.67483 | 0.187453 |      0.0 |  0.0 |  0.0 | 141.875122 |      0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 | 0.036902 | 0.000013 | 0.066216 | 0.335751 | 0.235948 |   0.0 | 0.187453 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 0.664287 | 0.184524 |      0.0 |  0.0 |  0.0 | 140.884764 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.030213 | 0.000008 | 0.066458 | 0.327422 | 0.240186 |   0.0 | 0.184524 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 |      0.0 | 0.654794 | 0.181887 |      0.0 |  0.0 |  0.0 | 139.908407 |      0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 | 0.024736 | 0.000005 | 0.066665 | 0.319056 | 0.244333 |   0.0 | 0.181887 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033434 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 0.666919 | 0.185255 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.041012 | 0.000003 | 0.066837 | 0.310678 | 0.248389 |   0.0 | 0.185255 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 |      0.0 | 0.674893 |  0.18747 | 0.766603 |  0.0 |  0.0 | 138.559658 |      0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 | 0.052999 | 0.000002 | 0.066984 | 0.302538 |  0.25237 |   0.0 |  0.18747 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 |      0.0 | 0.661405 | 0.183724 |  0.74963 |  0.0 |  0.0 | 137.632083 |      0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 | 0.043392 | 0.000001 | 0.067107 | 0.294626 | 0.256278 |   0.0 | 0.183724 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 |      0.0 | 0.649552 | 0.180431 | 0.687585 |  0.0 |  0.0 | 136.717338 |      0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 | 0.035526 | 0.000001 | 0.067202 | 0.286722 | 0.260101 |   0.0 | 0.180431 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 | 0.204485 | 0.658192 | 0.182831 | 0.521608 |  0.0 |  0.0 | 136.338253 | 0.204485 |      0.0 | 0.068359 | 0.200239 | 0.633587 | 0.048237 |      0.0 | 0.067269 | 0.278844 |  0.26384 |   0.0 | 0.182831 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 |  1.27835 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.056289 | 0.668723 | 0.185757 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.056289 |      0.0 | 0.068169 | 0.197146 | 0.631691 | 0.062681 |      0.0 | 0.067318 | 0.271214 |  0.26751 |   0.0 | 0.185757 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 |      0.0 | 0.658587 | 0.182941 |      0.0 |  0.0 |  0.0 |  133.98234 |      0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 | 0.056251 |      0.0 |  0.06735 | 0.263872 | 0.271115 |   0.0 | 0.182941 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 |      0.0 | 0.644356 | 0.178988 |      0.0 |  0.0 |  0.0 | 132.218945 |      0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 | 0.046054 |      0.0 | 0.067352 | 0.256325 | 0.274625 |   0.0 | 0.178988 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 |      0.0 | 0.631272 | 0.175353 |      0.0 |  0.0 |  0.0 | 129.797092 |      0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 | 0.037706 |      0.0 | 0.067313 | 0.248236 | 0.278017 |   0.0 | 0.175353 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 |      0.0 | 0.618762 | 0.171878 |      0.0 |  0.0 |  0.0 | 127.414065 |      0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 | 0.030871 |      0.0 | 0.067224 | 0.239396 | 0.281271 |   0.0 | 0.171878 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143048 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |      0.0 | 0.606442 | 0.168456 |      0.0 |  0.0 |  0.0 | 124.635441 |      0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 | 0.025275 |      0.0 | 0.067082 | 0.229713 | 0.284373 |   0.0 | 0.168456 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 |      0.0 | 0.594112 | 0.165031 |      0.0 |  0.0 |  0.0 | 122.150125 |      0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 | 0.020693 |      0.0 | 0.066883 |  0.21922 | 0.287315 |   0.0 | 0.165031 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 |      0.0 | 0.581716 | 0.161588 |      0.0 |  0.0 |  0.0 | 119.985132 |      0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 | 0.016942 |      0.0 |  0.06663 | 0.208046 | 0.290098 |   0.0 | 0.161588 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 |      0.0 | 0.569468 | 0.158185 |      0.0 |  0.0 |  0.0 | 118.139837 |      0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 | 0.013871 |      0.0 | 0.066332 | 0.196528 | 0.292737 |   0.0 | 0.158185 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179366 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 |      0.0 | 0.557558 | 0.154877 |      0.0 |  0.0 |  0.0 | 116.932423 |      0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 | 0.011357 |      0.0 |    0.066 | 0.184951 |  0.29525 |   0.0 | 0.154877 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 |      0.0 | 0.546245 | 0.151735 |      0.0 |  0.0 |  0.0 | 116.038409 |      0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 | 0.009298 |      0.0 | 0.065647 | 0.173638 | 0.297662 |   0.0 | 0.151735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 |      0.0 | 0.535741 | 0.148817 |      0.0 |  0.0 |  0.0 | 115.332585 |      0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 | 0.007613 |      0.0 | 0.065286 | 0.162844 | 0.299998 |   0.0 | 0.148817 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016211 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 |      0.0 | 0.526093 | 0.146137 |      0.0 |  0.0 |  0.0 | 114.678011 |      0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 | 0.006233 |      0.0 | 0.064923 | 0.152666 | 0.302271 |   0.0 | 0.146137 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 |      0.0 | 0.517266 | 0.143685 |      0.0 |  0.0 |  0.0 | 114.031165 |      0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 | 0.005103 |      0.0 | 0.064561 | 0.143114 | 0.304487 |   0.0 | 0.143685 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 |      0.0 |  0.50918 | 0.141439 |      0.0 |  0.0 |  0.0 | 113.391444 |      0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 | 0.004178 |      0.0 | 0.064201 | 0.134151 | 0.306649 |   0.0 | 0.141439 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 |      0.0 | 0.501753 | 0.139376 |      0.0 |  0.0 |  0.0 | 112.758402 |      0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 | 0.003421 |      0.0 | 0.063843 | 0.125732 | 0.308758 |   0.0 | 0.139376 |

.. _lland_v2_acker_routing:

:ref:`acre (routing) <lland_v1_acker_routing>`
______________________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_acker_routing` for arable land (|ACKER|) with relevant
inflow exactly:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.3
    >>> test('lland_v2_acker_routing',
    ...      axis1=(inputs.nied, fluxes.qah),
    ...      axis2=states.bowa)
    |                date | nied | teml |       pet |  qz |  qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.3 | 1.08 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 1.080495 | 0.300138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.3 | 0.300138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.3 | 1.08 |   0.0 | 20.2 |       0.0 |  0.0 |  0.03912 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 1.081462 | 0.300406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.3 | 0.300406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.3 | 1.08 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 1.082394 | 0.300665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.3 | 0.300665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.3 | 1.08 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 1.083291 | 0.300914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.3 | 0.300914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.3 | 1.08 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 1.084156 | 0.301154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.3 | 0.301154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.3 | 1.08 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041104 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  1.08499 | 0.301386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.3 | 0.301386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.3 | 1.08 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 1.085793 | 0.301609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.3 | 0.301609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.3 | 1.08 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 1.086566 | 0.301824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.3 | 0.301824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.3 | 1.08 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 1.087306 |  0.30203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.3 |  0.30203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.3 | 1.08 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 1.088003 | 0.302223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.3 | 0.302223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.3 | 1.08 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 1.088643 | 0.302401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.3 | 0.302401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.3 | 1.08 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 1.089219 | 0.302561 |      0.0 |  0.0 |  0.0 |  17.837971 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.3 | 0.302561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.3 | 1.08 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 1.089727 | 0.302702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.3 | 0.302702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.3 | 1.08 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  1.09018 | 0.302828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.3 | 0.302828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.3 | 1.08 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 1.090586 | 0.302941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.3 | 0.302941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.3 | 1.08 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 1.090942 | 0.303039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.3 | 0.303039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.3 | 1.08 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731934 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 1.091263 | 0.303129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.3 | 0.303129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.3 | 1.08 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 1.091564 | 0.303212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.3 | 0.303212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.3 | 1.08 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  1.09363 | 0.303786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.3 | 0.303786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.3 | 1.08 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 1.114046 | 0.309457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.3 | 0.309457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.3 | 1.08 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 1.140472 | 0.316798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.3 | 0.316798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.3 | 1.08 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039382 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 1.168924 | 0.324701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.3 | 0.324701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.3 | 1.08 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  1.25003 | 0.347231 | 0.780374 |  0.0 |  0.0 |  44.225069 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.3 | 0.347231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.3 | 1.08 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 1.290341 | 0.358428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.3 | 0.358428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.3 | 1.08 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 1.263571 | 0.350992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.3 | 0.350992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.3 | 1.08 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 1.265428 | 0.351508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.3 | 0.351508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.3 | 1.08 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038792 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 1.278953 | 0.355265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.3 | 0.355265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.3 | 1.08 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 1.371718 | 0.381033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.3 | 0.381033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.3 | 1.08 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 1.464279 | 0.406744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.3 | 0.406744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.3 | 1.08 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 1.481551 | 0.411542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.3 | 0.411542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.3 | 1.08 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 1.471634 | 0.408787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.3 | 0.408787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.3 | 1.08 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  1.42564 | 0.396011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.3 | 0.396011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.3 | 1.08 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   1.3746 | 0.381833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.3 | 0.381833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.3 | 1.08 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 1.332616 | 0.370171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.3 | 0.370171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.3 | 1.08 |  0.48 | 20.0 |       0.0 |  0.0 |   0.9844 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 1.298815 | 0.360782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.3 | 0.360782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.3 | 1.08 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 1.271684 | 0.353246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.3 | 0.353246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.3 | 1.08 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 1.286673 | 0.357409 | 0.125043 |  0.0 |  0.0 |  68.927692 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.3 | 0.357409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.3 | 1.08 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 1.368705 | 0.380196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.3 | 0.380196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.3 | 1.08 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 1.403726 | 0.389924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.3 | 0.389924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.3 | 1.08 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 1.901088 |  0.52808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.3 |  0.52808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.3 | 1.08 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 3.948463 | 1.096795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.3 | 1.096795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.3 | 1.08 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.504942 | 4.871851 | 1.353292 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.600789 | 0.904153 | 0.064494 | 0.140463 | 0.594939 | 0.837653 | 2.869278 |  0.02348 | 0.006795 | 0.054646 |   0.3 | 1.353292 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.3 | 1.08 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.045681 |  4.01515 | 1.115319 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.043686 | 0.001996 | 0.067538 | 0.186956 | 0.625375 | 0.923811 | 1.903847 | 0.025555 | 0.021764 | 0.060173 |   0.3 | 1.115319 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.3 | 1.08 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.527094 | 3.561423 | 0.989284 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.604289 | 0.922805 | 0.068447 | 0.201685 | 0.634469 | 0.998043 | 1.351716 | 0.027625 | 0.038197 | 0.065843 |   0.3 | 0.989284 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.3 | 1.08 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.227705 | 4.498899 | 1.249694 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.861643 | 5.366061 | 0.071072 | 0.246232 |  0.66072 | 1.132039 | 2.129639 |  0.02968 |  0.05591 | 0.071631 |   0.3 | 1.249694 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.3 | 1.08 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  6.97553 | 5.884338 | 1.634538 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.856642 | 5.118889 |  0.07804 | 0.377966 | 0.730399 | 1.263826 | 3.350409 |  0.03187 | 0.080394 |  0.07784 |   0.3 | 1.634538 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.3 | 1.08 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 |  3.09949 |  5.92592 | 1.646089 |  0.78175 |  0.0 |  0.0 | 169.469597 | 1.677366 | 1.422124 |  0.08319 | 0.486735 | 0.781897 | 1.354495 | 3.258614 | 0.034249 | 0.113973 |  0.08459 |   0.3 | 1.646089 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.3 | 1.08 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 0.948023 | 4.937116 | 1.371421 | 0.782063 |  0.0 |  0.0 | 169.380125 | 0.948023 |      0.0 | 0.084735 | 0.521116 | 0.797348 | 1.344716 | 2.233012 | 0.036674 | 0.151109 | 0.091605 |   0.3 | 1.371421 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.3 | 1.08 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 2.488688 | 4.281733 |  1.18937 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.598182 | 0.890506 |  0.08469 |  0.52011 | 0.796901 | 1.333698 | 1.544122 | 0.039016 | 0.186271 | 0.098625 |   0.3 |  1.18937 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.3 | 1.08 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 3.918425 | 1.088451 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.351893 | 1.120521 | 0.041268 |  0.21912 | 0.105622 |   0.3 | 1.088451 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.3 | 1.08 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 0.249877 | 3.427956 |  0.95221 | 0.782239 |  0.0 |  0.0 | 170.639599 | 0.249877 |      0.0 | 0.085875 | 0.546974 | 0.808745 | 1.242434 | 0.699367 | 0.043439 | 0.250108 | 0.112609 |   0.3 |  0.95221 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.3 | 1.08 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 |  2.99484 |   0.8319 | 0.782169 |  0.0 |  0.0 | 169.345118 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.047903 | 0.424187 | 0.045495 | 0.277747 | 0.119508 |   0.3 |   0.8319 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.3 | 1.08 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 0.242047 | 2.701283 | 0.750356 | 0.781959 |  0.0 |  0.0 | 168.284125 | 0.242047 |      0.0 | 0.084673 | 0.519716 | 0.796726 | 0.888843 | 0.257283 | 0.047421 | 0.301457 | 0.126278 |   0.3 | 0.750356 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.3 | 1.08 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 2.603064 | 0.723073 | 0.781681 |  0.0 |  0.0 | 168.578821 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 0.857129 | 0.162134 | 0.049225 | 0.321653 | 0.132923 |   0.3 | 0.723073 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.3 | 1.08 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061944 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.189947 | 2.534205 | 0.703946 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.189947 |      0.0 | 0.084289 | 0.511125 | 0.792894 | 0.820771 | 0.103491 | 0.050931 | 0.339529 | 0.139482 |   0.3 | 0.703946 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.3 | 1.08 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 2.393091 | 0.664748 | 0.705926 |  0.0 |  0.0 | 166.216859 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.696563 | 0.062771 | 0.052545 | 0.355257 | 0.145956 |   0.3 | 0.664748 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.3 | 1.08 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 |      0.0 | 2.270388 | 0.630663 | 0.471181 |  0.0 |  0.0 | 164.867718 |      0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 | 0.577717 | 0.038072 |  0.05405 | 0.368242 | 0.152307 |   0.3 | 0.630663 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.3 | 1.08 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 2.168702 | 0.602417 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.472994 | 0.023092 | 0.055451 | 0.378635 | 0.158529 |   0.3 | 0.602417 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.3 | 1.08 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 |      0.0 | 2.089088 | 0.580302 |      0.0 |  0.0 |  0.0 | 161.377452 |      0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 | 0.387255 | 0.014006 | 0.056746 | 0.386466 | 0.164615 |   0.3 | 0.580302 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.3 | 1.08 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 2.108002 | 0.585556 |      0.0 |  0.0 |  0.0 | 161.294055 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.399248 | 0.008495 | 0.057936 |  0.39177 | 0.170553 |   0.3 | 0.585556 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.3 | 1.08 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.014983 | 2.121382 | 0.589273 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.014983 |      0.0 | 0.080647 | 0.431887 |  0.75647 | 0.405172 | 0.005153 | 0.059044 | 0.395629 | 0.176385 |   0.3 | 0.589273 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.3 | 1.08 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 2.056673 | 0.571298 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.333039 | 0.003125 | 0.060079 | 0.398311 | 0.182119 |   0.3 | 0.571298 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.3 | 1.08 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 |      0.0 | 2.002391 |  0.55622 |      0.0 |  0.0 |  0.0 | 155.593339 |      0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 | 0.272669 | 0.001896 | 0.061023 |  0.39909 | 0.187714 |   0.3 |  0.55622 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.3 | 1.08 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 1.957158 | 0.543655 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.223243 |  0.00115 |  0.06187 | 0.397747 | 0.193148 |   0.3 | 0.543655 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.3 | 1.08 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 |      0.0 | 1.918885 | 0.533024 |      0.0 |  0.0 |  0.0 | 151.294476 |      0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 | 0.182776 | 0.000697 | 0.062621 | 0.394374 | 0.198417 |   0.3 | 0.533024 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.3 | 1.08 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 1.886179 | 0.523939 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.149644 | 0.000423 | 0.063282 | 0.389303 | 0.203526 |   0.3 | 0.523939 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.3 | 1.08 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 |      0.0 | 1.858035 | 0.516121 |      0.0 |  0.0 |  0.0 | 148.167947 |      0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 | 0.122518 | 0.000257 | 0.063864 | 0.382909 | 0.208487 |   0.3 | 0.516121 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.3 | 1.08 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 1.833875 |  0.50941 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.100309 | 0.000156 | 0.064379 |  0.37571 | 0.213321 |   0.3 |  0.50941 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.3 | 1.08 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 |      0.0 | 1.813188 | 0.503663 |      0.0 |  0.0 |  0.0 | 145.983055 |      0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 | 0.082126 | 0.000094 | 0.064839 | 0.368084 | 0.218044 |   0.3 | 0.503663 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.3 | 1.08 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 1.795434 | 0.498732 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.067239 | 0.000057 | 0.065249 | 0.360222 | 0.222667 |   0.3 | 0.498732 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.3 | 1.08 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 |      0.0 | 1.780075 | 0.494465 |      0.0 |  0.0 |  0.0 | 143.898607 |      0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 | 0.055051 | 0.000035 | 0.065614 | 0.352185 |  0.22719 |   0.3 | 0.494465 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.3 | 1.08 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 | 1.766662 | 0.490739 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 | 0.045072 | 0.000021 | 0.065935 | 0.344017 | 0.231617 |   0.3 | 0.490739 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.3 | 1.08 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |      0.0 |  1.75483 | 0.487453 |      0.0 |  0.0 |  0.0 | 141.875122 |      0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 | 0.036902 | 0.000013 | 0.066216 | 0.335751 | 0.235948 |   0.3 | 0.487453 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.3 | 1.08 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 1.744287 | 0.484524 |      0.0 |  0.0 |  0.0 | 140.884764 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.030213 | 0.000008 | 0.066458 | 0.327422 | 0.240186 |   0.3 | 0.484524 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 |      0.0 | 1.734794 | 0.481887 |      0.0 |  0.0 |  0.0 | 139.908407 |      0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 | 0.024736 | 0.000005 | 0.066665 | 0.319056 | 0.244333 |   0.3 | 0.481887 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.3 | 1.08 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033434 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 1.746919 | 0.485255 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.041012 | 0.000003 | 0.066837 | 0.310678 | 0.248389 |   0.3 | 0.485255 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.3 | 1.08 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 |      0.0 | 1.754893 |  0.48747 | 0.766603 |  0.0 |  0.0 | 138.559658 |      0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 | 0.052999 | 0.000002 | 0.066984 | 0.302538 |  0.25237 |   0.3 |  0.48747 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 |      0.0 | 1.741405 | 0.483724 |  0.74963 |  0.0 |  0.0 | 137.632083 |      0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 | 0.043392 | 0.000001 | 0.067107 | 0.294626 | 0.256278 |   0.3 | 0.483724 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.3 | 1.08 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 |      0.0 | 1.729552 | 0.480431 | 0.687585 |  0.0 |  0.0 | 136.717338 |      0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 | 0.035526 | 0.000001 | 0.067202 | 0.286722 | 0.260101 |   0.3 | 0.480431 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.3 | 1.08 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 | 0.204485 | 1.738192 | 0.482831 | 0.521608 |  0.0 |  0.0 | 136.338253 | 0.204485 |      0.0 | 0.068359 | 0.200239 | 0.633587 | 0.048237 |      0.0 | 0.067269 | 0.278844 |  0.26384 |   0.3 | 0.482831 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.3 | 1.08 |  0.48 | 12.2 |  0.201608 |  0.0 |  1.27835 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.056289 | 1.748723 | 0.485757 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.056289 |      0.0 | 0.068169 | 0.197146 | 0.631691 | 0.062681 |      0.0 | 0.067318 | 0.271214 |  0.26751 |   0.3 | 0.485757 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.3 | 1.08 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 |      0.0 | 1.738587 | 0.482941 |      0.0 |  0.0 |  0.0 |  133.98234 |      0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 | 0.056251 |      0.0 |  0.06735 | 0.263872 | 0.271115 |   0.3 | 0.482941 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.3 | 1.08 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 |      0.0 | 1.724356 | 0.478988 |      0.0 |  0.0 |  0.0 | 132.218945 |      0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 | 0.046054 |      0.0 | 0.067352 | 0.256325 | 0.274625 |   0.3 | 0.478988 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.3 | 1.08 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 |      0.0 | 1.711272 | 0.475353 |      0.0 |  0.0 |  0.0 | 129.797092 |      0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 | 0.037706 |      0.0 | 0.067313 | 0.248236 | 0.278017 |   0.3 | 0.475353 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.3 | 1.08 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 |      0.0 | 1.698762 | 0.471878 |      0.0 |  0.0 |  0.0 | 127.414065 |      0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 | 0.030871 |      0.0 | 0.067224 | 0.239396 | 0.281271 |   0.3 | 0.471878 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.3 | 1.08 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143048 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |      0.0 | 1.686442 | 0.468456 |      0.0 |  0.0 |  0.0 | 124.635441 |      0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 | 0.025275 |      0.0 | 0.067082 | 0.229713 | 0.284373 |   0.3 | 0.468456 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.3 | 1.08 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 |      0.0 | 1.674112 | 0.465031 |      0.0 |  0.0 |  0.0 | 122.150125 |      0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 | 0.020693 |      0.0 | 0.066883 |  0.21922 | 0.287315 |   0.3 | 0.465031 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.3 | 1.08 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 |      0.0 | 1.661716 | 0.461588 |      0.0 |  0.0 |  0.0 | 119.985132 |      0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 | 0.016942 |      0.0 |  0.06663 | 0.208046 | 0.290098 |   0.3 | 0.461588 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.3 | 1.08 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 |      0.0 | 1.649468 | 0.458185 |      0.0 |  0.0 |  0.0 | 118.139837 |      0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 | 0.013871 |      0.0 | 0.066332 | 0.196528 | 0.292737 |   0.3 | 0.458185 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.3 | 1.08 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179366 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 |      0.0 | 1.637558 | 0.454877 |      0.0 |  0.0 |  0.0 | 116.932423 |      0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 | 0.011357 |      0.0 |    0.066 | 0.184951 |  0.29525 |   0.3 | 0.454877 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.3 | 1.08 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 |      0.0 | 1.626245 | 0.451735 |      0.0 |  0.0 |  0.0 | 116.038409 |      0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 | 0.009298 |      0.0 | 0.065647 | 0.173638 | 0.297662 |   0.3 | 0.451735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.3 | 1.08 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 |      0.0 | 1.615741 | 0.448817 |      0.0 |  0.0 |  0.0 | 115.332585 |      0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 | 0.007613 |      0.0 | 0.065286 | 0.162844 | 0.299998 |   0.3 | 0.448817 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.3 | 1.08 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016211 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 |      0.0 | 1.606093 | 0.446137 |      0.0 |  0.0 |  0.0 | 114.678011 |      0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 | 0.006233 |      0.0 | 0.064923 | 0.152666 | 0.302271 |   0.3 | 0.446137 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.3 | 1.08 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 |      0.0 | 1.597266 | 0.443685 |      0.0 |  0.0 |  0.0 | 114.031165 |      0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 | 0.005103 |      0.0 | 0.064561 | 0.143114 | 0.304487 |   0.3 | 0.443685 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.3 | 1.08 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 |      0.0 |  1.58918 | 0.441439 |      0.0 |  0.0 |  0.0 | 113.391444 |      0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 | 0.004178 |      0.0 | 0.064201 | 0.134151 | 0.306649 |   0.3 | 0.441439 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 |      0.0 | 1.581753 | 0.439376 |      0.0 |  0.0 |  0.0 | 112.758402 |      0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 | 0.003421 |      0.0 | 0.063843 | 0.125732 | 0.308758 |   0.3 | 0.439376 |

:ref:`water <lland_v1_wasser>`
______________________________

As discussed in the documentation of |lland_v1|, the handling of
evaporation from water surfaces can be problematic.  |lland_v2| offers
a smoothing option for the calculation of |ET0| (see method
|Calc_ET0_WET0_V1|) that is freely configurable via the "delay weighting
factor" |WfET0|. In principle, you could apply this mechanism to all
land-use classes.  However, its original intention is to take the temporal
persistence of (large) water bodies into account.  We demonstrate this
functionality is by setting the weighting parameter |WfET0| to a value
smaller than one and defining a suitable "old" evaporation value (|WET0|):

.. integration-test::

    >>> lnk(WASSER)
    >>> wfet0(0.01)
    >>> test.inits.wet0 = 1.0
    >>> inlet.sequences.sim.series = 0.0
    >>> test('lland_v2_wasser',
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |        qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.990403 | 0.495201 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 |  0.98089 | 0.490445 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.971469 | 0.485735 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.962138 | 0.481069 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.952905 | 0.476452 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.943787 | 0.471893 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.935517 | 0.467758 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 | 0.933893 | 0.466946 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 0.942032 | 0.471016 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 0.961882 | 0.480941 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 0.985321 |  0.49266 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 1.012947 | 0.506473 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 1.023322 | 0.511661 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 1.039589 | 0.519795 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 1.058784 | 0.529392 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 1.057762 | 0.528881 |     0.24 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 1.054504 | 0.527252 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 1.048505 | 0.524252 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 1.041024 | 0.520512 | 0.520512 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.039488 |  0.288747 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.288747 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 1.031509 | 0.515755 | 0.515755 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.204245 |  1.723402 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.723402 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 | 1.021592 | 0.510796 | 0.510796 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.969204 |  0.824779 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.824779 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 |  1.01177 | 0.505885 | 0.505885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.374115 |   1.49281 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   1.49281 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 1.002044 | 0.501022 | 0.501022 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 12.218978 |  3.394161 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  3.394161 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 | 0.992414 | 0.496207 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.98288 |  0.49144 |  0.49144 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.34856 |  0.096822 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.096822 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 | 0.973441 |  0.48672 |  0.48672 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   3.11328 |    0.8648 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |    0.8648 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.964094 | 0.482047 | 0.482047 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.037953 |  0.566098 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.566098 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 |  0.95484 |  0.47742 |  0.47742 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  12.00258 |   3.33405 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   3.33405 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.945677 | 0.472838 | 0.472838 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.727162 |  1.035323 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.035323 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.936604 | 0.468302 | 0.468302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.611698 |  1.003249 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.003249 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.927987 | 0.463994 | 0.463994 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.976006 |  0.271113 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.271113 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.923766 | 0.461883 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 |  0.92671 | 0.463355 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.925166 | 0.462583 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 | 0.925758 | 0.462879 | 0.462879 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.017121 |  0.004756 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.004756 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 0.941365 | 0.470682 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 |  0.94545 | 0.472725 | 0.472725 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.847275 |  1.068687 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.068687 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 0.971282 | 0.485641 | 0.485641 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.594359 |  1.831766 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.831766 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 | 0.977753 | 0.488877 | 0.488877 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.831123 |  0.230868 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.230868 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.976419 | 0.488209 | 0.488209 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 24.351791 |  6.764386 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.764386 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.975611 | 0.487806 | 0.487806 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 44.992194 | 12.497832 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 12.497832 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 0.977367 | 0.488683 | 0.488683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  9.351317 |  2.597588 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.597588 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.973959 | 0.486979 | 0.486979 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.833021 |  1.064728 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.064728 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.965386 | 0.482693 | 0.482693 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.517307 |  2.365919 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.365919 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.956102 | 0.478051 | 0.478051 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 21.721949 |  6.033875 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.033875 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 | 0.946911 | 0.473456 | 0.473456 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 18.006544 |  5.001818 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  5.001818 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.937807 | 0.468904 | 0.468904 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  7.091096 |  1.969749 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.969749 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.928788 | 0.464394 | 0.464394 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.815606 |  0.504335 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.504335 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.919859 |  0.45993 |  0.45993 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.42007 |  1.505575 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.505575 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.911019 |  0.45551 |  0.45551 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   2.78449 |   0.77347 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.77347 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.902264 | 0.451132 | 0.451132 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.148868 |  0.041352 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.041352 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.893598 | 0.446799 |     0.24 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.885023 | 0.442512 | 0.442512 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.157488 |  0.043747 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.043747 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.876539 |  0.43827 |  0.43827 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   2.44173 |  0.678258 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.678258 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.868393 | 0.434197 | 0.434197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.045803 |  0.012723 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.012723 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.861591 | 0.430795 |     0.24 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.85767 | 0.428835 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 0.865902 | 0.432951 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 0.874682 | 0.437341 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 0.887158 | 0.443579 | 0.443579 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.676421 |   0.74345 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.74345 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 0.899791 | 0.449895 | 0.449895 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.390105 |  0.108362 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.108362 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 0.909429 | 0.454714 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 0.931493 | 0.465746 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 0.941584 | 0.470792 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.953396 | 0.476698 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 0.957006 | 0.478503 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.953461 |  0.47673 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.945025 | 0.472513 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.935925 | 0.467963 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.926909 | 0.463455 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.917977 | 0.458989 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.909131 | 0.454565 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.900368 | 0.450184 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 | 0.891703 | 0.445851 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.883125 | 0.441563 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.874628 | 0.437314 | 0.437314 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.122686 |  0.311857 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.311857 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 | 0.866216 | 0.433108 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.857893 | 0.428946 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.850555 | 0.425277 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.847617 | 0.423809 | 0.423809 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.416191 |  0.115609 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.115609 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 | 0.851924 | 0.425962 | 0.425962 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.054038 |   0.01501 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.01501 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 | 0.864172 | 0.432086 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 |  0.88415 | 0.442075 | 0.442075 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.037925 |  0.010535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.010535 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 0.908874 | 0.454437 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 0.933351 | 0.466675 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 0.966878 | 0.483439 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 0.994743 | 0.497372 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 1.016243 | 0.508122 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 1.031274 | 0.515637 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.032755 | 0.516378 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 1.027664 | 0.513832 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 1.018706 | 0.509353 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 1.008877 | 0.504439 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.999131 | 0.499566 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 | 0.989478 | 0.494739 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.979923 | 0.489961 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |       0.0 |

:ref:`water (routing) <lland_v1_wasser_routing>`
________________________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_wasser_routing` for water areas of type |WASSER|
with relevant inflow exactly:

.. integration-test::

    >>> lnk(WASSER)
    >>> inlet.sequences.sim.series = 0.3
    >>> wfet0(1.0)
    >>> test.inits.wet0 = 0.0
    >>> test('lland_v2_wasser_routing',
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |       pet |  qz |  qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.3 | 1.08 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020141 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.059859 | 0.294405 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294405 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.3 | 1.08 |   0.0 | 20.2 |  0.0 |  0.0 |  0.03912 |  0.01956 |  0.01956 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.06044 | 0.294567 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294567 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.3 | 1.08 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060604 | 0.294612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294612 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.3 | 1.08 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060802 | 0.294667 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294667 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.3 | 1.08 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060604 | 0.294612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294612 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.3 | 1.08 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041104 | 0.020552 | 0.020552 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.059448 | 0.294291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294291 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.3 | 1.08 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.058382 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.021618 | 0.283783 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.283783 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.3 | 1.08 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 | 0.386575 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.693425 | 0.192618 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.192618 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.3 | 1.08 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.873907 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.206093 | 0.057248 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.057248 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.3 | 1.08 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.3 | 1.08 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.3 | 1.08 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.3 | 1.08 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 | 1.025236 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.054764 | 0.015212 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.015212 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.3 | 1.08 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.3 | 1.08 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.3 | 1.08 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.478302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.841698 | 0.233805 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.233805 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.3 | 1.08 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731934 | 0.365967 | 0.365967 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.714033 | 0.198343 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.198343 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.3 | 1.08 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | 0.227314 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.852686 | 0.236857 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.236857 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.3 | 1.08 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.489803 | 0.691612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.691612 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.3 | 1.08 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  7.755221 | 2.154228 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.154228 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.3 | 1.08 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.540115 | 1.261143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.261143 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.3 | 1.08 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039382 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.940309 | 1.927864 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.927864 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.3 | 1.08 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.780374 | 3.827882 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 3.827882 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.3 | 1.08 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.180505 | 0.327918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.327918 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.3 | 1.08 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.900505 | 0.527918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.527918 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.3 | 1.08 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.660505 | 1.294585 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.294585 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.3 | 1.08 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.580604 | 0.994612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.994612 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.3 | 1.08 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  13.54067 | 3.761297 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 3.761297 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.3 | 1.08 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.260736 | 1.461315 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.461315 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.3 | 1.08 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.140802 | 1.428001 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.428001 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.3 | 1.08 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   2.48254 | 0.689595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.689595 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.3 | 1.08 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.947078 | 0.263077 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.263077 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.3 | 1.08 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.470892 | 0.130803 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.130803 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.3 | 1.08 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.693848 | 0.192736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.192736 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.3 | 1.08 |  0.48 | 20.0 |  0.0 |  0.0 |   0.9844 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0678 | 0.296611 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.296611 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.3 | 1.08 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 |      1.2 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.3 | 1.08 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.725043 | 1.312512 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.312512 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.3 | 1.08 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.395689 |  1.77658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  1.77658 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.3 | 1.08 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.590795 | 0.441887 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.441887 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.3 | 1.08 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 25.497849 | 7.082736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 7.082736 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.3 | 1.08 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 46.112149 | 12.80893 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 12.80893 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.3 | 1.08 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  10.34443 | 2.873453 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.873453 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.3 | 1.08 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.08171 | 1.411586 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.411586 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.3 | 1.08 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 10.021679 |   2.7838 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |   2.7838 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.3 | 1.08 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 23.261476 | 6.461521 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 6.461521 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.3 | 1.08 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  19.54151 | 5.428197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 5.428197 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.3 | 1.08 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   8.62175 | 2.394931 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.394931 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.3 | 1.08 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.342063 | 0.928351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.928351 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.3 | 1.08 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.942028 | 1.928341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.928341 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.3 | 1.08 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.302063 | 1.195018 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.195018 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.3 | 1.08 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.662239 | 0.461733 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.461733 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.3 | 1.08 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.302169 | 0.361713 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.361713 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.3 | 1.08 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.661959 | 0.461655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.461655 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.3 | 1.08 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.941681 | 1.094912 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.094912 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.3 | 1.08 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061944 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.529028 |  0.42473 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  0.42473 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.3 | 1.08 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.225926 | 0.340535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.340535 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.3 | 1.08 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.845255 | 0.234793 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.234793 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.3 | 1.08 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.239541 | 0.066539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.066539 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.3 | 1.08 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.568057 | 0.157794 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.157794 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.3 | 1.08 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.138849 | 0.871903 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.871903 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.3 | 1.08 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.844795 | 0.234665 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.234665 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.3 | 1.08 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.508217 | 0.141171 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.141171 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.3 | 1.08 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 |     1.44 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.3 | 1.08 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.109687 | 0.030468 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.030468 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.3 | 1.08 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.018616 | 0.005171 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.005171 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.3 | 1.08 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.422793 | 0.117442 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.117442 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.3 | 1.08 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.778757 | 0.216321 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.216321 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.3 | 1.08 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.025048 | 0.284735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.284735 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.3 | 1.08 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062487 | 0.295135 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295135 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.3 | 1.08 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062846 | 0.295235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295235 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.3 | 1.08 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063137 | 0.295316 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295316 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.3 | 1.08 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063357 | 0.295377 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295377 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.3 | 1.08 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063542 | 0.295428 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295428 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.3 | 1.08 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0631 | 0.295306 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295306 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.3 | 1.08 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033434 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.623283 |  0.72869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  0.72869 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.3 | 1.08 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.06332 | 0.295367 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295367 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.3 | 1.08 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.017954 | 0.282765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.282765 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.3 | 1.08 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.641608 | 0.456002 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.456002 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.3 | 1.08 |  0.48 | 12.2 |  0.0 |  0.0 |  1.27835 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.920825 | 0.255785 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.255785 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.3 | 1.08 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.16167 | 0.044908 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.044908 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.3 | 1.08 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.128993 | 0.035831 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.035831 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.3 | 1.08 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.3 | 1.08 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.3 | 1.08 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143048 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.3 | 1.08 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.3 | 1.08 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.3 | 1.08 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.3 | 1.08 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179366 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.490317 | 0.136199 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.136199 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.3 | 1.08 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.818154 | 0.227265 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.227265 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.3 | 1.08 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.01409 | 0.281692 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.281692 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.3 | 1.08 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062098 | 0.295027 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295027 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.3 | 1.08 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062846 | 0.295235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295235 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.3 | 1.08 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0631 | 0.295306 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295306 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |

:ref:`water (negative runoff) <lland_v1_wasser_negq>`
_____________________________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_wasser_negq` for water areas of type |WASSER|
exactly (with :math:`NegQ = False`):

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test('lland_v2_wasser_negq',
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |        qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020141 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.020141 | -0.005595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005595 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 |  0.03912 |  0.01956 |  0.01956 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.01956 | -0.005433 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005433 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019396 | -0.005388 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005388 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019198 | -0.005333 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005333 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019396 | -0.005388 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005388 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041104 | 0.020552 | 0.020552 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.020552 | -0.005709 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005709 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.058382 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.058382 | -0.016217 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.016217 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 | 0.386575 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.386575 | -0.107382 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.107382 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.873907 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.873907 | -0.242752 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.242752 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 | 1.463511 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.463511 | -0.406531 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.406531 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 | 1.652872 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.652872 | -0.459131 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.459131 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 | 1.873973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.873973 | -0.520548 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.520548 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 | 1.025236 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.025236 | -0.284788 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.284788 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 | 1.325006 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.325006 | -0.368057 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.368057 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 | 1.479524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.479524 | -0.410979 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.410979 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.478302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.238302 | -0.066195 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.066195 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731934 | 0.365967 | 0.365967 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.365967 | -0.101657 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.101657 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | 0.227314 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.227314 | -0.063143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.063143 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.409803 |  0.391612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.391612 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.675221 |  1.854228 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.854228 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.460115 |  0.961143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.961143 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039382 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.860309 |  1.627864 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.627864 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 12.700374 |  3.527882 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  3.527882 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.100505 |  0.027918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.027918 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.820505 |  0.227918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.227918 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.580505 |  0.994585 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.994585 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.500604 |  0.694612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.694612 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  12.46067 |  3.461297 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  3.461297 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.180736 |  1.161315 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.161315 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.060802 |  1.128001 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.128001 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.40254 |  0.389595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.389595 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.132922 | -0.036923 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.036923 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.609108 | -0.169197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.169197 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.386152 | -0.107264 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.107264 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 |   0.9844 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0122 | -0.003389 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.003389 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 | 1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.123189 | -0.311997 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.311997 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.645043 |  1.012512 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.012512 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.315689 |   1.47658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   1.47658 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.510795 |  0.141887 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.141887 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 24.417849 |  6.782736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.782736 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 45.032149 |  12.50893 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  12.50893 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   9.26443 |  2.573453 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.573453 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   4.00171 |  1.111586 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.111586 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.941679 |    2.4838 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |    2.4838 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 22.181476 |  6.161521 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.161521 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  18.46151 |  5.128197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  5.128197 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   7.54175 |  2.094931 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.094931 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.262063 |  0.628351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.628351 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.862028 |  1.628341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.628341 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.222063 |  0.895018 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.895018 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.582239 |  0.161733 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.161733 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.222169 |  0.061714 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.061714 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.581959 |  0.161655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.161655 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.861681 |  0.794911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.794911 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061944 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.449028 |   0.12473 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.12473 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.145926 |  0.040535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.040535 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.234745 | -0.065207 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.065207 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.840459 | -0.233461 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.233461 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.511943 | -0.142206 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.142206 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.058849 |  0.571903 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.571903 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.235205 | -0.065335 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.065335 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.571783 | -0.158829 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.158829 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 | 1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.197919 | -0.332755 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.332755 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.970313 | -0.269532 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.269532 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.061384 | -0.294829 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.294829 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.657207 | -0.182558 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.182558 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.301243 | -0.083679 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.083679 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.054952 | -0.015265 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.015265 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017513 | -0.004865 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004865 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017154 | -0.004765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004765 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016863 | -0.004684 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004684 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016643 | -0.004623 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004623 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016458 | -0.004572 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004572 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0169 | -0.004694 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004694 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033434 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.543283 |   0.42869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.42869 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.01668 | -0.004633 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004633 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.062046 | -0.017235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017235 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.561608 |  0.156002 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.156002 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 |  1.27835 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.159175 | -0.044215 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.044215 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.91833 | -0.255092 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.255092 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.951007 | -0.264169 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.264169 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 | 1.678286 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.678286 | -0.466191 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.466191 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 | 1.678257 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.678257 | -0.466183 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.466183 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143048 | 2.143048 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.143048 | -0.595291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.595291 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 | 1.876679 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.876679 |   -0.5213 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   -0.5213 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 | 1.572383 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.572383 | -0.436773 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.436773 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 | 1.259666 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.259666 | -0.349907 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.349907 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179366 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.589683 | -0.163801 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.163801 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.261846 | -0.072735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.072735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.06591 | -0.018308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.018308 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017902 | -0.004973 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004973 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017154 | -0.004765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004765 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0169 | -0.004694 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004694 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |

:ref:`lakes <lland_v1_see>`
___________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_see` for water areas of type |SEE| exactly (with
:math:`NegQ = True`):

.. integration-test::

    >>> lnk(SEE)
    >>> negq(False)
    >>> test('lland_v2_see',
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020141 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0001 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 |  0.03912 |  0.01956 | 0.019263 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01956 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000297 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.018908 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000488 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.018523 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019198 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000675 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.018536 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00086 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041104 | 0.020552 | 0.019502 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020552 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00105 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.056949 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058382 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001433 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 |  0.38294 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.003635 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.864033 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.873907 |   0.0 |   0.0 |   0.0 |   0.0 | -0.009874 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 | 1.442102 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.463511 |   0.0 |   0.0 |   0.0 |   0.0 | -0.021409 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |  1.61617 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.652872 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036702 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 | 1.820088 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.873973 |   0.0 |   0.0 |   0.0 |   0.0 | -0.053885 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 |  0.95747 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.025236 |   0.0 |   0.0 |   0.0 |   0.0 | -0.067766 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 | 1.246219 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.325006 |   0.0 |   0.0 |   0.0 |   0.0 | -0.078787 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 | 1.387567 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.479524 |   0.0 |   0.0 |   0.0 |   0.0 | -0.091957 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.378724 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.238302 |   0.0 |   0.0 |   0.0 |   0.0 | -0.099578 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731934 | 0.365967 | 0.264372 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.365967 |   0.0 |   0.0 |   0.0 |   0.0 | -0.101594 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 |  0.12378 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.227314 |   0.0 |   0.0 |   0.0 |   0.0 | -0.103534 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 |  0.05359 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.409803 |   0.0 |   0.0 |   0.0 |   0.0 | -0.096607 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 |  -0.0106 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  6.675221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.055379 |   0.0 |      0.0 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.015455 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.460115 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00443 |   0.0 |      0.0 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039382 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.042004 | 0.011668 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.860309 |   0.0 |   0.0 |   0.0 |   0.0 |  0.042004 |   0.0 | 0.011668 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.133983 | 0.037218 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 12.700374 |   0.0 |   0.0 |   0.0 |   0.0 |  0.133983 |   0.0 | 0.037218 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.196231 | 0.054509 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.100505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.196231 |   0.0 | 0.054509 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.198867 | 0.055241 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.820505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.198867 |   0.0 | 0.055241 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.218806 |  0.06078 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.580505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.218806 |   0.0 |  0.06078 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.246874 | 0.068576 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.500604 |   0.0 |   0.0 |   0.0 |   0.0 |  0.246874 |   0.0 | 0.068576 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.318934 | 0.088593 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  12.46067 |   0.0 |   0.0 |   0.0 |   0.0 |  0.318934 |   0.0 | 0.088593 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.398484 |  0.11069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.180736 |   0.0 |   0.0 |   0.0 |   0.0 |  0.398484 |   0.0 |  0.11069 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.435521 | 0.120978 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.060802 |   0.0 |   0.0 |   0.0 |   0.0 |  0.435521 |   0.0 | 0.120978 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.458346 | 0.127318 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   1.40254 |   0.0 |   0.0 |   0.0 |   0.0 |  0.458346 |   0.0 | 0.127318 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.460089 | 0.127802 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.132922 |   0.0 |   0.0 |   0.0 |   0.0 |  0.460089 |   0.0 | 0.127802 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.451815 | 0.125504 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.609108 |   0.0 |   0.0 |   0.0 |   0.0 |  0.451815 |   0.0 | 0.125504 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.44237 | 0.122881 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386152 |   0.0 |   0.0 |   0.0 |   0.0 |   0.44237 |   0.0 | 0.122881 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 |   0.9844 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.435989 | 0.121108 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0122 |   0.0 |   0.0 |   0.0 |   0.0 |  0.435989 |   0.0 | 0.121108 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 | 1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.425993 | 0.118332 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.123189 |   0.0 |   0.0 |   0.0 |   0.0 |  0.425993 |   0.0 | 0.118332 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.434341 |  0.12065 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.645043 |   0.0 |   0.0 |   0.0 |   0.0 |  0.434341 |   0.0 |  0.12065 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.474613 | 0.131837 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.315689 |   0.0 |   0.0 |   0.0 |   0.0 |  0.474613 |   0.0 | 0.131837 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.498838 | 0.138566 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.510795 |   0.0 |   0.0 |   0.0 |   0.0 |  0.498838 |   0.0 | 0.138566 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.618095 | 0.171693 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 24.417849 |   0.0 |   0.0 |   0.0 |   0.0 |  0.618095 |   0.0 | 0.171693 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.957635 |  0.26601 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 45.032149 |   0.0 |   0.0 |   0.0 |   0.0 |  0.957635 |   0.0 |  0.26601 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.21794 | 0.338317 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   9.26443 |   0.0 |   0.0 |   0.0 |   0.0 |   1.21794 |   0.0 | 0.338317 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.271778 | 0.353272 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   4.00171 |   0.0 |   0.0 |   0.0 |   0.0 |  1.271778 |   0.0 | 0.353272 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.323559 | 0.367655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.941679 |   0.0 |   0.0 |   0.0 |   0.0 |  1.323559 |   0.0 | 0.367655 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.465339 | 0.407039 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 22.181476 |   0.0 |   0.0 |   0.0 |   0.0 |  1.465339 |   0.0 | 0.407039 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.65293 | 0.459147 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  18.46151 |   0.0 |   0.0 |   0.0 |   0.0 |   1.65293 |   0.0 | 0.459147 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.765761 | 0.490489 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   7.54175 |   0.0 |   0.0 |   0.0 |   0.0 |  1.765761 |   0.0 | 0.490489 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.796922 | 0.499145 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.262063 |   0.0 |   0.0 |   0.0 |   0.0 |  1.796922 |   0.0 | 0.499145 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.819491 | 0.505414 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.862028 |   0.0 |   0.0 |   0.0 |   0.0 |  1.819491 |   0.0 | 0.505414 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.846559 | 0.512933 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.222063 |   0.0 |   0.0 |   0.0 |   0.0 |  1.846559 |   0.0 | 0.512933 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.84709 | 0.513081 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.582239 |   0.0 |   0.0 |   0.0 |   0.0 |   1.84709 |   0.0 | 0.513081 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.83271 | 0.509086 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.222169 |   0.0 |   0.0 |   0.0 |   0.0 |   1.83271 |   0.0 | 0.509086 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.818478 | 0.505133 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.581959 |   0.0 |   0.0 |   0.0 |   0.0 |  1.818478 |   0.0 | 0.505133 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.817535 | 0.504871 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.861681 |   0.0 |   0.0 |   0.0 |   0.0 |  1.817535 |   0.0 | 0.504871 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061944 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.815901 | 0.504417 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.449028 |   0.0 |   0.0 |   0.0 |   0.0 |  1.815901 |   0.0 | 0.504417 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.80079 |  0.50022 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.145926 |   0.0 |   0.0 |   0.0 |   0.0 |   1.80079 |   0.0 |  0.50022 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.782427 | 0.495119 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.234745 |   0.0 |   0.0 |   0.0 |   0.0 |  1.782427 |   0.0 | 0.495119 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.759337 | 0.488705 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.840459 |   0.0 |   0.0 |   0.0 |   0.0 |  1.759337 |   0.0 | 0.488705 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.735106 | 0.481974 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.511943 |   0.0 |   0.0 |   0.0 |   0.0 |  1.735106 |   0.0 | 0.481974 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.725559 | 0.479322 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.058849 |   0.0 |   0.0 |   0.0 |   0.0 |  1.725559 |   0.0 | 0.479322 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.717443 | 0.477067 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.235205 |   0.0 |   0.0 |   0.0 |   0.0 |  1.717443 |   0.0 | 0.477067 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.696336 | 0.471205 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.571783 |   0.0 |   0.0 |   0.0 |   0.0 |  1.696336 |   0.0 | 0.471205 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 | 1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.670648 | 0.464069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.197919 |   0.0 |   0.0 |   0.0 |   0.0 |  1.670648 |   0.0 | 0.464069 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.64324 | 0.456455 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.970313 |   0.0 |   0.0 |   0.0 |   0.0 |   1.64324 |   0.0 | 0.456455 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.61678 | 0.449106 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.061384 |   0.0 |   0.0 |   0.0 |   0.0 |   1.61678 |   0.0 | 0.449106 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.592146 | 0.442263 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.657207 |   0.0 |   0.0 |   0.0 |   0.0 |  1.592146 |   0.0 | 0.442263 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.571539 | 0.436539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.301243 |   0.0 |   0.0 |   0.0 |   0.0 |  1.571539 |   0.0 | 0.436539 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.554132 | 0.431703 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.054952 |   0.0 |   0.0 |   0.0 |   0.0 |  1.554132 |   0.0 | 0.431703 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.538308 | 0.427308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017513 |   0.0 |   0.0 |   0.0 |   0.0 |  1.538308 |   0.0 | 0.427308 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.522829 | 0.423008 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 |  1.522829 |   0.0 | 0.423008 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.507507 | 0.418752 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016863 |   0.0 |   0.0 |   0.0 |   0.0 |  1.507507 |   0.0 | 0.418752 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.49234 | 0.414539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016643 |   0.0 |   0.0 |   0.0 |   0.0 |   1.49234 |   0.0 | 0.414539 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.477327 | 0.410369 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016458 |   0.0 |   0.0 |   0.0 |   0.0 |  1.477327 |   0.0 | 0.410369 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.462461 | 0.406239 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 |  1.462461 |   0.0 | 0.406239 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.447741 |  0.40215 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.447741 |   0.0 |  0.40215 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033434 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.440942 | 0.400262 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.543283 |   0.0 |   0.0 |   0.0 |   0.0 |  1.440942 |   0.0 | 0.400262 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.434187 | 0.398385 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01668 |   0.0 |   0.0 |   0.0 |   0.0 |  1.434187 |   0.0 | 0.398385 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.419749 | 0.394375 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.419749 |   0.0 | 0.394375 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.405228 | 0.390341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.062046 |   0.0 |   0.0 |   0.0 |   0.0 |  1.405228 |   0.0 | 0.390341 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.393737 | 0.387149 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.561608 |   0.0 |   0.0 |   0.0 |   0.0 |  1.393737 |   0.0 | 0.387149 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 |  1.27835 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.381865 | 0.383851 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.159175 |   0.0 |   0.0 |   0.0 |   0.0 |  1.381865 |   0.0 | 0.383851 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.362748 | 0.378541 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.91833 |   0.0 |   0.0 |   0.0 |   0.0 |  1.362748 |   0.0 | 0.378541 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.339888 | 0.372191 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.951007 |   0.0 |   0.0 |   0.0 |   0.0 |  1.339888 |   0.0 | 0.372191 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 | 1.678286 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.313469 | 0.364853 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678286 |   0.0 |   0.0 |   0.0 |   0.0 |  1.313469 |   0.0 | 0.364853 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 | 1.678257 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.283701 | 0.356584 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678257 |   0.0 |   0.0 |   0.0 |   0.0 |  1.283701 |   0.0 | 0.356584 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143048 | 2.143048 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.251913 | 0.347754 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.143048 |   0.0 |   0.0 |   0.0 |   0.0 |  1.251913 |   0.0 | 0.347754 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 | 1.876679 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.21946 | 0.338739 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.876679 |   0.0 |   0.0 |   0.0 |   0.0 |   1.21946 |   0.0 | 0.338739 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 | 1.572383 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.190169 | 0.330603 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.572383 |   0.0 |   0.0 |   0.0 |   0.0 |  1.190169 |   0.0 | 0.330603 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 | 1.259666 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.16424 |   0.3234 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.259666 |   0.0 |   0.0 |   0.0 |   0.0 |   1.16424 |   0.0 |   0.3234 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179366 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.14346 | 0.317628 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.589683 |   0.0 |   0.0 |   0.0 |   0.0 |   1.14346 |   0.0 | 0.317628 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.127849 | 0.313291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.261846 |   0.0 |   0.0 |   0.0 |   0.0 |  1.127849 |   0.0 | 0.313291 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.114997 | 0.309722 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06591 |   0.0 |   0.0 |   0.0 |   0.0 |  1.114997 |   0.0 | 0.309722 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.103486 | 0.306524 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017902 |   0.0 |   0.0 |   0.0 |   0.0 |  1.103486 |   0.0 | 0.306524 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.092332 | 0.303426 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 |  1.092332 |   0.0 | 0.303426 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.081294 | 0.300359 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 |  1.081294 |   0.0 | 0.300359 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.070366 | 0.297324 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.070366 |   0.0 | 0.297324 |

:ref:`streams <lland_v1_fluss>`
_______________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_fluss` for water areas of type |FLUSS| exactly (with
:math:`NegQ = True`):

.. integration-test::

    >>> lnk(FLUSS)
    >>> test('lland_v2_fluss',
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |       evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |       qah |       qa | inzp | wats | waes | bowa |     qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |     qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 |  0.018255 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.020141 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020141 |       0.0 |   0.0 |   0.0 |  0.0 | -0.001886 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 |  0.03912 |  0.01956 |  0.014419 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.01956 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01956 |       0.0 |   0.0 |   0.0 |  0.0 | -0.005141 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 |  0.011657 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019396 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 | -0.007739 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 |  0.009364 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019198 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019198 |       0.0 |   0.0 |   0.0 |  0.0 | -0.009834 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 |  0.007846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019396 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01155 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041104 | 0.020552 |  0.007472 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.020552 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020552 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01308 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 |  0.040404 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.058382 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.058382 |       0.0 |   0.0 |   0.0 |  0.0 | -0.017978 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 |  0.330537 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.386575 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386575 |       0.0 |   0.0 |   0.0 |  0.0 | -0.056038 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 |  0.712312 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.873907 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.873907 |       0.0 |   0.0 |   0.0 |  0.0 | -0.161595 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 |  1.117577 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.463511 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.463511 |       0.0 |   0.0 |   0.0 |  0.0 | -0.345934 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |  1.086622 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.652872 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.652872 |       0.0 |   0.0 |   0.0 |  0.0 | -0.566251 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 |  1.090045 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.873973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.873973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.783929 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 |  0.123203 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.025236 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.025236 |       0.0 |   0.0 |   0.0 |  0.0 | -0.902033 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 |  0.372566 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.325006 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.325006 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.95244 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 |  0.445078 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.479524 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.479524 |       0.0 |   0.0 |   0.0 |  0.0 | -1.034446 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | -0.520578 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.238302 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.238302 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.99888 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731934 | 0.365967 |    -0.507 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.365967 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.365967 |       0.0 |   0.0 |   0.0 |  0.0 | -0.872967 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | -0.540764 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.227314 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.227314 |       0.0 |   0.0 |   0.0 |  0.0 | -0.768078 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | -0.352311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.409803 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  1.290681 |  0.119122 |   0.0 |   0.0 |  0.0 | -0.527888 |   0.02538 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 |  0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.675221 |  0.919074 | 0.255298 |  0.0 |  0.0 |  0.0 |  0.0 |  1.850192 |  4.825029 |   0.0 |   0.0 |  0.0 | -0.145837 |  1.064911 |   0.0 |   0.0 |  0.0 |   0.0 | 0.255298 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 |  0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.460115 |  2.091991 | 0.581109 |  0.0 |  0.0 |  0.0 |  0.0 |  1.710992 |  1.749123 |   0.0 |   0.0 |  0.0 |  0.202945 |  1.889046 |   0.0 |   0.0 |  0.0 |   0.0 | 0.581109 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039382 | 0.019691 |  0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.860309 |  2.807552 | 0.779876 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829361 |  4.030949 |   0.0 |   0.0 |  0.0 |  0.487393 |  2.320159 |   0.0 |   0.0 |  0.0 |   0.0 | 0.779876 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 |  0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.700374 |  5.170333 | 1.436203 |  0.0 |  0.0 |  0.0 |  0.0 |  1.921262 | 10.779112 |   0.0 |   0.0 |  0.0 |  0.739258 |  4.431075 |   0.0 |   0.0 |  0.0 |   0.0 | 1.436203 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.100505 |  5.415219 | 1.504228 |  0.0 |  0.0 |  0.0 |  0.0 |  0.100505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.782998 |  4.632221 |   0.0 |   0.0 |  0.0 |   0.0 | 1.504228 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.820505 |  3.536298 | 0.982305 |  0.0 |  0.0 |  0.0 |  0.0 |  0.820505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.726714 |  2.809584 |   0.0 |   0.0 |  0.0 |   0.0 | 0.982305 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.580505 |  2.928372 | 0.813437 |  0.0 |  0.0 |  0.0 |  0.0 |   1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 |  0.828023 |  2.100349 |   0.0 |   0.0 |  0.0 |   0.0 | 0.813437 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038792 | 0.019396 |  0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.500604 |  2.779855 | 0.772182 |  0.0 |  0.0 |  0.0 |  0.0 |  1.600097 |  0.900507 |   0.0 |   0.0 |  0.0 |  0.978544 |  1.801312 |   0.0 |   0.0 |  0.0 |   0.0 | 0.772182 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |   0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  12.46067 |  4.622021 | 1.283895 |  0.0 |  0.0 |  0.0 |  0.0 |  1.919747 | 10.540922 |   0.0 |   0.0 |  0.0 |  1.121149 |  3.500872 |   0.0 |   0.0 |  0.0 |   0.0 | 1.283895 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 |  0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.180736 |  5.791671 | 1.608797 |  0.0 |  0.0 |  0.0 |  0.0 |  1.760808 |  2.419928 |   0.0 |   0.0 |  0.0 |  1.251025 |  4.540646 |   0.0 |   0.0 |  0.0 |   0.0 | 1.608797 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 |  0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.060802 |  5.024932 | 1.395814 |  0.0 |  0.0 |  0.0 |  0.0 |  1.753743 |  2.307059 |   0.0 |   0.0 |  0.0 |  1.342771 |  3.682161 |   0.0 |   0.0 |  0.0 |   0.0 | 1.395814 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |   0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   1.40254 |  4.047727 | 1.124369 |  0.0 |  0.0 |  0.0 |  0.0 |  1.287008 |  0.115532 |   0.0 |   0.0 |  0.0 |  1.373556 |  2.674171 |   0.0 |   0.0 |  0.0 |   0.0 | 1.124369 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 |  0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.132922 |  2.867695 | 0.796582 |  0.0 |  0.0 |  0.0 |  0.0 | -0.132922 |       0.0 |   0.0 |   0.0 |  0.0 |  1.224886 |  1.642809 |   0.0 |   0.0 |  0.0 |   0.0 | 0.796582 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 |  0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.609108 |  1.930575 | 0.536271 |  0.0 |  0.0 |  0.0 |  0.0 | -0.609108 |       0.0 |   0.0 |   0.0 |  0.0 |   0.93416 |  0.996414 |   0.0 |   0.0 |  0.0 |   0.0 | 0.536271 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 |  0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.386152 |   1.27965 | 0.355458 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386152 |       0.0 |   0.0 |   0.0 |  0.0 |  0.675294 |  0.604356 |   0.0 |   0.0 |  0.0 |   0.0 | 0.355458 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 |   0.9844 |   0.4922 |    0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0122 |  0.884469 | 0.245686 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0122 |       0.0 |   0.0 |   0.0 |  0.0 |  0.517909 |   0.36656 |   0.0 |   0.0 |  0.0 |   0.0 | 0.245686 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 |  1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.123189 |  0.540098 | 0.150027 |  0.0 |  0.0 |  0.0 |  0.0 | -1.123189 |       0.0 |   0.0 |   0.0 |  0.0 |  0.317768 |   0.22233 |   0.0 |   0.0 |  0.0 |   0.0 | 0.150027 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 |  0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.645043 |  0.867169 |  0.24088 |  0.0 |  0.0 |  0.0 |  0.0 |  1.725655 |  1.919389 |   0.0 |   0.0 |  0.0 |  0.323372 |  0.543797 |   0.0 |   0.0 |  0.0 |   0.0 |  0.24088 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 |  1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.315689 |  2.008267 | 0.557852 |  0.0 |  0.0 |  0.0 |  0.0 |  1.811878 |  3.503811 |   0.0 |   0.0 |  0.0 |  0.585638 |   1.42263 |   0.0 |   0.0 |  0.0 |   0.0 | 0.557852 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 |  0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.510795 |   2.18105 | 0.605847 |  0.0 |  0.0 |  0.0 |  0.0 |  0.510795 |       0.0 |   0.0 |   0.0 |  0.0 |  0.686066 |  1.494984 |   0.0 |   0.0 |  0.0 |   0.0 | 0.605847 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 |  0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 24.417849 |  6.481785 | 1.800496 |  0.0 |  0.0 |  0.0 |  0.0 |  1.959046 | 22.458802 |   0.0 |   0.0 |  0.0 |  0.789929 |  5.691856 |   0.0 |   0.0 |  0.0 |   0.0 | 1.800496 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 |  0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 45.032149 |  17.68086 |  4.91135 |  0.0 |  0.0 |  0.0 |  0.0 |  1.977794 | 43.054355 |   0.0 |   0.0 |  0.0 |   1.00361 | 16.677251 |   0.0 |   0.0 |  0.0 |   0.0 |  4.91135 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |   0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   9.26443 | 20.625552 |  5.72932 |  0.0 |  0.0 |  0.0 |  0.0 |   1.89206 |   7.37237 |   0.0 |   0.0 |  0.0 |   1.17217 | 19.453382 |   0.0 |   0.0 |  0.0 |   0.0 |  5.72932 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |   0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   4.00171 | 14.898206 | 4.138391 |  0.0 |  0.0 |  0.0 |  0.0 |  1.750107 |  2.251603 |   0.0 |   0.0 |  0.0 |  1.289369 | 13.608837 |   0.0 |   0.0 |  0.0 |   0.0 | 4.138391 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 |  0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.941679 | 11.549032 | 3.208064 |  0.0 |  0.0 |  0.0 |  0.0 |  1.888164 |  7.053515 |   0.0 |   0.0 |  0.0 |  1.385817 | 10.163215 |   0.0 |   0.0 |  0.0 |   0.0 | 3.208064 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 |  0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 22.181476 | 13.229438 | 3.674844 |  0.0 |  0.0 |  0.0 |  0.0 |  1.954917 | 20.226558 |   0.0 |   0.0 |  0.0 |  1.483128 | 11.746309 |   0.0 |   0.0 |  0.0 |   0.0 | 3.674844 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |   0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  18.46151 |  15.86018 | 4.405606 |  0.0 |  0.0 |  0.0 |  0.0 |  1.945833 | 16.515677 |   0.0 |   0.0 |  0.0 |  1.567799 | 14.292382 |   0.0 |   0.0 |  0.0 |   0.0 | 4.405606 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |   0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   7.54175 | 14.486291 |  4.02397 |  0.0 |  0.0 |  0.0 |  0.0 |  1.867405 |  5.674346 |   0.0 |   0.0 |  0.0 |  1.628979 | 12.857312 |   0.0 |   0.0 |  0.0 |   0.0 |  4.02397 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 |  0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.262063 | 10.615291 | 2.948692 |  0.0 |  0.0 |  0.0 |  0.0 |  1.557926 |  0.704138 |   0.0 |   0.0 |  0.0 |  1.643215 |  8.972076 |   0.0 |   0.0 |  0.0 |   0.0 | 2.948692 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 |  0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.862028 |  8.081246 | 2.244791 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829411 |  4.032618 |   0.0 |   0.0 |  0.0 |   1.65318 |  6.428066 |   0.0 |   0.0 |  0.0 |   0.0 | 2.244791 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 |  0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.222063 |  6.624871 | 1.840242 |  0.0 |  0.0 |  0.0 |  0.0 |   1.68964 |  1.532424 |   0.0 |   0.0 |  0.0 |  1.672035 |  4.952836 |   0.0 |   0.0 |  0.0 |   0.0 | 1.840242 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 |  0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582239 |  4.852022 | 1.347784 |  0.0 |  0.0 |  0.0 |  0.0 |  0.582239 |       0.0 |   0.0 |   0.0 |  0.0 |  1.571514 |  3.280508 |   0.0 |   0.0 |  0.0 |   0.0 | 1.347784 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 |  0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222169 |  3.348196 | 0.930054 |  0.0 |  0.0 |  0.0 |  0.0 |  0.222169 |       0.0 |   0.0 |   0.0 |  0.0 |  1.358467 |  1.989729 |   0.0 |   0.0 |  0.0 |   0.0 | 0.930054 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 |  0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.581959 |  2.393018 | 0.664727 |  0.0 |  0.0 |  0.0 |  0.0 |  0.581959 |       0.0 |   0.0 |   0.0 |  0.0 |  1.186187 |  1.206832 |   0.0 |   0.0 |  0.0 |   0.0 | 0.664727 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 |  0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.861681 |  2.166761 | 0.601878 |  0.0 |  0.0 |  0.0 |  0.0 |  1.650555 |  1.211126 |   0.0 |   0.0 |  0.0 |  1.176737 |  0.990025 |   0.0 |   0.0 |  0.0 |   0.0 | 0.601878 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061944 | 0.030972 |  0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.449028 |  1.969075 | 0.546965 |  0.0 |  0.0 |  0.0 |  0.0 |  0.449028 |       0.0 |   0.0 |   0.0 |  0.0 |  1.150098 |  0.818977 |   0.0 |   0.0 |  0.0 |   0.0 | 0.546965 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 |  0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.145926 |  1.491364 | 0.414268 |  0.0 |  0.0 |  0.0 |  0.0 |  0.145926 |       0.0 |   0.0 |   0.0 |  0.0 |  0.994629 |  0.496735 |   0.0 |   0.0 |  0.0 |   0.0 | 0.414268 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 |  0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.234745 |  1.106419 | 0.307339 |  0.0 |  0.0 |  0.0 |  0.0 | -0.234745 |       0.0 |   0.0 |   0.0 |  0.0 |  0.805134 |  0.301285 |   0.0 |   0.0 |  0.0 |   0.0 | 0.307339 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 |  0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.840459 |  0.742647 | 0.206291 |  0.0 |  0.0 |  0.0 |  0.0 | -0.840459 |       0.0 |   0.0 |   0.0 |  0.0 |  0.559908 |  0.182738 |   0.0 |   0.0 |  0.0 |   0.0 | 0.206291 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 |  0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.511943 |  0.447668 | 0.124352 |  0.0 |  0.0 |  0.0 |  0.0 | -0.511943 |       0.0 |   0.0 |   0.0 |  0.0 |  0.336832 |  0.110837 |   0.0 |   0.0 |  0.0 |   0.0 | 0.124352 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 |  1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.058849 |  0.555989 | 0.154441 |  0.0 |  0.0 |  0.0 |  0.0 |  1.514292 |  0.544558 |   0.0 |   0.0 |  0.0 |  0.372739 |   0.18325 |   0.0 |   0.0 |  0.0 |   0.0 | 0.154441 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 |  1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.235205 |   0.62521 | 0.173669 |  0.0 |  0.0 |  0.0 |  0.0 | -0.235205 |       0.0 |   0.0 |   0.0 |  0.0 |  0.415821 |  0.209389 |   0.0 |   0.0 |  0.0 |   0.0 | 0.173669 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 |  0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.571783 |  0.393289 | 0.109247 |  0.0 |  0.0 |  0.0 |  0.0 | -0.571783 |       0.0 |   0.0 |   0.0 |  0.0 |  0.266288 |  0.127001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.109247 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 |  1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.197919 |  0.132761 | 0.036878 |  0.0 |  0.0 |  0.0 |  0.0 | -1.197919 |       0.0 |   0.0 |   0.0 |  0.0 |  0.055731 |   0.07703 |   0.0 |   0.0 |  0.0 |   0.0 | 0.036878 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 |  0.866834 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.970313 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.970313 |       0.0 |   0.0 |   0.0 |  0.0 | -0.150201 |  0.046721 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 |  0.782331 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.061384 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.061384 |       0.0 |   0.0 |   0.0 |  0.0 | -0.307391 |  0.028338 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 |  0.268181 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.657207 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.657207 |       0.0 |   0.0 |   0.0 |  0.0 | -0.406214 |  0.017188 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | -0.106706 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.301243 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.301243 |       0.0 |   0.0 |   0.0 |  0.0 | -0.418374 |  0.010425 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 |   -0.3128 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.054952 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.054952 |       0.0 |   0.0 |   0.0 |  0.0 | -0.374076 |  0.006323 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | -0.291374 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017513 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017513 |       0.0 |   0.0 |   0.0 |  0.0 | -0.312722 |  0.003835 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | -0.239696 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017154 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 | -0.259176 |  0.002326 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | -0.197004 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016863 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016863 |       0.0 |   0.0 |   0.0 |  0.0 | -0.215278 |  0.001411 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | -0.161792 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016643 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016643 |       0.0 |   0.0 |   0.0 |  0.0 | -0.179291 |  0.000856 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | -0.132814 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016458 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016458 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.14979 |  0.000519 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 | -0.108448 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0169 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.125663 |  0.000315 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |  -0.08879 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.105954 |  0.000191 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033434 | 0.016717 |  0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.543283 |  0.079252 | 0.022014 |  0.0 |  0.0 |  0.0 |  0.0 |  1.352031 |  0.191252 |   0.0 |   0.0 |  0.0 |  0.038388 |  0.040864 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022014 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |   0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.01668 |  0.207615 | 0.057671 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01668 |       0.0 |   0.0 |   0.0 |  0.0 |  0.148326 |  0.059289 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057671 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |  0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |  0.154349 | 0.042875 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 |  0.118388 |  0.035961 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042875 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124092 | 0.062046 |  0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.062046 |  0.111441 | 0.030956 |  0.0 |  0.0 |  0.0 |  0.0 | -0.062046 |       0.0 |   0.0 |   0.0 |  0.0 |   0.08963 |  0.021811 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030956 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 |  0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.561608 |  0.133773 | 0.037159 |  0.0 |  0.0 |  0.0 |  0.0 |  0.561608 |       0.0 |   0.0 |   0.0 |  0.0 |  0.120543 |  0.013229 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037159 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 |  1.27835 | 0.639175 |  0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.159175 |  0.141015 | 0.039171 |  0.0 |  0.0 |  0.0 |  0.0 | -0.159175 |       0.0 |   0.0 |   0.0 |  0.0 |  0.132991 |  0.008024 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039171 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |   1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.91833 |  0.013799 | 0.003833 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.91833 |       0.0 |   0.0 |   0.0 |  0.0 |  0.008932 |  0.004867 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003833 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 |  1.271747 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.951007 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.951007 |       0.0 |   0.0 |   0.0 |  0.0 | -0.162212 |  0.002952 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 |  1.306768 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.678286 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678286 |       0.0 |   0.0 |   0.0 |  0.0 | -0.373309 |   0.00179 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 |  1.069485 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.678257 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678257 |       0.0 |   0.0 |   0.0 |  0.0 | -0.609858 |  0.001086 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143048 |  1.296651 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.143048 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -2.143048 |       0.0 |   0.0 |   0.0 |  0.0 | -0.847056 |  0.000659 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 |  0.820046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.876679 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.876679 |       0.0 |   0.0 |   0.0 |  0.0 | -1.057033 |  0.000399 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 |  0.395514 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.572383 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.572383 |       0.0 |   0.0 |   0.0 |  0.0 | -1.177111 |  0.000242 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 |  0.040338 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.259666 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.259666 |       0.0 |   0.0 |   0.0 |  0.0 | -1.219474 |  0.000147 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179366 | 0.589683 | -0.574241 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.589683 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.589683 |       0.0 |   0.0 |   0.0 |  0.0 | -1.164013 |  0.000089 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | -0.767301 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.261846 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.261846 |       0.0 |   0.0 |   0.0 |  0.0 | -1.029202 |  0.000054 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 | -0.805811 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.06591 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.06591 |       0.0 |   0.0 |   0.0 |  0.0 | -0.871754 |  0.000033 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | -0.703261 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017902 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017902 |       0.0 |   0.0 |   0.0 |  0.0 | -0.721183 |   0.00002 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | -0.576463 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017154 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.59363 |  0.000012 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 | -0.472201 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0169 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.489109 |  0.000007 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | -0.386541 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.403518 |  0.000004 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

:ref:`sealed surfaces <lland_v1_vers>`
______________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_vers` for land use type |VERS| exactly:

.. integration-test::

    >>> lnk(VERS)
    >>> test('lland_v2_vers',
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi | evb |   wgtf |    wnied |   schmpot | schm |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |       qah |       qa |     inzp | wats | waes | bowa |    qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |     qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.0 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 |  0.03912 |  0.01956 |      0.0 | 0.0 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.0 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.0 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.0 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041104 | 0.020552 |      0.0 | 0.0 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.0 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.0 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.0 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.0 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 | 0.0 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.0 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 | 0.0 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.0 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.0 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 | 0.0 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731934 | 0.365967 |      0.0 | 0.0 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.0 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |  0.0 |      0.76 |  0.071177 | 0.019771 | 0.649803 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071177 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019771 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 |  6.569803 |  0.0 |  0.0 | 0.0 |  0.0 |  6.569803 |  1.303993 |  0.36222 | 0.755221 |  0.0 |  0.0 |  0.0 | 1.847788 |  4.722014 |   0.0 |   0.0 |  0.0 | 0.297915 |  1.006079 |   0.0 |   0.0 |  0.0 |   0.0 |  0.36222 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 |  3.435221 |  0.0 |  0.0 | 0.0 |  0.0 |  3.435221 |  2.395771 | 0.665492 | 0.780115 |  0.0 |  0.0 |  0.0 | 1.708898 |  1.726323 |   0.0 |   0.0 |  0.0 | 0.565852 |  1.829919 |   0.0 |   0.0 |  0.0 |   0.0 | 0.665492 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039382 | 0.019691 | 0.019691 | 0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 |  5.860115 |  0.0 |  0.0 | 0.0 |  0.0 |  5.860115 |  3.064476 | 0.851243 | 0.780309 |  0.0 |  0.0 |  0.0 | 1.829355 |   4.03076 |   0.0 |   0.0 |  0.0 | 0.784332 |  2.280144 |   0.0 |   0.0 |  0.0 |   0.0 | 0.851243 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 12.700309 |  0.0 |  0.0 | 0.0 |  0.0 | 12.700309 |  5.389127 |  1.49698 | 0.780374 |  0.0 |  0.0 |  0.0 | 1.921262 | 10.779047 |   0.0 |   0.0 |  0.0 |  0.98237 |  4.406757 |   0.0 |   0.0 |  0.0 |   0.0 |  1.49698 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 |  0.100374 |  0.0 |  0.0 | 0.0 |  0.0 |  0.100374 |  5.599489 | 1.555414 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.100374 |       0.0 |   0.0 |   0.0 |  0.0 | 0.982029 |   4.61746 |   0.0 |   0.0 |  0.0 |   0.0 | 1.555414 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 |  0.820505 |  0.0 |  0.0 | 0.0 |  0.0 |  0.820505 |  3.690286 | 1.025079 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.820505 |       0.0 |   0.0 |   0.0 |  0.0 | 0.889655 |  2.800631 |   0.0 |   0.0 |  0.0 |   0.0 | 1.025079 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  3.580505 |  0.0 |  0.0 | 0.0 |  0.0 |  3.580505 |  3.056347 | 0.848985 | 0.780505 |  0.0 |  0.0 |  0.0 |  1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 | 0.961428 |  2.094919 |   0.0 |   0.0 |  0.0 |   0.0 | 0.848985 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038792 | 0.019396 | 0.019396 | 0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  2.500505 |  0.0 |  0.0 | 0.0 |  0.0 |  2.500505 |  2.885765 | 0.801601 | 0.780604 |  0.0 |  0.0 |  0.0 | 1.600081 |  0.900424 |   0.0 |   0.0 |  0.0 | 1.087765 |     1.798 |   0.0 |   0.0 |  0.0 |   0.0 | 0.801601 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 | 12.460604 |  0.0 |  0.0 | 0.0 |  0.0 | 12.460604 |  4.709405 | 1.308168 |  0.78067 |  0.0 |  0.0 |  0.0 | 1.919747 | 10.540857 |   0.0 |   0.0 |  0.0 |  1.21057 |  3.498835 |   0.0 |   0.0 |  0.0 |   0.0 | 1.308168 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 |   4.18067 |  0.0 |  0.0 | 0.0 |  0.0 |   4.18067 |  5.863622 | 1.628784 | 0.780736 |  0.0 |  0.0 |  0.0 | 1.760804 |  2.419866 |   0.0 |   0.0 |  0.0 | 1.324236 |  4.539386 |   0.0 |   0.0 |  0.0 |   0.0 | 1.628784 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 |  4.060736 |  0.0 |  0.0 | 0.0 |  0.0 |  4.060736 |  5.084082 | 1.412245 | 0.780802 |  0.0 |  0.0 |  0.0 | 1.753739 |  2.306997 |   0.0 |   0.0 |  0.0 | 1.402711 |  3.681371 |   0.0 |   0.0 |  0.0 |   0.0 | 1.412245 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 |  1.420802 |  0.0 |  0.0 | 0.0 |  0.0 |  1.420802 |  4.099108 | 1.138641 |  0.76254 |  0.0 |  0.0 |  0.0 | 1.296172 |   0.12463 |   0.0 |   0.0 |  0.0 | 1.423489 |  2.675619 |   0.0 |   0.0 |  0.0 |   0.0 | 1.138641 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 |   0.08254 |  0.0 |  0.0 | 0.0 |  0.0 |   0.08254 |  2.932078 | 0.814466 | 0.547078 |  0.0 |  0.0 |  0.0 |  0.08254 |       0.0 |   0.0 |   0.0 |  0.0 | 1.286749 |  1.645329 |   0.0 |   0.0 |  0.0 |   0.0 | 0.814466 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.0 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  2.058675 | 0.571854 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 1.060733 |  0.997943 |   0.0 |   0.0 |  0.0 |   0.0 | 0.571854 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.0 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.473737 | 0.409371 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.868455 |  0.605283 |   0.0 |   0.0 |  0.0 |   0.0 | 0.409371 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 |   0.9844 |   0.4922 |     0.48 | 0.0 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.078153 | 0.299487 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.71103 |  0.367123 |   0.0 |   0.0 |  0.0 |   0.0 | 0.299487 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.0 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.804814 | 0.223559 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.582143 |  0.222671 |   0.0 |   0.0 |  0.0 |   0.0 | 0.223559 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 |      3.52 |  0.0 |  0.0 | 0.0 |  0.0 |      3.52 |  1.156758 | 0.321322 | 0.125043 |  0.0 |  0.0 |  0.0 | 1.715909 |  1.804091 |   0.0 |   0.0 |  0.0 | 0.637319 |  0.519439 |   0.0 |   0.0 |  0.0 |   0.0 | 0.321322 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.0 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 |  6.405043 |  0.0 |  0.0 | 0.0 |  0.0 |  6.405043 |  2.457156 | 0.682543 |      0.0 |  0.0 |  0.0 |  0.0 | 1.843873 |   4.56117 |   0.0 |   0.0 |  0.0 | 0.844819 |  1.612337 |   0.0 |   0.0 |  0.0 |   0.0 | 0.682543 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.0 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 |      0.52 |  0.0 |  0.0 | 0.0 |  0.0 |      0.52 |  2.702734 |  0.75076 |      0.0 |  0.0 |  0.0 |  0.0 |     0.52 |       0.0 |   0.0 |   0.0 |  0.0 | 0.901931 |  1.800804 |   0.0 |   0.0 |  0.0 |   0.0 |  0.75076 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 |     24.04 |  0.0 |  0.0 | 0.0 |  0.0 |     24.04 |  6.764387 | 1.878996 | 0.377849 |  0.0 |  0.0 |  0.0 | 1.958403 | 22.081597 |   0.0 |   0.0 |  0.0 |  0.96741 |  5.796977 |   0.0 |   0.0 |  0.0 |   0.0 | 1.878996 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 45.057849 |  0.0 |  0.0 | 0.0 |  0.0 | 45.057849 | 17.827296 | 4.952027 | 0.352149 |  0.0 |  0.0 |  0.0 | 1.977806 | 43.080042 |   0.0 |   0.0 |  0.0 | 1.148864 | 16.678432 |   0.0 |   0.0 |  0.0 |   0.0 | 4.952027 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 |  9.392149 |  0.0 |  0.0 | 0.0 |  0.0 |  9.392149 | 20.776864 | 5.771351 |  0.22443 |  0.0 |  0.0 |  0.0 | 1.893528 |  7.498621 |   0.0 |   0.0 |  0.0 | 1.291233 | 19.485632 |   0.0 |   0.0 |  0.0 |   0.0 | 5.771351 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 |   3.74443 |  0.0 |  0.0 | 0.0 |  0.0 |   3.74443 | 14.985386 | 4.162607 |  0.48171 |  0.0 |  0.0 |  0.0 | 1.732937 |  2.011494 |   0.0 |   0.0 |  0.0 |  1.38537 | 13.600016 |   0.0 |   0.0 |  0.0 |   0.0 | 4.162607 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 |   8.68171 |  0.0 |  0.0 | 0.0 |  0.0 |   8.68171 | 11.522469 | 3.200686 | 0.741679 |  0.0 |  0.0 |  0.0 | 1.884815 |  6.796894 |   0.0 |   0.0 |  0.0 | 1.462598 | 10.059871 |   0.0 |   0.0 |  0.0 |   0.0 | 3.200686 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 22.141679 |  0.0 |  0.0 | 0.0 |  0.0 | 22.141679 |  13.17456 |   3.6596 | 0.781476 |  0.0 |  0.0 |  0.0 | 1.954836 | 20.186843 |   0.0 |   0.0 |  0.0 |  1.54569 |  11.62887 |   0.0 |   0.0 |  0.0 |   0.0 |   3.6596 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 | 18.461476 |  0.0 |  0.0 | 0.0 |  0.0 | 18.461476 | 15.832992 | 4.398053 |  0.78151 |  0.0 |  0.0 |  0.0 | 1.945833 | 16.515642 |   0.0 |   0.0 |  0.0 | 1.619013 | 14.213979 |   0.0 |   0.0 |  0.0 |   0.0 | 4.398053 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |   7.54151 |  0.0 |  0.0 | 0.0 |  0.0 |   7.54151 | 14.480611 | 4.022392 |  0.78175 |  0.0 |  0.0 |  0.0 | 1.867401 |  5.674109 |   0.0 |   0.0 |  0.0 |  1.67091 | 12.809702 |   0.0 |   0.0 |  0.0 |   0.0 | 4.022392 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 |   2.26175 |  0.0 |  0.0 | 0.0 |  0.0 |   2.26175 | 10.620641 | 2.950178 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.557865 |  0.703886 |   0.0 |   0.0 |  0.0 | 1.677538 |  8.943103 |   0.0 |   0.0 |  0.0 |   0.0 | 2.950178 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 |  5.862063 |  0.0 |  0.0 | 0.0 |  0.0 |  5.862063 |  8.091731 | 2.247703 | 0.782028 |  0.0 |  0.0 |  0.0 | 1.829412 |  4.032652 |   0.0 |   0.0 |  0.0 | 1.681277 |  6.410455 |   0.0 |   0.0 |  0.0 |   0.0 | 2.247703 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 |  3.222028 |  0.0 |  0.0 | 0.0 |  0.0 |  3.222028 |  6.637192 | 1.843664 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.689637 |  1.532392 |   0.0 |   0.0 |  0.0 | 1.695038 |  4.942153 |   0.0 |   0.0 |  0.0 |   0.0 | 1.843664 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 |  0.582063 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582063 |  4.864354 | 1.351209 | 0.782239 |  0.0 |  0.0 |  0.0 | 0.582063 |       0.0 |   0.0 |   0.0 |  0.0 | 1.590331 |  3.274023 |   0.0 |   0.0 |  0.0 |   0.0 | 1.351209 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 |  0.222239 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222239 |   3.35966 | 0.933239 | 0.782169 |  0.0 |  0.0 |  0.0 | 0.222239 |       0.0 |   0.0 |   0.0 |  0.0 | 1.373864 |  1.985796 |   0.0 |   0.0 |  0.0 |   0.0 | 0.933239 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 |  0.582169 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582169 |  2.403264 | 0.667573 | 0.781959 |  0.0 |  0.0 |  0.0 | 0.582169 |       0.0 |   0.0 |   0.0 |  0.0 | 1.198819 |  1.204446 |   0.0 |   0.0 |  0.0 |   0.0 | 0.667573 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 |  2.861959 |  0.0 |  0.0 | 0.0 |  0.0 |  2.861959 |   2.17573 | 0.604369 | 0.781681 |  0.0 |  0.0 |  0.0 | 1.650589 |   1.21137 |   0.0 |   0.0 |  0.0 |   1.1871 |  0.988629 |   0.0 |   0.0 |  0.0 |   0.0 | 0.604369 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061944 | 0.030972 | 0.030972 | 0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 |  0.461681 |  0.0 |  0.0 | 0.0 |  0.0 |  0.461681 |  1.977946 | 0.549429 | 0.769028 |  0.0 |  0.0 |  0.0 | 0.461681 |       0.0 |   0.0 |   0.0 |  0.0 | 1.159771 |  0.818175 |   0.0 |   0.0 |  0.0 |   0.0 | 0.549429 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 |  0.209028 |  0.0 |  0.0 | 0.0 |  0.0 |  0.209028 |  1.505815 | 0.418282 | 0.705926 |  0.0 |  0.0 |  0.0 | 0.209028 |       0.0 |   0.0 |   0.0 |  0.0 | 1.009567 |  0.496248 |   0.0 |   0.0 |  0.0 |   0.0 | 0.418282 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.145867 | 0.318296 | 0.471181 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.844878 |   0.30099 |   0.0 |   0.0 |  0.0 |   0.0 | 0.318296 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.0 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.874287 | 0.242857 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.691727 |  0.182559 |   0.0 |   0.0 |  0.0 |   0.0 | 0.242857 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.0 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.677066 | 0.188074 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.566338 |  0.110728 |   0.0 |   0.0 |  0.0 |   0.0 | 0.188074 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.0 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 |      2.32 |  0.0 |  0.0 | 0.0 |  0.0 |      2.32 |  0.837794 | 0.232721 |      0.0 |  0.0 |  0.0 |  0.0 | 1.568966 |  0.751034 |   0.0 |   0.0 |  0.0 | 0.610618 |  0.227176 |   0.0 |   0.0 |  0.0 |   0.0 | 0.232721 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.0 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 |      0.04 |  0.0 |  0.0 | 0.0 |  0.0 |      0.04 |  0.914426 | 0.254007 |      0.0 |  0.0 |  0.0 |  0.0 |     0.04 |       0.0 |   0.0 |   0.0 |  0.0 | 0.641144 |  0.273282 |   0.0 |   0.0 |  0.0 |   0.0 | 0.254007 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.0 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.694183 | 0.192828 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.528429 |  0.165754 |   0.0 |   0.0 |  0.0 |   0.0 | 0.192828 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 0.0 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.533176 | 0.148104 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.432641 |  0.100535 |   0.0 |   0.0 |  0.0 |   0.0 | 0.148104 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 | 0.0 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.415194 | 0.115332 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.354216 |  0.060977 |   0.0 |   0.0 |  0.0 |   0.0 | 0.115332 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 0.0 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.326992 | 0.090831 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.290008 |  0.036985 |   0.0 |   0.0 |  0.0 |   0.0 | 0.090831 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 | 0.0 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.259871 | 0.072186 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.237438 |  0.022432 |   0.0 |   0.0 |  0.0 |   0.0 | 0.072186 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.0 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.208004 | 0.057779 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.194398 |  0.013606 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057779 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.0 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.167412 | 0.046503 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.15916 |  0.008252 |   0.0 |   0.0 |  0.0 |   0.0 | 0.046503 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 | 0.0 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.135314 | 0.037587 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.130309 |  0.005005 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037587 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.109724 | 0.030479 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.106688 |  0.003036 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030479 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.0 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |   0.08919 | 0.024775 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.087349 |  0.001841 |   0.0 |   0.0 |  0.0 |   0.0 | 0.024775 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.0 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.072632 | 0.020176 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071515 |  0.001117 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020176 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.0 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.059229 | 0.016452 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.058552 |  0.000677 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016452 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.048349 |  0.01343 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.047938 |  0.000411 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01343 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.039497 | 0.010972 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.039248 |  0.000249 |   0.0 |   0.0 |  0.0 |   0.0 | 0.010972 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033434 | 0.016717 | 0.016717 | 0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |  0.0 |      0.76 |  0.103462 | 0.028739 | 0.783283 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.103311 |  0.000151 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028739 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.151263 | 0.042018 | 0.766603 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151171 |  0.000092 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042018 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.123824 | 0.034396 |  0.74963 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.123769 |  0.000056 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034396 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 | 0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.101367 | 0.028157 | 0.687585 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.101333 |  0.000034 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028157 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 |  0.727585 |  0.0 |  0.0 | 0.0 |  0.0 |  0.727585 |  0.151126 | 0.041979 | 0.521608 |  0.0 |  0.0 |  0.0 | 0.727585 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151106 |   0.00002 |   0.0 |   0.0 |  0.0 |   0.0 | 0.041979 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 |  1.27835 | 0.639175 | 0.639175 | 0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 |  0.201608 |  0.0 |  0.0 | 0.0 |  0.0 |  0.201608 |  0.206356 | 0.057321 | 0.160825 |  0.0 |  0.0 |  0.0 | 0.201608 |       0.0 |   0.0 |   0.0 |  0.0 | 0.206344 |  0.000012 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057321 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.0 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.186612 | 0.051837 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.186604 |  0.000008 |   0.0 |   0.0 |  0.0 |   0.0 | 0.051837 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.0 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.152783 |  0.04244 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.152779 |  0.000005 |   0.0 |   0.0 |  0.0 |   0.0 |  0.04244 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.125087 | 0.034746 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.125084 |  0.000003 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034746 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 0.0 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.102412 | 0.028448 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.102411 |  0.000002 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028448 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143048 |      0.0 | 0.0 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.083848 | 0.023291 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.083847 |  0.000001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023291 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 0.0 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.068648 | 0.019069 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.068648 |  0.000001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019069 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 0.0 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.056204 | 0.015612 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.056204 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015612 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 0.0 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.046016 | 0.012782 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.046016 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012782 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179366 | 0.589683 |      0.0 | 0.0 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.037675 | 0.010465 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.037675 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.010465 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.0 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.030846 | 0.008568 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.030845 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.008568 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.0 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.025254 | 0.007015 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.025254 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.007015 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.0 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.020676 | 0.005743 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.020676 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.005743 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.016928 | 0.004702 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.016928 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004702 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |   0.01386 |  0.00385 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.01386 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00385 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.011347 | 0.003152 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.011347 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003152 |

:ref:`acre <lland_v1_acker_qkap>`
_________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_acker_qkap` for modified capillary rise settings
exactly:

.. integration-test::

    >>> lnk(ACKER)
    >>> kapgrenz(option='FK')
    >>> rbeta(True)
    >>> test('lland_v2_acker_qkap',
    ...      axis1=(fluxes.qbb, fluxes.qkap),
    ...      axis2=states.bowa)
    |                date | nied | teml |       pet |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |      qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |  0.100707 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |       0.0 |     0.01 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.065025 |      0.0 |      0.0 |     0.01 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000246 |      0.0 | -0.000399 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |  0.097801 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 |  0.03912 |  0.01956 |      0.0 | 0.004846 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |       0.0 | 0.010033 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.130146 |      0.0 |      0.0 | 0.010033 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000722 |      0.0 | -0.001191 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004821 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 | 0.010065 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.195261 |      0.0 |      0.0 | 0.010065 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001177 |      0.0 | -0.001975 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.09599 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004786 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 | 0.010098 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.260377 |      0.0 |      0.0 | 0.010098 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001612 |      0.0 | -0.002751 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |  0.096981 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038792 | 0.019396 |      0.0 | 0.004851 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.01013 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.325396 |      0.0 |      0.0 |  0.01013 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002026 |      0.0 |  -0.00352 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |  0.102761 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041104 | 0.020552 |      0.0 | 0.005155 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |       0.0 | 0.010163 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.390078 |      0.0 |      0.0 | 0.010163 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002422 |      0.0 | -0.004281 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  0.291908 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 |  0.01469 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |       0.0 | 0.010195 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.445193 |      0.0 |      0.0 | 0.010195 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002801 |      0.0 | -0.005034 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 |  1.932875 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 |  0.09752 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |       0.0 | 0.010223 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   20.41745 |      0.0 |      0.0 | 0.010223 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003162 |      0.0 |  -0.00578 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 |  4.369536 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.220171 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |       0.0 | 0.010209 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   20.26707 |      0.0 |      0.0 | 0.010209 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003506 |      0.0 | -0.006519 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 |  7.317556 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.366109 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |       0.0 | 0.010134 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.970828 |      0.0 |      0.0 | 0.010134 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003831 |      0.0 |  -0.00725 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 |  8.264362 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 | 0.407671 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |       0.0 | 0.009985 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.633171 |      0.0 |      0.0 | 0.009985 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004135 |      0.0 | -0.007974 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 |  9.369867 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.454685 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |       0.0 | 0.009817 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.248669 |      0.0 |      0.0 | 0.009817 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004416 |      0.0 |  -0.00869 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 |  5.126178 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.24406 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |       0.0 | 0.009624 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.074984 |      0.0 |      0.0 | 0.009624 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004675 |      0.0 |   -0.0094 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 |   6.62503 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.312677 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |       0.0 | 0.009537 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   18.83277 |      0.0 |      0.0 | 0.009537 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004914 |      0.0 | -0.010102 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 |  7.397619 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.344861 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |       0.0 | 0.009416 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.558493 |      0.0 |      0.0 | 0.009416 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005136 |      0.0 | -0.010798 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 |   2.39151 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 | 0.054764 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |       0.0 | 0.009279 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   18.57445 |      0.0 |      0.0 | 0.009279 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005342 |      0.0 | -0.011487 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 |  1.829834 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731934 | 0.365967 |      0.0 | 0.084172 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |       0.0 | 0.009287 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.560991 |      0.0 |      0.0 | 0.009287 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005534 |      0.0 | -0.012168 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  1.136569 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.052246 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |       0.0 |  0.00928 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.579465 |      0.0 |      0.0 |  0.00928 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005717 |      0.0 | -0.012843 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  0.750986 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |   0.02131 |  0.00929 |      0.0 |      0.0 | 0.08 |   0.02131 |      0.0 |      0.0 | 0.649803 |  0.0 |  0.0 |  19.388865 |  0.02131 |      0.0 |  0.00929 |      0.0 |    -0.08 | 0.001996 |      0.0 | 0.005891 |      0.0 | -0.013511 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |  0.223895 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 |  0.221019 | 0.009694 |      0.0 |      0.0 | 0.08 |  0.221019 | 0.016094 | 0.004471 | 0.755221 |  0.0 |  0.0 |  25.807955 | 0.221019 |      0.0 | 0.009694 |      0.0 |    -0.08 |   0.0242 |      0.0 | 0.006067 |      0.0 | -0.014173 |   0.0 | 0.004471 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |  0.099425 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 |  0.141928 | 0.012904 |      0.0 |      0.0 | 0.08 |  0.141928 | 0.043965 | 0.012212 | 0.780115 |  0.0 |  0.0 |  29.168344 | 0.141928 |      0.0 | 0.012904 |      0.0 |    -0.08 |  0.05247 |      0.0 | 0.006322 |      0.0 | -0.014828 |   0.0 | 0.012212 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |  0.098454 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039382 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 |   0.28445 | 0.014584 |      0.0 |      0.0 | 0.08 |   0.28445 | 0.073242 | 0.020345 | 0.780309 |  0.0 |  0.0 |  34.809425 |  0.28445 |      0.0 | 0.014584 |      0.0 |    -0.08 | 0.082034 |      0.0 | 0.006685 |      0.0 | -0.015476 |   0.0 | 0.020345 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |  0.098128 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 |   0.80163 | 0.017405 |      0.0 |      0.0 | 0.08 |   0.80163 | 0.158182 |  0.04394 | 0.780374 |  0.0 |  0.0 |  46.770699 |  0.80163 |      0.0 | 0.017405 |      0.0 |    -0.08 | 0.167162 |      0.0 | 0.007139 |      0.0 | -0.016118 |   0.0 |  0.04394 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 |  0.007364 | 0.023385 |      0.0 |      0.0 | 0.08 |  0.007364 | 0.198818 | 0.055227 | 0.780505 |  0.0 |  0.0 |  46.920324 | 0.007364 |      0.0 | 0.023385 |      0.0 |    -0.08 | 0.207785 |      0.0 | 0.007787 |      0.0 | -0.016754 |   0.0 | 0.055227 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |  0.097474 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 |  0.060883 |  0.02346 |      0.0 |      0.0 | 0.08 |  0.060883 | 0.167633 | 0.046565 | 0.780505 |  0.0 |  0.0 |  47.736487 | 0.060883 |      0.0 |  0.02346 |      0.0 |    -0.08 | 0.176467 |      0.0 |  0.00855 |      0.0 | -0.017383 |   0.0 | 0.046565 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |  0.097474 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |   0.27871 | 0.023868 |      0.0 |      0.0 | 0.08 |   0.27871 | 0.167196 | 0.046443 | 0.780505 |  0.0 |  0.0 |  51.094413 |  0.27871 |      0.0 | 0.023868 |      0.0 |    -0.08 | 0.175916 |      0.0 | 0.009287 |      0.0 | -0.018006 |   0.0 | 0.046443 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |  0.096981 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038792 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.207202 | 0.025547 |      0.0 |      0.0 | 0.08 |  0.207202 | 0.179268 | 0.049797 | 0.780604 |  0.0 |  0.0 |   53.44217 | 0.207202 |      0.0 | 0.025547 |      0.0 |    -0.08 | 0.187852 |      0.0 | 0.010039 |      0.0 | -0.018623 |   0.0 | 0.049797 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |  0.096652 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.187264 | 0.026721 |      0.0 |      0.0 | 0.08 |  1.187264 | 0.278264 | 0.077295 |  0.78067 |  0.0 |  0.0 |  64.768788 | 1.157728 | 0.029537 | 0.026721 |      0.0 |    -0.08 |  0.28038 | 0.006293 | 0.010824 |      0.0 | -0.019234 |   0.0 | 0.077295 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |  0.096321 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 |  0.457075 | 0.032384 |      0.0 |      0.0 | 0.08 |  0.457075 | 0.374843 | 0.104123 | 0.780736 |  0.0 |  0.0 |  68.539999 | 0.457075 |      0.0 | 0.032384 |      0.0 |    -0.08 | 0.373797 | 0.009146 | 0.011739 |      0.0 | -0.019839 |   0.0 | 0.104123 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.09599 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 |  0.472944 |  0.03427 |      0.0 |      0.0 | 0.08 |  0.472944 | 0.388281 | 0.107856 | 0.780802 |  0.0 |  0.0 |   72.17352 | 0.472944 |      0.0 |  0.03427 |      0.0 |    -0.08 | 0.390379 | 0.005547 | 0.012792 |      0.0 | -0.020437 |   0.0 | 0.107856 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |  0.187298 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 |  0.172327 | 0.036087 |      0.0 |      0.0 | 0.08 |  0.172327 | 0.373411 | 0.103725 |  0.76254 |  0.0 |  0.0 |  73.465909 | 0.172327 |      0.0 | 0.036087 |      0.0 |    -0.08 | 0.377192 | 0.003364 | 0.013884 |      0.0 |  -0.02103 |   0.0 | 0.103725 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  1.264612 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 |  0.010126 | 0.036733 |      0.0 |      0.0 | 0.08 |  0.010126 | 0.320272 | 0.088965 | 0.547078 |  0.0 |  0.0 |   73.58159 | 0.010126 |      0.0 | 0.036733 |      0.0 |    -0.08 | 0.324865 | 0.002041 | 0.014983 |      0.0 | -0.021617 |   0.0 | 0.088965 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 |  3.045538 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 |  0.04555 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |       0.0 | 0.036791 |      0.0 |      0.0 | 0.08 |       0.0 |  0.26195 | 0.072764 |      0.0 |  0.0 |  0.0 |  73.579249 |      0.0 |      0.0 | 0.036791 |      0.0 |    -0.08 | 0.266864 | 0.001238 | 0.016045 |      0.0 | -0.022198 |   0.0 | 0.072764 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 |  1.930758 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.283556 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |       0.0 |  0.03679 |      0.0 |      0.0 | 0.08 |       0.0 | 0.213525 | 0.059313 |      0.0 |  0.0 |  0.0 |  73.338903 |      0.0 |      0.0 |  0.03679 |      0.0 |    -0.08 |  0.21849 | 0.000751 | 0.017057 |      0.0 | -0.022773 |   0.0 | 0.059313 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 |  2.461001 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 |   0.9844 |   0.4922 |     0.48 | 0.008941 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |       0.0 | 0.036669 |      0.0 |      0.0 | 0.08 |       0.0 | 0.174014 | 0.048337 |      0.0 |  0.0 |  0.0 |  73.373293 |      0.0 |      0.0 | 0.036669 |      0.0 |    -0.08 | 0.178885 | 0.000455 | 0.018016 |      0.0 | -0.023342 |   0.0 | 0.048337 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 |  6.215945 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.823379 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |       0.0 | 0.036687 |      0.0 |      0.0 | 0.08 |       0.0 | 0.141755 | 0.039376 |      0.0 |  0.0 |  0.0 |  72.593227 |      0.0 |      0.0 | 0.036687 |      0.0 |    -0.08 | 0.146458 | 0.000276 | 0.018927 |      0.0 | -0.023906 |   0.0 | 0.039376 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 |  3.374783 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 |  0.436285 | 0.036297 |      0.0 |      0.0 | 0.08 |  0.436285 | 0.156256 | 0.043405 | 0.125043 |  0.0 |  0.0 |  75.720646 | 0.436285 |      0.0 | 0.036297 |      0.0 |    -0.08 |  0.16077 | 0.000168 | 0.019783 |      0.0 | -0.024464 |   0.0 | 0.043405 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 |  8.821555 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.720288 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 |  0.850271 |  0.03786 |      0.0 |      0.0 | 0.08 |  0.850271 | 0.245195 |  0.06811 |      0.0 |  0.0 |  0.0 |   80.59727 | 0.850271 |      0.0 |  0.03786 |      0.0 |    -0.08 | 0.249483 | 0.000102 | 0.020627 |      0.0 | -0.025017 |   0.0 |  0.06811 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 |  4.046025 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.007124 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 |  0.071497 | 0.040299 |      0.0 |      0.0 | 0.08 |  0.071497 | 0.281478 | 0.078188 |      0.0 |  0.0 |  0.0 |   81.07835 | 0.071497 |      0.0 | 0.040299 |      0.0 |    -0.08 | 0.285453 | 0.000062 | 0.021527 |      0.0 | -0.025564 |   0.0 | 0.078188 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 |  2.110757 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 |  3.847472 | 0.040539 |      0.0 |      0.0 | 0.08 |  3.847472 | 0.848321 | 0.235645 | 0.377849 |  0.0 |  0.0 | 101.310339 | 1.740089 | 2.107383 | 0.040539 |      0.0 |    -0.08 | 0.402939 | 0.449039 | 0.022449 |      0.0 | -0.026105 |   0.0 | 0.235645 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 |  2.239257 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 10.314783 | 0.050655 |  0.00135 | 0.456552 |  0.0 | 10.314783 | 3.105022 | 0.862506 | 0.352149 |  0.0 |  0.0 | 135.544849 | 1.903052 | 8.411731 | 0.050655 |  0.00135 | 0.456552 | 0.660585 | 2.444759 |  0.02358 | 0.000065 | -0.023968 |   0.0 | 0.862506 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 |  2.877848 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 |  2.700025 | 0.067772 | 0.190725 | 0.627724 |  0.0 |  2.700025 |  4.10496 | 1.140267 |  0.22443 |  0.0 |  0.0 | 141.350751 | 1.629633 | 1.070392 | 0.067772 | 0.190725 | 0.627724 | 0.860199 | 3.228424 | 0.025321 | 0.009348 | -0.018334 |   0.0 | 1.140267 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 |  1.591452 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 |  1.124141 | 0.070675 | 0.239314 | 0.656754 |  0.0 |  1.124141 | 3.149877 | 0.874966 |  0.48171 |  0.0 |  0.0 | 143.004297 | 1.110432 | 0.013709 | 0.070675 | 0.239314 | 0.656754 | 0.951049 | 2.154166 | 0.027463 | 0.028959 |  -0.01176 |   0.0 | 0.874966 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  0.291604 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 |  2.710266 | 0.071502 | 0.253811 | 0.665021 |  0.0 |  2.710266 | 2.641882 | 0.733856 | 0.741679 |  0.0 |  0.0 | 147.985407 | 1.631033 | 1.079233 | 0.071502 | 0.253811 | 0.665021 | 1.028696 | 1.538984 | 0.029591 | 0.049678 | -0.005067 |   0.0 | 0.733856 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |  0.092622 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 |  7.724215 | 0.073993 | 0.299162 | 0.689927 |  0.0 |  7.724215 | 3.640366 | 1.011213 | 0.781476 |  0.0 |  0.0 | 161.339789 | 1.870537 | 5.853678 | 0.073993 | 0.299162 | 0.689927 | 1.160312 | 2.375336 | 0.031696 | 0.071298 |  0.001724 |   0.0 | 1.011213 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |  0.092451 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  7.433494 |  0.08067 |  0.43237 | 0.756699 |  0.0 |  7.433494 | 5.113928 | 1.420535 |  0.78151 |  0.0 |  0.0 | 171.098031 | 1.865474 |  5.56802 |  0.08067 |  0.43237 | 0.756699 |  1.28858 | 3.683094 | 0.033923 | 0.099426 |  0.008905 |   0.0 | 1.420535 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |  0.091248 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.297279 | 0.085549 | 0.539546 |  0.80549 |  0.0 |  3.297279 | 5.145996 | 1.429443 |  0.78175 |  0.0 |  0.0 | 173.911677 |  1.69672 | 1.600559 | 0.085549 | 0.539546 |  0.80549 | 1.377348 | 3.579442 | 0.036323 | 0.136294 |  0.016588 |   0.0 | 1.429443 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |  0.089683 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 |  1.006677 | 0.086956 | 0.571889 | 0.819558 |  0.0 |  1.006677 | 4.069916 | 1.130532 | 0.782063 |  0.0 |  0.0 | 173.688346 | 1.006633 | 0.000044 | 0.086956 | 0.571889 | 0.819558 | 1.370611 | 2.459805 | 0.038758 | 0.176233 |  0.024508 |   0.0 | 1.130532 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |  0.089858 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 |  2.637943 | 0.086844 | 0.569299 | 0.818442 |  0.0 |  2.637943 | 3.358087 | 0.932802 | 0.782028 |  0.0 |  0.0 | 175.437882 | 1.620917 | 1.017026 | 0.086844 | 0.569299 | 0.818442 | 1.362163 | 1.708644 | 0.041106 |  0.21376 |  0.032414 |   0.0 | 0.932802 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |  0.089683 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 |   1.47069 | 0.087719 | 0.589694 | 0.827189 |  0.0 |   1.47069 | 2.965027 | 0.823619 | 0.782063 |  0.0 |  0.0 | 175.684618 | 1.320047 | 0.150643 | 0.087719 | 0.589694 | 0.827189 |  1.38089 | 1.251921 | 0.043358 |  0.24858 |  0.040278 |   0.0 | 0.823619 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |  0.088805 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 |  0.263876 | 0.087842 | 0.592589 | 0.828423 |  0.0 |  0.263876 | 2.432273 | 0.675631 | 0.782239 |  0.0 |  0.0 | 174.493951 | 0.263876 |      0.0 | 0.087842 | 0.592589 | 0.828423 | 1.270946 | 0.786505 | 0.045525 | 0.281182 |  0.048114 |   0.0 | 0.675631 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |  0.089157 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 |  0.098934 | 0.087247 | 0.578661 |  0.82247 |  0.0 |  0.098934 | 1.963553 | 0.545431 | 0.782169 |  0.0 |  0.0 | 173.128878 | 0.098934 |      0.0 | 0.087247 | 0.578661 |  0.82247 | 1.072948 |  0.47704 | 0.047574 | 0.310142 |  0.055849 |   0.0 | 0.545431 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |  0.090207 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 |  0.254662 | 0.086564 | 0.562828 | 0.815644 |  0.0 |  0.254662 | 1.648177 | 0.457827 | 0.781959 |  0.0 |  0.0 | 171.991348 | 0.254662 |      0.0 | 0.086564 | 0.562828 | 0.815644 | 0.910974 | 0.289339 | 0.049492 | 0.334929 |  0.063443 |   0.0 | 0.457827 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |  0.091593 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 |  1.243591 | 0.085996 | 0.549747 | 0.809957 |  0.0 |  1.243591 | 1.543981 | 0.428884 | 0.781681 |  0.0 |  0.0 | 172.164015 | 1.195877 | 0.047714 | 0.085996 | 0.549747 | 0.809957 | 0.880153 | 0.185659 | 0.051286 | 0.355984 |  0.070899 |   0.0 | 0.428884 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |  0.154861 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061944 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 |  0.199221 | 0.086082 | 0.551726 |  0.81082 |  0.0 |  0.199221 | 1.471016 | 0.408616 | 0.769028 |  0.0 |  0.0 | 170.977847 | 0.199221 |      0.0 | 0.086082 | 0.551726 |  0.81082 | 0.844044 | 0.121216 | 0.052981 | 0.374519 |  0.078257 |   0.0 | 0.408616 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  0.470369 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 |  0.088682 | 0.085489 | 0.538179 | 0.804889 |  0.0 |  0.088682 |  1.32115 | 0.366986 | 0.705926 |  0.0 |  0.0 | 169.669636 | 0.088682 |      0.0 | 0.085489 | 0.538179 | 0.804889 | 0.716805 | 0.073521 | 0.054581 | 0.390727 |  0.085517 |   0.0 | 0.366986 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  1.173726 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 | 0.084835 | 0.523369 | 0.798348 |  0.0 |       0.0 | 1.191989 | 0.331108 | 0.471181 |  0.0 |  0.0 | 168.263084 |      0.0 |      0.0 | 0.084835 | 0.523369 | 0.798348 |  0.59464 | 0.044593 | 0.056072 | 0.404042 |  0.092642 |   0.0 | 0.331108 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 |  4.202296 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.363261 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |       0.0 | 0.084132 |   0.5076 | 0.791315 |  0.0 |       0.0 | 1.085618 | 0.301561 |      0.0 |  0.0 |  0.0 | 166.516777 |      0.0 |      0.0 | 0.084132 |   0.5076 | 0.791315 |  0.48685 | 0.027047 | 0.057458 | 0.414635 |  0.099629 |   0.0 | 0.301561 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 |  4.359715 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.502927 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |       0.0 | 0.083258 | 0.488247 | 0.782584 |  0.0 |       0.0 | 1.002754 | 0.278543 |      0.0 |  0.0 |  0.0 | 164.659761 |      0.0 |      0.0 | 0.083258 | 0.488247 | 0.782584 | 0.398599 | 0.016405 | 0.058737 | 0.422546 |  0.106468 |   0.0 | 0.278543 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 |  5.305753 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 |  0.25617 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 |  0.914244 |  0.08233 | 0.467944 | 0.773299 |  0.0 |  0.914244 | 1.022793 | 0.284109 |      0.0 |  0.0 |  0.0 | 164.485775 | 0.914244 |      0.0 |  0.08233 | 0.467944 | 0.773299 | 0.411968 |  0.00995 |  0.05991 | 0.427816 |  0.113149 |   0.0 | 0.284109 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 |  5.376027 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269918 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 |  0.015591 | 0.082243 | 0.466056 | 0.772429 |  0.0 |  0.015591 | 1.037145 | 0.288096 |      0.0 |  0.0 |  0.0 | 162.919538 | 0.015591 |      0.0 | 0.082243 | 0.466056 | 0.772429 | 0.418853 | 0.006035 | 0.061001 | 0.431543 |  0.119713 |   0.0 | 0.288096 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 |  4.658915 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.560059 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |       0.0 |  0.08146 |  0.44918 | 0.764598 |  0.0 |       0.0 | 0.970152 | 0.269487 |      0.0 |  0.0 |  0.0 | 161.064241 |      0.0 |      0.0 |  0.08146 |  0.44918 | 0.764598 | 0.344294 |  0.00366 | 0.062018 | 0.434011 |  0.126169 |   0.0 | 0.269487 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 |  7.789594 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.171448 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |       0.0 | 0.080532 |  0.42946 | 0.755321 |  0.0 |       0.0 | 0.914023 | 0.253895 |      0.0 |  0.0 |  0.0 | 158.627479 |      0.0 |      0.0 | 0.080532 |  0.42946 | 0.755321 | 0.281884 |  0.00222 | 0.062944 | 0.434501 |  0.132475 |   0.0 | 0.253895 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 |  4.851567 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 | 0.946734 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |       0.0 | 0.079314 | 0.404012 | 0.743137 |  0.0 |       0.0 | 0.867307 | 0.240919 |      0.0 |  0.0 |  0.0 | 156.454282 |      0.0 |      0.0 | 0.079314 | 0.404012 | 0.743137 | 0.230787 | 0.001347 | 0.063771 |  0.43279 |  0.138612 |   0.0 | 0.240919 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 |   5.30692 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.033386 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 | 0.078227 | 0.381758 | 0.732271 |  0.0 |       0.0 | 0.827819 |  0.22995 |      0.0 |  0.0 |  0.0 |  154.22864 |      0.0 |      0.0 | 0.078227 | 0.381758 | 0.732271 | 0.188952 | 0.000817 | 0.064503 | 0.428975 |  0.144573 |   0.0 |  0.22995 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 |  3.286036 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 | 0.638397 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |       0.0 | 0.077114 | 0.359406 | 0.721143 |  0.0 |       0.0 | 0.794106 | 0.220585 |      0.0 |  0.0 |  0.0 | 152.432579 |      0.0 |      0.0 | 0.077114 | 0.359406 | 0.721143 | 0.154701 | 0.000495 | 0.065145 |   0.4234 |  0.150365 |   0.0 | 0.220585 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  1.506216 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.292049 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |       0.0 | 0.076216 | 0.341699 | 0.712163 |  0.0 |       0.0 | 0.765119 | 0.212533 |      0.0 |  0.0 |  0.0 | 151.010451 |      0.0 |      0.0 | 0.076216 | 0.341699 | 0.712163 | 0.126659 |   0.0003 | 0.065706 | 0.416454 |     0.156 |   0.0 | 0.212533 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  0.274762 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053189 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 | 0.075505 | 0.327892 | 0.705052 |  0.0 |       0.0 | 0.740253 | 0.205626 |      0.0 |  0.0 |  0.0 | 149.848812 |      0.0 |      0.0 | 0.075505 | 0.327892 | 0.705052 | 0.103699 | 0.000182 | 0.066201 | 0.408672 |  0.161498 |   0.0 | 0.205626 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |  0.087565 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 | 0.016928 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |       0.0 | 0.074924 | 0.316756 | 0.699244 |  0.0 |       0.0 | 0.718977 | 0.199716 |      0.0 |  0.0 |  0.0 | 148.740959 |      0.0 |      0.0 | 0.074924 | 0.316756 | 0.699244 | 0.084902 | 0.000111 | 0.066641 | 0.400446 |  0.166878 |   0.0 | 0.199716 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016559 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.07437 | 0.306255 | 0.693705 |  0.0 |       0.0 | 0.700731 | 0.194648 |      0.0 |  0.0 |  0.0 | 147.650069 |      0.0 |      0.0 |  0.07437 | 0.306255 | 0.693705 | 0.069512 | 0.000067 | 0.067031 | 0.391974 |  0.172147 |   0.0 | 0.194648 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |  0.084317 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016256 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 | 0.073825 | 0.296031 |  0.68825 |  0.0 |       0.0 |  0.68496 | 0.190267 |      0.0 |  0.0 |  0.0 | 146.575706 |      0.0 |      0.0 | 0.073825 | 0.296031 |  0.68825 | 0.056911 | 0.000041 | 0.067376 | 0.383322 |  0.177309 |   0.0 | 0.190267 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |  0.083215 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.016022 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |       0.0 | 0.073288 | 0.286076 | 0.682879 |  0.0 |       0.0 | 0.671197 | 0.186444 |      0.0 |  0.0 |  0.0 | 145.517442 |      0.0 |      0.0 | 0.073288 | 0.286076 | 0.682879 | 0.046595 | 0.000025 | 0.067677 | 0.374534 |  0.182367 |   0.0 | 0.186444 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |  0.082289 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015822 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |       0.0 | 0.072759 | 0.276382 | 0.677587 |  0.0 |       0.0 | 0.659069 | 0.183075 |      0.0 |  0.0 |  0.0 | 144.474892 |      0.0 |      0.0 | 0.072759 | 0.276382 | 0.677587 | 0.038149 | 0.000015 | 0.067938 | 0.365647 |   0.18732 |   0.0 | 0.183075 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016224 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 | 0.072237 | 0.266941 | 0.672374 |  0.0 |       0.0 | 0.648271 | 0.180075 |      0.0 |  0.0 |  0.0 | 143.447115 |      0.0 |      0.0 | 0.072237 | 0.266941 | 0.672374 | 0.031234 | 0.000009 |  0.06816 | 0.356696 |  0.192173 |   0.0 | 0.180075 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016271 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.071724 | 0.257741 | 0.667236 |  0.0 |       0.0 | 0.638558 | 0.177377 |      0.0 |  0.0 |  0.0 | 142.434144 |      0.0 |      0.0 | 0.071724 | 0.257741 | 0.667236 | 0.025572 | 0.000006 | 0.068346 | 0.347709 |  0.196925 |   0.0 | 0.177377 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |  0.083584 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033434 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 |  0.228252 | 0.071217 |  0.24878 | 0.662171 |  0.0 |  0.228252 | 0.651109 | 0.180864 | 0.783283 |  0.0 |  0.0 | 141.983724 | 0.228252 |      0.0 | 0.071217 |  0.24878 | 0.662171 | 0.042313 | 0.000003 | 0.068499 | 0.338714 |   0.20158 |   0.0 | 0.180864 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |    0.0834 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |       0.0 | 0.070992 |  0.24483 | 0.659919 |  0.0 |       0.0 | 0.659385 | 0.183163 | 0.766603 |  0.0 |  0.0 | 141.007984 |      0.0 |      0.0 | 0.070992 |  0.24483 | 0.659919 | 0.054642 | 0.000002 | 0.068626 | 0.329965 |  0.206151 |   0.0 | 0.183163 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.070504 | 0.236344 |  0.65504 |  0.0 |       0.0 | 0.645562 | 0.179323 |  0.74963 |  0.0 |  0.0 | 140.046095 |      0.0 |      0.0 | 0.070504 | 0.236344 |  0.65504 | 0.044737 | 0.000001 | 0.068729 | 0.321453 |  0.210642 |   0.0 | 0.179323 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  0.310229 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124092 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |       0.0 | 0.070023 | 0.228078 |  0.65023 |  0.0 |       0.0 | 0.633425 | 0.175951 | 0.687585 |  0.0 |  0.0 | 139.097764 |      0.0 |      0.0 | 0.070023 | 0.228078 |  0.65023 | 0.036627 | 0.000001 | 0.068804 | 0.312954 |   0.21504 |   0.0 | 0.175951 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  1.391958 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 |  0.210201 | 0.069549 | 0.220024 | 0.645489 |  0.0 |  0.210201 |  0.64236 | 0.178433 | 0.521608 |  0.0 |  0.0 | 138.680086 | 0.210201 |      0.0 | 0.069549 | 0.220024 | 0.645489 | 0.049674 |      0.0 | 0.068852 | 0.304487 |  0.219347 |   0.0 | 0.178433 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 |  3.195876 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 |  1.27835 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 |  0.057838 |  0.06934 | 0.216508 |   0.6434 |  0.0 |  0.057838 | 0.653239 | 0.181455 | 0.160825 |  0.0 |  0.0 | 137.894608 | 0.057838 |      0.0 |  0.06934 | 0.216508 |   0.6434 | 0.064503 |      0.0 |  0.06888 | 0.296279 |  0.223576 |   0.0 | 0.181455 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 |  5.191651 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.720177 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 | 0.068947 | 0.209947 | 0.639473 |  0.0 |       0.0 | 0.642876 | 0.178577 |      0.0 |  0.0 |  0.0 | 136.256065 |      0.0 |      0.0 | 0.068947 | 0.209947 | 0.639473 | 0.057878 |      0.0 | 0.068893 | 0.288371 |  0.227734 |   0.0 | 0.178577 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 |  7.155036 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.901722 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |       0.0 | 0.068128 | 0.196478 |  0.63128 |  0.0 |       0.0 | 0.628309 |  0.17453 |      0.0 |  0.0 |  0.0 | 134.458457 |      0.0 |      0.0 | 0.068128 | 0.196478 |  0.63128 | 0.047387 |      0.0 | 0.068876 | 0.280256 |   0.23179 |   0.0 |  0.17453 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 |  8.391432 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 |  1.58643 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 | 0.067229 | 0.182048 | 0.622292 |  0.0 |       0.0 |  0.61492 | 0.170811 |      0.0 |  0.0 |  0.0 | 132.000457 |      0.0 |      0.0 | 0.067229 | 0.182048 | 0.622292 | 0.038797 |      0.0 | 0.068817 | 0.271585 |   0.23572 |   0.0 | 0.170811 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 |  8.391286 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 |  1.57939 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |       0.0 |    0.066 | 0.162921 | 0.610002 |  0.0 |       0.0 | 0.602119 | 0.167255 |      0.0 |  0.0 |  0.0 | 129.582144 |      0.0 |      0.0 |    0.066 | 0.162921 | 0.610002 | 0.031764 |      0.0 | 0.068709 |  0.26214 |  0.239506 |   0.0 | 0.167255 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 10.715238 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143048 |      0.0 | 2.007477 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |       0.0 | 0.064791 | 0.144806 | 0.597911 |  0.0 |       0.0 | 0.589507 | 0.163752 |      0.0 |  0.0 |  0.0 | 126.767159 |      0.0 |      0.0 | 0.064791 | 0.144806 | 0.597911 | 0.026006 |      0.0 | 0.068548 | 0.251821 |  0.243132 |   0.0 | 0.163752 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 |  9.383394 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.747864 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |       0.0 | 0.063384 | 0.124637 | 0.583836 |  0.0 |       0.0 | 0.576876 | 0.160243 |      0.0 |  0.0 |  0.0 | 124.247439 |      0.0 |      0.0 | 0.063384 | 0.124637 | 0.583836 | 0.021292 |      0.0 |  0.06833 | 0.240662 |  0.246592 |   0.0 | 0.160243 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 |  7.861915 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.456406 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |       0.0 | 0.062124 | 0.107459 | 0.571237 |  0.0 |       0.0 | 0.564164 | 0.156712 |      0.0 |  0.0 |  0.0 | 122.050213 |      0.0 |      0.0 | 0.062124 | 0.107459 | 0.571237 | 0.017433 |      0.0 | 0.068058 |  0.22879 |  0.249885 |   0.0 | 0.156712 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 |  6.298329 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.160821 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |       0.0 | 0.061025 | 0.093188 | 0.560251 |  0.0 |       0.0 | 0.551594 | 0.153221 |      0.0 |  0.0 |  0.0 | 120.174928 |      0.0 |      0.0 | 0.061025 | 0.093188 | 0.560251 | 0.014273 |      0.0 | 0.067741 | 0.216553 |  0.253027 |   0.0 | 0.153221 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 |  2.948416 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179366 | 0.589683 |      0.0 | 0.540928 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |       0.0 | 0.060087 | 0.081557 | 0.550875 |  0.0 |       0.0 | 0.539364 | 0.149823 |      0.0 |  0.0 |  0.0 |  118.94148 |      0.0 |      0.0 | 0.060087 | 0.081557 | 0.550875 | 0.011685 |      0.0 | 0.067391 | 0.204251 |  0.256038 |   0.0 | 0.149823 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  1.309232 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.239446 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |       0.0 | 0.059471 | 0.074193 | 0.544707 |  0.0 |       0.0 | 0.527746 | 0.146596 |      0.0 |  0.0 |  0.0 | 118.023663 |      0.0 |      0.0 | 0.059471 | 0.074193 | 0.544707 | 0.009567 |      0.0 | 0.067019 | 0.192219 |   0.25894 |   0.0 | 0.146596 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |   0.32955 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.060127 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 | 0.059012 | 0.068866 | 0.540118 |  0.0 |       0.0 | 0.516963 | 0.143601 |      0.0 |  0.0 |  0.0 | 117.295539 |      0.0 |      0.0 | 0.059012 | 0.068866 | 0.540118 | 0.007833 |      0.0 |  0.06664 | 0.180729 |  0.261761 |   0.0 | 0.143601 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |  0.089508 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016299 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |       0.0 | 0.058648 | 0.064736 | 0.536478 |  0.0 |       0.0 | 0.507069 | 0.140852 |      0.0 |  0.0 |  0.0 | 116.619379 |      0.0 |      0.0 | 0.058648 | 0.064736 | 0.536478 | 0.006413 |      0.0 | 0.066259 | 0.169884 |  0.264513 |   0.0 | 0.140852 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |  0.085771 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 |  0.01559 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.05831 | 0.060977 | 0.533097 |  0.0 |       0.0 | 0.498028 | 0.138341 |      0.0 |  0.0 |  0.0 | 115.951405 |      0.0 |      0.0 |  0.05831 | 0.060977 | 0.533097 | 0.005251 |      0.0 | 0.065879 | 0.159696 |  0.267202 |   0.0 | 0.138341 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |    0.0845 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015331 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 | 0.057976 | 0.057338 | 0.529757 |  0.0 |       0.0 | 0.489758 | 0.136044 |      0.0 |  0.0 |  0.0 | 115.291004 |      0.0 |      0.0 | 0.057976 | 0.057338 | 0.529757 | 0.004299 |      0.0 | 0.065502 | 0.150126 |  0.269831 |   0.0 | 0.136044 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |  0.084864 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015369 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.057646 | 0.053814 | 0.526455 |  0.0 |       0.0 | 0.482173 | 0.133937 |      0.0 |  0.0 |  0.0 |  114.63772 |      0.0 |      0.0 | 0.057646 | 0.053814 | 0.526455 |  0.00352 |      0.0 | 0.065127 | 0.141125 |  0.272401 |   0.0 | 0.133937 |

:ref:`acre (winter) <lland_v1_acker_winter>`
____________________________________________

The following calculation reproduces the outflow values of the integration
test :ref:`lland_v1_acker_winter` for a snow event exactly (note that
we need to adapt the |PET| values to the changed |TemL| values):

.. integration-test::

    >>> inputs.teml.series = numpy.linspace(-20.0, 20.0, 96)
    >>> inputs.pet.series = (
    ...     0.03733, 0.037872, 0.038415, 0.038955, 0.039492, 0.040028, 0.110905,
    ...     0.71239, 1.641935, 2.661377, 3.014447, 3.343815, 1.81419, 2.535277,
    ...     3.137525, 1.010042, 0.776177, 0.503072, 0.346957, 0.105225, 0.047837,
    ...     0.048345, 0.048848, 0.04935, 0.04985, 0.05035, 0.050845, 0.051342,
    ...     0.051835, 0.052327, 0.102702, 0.69626, 1.692095, 1.078725, 1.382545,
    ...     3.516795, 1.935692, 5.120597, 2.372527, 1.267695, 1.365632, 1.772603,
    ...     0.997112, 0.185172, 0.0595, 0.059965, 0.060428, 0.06089, 0.06135,
    ...     0.06181, 0.062267, 0.062723, 0.063177, 0.06363, 0.10774, 0.325317,
    ...     0.811522, 2.973865, 3.12363, 3.798742, 3.874722, 3.373822, 5.667417,
    ...     3.434725, 3.787713, 2.368437, 1.120002, 0.211415, 0.070242, 0.07067,
    ...     0.0711, 0.071525, 0.07195, 0.072375, 0.072795, 0.073217, 0.073637,
    ...     0.074055, 0.2705, 1.23095, 2.866595, 4.66193, 6.271463, 6.833792,
    ...     6.738053, 8.078045, 7.222285, 6.044787, 4.837587, 2.353282, 1.092032,
    ...     0.289745, 0.080167, 0.080565, 0.080962, 0.081357)
    >>> test('lland_v2_acker_winter',
    ...      axis1=(inputs.nied, fluxes.wada),
    ...      axis2=(states.waes, states.wats))
    |                date | nied |       teml |      pet |  qz | qzh |  nkor |       tkor |      nbes |      sbes |      et0 |     evpo |      evi |      evb |      wgtf |     wnied |   schmpot |     schm |      wada |      qdb |     qib1 |     qib2 |      qbb | qkap |     qdgz |      qah |       qa |     inzp |       wats |       waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |      qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 |      -20.0 |  0.03733 | 0.0 | 0.0 |   0.0 |      -19.2 |       0.0 |       0.0 | 0.014932 | 0.007466 |      0.0 | 0.001844 |   -3.2064 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 |     0.01 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.068156 |      0.0 |      0.0 |     0.01 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000246 |      0.0 | -0.000399 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | -19.578947 | 0.037872 | 0.0 | 0.0 |   0.0 | -18.778947 |       0.0 |       0.0 | 0.015149 | 0.007574 |      0.0 | 0.001877 | -3.136084 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010034 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.136245 |      0.0 |      0.0 | 0.010034 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000722 |      0.0 | -0.001191 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | -19.157895 | 0.038415 | 0.0 | 0.0 |   0.0 | -18.357895 |       0.0 |       0.0 | 0.015366 | 0.007683 |      0.0 |  0.00191 | -3.065768 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010068 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.204267 |      0.0 |      0.0 | 0.010068 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001177 |      0.0 | -0.001975 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | -18.736842 | 0.038955 | 0.0 | 0.0 |   0.0 | -17.936842 |       0.0 |       0.0 | 0.015582 | 0.007791 |      0.0 | 0.001943 | -2.995453 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010102 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.272221 |      0.0 |      0.0 | 0.010102 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001612 |      0.0 | -0.002751 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | -18.315789 | 0.039492 | 0.0 | 0.0 |   0.0 | -17.515789 |       0.0 |       0.0 | 0.015797 | 0.007898 |      0.0 | 0.001976 | -2.925137 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010136 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.340109 |      0.0 |      0.0 | 0.010136 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002027 |      0.0 |  -0.00352 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | -17.894737 | 0.040028 | 0.0 | 0.0 |   0.0 | -17.094737 |       0.0 |       0.0 | 0.016011 | 0.008006 |      0.0 |  0.00201 | -2.854821 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01017 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.407929 |      0.0 |      0.0 |  0.01017 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002423 |      0.0 | -0.004281 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | -17.473684 | 0.110905 | 0.0 | 0.0 |   0.0 | -16.673684 |       0.0 |       0.0 | 0.044362 | 0.022181 |      0.0 | 0.005586 | -2.784505 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010204 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |   20.47214 |      0.0 |      0.0 | 0.010204 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002802 |      0.0 | -0.005034 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | -17.052632 |  0.71239 | 0.0 | 0.0 |   0.0 | -16.252632 |       0.0 |       0.0 | 0.284956 | 0.142478 |      0.0 | 0.035988 | -2.714189 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010236 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.505916 |      0.0 |      0.0 | 0.010236 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003164 |      0.0 |  -0.00578 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | -16.631579 | 1.641935 | 0.0 | 0.0 |   0.0 | -15.831579 |       0.0 |       0.0 | 0.656774 | 0.328387 |      0.0 | 0.083077 | -2.643874 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010253 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.492585 |      0.0 |      0.0 | 0.010253 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003509 |      0.0 | -0.006519 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | -16.210526 | 2.661377 | 0.0 | 0.0 |   0.0 | -15.410526 |       0.0 |       0.0 | 1.064551 | 0.532275 |      0.0 | 0.134574 | -2.573558 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010246 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.427765 |      0.0 |      0.0 | 0.010246 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003838 |      0.0 |  -0.00725 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | -15.789474 | 3.014447 | 0.0 | 0.0 |   0.0 | -14.989474 |       0.0 |       0.0 | 1.205779 | 0.602889 |      0.0 | 0.151965 | -2.503242 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010214 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.345586 |      0.0 |      0.0 | 0.010214 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004149 |      0.0 | -0.007974 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | -15.368421 | 3.343815 | 0.0 | 0.0 |   0.0 | -14.568421 |       0.0 |       0.0 | 1.337526 | 0.668763 |      0.0 | 0.167918 | -2.432926 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010173 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.247494 |      0.0 |      0.0 | 0.010173 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004444 |      0.0 |  -0.00869 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | -14.947368 |  1.81419 | 0.0 | 0.0 |   0.0 | -14.147368 |       0.0 |       0.0 | 0.725676 | 0.362838 |      0.0 | 0.090683 | -2.362611 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010124 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.226688 |      0.0 |      0.0 | 0.010124 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004722 |      0.0 |   -0.0094 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | -14.526316 | 2.535277 | 0.0 | 0.0 |   0.0 | -13.726316 |       0.0 |       0.0 | 1.014111 | 0.507055 |      0.0 | 0.126601 | -2.292295 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010113 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.169974 |      0.0 |      0.0 | 0.010113 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004986 |      0.0 | -0.010102 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | -14.105263 | 3.137525 | 0.0 | 0.0 |   0.0 | -13.305263 |       0.0 |       0.0 |  1.25501 | 0.627505 |      0.0 | 0.156253 | -2.221979 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010085 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.083635 |      0.0 |      0.0 | 0.010085 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005235 |      0.0 | -0.010798 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | -13.684211 | 1.010042 | 0.0 | 0.0 |  0.24 | -12.884211 |       0.0 |       0.0 | 0.404017 | 0.202008 | 0.202008 |      0.0 | -2.151663 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010042 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.037992 |        0.0 |        0.0 |  20.153594 |      0.0 |      0.0 | 0.010042 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00547 |      0.0 | -0.011487 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | -13.263158 | 0.776177 | 0.0 | 0.0 |   0.0 | -12.463158 |       0.0 |       0.0 | 0.310471 | 0.155235 | 0.037992 | 0.029172 | -2.081347 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010077 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.194345 |      0.0 |      0.0 | 0.010077 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005694 |      0.0 | -0.012168 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | -12.842105 | 0.503072 | 0.0 | 0.0 |   0.0 | -12.042105 |       0.0 |       0.0 | 0.201229 | 0.100614 |      0.0 | 0.025083 | -2.011032 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010097 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.239165 |      0.0 |      0.0 | 0.010097 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005908 |      0.0 | -0.012843 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | -12.421053 | 0.346957 | 0.0 | 0.0 |  1.56 | -11.621053 |      0.76 |      0.76 | 0.138783 | 0.069391 | 0.069391 |      0.0 | -1.940716 | -0.018459 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01012 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.730609 |       0.76 |       0.76 |  20.309046 |      0.0 |      0.0 |  0.01012 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006113 |      0.0 | -0.013511 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 |      -12.0 | 0.105225 | 0.0 | 0.0 |  6.72 |      -11.2 |  6.650609 |  6.650609 |  0.04209 | 0.021045 | 0.021045 |      0.0 |   -1.8704 | -0.155677 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010155 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.778955 |   7.410609 |   7.410609 |  20.378891 |      0.0 |      0.0 | 0.010155 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00631 |      0.0 | -0.014173 |   0.0 |      0.0 |
    | 2000-01-01 20:00:00 |  2.9 | -11.578947 | 0.047837 | 0.0 | 0.0 |  3.48 | -10.778947 |  3.458955 |  3.458955 | 0.019135 | 0.009567 | 0.009567 |      0.0 | -1.800084 | -0.077923 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010189 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.790433 |  10.869564 |  10.869564 |  20.448702 |      0.0 |      0.0 | 0.010189 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006498 |      0.0 | -0.014828 |   0.0 |      0.0 |
    | 2000-01-01 21:00:00 |  4.9 | -11.157895 | 0.048345 | 0.0 | 0.0 |  5.88 | -10.357895 |  5.870433 |  5.870433 | 0.019338 | 0.009669 | 0.009669 |      0.0 | -1.729768 | -0.127083 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010224 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.790331 |  16.739996 |  16.739996 |  20.518477 |      0.0 |      0.0 | 0.010224 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006679 |      0.0 | -0.015476 |   0.0 |      0.0 |
    | 2000-01-01 22:00:00 | 10.6 | -10.736842 | 0.048848 | 0.0 | 0.0 | 12.72 |  -9.936842 | 12.710331 | 12.710331 | 0.019539 |  0.00977 |  0.00977 |      0.0 | -1.659453 | -0.263968 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010259 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.79023 |  29.450327 |  29.450327 |  20.588218 |      0.0 |      0.0 | 0.010259 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006853 |      0.0 | -0.016118 |   0.0 |      0.0 |
    | 2000-01-01 23:00:00 |  0.1 | -10.315789 |  0.04935 | 0.0 | 0.0 |  0.12 |  -9.515789 |   0.11023 |   0.11023 |  0.01974 |  0.00987 |  0.00987 |      0.0 | -1.589137 | -0.002192 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010294 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.79013 |  29.560558 |  29.560558 |  20.657924 |      0.0 |      0.0 | 0.010294 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00702 |      0.0 | -0.016754 |   0.0 |      0.0 |
    | 2000-01-02 00:00:00 |  0.7 |  -9.894737 |  0.04985 | 0.0 | 0.0 |  0.84 |  -9.094737 |   0.83013 |   0.83013 |  0.01994 |  0.00997 |  0.00997 |      0.0 | -1.518821 | -0.015779 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010329 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.79003 |  30.390688 |  30.390688 |  20.727595 |      0.0 |      0.0 | 0.010329 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00718 |      0.0 | -0.017383 |   0.0 |      0.0 |
    | 2000-01-02 01:00:00 |  3.0 |  -9.473684 |  0.05035 | 0.0 | 0.0 |   3.6 |  -8.673684 |   3.59003 |   3.59003 |  0.02014 |  0.01007 |  0.01007 |      0.0 | -1.448505 |  -0.06508 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010364 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.78993 |  33.980718 |  33.980718 |  20.797231 |      0.0 |      0.0 | 0.010364 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007335 |      0.0 | -0.018006 |   0.0 |      0.0 |
    | 2000-01-02 02:00:00 |  2.1 |  -9.052632 | 0.050845 | 0.0 | 0.0 |  2.52 |  -8.252632 |   2.50993 |   2.50993 | 0.020338 | 0.010169 | 0.010169 |      0.0 | -1.378189 | -0.043291 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010399 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.789831 |  36.490648 |  36.490648 |  20.866833 |      0.0 |      0.0 | 0.010399 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007483 |      0.0 | -0.018623 |   0.0 |      0.0 |
    | 2000-01-02 03:00:00 | 10.4 |  -8.631579 | 0.051342 | 0.0 | 0.0 | 12.48 |  -7.831579 | 12.469831 | 12.469831 | 0.020537 | 0.010268 | 0.010268 |      0.0 | -1.307874 | -0.204106 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010433 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.789732 |  48.960479 |  48.960479 |  20.936399 |      0.0 |      0.0 | 0.010433 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007626 |      0.0 | -0.019234 |   0.0 |      0.0 |
    | 2000-01-02 04:00:00 |  3.5 |  -8.210526 | 0.051835 | 0.0 | 0.0 |   4.2 |  -7.410526 |  4.189732 |  4.189732 | 0.020734 | 0.010367 | 0.010367 |      0.0 | -1.237558 | -0.064891 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010468 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.789633 |   53.15021 |   53.15021 |  21.005931 |      0.0 |      0.0 | 0.010468 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007764 |      0.0 | -0.019839 |   0.0 |      0.0 |
    | 2000-01-02 05:00:00 |  3.4 |  -7.789474 | 0.052327 | 0.0 | 0.0 |  4.08 |  -6.989474 |  4.069633 |  4.069633 | 0.020931 | 0.010465 | 0.010465 |      0.0 | -1.167242 | -0.059449 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010503 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.789535 |  57.219843 |  57.219843 |  21.075428 |      0.0 |      0.0 | 0.010503 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007897 |      0.0 | -0.020437 |   0.0 |      0.0 |
    | 2000-01-02 06:00:00 |  1.2 |  -7.368421 | 0.102702 | 0.0 | 0.0 |  1.44 |  -6.568421 |  1.429535 |  1.429535 | 0.041081 |  0.02054 |  0.02054 |      0.0 | -1.096926 | -0.019625 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010538 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.77946 |  58.649378 |  58.649378 |   21.14489 |      0.0 |      0.0 | 0.010538 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008025 |      0.0 |  -0.02103 |   0.0 |      0.0 |
    | 2000-01-02 07:00:00 |  0.1 |  -6.947368 |  0.69626 | 0.0 | 0.0 |  0.12 |  -6.147368 |   0.09946 |   0.09946 | 0.278504 | 0.139252 | 0.139252 |      0.0 | -1.026611 | -0.001278 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010572 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.660748 |  58.748837 |  58.748837 |  21.214318 |      0.0 |      0.0 | 0.010572 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008148 |      0.0 | -0.021617 |   0.0 |      0.0 |
    | 2000-01-02 08:00:00 |  0.0 |  -6.526316 | 1.692095 | 0.0 | 0.0 |   0.0 |  -5.726316 |       0.0 |       0.0 | 0.676838 | 0.338419 | 0.338419 |      0.0 | -0.956295 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010607 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.322329 |  58.748837 |  58.748837 |  21.283711 |      0.0 |      0.0 | 0.010607 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008267 |      0.0 | -0.022198 |   0.0 |      0.0 |
    | 2000-01-02 09:00:00 |  0.0 |  -6.105263 | 1.078725 | 0.0 | 0.0 |   0.0 |  -5.305263 |       0.0 |       0.0 |  0.43149 | 0.215745 | 0.215745 |      0.0 | -0.885979 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010642 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.106584 |  58.748837 |  58.748837 |  21.353069 |      0.0 |      0.0 | 0.010642 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008382 |      0.0 | -0.022773 |   0.0 |      0.0 |
    | 2000-01-02 10:00:00 |  0.4 |  -5.684211 | 1.382545 | 0.0 | 0.0 |  0.48 |  -4.884211 |       0.0 |       0.0 | 0.553018 | 0.276509 | 0.276509 |      0.0 | -0.815663 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010677 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.310075 |  58.748837 |  58.748837 |  21.422392 |      0.0 |      0.0 | 0.010677 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008493 |      0.0 | -0.023342 |   0.0 |      0.0 |
    | 2000-01-02 11:00:00 |  0.1 |  -5.263158 | 3.516795 | 0.0 | 0.0 |  0.12 |  -4.463158 |       0.0 |       0.0 | 1.406718 | 0.703359 | 0.430075 | 0.072092 | -0.745347 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010711 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  58.748837 |  58.748837 |  21.419589 |      0.0 |      0.0 | 0.010711 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008601 |      0.0 | -0.023906 |   0.0 |      0.0 |
    | 2000-01-02 12:00:00 |  3.6 |  -4.842105 | 1.935692 | 0.0 | 0.0 |  4.32 |  -4.042105 |      3.52 |      3.52 | 0.774277 | 0.387138 | 0.387138 |      0.0 | -0.675032 | -0.029737 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01071 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.412862 |  62.268837 |  62.268837 |  21.488879 |      0.0 |      0.0 |  0.01071 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008703 |      0.0 | -0.024464 |   0.0 |      0.0 |
    | 2000-01-02 13:00:00 |  5.9 |  -4.421053 | 5.120597 | 0.0 | 0.0 |  7.08 |  -3.621053 |  6.692862 |  6.692862 | 2.048239 | 1.024119 |      0.8 | 0.059298 | -0.604716 | -0.050652 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010744 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  68.961699 |  68.961699 |  21.498836 |      0.0 |      0.0 | 0.010744 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008802 |      0.0 | -0.025017 |   0.0 |      0.0 |
    | 2000-01-02 14:00:00 |  1.1 |       -4.0 | 2.372527 | 0.0 | 0.0 |  1.32 |       -3.2 |      0.52 |      0.52 | 0.949011 | 0.474505 | 0.474505 |      0.0 |   -0.5344 | -0.003478 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010749 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.325495 |  69.481699 |  69.481699 |  21.568087 |      0.0 |      0.0 | 0.010749 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008897 |      0.0 | -0.025564 |   0.0 |      0.0 |
    | 2000-01-02 15:00:00 | 20.7 |  -3.578947 | 1.267695 | 0.0 | 0.0 | 24.84 |  -2.778947 | 24.365495 | 24.365495 | 0.507078 | 0.253539 | 0.253539 |      0.0 | -0.464084 | -0.141515 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010784 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.546461 |  93.847194 |  93.847194 |  21.637303 |      0.0 |      0.0 | 0.010784 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008988 |      0.0 | -0.026105 |   0.0 |      0.0 |
    | 2000-01-02 16:00:00 | 37.9 |  -3.157895 | 1.365632 | 0.0 | 0.0 | 45.48 |  -2.357895 | 45.226461 | 45.226461 | 0.546253 | 0.273126 | 0.273126 |      0.0 | -0.393768 | -0.222876 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010819 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.526874 | 139.073655 | 139.073655 |  21.706484 |      0.0 |      0.0 | 0.010819 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009077 |      0.0 | -0.026642 |   0.0 |      0.0 |
    | 2000-01-02 17:00:00 |  8.2 |  -2.736842 | 1.772603 | 0.0 | 0.0 |  9.84 |  -1.936842 |  9.566874 |  9.566874 | 0.709041 | 0.354521 | 0.354521 |      0.0 | -0.323453 | -0.038727 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010853 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.445479 | 148.640528 | 148.640528 |  21.775631 |      0.0 |      0.0 | 0.010853 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009162 |      0.0 | -0.027173 |   0.0 |      0.0 |
    | 2000-01-02 18:00:00 |  3.6 |  -2.315789 | 0.997112 | 0.0 | 0.0 |  4.32 |  -1.515789 |  3.965479 |  3.965479 | 0.398845 | 0.199422 | 0.199422 |      0.0 | -0.253137 | -0.012563 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010888 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.600578 | 152.606008 | 152.606008 |  21.844743 |      0.0 |      0.0 | 0.010888 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009246 |      0.0 | -0.027698 |   0.0 |      0.0 |
    | 2000-01-02 19:00:00 |  7.5 |  -1.894737 | 0.185172 | 0.0 | 0.0 |   9.0 |  -1.094737 |  8.800578 |  8.800578 | 0.074069 | 0.037034 | 0.037034 |      0.0 | -0.182821 | -0.020136 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010922 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.762966 | 161.406585 | 161.406585 |  21.913821 |      0.0 |      0.0 | 0.010922 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009327 |      0.0 | -0.028219 |   0.0 |      0.0 |
    | 2000-01-02 20:00:00 | 18.5 |  -1.473684 |   0.0595 | 0.0 | 0.0 |  22.2 |  -0.673684 | 22.162966 | 22.162966 |   0.0238 |   0.0119 |   0.0119 |      0.0 | -0.112505 | -0.031205 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010957 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |   0.7881 | 183.569551 | 183.569551 |  21.982864 |      0.0 |      0.0 | 0.010957 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009405 |      0.0 | -0.028734 |   0.0 |      0.0 |
    | 2000-01-02 21:00:00 | 15.4 |  -1.052632 | 0.059965 | 0.0 | 0.0 | 18.48 |  -0.252632 |   18.4681 |   18.4681 | 0.023986 | 0.011993 | 0.011993 |      0.0 | -0.042189 | -0.009751 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010991 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.788007 | 202.037651 | 202.037651 |  22.051873 |      0.0 |      0.0 | 0.010991 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009482 |      0.0 | -0.029244 |   0.0 |      0.0 |
    | 2000-01-02 22:00:00 |  6.3 |  -0.631579 | 0.060428 | 0.0 | 0.0 |  7.56 |   0.168421 |  7.548007 |  6.912385 | 0.024171 | 0.012086 | 0.012086 |      0.0 |  0.028126 |  0.002881 |  0.092837 | 0.092837 |       0.0 |      0.0 | 0.011026 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787914 | 208.857199 | 209.585658 |  22.120847 |      0.0 |      0.0 | 0.011026 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009556 |      0.0 | -0.029749 |   0.0 |      0.0 |
    | 2000-01-02 23:00:00 |  1.9 |  -0.210526 |  0.06089 | 0.0 | 0.0 |  2.28 |   0.589474 |  2.267914 |  1.599476 | 0.024356 | 0.012178 | 0.012178 |      0.0 |  0.098442 |   0.00362 |  0.305576 | 0.305576 |       0.0 |      0.0 |  0.01106 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787822 | 210.151099 | 211.853572 |  22.189786 |      0.0 |      0.0 |  0.01106 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009629 |      0.0 | -0.030249 |   0.0 |      0.0 |
    | 2000-01-03 00:00:00 |  4.9 |   0.210526 |  0.06135 | 0.0 | 0.0 |  5.88 |   1.010526 |  5.867822 |  2.903028 |  0.02454 |  0.01227 |  0.01227 |      0.0 |  0.168758 |  0.018675 |  0.561176 | 0.561176 |       0.0 |      0.0 | 0.011095 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.78773 | 212.492951 | 217.721394 |  22.258691 |      0.0 |      0.0 | 0.011095 |      0.0 |    -0.08 |      0.0 |      0.0 |   0.0097 |      0.0 | -0.030744 |   0.0 |      0.0 |
    | 2000-01-03 01:00:00 |  2.7 |   0.631579 |  0.06181 | 0.0 | 0.0 |  3.24 |   1.431579 |   3.22773 |  0.917355 | 0.024724 | 0.012362 | 0.012362 |      0.0 |  0.239074 |  0.016593 |  0.765468 | 0.765468 |       0.0 |      0.0 | 0.011129 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787638 | 212.644838 | 220.949124 |  22.327562 |      0.0 |      0.0 | 0.011129 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009768 |      0.0 | -0.031234 |   0.0 |      0.0 |
    | 2000-01-03 02:00:00 |  0.5 |   1.052632 | 0.062267 | 0.0 | 0.0 |   0.6 |   1.852632 |  0.587638 |    0.0433 | 0.024907 | 0.012453 | 0.012453 |      0.0 |  0.309389 |   0.00439 |  0.939459 | 0.939459 |       0.0 |      0.0 | 0.011164 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787547 | 211.748679 | 221.536762 |  22.396398 |      0.0 |      0.0 | 0.011164 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009836 |      0.0 | -0.031719 |   0.0 |      0.0 |
    | 2000-01-03 03:00:00 |  0.2 |   1.473684 | 0.062723 | 0.0 | 0.0 |  0.24 |   2.273684 |  0.227547 |       0.0 | 0.025089 | 0.012545 | 0.012545 |      0.0 |  0.379705 |  0.002166 |  1.143327 | 1.143327 |       0.0 |      0.0 | 0.011198 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787455 | 210.605351 | 221.764309 |    22.4652 |      0.0 |      0.0 | 0.011198 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009901 |      0.0 |   -0.0322 |   0.0 |      0.0 |
    | 2000-01-03 04:00:00 |  0.5 |   1.894737 | 0.063177 | 0.0 | 0.0 |   0.6 |   2.694737 |  0.587455 |       0.0 | 0.025271 | 0.012635 | 0.012635 |      0.0 |  0.450021 |  0.006628 |  1.367212 | 1.367212 |       0.0 |      0.0 | 0.011233 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787365 | 209.238139 | 222.351764 |  22.533967 |      0.0 |      0.0 | 0.011233 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009965 |      0.0 | -0.032675 |   0.0 |      0.0 |
    | 2000-01-03 05:00:00 |  2.4 |   2.315789 |  0.06363 | 0.0 | 0.0 |  2.88 |   3.115789 |  2.867365 |       0.0 | 0.025452 | 0.012726 | 0.012726 |      0.0 |  0.520337 |  0.037405 |  1.669887 | 1.669887 |       0.0 |      0.0 | 0.011267 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787274 | 207.568252 | 225.219129 |    22.6027 |      0.0 |      0.0 | 0.011267 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010028 |      0.0 | -0.033146 |   0.0 |      0.0 |
    | 2000-01-03 06:00:00 |  0.4 |   2.736842 |  0.10774 | 0.0 | 0.0 |  0.48 |   3.536842 |  0.467274 |       0.0 | 0.043096 | 0.021548 | 0.021548 |      0.0 |  0.590653 |  0.006919 |  1.789138 | 1.789138 |       0.0 |      0.0 | 0.011301 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.778452 | 205.779115 | 225.686403 |  22.671399 |      0.0 |      0.0 | 0.011301 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010089 |      0.0 | -0.033612 |   0.0 |      0.0 |
    | 2000-01-03 07:00:00 |  0.2 |   3.157895 | 0.325317 | 0.0 | 0.0 |  0.24 |   3.957895 |  0.218452 |       0.0 | 0.130127 | 0.065063 | 0.065063 |      0.0 |  0.660968 |   0.00362 |  1.989786 | 1.989786 |       0.0 |      0.0 | 0.011336 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.734937 | 203.789329 | 225.904855 |  22.740063 |      0.0 |      0.0 | 0.011336 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010149 |      0.0 | -0.034074 |   0.0 |      0.0 |
    | 2000-01-03 08:00:00 |  0.0 |   3.578947 | 0.811522 | 0.0 | 0.0 |   0.0 |   4.378947 |       0.0 |       0.0 | 0.324609 | 0.162304 | 0.162304 |      0.0 |  0.731284 |       0.0 |  2.189474 | 2.189474 |       0.0 |      0.0 |  0.01137 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.572632 | 201.599855 | 225.904855 |  22.808693 |      0.0 |      0.0 |  0.01137 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010208 |      0.0 | -0.034531 |   0.0 |      0.0 |
    | 2000-01-03 09:00:00 |  0.0 |        4.0 | 2.973865 | 0.0 | 0.0 |   0.0 |        4.8 |       0.0 |       0.0 | 1.189546 | 0.594773 | 0.572632 |   0.0062 |    0.8016 |       0.0 |       2.4 |      2.4 |       0.0 |      0.0 | 0.011404 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 199.199855 | 225.904855 |  22.871089 |      0.0 |      0.0 | 0.011404 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010265 |      0.0 | -0.034983 |   0.0 |      0.0 |
    | 2000-01-03 10:00:00 |  0.3 |   4.421053 |  3.12363 | 0.0 | 0.0 |  0.36 |   5.221053 |       0.0 |       0.0 | 1.249452 | 0.624726 |     0.36 | 0.074326 |  0.871916 |       0.0 |  2.610526 | 2.610526 |       0.0 |      0.0 | 0.011436 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 196.589329 | 225.904855 |  22.865327 |      0.0 |      0.0 | 0.011436 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010322 |      0.0 | -0.035431 |   0.0 |      0.0 |
    | 2000-01-03 11:00:00 |  2.6 |   4.842105 | 3.798742 | 0.0 | 0.0 |  3.12 |   5.642105 |      2.32 |       0.0 | 1.519497 | 0.759748 | 0.759748 |      0.0 |  0.942232 |  0.054804 |  2.985136 | 2.985136 |       0.0 |      0.0 | 0.011433 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.040252 | 193.604193 | 228.224855 |  22.933895 |      0.0 |      0.0 | 0.011433 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010376 |      0.0 | -0.035875 |   0.0 |      0.0 |
    | 2000-01-03 12:00:00 |  0.7 |   5.263158 | 3.874722 | 0.0 | 0.0 |  0.84 |   6.063158 |  0.080252 |       0.0 | 1.549889 | 0.774944 | 0.774944 |      0.0 |  1.012547 |  0.002037 |  3.037678 | 3.037678 |       0.0 |      0.0 | 0.011467 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.025056 | 190.566514 | 228.305106 |  23.002428 |      0.0 |      0.0 | 0.011467 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010428 |      0.0 | -0.036314 |   0.0 |      0.0 |
    | 2000-01-03 13:00:00 |  0.3 |   5.684211 | 3.373822 | 0.0 | 0.0 |  0.36 |   6.484211 |       0.0 |       0.0 | 1.349529 | 0.674764 | 0.385056 | 0.081783 |  1.082863 |       0.0 |  3.242105 | 3.242105 |       0.0 |      0.0 | 0.011501 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 187.324409 | 228.305106 |  22.989143 |      0.0 |      0.0 | 0.011501 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.01048 |      0.0 | -0.036749 |   0.0 |      0.0 |
    | 2000-01-03 14:00:00 |  0.3 |   6.105263 | 5.667417 | 0.0 | 0.0 |  0.36 |   6.905263 |       0.0 |       0.0 | 2.266967 | 1.133483 |     0.36 | 0.218231 |  1.153179 |       0.0 |  3.452632 | 3.452632 |       0.0 |      0.0 | 0.011495 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 183.871778 | 228.305106 |  22.839418 |      0.0 |      0.0 | 0.011495 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.01053 |      0.0 | -0.037179 |   0.0 |      0.0 |
    | 2000-01-03 15:00:00 |  0.0 |   6.526316 | 3.434725 | 0.0 | 0.0 |   0.0 |   7.326316 |       0.0 |       0.0 |  1.37389 | 0.686945 |      0.0 | 0.192616 |  1.223495 |       0.0 |  3.663158 | 3.663158 |       0.0 |      0.0 |  0.01142 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  180.20862 | 228.305106 |  22.715382 |      0.0 |      0.0 |  0.01142 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010575 |      0.0 | -0.037605 |   0.0 |      0.0 |
    | 2000-01-03 16:00:00 |  0.0 |   6.947368 | 3.787713 | 0.0 | 0.0 |   0.0 |   7.747368 |       0.0 |       0.0 | 1.515085 | 0.757543 |      0.0 | 0.211316 |  1.293811 |       0.0 |  3.873684 | 3.873684 |       0.0 |      0.0 | 0.011358 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 176.334936 | 228.305106 |  22.572709 |      0.0 |      0.0 | 0.011358 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010614 |      0.0 | -0.038027 |   0.0 |      0.0 |
    | 2000-01-03 17:00:00 |  0.0 |   7.368421 | 2.368437 | 0.0 | 0.0 |   0.0 |   8.168421 |       0.0 |       0.0 | 0.947375 | 0.473687 |      0.0 | 0.131346 |  1.364126 |       0.0 |  4.084211 | 4.084211 |       0.0 |      0.0 | 0.011286 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 172.250725 | 228.305106 |  22.510076 |      0.0 |      0.0 | 0.011286 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010649 |      0.0 | -0.038444 |   0.0 |      0.0 |
    | 2000-01-03 18:00:00 |  0.0 |   7.789474 | 1.120002 | 0.0 | 0.0 |   0.0 |   8.589474 |       0.0 |       0.0 | 0.448001 |    0.224 |      0.0 | 0.061948 |  1.434442 |       0.0 |  4.294737 | 4.294737 |       0.0 |      0.0 | 0.011255 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 167.955988 | 228.305106 |  22.516874 |      0.0 |      0.0 | 0.011255 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010679 |      0.0 | -0.038858 |   0.0 |      0.0 |
    | 2000-01-03 19:00:00 |  0.0 |   8.210526 | 0.211415 | 0.0 | 0.0 |   0.0 |   9.010526 |       0.0 |       0.0 | 0.084566 | 0.042283 |      0.0 | 0.011697 |  1.504758 |       0.0 |  4.505263 | 4.505263 |       0.0 |      0.0 | 0.011258 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 163.450725 | 228.305106 |  22.573918 |      0.0 |      0.0 | 0.011258 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010707 |      0.0 | -0.039267 |   0.0 |      0.0 |
    | 2000-01-03 20:00:00 |  0.0 |   8.631579 | 0.070242 | 0.0 | 0.0 |   0.0 |   9.431579 |       0.0 |       0.0 | 0.028097 | 0.014048 |      0.0 | 0.003896 |  1.575074 |       0.0 |  4.715789 | 4.715789 |  6.076197 | 0.232292 | 0.011287 |      0.0 |      0.0 | 0.08 | 0.232292 |      0.0 |      0.0 |      0.0 | 158.734936 |  222.22891 |  28.482641 | 0.232292 |      0.0 | 0.011287 |      0.0 |    -0.08 | 0.021755 |      0.0 | 0.010735 |      0.0 | -0.039673 |   0.0 |      0.0 |
    | 2000-01-03 21:00:00 |  0.0 |   9.052632 |  0.07067 | 0.0 | 0.0 |   0.0 |   9.852632 |       0.0 |       0.0 | 0.028268 | 0.014134 |      0.0 | 0.004874 |  1.645389 |       0.0 |  4.926316 | 4.926316 |  6.896842 |  0.33268 | 0.014241 |      0.0 |      0.0 | 0.08 |  0.33268 | 0.040081 | 0.011134 |      0.0 |  153.80862 | 215.332068 |  35.107688 |  0.33268 |      0.0 | 0.014241 |      0.0 |    -0.08 | 0.069321 |      0.0 | 0.010835 |      0.0 | -0.040074 |   0.0 | 0.011134 |
    | 2000-01-03 22:00:00 |  0.0 |   9.473684 |   0.0711 | 0.0 | 0.0 |   0.0 |  10.273684 |       0.0 |       0.0 |  0.02844 |  0.01422 |      0.0 | 0.005925 |  1.715705 |       0.0 |  5.136842 | 5.136842 |  7.191579 | 0.426244 | 0.017554 |      0.0 |      0.0 | 0.08 | 0.426244 | 0.096433 | 0.026787 |      0.0 | 148.671778 | 208.140489 |  41.929544 | 0.426244 |      0.0 | 0.017554 |      0.0 |    -0.08 | 0.125822 |      0.0 | 0.011082 |      0.0 | -0.040471 |   0.0 | 0.026787 |
    | 2000-01-03 23:00:00 |  0.0 |   9.894737 | 0.071525 | 0.0 | 0.0 |   0.0 |  10.694737 |       0.0 |       0.0 |  0.02861 | 0.014305 |      0.0 | 0.006948 |  1.786021 |       0.0 |  5.347368 | 5.347368 |  7.486316 | 0.531265 | 0.020965 |      0.0 |      0.0 | 0.08 | 0.531265 | 0.160732 | 0.044648 |      0.0 | 143.324409 | 200.654173 |  48.936682 | 0.531265 |      0.0 | 0.020965 |      0.0 |    -0.08 | 0.190115 |      0.0 | 0.011482 |      0.0 | -0.040864 |   0.0 | 0.044648 |
    | 2000-01-04 00:00:00 |  0.0 |  10.315789 |  0.07195 | 0.0 | 0.0 |   0.0 |  11.115789 |       0.0 |       0.0 |  0.02878 |  0.01439 |      0.0 |  0.00793 |  1.856337 |       0.0 |  5.557895 | 5.557895 |  7.781053 | 0.648613 | 0.024468 |      0.0 |      0.0 | 0.08 | 0.648613 | 0.233722 | 0.064923 |      0.0 | 137.766514 |  192.87312 |  56.116723 | 0.648613 |      0.0 | 0.024468 |      0.0 |    -0.08 | 0.262945 |      0.0 |  0.01203 |      0.0 | -0.041254 |   0.0 | 0.064923 |
    | 2000-01-04 01:00:00 |  0.0 |  10.736842 | 0.072375 | 0.0 | 0.0 |   0.0 |  11.536842 |       0.0 |       0.0 |  0.02895 | 0.014475 |      0.0 | 0.008857 |  1.926653 |       0.0 |  5.768421 | 5.768421 |  8.075789 | 0.779242 | 0.028058 |      0.0 |      0.0 | 0.08 | 0.779242 | 0.316175 | 0.087826 |      0.0 | 131.998093 | 184.797331 |  63.456354 | 0.779242 |      0.0 | 0.028058 |      0.0 |    -0.08 | 0.345089 |      0.0 | 0.012725 |      0.0 | -0.041639 |   0.0 | 0.087826 |
    | 2000-01-04 02:00:00 |  0.0 |  11.157895 | 0.072795 | 0.0 | 0.0 |   0.0 |  11.957895 |       0.0 |       0.0 | 0.029118 | 0.014559 |      0.0 | 0.009721 |  1.996968 |       0.0 |  5.978947 | 5.978947 |  8.370526 | 0.924203 | 0.031728 |      0.0 |      0.0 | 0.08 | 0.924203 | 0.408906 | 0.113585 |      0.0 | 126.019146 | 176.426804 |  70.941229 | 0.924203 |      0.0 | 0.031728 |      0.0 |    -0.08 | 0.437364 |      0.0 | 0.013563 |      0.0 | -0.042021 |   0.0 | 0.113585 |
    | 2000-01-04 03:00:00 |  1.3 |  11.578947 | 0.073217 | 0.0 | 0.0 |  1.56 |  12.378947 |      0.76 |       0.0 | 0.029287 | 0.014643 | 0.014643 |      0.0 |  2.067284 |  0.039389 |  6.307406 | 6.307406 |  9.590368 | 1.208247 | 0.035471 |      0.0 |      0.0 | 0.08 | 1.208247 | 0.528642 | 0.146845 | 0.785357 |  119.71174 | 167.596436 |  79.367879 | 1.172355 | 0.035892 | 0.035471 |      0.0 |    -0.08 | 0.548853 | 0.007647 | 0.014541 |      0.0 | -0.042399 |   0.0 | 0.146845 |
    | 2000-01-04 04:00:00 |  0.0 |       12.0 | 0.073637 | 0.0 | 0.0 |   0.0 |       12.8 |       0.0 |       0.0 | 0.029455 | 0.014727 | 0.014727 |      0.0 |    2.1376 |       0.0 |       6.4 |      6.4 |      8.96 | 1.277013 | 0.039684 |      0.0 |      0.0 | 0.08 | 1.277013 | 0.662858 | 0.184127 | 0.770629 |  113.31174 | 158.636436 |  87.091182 | 1.216923 |  0.06009 | 0.039684 |      0.0 |    -0.08 | 0.666049 | 0.023917 | 0.015666 |      0.0 | -0.042773 |   0.0 | 0.184127 |
    | 2000-01-04 05:00:00 |  0.0 |  12.421053 | 0.074055 | 0.0 | 0.0 |   0.0 |  13.221053 |       0.0 |       0.0 | 0.029622 | 0.014811 | 0.014811 |      0.0 |  2.207916 |       0.0 |  6.610526 | 6.610526 |  9.254737 | 1.473731 | 0.043546 |      0.0 |      0.0 | 0.08 | 1.473731 | 0.807275 | 0.224243 | 0.755818 | 106.701214 | 149.381699 |  94.908643 |  1.32145 | 0.152281 | 0.043546 |      0.0 |    -0.08 | 0.775695 | 0.057792 | 0.016932 |      0.0 | -0.043144 |   0.0 | 0.224243 |
    | 2000-01-04 06:00:00 |  0.0 |  12.842105 |   0.2705 | 0.0 | 0.0 |   0.0 |  13.642105 |       0.0 |       0.0 |   0.1082 |   0.0541 |   0.0541 |      0.0 |  2.278232 |       0.0 |  6.821053 | 6.821053 |  9.549474 | 1.690531 | 0.047454 |      0.0 |      0.0 | 0.08 | 1.690531 |  0.98021 | 0.272281 | 0.701718 |  99.880161 | 139.832225 | 102.800131 |  1.40847 | 0.282061 | 0.047454 |      0.0 |    -0.08 | 0.882773 | 0.122622 | 0.018326 |      0.0 |  -0.04351 |   0.0 | 0.272281 |
    | 2000-01-04 07:00:00 |  0.7 |  13.263158 |  1.23095 | 0.0 | 0.0 |  0.84 |  14.063158 |  0.741718 |       0.0 |  0.49238 |  0.24619 |  0.24619 |      0.0 |  2.348547 |  0.043672 |  7.162334 | 7.162334 | 10.768985 | 2.120548 |   0.0514 | 0.004217 | 0.464001 |  0.0 | 2.120548 | 1.219604 | 0.338779 |  0.55381 |  92.717827 | 129.804958 | 110.928951 | 1.528424 | 0.592124 |   0.0514 | 0.004217 | 0.464001 |   0.9893 | 0.251419 | 0.019844 | 0.000204 | -0.041162 |   0.0 | 0.338779 |
    | 2000-01-04 08:00:00 |  0.4 |  13.684211 | 2.866595 | 0.0 | 0.0 |  0.48 |  14.484211 |   0.23381 |       0.0 | 1.146638 | 0.573319 | 0.573319 |      0.0 |  2.418863 |  0.014179 |  7.284557 | 7.284557 |  10.43219 | 2.264101 | 0.055464 | 0.032517 | 0.504645 |  0.0 | 2.264101 | 1.487021 | 0.413062 | 0.226681 |   85.43327 | 119.606579 | 118.504413 | 1.558324 | 0.705777 | 0.055464 | 0.032517 | 0.504645 | 1.089827 | 0.409691 | 0.021483 | 0.001955 | -0.035933 |   0.0 | 0.413062 |
    | 2000-01-04 09:00:00 |  0.1 |  14.105263 |  4.66193 | 0.0 | 0.0 |  0.12 |  14.905263 |       0.0 |       0.0 | 1.864772 | 0.932386 | 0.346681 | 0.534993 |  2.489179 |       0.0 |  7.452632 | 7.452632 | 10.433684 | 2.476523 | 0.059252 |  0.07164 | 0.542522 |  0.0 | 2.476523 | 1.741301 | 0.483695 |      0.0 |  77.980639 | 109.172894 | 125.253168 | 1.596208 | 0.880315 | 0.059252 |  0.07164 | 0.542522 | 1.178299 | 0.563379 | 0.023233 | 0.006756 | -0.030366 |   0.0 | 0.483695 |
    | 2000-01-04 10:00:00 |  0.4 |  14.526316 | 6.271463 | 0.0 | 0.0 |  0.48 |  15.326316 |       0.0 |       0.0 | 2.508585 | 1.254293 |     0.48 | 0.718791 |  2.559495 |       0.0 |  7.663158 | 7.663158 | 10.728421 | 2.757158 | 0.062627 | 0.114213 | 0.576266 |  0.0 | 2.757158 | 2.012586 | 0.559052 |      0.0 |  70.317481 |  98.444473 | 131.752534 | 1.637308 |  1.11985 | 0.062627 | 0.114213 | 0.576266 | 1.257902 | 0.739119 | 0.025073 |  0.01499 | -0.024497 |   0.0 | 0.559052 |
    | 2000-01-04 11:00:00 |  0.0 |  14.947368 | 6.833792 | 0.0 | 0.0 |   0.0 |  15.747368 |       0.0 |       0.0 | 2.733517 | 1.366758 |      0.0 | 1.285647 |  2.629811 |       0.0 |  7.873684 | 7.873684 | 11.023158 | 3.055547 | 0.065876 | 0.161031 | 0.608763 |  0.0 | 3.055547 | 2.310271 | 0.641742 |      0.0 |  62.443797 |  87.421316 | 137.598827 | 1.672726 |  1.38282 | 0.065876 | 0.161031 | 0.608763 | 1.329994 | 0.944954 | 0.026984 | 0.026697 | -0.018358 |   0.0 | 0.641742 |
    | 2000-01-04 12:00:00 |  0.0 |  15.368421 | 6.738053 | 0.0 | 0.0 |   0.0 |  16.168421 |       0.0 |       0.0 | 2.695221 | 1.347611 |      0.0 | 1.280593 |  2.700126 |       0.0 |  8.084211 | 8.084211 | 11.317895 | 3.357424 | 0.068799 | 0.207493 | 0.637994 |  0.0 | 3.357424 | 2.628875 | 0.730243 |      0.0 |  54.359586 |  76.103421 | 143.364418 | 1.702153 | 1.655271 | 0.068799 | 0.207493 | 0.637994 | 1.394876 |  1.17529 | 0.028953 | 0.041728 | -0.011972 |   0.0 | 0.730243 |
    | 2000-01-04 13:00:00 |  0.0 |  15.789474 | 8.078045 | 0.0 | 0.0 |   0.0 |  16.589474 |       0.0 |       0.0 | 3.231218 | 1.615609 |      0.0 | 1.548635 |  2.770442 |       0.0 |  8.294737 | 8.294737 | 11.612632 | 3.683597 | 0.071682 | 0.257006 | 0.666822 |  0.0 | 3.683597 | 2.966573 | 0.824048 |      0.0 |  46.064849 |  64.490789 | 148.749308 | 1.728526 | 1.955071 | 0.071682 | 0.257006 | 0.666822 | 1.453046 | 1.428024 | 0.030967 | 0.059898 | -0.005361 |   0.0 | 0.824048 |
    | 2000-01-04 14:00:00 |  0.0 |  16.210526 | 7.222285 | 0.0 | 0.0 |   0.0 |  17.010526 |       0.0 |       0.0 | 2.888914 | 1.444457 |      0.0 | 1.394364 |  2.840758 |       0.0 |  8.505263 | 8.505263 | 11.907368 | 4.023341 | 0.074375 | 0.306334 | 0.693747 |  0.0 | 4.023341 | 3.323553 | 0.923209 |      0.0 |  37.559586 |  52.583421 | 154.164516 |  1.75145 | 2.271891 | 0.074375 | 0.306334 | 0.693747 | 1.505129 | 1.702903 | 0.033019 | 0.081041 |  0.001461 |   0.0 | 0.923209 |
    | 2000-01-04 15:00:00 |  0.0 |  16.631579 | 6.044787 | 0.0 | 0.0 |   0.0 |  17.431579 |       0.0 |       0.0 | 2.417915 | 1.208957 |      0.0 | 1.174275 |  2.911074 |       0.0 |  8.715789 | 8.715789 | 12.202105 | 4.397436 | 0.077082 | 0.358769 | 0.720823 |  0.0 | 4.397436 | 3.702346 |  1.02843 |      0.0 |  28.843797 |  40.381316 | 159.638236 | 1.772595 | 2.624841 | 0.077082 | 0.358769 | 0.720823 |  1.55176 | 2.001982 | 0.035102 | 0.105017 |  0.008485 |   0.0 |  1.02843 |
    | 2000-01-04 16:00:00 |  0.0 |  17.052632 | 4.837587 | 0.0 | 0.0 |   0.0 |  17.852632 |       0.0 |       0.0 | 1.935035 | 0.967517 |      0.0 | 0.944905 |  2.981389 |       0.0 |  8.926316 | 8.926316 | 12.496842 | 4.814267 | 0.079819 | 0.414505 | 0.748191 |  0.0 | 4.814267 | 4.110095 | 1.141693 |      0.0 |  19.917481 |  27.884473 | 165.133391 | 1.792284 | 3.021983 | 0.079819 | 0.414505 | 0.748191 | 1.593634 | 2.331674 | 0.037217 | 0.131861 |  0.015709 |   0.0 | 1.141693 |
    | 2000-01-04 17:00:00 |  0.0 |  17.473684 | 2.353282 | 0.0 | 0.0 |   0.0 |  18.273684 |       0.0 |       0.0 | 0.941313 | 0.470656 |      0.0 | 0.461858 |  3.051705 |       0.0 |  9.136842 | 9.136842 | 12.791579 | 5.281461 | 0.082567 | 0.473095 | 0.775667 |  0.0 | 5.281461 | 4.554369 | 1.265102 |      0.0 |  10.780639 |  15.092894 | 170.850323 | 1.810658 | 3.470803 | 0.082567 | 0.473095 | 0.775667 | 1.631364 | 2.698915 | 0.039362 | 0.161592 |  0.023134 |   0.0 | 1.265102 |
    | 2000-01-04 18:00:00 |  0.0 |  17.894737 | 1.092032 | 0.0 | 0.0 |   0.0 |  18.694737 |       0.0 |       0.0 | 0.436813 | 0.218406 |      0.0 | 0.215251 |  3.122021 |       0.0 |  9.347368 | 9.347368 | 13.086316 | 5.828459 | 0.085425 | 0.536729 | 0.804252 |  0.0 | 5.828459 | 5.047535 | 1.402093 |      0.0 |    1.43327 |   2.006579 | 176.466522 | 1.828428 | 4.000031 | 0.085425 | 0.536729 | 0.804252 | 1.665529 | 3.115387 |  0.04154 | 0.194314 |  0.030764 |   0.0 | 1.402093 |
    | 2000-01-04 19:00:00 |  0.0 |  18.315789 | 0.289745 | 0.0 | 0.0 |   0.0 |  19.115789 |       0.0 |       0.0 | 0.115898 | 0.057949 |      0.0 | 0.057323 |  3.192337 |       0.0 |  9.557895 |  1.43327 |  2.006579 |  0.92507 | 0.088233 | 0.601796 | 0.832333 |  0.0 |  0.92507 | 4.534067 | 1.259463 |      0.0 |        0.0 |        0.0 | 175.968346 |  0.92507 |      0.0 | 0.088233 | 0.601796 | 0.832333 | 1.610455 | 2.611216 | 0.043749 | 0.230047 |  0.038601 |   0.0 | 1.259463 |
    | 2000-01-04 20:00:00 |  0.0 |  18.736842 | 0.080167 | 0.0 | 0.0 |   0.0 |  19.536842 |       0.0 |       0.0 | 0.032067 | 0.016033 |      0.0 | 0.015855 |  3.262653 |       0.0 |  9.768421 |      0.0 |       0.0 |      0.0 | 0.087984 | 0.595925 | 0.829842 |  0.0 |      0.0 |   3.3409 | 0.928028 |      0.0 |        0.0 |        0.0 |  174.43874 |      0.0 |      0.0 | 0.087984 | 0.595925 | 0.829842 | 1.399579 | 1.583782 | 0.045912 | 0.265139 |  0.046486 |   0.0 | 0.928028 |
    | 2000-01-04 21:00:00 |  0.0 |  19.157895 | 0.080565 | 0.0 | 0.0 |   0.0 |  19.957895 |       0.0 |       0.0 | 0.032226 | 0.016113 |      0.0 | 0.015919 |  3.332968 |       0.0 |  9.978947 |      0.0 |       0.0 |      0.0 | 0.087219 | 0.578018 | 0.822194 |  0.0 |      0.0 | 2.504431 | 0.695675 |      0.0 |        0.0 |        0.0 | 172.935391 |      0.0 |      0.0 | 0.087219 | 0.578018 | 0.822194 | 1.145879 | 0.960613 | 0.047946 | 0.295752 |  0.054242 |   0.0 | 0.695675 |
    | 2000-01-04 22:00:00 |  0.0 |  19.578947 | 0.080962 | 0.0 | 0.0 |   0.0 |  20.378947 |       0.0 |       0.0 | 0.032385 | 0.016192 |      0.0 | 0.015981 |  3.403284 |       0.0 | 10.189474 |      0.0 |       0.0 |      0.0 | 0.086468 | 0.560596 | 0.814677 |  0.0 |      0.0 | 1.954266 | 0.542852 |      0.0 |        0.0 |        0.0 | 171.457669 |      0.0 |      0.0 | 0.086468 | 0.560596 | 0.814677 | 0.938166 | 0.582641 | 0.049842 |  0.32177 |  0.061846 |   0.0 | 0.542852 |
    | 2000-01-04 23:00:00 |  0.0 |       20.0 | 0.081357 | 0.0 | 0.0 |   0.0 |       20.8 |       0.0 |       0.0 | 0.032543 | 0.016271 |      0.0 | 0.016043 |    3.4736 |       0.0 |      10.4 |      0.0 |       0.0 |      0.0 | 0.085729 | 0.543645 | 0.807288 |  0.0 |      0.0 | 1.586083 | 0.440579 |      0.0 |        0.0 |        0.0 | 170.004963 |      0.0 |      0.0 | 0.085729 | 0.543645 | 0.807288 | 0.768105 |  0.35339 | 0.051611 | 0.343677 |    0.0693 |   0.0 | 0.440579 |
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
# ...from lland
from hydpy.models.lland import lland_model
from hydpy.models.lland import lland_masks
from hydpy.models.lland.lland_constants import *


class Model(modeltools.AdHocModel):
    """External PET/degree-day version of HydPy-L-Land."""
    INLET_METHODS = (
        lland_model.Pick_QZ_V1,
    )
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        lland_model.Calc_QZH_V1,
        lland_model.Calc_NKor_V1,
        lland_model.Calc_TKor_V1,
        lland_model.Calc_ET0_WET0_V1,
        lland_model.Calc_EvPo_V1,
        lland_model.Calc_NBes_Inzp_V1,
        lland_model.Calc_EvI_Inzp_V1,
        lland_model.Calc_SNRatio_V1,
        lland_model.Calc_SBes_V1,
        lland_model.Calc_WATS_V1,
        lland_model.Calc_WGTF_V1,
        lland_model.Calc_WNied_V1,
        lland_model.Calc_SchmPot_V1,
        lland_model.Calc_Schm_WATS_V1,
        lland_model.Calc_WaDa_WAeS_V1,
        lland_model.Calc_EvB_V1,
        lland_model.Calc_QKap_V1,
        lland_model.Calc_QBB_V1,
        lland_model.Calc_QIB1_V1,
        lland_model.Calc_QIB2_V1,
        lland_model.Calc_QDB_V1,
        lland_model.Calc_BoWa_V1,
        lland_model.Calc_QBGZ_V1,
        lland_model.Calc_QIGZ1_V1,
        lland_model.Calc_QIGZ2_V1,
        lland_model.Calc_QDGZ_V1,
        lland_model.Calc_QDGZ1_QDGZ2_V1,
        lland_model.Calc_QBGA_V1,
        lland_model.Calc_QIGA1_V1,
        lland_model.Calc_QIGA2_V1,
        lland_model.Calc_QDGA1_V1,
        lland_model.Calc_QDGA2_V1,
        lland_model.Calc_QAH_V1,
        lland_model.Calc_QA_V1,
    )
    ADD_METHODS = ()
    OUTLET_METHODS = (
        lland_model.Pass_QA_V1,
    )
    SENDER_METHODS = ()
    SUBMODELS = ()


class Masks(masktools.Masks):
    """Masks applicable to |lland_v2|."""
    # pylint: disable=no-member
    # bug of pylint 2.4?
    CLASSES = lland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
cythonizer.finalise()
