from propylean.equipments.generic_equipment_classes import _PressureChangers, _GasPressureChangers
from propylean import streams
import propylean.properties as prop

from math import pow
from propylean.validators import _Validators

# Start of final classes of pumps.
class CentrifugalPump(_PressureChangers):
    items = []    
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent Centrifugal Pump.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class
            min_flow:
                Required: No
                Type: int/float or tuple(value, unit) or VolumetricFlowrate(recommended)
                Acceptable values: Non-negative values
                Default value: None # TODO: Add industry standard
                Description: Minimum flow requirement of the pump

            NPSHr:
                Required: No
                Type: int/float or tuple(value, unit) or Length(recommended)
                Acceptable values: Non-negative values
                Default value: None
                Description: NPSHr of the pump
        
        PROPERTIES:
            NPSHa:
                Type: Length
                Description: NPSH available to the pump.
            
            head:
                Type: Length
                Description: Differential head generated by the pump.

            hydraulic_power:
                Type: Power
                Description: Hydraulic power generated by the pump.
            
            power/energy_in:
                Type: Power
                Description: Power required by the pump.

        
        RETURN VALUE:
            Type: CentrifugalPump
            Description: Returns an object of type CentrifugalPump with all properties of
                         a centrifugal pump used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> pump_1 = CentrifugalPump(tag="P1")
            >>> print(pump_1)
            Centrifugal Pump with tag: P1
        """
        super().__init__( **inputs)
        self._NPSHr = prop.Length()
        self._NPSHa = prop.Length()
        self._min_flow = prop.VolumetricFlowRate()
        del self.energy_out
        
        if 'min_flow' in inputs:
            self.min_flow = inputs['min_flow']
        if "NPSHr" in inputs:
            self.NPSHr = inputs['NPSHr']
        
        self._index = len(CentrifugalPump.items)
        CentrifugalPump.items.append(self)
    
    def __repr__(self):
        self = self._get_equipment_object(self)
        return "Centrifugal Pump with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())

    @property
    def min_flow(self):
        self = self._get_equipment_object(self)
        return self._min_flow
    @min_flow.setter
    def min_flow(self, value):
        _Validators.validate_arg_prop_value_type("min_flow", value, (prop.VolumetricFlowRate, int, float, tuple))
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.VolumetricFlowRate)
        if unit is None:
            unit = self._min_flow.unit
        self._min_flow = prop.VolumetricFlowRate(value, unit)
        self._update_equipment_object(self)

    @property
    def NPSHr(self):
        self = self._get_equipment_object(self)
        return self._NPSHr
    @NPSHr.setter
    def NPSHr(self, value):
        _Validators.validate_arg_prop_value_type("min_flow", value, (prop.Length, int, float, tuple))
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Length)
        if unit is None:
            unit = self._NPSHr.unit
        self._NPSHr = prop.Length(value, unit)
        self._update_equipment_object(self)
    
    @property
    def NPSHa(self):
        self = self._get_equipment_object(self)
        if self._inlet_material_stream_tag is None:
            raise Exception("Pump should be connected with MaterialStream at the inlet")
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        old_p_unit = self.inlet_pressure.unit
        self.inlet_pressure.unit = 'Pa'
        value = self.inlet_pressure.value/(9.8 * density.value)
        self.inlet_pressure.unit = old_p_unit
        return prop.Length(value, "m")

    @property
    def head(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        density = self._connected_stream_property_getter(is_inlet, "material", "density")
        density.unit = "kg/m^3"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = dp.value / (9.8 * density.value)
        return prop.Length(value, "m")
    @property
    def hydraulic_power(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Centrifugal Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        vol_flowrate = self._connected_stream_property_getter(is_inlet, "material", "vol_flowrate")
        vol_flowrate.unit = "m^3/h"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = vol_flowrate.value * dp.value / (3.6e3)
        return prop.Power(value, 'W')
    @property
    def power(self):
        self = self._get_equipment_object(self)
        self.hydraulic_power.unit = "W"
        value = self.hydraulic_power.value / self.efficiency.value
        return prop.Power(value, "W")
    @power.setter
    def power(self, value):
        #TODO Proived setting feature for power
        raise Exception("Pump power value setting is not yet supported. Modify differential pressure to get required power.")
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        # This is needed for property matching with the stream.
        _Validators.validate_arg_prop_value_type("energy_in", value, (prop.Power, int, float, tuple))
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)
    
    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('CentrifugalPump only supports energy inlet.')
            direction = 'in'
            stream_governed = False
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('CentrifugalPump only supports energy inlet.')
            direction = 'in'
            stream_type = "e"
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)


class PositiveDisplacementPump(_PressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent a Positive Displacement Pump.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class
            min_flow:
                Required: No
                Type: int/float or tuple(value, unit) or VolumetricFlowrate(recommended)
                Acceptable values: Non-negative values
                Default value: None # TODO: Add industry standard
                Description: Minimum flow requirement of the pump

            NPSHr:
                Required: No
                Type: int/float or tuple(value, unit) or Length(recommended)
                Acceptable values: Non-negative values
                Default value: None
                Description: NPSHr of the pump
        
        PROPERTIES:
            NPSHa:
                Type: Length
                Description: NPSH available to the pump.
            
            head:
                Type: Length
                Description: Differential head generated by the pump.

            accel_head:
                Type: Length
                Description: Acceleration head loss at the inlet of pump.
            
            power/energy_in:
                Type: Power
                Description: Power required by the pump.

        RETURN VALUE:
            Type: PositiveDisplacementPump
            Description: Returns an object of type PositiveDisplacementPump with all properties of
                         a positive displacement pump used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> pump_1 = PositiveDisplacementlPump(tag="P1")
            >>> print(pump_1)
            Positive Displacement Pump with tag: P1
        """
        super().__init__( **inputs)
        self._speed = prop.Frequency()
        self._NPSHr = prop.Length()
        if "NPSHr" in inputs:
            self.NPSHr = inputs['NPSHr']
        del self.energy_out

        self._index = len(PositiveDisplacementPump.items)
        PositiveDisplacementPump.items.append(self)
    
    def __repr__(self):
        self = self._get_equipment_object(self)
        return "Positive Displacement Pump with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())
    
    @property
    def NPSHa(self):
        self = self._get_equipment_object(self)
        if self._inlet_material_stream_tag is None:
            raise Exception("Pump should be connected with MaterialStream at the inlet")
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        old_p_unit = self.inlet_pressure.unit
        old_acc_head_unit = self.accel_head.unit
        self.inlet_pressure.unit = self.accel_head = 'Pa'
        
        value = (self.inlet_pressure.value - self.accel_head.value)/(9.8 * density.value)
        self.inlet_pressure.unit = old_p_unit
        self.accel_head.unit = old_acc_head_unit
        return prop.Length(value, "m")
    @property
    def NPSHr(self):
        self = self._get_equipment_object(self)
        return self._NPSHr
    @NPSHr.setter
    def NPSHr(self, value):
        _Validators.validate_arg_prop_value_type("min_flow", value, (prop.Length, int, float, tuple))
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Pressure)
        if unit is None:
            unit = self._NPSHr.unit
        self._NPSHr = prop.Length(value, unit)
        self._update_equipment_object(self)
    
    @property
    def head(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        density = self._connected_stream_property_getter(is_inlet, "material", "density")
        density.unit = "kg/m^3"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = dp.value / (9.8 * density.value)
        return prop.Length(value, "m")
    
    @property
    def accel_head(self):
        L, V = self._get_suction_length_velocity()
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        SG = density.value/1000
        k = 1
        head_loss = L.value * V.value * self.speed * SG / (k * 9.8) 
        return prop.Length(head_loss, 'm')
    
    @property
    def power(self):
        self = self._get_equipment_object(self)
        is_inlet = False if self._inlet_material_stream_index is None else True
        vol_flow = self._connected_stream_property_getter(is_inlet, "material", "vol_flowrate")
        vol_flow.unit = "gal/min"
        old_dp_unit = self.differential_pressure.unit
        self.differential_pressure.unit = 'psi'
        value = vol_flow.value * self.differential_pressure.value /(1714 * self.efficiency.value)
        return prop.Power(value, "hp")
    @power.setter
    def power(self, value):
        #TODO Proived setting feature for power
        raise Exception("Pump power value setting is not yet supported. Modify differential pressure to get required power.")
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        _Validators.validate_arg_prop_value_type("energy_in", value, (prop.Power, int, float, tuple))
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)
        
    
    @classmethod
    def list_objects(cls):
        return cls.items

    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('PositiveDisplacementPump only supports energy inlet.')
            direction = 'in'
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('PositiveDisplacementPump only supports energy inlet.')
            direction = 'in'
            stream_type = "e"
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)
# End of final classes of pumps

# Start of final classes of Compressors and TurboExpanders.
class CentrifugalCompressor(_GasPressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent a Centrifugal Compressors.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class  
            efficiency:
                Required: No
                Type: int or float (recommended)
                Acceptable values: Non-negative values
                Default value: 1.4
                Description: Efficiency of the compressor. Efficency can be set as
                             adiabatic/isentropic or polytropic efficiency.
            
            adiabatic_efficiency:
                Required: No
                Type: int or float (recommended)
                Acceptable values: Non-negative values
                Default value: 0.7
                Description: Efficiency of the compressor considering adiabatic/isentropic
                             compression.
            
            polytropic_efficiency:
                Required: No
                Type: int or float (recommended)
                Acceptable values: Non-negative values
                Default value: 0.7
                Description: Efficiency of the compressor considering polytropic
                             compression.

            polytropic_exponent:
                Required: No
                Type: int or float (recommended)
                Acceptable values: Non-negative values
                Default value: 1.4
                Description: Polytropic exponent of the gas.
        
        PROPERTIES:
            power:
                Type: Power
                Description: Power required by the compressor.

        
        RETURN VALUE:
            Type: CentrifugalCompressor
            Description: Returns an object of type CentrifugalCompressor with all properties of
                         a Centrifugal Compressor used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> CC_1 = CentrifugalCompressor(tag="P1")
            >>> print(CC_1)
            Centrifugal Compressor with tag: P1
        """
        super().__init__( **inputs)
        del self.energy_out
        self._index = len(CentrifugalCompressor.items)
        CentrifugalCompressor.items.append(self)
    
    def __repr__(self):
        self = self._get_equipment_object(self)
        return "Centrifugal Compressor with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())
    
    @property
    def temperature_increase(self):
        return super().temperature_change
    
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        # This is needed for property matching with the stream.
        _Validators.validate_arg_prop_value_type("energy_in", value, (prop.Power, int, float, tuple))
        _Validators.validate_non_negative_value("energy_in", value)
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)
    
    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('CentrifugalCompressor only supports energy inlet.')
            direction = 'in'
            stream_governed = False
            stream_type = "e"
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'out' in direction:
                raise Exception('CentrifugalCompressor only supports energy inlet.')
            direction = 'in'
            stream_type = 'e'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)

class TurboExpander(_GasPressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        super().__init__( **inputs)
        del self.energy_in
        self._index = len(TurboExpander.items)
        TurboExpander.items.append(self)
    
    def __repr__(self):
        self = self._get_equipment_object(self)
        return "TurboExpander with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())

    @property
    def temperature_decrease(self):
        return super().temperature_change
    @property
    def temperature_increase(self):
        t_increase = super().temperature_change
        t_increase.value *= -1
        return t_increase

    @property
    def energy_out(self):
        return self.power
    @energy_out.setter
    def energy_out(self, value):
        # This is needed for property matching with the stream.
        _Validators.validate_arg_prop_value_type("energy_out", value, (prop.Power, int, float, tuple))
        _Validators.validate_non_negative_value("energy_out", value)
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_out.unit
        self._energy_out = prop.Power(value, unit)
        self._update_equipment_object(self)

    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self,
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'in' in direction:
                raise Exception('TurboExpander only supports energy outlet.')
            direction = 'out'
            stream_type = 'e'
            stream_governed = False
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            if direction is not None and 'in' in direction:
                raise Exception('TurboExpander only supports energy outlet.')
            direction = 'out'
            stream_type = 'e'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)

# End of final classes of compressors