"""
Client for the kappa programming language through standard channel api
"""

import subprocess
import threading
import json
import abc
import kappy.kappa_common

from pkg_resources import resource_filename

class KappaStd(object):
    def __init__(self, path=None, delimiter='\x1e', args=None):
        """
        Kappa tools driver

        path -- where to find kappa executables
        delimiter -- What to use to delimit messages (must not appears in message body default '\x1e')
        args -- arguments to pass to kappa executables
        """
        self.delimiter = delimiter
        if not path:
            path = resource_filename(__name__,"bin/KaSimAgent")
        sim_args = [path,
                    "--delimiter",
                    "\\x{:02x}".format(ord(self.delimiter)),
                    "--log",
                    "-", ]
        if args:
            sim_args = sim_args + args
        self.lock = threading.Lock()
        self.message_id = 0
        self.popen = subprocess.Popen(sim_args,
                                      stdin=subprocess.PIPE,
                                      stdout=subprocess.PIPE,
                                      stderr=subprocess.STDOUT)

    def __del__(self):
        self.shutdown()

    def get_message_id(self):
        self.message_id += 1
        return self.message_id

    def dispatch(self, method, args=None):
        if args is not None:
            data = [method, args]
        else:
            data = method

        try:
            self.lock.acquire()
            message_id = self.get_message_id()
            message = {'id': message_id,'data': data}
            message = "{0}{1}".format(json.dumps(message), self.delimiter)
            self.popen.stdin.write(message.encode('utf-8'))
            self.popen.stdin.flush()
            buffer = bytearray()
            c = self.popen.stdout.read(1)
            while c != self.delimiter.encode('utf-8') and c:
                buffer.extend(c)
                c = self.popen.stdout.read(1)
            response = json.loads(buffer.decode('utf-8'))
            if response["id"] != message_id:
                raise KappaError("expect id {0} got {1}".format(response["id"], message_id))
            else:
                return self.projection(response)

        finally:
            self.lock.release()

    @abc.abstractmethod
    def projection(self, response): pass

    def shutdown(self):
        self.popen.stdin.close()
        self.popen.stdout.close()
        self.popen.kill()

    def projection(self,response):
        result_data = response["data"]["result_data"]
        data = result_data[1]
        if result_data[0] == "Ok":
            return data[1]
        else:
            raise kappy.kappa_common.KappaError(data)

    def project_parse(self,overwrites=[]):
        """
        Parses the project

        overwrites -- list of algebraic variables to overwrite
        Each element has the form {var : "variable_name", val : 42 }
        """
        return(self.dispatch("ProjectParse",overwrites))

    def file_create(self,file_object):
        """
        Add a file to the project

        file_object -- a Kappa_common.File
        """
        file_data = file_object.toJSON()
        return(self.dispatch("FileCreate",file_data))

    def file_delete(self,file_id):
        """
        Remove a file from the project
        """
        return(self.dispatch("FileDelete",file_id))

    def file_get(self,file_id):
        """
        Returns file file_id stored in the project
        """
        f = self.dispatch("FileGet",file_id)
        return(kappy.kappa_common.hydrate_file(f))

    def file_info(self):
        """
        Lists the files of the project (returns the FileMetadata
        """
        info = self.dispatch("FileCatalog")
        #return(list(map(hydrate_filemetadata,info)))
        return(map(kappy.kappa_common.hydrate_file_metadata, info))

    def simulation_delete(self):
        """
        Deletes running/paused simulation
        """
        return(self.dispatch("SimulationDelete"))

    def simulation_file_line(self,file_line_id):
        """
        Returns the file file_line_id generated by $PRINT perturbations
        """
        return(self.dispatch("SimulationDetailFileLine",file_line_id))

    def simulation_DIN(self,DIN_id):
        """
        Returns a given generated DIN
        """
        return(self.dispatch("SimulationDetailFluxMap",flux_map_id))

    def simulation_log_messages(self):
        """
        Returns simulation log
        """
        return(self.dispatch("SimulationDetailLogMessage"))

    def simulation_plot(self,limit=None):
        """
        Returns the plotted values of the simulation

        limit -- optionnal boundaries to only get a subplot
        format: { offset : 100, nb_points : 500 }
        returns the last points if offset is Null
        """
        if limit is not None:
            parameter = limit.toJSON()
        else:
            parameter = kappy.kappa_common.PlotLimit().toJSON()
        return(self.dispatch("SimulationDetailPlot",parameter))

    def simulation_snapshot(self,snapshot_id):
        """
        Returns a given generated snapshot
        """
        return(self.dispatch("SimulationDetailSnapshot",snapshot_id))

    def simulation_info(self):
        """
        Returns state and progress of the simulation
        """
        return(self.dispatch("SimulationInfo"))

    def simulation_info_file_line(self):
        """
        Lists files generated by $PRINT during the simulation
        """
        return(self.dispatch("SimulationCatalogFileLine"))

    def simulation_DINs(self):
        """
        Lists DIN generated during the simulation
        """
        return(self.dispatch("SimulationCatalogFluxMap"))

    def simulationsnapshots(self):
        """
        Lists snapshots generated during the simulation
        """
        return(self.dispatch("SimulationCatalogSnapshot"))

    def simulation_pause(self):
        """
        Pauses a simulation
        """
        return(self.dispatch("SimulationPause"))

    def simulation_perturbation(self,perturbation_code):
        """
        Fires a perturbation in a paused simulation
        """
        return(self.dispatch("SimulationPerturbation",
                              { "perturbation_code" : perturbation_code }))

    def simulation_start(self,simulation_parameter):
        """
        Launches a simulation from a parsed model

        simulation_parameter -- is described in kappa_common.SimulationParameter
        """
        return(self.dispatch("SimulationStart",
                             simulation_parameter.toJSON()))

    def simulation_continue(self,pause_condition):
        """
        Restarts a paused simulation
        """
        return(self.dispatch("SimulationContinue",pause_condition))
